; $Id: $
;#######################################################################
; 
; NAME:
;  dm_load_nxspe
;
; PURPOSE:
;  load nxspe file (binary spe+phx) returns zdat[nx,ny], err[nx,ny], xdat[nx or nx+1], ydat[ny or ny+1]
;  
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Andrei Savici
;  Oak Ridge National Lab
;  United States
;  saviciat@ornl.gov
;  April, 2020
; 
; Based on dm_load_spe.pro and dm_load_phx.pro
;
; HISTORY:
; 04/2010 Added flag to keep track of ki/kf correction
; 07/2011 Add finite value check for the mask
; 
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################
@dm_load_dcsdata  ;for dm_findbetween

;header = [ndet,nen,emin,emax,lamcorr,(fiex eief)]
pro dm_load_nxspe,file,data=data,err_data=err_data,tta=tta,dtt=dtt,phia=phia,dphi=dphi,energy=energy,eief=eief,tttit=tttit,phitit=phitit,entit=entit,$
    error=error,group_leader=group_leader,header=header,angi=angi,lamcorr=lamcorr
  
  error = 1b
  if (~file_test(file))then begin  
    ok = dialog_message("Can't open "+file,/error,dialog_parent=group_leader)
    return
  endif
  
  if (~H5F_IS_HDF5(file))then begin  
    ok = dialog_message(file+" is not in HDF5 format",/error,dialog_parent=group_leader)
    return
  endif
  
  catch, myerror
  if myerror ne 0 then begin
     error = 1b
     catch,/cancel
     ok = dialog_message("Missing elements in the NXSPE file. Please check format!",/error)
     return
  endif
  
  error       = 0b  ;clear error flag
  file_id     = H5F_OPEN(file)
  main_g_id   = H5G_GET_MEMBER_NAME(file_id,"/",0)
  id_data     = h5d_open(file_id,"/"+main_g_id+"/data/data")
  datatype_id = H5D_GET_TYPE(id_data)
  idd         = H5D_GET_SPACE(id_data)
  sSpace      = H5S_GET_SIMPLE_EXTENT_DIMS(idd)
  ny = sspace[0]
  nx = sspace[1]
  ;Note: data and err_data are transposed with respect to original spe format
  data = transpose(h5d_read(id_data))
  H5d_CLOSE,id_data
  
  id_energy = h5d_open(file_id,"/"+main_g_id+"/data/energy")
  energy    = h5d_read(id_energy)
  H5d_CLOSE,id_energy
  if arg_present(header) then begin
     if n_elements(energy) eq (ny+1) then $
        min_eng = min((energy[0:ny-1]+energy[1:ny])/2.,max=max_eng) $
     else $ 
        min_eng = min(energy,max=max_eng)
     header = [nx,ny,min_eng,max_eng]
  endif
 
  id_nxinfo = H5G_OPEN(FILE_ID,MAIN_G_ID+'/NXSPE_info')
  nxinfo_fields_num = H5G_GET_NUM_OBJS(ID_NXINFO)
  nxinfo_fields     = strarr(nxinfo_fields_num)
  for i=0,nxinfo_fields_num-1 do nxinfo_fields[i]=H5G_GET_OBJ_NAME_BY_IDX(id_nxinfo,i)
  
  if where(STRLOWCASE(NXINFO_FIELDS) eq 'ki_over_kf_scaling') eq -1 then begin
     lamcorr=0b
  endif else begin
     id_lamcorr = h5d_open(id_nxinfo,"ki_over_kf_scaling")
     lamcorr   =(h5d_read(id_lamcorr))[0]
     if (lamcorr ne 1b) then lamcorr=0b 
     H5d_CLOSE,id_lamcorr
  endelse
  ;check if data has constant energy steps or constant time steps
  diff = (energy-shift(energy,1))[1:n_elements(energy)-2]
  if (max(diff)-min(diff)) lt diff[0]*1e-5 then lamcorr+=2b ;add 2 for constant energy steps, so lamcorr=0 coresponds to DCS case
  if arg_present(header) then header = [header,lamcorr]
  
  if where(strupcase(NXINFO_FIELDS) eq 'FIXED_ENERGY') eq -1 then begin
     if ~arg_present(header) then $ 
        eief = float((dm_dialog_input('E_fixed (meV)',default=0,info=file+ ' does not seem to contain a fixed energy',dialog_parent=group_leader))[0])
  endif else begin
     id_eief = h5d_open(id_nxinfo,"fixed_energy")
     eief    = (h5d_read(id_eief))[0]
     H5d_CLOSE,id_eief
     if arg_present(header) then header = [header,eief]
  endelse

  if where(strupcase(NXINFO_FIELDS) eq 'PSI') eq -1 then begin
     angi = !VALUES.F_NAN
  endif else begin
     id_ang = h5d_open(id_nxinfo,"psi")
     angi   =(h5d_read(id_ang))[0]
     H5d_CLOSE,id_ang
  endelse  
  H5G_CLOSE, id_nxinfo
  
  if ~arg_present(data) then begin  ;header only
     H5F_CLOSE, file_id
     return
  endif
  
  id_error  = h5d_open(file_id,"/"+main_g_id+"/data/error")
  err_data  = transpose(h5d_read(id_error))
  H5d_CLOSE,id_error
  
  id_2t     = h5d_open(file_id,"/"+main_g_id+"/data/polar") 
  tta       = h5d_read(id_2t)
  H5d_CLOSE,id_2t
  
  id_d2t    = h5d_open(file_id,"/"+main_g_id+"/data/polar_width")
  dtt       = h5d_read(id_d2t)
  H5d_CLOSE,id_d2t
  
  id_phi    = h5d_open(file_id,"/"+main_g_id+"/data/azimuthal")
  phia      = h5d_read(id_phi)
  H5d_CLOSE,id_phi
  
  id_dphi   = h5d_open(file_id,"/"+main_g_id+"/data/azimuthal_width")
  dphi      = h5d_read(id_dphi)
  H5d_CLOSE,id_dphi
  
  H5F_CLOSE, file_id
end

;this procedure reads a spe file for dcs_mslice program
;Parameters:
;   open_path:  file path, string
;   open_file:  file name(s), string or string array
;Keywords:
; input:
;   avgsum:     0:arithmetic average 1:sum, effective when diffuse keyword is sest, default is self.bin_avgsum
;   background: if set, the loaded files are regarded as background files and will be summed
;   diffetyp:   only needed when diffuse keyword is set 0:single energy  1:multiple energy
;   diffuse:    if set, diffuse scattering mode
;   parent:     message widget parent,  default is self.tlb
;   title:      message title
;   eint_yn:    only applicable for diffuse mode, if set, I=sum(I[i,j]*dE[i,j]) over the specified time channels
;   e_range:    energy range - diffuse keyword set, default is self.e_range
;   t_chan:     0:all 1:elastic 2:specify erange  - efffective when diffuse keyword set, default is self.e_bin[self.samp_typ]
;   phxfile:    string containing the phx file name
;   eief:       if present and given a value, the procedure will not prompt for Ei or Ef value from user
;   masklimit:  a number indicating the threshhold below which the detector is masked, default:-1e20
; output:
;   error:      error flag 1 or 0
;   qty:        [ne,ndet] or [ne,ndet,nfiles]-diffuse keywords set [ne,ndet] if nfiles=1
;   dqty:       [ne,ndet] or [ne,ndet,nfiles]-diffuse keywords set [ne,ndet] if nfiles=1
;   ang:        [nfiles] - diffuse keyword set, not in spe file, need to be typed in by the user
;   eief:       Ei or Ef(depends on the instrument geometry), not in the spe file, need to be typed in by the user
;   emean:      [ne]
;   ewid:       diffuse:ewid[tchans] ; powder,single:[ntchan], the corresponding energy width for the time channels
;   info:       {ne:ne, ndet:ndet}  used for checking file compatibility
;   lamcorr:    lamcorr flag,0: no (const t bins) 1: yes (const t bins) 2: no (const E bins) 3: yes (const E bins)
;   weight:     weight or duration info , not in the file, need to be typed in by the user
;   kfactor:    [ntchan]-(ki/kf)  or a scalor
pro dcs_mslice::dm_load_nxspe,open_path,open_file,ang=ang,avgsum=avgsum,background=background,$
    diffetyp=diffetyp,diffuse=diffuse,eief=eief,e_range=e_range,eint_yn=eint_yn,emean=emean,$
    error=error,ewid=ewid,info=info,kfactor=kfactor,lamcorr=lamcorr,parent=parent,phxfile=phxfile,qty=qty,dqty=dqty,$
    scriptmode=scriptmode,t_chan=t_chan,title=title,masklimit=masklimit,adddata=adddata,debug=debug,weight=weight
    ;   eadjust:    0: none 1:automatic 2:specified
    ;   epadjust:   for eadjust=2, epadjust specified the apparent elastic peak position
    ;   epadcheckfirst: if set, only check the first file for the shifting of elastic peak, and use it for all files
    eadjust        = self.eadjust
    epadjust       = self.eadjust_spec
    epadcheckfirst = self.eadjust_checkfirst
    if n_elements(masklimit)   eq 0 then masklimit = -1e20
    if (eadjust eq 2) and (~ finite(epadjust)) then eadjust=0
    sch_ead        = (eadjust gt 0)
    cd,current=current
    if file_test(self.dirs[1],/directory) then cd,self.dirs[1]        
    if keyword_set(diffuse) then begin
       if n_elements(avgsum)   eq 0 then avgsum   = self.bin_avgsum
       if n_elements(t_chan)   eq 0 then t_chan   = self.e_bin[self.samp_typ]
       if n_elements(e_range)  ne 2 then e_range  = self.e_range
       if n_elements(diffetyp) eq 0 then diffetyp = (self.samp_typ eq 3)
       if diffetyp eq 0 then sch_ead = 0b
       if keyword_set(adddata) and t_chan ne 0 then t_chan = 2 ;forced to specify
    endif else begin
       if n_elements(diffetyp) eq 0 then diffetyp = 1     ;default is multiple energy
    endelse
    if n_elements(parent) eq 0 then parent = self.tlb
    if n_elements(title)  eq 0 then title  = 'Loading data ...'
    error   = 0b  ;initialize no error
    n_files = n_elements(open_file)
    n_read  = 0

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if obj_valid(mesg) then obj_destroy,mesg
       ok = dialog_message(!error_state.msg,/error)
       return
    endif


    ;ask for weight info
    myweight = 1.0
    if (arg_present(weight) or (n_files ne 1)) and (~keyword_set(scriptmode)) then begin
       myweight = !values.f_nan & default = '1'
       while(total(finite(myweight)) ne n_files) do begin
            myweight = dm_dialog_listinput(open_file,title='Please input weight info:',default=default,$
                 label='weight',dialog_parent=parent,reset=1,info=['The weight is for summing files and weighted mean average binning, not for normalizing the intensity.',$
                 "Press OK if you don't know the information."],xvalue=open_file)
            default = myweight & myweight = dm_to_number(myweight)
       endwhile
    endif
    
    ;rotation angles
    if keyword_set(diffuse) and (~ keyword_set(background)) then begin
       ang = fltarr(n_files)
    endif

    if keyword_set(debug) then t=systime(1)
    for i=0L,n_files-1L do begin
        file = open_path+self.pathsep+open_file[i]
        if i eq 0 then $
           mesg =  obj_new('dm_progress',title=title,message='Loading '+file,group_leader=parent) $
        else $
           mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)

        ;read the nxspe file
        dm_load_nxspe,file,data=data,err_data=err_data,tta=tta,dtt=dtt,phia=phia,dphi=dphi,energy=energy,eief=eief,tttit=tttit,$
        phitit=phitit,entit=entit,error=error,group_leader=parent,angi=angi,lamcorr=lamcorr
        if error ne 0 then begin
           if obj_valid(mesg) then obj_destroy,mesg
           return
        endif 
     
        if keyword_set(diffuse) and (~ keyword_set(background)) then ang[i]=angi
        
        ;check det masking
        ii = 0L & ncheck = n_elements(data[*,0]) & mskcount = ncheck
        while(mskcount eq ncheck) do begin
             mask = where(((data[*,ii] le masklimit) or (~finite(data[*,ii]))),mskcount,complement=tmpnomask)
             ii = ii+1    ;in case the first time channel is masked
        endwhile
        if mskcount gt 0 then begin
           if n_elements(nomask) eq 0 then begin
              nomask = tmpnomask 
           endif else begin
              nomask = dm_common(nomask,tmpnomask,count)
              if count eq 0 then tmp = temporary(nomask)
           endelse
           data[mask,*]     = !values.f_nan
           err_data[mask,*] = !values.f_nan
        endif          
        ;check time channel masking
        ii = 0L & ncheck = n_elements(data[0,*]) & mskcount = ncheck
        while(mskcount eq ncheck) do begin
             mask = where(((data[ii,*] le masklimit) or (~finite(data[ii,*]))),mskcount)
             ii = ii+1    ;in case the first detector is masked
        endwhile
        if mskcount gt 0 then begin
           data[*,mask]     = !values.f_nan
           err_data[*,mask] = !values.f_nan
        endif
        
        ndet = n_elements(tta)
        nen  = n_elements(energy)
        zdim = size(data,/dimension)

        if n_elements(info) eq 0 then begin
           info={ndet:ndet,nen:nen,nx:zdim[0],ny:zdim[1]}
        endif else begin
           if (info.ndet ne ndet) or (info.nen ne nen) or (info.nx ne zdim[0]) or (info.ny ne zdim[1]) then begin
              ok = dialog_message('Selected data files are of different types.',/error,dialog_parent=parent)
              error = 1b                          ;error occurs
              if obj_valid(mesg) then obj_destroy,mesg
              return
           endif
        endelse

        if i eq 0 then begin
           emean = temporary(energy)
           tth   = temporary(tta)
           dtth  = temporary(dtt)
           psi   = temporary(phia)
           dpsi  = temporary(dphi)
           ewid  = emean[1:nen-1]-emean[0:nen-2]
           if n_elements(ewid) ne zdim[1] then ewid = [ewid,2*ewid[nen-2]-ewid[nen-3]]
           if nen  eq zdim[1]+1 then emean = (emean[0:nen-2]+emean[1:nen-1])/2.

           if keyword_set(diffuse) then begin
              weight  = myweight
              all_tch = 1
              kfactor = 1.0
              ;search the proper time channels to be summed or averaged
              case t_chan of
                   0:          ;all time channels
                   1:  begin   ;elastic peak
                       if n_elements(nomask) ne 0 then $
                          tmp_y    = total(data[nomask,*],1,/double,/nan) $
                       else $
                          tmp_y    = total(data,1,/double,/nan)    
                       dm_gaussfit,tmp_y,params=params,ny=ny,fitnotgood=fitnotgood,debug=debug
                       if fitnotgood then begin
                          ok = dialog_message(["Can't locate the elastic peak from the first file.","All time channels are used instead."],$
                               dialog_parent=parent)
                       endif else begin
                          all_tch = 0     ;fitting is good
                          tchans  = params[1]-fix(2.5*params[2])+indgen(fix(5.0*params[2])+1)
                          index   = where(tchans lt 0, count)
                          if count ne 0 then tchans[index] = tchans[index]+ny
                          index   = where(tchans ge ny,count)
                          if count ne 0 then tchans[index] = tchans[index]-ny
                          emin    = min(emean[tchans],max=emax)
                          e_range = [emin,emax]
                       endelse
                       end
                   2:  begin   ;specified energy range
                       if n_elements(e_range) eq 2 then $
                          if total(finite(e_range)) eq 2 then begin
                             tchans = where((emean ge e_range[0]) and (emean le e_range[1]),count)
                             if count ne 0 then begin
                                all_tch = 0
                                if arg_present(kfactor) then begin
                                   if self.instrgeom eq 0 then begin   ;direct geometry
                                      kfactor = sqrt(eief/(eief-mean(e_range))) ;ki/kf_avg
                                   endif else begin   ;inverse geometry
                                      kfactor = sqrt((eief+mean(e_range))/eief) ;ki_avg/kf
                                   endelse
                                endif
                                break
                             endif
                          endif
                       ok = dialog_message(['The energy range specified is invalid.','All time channels are used instead.'],$
                                dialog_parent=parent)
                       end
                   else:
              endcase
              if all_tch eq 1 then begin
                 tchans = indgen(zdim[1])
                 emin   = min(emean,max=emax)
                 e_range = [emin,emax]
              endif
              ewid = ewid[tchans]
              nitem = (avgsum eq 1)?1.0:(1.0>(n_elements(ewid)))
              if keyword_set(eint_yn) then factor=1.0 else factor = 1.0/nitem
              if (diffetyp eq 1) and arg_present(kfactor) then begin
                 if self.instrgeom eq 0 then begin   ;direct geometry
                    kfactor = (sqrt(eief/(eief-emean)))[tchans]
                 endif else begin   ;inverse geometry
                    kfactor = (sqrt((eief+emean)/eief))[tchans]
                 endelse
              endif
           endif else begin
              qty    = 0.0
              dqty   = 0.0
              weight = 0.0
           endelse
        endif

        if sch_ead then begin          ;elastic peak adjustment
           erotate = 0
           i_el = dm_findbetween(emean,0)
           if (~ finite(i_el)) then $
              ok = dialog_message('No E=0 in the original data. Elastic peak position adjustment is skipped.',dialog_parent=parent,title='Warning: '+open_file[i]) 
           if eadjust eq 1 then begin
              if n_elements(nomask) ne 0 then $
                 tmp_y    = total(data[nomask,*],1,/double,/nan) $
              else $
                 tmp_y    = total(data,1,/double,/nan)
              dm_gaussfit,tmp_y,params=params,fitnotgood=fitnotgood,debug=debug
              if fitnotgood then begin;somehow the fitting is not right use all time channels
                 ok = dialog_message(["Can't locate the elastic peak.","Elastic peak adjustment is skipped."],$
                      dialog_parent=parent,title='Warning: '+open_file[i])
                 i_eadj = !values.f_nan
              endif else begin
                 i_eadj = fix(params[1]+0.5)
              endelse
           endif else if eadjust eq 2 then begin
              i_eadj = dm_findbetween(emean,epadjust)
              if (~ finite(i_eadj)) then $
                 ok = dialog_message('No E='+dm_to_string(epadjust)+' in the original data. Elastic peak position adjustment is skipped.',$
                      dialog_parent=parent,title='Warning: '+open_file[i])
           endif
           if finite(i_el) and finite(i_eadj) then erotate = fix(i_el-i_eadj) $
           else tmp= temporary(erotate)  ;destroy erotate
           if keyword_set(epadcheckfirst) and n_elements(erotate) ne 0 then sch_ead = 0b   
        endif
        if n_elements(erotate) ne 0 then begin
           data  = shift(data,0,erotate)
           err_data   = shift(err_data,0,erotate)
        endif

        ;add or append data
        if keyword_set(diffuse) then begin
           data  = data[*,tchans]
           err_data = err_data[*,tchans]
           if diffetyp eq 0 then begin
              if keyword_set(eint_yn) then begin
                 for j=0L,n_elements(data[*,0])-1 do begin
                     data[j,*] = data[j,*]*ewid
                     err_data[j,*] = err_data[j,*]*ewid
                 endfor
              endif
              if i eq 0 then begin
                 qty  = total(data,2,/nan)*factor
                 dqty = sqrt(total(err_data^2,2,/nan))*factor
              endif else begin
                 qty  = [[qty],[total(data,2,/nan)*factor]]
                 dqty = [[dqty],[sqrt(total(err_data^2,2,/nan))*factor]]
              endelse
           endif else begin
              if i eq 0 then begin
                 qty  = transpose(data)
                 dqty = transpose(err_data)
              endif else begin
                 qty  = [[[qty]],[[transpose(data)]]]
                 dqty = [[[dqty]],[[transpose(err_data)]]]
              endelse
           endelse
        endif else begin
           err_data = err_data^2  ;error bar
           ;handle maskings
           ii = 0L
           while(total(finite(qty[*,ii])) eq 0) do ii=ii+1    ;in case the first time channel is masked
           ind = where(~finite(qty[*,ii]),count)
           if count ne 0 then begin
              qty[ind,*]  = data[ind,*]*weight
              dqty[ind,*] = err_data[ind,*]*((weight+myweight[i])^2-myweight[i]^2)
           endif
           ii = 0L
           while(total(finite(data[*,ii])) eq 0) do ii=ii+1    ;in case the first time channel is masked
           mask = where(~finite(data[*,ii]),mskcount)
           if (mskcount gt 0) and (i gt 0) then begin
              data[mask,*] = qty[mask,*]/weight
              err_data[mask,*] = dqty[mask,*]*((weight+myweight[i])^2-weight^2)/(weight^2*myweight[i]^2) 
           endif
           qty    = temporary(qty)+temporary(data)*myweight[i]
           dqty   = temporary(dqty)+temporary(err_data)*(myweight[i]^2)
           weight = temporary(weight)+myweight[i]
        endelse
        n_read = n_read+1
        if (~ obj_valid(mesg)) then break
    endfor
 
    ;ask for rotation angles
    if keyword_set(diffuse) and (~ keyword_set(background)) and (~keyword_set(scriptmode)) then begin
       default = dm_to_string(ang) & ang = !values.f_nan 
       while(total(finite(ang)) ne n_files) do begin
          ang = dm_dialog_listinput(open_file,title='Please input rotation angles:',default=default,$
                label='angle',dialog_parent=parent,reset=0,xvalue=open_file)
          default = ang & ang = dm_to_number(ang)
       endwhile
    endif
    ;if n_read ne n_files and n_elements(ang) ne 0 then ang = ang[0:n_read-1]

    if (~ keyword_set(diffuse)) then begin
       if arg_present(kfactor) then begin
          if self.instrgeom eq 0 then begin   ;direct geometry
             kfactor = sqrt(eief/(eief-emean))
          endif else begin   ;inverse geometry
             kfactor = sqrt((eief+emean)/eief)
          endelse
       endif
       if weight eq 0 then weight = 1.0
       qty  = qty/weight
       dqty = sqrt(dqty)/weight
       qty  = transpose(qty)   ;->[nen,ndet]
       dqty = transpose(dqty)  ;->[nen,ndet]
    endif else begin
       if diffetyp eq 1 then emean=emean[tchans] $
       else begin
          if t_chan eq 2 then emean=mean(emean[tchans]) $
          else emean = 0.0
       endelse
       if keyword_set(background) and (N_elements(weight) gt 1) then begin    ;sum the files
          ind  = (size(qty))[0]
          totw = total(weight)
          if totw le 0 then totw = 1.0
          weight = weight/totw
          for i=0,n_elements(weight)-1 do begin
              qty[*,i]  = qty[*,i]*weight[i]
              dqty[*,i] = (dqty[*,i]*weight[i])^2
          endfor
          qty    = total(qty,ind,/nan)          ;not bullet_proof for different masking
          dqty   = sqrt(total(dqty,ind,/nan))
          weight = totw
       endif
    endelse
    
    ;read the information about angles (used to be in phx file)
    if n_elements(tth) eq 0 then return ;no data available
    ptr_free,self.detPosPtr
    self.detPosPtr = ptr_new({two_theta:temporary(tth),dtwo_theta:temporary(dtth),$
                     psi:temporary(psi),dpsi:temporary(dpsi)})
    self->my_widget_control,'nxspe_savephxBut',widget_info(self.tlb,find_by_uname='ftoolMenu'),/sensitive

    if keyword_set(debug) then print,"load time is ",systime(1)-t,'seconds'
    if obj_valid(mesg) then obj_destroy,mesg
    cd,current
end

;the following procedure is for loading detector dependent background file for dcs_mslice
pro dcs_mslice::dm_load_nxspe_detbackground,open_path,open_file,error=error,qty=qty,dqty=dqty,parent=parent
    self->dm_load_nxspe,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,e_range=self.bgeran,avgsum=0,$
        title='loading background file...',parent=parent,error=error,/diffuse,/background,diffetyp=0
end

;the following procedure is for loading time channel background file for dcs_mslice
pro dcs_mslice::dm_load_nxspe_tchanbackground,open_path,open_file,error=error,qty=qty,dqty=dqty,$
    parent=parent,emean=emean
    self->dm_load_nxspe,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,e_range=self.dcor_eqran[0:1],$
        avgsum=0,title='loading background file...',parent=parent,error=error,/diffuse,/background,$
        diffetyp=1,emean=emean,eief=eief,phxfile=phxfile
end