; $Id: dm_dialog_listinput.pro,v 1.15 2017/07/11 14:32:13 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_dialog_listinput
;
; PURPOSE:
;  a dialog input for data lists
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  July, 2017
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

pro dm_dialog_listinput_Exit,tlb,tmp=tmp
    ;do nothing, leave the job to obj_destroy
end

;widget event handler
pro dm_dialog_listinput_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;object event handler
pro dm_dialog_listinput::event, event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    widget_control,/hourglass
    ;error handling
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error)
       return
    end
    ;make the dialog window always on top of the screen
    if tag_names(event, /STRUCTURE_NAME) eq 'WIDGET_KBRD_FOCUS' then begin
       if event.id eq self.tlb then self->totop
    endif
    eventname = widget_info(event.id,/uname)
    eventname = strlowcase(eventname[0])
    case eventname of
       'input':begin
            ;if event.type eq 3 then begin
            ;   widget_control,event.id,set_table_select=[-1,-1,-1,-1]
            ;   if (!version.os_family ne 'Windows') and (event.y ge 0) then widget_control,event.id,set_table_view=[0,0>(event.y-14)]
            ;endif
            if event.type ne 0 then return
            if event.ch ne 13 and event.ch ne 10 then return
            widget_control,event.id,get_value=tmp
            if self.type eq 0 then begin
               if finite(dm_to_number(tmp[event.x,event.y]),/nan) then begin
                  ok = dialog_message('The value entered is not a valid number.',/error,dialog_parent=self.tlb)      
                  tmp[event.x,event.y] = (*self.value)[event.x,event.y]       
                  widget_control,event.id,set_value=tmp
                  event.y = event.y-1
               endif
            endif
            widget_control,event.id,set_table_select=[-1,-1,-1,-1]
            if event.y lt 0 then return
            event.y = (event.y+1) mod self.n_row
            widget_control,event.id,edit_cell=[event.y,event.x mod self.n_col]
            ;if (!version.os_family ne 'Windows') then widget_control,event.id,set_table_view=[0,0>(event.y-14)]
            end
       'cancel': begin
            self.cancel = (1b)
            widget_control,self.tlb,/destroy
            return
            end     
       'ok':begin
            table = widget_info(self.tlb,find_by_uname='input') 
            widget_control,table,edit_cell=[0,0]  ;to force a content update, effective in Windows system
            widget_control,table,get_value=tmp
            (*self.value) = tmp
            widget_control,self.tlb,/destroy
            end
       'reset':begin
            if strlen(self.reset) ne 0 then $
               (*self.value)[*] = self.reset $
            else $
               (*self.value) = (*self.default)
            input = widget_info(self.tlb,find_by_uname='input')
            widget_control,input,set_value=reform(*self.value,self.n_col,self.n_row)
            end
       'read':begin
            file = dm_choose_file('txt',/read,dialog_parent=self.tlb)
            if strlen(file) eq 0 then break
            openr,unit,file,/get_lun,error=openerr
            if openerr ne 0 then begin
               ok = dialog_message("Can't read "+file+'.',/error)
               break
            endif
            tmp = ''
            sep = string('A4'xb)
            if self.type eq 0 then sep=sep+' ,'+ string(9b)
            while (~ eof(unit)) do begin
                readf,unit,tmp
                tmp1 = strsplit(tmp,sep,/extract)
                if n_elements(in) eq 0 then in=tmp1 else in=[[in],[tmp1]]
            endwhile
            free_lun,unit
            if n_elements(in) ne 0 then begin
               in      = reform(in)
               in_size = size(in,/dimension)
               n_dim   = n_elements(in_size)
               if (n_dim ge 2) and (ptr_valid(self.xvalue)) and (n_elements(in) gt 2) then begin
                  if self.type eq 0 then begin
                     in  = float(in)
                     ind = sort(in[0,*])
                     in  = in[*,ind]
                     for i=0,self.n_col-1 do (*self.value)[i,*] = dm_to_string(interpol(in[i+1,*],in[0,*],(*self.xvalue)))
                  endif else begin
                     in = strtrim(in,2)
                     for i=0,n_elements((*self.value))-1 do begin
                         ind = where(strupcase(in[0,*]) eq strupcase((*self.xvalue)[i]),count)
                         if count ne 0 then (*self.value)[*,i] = in[1:self.n_col,ind[0]]
                     endfor
                  endelse
               endif else begin
                  if n_dim ge 2 then begin
                     tmp = min([self.n_col,in_size[0]])
                     for i=0,(self.n_row-1)<(in_size[1]-1) do $
                         (*self.value)[0:tmp-1,i] = in[0:tmp-1,i]
                  endif else begin
                     for i=0,(self.n_row-1)<(in_size[0]-1) do $
                         (*self.value)[0,i] = in[i]
                  endelse
               endelse
            endif
            input = widget_info(self.tlb,find_by_uname='input')
            widget_control,input,set_value=reform(*self.value,self.n_col,self.n_row)
            end
       'save':begin
            file = dm_choose_file('txt',/write,dialog_parent=self.tlb)
            if strlen(file) eq 0 then break
            openw,unit,file,/get_lun,error=openerr
            if openerr ne 0 then begin
               ok = dialog_message("Can't write in "+file+'.',/error)
               break
            endif
            table = widget_info(self.tlb,find_by_uname='input') 
            widget_control,table,edit_cell=[0,0]  ;to force a content update, effective in Windows system
            widget_control,table,set_table_select=[-1,-1,-1,-1]
            widget_control,table,get_value=tmp
            (*self.value) = tmp
            if ptr_valid(self.xvalue) then begin
               if self.type eq 0 then sep=string(9b) else sep=string('A4'xb)
               for i=0,self.n_row-1 do begin
                   tmp = ''
                   for j=0,self.n_col-1 do tmp = tmp+sep+(*self.value)[j,i]
                   printf,unit,(*self.xvalue)[i]+tmp
               endfor
            endif else begin
               for i=0,self.n_row-1 do begin
                   tmp = ''
                   for j=0,self.n_col-1 do tmp = tmp+(*self.value)[j,i]+string(9b)
                   printf,unit,tmp
               endfor
            endelse
            free_lun,unit
            end
       else:
    endcase
end

;get properties
pro dm_dialog_listinput::getproperty,cancel=cancel,value=value
    if arg_present(cancel) then cancel=self.cancel
    if arg_present(value) then begin
       value = reform(*self.value)
       if self.type eq 0 then value=dm_to_number(value)
       if n_elements(value) eq 1 then value=value[0]
    endif
end

pro dm_dialog_listinput::totop
    widget_control,self.tlb,kbrd_focus_event=0
    widget_control,self.tlb,/show,iconify=0
    wid = widget_info(self.tlb,find_by_uname='ok')
    if wid gt 0 then widget_control,wid,/input_focus
    widget_control,self.tlb,/kbrd_focus_event
end

;destroy the object,life cycle method
pro dm_dialog_listinput::Cleanup
    if widget_info(self.extbase,/valid_id) then widget_control,self.extbase,/destroy
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    ptr_free,self.default,self.value,self.xvalue
end

;constructor,lifecycle method
;syntax:
;   obj=obj_new('dm_dialog_listinput',['item 0','item 1'],default=['red','blue'],label='Color')
;parameter:
;   name:               a string array of names [nrow]
;keyword:
;   cancel:             if arg_presnt, a cancel button will be available
;   cbstring:           if present, use this string instead of "Cancel" for the cancel button
;   default:            default value, or rather starting value [ncol,nrow]
;   dialog_parent:      dialog parent, tlb will be centered on dialog parent
;   info:               informations to be shown at the bottom of tlb
;   label:              column label
;   reset:              reset value, scalor
;   title:              title of tlb
;   xsize:              xsize of the table column width
;   xvalue:             a number or string array assigned to name, when save, this value will also be saved
;                       when read, if present, this value will be used to locate the name(interpolation for numbers)
;                       if not present, read  will be done according to order
function dm_dialog_listinput::Init,name,label=label,cancel=cancel,cbstring=cbstring,default=default,dialog_parent=dialog_parent,info=info,reset=reset,title=title,xsize=xsize,xvalue=xvalue

    self.type = 1     ;default type is string
    if n_elements(name)  eq 0 then name=''
    if n_elements(label) eq 0 then label=['Value'] else label=[label]
    self.n_row = n_elements(name)
    self.n_col = n_elements(label)
    if n_elements(title) eq 0 then title='Input:'
    if n_elements(xsize) eq 0 then xsize=80
    if n_elements(reset) ne 0 then self.reset=dm_to_string(reset[0])
    if n_elements(default) ne 0 then begin
       type = size(default,/type)
       if type ne 7 then self.type=0
       mydefault = dm_to_string(default)
    endif else mydefault=''
    n_default  = size(mydefault,/dimensions)
    if n_elements(n_default) eq 1 then begin
       mydefault = reform(mydefault,1,n_elements(mydefault))
       n_default = [1,n_elements(mydefault)]
    endif
    self.value = ptr_new(strarr(self.n_col,self.n_row))
    if (n_elements(xvalue) eq self.n_row) then begin
       if size(xvalue[0],/type) eq 7 then $
          self.xvalue = ptr_new(strtrim(xvalue,2)) $
       else $
          self.xvalue = ptr_new(dm_to_string(xvalue))
    endif
    for i=0,self.n_row-1 do begin
        for j=0,self.n_col-1 do begin
           (*self.value)[j,i]=mydefault[j mod n_default[0],i mod n_default[1]]
        endfor
    endfor
    self.default = ptr_new(*self.value)
    self.extbase = widget_base(group_leader=dialog_parent,map=0,floating=(n_elements(dialog_parent) ne 0))
    tlb  = widget_base(title=title,/col,group_leader=self.extbase,/modal,/align_center)
    if !version.os_family ne 'Windows' then begin
       mesg = 'Press Return or Enter to insert a value.'
       void = widget_label(tlb,value=mesg,/align_center)
    endif
    row  = widget_base(tlb,/row,/align_center)
    if n_elements(name) eq 1 then name = [name]
    void = widget_table(row,row_labels=name,column_labels=label,uname='input',value=reform(*self.value,self.n_col,self.n_row),y_scroll_size=(15<(self.n_row)),$
           x_scroll_size=self.n_col,column_widths=xsize,/editable);,/all_events)
    n_info = n_elements(info)
    if n_info gt 0 then  begin
       col = widget_base(tlb,/col,/align_center)
       for i=0,n_info-1 do $
           label = widget_label(col,value=info[i],/align_left)
    endif
    row  = widget_base(tlb,/row,/align_center)
    void = widget_button(row,value='Save',uname='save')
    if float(!version.release) ge 5.6 then widget_control,void,tooltip='Save to a text file.'
    void = widget_label(row,value='',xsize=15)
    void = widget_button(row,value='Read',uname='read')
    if float(!version.release) ge 5.6 then widget_control,void,tooltip='Read from a text file.'
    void = widget_label(row,value='',xsize=15)
    void = widget_button(row,value='Reset',uname='reset')
    if (float(!version.release) ge 5.6) and (n_elements(reset) ne 0) then widget_control,void,tooltip='Reset to '+dm_to_string(reset[0])+'.'
    void = widget_label(row,value='',xsize=15)
    if arg_present(cancel) then begin
       if n_elements(cbstring) eq 0 then $
          cbstring = 'Cancel' $
       else if strlen(cbstring) eq 0 then $
          cbstring = 'Cancel'
       void = widget_button(row,value=cbstring,uname='cancel')
       void = widget_label(row,value='',xsize=15)
    endif
    void = widget_button(row,value='OK',uname='ok')
    dm_center_kid,tlb,dialog_parent   ;centering over the dialog_parent widget
    widget_control,tlb,/realize
    self.tlb = tlb
    widget_control,tlb,set_uvalue=self,/kbrd_focus_event
    xmanager,'dm_dialog_listinput',tlb,cleanup='dm_dialog_listinput_Exit'
    return,1
end

;class definition
pro dm_dialog_listinput__define
    void={dm_dialog_listinput    , $    ;thie object class name
        extbase:               0L, $    ;group leader of the modal tlb
        tlb:                   0L, $    ;top level base
        n_row:                 0s, $    ;number of rows
        n_col:                 0s, $    ;number of columns
        type:                  0s, $    ;data type 0:float 1:string
        reset:                 '', $    ;reset value
        default:        ptr_new(), $    ;default values for reset button
        value:          ptr_new(), $    ;save the results
        xvalue:         ptr_new(), $    ;save the identifying xvalue
        cancel:                0b  $    ;flag to show if cancel button has been pressed
    }
end

;example
;   result = dm_dialog_listinput(['item 0','item 1'],default=1,label='Factor',$
;               info=['The factor must be greater than 0','and less than 1'])
;   result = dm_dialog_listinput(['item 0','item 1'],default=['red','blue'],label='Color')
function dm_dialog_listinput,name,_ref_extra=extra
    tmp = obj_new('dm_dialog_listinput',name,_extra=extra)
    if obj_valid(tmp) then begin
       tmp->getproperty,value=value,_extra=extra
       obj_destroy,tmp
    endif
    if n_elements(value) ne 0 then return,value
end