; $Id: dm_copyftp.pro,v 1.9 2017/05/10 14:53:38 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_copyftp
;
; PURPOSE:
;  this program allows users to copy files from NCNR data repository to a local directory 
;
; CATEGORY:
;  dcs_mslice file tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;clear all objects and widget hierarchy
pro dm_copyftp_Exit,tlb
    widget_control,tlb,get_uvalue=state
    obj_destroy,[state.open_filesel,state.save_filesel]
    widget_control,tlb,/destroy
end

pro dm_copyftp_copy,state,ftpobj,list,open_path,save_path,mesgobj=mesgobj,incomplete=incomplete
    n_files = n_elements(list)
    while strmid(open_path,0,1,/reverse_offset) eq state.pathsep[1] do open_path = strmid(open_path,0,strlen(open_path)-1)
    while strmid(save_path,0,1,/reverse_offset) eq state.pathsep[0] do save_path = strmid(save_path,0,strlen(save_path)-1)
    for i=0L,n_files-1 do begin
        if (~ obj_valid(mesgobj)) then return
        if strpos(list[i],state.pathsep[1]) ne -1 then begin
           newdir = save_path+state.pathsep[0]+strmid(list[i],0,strlen(list[i])-1)
           mesgobj->update,message='Creating directory '+newdir,title='Copying...  '+dm_to_string(i+1)+'/'+dm_to_string(n_files)
           file_mkdir,newdir
           state.open_filesel->set_path,open_path+state.pathsep[1]+list[i],/ftp
           state.open_filesel->getproperty,dFiles=dFiles
           dm_copyftp_copy,state,ftpobj,dFiles,open_path+state.pathsep[1]+list[i],newdir,mesgobj=mesgobj,incomplete=incomplete
        endif else begin
           newfile = open_path+state.pathsep[1]+list[i] 
           mesgobj->update,message='Copying '+newfile,title='Copying...  '+dm_to_string(i+1)+'/'+dm_to_string(n_files)      
           ok = ftpobj->GetFileContent(newfile,localfilename=save_path+state.pathsep[0]+list[i])
           ;check if the files are incomplete for .ng0 
           if stregex(list[i],'\.ng0',/boolean,/fold_case) then begin
              info = ''
              dm_load_macs,save_path+state.pathsep[0]+list[i],data,npoints=npoints,/checkonly
              if n_elements(data) eq 0 then ndata = 0 else ndata = n_elements(data[*,0])
              if npoints eq 0 then info = ': empty file' $
              else if npoints ne ndata then info = ': header npoint='+dm_to_string(npoints)+', actual data point='+dm_to_string(ndata) 
              if strlen(info) ne 0 then begin
                 if n_elements(incomplete) eq 0 then incomplete = '      '+list[i]+info else incomplete = [incomplete,'      '+list[i]+info]
              endif
           endif
        endelse
    endfor
end

;event handler
pro dm_copyftp_event,event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=state
    state.open_filesel->getproperty,path=open_path,ftpobj=ftpobj,selected=selected,dFiles=dFiles,sep=sep2
    state.save_filesel->getproperty,path=save_path,sep=sep1
    state.pathsep = [sep1,sep2]
 
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       return
    end
    
    if strmid(strlowcase(tag_names(event,/structure)),0,10) eq 'dm_filesel' then event.id = -1  ;events from file selectors
    
    case (event.id) of
       state.tlb:   widget_control,event.id,scr_xsize=state.geom[0],scr_ysize=state.geom[1]
       state.exitBut:begin
          widget_control,event.handler,/destroy
          return
         end   
       state.copyBut:begin
          mesg = ''
          if n_elements(selected) eq 0 then mesg = [mesg,'Please select ftp files first.']
          if ~obj_valid(ftpobj) then mesg = [mesg,'Please copy files from the NCNR Data Repository.']
          if ~file_test(save_path,/directory,/write) then mesg = [mesg,save_path+' is unwritable.'] 
          if n_elements(mesg) ne 1 then begin
             ok = dialog_message(mesg[1:*],/error,/center,dialog_parent=event.handler)
             break
          endif
          mesgobj = obj_new('dm_progress',title='Copying...',message='Copying '+dFiles[selected[0]],group_leader=state.tlb)
          dm_copyftp_copy,state,ftpobj,dFiles[selected],open_path,save_path,mesgobj=mesgobj,incomplete=incomplete
          state.open_filesel->set_path,open_path,/ftp
          state.save_filesel->set_path,save_path
          state.open_filesel->clear_file,fileloc=dFiles[selected[0]]
          if obj_valid(mesgobj) then obj_destroy,mesgobj
          if n_elements(incomplete) ne 0 then begin
             if n_elements(incomplete) gt 25 then incomplete = [incomplete[0:20],'      ...']
             ss = (['','s'])[n_elements(incomplete) gt 1]
             ok = dialog_message(['The following file'+ss+' might be incomplete:',incomplete,'','Please check the file'+ss+', and try again later if necessary.',$
                  'It may take a couple of days for the ftp server to synchronize files.'],/center,dialog_parent=event.handler)
          endif
         end  
       else: begin  ;file type events and file selector events
          eventname = strlowcase(tag_names(event,/structure))
          case eventname of 
               'widget_button': begin
                                ind0 = where(state.ftypeBut eq event.id,count,complement=ind1)
                                if count eq 1 then begin
                                   state.open_filesel->set_filter,(['.dcs','.ng0,.bt9,.log','.*'])[ind0[0]]
                                   dm_toggle_menubut,check=state.ftypeBut[ind0],uncheck=state.ftypeBut[ind1]
                                   widget_control,state.tlb,base_set_title='Copy '+(['DCS ','MACS ',''])[ind0[0]]+'files from NCNR Data Repository'
                                endif
                                end                 
               'dm_filesel_newfilter': begin
                                if event.object ne state.open_filesel then break
                                if strmatch(event.filter,'.dcs',/fold_case) then dm_copyftp_event,{WIDGET_BUTTON,ID:state.ftypeBut[0],TOP:state.tlb,HANDLER:state.tlb,SELECT:1} $
                                else if strmatch(event.filter,'.ng0,.bt9',/fold_case) then dm_copyftp_event,{WIDGET_BUTTON,ID:state.ftypeBut[1],TOP:state.tlb,HANDLER:state.tlb,SELECT:1} $
                                else if strmatch(event.filter,'.\*',/fold_case) or strmatch(event.filter,'All Files',/fold_case) then dm_copyftp_event,{WIDGET_BUTTON,ID:state.ftypeBut[2],TOP:state.tlb,HANDLER:state.tlb,SELECT:1} $
                                else begin
                                   dm_toggle_menubut,uncheck=state.ftypeBut
                                   widget_control,state.tlb,base_set_title='Copy files from NCNR Data Repository'
                                endelse
                                end
               'dm_filesel_setpath': begin
                                if n_elements(ftpobj) ne 0 then begin
                                   if obj_valid(ftpobj) then break  
                                endif
                                if ~file_test(save_path,/directory,/write) then break
                                state.open_filesel->set_ftpbuffer,save_path
                                state.open_filesel->set_path,/ftp
                                state.save_filesel->set_path,setpathevent=0   ;no more setpathevent
                                widget_control,state.copyBut,/sensitive
                                end
               else:
          endcase
         end
    endcase
end

;main program
pro dm_copyftp,event,workDir=workDir,dataDir=dataDir,filter=filter
    registerName = 'dm_copyftp'
    if xregistered(registerName) then return   ;only allow one copy to be running at one time
    
    state={ group_leader:    0L, $   ;group leader
         tlb:                0L, $   ;top level base
         open_filesel: obj_new(),$   ;open file selector
         save_filesel: obj_new(),$   ;save file selector
         copyBut:            0L, $   ;copy button
         ftypeBut:    [0L,0L,0L],$   ;[DCS,MACS,all]
         exitBut:            0L, $   ;exit button
         pathsep:       ['',''], $   ;path separator 
         geom:    [0L,0L,0L,0L]  $   ;xsize,ysize,xoffset,yoffset
    }
    
    if n_elements(workDir) eq 0 then begin
       workDir = sourcepath()
       defsysv,'!dave_defaults',exists=exists
       if exists then exists = ptr_valid(!dave_defaults)
       if exists then begin
          if (file_test((*!dave_defaults).workDir,/directory,/write))[0] then workDir = (*!dave_defaults).workDir
       endif
    endif
    if ~file_test(workDir[0],/directory,/write) then setpathevent = 1b
    if n_elements(filter) eq 0 then filter = '.*'
    if strmatch(filter,'.ng0',/fold_case) or strmatch(filter,'.ng0,.bt9',/fold_case) then filter='.ng0,.bt9,.log' ;add .log files to MACS file type
    
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    if exists then icon = !DAVE_AUXILIARY_DIR+'ftp.ico' $
    else icon = file_which('ftp.ico',/include_current_dir)
    if dm_to_number(!version.release) ge 6.4 then icon_extra={bitmap:icon}
    
    if n_elements(event) ne 0 then state.group_leader = event.top
    if strmatch(filter,'.dcs',/fold_case) then name = 'DCS ' else if strmatch(filter,'.ng0,.bt9,.log',/fold_case) then name='MACS ' else name=''
    state.tlb = widget_base(title='Copy '+name+'files from NCNR Data Repository',/col,group_leader=state.group_leader,_extra=icon_extra,/tlb_size_event,mbar=mbar,xpad=0,ypad=1)
    
    ;menu bar
    filemenu          = widget_button(mbar,value='File',/menu)
    ftypemenu         = widget_button(filemenu,value='File Type',/menu)
    state.exitBut     = widget_button(filemenu,value='Exit',/separator)
    state.ftypeBut[0] = dm_widget_button(ftypemenu,value='DCS',set_button=strmatch(filter,'.dcs',/fold_case))
    state.ftypeBut[1] = dm_widget_button(ftypemenu,value='MACS',set_button=strmatch(filter,'.ng0,.bt9,.log',/fold_case))
    state.ftypeBut[2] = dm_widget_button(ftypemenu,value='All Files',set_button=strmatch(filter,'.\*',/fold_case))
     
    ;menu bar separator for windows system
    if !version.os_family eq 'Windows' then begin
       mbar_sep = widget_label(state.tlb,sensitive=0,/dynamic_resize,value=' ',scr_ysize=5)
       tmp  = 0
    endif else tmp = 4
    
    row     = widget_base(state.tlb,/row,ypad=0)
    openCol = widget_base(row,/col,ypad=0)
    cmdCol  = widget_base(row,/col,/align_center,ypad=0,xpad=0)
    saveCol = widget_base(row,/col,ypad=0)
    label   = widget_label(openCol,value='NCNR Data Repository')
    label   = widget_label(saveCol,value='local save directory')
    
    if n_elements(dataDir) eq 0 then dataDir = ''
    if ~stregex(dataDir,'^(ftp://ftp\.|https://)ncnr\.nist\.gov',/boolean,/fold_case) then dataDir = 'https://ncnr.nist.gov/pub/ncnrdata'
    state.open_filesel = obj_new('dm_filesel',parent=openCol,ysize=18+tmp,/frame,filter=filter,path=dataDir,/ncnrftp,group_leader=state.tlb)
    state.open_filesel->getproperty,ncnrftp=ncnrftp
    if ~ncnrftp then begin
       ok = dialog_message('Cannot establish connection to NCNR Data Repository.',/center,/error)
       obj_destroy,state.open_filesel
       widget_control,state.tlb,/destroy
       return
    endif
    state.save_filesel = obj_new('dm_filesel',parent=saveCol,ysize=18+tmp,/frame,path=workDir,group_leader=state.tlb,setpathevent=setpathevent)
    if keyword_set(setpathevent) then begin
       state.save_filesel->getproperty,path=save_path
       setpathevent = ~file_test(save_path,/directory,/write)
       if ~setpathevent then begin
          state.open_filesel->set_ftpbuffer,save_path
          state.open_filesel->set_path,/ftp
          state.save_filesel->set_path,setpathevent=0
       endif
    endif
    state.copyBut = widget_button(cmdCol,value='  Copy->  ',sensitive=~keyword_set(setpathevent))
    if float(!version.release) ge 5.6 then $
       widget_control,state.copyBut,tooltip='Choose the local directory for the files to be saved in, select the files or directory from the ftp server, then click this button to copy files. '

    dm_center_kid, state.tlb,state.group_leader,/side   ;centering the top level base
    widget_control,state.tlb,/realize
    geom = widget_info(state.tlb,/geometry)
    state.geom = [geom.scr_xsize,geom.scr_ysize,geom.xoffset,geom.yoffset]
    tlb  = state.tlb
    widget_control,state.tlb,set_uvalue=state,/no_copy
    xmanager,registerName,tlb,cleanup='dm_copyftp_Exit',/no_block
    if keyword_set(setpathevent) then ok = dialog_message('Please choose a writable local save directory first.',/center,dialog_parent=tlb)
end