; $Id: $
;#######################################################################
;
; NAME:
;  dm_choose_volumedata
;
; PURPOSE:
;  choose volume data
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   None
; keywords:
;   (xyzi)dat: data to be returned
;   ierr:      error bar of idat
;   (xyzi)tit: label of the data
;   iran:      for i data only, volume plot color bar range
;   iso3dtype: returns 0 if E is present, 1 if no E in x,y,z data
;   extra_tit: extra title info
;   weight:    weight for avgsum=-1
pro dcs_mslice::dm_choose_volumedata,xdat=xdat,ydat=ydat,zdat=zdat,idat=idat,ierr=ierr,iran=iran,xinfo=xinfo,xtit=xtit,yinfo=yinfo,ytit=ytit,zinfo=zinfo,ztit=ztit,itit=itit,$
    tran_invalid=tran_invalid,debug=debug,iso3dtype=iso3dtype,extra_tit=extra_tit,weight=weight,error=error
    if keyword_set(debug) then begin
       print,'&&choose volume data...'
       current=systime(/sec)
    endif
    
    ;weight
    if (self.bin_avgsum eq -1) or ((self.instrname eq 'macs') and keyword_set(self.macshistomode) and self.macs_rawintn) then self->dm_choose_trajdata,-4,data=weight
 
    ;intensity & error bar
    if arg_present(ierr) then begin
       ok_ierr = 1b
       self->dm_choose_trajdata,-1,data=idat,error=ierr,title=itit,debug=debug 
    endif else begin
       ok_ierr = 0b
       self->dm_choose_trajdata,-1,data=idat,title=itit,debug=debug 
    endelse
    ndat = n_elements(idat)
    ok_wght = (n_elements(weight) eq ndat)
       
    ;choose x, y, and z data
    self->dm_choose_trajdata,self.volsel_x[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=xdat,title=xtit,debug=debug,is_E=is_E_x,is_hkl=is_hkl_x,hkl_info=xinfo
    self->dm_choose_trajdata,self.volsel_y[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=ydat,title=ytit,debug=debug,is_E=is_E_y,is_hkl=is_hkl_y,hkl_info=yinfo
    self->dm_choose_trajdata,self.volsel_z[0>(self.samp_typ)]+((self.samp_typ eq -1)?2:0),data=zdat,title=ztit,debug=debug,is_E=is_E_z,is_hkl=is_hkl_z,hkl_info=zinfo
 
    is_hkl_tot = total([is_hkl_x,is_hkl_y,is_hkl_z])
    iso3dtype  = (is_hkl_tot eq 3)
    if self.samp_typ eq -1 then begin
       n_tot = 0
       for i=0, n_elements((*self.dataPtr).axis_info)-1 do n_tot = n_tot+(total(abs((*self.dataPtr).axis_info[i].vaxis)) ne 0)
    endif else n_tot = 2+((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
    if (is_hkl_tot gt 0) and (is_hkl_tot lt n_tot) then qoffset=0.
     
    extra_tit = ''
    
    nogrid = [0,0,0]
    if self.samp_typ eq -1 then begin
       ok_ewid = 0b

       ;thickness 
       nitem    = n_elements((*self.dataPtr).label)
       if nitem gt 5 then tran_invalid = bytarr(nitem-5)
       for i=0,nitem-6 do begin
           is_hkl_t = 0b
           if (total(finite((*self.volsel_tft)[i,*])) ne 0) then begin
              self->dm_choose_trajdata,(*self.volsel_t)[i]+2,index=index1,range=reform((*self.volsel_tft)[i,*]),empty=empty,debug=debug,unit=unit,label=label,is_hkl=is_hkl_t,hkl_info=hkl_info_t
              if empty then begin
                 if n_elements(emptyitem) eq 0 then emptyitem = label $
                 else emptyitem = emptyitem+' and '+label
                 index1 = -1 & tmp = temporary(index1)
                 tran_invalid[i] = 1b
              endif else if n_elements(index1) eq ndat then begin
                 index1 = -1 & tmp = temporary(index1)
              endif else begin
                 if n_elements(index) eq 0 then index = index1 $
                 else begin
                    index = dm_common(index,index1,/no_copy)
                 endelse
              endelse
              if ~tran_invalid[i] then begin
                 extra_tit = self->range_string(extra_tit,label,(*self.volsel_tft)[i,*],unit)
                 if is_hkl_t and (total(finite((*self.volsel_tft)[i,*],/nan)) eq 0) and (n_elements(qoffset) ne 0) then begin
                    if (abs((*self.volsel_tft)[i,1]-(*self.volsel_tft)[i,0]) le self.qoffsetthreshold) then qoffset = qoffset+hkl_info_t.vaxis*mean((*self.volsel_tft)[i,*])
                 endif
              endif
           endif
       endfor
       if n_elements(index) ne 0 then begin
          if index[0] eq -1 then begin
             emptyitem = 'combined thickness'
             if n_elements(tran_invalid) ne 0 then begin
                tran_invalid[*] = 1b
                extra_tit = ''
             endif
          endif else begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             index = -1
          endelse
       endif
       if is_E_x then begin
          nogrid[0] = (finite(self.vol_xstep,/nan) or (self.vol_xstep eq 0))
          if nogrid[0] then uniq_E_x = xdat[uniq(xdat,sort(xdat))] 
       endif
       if is_E_y then begin
          nogrid[1] = (finite(self.vol_ystep,/nan) or (self.vol_ystep eq 0))
          if nogrid[1] then uniq_E_y = ydat[uniq(ydat,sort(ydat))] 
       endif
       if is_E_z then begin
          nogrid[2] = (finite(self.vol_zstep,/nan) or (self.vol_zstep eq 0))
          if nogrid[2] then uniq_E_z = zdat[uniq(zdat,sort(zdat))] 
       endif
    endif else begin     
       ;ewid
       if self.eint_yn and (self.samp_typ ne 2) then begin
          self->dm_choose_trajdata,-2,data=ewid
          emax = max((*self.projStrPtr).en,min=emin)
          dE   = emax-emin    ;default energy integration width
       endif
       ok_ewid = (n_elements(ewid) eq ndat)
       
       if is_E_x then begin
          nogrid[0] = (finite(self.vol_xstep,/nan) or (self.vol_xstep eq 0))
          if nogrid[0] then uniq_E_x = xdat[*,0,0]
       endif
       if is_E_y then begin
          nogrid[1] = (finite(self.vol_ystep,/nan) or (self.vol_ystep eq 0))
          if nogrid[1] then uniq_E_y = ydat[*,0,0]
       endif
       if is_E_z then begin
          nogrid[2] = (finite(self.vol_zstep,/nan) or (self.vol_zstep eq 0))
          if nogrid[2] then uniq_E_z = zdat[*,0,0]
       endif
       
       tran_invalid = bytarr(3)
       is_E_t = bytarr(3)
       check_fold = bytarr(10) ;u1,u2,u3,E,T,H,...
       is_fold2 = ((self.samp_typ gt 0) and (self.fold_type eq 1))
       is_u3 = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
       check_fold[0] = (is_fold2 and self.view_u1fold and finite((*(self.view_u1foldcenter))[0]))
       check_fold[1] = (is_fold2 and self.view_u2fold and finite((*(self.view_u2foldcenter))[0]))
       check_fold[2] = (is_fold2 and is_u3 and self.view_u3fold and finite((*(self.view_u3foldcenter))[0]))
       fold_center = [(*(self.view_u1foldcenter))[0],(*(self.view_u2foldcenter))[0],(*(self.view_u3foldcenter))[0],fltarr(7)]
       
       ;thickness
       ;t1
       if self.disp_flag[3] and (total(finite(self.volsel_tft1)) ne 0) then begin
          self->dm_check_thickness,self.volsel_t1[self.samp_typ],range=self.volsel_tft1,ndat=ndat,index=index1,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.volsel_t1[self.samp_typ]],foldcenter=fold_center[self.volsel_t1[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[0] = is_E
          tran_invalid[0] = traninvalid
       endif
       ;t2
       if self.disp_flag[4] and (total(finite(self.volsel_tft2)) ne 0) then begin
          self->dm_check_thickness,self.volsel_t2[self.samp_typ],range=self.volsel_tft2,ndat=ndat,index=index2,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.volsel_t2[self.samp_typ]],foldcenter=fold_center[self.volsel_t2[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[1] = is_E
          tran_invalid[1] = traninvalid
       endif
       ;t3
       if self.disp_flag[5] and (total(finite(self.volsel_tft3)) ne 0) then begin
          self->dm_check_thickness,self.volsel_t3[self.samp_typ],range=self.volsel_tft3,ndat=ndat,index=index3,is_E=is_E,traninvalid=traninvalid,emptyitem=emptyitem,$
              extra_tit=extra_tit,checkfold=check_fold[self.volsel_t3[self.samp_typ]],foldcenter=fold_center[self.volsel_t3[self.samp_typ]],qoffset=qoffset,debug=debug
          is_E_t[2] = is_E
          tran_invalid[2] = traninvalid
       endif
    
       if n_elements(index1) ne 0 then index = temporary(index1)
       if n_elements(index2) ne 0 then begin
          if n_elements(index) ne 0 then index = dm_common(index,index2,/no_copy) $
          else index = temporary(index2)
       endif
       if n_elements(index3) ne 0 then begin
          if n_elements(index) ne 0 then index = dm_common(index,index3,/no_copy) $
          else index = temporary(index3)
       endif
    
       if n_elements(index) ne 0 then begin
          if index[0] eq -1 then begin
             emptyitem = 'combined thickness'
             tran_invalid[*] = 1b
             extra_tit = ''
          endif else if n_elements(idat) ne n_elements(index) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
          endif
          index = -1
       endif
       
       ;fold to both sides
       if check_fold[self.volsel_x[self.samp_typ]] then begin
          xdat = [xdat,2.0*fold_center[self.volsel_x[self.samp_typ]]-xdat]
          ydat = [ydat,ydat]
          zdat = [zdat,zdat]
          idat = [idat,idat]
          if ok_ierr then ierr = [ierr,ierr]
          if ok_ewid then ewid = [ewid,ewid]
          if ok_wght then weight = [weight,weight]
       endif
       if check_fold[self.volsel_y[self.samp_typ]] then begin
          xdat = [xdat,xdat]
          ydat = [ydat,2.0*fold_center[self.volsel_y[self.samp_typ]]-ydat]
          zdat = [zdat,zdat]
          idat = [idat,idat]
          if ok_ierr then ierr = [ierr,ierr]
          if ok_ewid then ewid = [ewid,ewid]
          if ok_wght then weight = [weight,weight]
       endif
       if check_fold[self.volsel_z[self.samp_typ]] then begin
          xdat = [xdat,xdat]
          ydat = [ydat,ydat]
          zdat = [zdat,2.0*fold_center[self.volsel_z[self.samp_typ]]-zdat]
          idat = [idat,idat]
          if ok_ierr then ierr = [ierr,ierr]
          if ok_ewid then ewid = [ewid,ewid]
          if ok_wght then weight = [weight,weight]
       endif
   
       ;extra title info
       if strlen(extra_tit) eq 0 then sep='' else sep=', '
       if (self.instrname eq 'dcs') and (total(self.dcsdetmask[1:3]) ne 0) then begin  ;add mask info for DCS in the title extra
          ind = where(1-self.dcsdetmask[1:3],count)
          if count gt 0 then begin
             extra_tit = extra_tit+sep+(['Lower','Central','Upper'])[ind[0]]
             for i=1,count-1 do extra_tit = extra_tit+'+'+(['Lower','Central','Upper'])[ind[i]]
             extra_tit = extra_tit+' Bank'+(['','s'])[count gt 1]
          endif
       endif
       if (self.instrname eq 'macs') and (self.macsintn_typ ne 0) then begin
          extra_tit = extra_tit+sep+'Detector='+(['SPEC','DIFF','SPEC+DIFF'])[self.macsintn_typ]
       endif
       
       ;zero error for dcs, macs, and wand data
       if total(finite(self.zero_error,/nan)) eq 0 then zero_error = self.zero_error[0]*self.zero_error[2]
       bin_zeroerror = self.bin_zeroerror*((self.instrname eq 'dcs') or (self.instrname eq 'macs') or (self.instrname eq 'wand'))
    endelse

    if n_elements(qoffset) eq 3 then begin
       if is_hkl_x then xtit = self->dm_adjust_xyztit(qoffset,title=xtit,hkl_info=xinfo)
       if is_hkl_y then ytit = self->dm_adjust_xyztit(qoffset,title=ytit,hkl_info=yinfo)
       if is_hkl_z then ztit = self->dm_adjust_xyztit(qoffset,title=ztit,hkl_info=zinfo)
    endif

    if n_elements(emptyitem) ne 0 then $
       ok = dialog_message(['The specified '+emptyitem+' range covers no data.','The entire possible range is used instead.'],dialog_parent=self.tlb,/center)

    ;x,y,z range
    n_xran = total(finite(self.vol_xran))
    n_yran = total(finite(self.vol_yran))
    n_zran = total(finite(self.vol_zran))
    if n_xran eq 2 then begin
       index = where((xdat le max(self.vol_xran)) and (xdat ge min(self.vol_xran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          xdat = xdat[index]
          ydat = ydat[index]
          zdat = zdat[index]
          idat = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          xstart = min(self.vol_xran)
       endif
    endif else begin
       if finite(self.vol_xran[0]) then begin
          index = where(xdat ge self.vol_xran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             xstart = self.vol_xran[0]
          endif
       endif else if finite(self.vol_xran[1]) then begin
          index = where(xdat le self.vol_xran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_yran eq 2 then begin
       index  = where((ydat le max(self.vol_yran)) and (ydat ge min(self.vol_yran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          xdat = xdat[index]
          ydat = ydat[index]
          zdat = zdat[index]
          idat = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          ystart = min(self.vol_yran)
       endif
    endif else begin
       if finite(self.vol_yran[0]) then begin
          index = where(ydat ge self.vol_yran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             ystart = self.vol_yran[0]
          endif
       endif else if finite(self.vol_yran[1]) then begin
          index = where(ydat le self.vol_yran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse
    if n_zran eq 2 then begin
       index  = where((zdat le max(self.vol_zran)) and (zdat ge min(self.vol_zran)),count)
       if (count gt 0) and (count ne n_elements(idat)) then begin
          xdat = xdat[index]
          ydat = ydat[index]
          zdat = zdat[index]
          idat = idat[index]
          if ok_ierr then ierr = ierr[index]
          if ok_ewid then ewid = ewid[index]
          if ok_wght then weight = weight[index]
          index = -1
          zstart = min(self.vol_zran)
       endif
    endif else begin
       if finite(self.vol_zran[0]) then begin
          index = where(zdat ge self.vol_zran[0],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
             zstart = self.vol_zran[0]
          endif
       endif else if finite(self.vol_zran[1]) then begin
          index = where(zdat le self.vol_zran[1],count)
          if (count gt 0) and (count ne n_elements(idat)) then begin
             xdat = xdat[index]
             ydat = ydat[index]
             zdat = zdat[index]
             idat = idat[index]
             if ok_ierr then ierr = ierr[index]
             if ok_ewid then ewid = ewid[index]
             if ok_wght then weight = weight[index]
             index = -1
          endif
       endif
    endelse

    ;rebinning according to the steps
    xmax  = max(xdat,min=xmin)
    ymax  = max(ydat,min=ymin)
    zmax  = max(zdat,min=zmin)
    xstep = (xmax-xmin)/200.  ;default in case of xstep absent
    ystep = (ymax-ymin)/200.  ;default in case of ystep absent
    zstep = (zmax-zmin)/200.  ;default in case of zstep absent
    if self.instrname eq 'macs' then begin
       if n_elements(uniq_E_x) ne 0 then begin
          index = where((uniq_E_x ge xmin) and (uniq_E_x le xmax),count)
          if count ne 0 then uniq_E_x = uniq_E_x[index]
          uniq_E_x = round(uniq_E_x*(1000d),/L64)*0.001
          uniq_E_x = uniq_E_x[uniq(uniq_E_x,sort(uniq_E_x))]
       endif
       if n_elements(uniq_E_y) ne 0 then begin
          index = where((uniq_E_y ge ymin) and (uniq_E_y le ymax),count)
          if count ne 0 then uniq_E_y = uniq_E_y[index]
          uniq_E_y = round(uniq_E_y*(1000d),/L64)*0.001
          uniq_E_y = uniq_E_y[uniq(uniq_E_y,sort(uniq_E_y))]
       endif
       if n_elements(uniq_E_z) ne 0 then begin
          index = where((uniq_E_z ge zmin) and (uniq_E_z le zmax),count)
          if count ne 0 then uniq_E_z = uniq_E_z[index]
          uniq_E_z = round(uniq_E_z*(1000d),/L64)*0.001
          uniq_E_z = uniq_E_z[uniq(uniq_E_z,sort(uniq_E_z))]
       endif
    endif
    nuex = n_elements(uniq_E_x)
    nuey = n_elements(uniq_E_y)
    nuez = n_elements(uniq_E_z)
    if nogrid[0] and (nuex ge 2) then xstep = max([xstep,mean(uniq_E_x[1:(nuex-1)]-uniq_E_x[0:(nuex-2)])])
    if nogrid[1] and (nuey ge 2) then ystep = max([ystep,mean(uniq_E_y[1:(nuey-1)]-uniq_E_y[0:(nuey-2)])])
    if nogrid[2] and (nuez ge 2) then zstep = max([zstep,mean(uniq_E_z[1:(nuez-1)]-uniq_E_z[0:(nuez-2)])])
    if finite(self.vol_xstep) and (self.vol_xstep ne 0) then xstep = abs(self.vol_xstep)
    if finite(self.vol_ystep) and (self.vol_ystep ne 0) then ystep = abs(self.vol_ystep)
    if finite(self.vol_zstep) and (self.vol_zstep ne 0) then zstep = abs(self.vol_zstep)
    
    ;figure out dE
    if ok_ewid then begin 
       if is_E_x then begin
          dE = xstep
       endif else if is_E_y then begin
          dE = ystep
       endif else if is_E_z then begin
          dE = zstep
       endif else if self.disp_flag[3] then begin
          tmp_ran = [!values.f_nan,!values.f_nan]
          if self.disp_flag[3] and is_E_t[0] then tmp_ran = self.volsel_tft1
          if self.disp_flag[4] and is_E_t[1] then tmp_ran = self.volsel_tft2
          if self.disp_flag[5] and is_E_t[2] then tmp_ran = self.volsel_tft3
          if finite(tmp_ran[0],/nan) then tmp_ran[0]=-1e30
          if finite(tmp_ran[1],/nan) then tmp_ran[1]=1e30
          tmp_ran = [min(tmp_ran),max(tmp_ran)]
          if tmp_ran[0] lt emin then tmp_ran[0]=emin
          if tmp_ran[1] gt emax then tmp_ran[1]=emax
          if tmp_ran[1] gt tmp_ran[0] then dE=tmp_ran[1]-tmp_ran[0]
       endif
       if dE le 0 then dE=1.0
    endif   
    
    if self.bin_checkstep then error = dm_check_step(xdat,xstep,label=xtit,group_leader=self.tlb,nmax=1e3)
    if keyword_set(error) then return else if self.bin_checkstep then error = dm_check_step(ydat,ystep,label=ytit,group_leader=self.tlb,nmax=1e3) 
    if keyword_set(error) then return else if self.bin_checkstep then error = dm_check_step(zdat,zstep,label=ztit,group_leader=self.tlb,nmax=1e3)
    if keyword_set(error) then return
    
    checkfinite = self.ftypecheckmask[0] or (strlen(self.userf_str) ne 0)
    dm_volumegrid_bin,xstep,ystep,zstep,xdat,ydat,zdat,idat,ierr=ierr,avgsum=self.bin_avgsum,xstart=xstart,ystart=ystart,zstart=zstart,bintr=self.binthreshold,group_leader=self.tlb,$
       conststep=self.bin_conststep,checkfinite=checkfinite,extlib=self.bin_extlib,debug=debug,ewid=ewid,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,$
       zoerrestrange=self.bin_statrange[0]*self.bin_statrange[1],estapplyall=self.bin_statrange[2],foundrange=foundrange

    if n_elements(foundrange) ne 0 then self.bin_statrange[3:4] = [foundrange,2]

    ;smooth regular grid data
    index  = where(finite(idat,/nan),count) ;bins with no data
    refill = (count ne 0)                   ;flag for refilling
    if self.vol_smooth gt 0 then begin      ;smooth data only on plotting
       tmp = size(idat,/dimension)
       if (n_elements(tmp) ge 2) and (min(tmp) gt 3) then begin
          if ok_ierr then ierr = ierr^2
          if n_elements(tmp) eq 3 then begin
             vol_sm_kernel = fltarr(3,3,3)+self.slice_sm_kernel[0,0]
             vol_sm_kernel[*,*,1] = self.slice_sm_kernel
             vol_sm_kernel[1,1,[0,2]] = self.slice_sm_kernel[0,1]
          endif else $
             vol_sm_kernel = self.slice_sm_kernel
          if self.idl_version ge 6.2 then begin
             for i=0, self.vol_smooth-1 do begin
                 idat = convol(idat,vol_sm_kernel,/edge_truncate,/nan,/normalize)
                 if ok_ierr then ierr = convol(ierr,vol_sm_kernel,/edge_truncate,/nan,/normalize)
                 if refill then begin
                    idat[index] = !values.f_nan
                    if ok_ierr then ierr[index] = !values.f_nan 
                 endif
             endfor
          endif else begin
             for i=0, self.vol_smooth-1 do begin
                 ;put 0 in bins with no data
                 if refill then begin
                    idat[index] = 0.0
                    if ok_ierr then ierr[index] = 0.0
                 endif
                 idat = convol(idat,vol_sm_kernel,total(vol_sm_kernel),/edge_truncate)
                 if ok_ierr then ierr = convol(ierr,vol_sm_kernel,total(vol_sm_kernel),/edge_truncate)
             endfor
             if refill then begin
                idat[index] = !values.f_nan
                if ok_ierr then ierr[index] = !values.f_nan
             endif
          endelse
          if ok_ierr then ierr = sqrt(ierr)
       endif
    endif
    index = -1
    ;intensity range
    oldimax = max(idat,min=oldimin,/nan)
    imax = oldimax & imin = oldimin
    if finite(self.vol_iran[0]) then $
       if self.vol_iran[0] lt oldimax then imin=self.vol_iran[0]
    if finite(self.vol_iran[1]) then $
       if self.vol_iran[1] gt oldimin then imax=self.vol_iran[1]
    iran = [imin,imax]
    
    tmp = size(idat,/dimension)
    if n_elements(tmp) ne 3 then begin
       nx = n_elements(xdat) & ny = n_elements(ydat) & nz = n_elements(zdat)
       idat = reform(idat,[nx,ny,nz])
       if ok_ierr then ierr = reform(ierr,[nx,ny,nz])
       if ok_wght then weight = reform(weight,[nx,ny,nz])
    endif
    
    if ok_ewid then begin
       idat = idat*dE
       if ok_ierr then ierr = ierr*dE
    endif
  
    if keyword_set(debug) then print,'&&choose volume data finished in ',systime(/sec)-current,' sec.'
end