; $Id: $
;#######################################################################
;
; NAME:
;  dm_calc_projection
;
; PURPOSE:
;  calculate projection
;  apply masking, rebinning, and detector efficiency adjustment to data
;
; CATEGORY:
;  dcs_mslice
;
; HISTORY:
;  02/2010: added NXSPE support - Andrei Savici (saviciat@ornl.gov)   
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;convert energy (meV) to wavelength (A)
function dm_e2lambda,En,double=double
    if keyword_set(double) then return,sqrt((81.80421d)/En) else return,sqrt(81.8042/En)
end

;convert energy (meV) to velocity (m/s)
function dm_e2velocity,En,double=double
    if keyword_set(double) then return,sqrt(En/5.2270376d-6) else return,sqrt(En/5.2270376e-6)
end

;convert wavelength (A) to energy (meV)
function dm_lambda2e,lambda,double=double
    if keyword_set(double) then return,(81.80421d)/(lambda)^2 else return,81.8042/(lambda)^2
end

function dm_rangestring,values,step=step,unit=unit,tolerance=tolerance
    if n_elements(values) eq 0 then return,''
    if n_elements(step) eq 0   then step=0.0 else step=abs(step) 
    if n_elements(unit) eq 0   then unit=''
    if n_elements(tolerance) eq 0 then tolerance = step/1e4
    tmp =  values[uniq(values,sort(values))]
    n_tmp = n_elements(tmp)
    if step eq 0 then return,dm_to_string(tmp[0])+unit+((n_tmp gt 1)?' to '+dm_to_string(tmp[n_tmp-1])+unit:'')
    outstr = dm_to_string(tmp[0])+unit
    vstart = tmp[0] & cnt = 0L
    for i=1L,n_tmp-1 do begin
        cnt = cnt+1
        if abs(vstart+cnt*step-tmp[i]) gt tolerance then begin
           if abs(tmp[i-1]-vstart) gt tolerance then outstr = outstr+' to '+dm_to_string(tmp[i-1])+unit
           outstr = outstr+', '+dm_to_string(tmp[i])+unit
           vstart = tmp[i] & cnt=0L
        endif   
    endfor
    if abs(tmp[i-1]-vstart) gt tolerance then outstr = outstr+' to '+dm_to_string(tmp[i-1])+unit
    return,outstr
end

;shift histogram data for macs
;parameter:
;  nchan:           shift number of time channels, if not present, use t0_0,t0_1,and t_wid to calculate it
;keywords:
;  t0_0,t0_1,t_wid: t0_0-target t0, t0-1-actual t0, t_wid-tbin width
;  qty:             [ntchan, *, *]
;  dqty:            [ntchan, *, *]
;  histo_weight:    [ntchan, *]
pro dm_shift_histochan,nchan,t0_0=t0_0,t0_1=t0_1,t_wid=t_wid,qty=qty,dqty=dqty,histo_weight=histo_weight
    if (n_elements(nchan) eq 0) and (n_elements(t0_0) ne 0) and (n_elements(t0_1) ne 0) and (n_elements(t_wid) ne 0) then begin
       nchan = round((t0_1-t0_0)/t_wid) 
    endif
    if n_elements(nchan) eq 0 then return
    if size(qty,/n_dim)  gt 1 then qty  = shift(qty,[nchan,intarr(size(qty,/n_dim)-1)])
    if size(dqty,/n_dim) gt 1 then dqty = shift(dqty,[nchan,intarr(size(dqty,/n_dim)-1)])
    if size(histo_weight,/n_dim) gt 1 then histo_weight = shift(histo_weight,[nchan,intarr(size(histo_weight,/n_dim)-1)])
end

;returns start,end,step value
;tolerance is used to determine the step, default is 0.01
;if no step is available then step=!values.f_nan
function dm_startend,value,resolution=resolution,tolerance=tolerance
    n_val = n_elements(value)
    if n_val eq 0 then return,[!values.f_nan,!values.f_nan,!values.f_nan]
    if n_elements(resolution) eq 0 then resolution = 2
    if n_elements(tolerance) eq 0 then tolerance = 2.0/((10d)^resolution)
    val = round(value*((10d)^resolution)+(0.0001d),/L64)/((10d)^resolution)
    if n_val gt 1 then begin
       val = val[uniq(val,sort(val))]
       n_val = n_elements(val)
    endif
    minv = min(val,max=maxv)
    step = !values.f_nan
    avgv = mean(val)
    if max(abs(val-avgv)) gt tolerance then begin
       if n_val ge 2 then begin
          step = (val[n_val-1]-val[0])/(n_val-1.0)
          tmpv = val[0]+findgen(n_val)*step
          if max(abs(tmpv-val)) gt tolerance then step = !values.f_nan $
          else step = dm_to_number(dm_to_string(step,resolution=resolution))
       endif
    endif else step = 0.0
    if step eq 0 then begin
       minv = avgv
       maxv = avgv
    endif
    minv = dm_to_number(dm_to_string(minv,resolution=resolution))
    maxv = dm_to_number(dm_to_string(maxv,resolution=resolution))
    return,[minv,maxv,step]
end

;This function calculates the MACS monitor lambda/2 correction
;parameter:
;     Ei:       [ndat], incident energy
;     monitor:  [ndat], corresponding monitor counts
;     cfx:      [ndat], corresponding CFX status 0-no filter, correction will be applied
;keywords:
;     extra keywords for poly_fit
;return:
;     [ndat], new monitor counts
function dm_macs_lambda2correction,Ei,monitor,cfx,_ref_extra=_extra
    index  = where(cfx eq 0,count)
    if count eq 0 then return,monitor
    ;factor is from Jose's Intensitiesl_l2_Sim_nov.txt file
    factor = [[2,   2.378025665d],[2.5, 1.793602670d],[3,   1.490934555d],[3.5, 1.252089079d],[4,   1.127126095d],[4.5, 0.959301163d],[5,   0.792585170d],[5.5, 0.649712929d],$
              [6,   0.582762512d],[6.5, 0.521160485d],[7,   0.483133752d],[7.5, 0.444736852d],[8,   0.416914118d],[8.5, 0.390288542d],[9,   0.362685682d],[9.5, 0.347918053d],$
              [10,  0.339471838d],[10.5,0.327112776d],[11,  0.325793541d],[11.5,0.305038794d],[12,  0.292795409d],[12.5,0.284034166d],[13,  0.264029518d],[13.5,0.268863169d],$
              [14,  0.259363234d],[14.5,0.247959631d],[15,  0.246067755d],[15.5,0.237768686d],[16,  0.225825977d],[16.5,0.228803554d],[17,  0.229217028d],[17.5,0.235338426d],$
              [18,  0.236590056d],[18.5,0.233790439d],[19,  0.242718897d],[19.5,0.242768485d],[20,  0.242982347d]]
    parm   = poly_fit(factor[0,*],factor[1,*],7,_extra=_extra)
    newmon = monitor
    newmon[index] = newmon[index]/(1.0+0.5*float(poly(Ei[index],parm)))
    return,newmon
end

;construct U1, U2, and U3 from lattice parameters, scattering plane vector U&V, and viewing axes
pro dm_u1u2u3,lattparm,U,V,viewU1,viewU2,viewU3,U1=U1,U2=U2,U3=U3,abc_r=abc_r,error=error
    abc_r = dm_basis_r(lattparm,error=error)        ;direct -> reciprocal basis vectors
    if keyword_set(error) then return
    uvw = dblarr(3,3)
    uvw[*,0] = abc_r#U
    uvw[*,1] = abc_r#V
    uvw[*,0] = uvw[*,0]/norm(uvw[*,0],/double)
    uvw[*,1] = uvw[*,1]-uvw[*,0]#(transpose(uvw[*,1])#uvw[*,0])
    uvw[*,1] = uvw[*,1]/norm(uvw[*,1],/double)
    uvw[*,2] = crossp(uvw[*,0],uvw[*,1])
    abcr = transpose(uvw)#(abc_r)     ;rewrite reciprocal lattice parameters in the (u,v,w) frame
    U1   = abcr#viewU1
    U2   = abcr#viewU2
    itmp = where(abs(U1) lt 1e-15,count)    & if count gt 0 then U1[itmp] = 0d   ;to avoid the floating underflow condition
    itmp = where(abs(U2) lt 1e-15,count)    & if count gt 0 then U2[itmp] = 0d   ;to avoid the floating underflow condition
    if n_elements(viewU3) eq 3 then begin
       U3 = abcr#viewu3
       itmp = where(abs(U3) lt 1e-15,count) & if count gt 0 then U3[itmp] = 0d   ;to avoid the floating underflow condition
    endif else U3 = [0d,0d,0d]
end

pro dm_mismatch,empty_count=empty_count,mesg=mesg,psi=psi,en=en,kid=kid,a3=a3
    empty_count = empty_count+1
    if (empty_count gt 30) or ((n_elements(psi) lt 2) and ((n_elements(en) lt 2) or (n_elements(kid) lt 2))) then return
    if (n_elements(en) gt 1) and (n_elements(kid) gt 1) then begin
       if n_elements(a3) gt 1 then $
          mismatchmesg = 'No match for '+'E='+dm_to_string(en[0])+', kid='+dm_to_string(kid[0])+', A3='+dm_to_string(a3[0])+', the closest empty can data (E='+$
                          dm_to_string(en[1:*],sep=',')+', kid='+dm_to_string(kid[1:*],sep=',')+', A3='+dm_to_string(a3[1:*],sep=',')+') is used.' $
       else $
          mismatchmesg = 'No match for '+'E='+dm_to_string(en[0])+', kid='+dm_to_string(kid[0])+', the closest empty can data (E='+$
                          dm_to_string(en[1:*],sep=',')+', kid='+dm_to_string(kid[1:*],sep=',')+') is used.'
    endif
    if n_elements(psi) gt 1 then mismatchmesg = 'No match for psi='+dm_to_string(psi[0])+', the closest empty can data (psi='+dm_to_string(psi[1:*],sep=',')+') is used.'
    if n_elements(mesg) eq 0 then mesg = mismatchmesg else mesg = [temporary(mesg),mismatchmesg]
end

;check if the common input parameters are all set for single crystal and diffuse scattering type and check orthogonality if necessary, reconstruct and return the viewing axes U1,U2,U3
pro dcs_mslice::dm_check_crystparm,U1=U1,U2=U2,U3=U3,error=error
    extravaxis = (self.extravaxis_yn[0] and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
    mesg = ''
    if total(finite(self.latt_parm,/nan)) ne 0 then mesg = [mesg,'Missing lattice parameters.']
    if (self.instrname eq 'macs') and finite(self.orie_psi,/nan) then orie_psi = 0.0 else orie_psi = self.orie_psi
    if total([finite(self.orie_u,/nan),finite(self.orie_v,/nan),finite(orie_psi,/nan)]) ne 0 then mesg = [mesg,'Missing crystal orientation parameters.'] $ 
    else if norm(crossp(self.orie_u,self.orie_v)) eq 0 then mesg = [mesg,'Parallel cyrstal orientation axes u and v.']
    if (total([finite(self.view_u1,/nan),finite(self.view_u2,/nan)]) ne 0) or ((total(finite(self.view_u3,/nan)) ne 0) and extravaxis) then begin
       mesg = [mesg,'Missing viewing axes parameters.']
    endif else begin
       if norm(crossp(self.view_u1,self.view_u2)) eq 0 then begin
          mesg = [mesg,'Parallel viewing axes u1 and u2.']
       endif
       if extravaxis then begin
          if norm(crossp(self.view_u1,self.view_u3)) eq 0 then mesg = [mesg,'Parallel viewing axes u1 and u3.'] $
          else if norm(crossp(self.view_u2,self.view_u3)) eq 0 then mesg = [mesg,'Parallel viewing axes u2 and u3.'] $
          else if abs(transpose(self.view_u1)#crossp(self.view_u2,self.view_u3)) lt 1e-15 then mesg = [mesg,'Three viewing axes are in the same plane.']
       endif
    endelse
    if n_elements(mesg) gt 1 then begin ;fatal errors
       ok = dialog_message(mesg[1:*],/error,dialog_parent=self.tlb,/center)
       error = 1b
       return
    endif
    dm_u1u2u3,self.latt_parm,self.orie_u,self.orie_v,self.view_u1,self.view_u2,self.view_u3,U1=U1,U2=U2,U3=U3,error=error
    if keyword_set(error) then return  ;error encountered, no further calcuation
    if ~extravaxis then U3 = [0d,0d,0d]
    if (self.instrname eq 'macs') or (self.instrname eq 'wand') and ~self->viewonscattplane() then mesg = [mesg,'The viewing axes are not on the scattering plane.']
    if self.view_ortharbi eq 0 then begin    ;check orthogonality
       if (abs(transpose(U1)#U2) gt 1e-8) or (extravaxis and (max(abs([transpose(U2)#U3,transpose(U3)#U1])) gt 1e-8)) then mesg = [mesg,'The viewing axes are not orthogonal.']
    endif
    if n_elements(mesg) gt 1 then ok = dialog_message(mesg[1:*],dialog_parent=self.tlb,/center)  ;non-fatal error
end

;Masking a3 for MACS
pro dcs_mslice::dm_maska3,a3mask,Ei,Ef,en,a3,kidney,qty=qty,err=err,data_hfield=data_hfield,data_specify=data_specify,data_temperature=data_temperature,weight=weight,histo_weight=histo_weight
    nen = n_elements(en)
    tmp = bytarr(nen)
    tmpstrs = (*a3mask)
    ;check for > and >= expression
    repeat begin
       tmp1 = stregex(tmpstrs,'>(=*)[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
       if len[0] gt 0 then begin
          tmp2 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
          if len[1] gt 0 then ind = where(a3 ge tmp2,count) $  ;'>='
          else ind = where(a3 gt tmp2,count)                   ;'>'
          if count gt 0 then tmp[ind] = 1b
          tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
       endif
    endrep until len[0] le 0
    ;check for < and <= expression
    repeat begin
       tmp1 = stregex(tmpstrs,'<(=*)[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
       if len[0] gt 0 then begin
          tmp2 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
          if len[1] gt 0 then ind = where(a3 le tmp2,count) $  ;'<='
          else ind = where(a3 lt tmp2,count)                   ;'<'
          if count gt 0 then tmp[ind] = 1b
          tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
       endif
    endrep until len[0] le 0
    ;check for : expression
    repeat begin
       tmp1 = stregex(tmpstrs,'(-?[0-9.]+)[ '+string(9b)+']*:[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
       if len[0] gt 0 then begin
          tmp2 = dm_to_number(strmid(tmpstrs,tmp1[1],len[1]))
          tmp3 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
          ind = where((a3 ge tmp2) and (a3 le tmp3),count)
          if count gt 0 then tmp[ind] = 1b
          tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
       endif
    endrep until len[0] le 0
    ;check for xN+y expression, where N is an integer
    repeat begin
       tmp1 = stregex(tmpstrs,'(\-* *[0-9.]+) *n *([\+\-]* *[0-9.]*)',/subexpr,length=len,/fold_case)
       if len[0] gt 0 then begin
          tmp2 = dm_to_number(strmid(tmpstrs,tmp1[1],len[1]))
          if tmp2 ne 0 then begin
             if len[2] gt 0 then tmp3 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2])) else tmp3 = 0
             tmp4 = (a3-tmp3)/tmp2
             ind  = where(abs(tmp2*(tmp4-round(tmp4))) le 0.001,count)
             if count gt 0 then tmp[ind] = 1b
          endif
          tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
       endif
    endrep until len[0] le 0
    ;check for single values
    tmp1 = strsplit(tmpstrs,'[a-zA-Z '+string(9b)+',:<>]',/regex,/extract,count=count)
    for j=0,count-1 do begin
        ind = where(abs(a3-dm_to_number(tmp1[j])) lt 0.0001,count)
        if count gt 0 then tmp[ind] = 1b
    endfor
    ind = where(tmp eq 0,count,complement=complement,ncomplement=ncomplement) & tmp=0
    if (count ne 0) and (count ne nen) then begin
       Ei = Ei[ind]
       Ef = Ef[ind]
       en = en[ind]
       a3 = a3[ind]
       kidney = kidney[ind]
       if n_elements(qty)              gt nen then qty = qty[*,*,ind]
       if n_elements(err)              gt nen then err = err[*,*,ind]
       if n_elements(data_temperature) gt nen then data_temperature = data_temperature[ind,*]
       if n_elements(data_hfield)      eq nen then data_hfield  = data_hfield[ind]
       if n_elements(data_specify)     eq nen then data_specify = data_specify[ind]
       if n_elements(weight)           eq nen then weight       = weight[ind]
       if n_elements(histo_weight)     gt nen then histo_weight = histo_weight[*,ind]
       nen = count
    endif else if count eq 0 then begin
       is_bg = (a3mask eq self.bga3mask)
       mesg = 'All'+(['',' background'])[is_bg]+' A3 angles were masked. The mask has been cleared.'
       if is_bg and (total(ptr_valid(self.bgdata)) gt 1) then mesg = mesg+' You might need to recalculate the projection.'
       ok = dialog_message(mesg,dialog_parent=self.tlb,/center)
       if is_bg then self->clear_mask,/bga3 else self->clear_mask,/a3
    endif
end

;empty can subtraction,use weighted mean
;keyword:
;    raw: if set, intensity is unnormalized, for MACS event mode data only
;    weight0,histo_weight0: weight info for qty, only used when raw=1
pro dcs_mslice::dm_subbg,qty,err,nen=nen,n_file=n_file,angle=angle,ssfac=ssfac,en=en,Ei=Ei,Ef=Ef,kidney=kidney,monitor=monitor,mesg=mesg,tchan_count=tchan_count,tchan_ind=tchan_ind,$
    empty_count=empty_count,bgset=bgset,debug=debug,raw=raw,weight0=weight0,histo_weight0=histo_weight0
    if n_elements(err) eq 0 then err = fltarr(1>(size(qty,/dimension)))
    if n_elements(angle) eq 0 then angle = (*self.dataStrPtr).rotation
    n_bgset = total(ptr_valid(self.bgdata))
    if (total(finite(self.empcanfactor[0:(n_bgset-1)])) eq n_bgset) and (n_bgset gt 1) then factor=self.empcanfactor else factor=make_array(n_elements(self.empcanfactor),value=1.0)
    if size(ssfac,/n_dim) eq 2 then ssfac_1d=ssfac[0,*] else ssfac_1d=ssfac
    if n_elements(empty_count) eq 0 then empty_count = 0L  ;limit the number of empty can mismatch error messages
    if n_elements(bgset) eq 0 then bgset = 0 else bgset = bgset[0]
    bgptr = self.bgdata[bgset]
    bgqty = (*bgptr).qty*factor[bgset]
    bgerr = (*bgptr).err*abs(factor[bgset])
    bgang = (*bgptr).rotation
    if self.instrname eq 'macs' then begin
       bgei   = (*bgptr).Ei
       bgef   = (*bgptr).Ef
       bgen   = (*bgptr).Ei-(*bgptr).Ef
       bgkid  = (*bgptr).kidney
       is_raw = keyword_set(raw) and (n_elements(weight0) ne 0) and (n_elements(histo_weight0) ne 0) 
       if self.macs_lamb2 then weight = dm_macs_lambda2correction(bgEi,(*bgptr).weight[*,self.mon_typ[1]],(*bgptr).cfx) $
       else weight = (*bgptr).weight[*,self.mon_typ[1]]
       histo_weight = (*bgptr).histo_weight 
       if ~is_raw then begin
          tmpind = where(weight eq 0,cnt)
          if cnt ne 0 then weight[tmpind] = 1.0
          if n_elements(monitor) eq 0 then monitor = (self.macs_mon[self.mon_sumfit[1],self.mon_typ[1]])
          monfac = monitor/weight
          if cnt ne 0 then weight[tmpind] = 0.0
          for i=0LL,n_elements(bgen)-1L do begin
              bgqty[*,*,i] = bgqty[*,*,i]*monfac[i]
              bgerr[*,*,i] = bgerr[*,*,i]*monfac[i]
          endfor
       endif
       if (self.macshistomode eq 2) and (self.macshistot0[0] ne self.macshistot0[1]) then begin ;need to shift empty can time channel
          bgqty = transpose(bgqty,[1,0,2])
          bgerr = transpose(bgerr,[1,0,2])
          dm_shift_histochan,t0_0=self.macshistot0[0],t0_1=self.macshistot0[1],t_wid=self.macshistowidth,qty=bgqty,dqty=bgerr
          bgqty = transpose(bgqty,[1,0,2])
          bgerr = transpose(bgerr,[1,0,2])
       endif
       if ptr_valid(self.bga3mask) then self->dm_maska3,self.bga3mask,bgei,bgef,bgen,bgang,bgkid,qty=bgqty,err=bgerr,weight=weight,histo_weight=histo_weight
       tmp_tol = abs(self.macs_tolerance[*,0])
       tmpind  = where(tmp_tol eq 0,cnt)
       if cnt ne 0 then tmp_tol[tmpind] = 1e-6
       dims = size(bgqty,/dim)
       if n_elements(dims) eq 2 then dims = [dims,1]
       if size(weight,/n_dim) eq 0 then weight = [weight]        ;make sure it is an array
       if total(abs(size(histo_weight,/dim)-dims[1:2])) eq 0 then begin
          weight = histo_weight*transpose(rebin(weight,dims[2],dims[1]))
          is_histo = 1b
          if is_raw then begin
             dims1 = size(qty,/dim)
             if n_elements(dims1) eq 2 then dims1 = [dims1,1]
             if size(weight0,/n_dim) eq 0 then weight1 = [weight0] else weight1 = weight0
             weight1 = histo_weight0*transpose(rebin(weight1,dims1[2],dims1[1]))
          endif
       endif else is_histo = 0b
       for i=0L,nen-1 do begin
           mismatch = 0b
           if self.empt_tol_optn eq 0 then begin
              if self.samp_typ eq 0 then $
                 index = where((abs(bgei-Ei[i]) le tmp_tol[0]) and (abs(bgef-Ef[i]) le tmp_tol[0]) and (abs(bgkid-kidney[i]) le tmp_tol[1]),count) $
              else $
                 index = where((abs(bgei-Ei[i]) le tmp_tol[0]) and (abs(bgef-Ef[i]) le tmp_tol[0]) and (abs(bgkid-kidney[i]) le tmp_tol[1]) and (abs(bgang-angle[i]) le tmp_tol[2]),count)
              if count eq 0 then begin
                 if self.samp_typ eq 0 then begin
                    tmp1 = min(abs(bgei-Ei[i])/tmp_tol[0]+abs(bgef-Ef[i])/tmp_tol[0]+abs(bgkid-kidney[i])/tmp_tol[1],imin)
                    index = where((abs(bgei-Ei[i]) eq abs(bgei[imin]-Ei[i])) and (abs(bgef-Ef[i]) eq abs(bgef[imin]-Ef[i])) and (abs(bgkid-kidney[i]) eq abs(bgkid[imin]-kidney[i])),count)
                 endif else begin
                    tmp1 = min(abs(bgei-Ei[i])/tmp_tol[0]+abs(bgef-Ef[i])/tmp_tol[0]+abs(bgkid-kidney[i])/tmp_tol[1]+abs(bgang-angle[i])/tmp_tol[2],imin)
                    index = where((abs(bgei-Ei[i]) eq abs(bgei[imin]-Ei[i])) and (abs(bgef-Ef[i]) eq abs(bgef[imin]-Ef[i])) and (abs(bgkid-kidney[i]) eq abs(bgkid[imin]-kidney[i]) and (abs(bgang-angle[i]) eq abs(bgang[imin]-angle[i]))),count)
                 endelse
                 mismatch = 1b
              endif
           endif else begin
              if self.samp_typ eq 0 then begin
                 tmp1 = min(abs(bgei-Ei[i])/tmp_tol[0]+abs(bgef-Ef[i])/tmp_tol[0]+abs(bgkid-kidney[i])/tmp_tol[1],imin)
                 index = where((abs(bgei-bgei[imin]) le tmp_tol[0]) and (abs(bgef-bgef[imin]) le tmp_tol[0]) and (abs(bgkid-bgkid[imin]) le tmp_tol[1]),count)
              endif else begin
                 tmp1 = min(abs(bgei-Ei[i])/tmp_tol[0]+abs(bgef-Ef[i])/tmp_tol[0]+abs(bgkid-kidney[i])/tmp_tol[1]+abs(bgang-angle[i])/tmp_tol[2],imin)
                 index = where((abs(bgei-bgei[imin]) le tmp_tol[0]) and (abs(bgef-bgef[imin]) le tmp_tol[0]) and (abs(bgkid-bgkid[imin]) le tmp_tol[1]) and (abs(bgang-bgang[imin]) le tmp_tol[2]),count)
              endelse
           endelse
           tmp_err = (bgerr[*,*,index])^2
           if is_raw then begin
              tmp_wgt = reform(transpose(rebin(weight[*,index],dims[1],count,dims[0]),[2,0,1]))
              tmp_qty = reform(bgqty[*,*,index])
              tmp_err = reform(tmp_err)
              if count gt 1 then begin
                 tmp_wgt = total(tmp_wgt,3)
                 tmp_qty = total(tmp_qty,3)
                 tmp_err = total(tmp_err,3)
              endif
              tmp_ind = where(tmp_wgt eq 0,tmp_cnt)
              if tmp_cnt ne 0 then tmp_wgt[tmp_ind] = 1.0  ;make sure not divided by 0.0
              tmp_wgt = transpose(rebin(weight1[*,i],dims1[1],dims1[0]))/tmp_wgt
              qty[*,*,i] = qty[*,*,i]-ssfac*tmp_wgt*tmp_qty
              err[*,*,i] = sqrt((err[*,*,i])^2+(ssfac^2)*(tmp_wgt^2)*tmp_err)
           endif else begin
              if count gt 1 then begin
                 if is_histo then tmp_wgt = transpose(rebin(weight[*,index],dims[1],count,dims[0]),[2,0,1]) else tmp_wgt = transpose(rebin(weight[index],count,dims[1],dims[0]))
                 tmp_qty = total(bgqty[*,*,index]*tmp_wgt,3,nan=self.ftypecheckmask[0])
                 tmp_err = total(tmp_err*tmp_wgt^2,3,nan=self.ftypecheckmask[0])
                 if is_histo then begin
                    tmp_wgt = total(tmp_wgt,3)
                    tmp_ind = where(tmp_wgt eq 0,tmp_cnt)
                    if tmp_cnt ne 0 then tmp_wgt[tmp_ind] = 1.0  ;make sure not divided by 0.0
                 endif else tmp_wgt = total(weight[index])
                 if self.ftypecheckmask[0] then begin            ;make sure sum of all NAN still returns NAN, not 0.0
                    tmp_ind = where(total(finite(bgqty[*,*,index]),3) eq 0,tmp_cnt)
                    if tmp_cnt ne 0 then begin
                       tmp_qty[tmp_ind] = !values.f_nan
                       tmp_err[tmp_ind] = !values.f_nan
                    endif
                 endif
              endif else begin
                 tmp_qty = reform(bgqty[*,*,index])
                 tmp_wgt = 1.0
              endelse      
              qty[*,*,i] = qty[*,*,i]-ssfac/tmp_wgt*tmp_qty
              err[*,*,i] = sqrt((err[*,*,i])^2+(ssfac^2)/(tmp_wgt^2)*tmp_err)
           endelse
           tmp_err = 0 & tmp_wgt = 0
           if mismatch then begin ;mismatch warning message
              if self.samp_typ eq 0 then dm_mismatch,empty_count=empty_count,mesg=mesg,en=[en[i],bgen[index]],kid=[kidney[i],bgkid[index]] $
              else dm_mismatch,empty_count=empty_count,mesg=mesg,en=[en[i],bgen[index]],kid=[kidney[i],bgkid[index]],a3=[angle[i],bgang[index]]
           endif
           ;if keyword_set(debug) then print,'E='+dm_to_string(en[i])+','+dm_to_string(bgen[index])+' kid='+dm_to_string(kidney[i])+','+dm_to_string(bgkid[index])+' A3='+dm_to_string(angle[i])+','+dm_to_string(bgang[index])
       endfor
    endif else begin
       weight = (*bgPtr).weight
       dims   = size(bgqty,/dim)
       if self.samp_typ ge 2 then begin    ;diffuse scattering type
          if n_elements(bgang) eq 1 then begin
             if self.samp_typ eq 2 then begin
                for i=0L,n_file-1L do begin
                    qty[*,i] = qty[*,i]-bgqty[*,0]*ssfac
                    err[*,i] = sqrt((err[*,i])^2+(bgerr[*,0])^2*(ssfac^2))
                endfor
             endif else begin
                for i=0L,n_file-1L do begin
                    qty[*,*,i] = qty[*,*,i]-bgqty[*,*,0]*ssfac
                    err[*,*,i] = sqrt((err[*,*,i])^2+(bgerr[*,*,0])^2*(ssfac^2))
                    if tchan_count ne 0 then begin   ;avoid double-subtraction
                       qty[tchan_ind,*,i] = qty[tchan_ind,*,i]+bgqty[tchan_ind,*,0]*ssfac_1d
                       ;error bar is not changed
                    endif
                endfor
             endelse
          endif else begin
             for i=0L,n_file-1L do begin
                 mismatch = 0b
                 tmp1 = min(abs(bgang-angle[i]),imin) & tmp1 = bgang[imin]
                 if self.empt_tol_optn eq 0 then begin
                    index = where(abs(bgang-angle[i]) le self.psi_tolerance[0],count)
                    if count eq 0 then begin
                       index = where(bgang eq tmp1,count)
                       mismatch = 1b
                    endif
                 endif else index = where(abs(bgang-tmp1) le self.psi_tolerance[0],count)
                 if self.samp_typ eq 2 then begin
                    tmp_err = (bgerr[*,index])^2
                    if count gt 1 then begin
                       tmp_wg0 = total(weight[index])
                       tmp_wgt = transpose(rebin(weight[index],count,dims[0]))
                       tmp_qty = total(bgqty[*,index]*tmp_wgt,2,nan=self.ftypecheckmask[0])
                       tmp_err = total(tmp_err*tmp_wgt^2,2,nan=self.ftypecheckmask[0])
                       if self.ftypecheckmask[0] then begin  ;make sure sum of all NAN still returns NAN, not 0.0
                          tmp_ind = where(total(finite(bgqty[*,index]),2) eq 0,tmp_cnt)
                          if tmp_cnt ne 0 then begin
                             tmp_qty[tmp_ind] = !values.f_nan
                             tmp_err[tmp_ind] = !values.f_nan
                          endif
                       endif
                    endif else begin
                       tmp_qty = bgqty[*,index]
                       tmp_wg0 = 1.0
                    endelse
                    qty[*,i] = qty[*,i]-ssfac/tmp_wg0*tmp_qty
                    err[*,i] = sqrt((err[*,i])^2+(ssfac^2)/(tmp_wg0^2)*tmp_err)
                 endif else begin
                    tmp_err = (bgerr[*,*,index])^2
                    if count gt 1 then begin
                       tmp_wg0 = total(weight[index])
                       tmp_wgt = transpose(rebin(weight[index],count,dims[1],dims[0]))
                       tmp_qty = total(bgqty[*,*,index]*tmp_wgt,3,nan=self.ftypecheckmask[0])
                       tmp_err = total(tmp_err*tmp_wgt^2,3,nan=self.ftypecheckmask[0])
                       if self.ftypecheckmask[0] then begin  ;make sure sum of all NAN still returns NAN, not 0.0
                          tmp_ind = where(total(finite(bgqty[*,*,index]),3) eq 0,tmp_cnt)
                          if tmp_cnt ne 0 then begin
                             tmp_qty[tmp_ind] = !values.f_nan
                             tmp_err[tmp_ind] = !values.f_nan
                          endif
                       endif
                    endif else begin
                       tmp_qty = bgqty[*,*,index]
                       tmp_wg0 = 1.0
                    endelse
                    qty[*,*,i] = qty[*,*,i]-ssfac/tmp_wg0*tmp_qty
                    err[*,*,i] = sqrt((err[*,*,i])^2+(ssfac^2)/(tmp_wg0^2)*tmp_err)
                    if tchan_count ne 0 then begin   ;avoid double-subtraction
                       qty[tchan_ind,*,i] = qty[tchan_ind,*,i]+ssfac_1d/tmp_wg0*tmp_qty
                       ;error bar is not changed
                    endif
                 endelse
                 tmp_err = 0 & tmp_wgt = 0
                 if mismatch then dm_mismatch,empty_count=empty_count,mesg=mesg,psi=[angle[i],bgang[index]] ;mismatch warning message
             endfor
          endelse
       endif else begin
          for i=0L,n_file-1L do begin
              qty[*,*,i] = qty[*,*,i]-(bgqty)*ssfac
              err[*,*,i] = sqrt((err[*,*,i])^2+(bgerr)^2*(ssfac^2))
          endfor
          if tchan_count ne 0 then begin
             for i=0L,n_file-1L do qty[tchan_ind,*,i] = qty[tchan_ind,*,i]+(bgqty[tchan_ind,*])*ssfac_1d
             ;error bar is not changed
          endif
       endelse
    endelse
    bgqty = 0 & bgerr = 0
end

;keywords:
;     temperature: if present, no prompt for temperautre, useful in scripts
;     dwu2:        if present, no prompt for Debye-Waller factro correction u^2, useful in scripts
pro dcs_mslice::dm_calc_projection,temperature=temperature,dwu2=dwu2,debug=debug
    self.error  = 1b    ;initialize the error flag, it will be cleared at the end of the program
    elastic_tol = 0.005 ;tolerance value for elastic energy
    if keyword_set(debug) then begin
       print,'**Calculate projection...'
       current = systime(/sec)
    end

    case self.instrname of
      'dcs':begin     ;dcs
            if (~ ptr_valid(self.detPosPtr)) then self.detPosPtr=dm_load_detpos(/dcs,adjust2th=self.tthcorr)
            end
      'macs':begin    ;macs
            if (~ ptr_valid(self.detPosPtr)) then self.detPosPtr=dm_load_detpos(/macs)
            elastic_tol = abs(self.macs_tolerance[0,0])>(elastic_tol)<0.05  ;tolerance value for elastic energy can be changed from the empty can subtraction tolerance value
            end
      'spe':begin     ;spe
            if (~ ptr_valid(self.detPosPtr)) then begin
               self->dm_load_phx,phxfile=phxfile
               if strlen(phxfile) ne 0 then self.phxfile=phxfile
            endif
            end
      'nxspe':begin   ;nxspe
            if (~ ptr_valid(self.detPosPtr)) then begin
               self->dm_load_phx,phxfile=phxfile
            endif
            end
      'inx':begin     ;inx
            if (~ ptr_valid(self.detPosPtr)) then begin
               self.detPosPtr = dm_load_detpos(/neat)
               ok = dialog_message(['Detector information is not available, using NEAT instrument info instead.','',$
                    'Or you can load an existing INX file and clear data to save the detector information.'],title='Please note:',dialog_parent=self.tlb,/center)
            endif
            end
      'wand':begin    ;wand
            if (~ ptr_valid(self.detPosPtr)) then self.detPosPtr=dm_load_detpos(/wand,adjust2th=self.tthcorr)
            end
      else:
    endcase

    if (~ ptr_valid(self.detPosPtr)) then begin
       info = ['Detector information is not available. The projection calculation is aborted.']
       if ((self.instrname eq 'nxspe') or (self.instrname eq 'spe')) and (~ptr_valid(self.dataStrPtr)) then $
       info = [info,'','You can load an existing '+self.instrname+' file and clear data to save the detector information.']
       ok   = dialog_message(info,/error,dialog_parent=self.tlb,/center)
       return
    endif

    if (self.instrname eq 'macs') and finite(self.orie_psi,/nan) then orie_psi = 0.0 else orie_psi = self.orie_psi
    
    extravaxis = (self.extravaxis_yn[0] and (self.instrname ne 'macs') and (self.instrname ne 'wand'))

    ;check if the common input parameters are all set for single crystal and diffuse scattering type
    if self.samp_typ ne 0 then begin
       self->dm_check_crystparm,U1=U1,U2=U2,U3=U3,error=error
       if keyword_set(error) then return  ;error encountered, no further calculation
    endif
    
    if (strlen(self.userf_str) ne 0) then begin ;remove comment lines, just in case
       userf_str = dm_strip_comments(strsplit(self.userf_str,'&',/extract),/combine)
    endif else userf_str = ''

    data_temperature = 0.0  ;data temperature, for MACS, DCS, and WAND
    data_hfield      = 0.0  ;magnetic field, for MACS and DCS
    data_specify     = 0.0  ;specified data, for MACS
    weight           = 1.0
    histo_weight     = 1.0  ;MACS event mode only
    if (~ ptr_valid(self.dataStrPtr)) then begin    ;no data loaded yet
       if self.instrname eq 'macs' then begin
          eief = 5.0
          if (finite(self.eief)) then $
             if self.eief gt 0 then eief=dm_to_number(dm_to_string(self.eief,resolution=3))
          fts = ['from','to','step']
          name = ['fixed '+(['Ei','Ef'])[self.instrgeom]+' (meV)','E '+fts+' (meV)','Kidney '+fts,'Optimize kidney step']
          default = [dm_to_string([eief,self.estartend]),self.kidney[0:2],(['0','1'])[strmatch(self.kidney[3],'no',/fold_case)]]    
          info = ['Leave kidney range blank for full allowable kidney range.','The equivalent optimized kidney step might be slightly','different from the target step.']
          if orie_psi ne 0 then info = ['A3 offset angle of '+dm_to_string(orie_psi)+' will not be used in this calculation.',info]
          if ~lmgr(/vm) then info = [info,'Use UL and LL for the upper and lower kidney limit,','and CN for the mean kidney center position at each energy.','For example: ul-5, ll+5, cn-4.']
          info = [info,'Kidney offset = '+dm_to_string(self.kidney_offset,resolution=3)+string('b0'XB)]
          if self.samp_typ eq 1 then begin
             name = [name,'A3 '+fts]
             default = [default,dm_to_string(self.psi[0:2])]
             if self.datr_yn then begin
                name    = [name,'Dark Angle','Dark Angle Width'] 
                default = [default,dm_to_string(self.datr_psi),dm_to_string(self.datr_width)]
                info    = [info,'Dark angle position is relative to the crystal axis u.','Counterclockwise rotation is positive.']
             endif
          endif
          ok = dm_getkidneyrange(5,eimat=eimat)
          info = [info,'Ei'+(['','=Ef+E'])[self.instrgeom]+' should be in the range of ['+dm_to_string(min(eimat))+', '+dm_to_string(max(eimat))+'] meV.']
          notok = 1b
          is_droplist = bytarr(n_elements(name)) & is_droplist[7] = 1
          while notok do begin
               if n_elements(anstr) eq 0 then default=default else default=anstr
               anstr = dm_dialog_input(name+':',default=default,dialog_parent=self.tlb,/align_center,info=info,cancel=cancel,droplist_content=ptr_new(['Yes','No']),is_droplist=is_droplist)
               if keyword_set(cancel) then begin
                  self.error = 0
                  return
               endif
               ans = dm_to_number(anstr,/float)
               ind_nan = where(finite(ans,/nan),cnt_nan)
               if cnt_nan eq total(finite(ans[[4,5,7]],/nan)) then begin
                  if ans[0] gt 0 then notok = 0b else tmp_info = 'Negative '+name[0]+'.'
                  if (total(finite(ans[4:5],/nan)) ne 0) and (ans[6] eq 0) then begin
                     notok = (~strmatch(anstr[4],anstr[5],/fold_case))
                     if notok then tmp_info = 'Incorrect kidney step.'
                  endif
               endif else begin
                  tmp_ind  = where((ind_nan ne 4) and (ind_nan ne 5) and (ind_nan ne 7))
                  tmp_info = 'Missing '+name[ind_nan[tmp_ind]]+'.'
               endelse
               if notok and n_elements(tmp_info) ne 0 then ok = dialog_message(tmp_info,/error,dialog_parent=self.tlb,/center)
          endwhile
          kid_fnt = finite(ans[4:5])
          if (~lmgr(/vm)) and stregex(anstr[4],'ul|ll|cn',/boolean,/fold_case) then kid_fnt[0] = 0  ;need to replace ul, ll, and cn
          if (~lmgr(/vm)) and stregex(anstr[5],'ul|ll|cn',/boolean,/fold_case) then kid_fnt[1] = 0
          ans[[3,6]] = abs(ans[[3,6]])
          if ans[2] lt ans[1] then ans[1:2] = ans[[2,1]]
          if ans[3] eq 0 then ans[2] = ans[1] $
          else if ans[1] eq ans[2] then ans[3] = 0
          if strmatch(anstr[4],anstr[5],/fold_case) and (strlen(anstr[4]) ne 0) then anstr[6] = '0'
          if kid_fnt[0] and kid_fnt[1] then begin
             if ans[5] lt ans[4] then anstr[4:5] = anstr[[5,4]]
             if ans[6] eq 0 then anstr[5] = anstr[4] $
             else if ans[4] eq ans[5] then anstr[6] = '0'
          endif
          eief           = ans[0]
          self.eief      = ans[0] 
          self.estartend = ans[1:3]
          self.kidney    = anstr[4:7]
          nen            = 1>((self.estartend[2] eq 0)?1:round(1+(self.estartend[1]-self.estartend[0])/self.estartend[2]))
          if self.samp_typ eq 1 then begin
             ans[10] = abs(ans[10])
             if ans[9] lt ans[8] then ans[8:9] = ans[[9,8]]
             if ans[10] eq 0 then ans[9] = ans[8] $
             else if ans[8] eq ans[9] then ans[10] = 0
             self.psi[0:2] = ans[8:10]
             na3 = 1>((self.psi[2] eq 0)?1:round(1+(self.psi[1]-self.psi[0])/self.psi[2]))  
             if self.datr_yn then begin
                self.datr_psi   = ans[11]
                self.datr_width = abs(ans[12])
             endif
          endif else na3 = 1
          for i=0L,nen-1L do begin
              tmp_en = self.estartend[0]+self.estartend[2]*i
              if self.instrgeom eq 0 then tmp_ei = eief $ ;direct geometry
              else tmp_ei = eief+tmp_en                   ;inverse geometry
              kidran = dm_to_number(self.kidney[0:1]) 
              kidlim = dm_getkidneyrange(tmp_ei,offset=self.kidney_offset,/notcheckeimax,/notcheckeimin)
              if finite(self.kidney_backlash) then kidlim[0] = kidlim[0]+self.kidney_backlash
              if ~kid_fnt[0] then begin
                 if (~lmgr(/vm)) and (strlen(anstr[4]) ne 0) then kidran[0] = dm_calckidney(anstr[4],kidlim) else kidran[0] = kidlim[0]
              endif
              if ~kid_fnt[1] then begin
                 if (~lmgr(/vm)) and (strlen(anstr[5]) ne 0) then kidran[1] = dm_calckidney(anstr[5],kidlim) else kidran[1] = kidlim[1]
              endif
              kstep = dm_to_number(self.kidney[2])
              if strmatch(self.kidney[3],'yes',/fold_case) then kstep = dm_optimizekidstep(kstep,kidran)
              nkid = (kstep eq 0)?1:(floor(abs(kidran[1]-kidran[0])/kstep)+1)              
              for j=0L,nkid-1L do begin
                  tmp_kid = (kidran[0])>(kidran[0]+kstep*j)<(kidran[1])
                  if n_elements(a3) eq 0 then begin
                     a3     = self.psi[0]+self.psi[2]*findgen(na3)         
                     en     = replicate(tmp_en,na3)
                     Ei     = replicate(tmp_ei,na3)
                     kidney = replicate(tmp_kid,na3)
                  endif else begin
                     a3     = [a3,self.psi[0]+self.psi[2]*findgen(na3)]
                     en     = [en,replicate(tmp_en,na3)]
                     Ei     = [Ei,replicate(tmp_ei,na3)]
                     kidney = [kidney,replicate(tmp_kid,na3)]
                  endelse
              endfor
          endfor
          nen = n_elements(en)
          Ef  = Ei-en
          psi = 90.0-temporary(a3)
          ;check kidney
          index = dm_check_macskidney(kidney,Ei,count,complement=bindex,offset=self.kidney_offset,/spline)
          if (count eq 0) then begin
             ok = dialog_message('All kidney angles are out of valid range.',/error,dialog_parent=self.tlb,/center)
             return 
          endif else if count ne nen then begin
             bad_en  = en[bindex]
             bad_kid = kidney[bindex]
             Ei  = Ei[index]
             Ef  = Ef[index]
             en  = en[index]
             psi = psi[index]
             kidney = kidney[index]
             tmp = bad_en[uniq(bad_en,sort(bad_en))]
             if n_elements(mesg) eq 0 then mesg = 'The following kidney angles are out of valid range.' $
             else mesg = [mesg,'','The following kidney angles are out of valid range.']
             for i=0L,n_elements(tmp)-1L do begin
                 index = where(bad_en eq tmp[i])
                 tmpmin = min(bad_kid[index],max=tmpmax)
                 mesg = [mesg,'E='+dm_to_string(tmp[i])+' meV, kidney='+dm_rangestring(bad_kid[index],step=dm_to_number(self.kidney[2]),unit=string('b0'XB),tolerance=0.0001)]
             endfor
             mesg = [mesg,'These kidney angles have been removed from projection calculation.','']
             nen = count
          endif
          index = 0  & bindex = 0
       endif else begin
          ;ask for Ei or Ef according to instrument geometry
          eief = 5.0
          if (finite(self.eief)) then $
             if self.eief gt 0 then eief=self.eief
          if self.samp_typ eq 2 then begin             ;diffuse scattering single E
             case self.e_bin[self.samp_typ] of
               0:  en = 0.0                                      ;use elastic energy for all time channels
               1:  en = 0.0                                      ;use elastic energy for elastic peak
               2:  en = (self.e_range[0]+self.e_range[1])/2.0    ;use the mean energy for specified energy range
               else: en = 0.0
             endcase
             askeran = 0b
             if self.instrname eq 'wand' then $
                info = ['detector offset angle='+dm_to_string(-(*self.detPosPtr).two_theta[0],resolution=3)+string('b0'XB)] $
             else $
                info = ['E='+dm_to_string(en)+' meV']
          endif else if self.samp_typ eq 3 then begin  ;diffuse scattering multiple E
             askeran = 1b
             case self.e_bin[3] of
               0:  info = (['-Ei< E <Ei','-Ef< E <Ef'])[self.instrgeom]
               1:  begin
                   info = ['E~0 meV']
                   askeran = 0b
                   end
               2:  begin
                   info = [dm_to_string(self.e_range[0])+'< E <'+dm_to_string(self.e_range[1])]
                   if strlen(userf_str) ne 0 then self.estartend[0:1] = self.e_range[0:1]
                   end
               else:
             endcase
          endif else begin
             info = (['-Ei< E <Ei','-Ef< E <Ef'])[self.instrgeom]
             askeran = 1b
          endelse
          if (self.instrname eq 'dcs') or (self.instrname eq 'wand') then begin    ;for DCS convert energy(meV) to wavelength(angstrom)
             eief  = dm_e2lambda(eief)
             title = "Incident wavelength:"
          endif else $
             title = (["Incident energy:","Final energy:"])[self.instrgeom]
          notok = 1b
          while notok do begin
             if askeran and strlen(userf_str) ne 0 then begin
                if (self.instrname eq 'dcs') or (self.instrname eq 'wand') then begin
                   eief = dm_dialog_input(['wavelength ('+string('c5'XB)+'):','E_start:','E_end:','n_E:'],/float,default=[finite(eief[0])?round(eief[0]*1000.0,/L64)/1000.0:eief[0],self.estartend],$
                          dialog_parent=self.tlb,info='E_start< E <E_end, n_E is the sampling number.',/lambda2e,title=title,/align_center,cancel=cancel)
                endif else begin
                   eief = dm_dialog_input([(['Ei','Ef'])[self.instrgeom]+'(meV):','E_start:','E_end:','n_E:'],/float,default=[finite(eief[0])?round(eief[0]*1000.0,/L64)/1000.0:eief[0],self.estartend],$
                          dialog_parent=self.tlb,info='E_start< E <E_end, n_E is the sampling number.',/e2lambda,title=title,/align_center,cancel=cancel)
                endelse
             endif else begin
                if (self.instrname eq 'dcs') or (self.instrname eq 'wand') then begin
                   eief = dm_dialog_input('wavelength ('+string('c5'XB)+'):',/float,default=finite(eief)?round(eief*1000.0,/L64)/1000.0:eief,dialog_parent=self.tlb,info=info,/lambda2e,title=title,$
                          /align_center,cancel=cancel)
                endif else begin
                   eief = dm_dialog_input((['Ei','Ef'])[self.instrgeom]+'(meV):',/float,title=title,default=finite(eief)?round(eief*1000.0,/L64)/1000.0:eief,dialog_parent=self.tlb,info=info,/e2lambda,$
                          /align_center,cancel=cancel)
                endelse
             endelse
             if keyword_set(cancel) then begin
                self.error = 0
                return
             endif
             if (total(finite(eief)) eq n_elements(eief)) then begin
                if eief[0] gt 0 then notok = 0b
             endif
          endwhile
          if (self.instrname eq 'dcs') or (self.instrname eq 'wand') then eief[0] =  dm_lambda2e(eief[0])    ;for DCS and WAND convert wavelength to energy             
          self.eief = eief[0]  ;save ei or ef information
          if askeran and strlen(userf_str) ne 0 then begin
             if eief[3] lt 1 then eief[3] = 81
             if self.instrgeom eq 0 then eief[2] = min([eief[2],eief[0]]) $
             else eief[1] = max([eief[1],-eief[0]])
             en = eief[1] + findgen(eief[3])/(eief[3]-1)*(eief[2]-eief[1]) +(2*self.instrgeom-1)*1e-6
             self.estartend = eief[1:3]
             eief = eief[0]
          endif else begin
             nnen = 81.           ;sampling number
             if self.samp_typ eq 2 then begin             ;diffuse scattering single E
                if (self.instrgeom eq 0) and (en gt eief) then begin
                   ok = dialog_message('Incident energy is less than the energy transfer: E>Ei. E=0 is used.',/error,dialog_parent=self.tlb,/center)
                   en = 0.0
                endif
                if (self.instrgeom eq 1) and (en lt -eief) then begin
                   ok = dialog_message('Neutron energy gain is more than the final energy: E<-Ef. E=0 is used.',/error,dialog_parent=self.tlb,/center)
                   en = 0.0
                endif
             endif else if self.samp_typ eq 3 then begin  ;diffuse scattering multiple E
                case self.e_bin[3] of
                  0:  en = (findgen(nnen)/((nnen-1)/2.)-1)*eief+(2*self.instrgeom-1)*1e-6         ;[-Ei,...,Ei] or [-Ef,...Ef]
                  1:  begin
                      nnen = 21.
                      ;use empirical DCS energy resolution as the energy range
                      dE = 0.02008*eief^(1.4735)
                      en = (findgen(nnen)/((nnen-1)/2.)-1)*dE
                      end
                  2:  begin
                      if self.instrgeom eq 0 then begin
                         if min(self.e_range) gt eief then begin
                            ok = dialog_message('Incident energy is smaller than the specified energy range. Elastic peak is used.',/error,dialog_parent=self.tlb,/center)
                            ;use empirical DCS energy resolution as the energy range
                            dE = 0.02008*eief^(1.4735)
                            en = (findgen(nnen)/((nnen-1)/2.)-1)*dE
                         endif else begin
                            eran = [self.e_range[0],min([self.e_range[1],eief])]
                            en   = eran[0] + findgen(nnen)/(nnen-1)*(eran[1]-eran[0])-1e-6
                         endelse
                      endif else begin
                         if max(self.e_range) lt -eief then begin
                            ok = dialog_message('Neutron energy gain in the specified energy range is more than the final energy. Elastic peak is used.',/error,dialog_parent=self.tlb,/center)
                            ;use empirical DCS energy resolution as the energy range
                            dE = 0.02008*eief^(1.4735)
                            en = (findgen(nnen)/((nnen-1)/2.)-1)*dE
                         endif else begin
                            eran = [max(self.e_range[0],-eief),self.e_range[1]]
                            en   = eran[0] + findgen(nnen)/(nnen-1)*(eran[1]-eran[0])+1e-6
                         endelse
                      endelse
                      end
                  else:
                endcase
             endif else begin
                en = (findgen(nnen)/((nnen-1)/2.)-1)*eief +(2*self.instrgeom-1)*1e-6         ;[-Ei,...,Ei] or [-Ef,...Ef]
             endelse
          endelse
          if self.samp_typ ge 2 then begin ;psi range for diffuse scattering type
             notok = 1b
             if total(finite(self.psi[0:2])) gt 0 then psi=self.psi[0:2] else psi = [0,50,2]
             name = ['Start:','End:','Step:']
             if self.datr_yn then begin
                psi    = [psi,self.datr_psi,self.datr_width]
                name   = [name,'Dark Angle:','Dark Angle Width:'] 
                dainfo = ['Dark angle position is relative to the crystal axis u.','Counterclockwise rotation is positive.']
             endif
             while notok do begin
                psi = dm_dialog_input(name,title='Psi range:',default=psi,/float,dialog_parent=self.tlb,info=dainfo,cancel=cancel)
                if keyword_set(cancel) then begin
                   self.error = 0
                   return
                endif
                notok = (total(finite(psi[0:2],/nan)) ne 0)
             endwhile
             psi[2] = abs(psi[2])
             min_a  = min(psi[0:1],max=max_a)
             if psi[2] eq 0 then max_a = min_a $
             else if min_a eq max_a then psi[2] = 0
             self.psi[0:2] = [min_a,max_a,psi[2]]
             if self.datr_yn then begin
                self.datr_psi   = psi[3]
                self.datr_width = abs(psi[4])
             endif
             if psi[2] eq 0 then psi = min_a $
             else psi = min_a+findgen(fix((max_a-min_a)/psi[2])+1)*psi[2]
             n_file = n_elements(psi)
          endif
       endelse
       qty  = 0.
       err  = 0.
       ewid = 0.
       time = 0.
    endif else begin
       zero_error       = self.zero_error[1]
       data_temperature = (*self.dataStrPtr).temperature  ;data temperture, for MACS, DCS and WAND   
       data_hfield      = (*self.dataStrPtr).hfield       ;magnetic field, for MACS and DCS 
       data_specify     = (*self.dataStrPtr).specify      ;specified data, for MACS   
       if self.instrname eq 'macs' then begin
          macshistonchan = self.macshistonchan-(self.macshistocut?dm_macs_histotail(self.macshistonchan):0)
          qty = (*self.dataStrPtr).qty
          err = (*self.dataStrPtr).err
          Ei  = (*self.dataStrPtr).Ei
          Ef  = (*self.dataStrPtr).Ef
          psi = 90.0-((*self.dataStrPtr).rotation+orie_psi)  ;actually A3,orie_psi is the A3 offset angle, 90-a3 due to u definition and rotation definition  
          en  = Ei-Ef
          eief = self.eief
          kidney = (*self.dataStrPtr).kidney
          if self.macs_ptaionly then ptai = (*self.dataStrPtr).ptai
          if self.macs_lamb2 then weight = dm_macs_lambda2correction(Ei,(*self.dataStrPtr).weight[*,self.mon_typ[1]],(*self.dataStrPtr).cfx) $
          else weight = (*self.dataStrPtr).weight[*,self.mon_typ[1]]
          histo_weight = (*self.dataStrPtr).histo_weight
          tmp_size = size(qty,/dim)
          macshistodata = ((tmp_size[0] eq 40) and (tmp_size[1] eq macshistonchan))
          if ~(keyword_set(self.macshistomode) and self.macs_rawintn) then begin
             tmpind = where(weight eq 0,cnt)
             if cnt ne 0 then weight[tmpind] = 1.0
             monfac = (self.macs_mon[self.mon_sumfit[1],self.mon_typ[1]])/weight
             if cnt ne 0 then weight[tmpind] = 0.0
             for i=0LL,n_elements(monfac)-1 do begin
                 qty[*,*,i] = qty[*,*,i]*monfac[i]
                 err[*,*,i] = err[*,*,i]*monfac[i]
             endfor
             zero_error = zero_error*min(monfac)
             monfac = 0
          endif          
          ewid = 1
       endif else begin
          ewid = (*self.dataStrPtr).ewid
          eief = (*self.dataStrPtr).eief & self.eief = eief
          qty  = (*self.dataStrPtr).qty
          err  = (*self.dataStrPtr).err
          weight = (*self.dataStrPtr).weight
          while (~ finite(eief)) do begin
               eief = dm_dialog_input((['Ei','Ef'])[self.instrgeom]+'(meV):',/float,default=eief,dialog_parent=self.tlb,/e2lambda)
               (*self.dataStrPtr).eief = eief
               self.eief = eief
          endwhile
          en   = (*self.dataStrPtr).e 
          time = (*self.dataStrPtr).time
          if self.samp_typ ge 2 then begin                      ;diffuse scattering type
             psi = (*self.dataStrPtr).rotation                  ;angle 2-rotation
             self.orie_offset = orie_psi-self.psi[3]            ;save the offset information
             psi = psi+self.orie_offset                         ;adjust the offset
             n_file = n_elements(psi)
             tmp_psi = dm_startend(psi,resolution=2)
             if (tmp_psi[2] ne 0) or (self.pow2cryexpan eq 0) then self.psi[0:2] = tmp_psi
          endif else begin
             if size(qty,/n_dim) eq 3 then n_file = (size(qty,/dimen))[2] else n_file = 1   ;appilcable for dcs when temperature or magnetic field axis is enbaled
             if (self.samp_typ eq 1) and finite((*self.dataStrPtr).rotation) then self.orie_offset = orie_psi-(*self.dataStrPtr).rotation
          endelse
       endelse
       if size(weight,/n_dim) eq 0 then weight = reform(weight,1)  ;force it to be an array to avoid rebin error
    endelse

    det_tt  = (*self.detPosPtr).two_theta 
    det_psi = (*self.detPosPtr).psi
    nen     = n_elements(en)
    ndet    = n_elements(det_tt)

    if self.intn_typ eq (n_elements(self.intn_typ_str)-2) then begin  ;J(Q,y), need to ask for mass
       Mjqy = !values.f_nan
       while (finite(Mjqy,/nan)) do begin
          Mjqy = dm_dialog_input('Mass:',/float,default=self.mjqy,dialog_parent=self.tlb,cancel=cancel,info=['The mass is in atomic mass unit.','It is for J(Q,y) calculation.'],title=(['Enter mass:','Please confirm:'])[finite(self.mjqy)])
          if keyword_set(cancel) then begin
             self.error = 0
             return
          endif
       endwhile
       self.mjqy = Mjqy
    endif
   
    widget_control,/hourglass
    
    ;user function executed before empty can subtraction
    if (strlen(userf_str) ne 0) and (self.userf_before eq 2) then begin
       if self.userfwarning then begin
          if lmgr(/runtime) then usertmp = 'User Macro' else usertmp = 'User Macro/Function/Procedure'
          if n_elements(mesg) eq 0 then mesg = [usertmp+' is specified.',''] $
          else mesg = [usertmp+' is specified.','',mesg]
          self.userfwarning = 0b
       endif
       ok = execute(userf_str,1,1)
       if ~ok then begin
          errmsg = strmid(!error_state.msg,0,strlen(!error_state.msg)-1)+' in the user macro.'
          ok = dialog_message(errmsg,/error,dialog_parent=self.tlb,/center)
          return  ;error encountered
       endif
       if n_elements(err) ne n_elements(qty) then begin ;in case qty and err are not both defined in user functions
          if n_elements(err) eq 1 then begin
             err = replicate(err,size(qty,/dimension))
          endif else begin
             help,qty,err,output=tmp
             ok = dialog_message(['Incompatible array sizes of intensity and error bar, please check the user macro.','',tmp],/error,dialog_parent=self.tlb,/center)
             return
          endelse
       endif
    endif
    
    if ptr_valid(self.dataStrPtr) then begin
       if (self.samp_typ eq 0) and self.ftypecheckmask[0] and (self.instrname ne 'macs') then begin
          i = 0L
          while(total(finite(qty[i,*,*])) eq 0) do i=i+1    ;in case the first time channel is masked
          qtydet =  reform(qty[i,*,*])                      ;save the unadjust qty info for later mask determination
       endif
       ;subtract overall background rate
       if ptr_valid(self.bgratePtr) then begin
          bgrate = (*self.bgratePtr)
          nrate  = n_elements(bgrate)
          if nrate gt 1 then bgrate = bgrate[*,0]
          case self.instrname of
            'dcs':begin   ;dcs
                  if nrate eq 1 then begin    ;overall rate
                     if ptr_valid(self.eff) then tmp_eff=(*self.eff) else tmp_eff=fltarr(ndet)+1.
                     tmp_value = bgrate/3600./(self.monrate)*temporary(tmp_eff)
                  endif else $
                     tmp_value = bgrate/3600./(self.monrate)
                  if self.samp_typ ge 2 then begin      ;diffuse scattering type
                     if self.samp_typ eq 2 then begin   ;single energy
                        if self.eint_yn then $
                           tmp_value = tmp_value*total(ewid) $
                        else $
                           tmp_value = tmp_value
                        for i=0L,n_file-1 do qty[*,i]=qty[*,i]-tmp_value 
                     endif else begin                    ;multiple energy
                        for i=0L,n_file-1L do begin
                            for j=0L,nen-1L do $
                                qty[j,*,i] = qty[j,*,i]-tmp_value
                        endfor
                     endelse
                  endif else begin
                     if n_file ne 1 then tmp_value = rebin(tmp_value,ndet,n_file)
                     for i=0L,nen-1L do $
                         qty[i,*,*] = qty[i,*,*]-tmp_value
                  endelse
                  tmp_value = 1     ;free up the memory
                  end
            else:
          endcase
       endif
       ;subtract detector dependent background data
       if ptr_valid(self.bgdetsPtr) then begin
          tmp_value = (*self.bgdetsPtr)
          if self.samp_typ ge 2 then begin
             if self.eint_yn then $
                tmp_value = tmp_value*total(ewid)
             if self.samp_typ eq 2 then begin
                for i=0L,n_file-1L do $
                    qty[*,i] = qty[*,i]-tmp_value
             endif else begin
                tmp_value = (fltarr(nen)+1.0)#tmp_value
                for i=0L,n_file-1L do $
                    qty[*,*,i] = qty[*,*,i]-tmp_value
             endelse
          endif else begin
             for i=0L,n_file-1L do begin
                 qty[*,*,i] = qty[*,*,i]-(fltarr(nen)+1.0)#tmp_value
             endfor
          endelse
          tmp_value = 1     ;free up the memory
       endif
       ;subtract time channel background data
       if ptr_valid(self.bgtchnPtr) then begin
          tmp_emin  = min((*self.bgtchnPtr).emean,max=tmp_emax)
          tmp_nen   = n_elements((*self.bgtchnPtr).emean)
          tchan_ind = where(en ge tmp_emin and en le tmp_emax,tchan_count)
          if tchan_count gt tmp_nen then tchan_ind = tchan_ind[0:(tmp_nen-1)]   ;errorproof
          if tchan_count ne 0 then begin
             if self.samp_typ eq 3 then begin
                for i=0L,n_file-1L do $
                    for j=0L,ndet-1L do $
                        qty[tchan_ind,j,i] = qty[tchan_ind,j,i]-(*self.bgtchnPtr).qty
             endif else begin
                if n_file eq 1 then tmp_value = (*self.bgtchnPtr).qty $
                else tmp_value = (*self.bgtchnPtr).qty#(fltarr(n_file)+1.0)
                for i=0L,ndet-1L do qty[tchan_ind,i,*] = qty[tchan_ind,i,*]-tmp_value
                tmp_value = 1   ;free up the memory
             endelse
          endif
       endif else tchan_count = 0
       ;subtract empty can file
       if ptr_valid(self.bgdata[0]) then begin
          ssfac = 1.0     ;self-shielding factor
          if ptr_valid(self.ssfacPtr) then begin
             if (*self.ssfacPtr).det_or_en eq 0 then begin  ;detector dependent ssfac
                if ndet eq n_elements((*self.ssfacPtr).ssfac) then begin
                   ssfac = reform(make_array(nen,value=1d)#(*self.ssfacPtr).ssfac)
                endif else begin
                   ok = dialog_message('Incompatible detector dependent self-shielding factor array size ('+dm_to_string(n_elements((*self.ssfacPtr).ssfac))+') and detector number ('+dm_to_string(ndet)+$
                        '). ssfac=1.0 is used instead.',dialog_parent=self.tlb,/center)
                   widget_control,/hourglass
                endelse
             endif else begin   ;energy dependent ssfac
                if nen eq n_elements((*self.ssfacPtr).ssfac) then begin
                   ssfac = reform((*self.ssfacPtr).ssfac#make_array(ndet,value=1d))
                endif else begin
                   ok = dialog_message('Incompatible energy dependent self-shielding factor array size ('+dm_to_string(n_elements((*self.ssfacPtr).ssfac))+') and time channel number ('+dm_to_string(nen)+$
                      '). ssfac=1.0 is used instead.',dialog_parent=self.tlb,/center)
                   widget_control,/hourglass
                endelse
             endelse
          endif else if finite(self.ssfac) then ssfac = self.ssfac
          for i=0,n_elements(self.bgdata)-1 do begin
              if ptr_valid(self.bgdata[i]) then begin
                 if (self.instrname eq 'macs') and keyword_set(self.macshistomode) and self.macs_rawintn then begin
                    self->dm_subbg,qty,err,nen=nen,n_file=n_file,ssfac=ssfac,en=en,Ei=Ei,Ef=Ef,kidney=kidney,mesg=mesg,tchan_count=tchan_count,$
                          tchan_ind=tchan_ind,empty_count=empty_count,debug=debug,bgset=i,/raw,weight0=weight,histo_weight0=histo_weight
                 endif else begin
                    self->dm_subbg,qty,err,nen=nen,n_file=n_file,ssfac=ssfac,en=en,Ei=Ei,Ef=Ef,kidney=kidney,mesg=mesg,tchan_count=tchan_count,$
                          tchan_ind=tchan_ind,empty_count=empty_count,debug=debug,bgset=i
                 endelse
              endif
          endfor
       endif
       if n_elements(empty_count) ne 0 then begin
          if empty_count gt 30 then mesg = [mesg,'...',dm_to_string(empty_count-30)+' more mismatch'+(['es are',' is'])[empty_count eq 31]+' not listed.']
          if empty_count gt 0  then mesg = [mesg,'Try increasing the empty can subtraction tolerance value in the parameter menu.','']
       endif
       ;efficiency adjustment
       if ptr_valid(self.eff) then begin
          if self.instrname eq 'macs' then begin
             if macshistodata then begin
                if ~self.macs_rawintn then begin  ;efficiency is applied to weight
                   for i=0,ndet*2-1 do begin
                       qty[i,*,*] = qty[i,*,*]*(*self.eff)[i mod 2,i/2]
                       err[i,*,*] = err[i,*,*]*(*self.eff)[i mod 2,i/2]
                   endfor
                endif
             endif else begin
                for i=0LL,nen-1L do begin
                    qty[*,*,i] = qty[*,*,i]*(*self.eff)
                    err[*,*,i] = err[*,*,i]*(*self.eff)
                endfor
             endelse
          endif else begin
             if self.samp_typ ge 2 then begin    ;diffuse scattering type
                if self.samp_typ eq 2 then begin
                   for i=0L,n_file-1L do begin
                       qty[*,i] = qty[*,i]/(*self.eff)
                       err[*,i] = err[*,i]/(*self.eff)
                   endfor
                endif else begin
                   for i=0L,n_file-1L do begin
                       for j=0L,nen-1L do begin
                           qty[j,*,i] = qty[j,*,i]/(*self.eff)
                           err[j,*,i] = err[j,*,i]/(*self.eff)
                       endfor
                   endfor
                endelse
             endif else begin             
                if n_file eq 1 then tmp_value = (*self.eff) else tmp_value = (*self.eff)#(fltarr(n_file)+1.0)
                for i=0L,nen-1L do begin
                    qty[i,*,*] = qty[i,*,*]/tmp_value
                    err[i,*,*] = err[i,*,*]/tmp_value
                endfor
                tmp_value = 1.0
             endelse
          endelse
       endif
       ;Debye-Waller adjustment
       if (self.intn_typ eq 4) or (self.intn_typ eq 5) then begin   
          ;ask for u^2
          oldu2 = 0. & u2_existed = 0b
          if n_tags((*self.dataStrPtr).info) ne 0 then begin
             u2names = tag_names((*self.dataStrPtr).info)
             index  = where(u2names eq 'DWU2',count)
             if count ne 0 then begin
                oldu2 = (*self.dataStrPtr).info.dwu2
                u2_existed = 1b
             endif
          endif
          if n_elements(dwu2) ne 0 then begin
             notvalid = 0b
             dwu2 = dwu2[0]       
          endif else $
             notvalid = 1b
          while notvalid do begin
             dwu2 = dm_dialog_input('u^2:',/float,default=oldu2,dialog_parent=self.tlb,title='Debye-Waller correction:',info='exp(u^2*Q^2) will be applied to correct for the Debye-Waller factor.')
             widget_control,/hourglass
             if finite(dwu2) then $
                if dwu2 ge 0 then begin
                   if u2_existed then (*self.dataStrPtr).info.dwu2=dwu2
                   notvalid = 0b ;valid u^2 given
                endif
          endwhile
       endif
       ;kfactor adjustment
       if (self.intn_typ eq 1) or (self.intn_typ eq 3) or (self.intn_typ eq 4) or (self.intn_typ eq 5) then begin   ;detailed balance factor
          kb = 8.617342e-2    ;Boltzmann constant in mev.k^-1
          if self.instrname ne 'macs' then begin
             if self.instrname eq 'dcs' and (total(self.extravaxis_yn[1:2]) ne 0) then begin
                if n_elements(temperature) eq n_elements((*self.dataStrPtr).filename) then begin
                   notvalid = 0b
                   temp = temperature
                endif else begin
                   notvalid = 1b
                   typ = (self.extravaxis_yn[0]?self.macstemp_typ:2)
                   old = data_temperature[typ,*]
                endelse
                while notvalid do begin
                   temp = dm_dialog_input((*self.dataStrPtr).filename+':',/float,default=old,dialog_parent=self.tlb,title='Enter temperature',info='The temperature is for detailed balance calculation.')
                   widget_control,/hourglass
                   if total(finite(temp,/nan)) eq 0 then begin
                      if total((temp le 0)) eq 0 then begin
                         (*self.dataStrPtr).temperature[typ,*] = temp ;is this really a good idea?
                         notvalid = 0b  ;valid temperature given
                      endif
                   endif
                endwhile
                dbal = reform(exp(-en#(0.5/kb/temp)))
             endif else begin
                ;ask for a sample temperature
                old = 300. & t_existed = 0b
                if n_tags((*self.dataStrPtr).info) ne 0 then begin
                   tnames = tag_names((*self.dataStrPtr).info)
                   index  = where(tnames eq 'TEMPERATURE',count)
                   if count ne 0 then begin
                      old = (*self.dataStrPtr).info.temperature
                      t_existed = 1b
                   endif
                endif
                if n_elements(temperature) ne 0 then begin
                   notvalid = 0b
                   temp = temperature[0]       
                endif else $
                   notvalid = 1b
                while notvalid do begin
                   temp = dm_dialog_input('sample temperature:',/float,default=old,dialog_parent=self.tlb,title='Detailed balance calculation')
                   widget_control,/hourglass
                   if finite(temp) then begin
                      if temp gt 0 then begin
                         if t_existed then (*self.dataStrPtr).info.temperature = temp
                         notvalid = 0b ;valid temperature given
                      endif
                   endif
                endwhile
                ;calculate detailed balance factor
                dbal = exp(-en/(2.0*kb*temp))
             endelse
          endif else begin
             dbal = exp(-en/(2.0*kb*data_temperature[*,self.macstemp_typ]))
          endelse
       endif
       case self.intn_typ of    ;same case for ((self.instrname eq 'nxspe') and (self.lamcorrnxspe eq 0)) 
          0:  if self.instrname eq 'macs' then $      ;S(Q,omega)
              tmp = sqrt(2.07212/Ef) $
              else $
              tmp = (*self.kfactor)^4                 ;S(Q,omega)
          1:  if self.instrname eq 'macs' then $      ;Symmetrized S(Q,omega)
              tmp = sqrt(2.07212/Ef)*dbal $
              else begin
              tmp = (*self.kfactor)  
              if n_elements(tmp) ne n_elements(dbal) then begin
              arr_size = size(dbal,/dim)
              tmp = rebin(tmp,arr_size[0],arr_size[1]) 
              endif
              tmp = tmp^4*dbal
              endelse            
          2:  tmp = (*self.kfactor)^3                 ;d2sigma/dOmegadEf
          3:  if self.instrname eq 'macs' then $      ;Chi(Q,omega)
              tmp = sqrt(2.07212/Ef)*(1-dbal^2) $
              else begin
              tmp = (*self.kfactor)
              if n_elements(tmp) ne n_elements(dbal) then begin
              arr_size = size(dbal,/dim)
              tmp = rebin(tmp,arr_size[0],arr_size[1])
              endif
              tmp = tmp^4*(1-dbal^2)
              endelse      
          4:  begin
              tmp = (*self.kfactor)^4*en
              if n_elements(tmp) ne n_elements(dbal) then begin
              arr_size = size(dbal,/dim)
              tmp = rebin(tmp,arr_size[0],arr_size[1])  
              endif
              tmp = tmp*(1-dbal^2)                    ;GDOS
              end
          5:  begin
              tmp = (*self.kfactor)^3*en  
              if n_elements(tmp) ne n_elements(dbal) then begin
              arr_size = size(dbal,/dim)
              tmp = rebin(tmp,arr_size[0],arr_size[1])
              endif  
              tmp = tmp*(1-dbal^2)                    ;VDOS
              end   
          6:  if self.instrname eq 'macs' then $      ;J(Q,y)
              tmp = sqrt(2.07212/Ef) $
              else $
              tmp = (*self.kfactor)^4
         else:tmp = 1.0                               ;no kfactor adjustment
       endcase
       if ((self.instrname eq 'nxspe') and (self.lamcorrnxspe eq 1)) then begin ;very VERY unlikely case
          case self.intn_typ of
            0:  tmp = (*self.kfactor)^3               ;S(Q,omega)
            1:  tmp = (*self.kfactor)^3*dbal          ;Symmetrized S(Q,omega)
            2:  tmp = (*self.kfactor)^2               ;d2sigma/dOmegadEf
            3:  tmp = (*self.kfactor)^3*(1-dbal^2)    ;Chi(Q,omega)
            4:  tmp = (*self.kfactor)^3*(1-dbal^2)*en ;GDOS
            5:  tmp = (*self.kfactor)^2*(1-dbal^2)*en ;VDOS   
            6:  tmp = (*self.kfactor)^3               ;J(Q,y)
           else:tmp = 1.0                             ;no kfactor adjustment
          endcase
       endif
       if ((self.instrname eq 'nxspe') and (self.lamcorrnxspe eq 2)) then begin
          case self.intn_typ of
            0:  tmp = (*self.kfactor)                 ;S(Q,omega)
            1:  tmp = (*self.kfactor)*dbal            ;Symmetrized S(Q,omega)
            2:  tmp = 1.0                             ;d2sigma/dOmegadEf
            3:  tmp = (*self.kfactor)*(1-dbal^2)      ;Chi(Q,omega)
            4:  tmp = (*self.kfactor)*(1-dbal^2)*en   ;GDOS
            5:  tmp = (1-dbal^2)*en                   ;VDOS   
            6:  tmp = (*self.kfactor)                 ;J(Q,y)
           else:tmp = 1.0                             ;no kfactor adjustment
          endcase
       endif
       if ((self.instrname eq 'nxspe') and (self.lamcorrnxspe eq 3)) then begin
          case self.intn_typ of
            0:  tmp = 1.0                             ;S(Q,omega)
            1:  tmp = dbal                            ;Symmetrized S(Q,omega)
            2:  tmp = 1.0/(*self.kfactor)             ;d2sigma/dOmegadEf
            3:  tmp = 1-dbal^2                        ;Chi(Q,omega)
            4:  tmp = (1-dbal^2)*en                   ;GDOS
            5:  tmp = (1-dbal^2)*en/(*self.kfactor)   ;VDOS   
            6:  tmp = 1.0                             ;J(Q,y)
           else:tmp = 1.0                             ;no kfactor adjustment
          endcase
       endif      
       if self.eint_yn and (self.samp_typ ne 2) then begin
          if n_elements(tmp) ne 1 then begin
             if n_elements(tmp) ne n_elements(ewid) then begin
                arr_size = size(tmp,/dim)
                ewid = rebin(ewid,arr_size[0],arr_size[1])
             endif
          endif
          tmp = temporary(tmp)*ewid
       endif
       ;energy dependent detector efficiency adjustment for direct geometry
       if (self.edep_yn eq 1) and (self.instrgeom eq 0) then begin
          he3_temp = 293.12                                                                     ;assume the temperature is 20C = 293 K
          sigma_acoeff = 1.01325*5333/1.38065/he3_temp/dm_e2lambda(25.3)*self.edep_det[1]*1e-2  ;5333 barn is the he-3 absorption cross section
          sigma_s = 1.01325*6/1.38065/he3_temp*self.edep_det[1]*1e-2                            ;6 barn is the he-3 total scattering cross section
          if self.instrname eq 'macs' then begin
             wl1 = dm_e2lambda(Ef)
          endif else begin
             wl0 = dm_e2lambda((*self.dataStrPtr).eief)                                         ;incident wavelength
             wl1 = wl0*(*self.kfactor)                                                          ;scattered wavelength
          endelse
          sumtd = sigma_acoeff*wl1+sigma_s
          ;deteff_inv=1.0/(1.0-exp(-det_thick*sumtd))
          deteff_inv = sumtd/(sumtd-sigma_s)/(1.0-exp(-self.edep_det[0]*sumtd))                 ;from Jeremy's Equation 9
          sumtd = 0 & wl1 = 0
          if n_elements(tmp) ne 1 then begin
             if n_elements(tmp) ne n_elements(deteff_inv) then begin
                arr_size = size(tmp,/dim)
                deteff_inv = rebin(deteff_inv,arr_size[0],arr_size[1])
             endif
          endif
          tmp = temporary(tmp)*temporary(deteff_inv)
       endif
       if n_elements(tmp) eq 1 then begin
          qty = temporary(qty)*tmp[0]
          err = temporary(err)*tmp[0]
       endif else begin
          if self.samp_typ ge 2 then begin
             if self.samp_typ eq 2 then begin      
                qty = temporary(qty)*tmp[0]                ;tmp is a scalor here, qty is a [ndet,nfile] arrray
                err = temporary(err)*tmp[0]
             endif else begin                  
                if n_elements(tmp) eq nen then begin       ;tmp is a [nen] array, qty is a [nen,ndet,nfile] array
                   for i=0L,ndet-1L do begin
                       for j=0L,n_file-1L do begin
                           qty[*,i,j] = qty[*,i,j]*tmp
                           err[*,i,j] = err[*,i,j]*tmp
                       endfor
                   endfor
                endif else begin                           ;tmp is a [nen,nfile] array, qty is a [nen,ndet,nfile] array
                   for i=0L,ndet-1L do begin
                       qty[*,i,*] = qty[*,i,*]*tmp
                       err[*,i,*] = err[*,i,*]*tmp
                   endfor
                endelse
             endelse
          endif else begin
             if self.instrname eq 'macs' then begin
                if macshistodata then begin                ;tmp is a [nen] array, qty is a [40,nchan,nen] array, only apply the factor to the spec intensity
                   for i=0,2*(ndet-1),2 do begin
                       for j=0L,macshistonchan-1 do begin
                           qty[i,j,*] = qty[i,j,*]*tmp
                           err[i,j,*] = err[i,j,*]*tmp
                       endfor
                   endfor
                endif else begin
                   for i=0LL,ndet-1L do begin              ;tmp is a [nen] array, qty is a [2,ndet,nen] array, only apply the factor to the spec intensity
                       qty[0,i,*] = qty[0,i,*]*tmp
                       err[0,i,*] = err[0,i,*]*tmp
                   endfor   
                endelse
             endif else begin                              ;tmp is a [nen[,nfile]] array, qty is a [nen,ndet[,nfile]] array
                if (n_file gt 1) and (n_elements(tmp) eq nen) then tmp = rebin(tmp,nen,n_file)
                for i=0L,ndet-1L do begin      
                    qty[*,i,*] = qty[*,i,*]*tmp
                    err[*,i,*] = err[*,i,*]*tmp
                endfor
             endelse
          endelse
       endelse
       zero_error = zero_error*min(tmp[where(tmp ne 0)])
       tmp = 0    ;free up the memory
       
       ;energy rebin
       if (self.e_bin[self.samp_typ] gt 1) and (self.samp_typ lt 2) and (self.instrname ne 'macs') then begin
          dm_grid_ebin,self.e_bin[self.samp_typ],qty,err,en,debug=debug,avgsum=self.bin_avgsum
          dm_grid_ebin,self.e_bin[self.samp_typ],ewid
          if (self.samp_typ eq 0) and (self.instrname eq 'dcs') then dm_grid_ebin,self.e_bin[self.samp_typ],time
       endif
       ;calculate dark angle correction factor
       if (self.samp_typ ge 2) and self.dcor_yn and ptr_valid(self.dcor_info) then begin
          tmp_qty = (*self.dcor_info).factor
          tmp_ang = (*self.dcor_info).psi
          tmp_fac = fltarr(n_file)
          for i=0L,n_file-1L do begin
              tmp1 = abs(tmp_ang-(*self.dataStrPtr).rotation[i])  ;find the closest angle index
              tmp1 = min(tmp1,index)    ;index = (0L)>(value_locate(tmp_ang,(*self.dataStrPtr).rotation[i])) ;sort tmp_ang first if using value_locate
              tmp_fac[i] = tmp_qty[index[0]]
          endfor
          if self.samp_typ eq 2 then begin
             for i=0L,n_file-1L do begin
                 qty[*,i] = qty[*,i]*(tmp_fac[i])
                 err[*,i] = err[*,i]*(tmp_fac[i])
             endfor
          endif else begin
             for i=0L,n_file-1L do begin
                 qty[*,*,i] = qty[*,*,i]*(tmp_fac[i])
                 err[*,*,i] = err[*,*,i]*(tmp_fac[i])
             endfor
          endelse
       endif 

       if (self.instrname eq 'macs') and self.macs_ptaionly then begin
          tmp = indgen(ndet)+1
          if macshistodata then begin 
             for i=0L,nen-1L do begin
                 ind = where(tmp ne ptai[i])
                 qty[[2*ind,2*ind+1],*,i] = !values.f_nan
                 err[[2*ind,2*ind+1],*,i] = !values.f_nan
             endfor
          endif else begin
             for i=0L,nen-1L do begin
                 ind = where(tmp ne ptai[i])
                 qty[*,ind,i] = !values.f_nan
                 err[*,ind,i] = !values.f_nan
             endfor
          endelse
       endif
    endif

    ;mask a3 for macs
    if (self.instrname eq 'macs') and ptr_valid(self.a3mask) then begin
       tmp_a3 = 90.0-psi-([0,orie_psi])[ptr_valid(self.dataStrPtr)]   ;from psi back to instrument a3
       self->dm_maska3,self.a3mask,Ei,Ef,en,tmp_a3,kidney,qty=qty,err=err,data_hfield=data_hfield,data_specify=data_specify,data_temperature=data_temperature,weight=weight,histo_weight=histo_weight
       psi = 90.0-tmp_a3-([0,orie_psi])[ptr_valid(self.dataStrPtr)]   ;from a3 back to psi
    endif

    ;mask detectors
    ;detector number mask
    if ptr_valid(self.mask) then begin
       tmp = bytarr(ndet)
       if (self.instrname eq 'macs') and (self.masktype eq 1) then tmp = bytarr(nen*ndet)
       tmp[*(self.mask)] = 1b 
       ind = where(tmp eq 0,count) & tmp=0
       if count ne 0 then begin
          if ptr_valid(self.dataStrPtr) then begin
             if self.instrname eq 'macs' then begin
                case self.masktype of 
                   0: begin
                      if macshistodata then begin 
                         tmpind = [2*ind,2*ind+1] & tmpind = tmpind[sort(tmpind)]
                         qty = qty[tmpind,*,*]
                         err = err[tmpind,*,*]
                      endif else begin
                         qty = qty[*,ind,*]
                         err = err[*,ind,*]
                      endelse
                      end
                   1: if n_elements(*(self.mask)) ne 0 then begin
                      if macshistodata then begin
                         if size(qty,/n_dimensions) eq 2 then begin
                            qty = reform(qty,[size(qty,/dim),1],/overwrite)
                            err = reform(err,[size(err,/dim),1],/overwrite)
                         endif
                         qty = transpose(qty,[1,0,2])
                         err = transpose(err,[1,0,2])
                         qty = reform(qty,macshistonchan,nen*ndet*2,/overwrite)
                         err = reform(err,macshistonchan,nen*ndet*2,/overwrite)
                         i_r = (*(self.mask) mod ndet)
                         i_c = (*(self.mask)/ndet)
                         ind = [i_c*ndet*2+i_r*2,i_c*ndet*2+i_r*2+1]
                         qty[*,ind] = !values.f_nan
                         err[*,ind] = !values.f_nan
                         ind = 0
                         qty = reform(qty,macshistonchan,ndet*2,nen,/overwrite)
                         err = reform(err,macshistonchan,ndet*2,nen,/overwrite)
                         qty = transpose(qty,[1,0,2])
                         err = transpose(err,[1,0,2])
                      endif else begin
                         qty = reform(qty,2,nen*ndet,/overwrite)
                         err = reform(err,2,nen*ndet,/overwrite)
                         qty[*,*(self.mask)] = !values.f_nan
                         err[*,*(self.mask)] = !values.f_nan
                         qty = reform(qty,2,ndet,nen,/overwrite)
                         err = reform(err,2,ndet,nen,/overwrite)
                      endelse
                      end
                   else:
                endcase
             endif else begin
                if self.samp_typ ge 2 then begin
                   if self.samp_typ eq 2 then begin
                      qty = qty[ind,*]
                      err = err[ind,*]
                   endif else begin
                      qty = qty[*,ind,*]
                      err = err[*,ind,*]
                   endelse
                endif else begin
                   qty = qty[*,ind,*]
                   err = err[*,ind,*]
                endelse
                if (self.samp_typ eq 0) and self.ftypecheckmask[0] then qtydet = qtydet[ind]
             endelse
          endif
          if (self.masktype eq 0) or (self.instrname ne 'macs') then begin
             det_tt  = det_tt[ind]
             det_psi = det_psi[ind]
             ndet    = count
          endif
          ind = 0
       endif else all_masked = 1b
    endif
    ;angular range mask
    if ptr_valid(self.mask_ar) then begin
       tmp = bytarr(ndet)
       if (self.instrname eq 'macs') then tmp = bytarr(nen*ndet)
       if ptr_valid(self.dataStrPtr) or (self.instrname ne 'macs') then begin
          if self.instrname eq 'macs' then begin  ;data available
             tmp_det_tt = det_tt#(fltarr(nen)+1)
             for i=0L,ndet-1 do tmp_det_tt[i,*] = kidney-76.0+det_tt[i]
          endif else $
             tmp_det_tt = det_tt
          tmpstrs = *(self.mask_ar)
          ;check for > and >= expression
          repeat begin
             tmp1 = stregex(tmpstrs,'>(=*)[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
             if len[0] gt 0 then begin
                tmp2 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
                if len[1] gt 0 then ind = where(tmp_det_tt ge tmp2,count) $  ;'>='
                else ind = where(tmp_det_tt gt tmp2,count)                   ;'>'
                if count gt 0 then tmp[ind] = 1b
                tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
             endif
          endrep until len[0] le 0 
          ;check for > and >= expression
          repeat begin  
             tmp1 = stregex(tmpstrs,'<(=*)[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
             if len[0] gt 0 then begin
                tmp2 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
                if len[1] gt 0 then ind = where(tmp_det_tt le tmp2,count) $  ;'<='
                else ind = where(tmp_det_tt lt tmp2,count)                   ;'<'
                if count gt 0 then tmp[ind] = 1b
                tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
             endif
          endrep until len[0] le 0 
          ;check for : expression
          repeat begin  
             tmp1 = stregex(tmpstrs,'(-?[0-9.]+)[ '+string(9b)+']*:[ '+string(9b)+']*(-?[0-9.]+)',/subexpr,length=len)
             if len[0] gt 0 then begin
                tmp2 = dm_to_number(strmid(tmpstrs,tmp1[1],len[1]))
                tmp3 = dm_to_number(strmid(tmpstrs,tmp1[2],len[2]))
                ind = where((tmp_det_tt ge tmp2) and (tmp_det_tt le tmp3),count) 
                if count gt 0 then tmp[ind] = 1b
                tmpstrs = strmid(tmpstrs,0,tmp1[0])+strmid(tmpstrs,tmp1[0]+len[0],strlen(tmpstrs)-tmp1[0]-len[0])
             endif
          endrep until len[0] le 0 
          ;check for single values
          tmp1 = strsplit(tmpstrs,'[a-zA-Z '+string(9b)+',:<>]',/regex,/extract,count=count)
          for j=0,count-1 do begin
              ind = where(abs(tmp_det_tt-dm_to_number(tmp1[j])) lt 0.0001,count)
              if count gt 0 then tmp[ind] = 1b
          endfor
       endif else begin ;'macs' without data 
          if n_elements(mesg) eq 0 then $ 
             mesg = ['The detector angular range mask has not been applied.',''] $
          else $
             mesg = [mesg,'The detector angular range mask has not been applied.','']
       endelse
       ind = where(tmp eq 0,count,complement=complement,ncomplement=ncomplement) & tmp=0 & tmp_det_tt = 0
       if count ne 0 then begin
          if ptr_valid(self.dataStrPtr) then begin
             if (self.instrname eq 'macs') then begin
                if ncomplement ne 0 then begin
                   if macshistodata then begin
                      if size(qty,/n_dimensions) eq 2 then begin
                         qty = reform(qty,[size(qty,/dim),1],/overwrite)
                         err = reform(err,[size(err,/dim),1],/overwrite)
                      endif
                      qty = transpose(qty,[1,0,2])
                      err = transpose(err,[1,0,2])
                      qty = reform(qty,macshistonchan,nen*ndet*2,/overwrite)
                      err = reform(err,macshistonchan,nen*ndet*2,/overwrite)
                      i_r = (complement mod ndet)
                      i_c = (complement/ndet)
                      complement = 0   
                      ind = [i_c*ndet*2+i_r*2,i_c*ndet*2+i_r*2+1]
                      qty[*,ind] = !values.f_nan
                      err[*,ind] = !values.f_nan
                      ind = 0
                      qty = reform(qty,macshistonchan,ndet*2,nen,/overwrite)
                      err = reform(err,macshistonchan,ndet*2,nen,/overwrite)
                      qty = transpose(qty,[1,0,2])
                      err = transpose(err,[1,0,2])
                  endif else begin
                      qty = reform(qty,2,nen*ndet,/overwrite)
                      err = reform(err,2,nen*ndet,/overwrite)
                      qty[*,complement] = !values.f_nan
                      err[*,complement] = !values.f_nan
                      complement = 0
                      qty = reform(qty,2,ndet,nen,/overwrite)
                      err = reform(err,2,ndet,nen,/overwrite)
                  endelse
                endif
             endif else begin
                if self.samp_typ ge 2 then begin
                   if self.samp_typ eq 2 then begin
                      qty = qty[ind,*]
                      err = err[ind,*]
                   endif else begin
                      qty = qty[*,ind,*]
                      err = err[*,ind,*]
                   endelse
                endif else begin
                   qty = qty[*,ind,*]
                   err = err[*,ind,*]
                endelse
                if (self.samp_typ eq 0) and self.ftypecheckmask[0] then qtydet = qtydet[ind]
             endelse
          endif
          if (self.instrname ne 'macs') then begin
             det_tt  = det_tt[ind]
             det_psi = det_psi[ind]
             ndet    = count
          endif
          ind = 0
       endif else all_masked = 1b
    endif
    if keyword_set(all_masked) then begin
       ok = dialog_message('All detectors were masked. The detector mask has been cleared.',dialog_parent=self.tlb,/center)
       self->clear_mask
    endif
    
    ;absorption correction, data need to be present
    if ptr_valid(self.absp_info) and ptr_valid(self.dataStrPtr) then begin
       ninfo = n_elements(*(self.absp_info))
       info  = [(*(self.absp_info))[0],(*(self.absp_info))[4:(ninfo-1)]]
       if info[0] ge 0 and info[0] lt n_elements(self.absp_npts) then begin
          if self.absp_asint then begin
             qty[*] = 1.0 & err[*] = 0.0 & weight[*] = 1.0 & histo_weight[*] = 1.0
          endif
          npts = (*self.absp_npts[info[0]])[*,(*(self.absp_info))[1]]
          if self.instrname eq 'macs' then begin
             current1 = systime(/sec)
             for i=0LL,nen-1L do begin
                 if info[0] eq 3 then  info[n_elements(info)-1] = (*(self.absp_info))[ninfo-1]+psi[i]-(90.0-self.psi[3]-orie_psi) ;flat plate geometry
                 tmp = dm_calc_shielding(info,([Ei[i],Ef[i]])[self.instrgeom],en[i],kidney[i]-76.0+det_tt,interp=(*(self.absp_info))[2],instrgeom=self.instrgeom,$
                       is_dcs=(self.instrname eq 'dcs'),npoints=npts,group_leader=self.tlb,mesgobj=mesgobj,stopped=stopped,l1factor=self.absp_l1factor,$
                       message='Calculating absorption correction. Please wait...'+dm_to_string(i+1)+'/'+dm_to_string(nen),/noquestion,transmission=self.absp_asint)
                 if i eq 0 then begin
                    time = (systime(/sec)-current1)*nen/60.
                    if time gt 1 then begin
                       ok = dialog_message(['The absorption calculation will take about '+dm_to_string(time,resolution=1)+' mintues. Do you still want to continue?',$
                            'Try reducing the number of sampling points.'],/question,dialog_parent=self.tlb)
                       if ok eq 'No' then stopped = 1b
                    endif
                 endif
                 if (~stopped) then begin
                    if macshistodata then begin
                       for j=0,ndet-1 do begin
                           qty[(2*j):(2*j+1),*,i] = qty[(2*j):(2*j+1),*,i]*tmp[j]
                           err[(2*j):(2*j+1),*,i] = err[(2*j):(2*j+1),*,i]*tmp[j]
                       endfor
                    endif else begin
                       qty[0,*,i] = qty[0,*,i]*tmp
                       qty[1,*,i] = qty[1,*,i]*tmp
                       err[0,*,i] = err[0,*,i]*tmp
                       err[1,*,i] = err[1,*,i]*tmp
                    endelse
                 endif else break
             endfor
          endif else begin
             det_tt1 = det_tt
             if self.psi_spe then begin ;convert SPE detector definition to DCS definition 
                sin_tth = sin(det_tt*!ddtor)
                tmp_psi = det_psi*!ddtor
                ind = where(sin_tth*cos(tmp_psi) gt 0,cnt)
                if cnt gt 0 then det_tt1[ind] = det_tt1[ind]*(-1.0)
                det_psi1 = asin(temporary(sin_tth)*sin(temporary(tmp_psi)))
             endif else begin
                det_psi1 = det_psi*!ddtor
             endelse
             ;convert det_tt to equatorial tt
             det_tt1 = (acos(cos(det_tt1*!ddtor)/cos(det_psi1))*(2.0*(det_tt1 gt 0)-1.0))/!ddtor
             if (~self.psi_0only) and ((*(self.absp_info))[3] eq 0) then begin
                if (*(self.absp_info))[2] then begin
                   det_psi1 = round(det_psi1*1d5)/1d5                                     ;precision up to 6e-4 degree
                   uniq_psi = det_psi1[uniq(det_psi1,sort(det_psi1))]
                endif
                if (n_elements(uniq_psi) lt 5) and (n_elements(uniq_psi) gt 0) then $     ;still allow tth interpolation
                   extra = {det_psi:det_psi1,uniq_psi:uniq_psi} $
                else $
                   extra = {det_psi:det_psi1}
             endif
             if (self.samp_typ lt 2) or (info[0] ne 3) then $
                tmp = dm_calc_shielding(info,eief,en,det_tt1,interp=(*(self.absp_info))[2],instrgeom=self.instrgeom,is_dcs=(self.instrname eq 'dcs'),npoints=npts,group_leader=self.tlb,$
                      stopped=stopped,transmission=self.absp_asint,_extra=extra,l1factor=self.absp_l1factor)
             if self.samp_typ ge 2 then begin    ;diffuse scattering type
                if self.samp_typ eq 2 then begin
                   for i=0L,n_file-1L do begin
                       if info[0] eq 3 then begin ;flat plate in diffuse scattering mode
                          info[n_elements(info)-1] = (*(self.absp_info))[ninfo-1]+psi[i]-orie_psi
                          tmp = dm_calc_shielding(info,eief,en,det_tt1,interp=(*(self.absp_info))[2],instrgeom=self.instrgeom,is_dcs=(self.instrname eq 'dcs'),npoints=npts,$
                                group_leader=self.tlb,transmission=self.absp_asint,l1factor=self.absp_l1factor,stopped=stopped,mesgobj=mesgobj,timefactor=n_file,$
                                message='Calculating absorption correction. Please wait...'+dm_to_string(i+1)+'/'+dm_to_string(n_file),noquestion=(i gt 0),_extra=extra)
                       endif
                       if (~stopped) then begin
                          qty[*,i] = qty[*,i]*tmp
                          err[*,i] = err[*,i]*tmp
                       endif else break
                   endfor
                endif else begin
                   for i=0L,n_file-1L do begin
                       if info[0] eq 3 then begin ;flat plate in diffuse scattering mode
                          info[n_elements(info)-1] = (*(self.absp_info))[ninfo-1]+psi[i]-orie_psi
                          tmp = dm_calc_shielding(info,eief,en,det_tt1,interp=(*(self.absp_info))[2],instrgeom=self.instrgeom,is_dcs=(self.instrname eq 'dcs'),npoints=npts,$
                                group_leader=self.tlb,transmission=self.absp_asint,l1factor=self.absp_l1factor,stopped=stopped,mesgobj=mesgobj,timefactor=n_file,$
                                message='Calculating absorption correction. Please wait...'+dm_to_string(i+1)+'/'+dm_to_string(n_file),noquestion=(i gt 0),_extra=extra)
                       endif
                       if (~stopped) then begin
                          qty[*,*,i] = qty[*,*,i]*tmp
                          err[*,*,i] = err[*,*,i]*tmp
                       endif else break
                   endfor
                endelse
             endif else begin
                if (~stopped) then begin
                   for i=0L,n_file-1L do begin
                       qty[*,*,i] = qty[*,*,i]*tmp
                       err[*,*,i] = err[*,*,i]*tmp
                   endfor
                endif
             endelse
             det_tt1  = 0
             det_psi1 = 0
          endelse
          if stopped then begin  ;change the menu to absorption->None
             self->absorption,-1
             tmpmesg = 'Absorption correction is disabled.'
             if (self.samp_typ ge 2) then begin  ;some intensity and error bars have been modified
                if (i gt 0) or self.absp_asint then tmpmesg = tmpmesg+' The intensities of some angles have been modified. Please recalculate projection'
             endif
             if n_elements(mesg) eq 0 then $
                mesg = [tmpmesg,''] $
             else $
                mesg = [mesg,tmpmesg,'']
          endif
          if obj_valid(mesgobj) then obj_destroy,mesgobj
          tmp = 1    ;free up the memory
          widget_control,/hourglass
       endif
    endif
    
    if self.samp_typ ge 1 then begin
       if self.samp_typ eq 1 then begin
          if self.instrname eq 'macs' then begin
             if self.macsintn_typ ne 0 then begin   ;'DIFF' or 'SPEC+DIFF' intensity, force Ef=Ei
                Ef = Ei
                en[*] = 0.0
             endif
          endif else begin
             psi = orie_psi
          endelse
          type = 1
       endif else type = 2 
    endif

    ptr_free,self.projStrPtr,self.powd_line
  
    ;user function executed before projection
    if (strlen(userf_str) ne 0) and (self.userf_before eq 1) then begin
       if self.userfwarning then begin
          if lmgr(/runtime) then usertmp = 'User Macro' else usertmp = 'User Macro/Function/Procedure'
          if n_elements(mesg) eq 0 then mesg = [usertmp+' is specified.',''] $
          else mesg = [usertmp+' is specified.','',mesg]
          self.userfwarning = 0b
       endif
       ok = execute(userf_str,1,1)
       if ~ok then begin
          errmsg = strmid(!error_state.msg,0,strlen(!error_state.msg)-1)+' in the user macro.'
          ok = dialog_message(errmsg,/error,dialog_parent=self.tlb,/center)
          return  ;error encountered
       endif
       if n_elements(err) ne n_elements(qty) then begin ;in case qty and err are not both defined in user functions
          if n_elements(err) eq 1 then begin
             err = replicate(err,size(qty,/dimension))
          endif else begin
             help,qty,err,output=tmp
             ok = dialog_message(['Incompatible array sizes of intensity and error bar, please check the user macro.','',tmp],/error,dialog_parent=self.tlb,/center)
             return
          endelse
       endif
    endif
    
    max_en = max(en,min=min_en)
    if (self.samp_typ eq 2) then begin
       if finite(self.e_range[0]) then min_en = (min_en)<(self.e_range[0])
       if finite(self.e_range[1]) then max_en = (max_en)>(self.e_range[1])
    endif

    case self.samp_typ of
     0:begin  ;powder
       ndat = n_elements(det_tt)
       ;handle the inherent masks in SPE file data
       if ptr_valid(self.dataStrPtr) and self.ftypecheckmask[0] and (self.instrname ne 'macs') then begin
          ind = where(finite(qtydet),count) & qtydet=0
          if count ne ndat then begin
             qty     = qty[*,ind,*] 
             err     = err[*,ind,*]
             det_tt  = det_tt[ind]
             det_psi = det_psi[ind]
             ndat    = count
          endif
       endif
       if self.instrname eq 'macs' then begin
          Q0 = dm_proj_spec_samp(Ei,Ef,det_tt,det_psi,type=0,kidney=kidney,debug=debug)               ;Q[ndat,ndet]
          Q1 = dm_proj_spec_samp(Ei,Ei,det_tt,det_psi,type=0,kidney=kidney,debug=debug)               ;Q[ndat,ndet]
          Q  = transpose([[[transpose(Q0)]],[[transpose(Q1)]]])                                       ;Q[2,ndat,ndet] 0-spec 1-diff
       endif else begin
          if ~self.negative2th then det_tt = abs(temporary(det_tt))
          Q = dm_proj_spec_samp(eief,en,det_tt,det_psi,type=0,debug=debug,instrgeom=self.instrgeom)   ;Q[nen,ndet]
       endelse
       if ptr_valid(self.powd_qvalue) and ((min_en le elastic_tol) and (max_en ge -elastic_tol)) then begin  ;powder line
          if self.instrname eq 'macs' then $
             maxQ = max(dm_proj_spec_samp(Ei,Ei,det_tt,det_psi,type=0,kidney=kidney),min=minQ) $
          else $
             maxQ = max(abs(sqrt(eief/2.0721246)*sin(det_tt/2*!dtor)*2.),min=minQ) 
          yval = [-2e27,1e-20,2e27]
          for i=0L,n_elements(*self.powd_qvalue)-1 do begin
             if ((*self.powd_qvalue)[i] lt minQ) then continue
             if ((*self.powd_qvalue)[i] gt maxQ) then break
             if n_elements(powd_x) ne 0 then begin
                powd_x = [powd_x,(*self.powd_qvalue)[i],(*self.powd_qvalue)[i],(*self.powd_qvalue)[i]]
                powd_y = [powd_y,yval]
             endif else begin
                powd_x = [(*self.powd_qvalue)[i],(*self.powd_qvalue)[i],(*self.powd_qvalue)[i]]
                powd_y = yval 
             endelse
             yval[[0,2]] = -yval[[0,2]]
          endfor
          if n_elements(powd_x) ne 0 then self.powd_line=ptr_new([[powd_x],[powd_y]],/no_copy)
       endif
       end
     else:begin  ;single crystal and diffuse scattering
       if self.pow2cryexpan and ((self.instrname eq 'macs') or (self.samp_typ gt 1)) and ptr_valid(self.dataStrPtr) then begin  ;powder to single crystal projection
          ;make sure there is only one A3 value in the original data
          if n_elements(uniq(round(psi),sort(round(psi)))) ne 1 then begin
             do_pow2cry = dialog_message('The data might not be powder data. Do you want to continue with the expansion of powder to single crystal projection?',title='Powder to Crystal',$
                  /question,dialog_parent=self.tlb)
          endif else do_pow2cry = 'Yes'
          if do_pow2cry eq 'Yes' then begin
             a3_nam = (['psi','A3'])(self.instrname eq 'macs')
             a3_ran = dm_dialog_input(['from','to','step']+':',title=a3_nam+' range:',default=self.psi[0:2],/float,dialog_parent=self.tlb,info=$
                      ['The '+a3_nam+' range is for expanding the powder data.','Press cancel to skip this expansion.'],cancel=cancel)
             if ~keyword_set(cancel) then begin
                if (total(finite(a3_ran,/nan)) eq 0) and (a3_ran[2] ne 0) then begin
                   a3_ran[2] = abs(a3_ran[2])
                   if a3_ran[1] lt a3_ran[0] then a3_ran[0:1] = a3_ran[[1,0]] 
                   n_a3   = floor((a3_ran[1]-a3_ran[0])/a3_ran[2])
                   n_psi  = n_elements(psi)
                   sz_qty = size(qty)
                   weight = reform(rebin(weight,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                   if self.instrname eq 'macs' then begin
                      qty    = reform(rebin(qty,[sz_qty[1:sz_qty[0]],(n_a3+1L)]),[sz_qty[1:2],n_psi*(n_a3+1L)])
                      err    = reform(rebin(err,[sz_qty[1:sz_qty[0]],(n_a3+1L)]),[sz_qty[1:2],n_psi*(n_a3+1L)])
                      en     = reform(rebin(en,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      Ei     = reform(rebin(Ei,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      Ef     = reform(rebin(Ef,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      psi    = reform(transpose(rebin(90.0-(a3_ran[0]+findgen(n_a3+1)*a3_ran[2]),[n_a3+1L,n_psi])),n_psi*(n_a3+1L))
                      kidney = reform(rebin(kidney,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      if self.extravaxis_yn[1] then data_temperature = transpose(reform(rebin(transpose(data_temperature),[3,n_psi,n_a3+1]),[3,n_psi*(n_a3+1L)]))
                      if self.extravaxis_yn[2] then data_hfield      = reform(rebin(data_hfield,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      if self.extravaxis_yn[0] then data_specify     = reform(rebin(data_specify,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                   endif else begin
                      qty    = reform(rebin(qty,[sz_qty[1:sz_qty[0]],(n_a3+1L)]),[sz_qty[1:(self.samp_typ-1)],n_psi*(n_a3+1L)])
                      err    = reform(rebin(err,[sz_qty[1:sz_qty[0]],(n_a3+1L)]),[sz_qty[1:(self.samp_typ-1)],n_psi*(n_a3+1L)])
                      psi    = reform(transpose(rebin(a3_ran[0]+findgen(n_a3+1)*a3_ran[2],[n_a3+1L,n_psi])),n_psi*(n_a3+1L))
                      if (self.instrname eq 'dcs') and self.extravaxis_yn[1] then data_temperature = reform(rebin(data_temperature,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                      if (self.instrname eq 'dcs') and self.extravaxis_yn[2] then data_hfield      = reform(rebin(data_hfield,[n_psi,n_a3+1]),n_psi*(n_a3+1L))
                   endelse
                endif else ok = dialog_message('Invalid '+a3_nam+' range. The powder to single crystal expansion has been skipped.',dialog_parent=self.tlb,/info)
                self.psi[0:2] = a3_ran
                if self.samp_typ ge 2 then n_file = n_elements(psi)
             endif
          endif else begin 
             self.pow2cryexpan = 0b
             ok = dialog_message('The powder to single crystal expansion has been disabled.',dialog_parent=self.tlb,/info)
          endelse
       endif
      
       if self.fold_type gt 1 then begin  ;symmetry rotation folding
          if self.view_u1fold then begin
             index = where(finite(*(self.view_u1foldcenter)),count)
             if count eq 0 then begin
                if n_elements(mesg) eq 0 then $
                   mesg = ['n-fold symmetry value is invalid.',''] $
                else $
                   mesg = [mesg,'n-fold symmetry value is invalid.','']  
             endif else begin
                n_fold = (*(self.view_u1foldcenter))[index[0]]
                if (self.fold_type eq 4) and (n_fold gt 1) then begin
                   i_max = self.numrot_optn*n_fold
                   n_psi = n_elements(psi)
                   tmpqty = qty & tmperr = err & i=1 & tmprot=360./n_fold
                   if self.numrot_optn eq 0 then tmprot = (1>fix((max(psi)-min(psi))/tmprot))*tmprot
                   while (max(psi)-min(psi) lt 360) or (i lt i_max) do begin
                      psi = [psi,psi[0:n_psi-1]+tmprot*i]
                      weight = [weight,weight]
                      if self.instrname eq 'macs' then begin
                         Ei = [Ei,Ei[0:(n_psi-1)]] & Ef = [Ef,Ef[0:(n_psi-1)]] & en = [en,en[0:(n_psi-1)]] & kidney = [kidney,kidney[0:(n_psi-1)]]
                         if self.extravaxis_yn[1] then data_temperature = [data_temperature,data_temperature[0:(n_psi-1),*]]
                         if self.extravaxis_yn[2] then data_hfield      = [data_hfield,data_hfield[0:(n_psi-1)]]
                         if self.extravaxis_yn[0] then data_specify     = [data_specify,data_specify[0:(n_psi-1)]]
                      endif else begin
                         if (self.instrname eq 'wand') and self.extravaxis_yn[0] then data_temperature = [data_temperature,data_temperature[0:(n_psi-1)]]
                         if (self.instrname eq 'dcs')  and self.extravaxis_yn[1] then data_temperature = [[data_temperature],[data_temperature[*,0:(n_psi-1)]]]
                         if (self.instrname eq 'dcs')  and self.extravaxis_yn[2] then data_hfield      = [data_hfield,data_hfield[0:(n_psi-1)]]
                      endelse
                      if (size(tmpqty,/n_dim) eq 2) and ((self.samp_typ eq 2) or (self.instrname ne 'dcs') or (total(self.extravaxis_yn[1:2]) eq 0)) then begin
                         qty = [[qty],[tmpqty]] & err = [[err],[tmperr]]
                      endif else begin
                         qty = [[[qty]],[[tmpqty]]] & err = [[[err]],[[tmperr]]]
                      endelse
                      i = i+1
                   endwhile
                   n_fold = 1
                   tmpqty = 0 & tmperr = 0
                   if self.instrname eq 'macs' then nen = n_elements(Ei)
                   if self.samp_typ ge 2 then    n_file = n_elements(psi)
                endif
             endelse
          endif
       endif

       if self.instrname eq 'macs' then begin
          Q = dm_proj_spec_samp(Ei,Ef,det_tt,det_psi,psi=psi,type=type,debug=debug,is_spe=self.psi_spe,kidney=kidney)
                                                               ;Q[3,ndat,ndet] in sample frame
       endif else begin
          Q = dm_proj_spec_samp(eief,en,det_tt,det_psi,psi=psi,type=type,debug=debug,is_spe=self.psi_spe,diffetyp=(self.samp_typ ne 2),instrgeom=self.instrgeom)
                                                               ;Q[3,nen,ndet,nfile] in sample frame                                                    
       endelse
       QV = dm_proj_view(Q,U1,U2,U3,no_copy=((self.intn_typ lt 4) or (self.intn_typ eq (n_elements(self.intn_typ_str)-1))),n_fold=n_fold,sr_type=self.fold_type-2,sr_startvec=self.symrot_startvec,debug=debug) ;project onto viewing axis

       ;dark angle trajectory or mask dark angle
       if ((self.samp_typ ge 2) or ((self.samp_typ eq 1) and (self.instrname eq 'macs'))) and finite(self.datr_psi) and finite(self.datr_width) and $
          ((self.datr_yn and (~ptr_valid(self.dataStrPtr))) or (self.mask_darkangle and ptr_valid(self.dataStrPtr))) then begin  
          if ~ptr_valid(self.dataStrPtr) then begin ;dark angle trajectory
             if self.instrname eq 'macs' then begin  ;special case, det range change with Ei
                tmp_nen = 1>((self.estartend[2] eq 0)?1:round(1+(self.estartend[1]-self.estartend[0])/self.estartend[2]))
                tmp_na3 = 1>((self.psi[2] eq 0)?1:round(1+(self.psi[1]-self.psi[0])/self.psi[2]))      
                tmp_a3  = self.psi[0]+findgen(tmp_na3)*self.psi[2]
                da_tt0  = -(90-tmp_a3+self.datr_psi)
                tmp_ndt = n_elements(det_tt)
                tmp_da3 = reform(tmp_a3#(fltarr(6)+1),tmp_na3*6)
                tmp_dtt = reform(da_tt0#(fltarr(6)+1)+(fltarr(tmp_na3)+1)#(((findgen(6)-2.5)/5)*self.datr_width),tmp_na3*6)    ;dark angle range [-0.5,0.5]*self.datr_width
                while (total(da_tt0 gt 180)  ne 0)  do da_tt0  = da_tt0-360*(da_tt0 gt 180)
                while (total(da_tt0 lt -180) ne 0)  do da_tt0  = da_tt0+360*(da_tt0 lt -180)
                while (total(tmp_dtt gt 180)  ne 0) do tmp_dtt = tmp_dtt-360*(tmp_dtt gt 180)
                while (total(tmp_dtt lt -180) ne 0) do tmp_dtt = tmp_dtt+360*(tmp_dtt lt -180)
                ind2 = where(abs((abs(da_tt0)-180)) le self.datr_width/2.0,count2,ncomplement=count1)
                da_en = 0.0 & da_ei = 0.0 & da_tt = 0.0 & tmp_psi = 0.0
                for j=0L,tmp_nen-1L do begin
                    tmp_en = self.estartend[0]+self.estartend[2]*j
                    if self.instrgeom eq 0 then tmp_ei = eief $  ;direct geometry
                    else tmp_ei = eief+tmp_en                    ;inverse geometry
                    ;figure out the detector angle range
                    kidran = dm_to_number(self.kidney[0:1]) 
                    kidlim = dm_getkidneyrange(tmp_ei,offset=self.kidney_offset,/notcheckeimax,/notcheckeimin)
                    if ~kid_fnt[0] then begin
                       if (~lmgr(/vm)) and (strlen(anstr[4]) ne 0) then kidran[0] = dm_calckidney(self.kidney[0],kidlim) else kidran[0] = kidlim[0]
                    endif
                    if ~kid_fnt[1] then begin
                       if (~lmgr(/vm)) and (strlen(anstr[5]) ne 0) then kidran[1] = dm_calckidney(self.kidney[1],kidlim) else kidran[1] = kidlim[1]
                    endif
                    kstep = dm_to_number(self.kidney[2])
                    if strmatch(self.kidney[3],'yes',/fold_case) then kstep = dm_optimizekidstep(kstep,kidran)
                    nkid = (kstep eq 0)?1:(floor(abs(kidran[1]-kidran[0])/kstep)+1)     
                    for k=0L,nkid-1L do begin
                        tmp_kid = (kidran[0])>(kidran[0]+kstep*k)<(kidran[1])
                        tmp = dm_check_macskidney(tmp_kid,tmp_ei,count,offset=self.kidney_offset,/spline)
                        if count eq 0 then continue
                        if count2 ne 0 then begin
                           da_tt1 = reform((det_tt-76+tmp_kid)#(fltarr(count2)+1.0),tmp_ndt*count2)
                           da_en   = [da_en,replicate(tmp_en,count2*tmp_ndt)]
                           da_ei   = [da_ei,replicate(tmp_ei,count2*tmp_ndt)]
                           da_tt   = [da_tt,da_tt1]
                           tmp_psi = [tmp_psi,reform((fltarr(tmp_ndt)+1.0)#(90-tmp_a3[ind2]),tmp_ndt*count2)] 
                        endif   
                        if count1 ne 0 then begin
                           tmp_ran = tmp_kid-76+det_tt[[0,tmp_ndt-1]]+[-0.5,0.5]
                           tmp_ind = where((tmp_dtt ge tmp_ran[0]) and (tmp_dtt le tmp_ran[1]),count)
                           if count eq 0 then continue else begin
                              da_en   = [da_en,replicate(tmp_en,count)]
                              da_ei   = [da_ei,replicate(tmp_ei,count)]
                              da_tt   = [da_tt,tmp_dtt[tmp_ind]]
                              tmp_psi = [tmp_psi,90-tmp_da3[tmp_ind]] 
                           endelse
                        endif    
                    endfor
                endfor
                n_da = n_elements(da_tt) & tmp_da3 = 0 & tmp_dtt = 0
                if n_da gt 1 then begin
                   da_ei = da_ei[1:n_da-1] & da_en = da_en[1:n_da-1] & da_tt = da_tt[1:n_da-1] & tmp_psi = tmp_psi[1:n_da-1]
                   if self.macsintn_typ ne 0 then begin
                      daQ = dm_proj_spec_samp(da_ei,da_ei,76,0,psi=tmp_psi,type=type,is_spe=self.psi_spe,kidney=da_tt)
                      da_en = fltarr(n_da)
                   endif else begin
                      daQ = dm_proj_spec_samp(da_ei,da_ei-da_en,76,0,psi=tmp_psi,type=type,is_spe=self.psi_spe,kidney=da_tt)
                   endelse
                endif
                da_ei = 0
             endif else begin
                ;figure out the detector angle range
                tmp_det_tt = det_tt
                tmp_detpsi = det_psi
                if self.psi_spe then begin
                   pos_ind = where((abs(tmp_detpsi) gt 90) and (abs(tmp_detpsi) lt 270),n_pos,complement=neg_ind,ncomplement=n_neg)       
                   if n_pos gt 0 then pos_max = max(tmp_det_tt[pos_ind],min=pos_min)
                   if n_neg gt 0 then neg_max = max(-tmp_det_tt[neg_ind],min=neg_min)
                endif else begin
                   pos_ind = where(tmp_det_tt gt 0,n_pos,complement=neg_ind,ncomplement=n_neg)
                   if n_pos gt 0 then pos_max = max(tmp_det_tt[pos_ind],min=pos_min)
                   if n_neg gt 0 then neg_max = max(tmp_det_tt[neg_ind],min=neg_min)
                endelse
                pos_ind = 0 & neg_ind = 0 & tmp_det_tt = 0 & tmp_detpsi = 0
                for i=0L,n_elements(psi)-1 do begin
                    da_tt = -(psi[i]+self.datr_psi)
                    while da_tt gt 180 do da_tt = da_tt-360
                    while da_tt lt -180 do da_tt = da_tt+360
                    if abs((abs(da_tt)-180)) le self.datr_width/2.0 then begin     ;blocking the incident beam
                       da_tt  = det_tt 
                       da_psi = det_psi
                    endif else begin
                       da_tt  = da_tt+(findgen(11)-5.)/10.*self.datr_width         ;dark angle range [-0.5,0.5]*self.datr_width
                       da_psi = fltarr(11)
                       tmp_ind = where(da_tt gt 180,tmp_count)
                       while tmp_count gt 0 do begin
                          da_tt[tmp_ind] = da_tt[tmp_ind]-360
                          tmp_ind = where(da_tt gt 180,tmp_count)
                       endwhile
                       tmp_ind = where(da_tt lt -180,tmp_count)
                       while tmp_count gt 0 do begin
                          da_tt[tmp_ind] = da_tt[tmp_ind]+360
                          tmp_ind = where(da_tt lt -180,tmp_count)
                       endwhile
                       tmp_ind1 = temporary(tmp_ind) ;destroy tmp_ind
                       if n_pos gt 0 then begin
                          tmp_ind1 = where((da_tt ge pos_min) and (da_tt le pos_max),n_da_tt1)
                          if n_da_tt1 gt 0 then tmp_ind = temporary(tmp_ind1)
                       endif
                       if n_neg gt 0 then begin
                          tmp_ind2 = where((da_tt ge neg_min) and (da_tt le neg_max),n_da_tt2)
                          if n_da_tt2 gt 0 then begin
                             if n_elements(tmp_ind) gt 0 then $
                                tmp_ind = [tmp_ind,temporary(tmp_ind2)] $
                             else $
                                tmp_ind = temporary(tmp_ind2)
                           endif
                       endif
                       if n_elements(tmp_ind) gt 0 then begin
                          da_tt  = da_tt[tmp_ind]
                          da_psi = da_psi[tmp_ind] 
                          if self.psi_spe then begin ;SPE type PSI 
                             ind_psi = where(da_tt gt 0,count_psi)
                             if count_psi gt 0 then da_psi[ind_psi] = 180.0
                             da_tt = abs(da_tt)
                          endif
                       endif else tmp_ind = temporary(da_tt)
                    endelse
                    if n_elements(da_tt) gt 0 then begin
                       daQ0 = dm_proj_spec_samp(eief,en,da_tt,da_psi,psi=psi[i],type=type,is_spe=self.psi_spe,diffetyp=(self.samp_typ eq 3),instrgeom=self.instrgeom)
                       daQ0 = reform(daQ0,3,n_elements(daQ0)/3,/overwrite)  
                       if n_elements(daQ) eq 0 then daQ = daQ0 else daQ=[[daQ],[daQ0]] 
                    endif      
                endfor
                tmp_det_tt = 0 & tmp_detpsi = 0
             endelse
             da_tt = 0 & da_psi = 0 & tmp_psi = 0
             if n_elements(daQ) ne 0 then daQV = dm_proj_view(daQ,U1,U2,U3,/no_copy,n_fold=n_fold,sr_type=self.fold_type-2,sr_startvec=self.symrot_startvec)
          endif else begin  ;mask dark angle
             for i=0LL,n_elements(psi)-1 do begin
                 da_tt = -(psi[i]+self.datr_psi)
                 while da_tt gt 180 do da_tt = da_tt-360
                 while da_tt lt -180 do da_tt = da_tt+360
                 if abs((abs(da_tt)-180)) le self.datr_width/2.0 then begin     ;blocking the incident beam
                    if self.instrname eq 'macs' then begin
                       qty[*,*,i] = !values.f_nan
                       err[*,*,i] = !values.f_nan
                    endif else begin
                       if self.samp_typ eq 2 then begin
                          qty[*,i] = !values.f_nan
                          err[*,i] = !values.f_nan
                       endif else begin
                          qty[*,*,i] = !values.f_nan
                          err[*,*,i] = !values.f_nan
                       endelse 
                    endelse
                 endif else begin
                    da_tt  = da_tt+self.datr_width*[-0.5,0.5]                   ;dark angle range [-0.5,0.5]*self.datr_width
                    if self.instrname eq 'macs' then begin
                       tmp_det = det_tt-76+kidney[i]  
                       ind = where(tmp_det ge da_tt[0] and tmp_det le da_tt[1],count)
                       if count ne 0 then begin                        
                          if macshistodata then begin
                             qty[[2*ind,2*ind+1],*,i] = !values.f_nan
                             err[[2*ind,2*ind+1],*,i] = !values.f_nan
                          endif else begin
                             qty[*,ind,i] = !values.f_nan
                             err[*,ind,i] = !values.f_nan
                          endelse
                       endif
                    endif else begin
                       ind = where(det_tt ge da_tt[0] and det_tt le da_tt[1],count)
                       if count ne 0 then begin
                          if self.samp_typ eq 2 then begin
                             qty[ind,i] = !values.f_nan
                             err[ind,i] = !values.f_nan
                          endif else begin
                             qty[*,ind,i] = !values.f_nan
                             err[*,ind,i] = !values.f_nan
                          endelse 
                       endif 
                    endelse
                 endelse
             endfor
          endelse 
       endif

       ;Al, Cu, or Stainless Steel powder line calculation
       if ptr_valid(self.powd_qvalue) and (min_en le elastic_tol) and (max_en ge -elastic_tol) then begin  ;must be elastic
          tmp_ei = eief
          if self.instrname eq 'macs' then begin
             if self.macsintn_typ eq 0 then begin       ;'SPEC' 
                maxQ = max(dm_proj_spec_samp(Ei,Ef,det_tt,det_psi,type=0,kidney=kidney),min=minQ) 
             endif else begin                           ;'DIFF' or 'SPEC+DIFF'
                maxQ = max(dm_proj_spec_samp(Ei,Ei,det_tt,det_psi,type=0,kidney=kidney),min=minQ)
                tmp_ei = Ei[uniq(Ei,sort(Ei))]
             endelse
          endif else begin
             maxQ = max(abs(sqrt(eief/2.0721246)*sin(det_tt/2*!dtor)*2.),min=minQ)
          endelse
          if self.powd_showall then begin
             tmppsi = findgen(361)*!dtor
          endif else begin
             psimax = max(psi,min=psimin) & npsi = 11>(round(psimax-psimin)+1)
             tmppsi = (psimin-0.5+findgen(npsi)/(npsi-1)*(psimax-psimin+1))*!dtor
          endelse
          tmpcospsi = cos(tmppsi)
          tmpsinpsi = sin(temporary(tmppsi))
          ;now figure out which detectors will intercept powder cones,use 0.5 degree as threshhold
          if self.instrname eq 'macs' then begin
             tmp_kid = kidney[uniq(kidney,sort(kidney))]
             n_kid = n_elements(tmp_kid) & n_det = n_elements(det_tt)
             tmp_det_tt = (fltarr(n_kid)+1)#det_tt
             tmp_detpsi = (fltarr(n_kid)+1)#det_psi
             for i=0L,n_det-1 do tmp_det_tt[*,i] = tmp_kid-76.0+det_tt[i]
             tmp_kid = 0
          endif else begin
             tmp_det_tt = det_tt
             tmp_detpsi = det_psi
          endelse
          for i=0L,n_elements(*self.powd_qvalue)-1 do begin
              if ((*self.powd_qvalue)[i] lt minQ) then continue
              if ((*self.powd_qvalue)[i] gt maxQ) then break
              for j=0L,n_elements(tmp_ei)-1 do begin
                  k = sqrt(tmp_ei[j]/2.07212)
                  tth = asin((*self.powd_qvalue)[i]/2./k)*2/!dtor
                  if self.powd_showall then begin
                     tmptth = tth*(2*(tmp_det_tt gt 0)-1)
                     tmpdetpsi = tmp_detpsi   
                  endif else begin
                     ind = where(abs(abs(tmp_det_tt)-tth) le 0.5,count)
                     if count eq 0 then continue
                     tmptth = tth*(2*(tmp_det_tt[ind] gt 0)-1)
                     tmpdetpsi = tmp_detpsi[ind]
                  endelse
                  tmp = dm_to_string(tmptth,resolution=3)+'_'+dm_to_string(tmpdetpsi,resolution=3)
                  ind = sort(tmp)
                  ind1 = uniq(tmp[ind])
                  tmptth = tmptth[ind[ind1]]
                  tmpdetpsi = tmpdetpsi[ind[ind1]]
                  for l=0L,n_elements(tmptth)-1 do begin
                      ;in psi=0 uvw frame
                      x = k*(1.0-cos(tmptth[l]*!dtor))
                      sin2th = sin(tmptth[l]*!dtor)
                      sinpsi = sin(tmpdetpsi[l]*!dtor)
                      cospsi = cos(tmpdetpsi[l]*!dtor)
                      if self.psi_spe then begin ;SPE type detector info
                         y = -k*sin2th*cospsi
                         z = -k*sin2th*sinpsi
                      endif else begin           ;DCS type detector info
                         if sin2th eq 0 then begin
                            y = 0.0
                            z = 0.0
                         endif else begin
                            y = k*sin2th*sqrt(1-(sinpsi/sin2th)^2)
                            z = -k*sinpsi
                         endelse
                      endelse
                      ;rotate to sample frame
                      xs = x*tmpcospsi+y*tmpsinpsi
                      ys = y*tmpcospsi-x*tmpsinpsi
                      zs = replicate(z,n_elements(xs))
                      if self.powd_showall then n_fold = 1
                      Vpow = dm_proj_view(transpose([[xs],[ys],[zs]]),U1,U2,U3,n_fold=n_fold,sr_type=self.fold_type-2,sr_startvec=self.symrot_startvec)
                      xs = reform(Vpow[0,*]) & ys = reform(Vpow[1,*]) & zs = reform(Vpow[2,*])
                      if n_elements(lines) ne 0 then lines=[[[lines]],[[[xs],[ys],[zs]]]] $
                      else lines=[[xs],[ys],[zs]]
                  endfor      
              endfor
          endfor
          tmp_det_tt = 0 & tmp_detpsi = 0 & tmpcospsi = 0 & tmpsinpsi = 0
          if n_elements(lines) ne 0 then self.powd_line = ptr_new(lines,/no_copy)
       endif
       
       ;check folding
       fold = [-1] & fcnt = [0.0] & fdir = [0]
       if self.view_u1fold and (self.fold_type lt 2) then begin
          nfc = n_elements(*(self.view_u1foldcenter))
          nfd = n_elements(*(self.view_u1folddir))
          index = where(finite(*(self.view_u1foldcenter)),count,complement=index1,ncomplement=count1)
          if self.fold_type eq 1 then begin
             if count gt 0 then begin
                *(self.view_u1foldcenter) = (*(self.view_u1foldcenter))[index[0]]
                if count gt 1 then begin
                   tmp = 'Only the first u1 folding center is used when folding to both sides.'
                   if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
                endif
             endif
             if (nfc gt 1) or (count1 gt 0) then reset_projection = 1b
          endif else begin
             if nfc ne nfd then begin
                *(self.view_u1folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u1foldcenter))+':',title='Set u1 folding direction',$
                       default =*(self.view_u1folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
               if ~keyword_set(debug) then begin
                  value = (['-','auto','+'])[(*self.view_u1folddir)[0]+1]
                  for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u1folddir)[i]+1]
                  dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU1Folddir'),select=0,set_value=[value,'auto','+']
               endif
             endif
             if count gt 0 then begin
                fold = [fold,intarr(count)]
                fcnt = [fcnt,(*(self.view_u1foldcenter))[index]]
                fdir = [fdir,(*(self.view_u1folddir))[index]]
             endif
             if count1 gt 0 then begin
                if count+count1 eq 1 then tmp = 'u1 folding center value is invalid.' $
                else tmp = 'u1 folding center value #'+dm_to_string(index1+1,separator=', ')+([' is',' are'])[count1 gt 1]+' invalid.'
                if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
             endif
          endelse
       endif
       if self.view_u2fold and (self.fold_type lt 2) then begin
          nfc = n_elements(*(self.view_u2foldcenter))
          nfd = n_elements(*(self.view_u2folddir))
          index = where(finite(*(self.view_u2foldcenter)),count,complement=index1,ncomplement=count1)
          if self.fold_type eq 1 then begin
             if count gt 0 then begin
                *(self.view_u2foldcenter) = (*(self.view_u2foldcenter))[index[0]]
                if count gt 1 then begin
                   tmp = 'Only the first u2 folding center is used when folding to both sides.'
                   if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
                endif
             endif
             if (nfc gt 1) or (count1 gt 0) then reset_projection = 1b
          endif else begin
             if nfc ne nfd then begin
                *(self.view_u2folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u2foldcenter))+':',title='Set u2 folding direction',$
                       default =*(self.view_u2folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
               if ~keyword_set(debug) then begin
                  value = (['-','auto','+'])[(*self.view_u2folddir)[0]+1]
                  for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u2folddir)[i]+1]
                  dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU2Folddir'),select=0,set_value=[value,'auto','+']
               endif
             endif
             if count gt 0 then begin
                fold = [fold,intarr(count)+1]
                fcnt = [fcnt,(*(self.view_u2foldcenter))[index]]
                fdir = [fdir,(*(self.view_u2folddir))[index]]
             endif
             if count1 gt 0 then begin
                if count+count1 eq 1 then tmp = 'u2 folding center value is invalid.' $
                else tmp = 'u2 folding center value #'+dm_to_string(index1+1,separator=', ')+([' is',' are'])[count1 gt 1]+' invalid.'
                if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
             endif
          endelse
       endif
       if extravaxis then begin
          if self.view_u3fold and (self.fold_type lt 2) then begin
             nfc = n_elements(*(self.view_u3foldcenter))
             nfd = n_elements(*(self.view_u3folddir))
             index = where(finite(*(self.view_u3foldcenter)),count,complement=index1,ncomplement=count1)
             if self.fold_type eq 1 then begin
                if count gt 0 then begin
                   *(self.view_u3foldcenter) = (*(self.view_u3foldcenter))[index[0]]
                   if count gt 1 then begin
                      tmp = 'Only the first u3 folding center is used when folding to both sides.'
                      if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
                   endif
                endif
                if (nfc gt 1) or (count1 gt 0) then reset_projection = 1b
             endif else begin
                if nfc ne nfd then begin
                   *(self.view_u3folddir) = (dm_dialog_input('center value '+dm_to_string(*(self.view_u3foldcenter))+':',title='Set u3 folding direction',$
                          default =*(self.view_u3folddir)+1,is_droplist=1+intarr(nfc),droplist_content=ptr_new(['-','auto','+']),xsize=100,dialog_parent=self.tlb,/return_number)-1)
                   if ~keyword_set(debug) then begin
                      value = (['-','auto','+'])[(*self.view_u3folddir)[0]+1]
                      for i=1,nfc-1 do value = value+','+(['-','auto','+'])[(*self.view_u3folddir)[i]+1]
                      dm_set_droplist,widget_info(self.projBas,find_by_uname='viewU3Folddir'),select=0,set_value=[value,'auto','+']
                   endif
                endif
                if count gt 0 then begin
                   fold = [fold,intarr(count)+2]
                   fcnt = [fcnt,(*(self.view_u3foldcenter))[index]]
                   fdir = [fdir,(*(self.view_u3folddir))[index]]
                endif
                if count1 gt 0 then begin
                   if count+count1 eq 1 then tmp = 'u3 folding center value is invalid.' $
                   else tmp = 'u3 folding center value #'+dm_to_string(index1+1,separator=', ')+([' is',' are'])[count1 gt 1]+' invalid.'
                   if n_elements(mesg) eq 0 then mesg = [tmp,''] else mesg = [mesg,tmp,'']
                endif
             endelse
          endif
       endif else begin
          QV = QV[0:1,*,*,*]
          if n_elements(daQV) ne 0 then daQV = daQV[0:1,*]
       endelse
       ;folding of viewing axis
       for i=1,n_elements(fold)-1 do begin
           xdat = QV[fold[i],*,*,*]    
           if n_elements(daQV) ne 0 then daxdat = daQV[fold[i],*]
           if ptr_valid(self.powd_line) and (~self.powd_showall) then pdxdat = (*self.powd_line)[*,fold[i],*]
           count = 0 & dacount = 0 & pdcount = 0
           case fdir[i] of 
                -1: begin    ;'-' side, unless no data to the '-' side
                    index = where(xdat gt fcnt[i],count,ncomplement=count1)
                    if n_elements(daQV) ne 0 then daindex = where(daxdat gt fcnt[i],dacount,ncomplement=dacount1)
                    if n_elements(pdxdat) ne 0 then pdindex = where(pdxdat gt fcnt[i],pdcount,ncomplement=pdcount1)
                    if count1 eq 0 then begin  ;no data to the '-' side
                       count = 0 & index = -1 & dacount = 0 & daindex = -1 & pdcount = 0 & pdindex = -1
                    endif
                    end
                 0: begin    ;auto
                    ;determine on which side of the folding center that the data has wider range
                    ;fold to greater range side   
                    index = where(xdat gt fcnt[i],count,complement=index1,ncomplement=count1)
                    if n_elements(daQV) ne 0 then daindex = where(daxdat gt fcnt[i],dacount,complement=daindex1,ncomplement=dacount1)
                    if n_elements(pdxdat) ne 0 then pdindex = where(pdxdat gt fcnt[i],pdcount,complement=pdindex1,ncomplement=pdcount1)
                    if (count ne 0) and (count1 ne 0) then begin
                       x_max = max(xdat[index],min=x_min)
                       p_ran = x_max-x_min
                       x_max = max(xdat[index1],min=x_min)
                       n_ran = x_max-x_min  
                       if p_ran ge n_ran then begin
                          index = temporary(index1) 
                          count = count1
                          if n_elements(daQV) ne 0 then begin
                             daindex = temporary(daindex1) 
                             dacount = dacount1
                          endif
                          if n_elements(pdxdat) ne 0 then begin
                             pdindex = temporary(pdindex1) 
                             pdcount = pdcount1
                          endif
                       endif else begin
                          index1 = -1 & daindex1 = -1 & pdindex1 = -1
                       endelse
                    endif else begin
                       tmp = temporary(p_ran) & tmp = temporary(n_ran) & count = 0
                    endelse
                    end
                 1: begin    ;'+' side,unless no data to the '+' side
                    index = where(xdat lt fcnt[i],count,ncomplement=count1)
                    if n_elements(daQV) ne 0 then daindex = where(daxdat lt fcnt[i],dacount,ncomplement=dacount1)
                    if n_elements(pdxdat) ne 0 then pdindex = where(pdxdat lt fcnt[i],pdcount,ncomplement=pdcount1)
                    if count1 eq 0 then begin  ;no data to the '+' side
                       count = 0 & index = -1 & dacount = 0 & daindex = -1 & pdcount = 0 & pdindex = -1
                    endif
                    end
              else:
           endcase
           if count ne 0 then begin
              xdat[index] = (2.*fcnt[i]-xdat[index]) & index = -1
              QV[fold[i],*,*,*] = temporary(xdat)
           endif
           if dacount ne 0 then begin
              daxdat[daindex] = (2.*fcnt[i]-daxdat[daindex]) & daindex = -1
              daQV[fold[i],*] = temporary(daxdat)
           endif
           if pdcount ne 0 then begin
              pdxdat[pdindex] = (2.*fcnt[i]-pdxdat[pdindex]) & pdindex = -1
              (*self.powd_line)[*,fold[i],*] = temporary(pdxdat)
           endif
       endfor
       end
    endcase

    if (self.intn_typ ge 4) and (self.intn_typ ne (n_elements(self.intn_typ_str)-1)) then begin
       if self.samp_typ gt 0 then Q = sqrt(total(Q^2,1))
       case self.intn_typ of 
            4: begin    ;GDOS
               if dwu2 eq 0 then factor = 0.48259649/Q^2 $
               else factor = 0.48259649/Q^2*exp(dwu2*Q^2)
               end
            5: begin    ;VDOS
               tthmax = max(det_tt,min=tthmin)
               if self.instrgeom eq 0 then efei = eief-en $
               else efei = eief+en
               root = sqrt(eief*efei)
               q4maxmin = (eief+efei-2.0*root*cos(tthmax*!dtor))^2-(eief+efei-2.0*root*cos(tthmin*!dtor))^2
               if dwu2 eq 0 then factor = (1./q4maxmin)#sin(det_tt*!dtor) $
               else factor = ((1./q4maxmin)#sin(det_tt*!dtor))*exp(dwu2*Q^2)
               end
            6: begin    ;J(Q,y)
               factor = 2.07212*1.008665*2/Mjqy*Q
               if self.samp_typ eq 0 then y = ((en#(fltarr(ndet)+1))-2.07212*1.008665/Mjqy*Q^2)/factor
               end
            else: factor = 1.0
       endcase
       if (n_elements(factor) ne 1) and (n_elements(factor) ne n_elements(qty)) and (n_elements(qty) ne 1) then begin  ;must be DCS and temperature or magnetic field axis enabled
          for i=0L,n_file-1L do begin
              qty[*,*,i] = qty[*,*,i]*factor
              err[*,*,i] = err[*,*,i]*factor
          endfor
       endif else begin
          qty = qty*factor
          err = err*factor
       endelse
       if n_elements(zero_error) ne 0 then zero_error = zero_error*min(factor[where(factor ne 0)])
       factor = 1.0
    endif

    ;user function executed after projection
    if (strlen(userf_str) ne 0) and (self.userf_before eq 0) then begin
       if self.userfwarning then begin
          if lmgr(/runtime) then usertmp = 'User Macro' else usertmp = 'User Macro/Function/Procedure'
          if n_elements(mesg) eq 0 then mesg = [usertmp+' is specified.',''] $
          else mesg = [usertmp+' is specified.','',mesg]
          self.userfwarning = 0b
       endif
       ok = execute(userf_str,1,1)
       if ~ok then begin
          errmsg = strmid(!error_state.msg,0,strlen(!error_state.msg)-1)+' in the user macro.'
          ok = dialog_message(errmsg,/error,dialog_parent=self.tlb,/center)
          return  ;error encountered
       endif
       if n_elements(err) ne n_elements(qty) then begin ;in case qty and err are not both defined in user functions
          if n_elements(err) eq 1 then begin
             err = replicate(err,size(qty,/dimension))
          endif else begin
             help,qty,err,output=tmp
             ok = dialog_message(['Incompatible array sizes of intensity and error bar, please check the user macro.','',tmp],/error,dialog_parent=self.tlb,/center)
             return
          endelse
       endif
    endif

    if self.instrname eq 'macs' then begin
       if ptr_valid(self.dataStrPtr) then begin
          if size(qty,/n_dimensions) eq 2 then begin
             qty = reform(qty,[size(qty,/dimensions),1],/overwrite)
             err = reform(err,[size(err,/dimensions),1],/overwrite)
          endif
          if macshistodata then begin
             qty = transpose(qty,[1,2,0])  ;[40,nchan,ndat]->[nchan,ndat,40]
             err = transpose(err,[1,2,0])
          endif else begin
             qty = transpose(qty,[0,2,1])  ;[2,20,ndat]->[2,ndat,20]
             err = transpose(err,[0,2,1])
          endelse
       endif
    endif
    
    if n_elements(zero_error) ne 0 then self.zero_error[2] = zero_error

    ;store projection
    if self.samp_typ eq 0 then begin        ;powder
       if n_elements(kidney) ne 0 then begin
          if self.intn_typ eq (n_elements(self.intn_typ_str)-2) then $  ;J(Q,y), not used yet
             self.projStrPtr = ptr_new({Q:temporary(Q),en:temporary(en),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),ewid:temporary(ewid),kidney:temporary(kidney),psi:temporary(psi),temperature:temporary(data_temperature),hfield:temporary(data_hfield),specify:temporary(data_specify),y:temporary(y),weight:temporary(weight),histo_weight:temporary(histo_weight)}) $
          else $
             self.projStrPtr = ptr_new({Q:temporary(Q),en:temporary(en),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),ewid:temporary(ewid),kidney:temporary(kidney),psi:temporary(psi),temperature:temporary(data_temperature),hfield:temporary(data_hfield),specify:temporary(data_specify),weight:temporary(weight),histo_weight:temporary(histo_weight)})
       endif else begin
          if self.intn_typ eq (n_elements(self.intn_typ_str)-2) then $  ;J(Q,y)
             self.projStrPtr = ptr_new({Q:temporary(Q),en:temporary(en),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),ewid:temporary(ewid),time:temporary(time),temperature:temporary(data_temperature),hfield:temporary(data_hfield),y:temporary(y),weight:temporary(weight)}) $
          else $
             self.projStrPtr = ptr_new({Q:temporary(Q),en:temporary(en),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),ewid:temporary(ewid),time:temporary(time),temperature:temporary(data_temperature),hfield:temporary(data_hfield),weight:temporary(weight)})
       endelse
    endif else begin                        ;single crystal    
       if n_elements(daQV) ne 0 then begin
          if n_elements(kidney) ne 0 then begin
             self.projStrPtr = ptr_new({V:temporary(QV),en:temporary(en),ewid:temporary(ewid),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),kidney:temporary(kidney),psi:temporary(psi),temperature:temporary(data_temperature),hfield:temporary(data_hfield),specify:temporary(data_specify),daQV:temporary(daQV),daen:temporary(da_en),weight:temporary(weight),histo_weight:temporary(histo_weight)})
          endif else begin
             self.projStrPtr = ptr_new({V:temporary(QV),en:temporary(en),ewid:temporary(ewid),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),temperature:temporary(data_temperature),daQV:temporary(daQV),weight:temporary(weight)})
          endelse
       endif else begin
          if n_elements(kidney) ne 0 then begin
             self.projStrPtr = ptr_new({V:temporary(QV),en:temporary(en),ewid:temporary(ewid),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),kidney:temporary(kidney),psi:temporary(psi),temperature:temporary(data_temperature),hfield:temporary(data_hfield),specify:temporary(data_specify),weight:temporary(weight),histo_weight:temporary(histo_weight)})
          endif else begin
             self.projStrPtr = ptr_new({V:temporary(QV),en:temporary(en),ewid:temporary(ewid),two_theta:temporary(det_tt),qty:temporary(qty),err:temporary(err),temperature:temporary(data_temperature),hfield:temporary(data_hfield),weight:temporary(weight)})
          endelse
       endelse
    endelse

    ;allow plots
    if (~ keyword_set(debug)) then $
       self->set_display_button  $
    else $
       print,'**Projection calculation finished in ',systime(/sec)-current,' sec.'
    if ptr_valid(self.dataStrPtr) and (self.bin_avgsum eq 1) and (self.e_bin[self.samp_typ] gt 1) and (self.samp_typ lt 2)then begin
       if n_elements(mesg) eq 0 then mesg='The current binning method is SUM.' $
       else mesg = [mesg,'The current binning method is SUM.']
    endif   
    if n_elements(mesg) ne 0 then ok = dialog_message([mesg,'','This message is only a reminder.'],dialog_parent=self.tlb,/center)
    if keyword_set(reset_projection) then self->reset_projection,/reset,/foldonly
    self.error = 0       ;clear the error flag
end