; $Id: dm_calc_deteff.pro,v 1.10 2014/05/20 14:21:03 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_calc_deteff
;
; PURPOSE:
;   calculate detector efficiency
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  December, 2020
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################


;this function calculates the intensity ratio of two macs detectors
;parameters:
;    a4_1:      [na4_1]
;    a4_2:      [na4_2]
;    int_1:     [2,na4_1]
;    int_2:     [2,na4_2]
;keywords:
;   ignore_a4:  [2], the A4 range to be ignored in the calculation
;   intavg:     if set, average the intensity first, then calculate the ratio, otherwise calculate the average of the ratio
function dm_macs_int_ratio,a4_1,a4_2,int_1,int_2,ignore_a4=ignore_a4,intavg=intavg
    if n_elements(ignore_a4) eq 2 then begin       
       indx = where((a4_1 lt ignore_a4[0]) or (a4_1 gt ignore_a4[1]),cnt)
       if cnt eq 0 then return,[!values.f_nan,!values.f_nan] $
       else begin
          a4_1  = a4_1[indx]
          int_1 = int_1[*,indx] 
       endelse
       indx = where((a4_2 lt ignore_a4[0]) or (a4_2 gt ignore_a4[1]),cnt)
       if cnt eq 0 then return,[!values.f_nan,!values.f_nan] $
       else begin
          a4_2  = a4_2[indx]
          int_2 = int_2[*,indx] 
       endelse
    endif
    range = [min(a4_2),max(a4_1)]
    indx0 = where((a4_1 ge range[0]) and (a4_1 le range[1]),cnt0) 
    indx1 = where((a4_2 ge range[0]) and (a4_2 le range[1]),cnt1)
    cnt   = (cnt0<cnt1)
    if cnt eq 0 then return,[!values.f_nan,!values.f_nan]
    if keyword_set(intavg) then begin
       int1  = total(int_1[*,indx0],2)/cnt0
       int2  = total(int_2[*,indx1],2)/cnt1
       ratio = int1/int2
    endif else $
       ratio = int_1[*,indx0[0:cnt-1]]/int_2[*,indx1[0:cnt-1]]
    if n_elements(ratio) gt 2 then ratio = total(ratio,2)/n_elements(ratio[0,*])
    return,ratio
end

; parameter:
;   qty:        [ne,ndet], value will be changed if maskdet exists, for MACS [2, ndet, ne]
; keyword:
;   kidney:     if present, this is for MACS file, maskdet will be the a4 range to be ignored
;   baddet_id:  for macs, the baddet_id are detector numbers that will be bypassed in the efficiency calculation, return !values.f_nan for those detectors
;   fit:        if set, fit by an gaussian and use the data in the gaussian fucntion range
;   maskdet:    detector indices to be masked, those detectors will be given a value of 1.0; for macs, this is the a4 range to be ignored
;   intavg:     for macs, if set, average the intensity first, then calculate the ratio, otherwise calculate the average of the intensity ratios
;   macssym:    for macs, if set, symmetrize the intensity
function dm_calc_deteff,qty,baddet_id=baddet_id,fit=fit,kidney=kidney,intavg=intavg,macssym=macssym,maskdet=maskdet,debug=debug
    nkid = n_elements(kidney)
    if nkid ne 0 then begin
       nbd    = n_elements(baddet_id)
       ndet   = (size(qty,/dim))[1]
       ;first sort the kidney angles
       index  = sort(kidney)
       tmpkid = kidney[index]
       tmpqty = qty[*,*,index]
       a4     = fltarr(ndet,nkid)
       for i=0,(ndet-1) do a4[i,*] = tmpkid-76.0+8.0*i
       eff = fltarr(2,ndet)+1.0 & eff[*,1:(ndet-1)] = !values.f_nan
       if nbd ne 0 then begin       ;replace the bad detector with the nearest good detector intensity
          tmp = bytarr(ndet)+1b
          tmp[baddet_id] = 0
          gooddet_id = where(tmp)   ;assuming not all detectors are bad
          for i=0,nbd-1 do begin
              tmp = min(abs(gooddet_id-baddet_id[i]),index)
              tmpqty[*,*,baddet_id[i]] = tmpqty[*,*,gooddet_id[index]]
          endfor
       endif
       for i=1,(ndet-1) do begin
           ratio = dm_macs_int_ratio(reform(a4[i-1,*]),reform(a4[i,*]),reform(tmpqty[*,i-1,*]),reform(tmpqty[*,i,*]),ignore_a4=maskdet,intavg=intavg)
           if total(finite(ratio)) eq 2 then begin
              eff[*,i] = ratio
              for j=0,1 do tmpqty[j,i,*] = tmpqty[j,i,*]*eff[j,i]
           endif
       endfor
       if keyword_set(macssym) then begin
          ;check if there's a break in matching det channels
          nind = where(finite(eff[0,*],/nan),nnan)
          if nnan ne 0 then begin   ;yes there's a break, apply a single factor to one side
             lind  = where(indgen(ndet) lt min(nind),nl)
             uind  = where(indgen(ndet) gt max(nind),nu)
             ;figure out the a4 range to be compared
             maxa4 = min([max(a4[uind,*]),abs(min(a4[lind,*]))]) 
             llind = where((a4 ge -maxa4) and (a4 le -maxa4+20),nll)
             uuind = where((a4 ge maxa4-20) and (a4 le maxa4),nuu)
             if (nll ne 0) and (nuu ne 0) then begin
                tmpqty = reform(tmpqty,2,n_elements(tmpqty[0,*,*]),/overwrite)
                tmp0   = total(tmpqty[*,llind],2)/nll
                tmp1   = total(tmpqty[*,uuind],2)/nuu
                ratio  = tmp0/tmp1
                for i=0,nu-1 do eff[*,uind[i]] = eff[*,uind[i]]*ratio
             endif
          endif else begin          ;no break, apply a sloping factor
             maxa4 = min([abs(min(a4[0,*])),max(a4[ndet-1,*])])
             lind  = (where((a4 ge -maxa4) and (a4 le -maxa4+20),nll) mod ndet)
             uind  = (where((a4 ge maxa4-20) and (a4 le maxa4),nuu) mod ndet)
             if (nll ne 0) and (nuu ne 0) then begin
                lind  = lind[uniq(lind,sort(lind))]
                uind  = uind[uniq(uind,sort(uind))]
                tmp0  = total(tmpqty[*,lind,*],2,/double) & tmp0 = total(tmp0,2)/n_elements(tmpqty[0,lind,*])
                tmp1  = total(tmpqty[*,uind,*],2,/double) & tmp1 = total(tmp1,2)/n_elements(tmpqty[0,uind,*])
                slope = (tmp1-tmp0)/(mean(uind)-mean(lind))
                inter = tmp0-slope*mean(lind)
                averg = inter+slope*mean(indgen(ndet))           
                for i=0,(ndet-1) do eff[*,i] = eff[*,i]*averg/(inter+slope*i)
             endif
          endelse
       endif
       if nbd ne 0 then eff[*,baddet_id] = !values.f_nan  
       for i=0,1 do eff[i,*] = eff[i,*]*total(finite(eff[i,*]))/total(eff[i,*],/nan) 
    endif else begin
       if n_elements(maskdet) ne 0 then qty[*,maskdet] = 0.0
       if keyword_set(fit) then begin
          ;first fit the det-sum and find out the elastic range
          y = total(qty,2,/double)
          dm_gaussfit,y,ny=ny,params=params,fitnotgood=fitnotgood,debug=debug
          if fitnotgood then begin
             eff  = total(qty,1,/double)
          endif else begin
             tmp  = shift(qty,ny/2-params[1],0)
             istart = (0>(ny/2-fix(2.5*params[2])))
             iend = ((ny-1)<(ny/2+fix(2.5*params[2])))
             tmp  = tmp[istart:iend,*] 
             eff  = total(tmp,1,/double)
          endelse
       endif else begin
          eff = total(qty,1,/double)
       endelse
       index0 = where(eff eq 0,count,complement=index1)
       if count ne 0 then begin
          minv = min(eff[index1])
          eff[index0] = minv
       endif
       eff = float(eff/mean(eff,/double))
       if n_elements(maskdet) ne 0 then eff[maskdet] = 1.0
    endelse
    return, eff
end