; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEreadDAVE1
;
; PURPOSE:
;   File reader for DAVE 1.x data files
;
; CATEGORY:
;   Input Output
;
; SUPERCLASSES:
;   IDLitReader
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEreadDAVE1::Init
;
; METHODS:
;   GetData
;   IsA
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; July 2004
;-
;###############################################################################


;===============================================================================
; DAVEreadDAVE1::GetData
; 
; PURPOSE:
;   Responsible for actually reading the data from the file
;
; PARAMETERS:
;   oData [out] - An IDLitData (or subclass) object which will contain
;                 the data read from the file.
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Successful Read
;    0 - Failure reading from the file.
function DAVEreadDAVE1::GetData, oData
compile_opt idl2

catch, iErr
if(iErr ne 0)then begin
    catch, /cancel
    goto, ioerr  ;; do any cleanup needed
endif
on_ioerror, ioerr

oTool = self->GetTool()
if (not OBJ_VALID(oTool)) then $
  return, 0


filename = self->GetFilename()

oData = obj_new('DAVEDataset',filename=filename)

return, obj_valid(oData)

;; use IDL restore to retrieve info from file
;restore,filename, /relaxed_structure_assignment
;
;; No davePtr? Can't continue
;if (~ptr_valid(davePtr) || (n_elements(*davePtr) eq 0)) then return, 0
;    
;; Convert the davePtr into an object hierarchy
;oData = self->DavePtrToObject(davePtr,filename)
;if (~obj_valid(oData)) then begin 
;    heap_free, davePtr
;    return, 0
;endif
;
;heap_free, davePtr
;;help,oData
;return, 1


ioerr: ;; IO Error handler
if(n_elements(unit) gt 0)then begin
    sStat = fStat(unit)
    if(sStat.open ne 0)then $
      free_lun, unit
endif
self->SignalError, !error_state.msg, severity=2
return, 0

end


;===============================================================================
; DAVEreadDAVE1::DavePtrToObject
; 
; PURPOSE:
;   Function for converting the data in a davePtr structure
;   into a object hierarchy consisting of IDLitDataContainer and/or
;   IDLitData ojects (or subclasses of these).
;
; PARAMETERS
;   davePtr - The davePtr structure obtained from a DAVE 1.x file.
;
;   filename - The filename from which the davePtr was restored from.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an object hierarchy representation of the data.
;    If unsuccessful, a null object.
;
;function DAVEreadDAVE1::DavePtrToObject, davePtr, filein
;compile_opt idl2
;
;; protect input argument from local changes
;filename = filein
;
;; both the davePtr and filename arguments are required
;if (n_params() ne 2) then begin
;    errorMsg = 'DAVEreadDAVE1::DavePtrToObject: Missing parameters'
;    self->SignalError, errorMsg, severity=2
;    return, obj_new()
;end 
;
;dims = (size((*(*(*davePtr).dataStrPtr).commonStr.histPtr).qty))[0]
;
;if (dims ne 1 && dims ne 2) then begin
;    errorMsg = 'Empty data block'
;    self->SignalError, errorMsg, severity=2
;    return, obj_new()
;endif
;
;; strip dir path and extension from the filename
;description = filename
;filename = file_basename(filename,'.dave',/fold_case)
;
;; iTools does not handle '_[0-9]' characters in filenames well eg _0, _1, etc
;; replace all '_' with '-'
;while ((pos = stregex(filename,'_[0-9]')) ne -1) do begin
;    strput, filename,'-',pos
;endwhile
;
;; Create a data container using filename as its name
;oData = obj_new('IDLitDataContainer', name=filename $
;                ,description=description,identifier='id'+filename $
;                ,type='DAVE1DATASET')
;
;; The instrument name and treatment history
;childData = (*(*davePtr).dataStrPtr).commonStr.instrument
;child = obj_new('IDLitData',childData,name='Instrument',type='DAVE1INST'$ 
;                ,description='Instrument Name')
;if (obj_valid(child)) then oData->add, child
;
;childData = (*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr)
;child = obj_new('IDLitData',childData,name='Treatment History',type='DAVE1HISTORY' $
;                ,description='Treatment history')
;if (obj_valid(child)) then oData->add, child
;
;
;;_________________________________________________
;; Deal with the descriPtr
;void = where(tag_names(*davePtr) eq 'DESCRIPTR', cnt)
;if (cnt eq 1) then begin 
;    ;; A descriPtr entry is present.
;    ;; If it is valid, create an entry for it
;    if (ptr_valid((*davePtr).descriPtr) && $
;        (n_elements(*(*davePtr).descriPtr) gt 0)) then begin
;        
;        descriPtrStr = (*(*davePtr).descriPtr)
;        ;; Create a container for the descriPtr and fill it with the
;        ;; descriPtr contents
;        oDescriPtr = obj_new('IDLitParameterSet', name='Sample Information Tag' $
;                             ,description='Sample information tag (descriPtr)'  $
;                             ,type='DESCRIPTR')
;        oQty = obj_new('IDLitData',descriPtrStr.qty,name=descriPtrStr.name,type=partype(descriPtrStr.qty),uvalue=0.0)
;        oErr = obj_new('IDLitData',descriPtrStr.err,name='Error in '+descriPtrStr.name,type=partype(descriPtrStr.err))
;        oUnits = obj_new('IDLitData',descriPtrStr.units,name='Units',type=partype(descriPtrStr.units))
;        oLegend = obj_new('IDLitData',descriPtrStr.legend,name='Description',type=partype(descriPtrStr.legend))
;        parameter_names = ['Description','Value','Error','Units']
;        oDescriPtr->add, [oLegend,oQty,oErr,oUnits],parameter_name=parameter_names
;        oData->add, oDescriPtr 
;    endif
;endif
;
;;_________________________________________________
;; Deal with commonStr
;
;; Create a container for the data (x,y,z axes)
;icon = (dims eq 1)? 'plot' : 'surface'
;oCommonCon = obj_new('IDLitParameterSet',name='Experimental Data' $
;                     ,description='Experimental Data Section',type='DAVE1COMMONSTR',icon=icon)
;oCommonCon->AddMetaData,'DatasetName',filename ; give it same name as the filename
;oData->Add, oCommonCon
;
;; First Independent Axis
;; Store in IDLitDataDave object and include additional metadata
;desc = (*(*davePtr).dataStrPtr).commonStr.xlabel
;xname = (strtrim(desc) eq '')? 'X Axis' : desc
;xAxisType = '0'
;data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).x
;dist = (strcmp((*(*davePtr).dataStrPtr).commonStr.xtype,'HISTOGRAM',4,/FOLD_CASE) gt 0)? $
;       'HISTOGRAM' : 'POINTS'
;oX = obj_new('IDLitDataDave',data,name=xname,type=partype(data),description=desc,axisType=xAxisType)
;oX->AddMetaData,'Long_name',desc
;oX->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.xunits
;oX->AddMetaData,'Distribution',dist
;
;oCommonCon->Add, oX, parameter_name=xAxisType
;
;if (dims eq 2) then begin
;    ;; Second Independent Axis
;    ;; Store in IDLitDataDave object and include additional metadata
;    desc = (*(*davePtr).dataStrPtr).commonStr.ylabel
;    yname = (strtrim(desc) eq '')? 'Y Axis' : desc
;    yAxisType = '1'
;    data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).y
;    dist = (strcmp((*(*davePtr).dataStrPtr).commonStr.ytype,'HISTOGRAM',4,/FOLD_CASE) gt 0)? $
;           'HISTOGRAM' : 'POINTS'
;    oY = obj_new('IDLitDataDave',data,name=yname,type=partype(data),description=desc,axisType=yAxisType)
;    oY->AddMetaData,'Long_name',desc
;    oY->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.yunits
;    oY->AddMetaData,'Distribution',dist
;    
;    oCommonCon->Add, oY, parameter_name=yAxistype
;endif
;
;; The data
;; Store in IDLitDataDave object using a suitable type and additional metadata.
;desc = (*(*davePtr).dataStrPtr).commonStr.histlabel
;zname = (strtrim(desc) eq '')? 'Data' : desc
;zAxisType = '2'
;axes = (dims eq 1)? xname : [xname,yname]
;data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).qty
;
;oZ = obj_new('IDLitDataDave',data,name=zname,type=partype(data),description=desc,axisType=zAxisType)
;oZ->AddMetaData,'Signal',1
;oZ->AddMetaData,'Axes',axes
;oZ->AddMetaData,'Long_name',desc
;oZ->AddMetaData,'Units',(*(*davePtr).dataStrPtr).commonStr.histunits
;
;oCommonCon->Add, oZ, parameter_name=zAxisType
;
;; The data error
;; Store in an IDLitDataDave object
;ename = 'Error'
;eAxisType = '3'
;desc = 'Uncertainty in '+zname
;data = (*(*(*davePtr).dataStrPtr).commonStr.histPtr).err
;oErr = obj_new('IDLitDataDave',data,name=ename,type=partype(data),description=desc,axisType=eAxisType)
;oCommonCon->Add, oErr, parameter_name=eAxisType
;
;
;;_________________________________________________
;; Deal with specficPtr
;specificPtr = (*(*davePtr).dataStrPtr).specificPtr
;if (ptr_valid(specificPtr) && $
;    (n_elements(specificPtr) gt 0)) then begin
;    
;    void = self->ReadSpecific(specificPtr,oData,name='Miscellaneous' $
;                          ,description='Instrument Specific Details' $
;                          ,type='DAVE1SPECIFICPTR')
;endif
;
;return, oData
;
;end



;===============================================================================
; DAVEreadDAVE1::ReadSpecific
; 
; PURPOSE:
;   Convert the contents of the Specific pointer into an object hierarchy.
;
; PARAMETERS
;   dataStr - The data structure to be converted
;
;   Container - The parent container in which object(s) of the dataStr
;               are to be placed.
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;
;function DAVEreadDAVE1::ReadSpecific, dataStr, parent, type=type, _EXTRA=etc
;compile_opt idl2
;
;switch size(dataStr,/tname) of
;    'UNDEFINED':
;    'OBJREF': break
;    
;    'POINTER': begin
;        if ptr_valid(dataStr) then begin
;            if (~keyword_set(type)) then type='POINTER'
;            void = self->ReadSpecific(*dataStr,parent,type=type,_EXTRA=etc) ; ,type='POINTER'
;        endif
;        break
;    end
;    
;    'STRUCT': begin
;        tags = tag_names(dataStr)
;        ntags = n_tags(dataStr)
;        if (ntags le 0) then break
;
;        if (~keyword_set(type)) then type='STRUCTURE'
;        child = obj_new('IDLitDataContainer',type=type, _EXTRA=etc) ;,type='STRUCTURE'
;        if (~obj_valid(child)) then break
;        
;        parent->add, child
;        
;        for i=0,ntags-1 do $
;          void = self->ReadSpecific(dataStr.(i),child,name=tags[i] $
;                                ,description=tags[i]) ; ,type=vartype(dataStr.(i))
;        
;        break
;    end
;    
;    else: begin
;        if (n_elements(dataStr) gt 0) then begin
;            axisType = 4   ; 'UNDEFINED' by default
;            extraType = ''   ; unspecified
;            if (~keyword_set(type)) then type=partype(dataStr)
;            if (strcmp(type,'POINTER')) then begin
;               extraType=type   ; store as extraType property
;               type=partype(dataStr)   ; and retrieve the proper type of the data stored in the heap
;            endif
;            void = where(size(dataStr,/type) eq [1,2,3,4,5,12,13,14,15],validNumericType)   ; what is the intrinsic data type?
;            void = where(size(dataStr,/n_dimensions) eq [1,2],oneORtwoDims)   ; is it a vector/array?
;            if (validNumericType && oneORtwoDims) then axisType=2   ; set as dependent data vector/array suitable for plotting
;
;            child = obj_new('IDLitDataDave',dataStr,type=type, extraType=extraType, axisType=axisType,_EXTRA=etc)
;            if (obj_valid(child)) then parent->add, child
;;            if (~keyword_set(type)) then type=partype(dataStr)
;;            child = obj_new('IDLitData',dataStr,type=type,_EXTRA=etc)
;;            if (obj_valid(child)) then parent->add, child
;        endif
;        break
;    end
;
;endswitch
;
;return, 1
;end


;===============================================================================
; DAVEreadDAVE1::IsA
; 
; PURPOSE:
;   Check the given file to see whether it is the proper type for this
;   reader.
;
; PARAMETERS
;   strFilename - The file to check
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - strFilename is a proper file type
;    0 - strFilename is cannot be handled by this reader
;
function DAVEreadDAVE1::IsA, strFilename
compile_opt idl2

; Basic error handler
catch, iErr
if(iErr ne 0)then begin
    catch, /cancel
    return, 0
endif

; Call the base class method to check for proper extension for this reader
if (~self->IDLitReader::IsA(strFilename)) then return, 0

; Check file contents to make sure this is a valid DAVE 1.x file.
; Using the IDL_Savefile class avoids the need to restore the file
oSaveFile = obj_new('IDL_Savefile', strFilename) ; if not a sav file error handler will trigger!
varNames = oSaveFile->Names()   ; retrieve variable names found in sav file
obj_destroy, oSaveFile

; A valid DAVE file must contain a variable called davePtr
res = where(varNames eq 'DAVEPTR', count)
if (count eq 0) then return, 0

return, 1
end

;===============================================================================
; DAVEreadDAVE1::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEreadDAVE1 class
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function DAVEreadDAVE1::Init, _REF_EXTRA=etc
compile_opt idl2

; Init superclass
if (self->IDLitReader::Init('dave'  $            ; <-- acceptable file extension(s)
                            ,NAME='DAVE 1.x' $
                            ,DESCRIPTION="DAVE 1.x data file (dave)" $
                            ,_EXTRA=etc) eq 0) then return, 0

; call setproperty method, if necessary.
if (n_elements(etc) gt 0) then $
  self->SetProperty, _EXTRA=etc

return, 1

end




;===============================================================================
pro DAVEreadDAVE1__Define
compile_opt idl2

void = {DAVEreadDAVE1, inherits IDLitReader}

end
