; $Id$
;###############################################################################
;
; NAME:
;  widget_rebin
;
; PURPOSE:
;   Perform a numerical convolution between two 1D datasets.
;   Based on convol_as_approx written by Rob Dimeo.
;
; CATEGORY:
;   DAVE Data Analysis
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   July, 2006
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;===============================================================================
; widget_rebin
; 
pro widget_rebin_event, event
compile_opt idl2
  
;begin error handler------------------------------------------------------------
; RTA - handler to catch misc. errors and provide a graceful exit so
;       that main app does not crash. Remember to switch off debug flag when debugging!
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'widget_rebin_event - Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif
;end error handler-------------------------------------------------------------

widget_control, event.top, get_uvalue = sPtr

case event.id of
   (*sPtr).wCan:  begin
      (*sPtr).bins.cancel = 1
      widget_control, event.top, /destroy
   end
   
   (*sPtr).wOK: begin
      widget_control, (*sPtr).wMin, get_value = xmin
      (*sPtr).bins.min = xmin
      widget_control, (*sPtr).wMax, get_value = xmax
      (*sPtr).bins.max = xmax
      widget_control, (*sPtr).wBin, get_value = binw
      (*sPtr).bins.binw = binw
      (*sPtr).bins.cancel = 0
      
      widget_control,event.top,/destroy
   end
   
else:
   
endcase

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;===============================================================================
; widget_rebin
;
; PURPOSE:
;   Present simple dialog to retrieve bin settings from a user.
;
; PARAMETERS:
;   xin [in] - The independent variable.
;
; KEYWORDS:
;   group_leader [in] - widget identifier supplied by the calling
;                      program to serve as group_leader
;
;   title [in] - The title to be used for the base widget
;
; RETURN VALUE:
;   - A structure ({xmin:xmin,xmax:xmax,xbin:dxmax,cancel:1})
;     specifying the bin settings and whether the cancel button was
;     pressed when exiting the dialog.
function widget_rebin, xin, group_leader = group_leader, title=title
compile_opt idl2

gl = (n_elements(group_leader) eq 0)? 0L : group_leader

if (n_elements(xin) eq 0) then begin
   msg = 'The independent axis data must be supplied'
   void = dialog_message(/error,msg,dialog_parent=gl)
   return, 0
endif

if ((size(xin))[0] ne 1) then begin
   msg = 'The independent axis data must be a vector'
   void = dialog_message(/error,msg,dialog_parent=gl)
   return, 0
endif

if (n_elements(title) eq 0) then title='Bin Settings'

if n_elements(group_leader) eq 0 then begin
   modal = 0
   wTLB = widget_base(/col,title =title,/tlb_frame_attr)
endif else begin
   modal = 1
   wTLB = widget_base(group_leader=group_leader,/col,/modal,title=title,/tlb_frame_attr)
endelse

wCB0 = widget_base(wTLB,/row)

x = float(xin)                  ; don't modify input!
nx = n_elements(x)
xmin = min(x,max=xmax)
dx = x[1:nx-1] - x[0:nx-2]
dxmin = min(dx, max=dxmax)

wCB1 = widget_base(wCB0,/col,/frame)

void = widget_label(wCB1,value = 'Current Settings')
void = cw_field(wCB1,/row,/float,value=xmin,title =   'Lower Limit  ',/noedit)
void = cw_field(wCB1,/row,/float,value=xmax,title =  'Upper Limit  ',/noedit)
void = cw_field(wCB1,/row,/float,value=dxmin,title = 'Min Bin Width',/noedit)
void = cw_field(wCB1,/row,/float,value=dxmax,title = 'Max Bin Width',/noedit)
void = cw_field(wCB1,/row,/integer,value=nx,title =  'Nos of Bins  ',/noedit)

wCB2 = widget_base(wCB0,/col,/frame)
void = widget_label(wCB2,value = 'Desired Settings')
;wMin = cw_field(wCB2,/row,/float,value =xmin,title =  'Minimum  ')
;wMax = cw_field(wCB2,/row,/float,value =xmax,title =  'Maximum  ')
;wBin = cw_field(wCB2,/row,/float,value =dxmax,title = 'Bin Width')
wMin = Fsc_field(wCB2,title='Minimum  ',value=Float(xmin),uname='WMIN' $
  ,/highlight,/cr_only,event_pro='widget_rebin_event')
wMax = Fsc_field(wCB2,title='Maximum  ',value=Float(xmax),uname='WMAX' $
  ,/highlight,/cr_only,event_pro='widget_rebin_event')
wBin = Fsc_field(wCB2,title='Bin Width',value=Float(dxmax),uname='WBIN' $
  ,/highlight,/cr_only,event_pro='widget_rebin_event')
;  prec = 3
;  wMin = Fsc_field(wCB2,title='Minimum  ',decimal=prec,value=Float(xmin),uname='WMIN' $
;    ,/highlight,/cr_only,event_pro='widget_rebin_event')
;  wMax = Fsc_field(wCB2,title='Maximum  ',decimal=prec,value=Float(xmax),uname='WMAX' $
;    ,/highlight,/cr_only,event_pro='widget_rebin_event')
;  wBin = Fsc_field(wCB2,title='Bin Width',decimal=prec,value=Float(dxmax),uname='WBIN' $
;    ,/highlight,/cr_only,event_pro='widget_rebin_event')

wOK = widget_button(wTLB,value = 'Accept')
wCan = widget_button(wTLB,value = 'Cancel')

bins = {min:float(xmin),max:float(xmax),binw:float(dxmax),cancel:1}
sPtr = ptr_new({bins:bins,wMin:wMin,wMax:wMax, wBin:wBin,wCan:wCan,wOK:wOK})

widget_control, wTLB, /realize, set_uvalue = sPtr
xmanager,'widget_rebin',wTLB,no_block=modal

retVal = (*sPtr).bins

ptr_free,sPtr

return,retVal

end
