; docformat = 'rst'

;+
; Common functionality for HDF 5 classes.
;
; :Properties:
;   parent
;     parent MGffH5 object
;   identifier
;     HDF 5 identifier for object
;   name
;     name of the object
;   fullname
;     full name of the object (including path)
;-


;= property access

;+
; Get properties.
;-
pro mgffh5base::getProperty, parent=parent, identifier=identifier, $
  name=name, fullname=fullname
  compile_opt strictarr

  if (Arg_present(parent)) then parent = self.parent
  if (Arg_present(identifier)) then identifier = self.id
  if (Arg_present(name)) then name = self.name
  if (Arg_present(fullname)) then begin
    if (Obj_valid(self.parent)) then begin
      self.parent->Getproperty, fullname=fullname
      fullname += '/' + self.name
    endif else fullname = self.name
  endif
end


;= overload operator methods

;+
; `HELP` overload common routine.
;
; :Returns:
;   string
;
; :Params:
;   varname : in, required, type=string
;     name of variable to provide HELP for
;
; :Keywords:
;   type : in, optional, type=string, default='NC_BASE'
;     type for object
;   specs : in, optional, type=string, default='<undefined>'
;     specs for object, depending on object type
;-
function mgffh5base::_overloadHelp, varname, type=type, specs=specs
  compile_opt strictarr

  _type = N_elements(type) eq 0L ? 'H5_BASE' : type
  _specs = N_elements(specs) eq 0L ? '<undefined>' : specs

  Return, String(varname, _type, _specs, format='(%"%-15s %-9s = %s")')
end


;= lifecycle methods

;+
; Creates HDF 5 object.
;
; :Returns:
;   1 for success, 0 for failure
;-
function mgffh5base::init, parent=parent, identifier=identifier, name=name
  compile_opt strictarr
  On_error, 2

  self.parent = N_elements(parent) eq 0L ? Obj_new() : parent
  self.id = N_elements(identifier) eq 0L ? '' : identifier
  self.name = N_elements(name) eq 0L ? '' : name

  Return, 1
end


;+
; Define instance variables and class inheritance.
;
; :Fields:
;   parent
;     parent `MGffH5` object
;   id
;     HDF 5 identifier for object
;   name
;     name of the object
;-
pro mgffh5base__define
  compile_opt strictarr

  define = { Mgffh5base, inherits Idl_object, $
    parent: Obj_new(), $
    id: 0L, $
    name: '' $
  }
end