; $Id$
;###############################################################################
;+
; NAME:
;   Quadrature
;
; PURPOSE:
;   Calculates the Elastic Incoherent Structure Factor. It is the ratio of the sum of the
;   elastic intensities to the sum of the quasielastic intensities.
;
; CATEGORY:
;   DAVE Utilities
;
; PARAMETERS
;   elInt - The elastic intensities. If only one peak is present then elInt is a vector. Otherwise
;           it is a 2D array where the second dimention is the number of peaks present.
;
;   qeInt - the quasielastic intensities. Again there may be one or more quasielastic peaks just as
;           as in the case for elInt.
;
;   elErr - the errors in elInt. Must have the same dimensions as elInt.
;
;   qeErr - the errors in qeInt. Must have the same dimensions as qeInt.
;
;
; KEYWORDS:
;   RetVal - specifies a named variable that will contain the results of the eisf calculation
;
;   RetErr - specifies a named variable that will contain the error in calculated eisf.
;
; Return Value:
;   returns 1 if successful or 0 otherwise.
;-
function Eisf, elInt, qeInt, elErr, qeErr, RetVal=calc, RetErr=calcErr, emsg=emsg
  compile_opt idl2

  ; Basic error Handler
  if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
      print,'Eisf: Error Encountered'
      emsg = !ERROR_STATE.MSG
      print, emsg
      catch, /cancel
      return, 0
    endif
  endif

  if (n_params() ne 4) then begin
    emsg = 'Eisf - incorrent number of function arguments!'
    print, emsg
    return, 0
  endif
  el_dim = size(elInt,/n_dimensions)
  qe_dim = size(qeInt,/n_dimensions)
  dimCheck = (el_dim eq 1 || el_dim eq 2) && (qe_dim eq 1 || qe_dim eq 2)
  if (~dimCheck) then begin
    emsg = 'Eisf - the elastic or quasielastic intensity have an incorect dimension'
    print, emsg
    return, 0
  endif
  if (el_dim ne size(elErr,/n_dimensions) || size(elInt,/n_elements) ne size(elErr,/n_elements)) then begin
    emsg = 'Eisf - the elastic intensity and error arguments must be the same size'
    print, emsg
    return, 0
  endif
  if (qe_dim ne size(qeErr,/n_dimensions) || size(qeInt,/n_elements) ne size(qeErr,/n_elements)) then begin
    emsg = 'Eisf - the quasielastic intensity and error arguments must be the same size'
    print, emsg
    return, 0
  endif

  ; the nos of elements in each elastic or quasielastic peak
  n_el = (size(elInt))[1]
  n_qe = (size(qeInt))[1]
  if (n_el ne n_qe) then begin
    emsg = 'Eisf - the elastic and quasielastic intensities have different lengths'
    print, emsg
    return, 0
  endif

  ; the number of elastic or quasielastic peaks
  el_peaks = (el_dim eq 1)? 1 : (size(elInt))[2]
  qe_peaks = (qe_dim eq 1)? 1 : (size(qeInt))[2]

  ; if more than one elastic peak, sum them
  elInt0 = elInt
  elErr0 = elErr
  if (el_peaks gt 1) then begin
    elInt0 = total(elInt,2)
    elErr0 = sqrt(total(elErr^2,2))
  endif

  ; if more than one quasielastic peak, sum them
  qeInt0 = qeInt
  qeErr0 = qeErr
  if (qe_peaks gt 1) then begin
    qeInt0 = total(qeInt,2)
    qeErr0 = sqrt(total(qeErr^2,2))
  endif

  ; calculate EISF and error.
  ; For error the fractional errors in must be added in quadrature
  ; if z = x/(x+y) then
  ; let w = x+y, then
  ; dZ/z = quadrature(dx/x, dw/w) and
  ; dz = z*quadrature(dx/x, dw/w) where dw = sqrt(dx^2 + dy^2)

  ; EISF
  w = elInt0 + qeInt0
  index = where(w eq 0, cnt)
  if (cnt gt 0) then w[index] = 1.0
  calc = elInt0/w

  ; error in EISF
  dw = sqrt(elErr0^2 + qeErr0^2)
  dxOx = elErr0/elInt0
  dwOw = dw/w
  calcErr = calc * sqrt(dxOx^2 + dwOw^2)

  return, 1
end