; $Id$
;
; THIS IS USED IN SEVERAL DAVE APPLICATIONS AND IS A USEFUL TOOL FOR DEVELOPMENT AND
; CODE TESTING.
;
;
;###############################################################################
;
;FILENAME:
;checkforobjinstance.pro
;
;PURPOSE:
;           Count the number of instances of a class and get
;           the object references.
;
;CATEGORY:
;           General OOP utility.
;
;METHODS IN FILE:
;
;    checkForObjInstance
;
;
; AUTHOR:
; Larry Kneller
; NIST Center for Neutron Research
; 100 Bureau Drive, Gaithersburg, MD 20899
; United States
; kneller@nist.gov  301-975-8839
; Jan 24,2005
;
; LICENSE:
; The software in this file is written by an employee of
; National Institute of Standards and Technology
; as part of the DAVE software project.
;
; The DAVE software package is not subject to copyright protection
; and is in the public domain. It should be considered as an
; experimental neutron scattering data reduction, visualization, and
; analysis system. As such, the authors assume no responsibility
; whatsoever for its use, and make no guarantees, expressed or
; implied, about its quality, reliability, or any other
; characteristic. The use of certain trade names or commercial
; products does not imply any endorsement of a particular product,
; nor does it imply that the named product is necessarily the best
; product for the stated purpose. We would appreciate acknowledgment
; if the DAVE software is used of if the code in this file is
; included in another product.
;
;###############################################################################
;   AUTHOR  Larry Kneller
;
;   PURPOSE Count the number of instances of a class and get
;           the object references.
;
;   USAGE   numberOfInstances = checkForObjInstance(className,obj=objlist)
;           WHERE className IS THE STRING NAME OF A CLASS AND
;           obj IS AN ARRAY OF THE OBJECT REFERENCES OF TYPE
;           className.
;
;   DATE    10/07/04
;
;
;   I am writing this to count the number of instances
;   of a given class so that I can incorporate it into
;   classes where I want to limit the number of instances
;   to exactly 1 (Singleton), but I am leaving it general
;   enough to simply count the number of instances.
;
;
;   See also Rob's posting "Re: Limiting the number of VM instances"
;   on the comp.lang.idl-pvwave newsgroup at Google on
;   2004-01-22 14:46:11 PST.

function checkForObjInstance,className,ref=ref,subclass=subclass
;
;NAME:
;        checkForObjInstance
;
;PURPOSE:
;       Find instances of a given class.
;PARAMETERS:
;       className   The name of the class of objects to search for.
;KEYWORDS:
;       ref         An array of object references that meet the selection criteria.
;       subclass    Flag to also search for subclasses.
;
;RETURN VALUE:
;       The number of object instances.

    if n_elements(subclass) eq 0 then subclass = 0

    help,className,output=out,struct=0

    if stregex(out,'UNDEFINED',/boolean) then begin
        print,'PLEASE ENTER A VALID CLASS NAME'
        ref=0
        return,0
    endif else begin

        ov = obj_valid()


        ;COUNT THE NUMBER OF VALID OBJECTS OF TYPE className
        count=0
        for i=0L,n_elements(ov)-1 do begin
            help,ov[i],output=op,struct=0
            segs = strsplit(op,/extract)

            if subclass ne 0 then begin
                ;NEW -- SELECTS THE CLASS AND ALL OBJECTS THAT ARE SUBCLASSES OF THE CLASS.
                if obj_isa(ov[i],className) then count++
            endif else begin

                ;ORIGINAL -- SELECTS ONLY ITEMS IN THE CLASS.
                if (stregex(segs[3],'('+className+')',/boolean,/fold_case)) $
                    then begin
                    count++
                    pos = stregex(segs[3],'('+className+')', length=len)
                endif
            endelse
        endfor;i
    endelse;UNDEFINED


    ;RETRIEVE REFERENCES TO ALL OF THE VALID OBJECTS
    ;OF CLASSNAME.
    if count ne 0 then begin
        ref = objarr(count)
        jref = 0

        for i=0L,n_elements(ov)-1 do begin
            if subclass ne 0 then begin
                ;FIND OBJECTS OF CLASS AND SUBCLASSES
                if obj_isa(ov[i],className) gt 0 then begin
                    ref[jref] = ov[i]
                    jref++
                endif
            endif else begin
                ;FIND OBJECTS OF SPECIFIED CLASS ONLY ;ORIGINAL
                help,ov[i],output=op,struct=0
                segs = strsplit(op,/extract)
                if (stregex(segs[3],'('+className+')',/boolean,/fold_case)) $
                    then begin
                    ref[jref] = ov[i]
                    jref++
                endif
            endelse
        endfor;i


    endif else ref = 0


    return,count
end;checkForObjInstance
