; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   ASCIIGrpDataset
;
; PURPOSE:
;   Data Class for representing SPE formated datasets
;
; CATEGORY:
;
; SUPERCLASSES:
;   GenericDataset
;
; SUBCLASSES:
;
; CREATION:
;   See ASCIIGrpDataset::Init
;
; METHODS:
;   GetProperty
;   SetProperty
;   ToDavePtr
;   ToASCIISPE
;   ToASCIIGrp
;   ToASCIIColumn
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; December 2008
;-
;###############################################################################



;===============================================================================
; ASCIIGrpDataset::Clone
; 
; PURPOSE:
;   return an object containing an axact copy of the data in the current object
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    success - a copy of the current object
;    failure - a null object
;
function ASCIIGrpDataset::Clone
compile_opt idl2

status = Self->toASCIIGrp(dataStr)

if (status) then begin
   filename = 'Copy of '+Self.filename
   Self->GetProperty, name=nameTag, treatment=trmt
   nameTag = 'Copy of '+nameTag
   oData = obj_new('ASCIIGrpDataset',dataStr=dataStr,nameTag=nameTag)
   if (obj_valid(oData)) then oData->SetProperty, filename=filename, treatment=trmt
   return, oData
end

return, obj_new() 

end



;===============================================================================
; ASCIIGrpDataset::Init
; 
; PURPOSE:
;   Initialization method for objects of ASCIIGrpDataset class. For the initialization to 
;   complete successfully, at least a filename or the dataStr keyword must be supplied.
;
; PARAMETERS
;   filename   - The name of a dave format file from which to retrieve the dataset.
;
;   nameTag    - A label to be associated with objects of this class. If undefined, it will 
;                default to the basename of the filename keyword, if that is set, or to 'UNTITLED'
;
;   dataStr    - a data structure containing a Grp dataset This is an anonymous data structure
;                containing the following fields:
;                   {xData: float[n], $    ; a vector representing first independent axis eg energy transfer or time-of-flight, etc
;                    yData: float[m], $    ; a vector representing second independent axis eg angle, wavevector, etc
;                    data:  float[n,m], $  ; a 2D array of dependent data values
;                    error: float[n,m], $  ; a 2D array of the uncertainty in the dependent data values
;                    xTitle: ''       , $  ; title for xData block
;                    yTitle: ''       , $  ; title for yData block
;                    dataTitle: ''    , $  ; title for data block
;                   }
;                Note: the xData and yData fields could be of size n+1 nd m+1 respectively if they represent histogram data.
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function ASCIIGrpDataset::Init, filename=filename, dataStr=dataStr, nameTag=nameTag, _REF_EXTRA=etc
compile_opt idl2

; Basic error handler
catch, theError
if (theError ne 0) then begin
    catch, /cancel
    
    goto, IOERROR
endif

filenameIsSet = n_elements(filename)
dataStrIsSet = n_elements(dataStr)
nameTagIsSet = n_elements(nameTag)

; Can't proceed if both filename and dataStr are undefined
if (~filenameIsSet && ~dataStrIsSet) then return, 0

; If starting from filename, read it in
if (filenameIsSet and ~dataStrIsSet) then begin
   
   openr, lun, filename, /get_lun  ; open file
   
   ;; first line after comments should contain size of
   ;; first independent axis.
   repeat begin
       line = ''
       readf, lun, line
   endrep  until (strmid(line,0,1) ne '#') 
   n1 = strtrim(strsplit(line,' ',/extract,count=ntoks),2)
   if (ntoks ne 1) then goto, IOERROR
   n1 = fix(n1[0])
   
   ;; Same again for size of 2nd independent axis.
   repeat begin
       line = ''
       readf, lun, line
   endrep  until (strmid(line,0,1) ne '#') 
   n2 = strtrim(strsplit(line,' ',/extract,count=ntoks),2)
   if (ntoks ne 1) then goto, IOERROR
   n2 = fix(n2[0])
   
   ;; First independent axis
   line = ''
   readf, lun, line
   if (strmid(line,0,1) ne '#') then goto, IOERROR
   xData = fltarr(n1)
   readf, lun, xData
   xtit = strtrim(strmid(line,1),2)
   
   ;; Second independent axis
   line = ''
   readf, lun, line
   if (strmid(line,0,1) ne '#') then goto, IOERROR
   yData = fltarr(n2)
   readf, lun, yData
   ytit = strtrim(strmid(line,1),2)
   
   ;; The data
   data = fltarr(n1,n2)
   err = fltarr(n1,n2)
   buf = fltarr(2*n1)
   index1 = indgen(n1)*2
   index2 = indgen(n1)*2+1
   for i=0,n2-1 do begin
       readf, lun, line
       readf, lun, buf
       data[*,i] = buf[index1]
       err[*,i] = buf[index2]
   endfor
   ztit = 'Intensity'
   
   free_lun, lun, /force           ; close file

   xMode = 'POINT'
   yMode = 'POINT'
   
   xName = (strtrim(xtit) eq '')? 'X' : xtit
   yName = (strtrim(ytit) eq '')? 'Y' : ytit
   datName = (strtrim(ztit) eq '')? 'Intensity' : ztit
   errName = 'Error'
   
   xDesc = xName
   yDesc = yName
   zDesc = datName
   
   i = strpos(filename,'.',/reverse_search)
   ext = (i lt 0)? '' : strmid(filename,i)
   ; strip dir path and extension from the filename and assign to nameTag, if required
   basename = file_basename(filename,ext,/fold_case)
   if (~nameTagisSet) then nameTag = basename
   
endif else if (dataStrIsSet) then begin
   xdata = dataStr.xData
   ydata = dataStr.yData
   data = dataStr.data
   err = dataStr.error
   xtit = dataStr.xTitle
   ytit = dataStr.yTitle
   ztit = dataStr.dataTitle
   
   zdim = size(data)
   d1 = zdim[1]
   d2 = zdim[2]
   nx = n_elements(xdata)
   ny = n_elements(ydata)

   ; Grp data should be point mode!
   xMode = 'POINT'
   yMode = 'POINT'
   if (nx eq (d1 +1)) then begin
      ;; this is histogram data wrt x so convert to point mode
      index = lindgen(d1)
      xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries
      nx = n_elements(xData)
   endif
   if (ny eq (d1 +1)) then begin
      ;; this is histogram data wrt y so convert to point mode
      index = lindgen(d2)
      yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries
      ny = n_elements(yDat)      
   endif

   xName = (strtrim(xtit) eq '')? 'X' : xtit
   yName = (strtrim(ytit) eq '')? 'Y' : ytit
   datName = (strtrim(ztit) eq '')? 'Intensity' : ztit
   errName = 'Error'
   
   xDesc = xName
   yDesc = yName
   zDesc = datName
   
endif

; Check for masked data points 
; For spe files, masked points have the value -1e20
; Replace masked points with a NaN
index = where(data eq -1e20, nMasked)
if (nMasked gt 0) then data[index] = !values.F_NAN


; If nameTag is not yet specified, set it to 'Untitled'
if (n_elements(nameTag) eq 0) then nameTag = 'UNTITLED'
description = (filenameIsSet)? filename : nameTag
basename = (filenameIsSet)? basename : nameTag

; iTools does not handle '_[0-9]' characters in nameTags well eg _0, _1, etc
; replace all '_' with '-'
;while ((pos = stregex(nameTag,'_[0-9]')) ne -1) do begin
;    strput, nameTag,'-',pos
;endwhile
   

; Call the base class init method to instantiate container
status = self->GenericDataset::Init(name=nameTag $
                ,description=description,identifier='id'+file_basename(nameTag),icon='surface' $
                ,type='ASCIIGRP', _EXTRA=etc)
if (status eq 0) then return, status

Self->AddMetaData,'DatasetName',basename ; give it same name as the file basename

xAxisType=0
yAxisType=1
datAxisType=2
errAxisType=3
oX = obj_new('IDLitDataDAVE', xdata, name=xName,type=partype(xdata), description=xDesc, axisType=xAxisType)
oY = obj_new('IDLitDataDAVE', ydata, name=yName,type=partype(ydata), description=yDesc, axisType=yAxisType)
oDat = obj_new('IDLitDataDAVE',data,name=datName,type=partype(data), description=zDesc, axisType=datAxisType)
oErr = obj_new('IDLitDataDAVE',err,name=errName,type=partype(err), description=errName, axisType=errAxisType)

;; metadata
oX->AddMetaData,'Long_name',xDesc
oX->AddMetaData,'Units',''
oX->AddMetaData,'Distribution',xMode

oY->AddMetaData,'Long_name',yDesc
oY->AddMetaData,'Units',''
oY->AddMetaData,'Distribution',yMode

oDat->AddMetaData,'Signal',1
oDat->AddMetaData,'Axes',[xName,yName]
oDat->AddMetaData,'Long_name',zDesc
oDat->AddMetaData,'Units',''

Self->Add, [oX,oY,oDat,oErr] $
            ,parameter_name=strtrim(string([xAxisType,yAxisType,datAxisType,errAxisType]),2)
            
; Add a blank treatment history object
child = obj_new('IDLitData','',name='Treatment History',type='HISTORY' $
                ,description='Treatment history')
if (obj_valid(child)) then begin
   Self->add, child
   Self.treatmentRef = child
endif

            
Self.axis1Ref = oX
Self.axis2Ref = oY
Self.dataRef = oDat
Self.errorRef = oErr
self.plotDataRef = Self
Self.filename = description
Self.nDims = 2

return, 1

IOERROR:
if (n_elements(lun) gt 0) then begin
    sStat = fStat(lun)
    if (sStat.open ne 0) then free_lun, lun, /force
endif

return, 0

end


;===============================================================================
pro ASCIIGrpDataset__Define
compile_opt idl2

void = {ASCIIGrpDataset $
        ,inherits GenericDataset $
        }

end