; $Id$
;###############################################################################
;
; NAME:
;   wd_ViewDetal
;
; PURPOSE:
;   A small utility widget program to display the data contained
;   within an IDLitData object
;
; CATEGORY:
;   DAVE Main Tool
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   Jan, 2006
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;###############################################################################



;===============================================================================
; DB_DispData_Event
;
; PURPOSE:
;   Event handler for events originating from the main UI
;
; PARAMETERS:
;   event [in] - the event structure to be handled
;
pro wd_viewdetail_event,event

uname = widget_info(event.id,/uname)

eventName = tag_names(event,/structure_name)
if (eventName eq 'WIDGET_BASE') then begin
    wID = widget_info(event.top, find_by_uname='TEXT')
;    geom = widget_info(event.top,/geometry)
;    help,geom,/struct
    widget_control, event.top, get_uvalue=uval
    newX = event.x - uval.xdiff
    newY = event.y - uval.ydiff
    if (widget_info(wID,/valid)) then $
      widget_control, wID, scr_xsize=newX, scr_ysize=newY
endif

case uname of
    'DISMISS': widget_control,event.top,/destroy
    else:
endcase

end
;-------------------------------------------------------------------------------



;===============================================================================
; WD_ViewDetail
;
; PURPOSE:
;   Main dialog builder for program
;
; PARAMETERS
;   oData [in;req] - IDLitData object whose data is to be displayed.
;
; KEYWORDS:
;   group_leader [in] - the WID of the parent widget from which this
;                       dialog was launched.
;
pro wd_viewDetail, oData, group_leader=grpl

;
if (~obj_valid(oData) || ~oData->GetData(data) || (n_elements(data) le 0)) then begin
    msg = 'Invalid data'
    void = dialog_message(msg,/error,dialog_parent=grpl,title='Data Details')
    return
endif
;
dims = (size(data))[0]
if (dims lt 0 || dims gt 2) then begin
    msg = 'Data content cannot be interpreted!'
    void = dialog_message(msg,/error,dialog_parent=grpl,title='Data Details')
    return
endif

buffer = ''

; Check for metadata
metaDataNames = oData->GetMetaDataNames(count=metaDataCnt)

if (metaDataCnt gt 0) then begin
    buffer = [buffer,'MetaData Section:']
    for i = 0,metaDataCnt-1 do begin
        void = oData->GetMetaData(metaDataNames[i],metaData)
        if (void) then begin
            tn = size(metaData,/tname)
            bad = (tn eq 'OBJREF' || tn eq 'POINTER' || tn eq 'STRUCT' || $
                   tn eq 'UNDEFINED' || tn eq 'COMPLEX' || tn eq 'DCOMPLEX')
            if (~bad) then $
              buffer = [buffer,metaDataNames[i]+' = '+string(metaData)]
        endif
    endfor
    buffer = [buffer,'']
endif

; The data
tn = size(data,/tname)
isAnInt = (tn eq 'BYTE' || tn eq 'INT' || tn eq 'LONG' || tn eq 'UINT' || $
           tn eq 'ULONG' || tn eq 'LONG64' || tn eq 'ULONG64')
isAFloat = (tn eq 'FLOAT' || tn eq 'DOUBLE')
isAString = (tn eq 'STRING')

; Deal with Scalars and Vectors
if (dims eq 0 || dims eq 1) then begin
    fmt = (dims eq 0)? '(G10.4)' : '(8(G11.4,1X))'
    case 1 of
        isAnInt: buffer = [buffer,strtrim(string(data,format=fmt),2)]
        isAFloat: buffer = [buffer,strtrim(string(data,format=fmt),2)]
        isAString: buffer = [buffer,data]
    endcase
endif


; Deal with 2D arrays
if (dims eq 2) then begin
    fmt = '(8(G11.4,1X))'
    size = 1.0e-3*n_elements(data)*4.0
    progressOn = 0
    if (2*size ge 20.0) then begin
        progressOn = 1
        title = 'Detail Display Progress'
        pTlb = keyword_set(grpl)? $
               widget_base(group_leader=grpl,tlb_frame_attr=1,modal=1,title=title,/col) : $
               widget_base(tlb_frame_attr=1,modal=0,title=title,/col)
        msg = 'The contents of data is '+strtrim(string(format='(G10.4)',2*size),2)+' KB!'
        msg = ['',msg,'Be patient while it is being formatted for display','']
        void = widget_text(pTlb,value=msg,xsize=max(strlen(msg)),ysize=5)
        geom = widget_info(pTLB,/geometry)
        slider = widget_slider(pTLB,/frame,minimum=0,maximum=100,xsize=geom.xsize)

        widget_control,pTLB,/realize
    endif
    nGroups = (size(data))[2]
    nGroupsTxt = strtrim(string(nGroups),2)
    fraction = 100.0/ngroups
    line = strjoin(replicate('_',50),/single)     ; create a line of underscores
    for i=0,nGroups-1 do begin
        grpHeader  = 'Group '+strtrim(string(i+1),2)+' of '+nGroupsTxt
        buffer = [temporary(buffer),line,grpHeader]
        buffer = [temporary(buffer),string(format=fmt,data[*,i])]
        if (progressOn) then widget_control, slider, set_value=fix((i+1)*fraction)
    endfor

    if (progressOn) then widget_control, pTlb, /destroy
endif

maxh = 30                       ; max text display height
height = n_elements(buffer) < maxh ; text display height
maxw = 60                       ; width of text widget
width = maxw < max(strlen(buffer))


oData->GetProperty, name=dataName 
title = 'Contents of ' + dataName
width = width > 1.75*strlen(title)
modal = 0
wTLB = widget_base(/col,title=title,/base_align_center $ ;,tlb_frame_attr=1 $
                   ,group_leader=grpl,modal=modal $
                   ,/tlb_size_events $
                  )

; Create the text display widget and populate it with the data
case (!version.os_family eq 'Windows') of
    1: font = 'Courier New*14'
    else: font = '-adobe-courier-bold-r-normal--12-120-75-75-m-70-iso8859-1'
endcase
scroll = ((width ge maxw) || (height ge maxh))? 1 : 0
wText = widget_text(wTLB,value=buffer,xsize=width,ysize=height,scroll=scroll,uname='TEXT',font=font)

; Create a close button
dismTip = 'Close this dialog'
wBut = widget_button(wTLB,value='Dismiss',uname='DISMISS',toolTip=dismTip)

geom0 = widget_info(wTLB,/geometry)
geom1 = widget_info(wText,/geometry)
xdiff = geom0.scr_xsize - geom1.scr_xsize
ydiff = geom0.scr_ysize - geom1.scr_ysize

centertlb, wTLB
widget_control,wTLB,/realize,set_uvalue={xdiff:xdiff,ydiff:ydiff}

xmanager,'WD_VIEWDETAIL',wTLB,event_handler='wd_viewdetail_event',/no_block ;,no_block=modal

return
end
