; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEvisSurface
;
; PURPOSE:
;   This file simply inherits and extends IDL's IDLitVisSurface visualization class
;
; CATEGORY:
;   DAVE Visualization
;
; SUPERCLASSES:
;   IDLitVisSurface
;
; SUBCLASSES:
;
; CREATION:
;
; METHODS:
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; September 2004
;-
;###############################################################################



;===============================================================================
; DAVEvisSurface::SetPaletteAttribute
; 
; PURPOSE:
;   Un/set the palette property. Overides
;   IDLitVisSurface::SetPaletteAttribute and is essentially the same
;   except the palette object is created in 'Data Maneger' container
;   rather than in the parent container of the 'Z' parameter. In
;   addition the palette defaults to rainbow rather than greyscale.
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
pro DAVEvisSurface::SetPaletteAttribute
compile_opt hidden

;; determine if a palette is currently useful or needed
oVertColor = self->getParameter('VERTEX COLORS')
vSuccess = (obj_valid(oVertColor) ? oVertColor->getData(vData) : 0)
vDims = (vSuccess ? size(vData,/n_dimensions) : 0)
dataDims = SIZE(vData, /DIMENSIONS)
oTextMap = self->getParameter('TEXTURE')
tSuccess = (obj_valid(oTextMap) ? oTextMap->getData(tData) : 0)
tDims = (tSuccess ? size(tData,/n_dimensions) : 0)
needPalette = (tDims EQ 2) || (vDims EQ 1) || $
              ((vDims eq 2) && (dataDims[0] gt 4))

;; do we have a valid palette?
oPalette = self->getParameter('PALETTE')
pSuccess = (obj_valid(oPalette) ? oPalette->getData(pData) : 0)

IF needPalette THEN BEGIN
    self->setPropertyAttribute,'VISUALIZATION_PALETTE',SENSITIVE=1
    ;; if we have a valid palette just return
    IF pSuccess THEN return

    ;; if no palette exists, create one!
    ;; first check to see if we have one lying around
    oTool = self->GetTool()
    oPalette = oTool->getByIdentifier(self._myPaletteID)
    IF obj_valid(oPalette) THEN BEGIN
        success = self->setData(oPalette,PARAMETER_NAME='PALETTE')
        self._myPalette = 1
    ENDIF ELSE BEGIN
        ;; create a temp IDLgrPalette and load color table 13
        ;; (rainbow) and extract the rgb values from it.
        oVoidPalette = obj_new('IDLgrPalette')
        oVoidPalette->LoadCT,13 ; use color table 13 - rainbow
        oVoidPalette->GetProperty, red_values=pr,green_values=pg,blue_values=pb
        obj_destroy,oVoidPalette

        ;; create an IDLitDataIDLPalette object using the rgb values
        ;; Create a palette parameter from the object
        oRainbowPalette = OBJ_NEW('IDLitDataIDLPalette' $
                                  ,transpose([[pr],[pg],[pb]]) $
                                  ,name='PALETTE')
        oRainbowPalette->SetProperty,/AUTO_DELETE ; should self-destruct when no longer needed
        success = self->setData(oRainbowPalette,PARAMETER_NAME='PALETTE')
        oPalette = self->getParameter('PALETTE')

        ;; add to the system's Data Manager directory
        oZ = self->GetParameter('Z')
        IF obj_valid(oZ) THEN BEGIN
            self->AddByIdentifier,'/DATA MANAGER',oPalette
        ENDIF

        self._myPalette = 1
        self._myPaletteID = oPalette->getFullIdentifier()
    ENDELSE
ENDIF ELSE BEGIN
    ;; if the palette was the default rainbow, delete it
    IF pSuccess && self._myPalette THEN BEGIN
        self->unSetParameter,'PALETTE'
        obj_destroy,oPalette
        self._myPaletteID = ''
    ENDIF
    self._myPalette = 0
    ;; do we still have a valid palette?
    oPalette = self->getParameter('PALETTE')
    pSuccess = (obj_valid(oPalette) ? oPalette->getData(pData) : 0)
    self->setPropertyAttribute,'VISUALIZATION_PALETTE',SENSITIVE=pSuccess
ENDELSE

END


;===============================================================================
; DAVEvisSurface::EnsureXYParameters
; 
; PURPOSE:
;   Ensure that X and Y parameters exist, based on the surface data 
;   dimensions. Overides IDLitVisSurface::EnsureXYParameters and is
;   essentially the same except that any data that are automatically
;   created are stored in the system 'Data Manager' and not underneath
;   the parent of the 'Z' parameter.
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
pro DAVEvisSurface::EnsureXYParameters

; Retrieve Z parameter first to get data dimensions.
oZParam = self->GetParameter('Z')
if (~OBJ_VALID(oZParam)) then return
if (~(oZParam->GetData(pData, /POINTER))) then return
dims = SIZE(*pData, /DIMENSIONS)

paramNames = ['X', 'Y']
for i=0,1 do begin
    ;; Check if parameter already exists.
    oParam = self->GetParameter(paramNames[i])
    if (~obj_valid(oParam)) then begin
        ;; Create and set the parameter.
        data = DINDGEN(dims[i])
        oData = OBJ_NEW('IDLitDataIDLVector', data, NAME=paramNames[i])
        oData->SetProperty, /AUTO_DELETE
        self->SetParameter, paramNames[i], oData, /NO_UPDATE
        
        ;; Add to data manager.
        oData = self->GetParameter(paramNames[i])

        self->AddByIdentifier,'/DATA MANAGER',oData
    endif
endfor

; Send a notification message to update UI
self->DoOnNotify, self->GetFullIdentifier(),"ADDITEMS", ''

end


;===============================================================================
; DAVEvisSurface::_ValidateXYParameter
; 
; PURPOSE:
;   Overide IDLitVisSurface::_ValidateXYParameter. To avoid errors,
;   make sure the number of elements in X or Y matches up with Z
;   data. DAVE extension deals with histogram data.
;
; PARAMETERS:
;   param [in] - 'X' or 'Y'
;
;   data [out] - Array containing the X or Y data, or undefined if not
;                successful.
;
; KEYWORDS:
;    Returns 1 for success, 0 otherwise. If success then the X or Y
;    data is contained in the data argument, otherwise it is
;    undefined. 
;
function DAVEvisSurface::_ValidateXYParameter, param, data

    compile_opt idl2, hidden

    ; First retrieve Z dimensions.
    oZ = self->GetParameter('Z')
    if (~OBJ_VALID(oZ) || ~oZ->GetData(pZ, /POINTER)) then $
        return, 0
    zdim = SIZE(*pZ, /DIMENSIONS)

    oParam = self->GetParameter(param)
    if (~OBJ_VALID(oParam) || ~oParam->GetData(mydata)) then $
        return, 0

    ;; if parameter contains histogram data, convert it into point mode
    if (~oParam->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        nx = n_elements(mydata)
        index = lindgen(nx-1)
        points = 0.5*(mydata[index]+mydata[index+1]) ; ==> mid values of the bin boundaries
        mydata = temporary(points)
    endif

    if (min(mydata, MAX=maxx) eq maxx) then $
        return, 0

    ndim = SIZE(mydata, /N_DIM)

    ; Vector X/Y?
    if (ndim eq 1) then begin
        ; Which dimension to check.
        zdim1 = (param eq 'X') ? zdim[0] : zdim[1]
        if (N_ELEMENTS(mydata) ne zdim1) then $
            return, 0
    endif else begin
        ; 2D X/Y
        dims = SIZE(mydata, /DIM)
        if (~ARRAY_EQUAL(dims, zdim)) then $
            return, 0
    endelse

    ; If we reach here then success.
    data = TEMPORARY(mydata)
    return, 1
end



;===============================================================================
; DAVEvisSurface::OnDataChangeUpdate
; 
; PURPOSE:
;   Extend IDLitVisSurface::OnDataChangeUpdate.
;   This method is called when the data associated with a parameter
;   has changed. When called, the visualization performs the
;   desired updates to reflect the change.
;
; PARAMETERS:
;   oSubject - The data item for the parameter that has changed.
;
;   parmName - The name of the parameter that has changed.
;
; KEYWORDS:
;   none
;
pro DAVEvisSurface::OnDataChangeUpdate, oSubject, parmName
compile_opt idl2

; Call base class method
self->IDLitVisSurface::OnDataChangeUpdate, oSubject, parmName

;; Perform additional tasks not handled by the base class method
case strupcase(parmName) of
    '<PARAMETER SET>': begin
        ;; IDLitVisSurface::OnDataChangeUpdate handles the 'X' and 'Y'
        ;; parameters during the handling of the 'Z' parameter. Need
        ;; to force an explicit call for 'X' and 'Y' 
        dataNames = ['X', 'Y']
        for i=0,n_elements(dataNames)-1 do begin
            oData = oSubject->GetByName(dataNames[i],count=nCount)
            if (nCount ne 0) then self->OnDataChangeUpdate,oData,dataNames[i]
        endfor
    end

    'Z': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label
            zLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then zLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              zLabel = (strlen(units) gt 0)? zLabel + ' ('+units+')' : zLabel

            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)

            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then begin
                ;; Ensure z-axis is located in the background (0,1,0)
                ;; instead of the default (0,0,0) for surface displays
                ;; using 3 axes.
                ;all = oAxes->Get(/ALL, ISA='IDLitVisAxis',count=count)
                ;if (count eq 3) then begin
                ;    all[2]->SetProperty, norm_location=[0d,1d,0d]
                    ;oAxes->_UpdateAxesRanges
                ;endif
                oAxes->SetProperty,ztitle=strtrim(zLabel)
            endif
        endif

        ;; If 'Z' changes and 'VERTICAL COLORS' is unset, then assign the
        ;; 'Z' parameter data to it.
        ;; Proceed if the 'VERTEX COLORS' parameter is not set
        if (~obj_valid(self->getParameter('VERTEX COLORS'))) then begin
            ;; Assign the 'Z' parameter data (oSubject) to the
            ;; 'VERTEX COLORS'  parameter.  This assignment will also
            ;; trigger an update. 
            void = self->SetData(oSubject,PARAMETER_NAME='VERTEX COLORS')
        endif
    end
    
    'Y': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label + unit
            yLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then yLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              yLabel = (strlen(units) gt 0)? yLabel + ' ('+units+')' : yLabel

            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)

            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,ytitle=strtrim(yLabel)
            
            ;; If histogram data, convert to point mode
;             if (oData->GetMetaData('Distribution',dist)) then begin
;                if ( strmid(strupcase(dist),0,4) eq 'HIST' && $
;                     self->_ValidateXYParameter('Y', yData)) then begin
                 
;                   len = n_elements(yData)
;                   index = lindgen(len-1)
;                   yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries

;                   self._oSurface->SetProperty, DATAY=temporary(yData)
;                endif
;             endif
        endif
    end
    
    'X': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; X label + unit
            xLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then xLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              xLabel = (strlen(units) gt 0)? xLabel + ' ('+units+')' : xLabel

            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)

            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,xtitle=strtrim(xLabel)
            
            ;; If histogram data, convert to point mode
;             if (oData->GetMetaData('Distribution',dist)) then begin
;                if ( strmid(strupcase(dist),0,4) eq 'HIST' && $ 
;                     self->_ValidateXYParameter('X', xData) ) then begin
                  
;                   len = n_elements(xData)
;                   index = lindgen(len-1)
;                   xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries

;                   self._oSurface->SetProperty, DATAX=temporary(xData)
;                endif
;             endif
        endif                    
    end
    
    else:
endcase

end


;===============================================================================
; DAVEvisSurface::Init
; 
; PURPOSE:
;   Init
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
function DAVEvisSurface::Init, _REF_EXTRA=etc
compile_opt idl2

print,'DAVEvisSurface::Init called...'

; Call base class init method
if (~self->IDLitVisSurface::Init(_EXTRA=etc)) then return, 0

; Set sensible default properties
self->SetProperty, shading=1, hidden_lines=1 ; Gouraud shading, hidden lines removal

return, 1
end


;===============================================================================
; DAVEvisSurface__define
; 
; PURPOSE:
;   DAVEvisSurface class structure definition
;
pro DAVEvisSurface__define

compile_opt idl2

struct = {DAVEvisSurface, $
         inherits IDLitVisSurface }

end
