; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopPlotProfile
;
; PURPOSE:
;   This file implements the DAVEopPlotProfile the class which simply
;   extends the IDLitopPlotProfile class. The IDLitopPlotProfile class
;   implements the operation that takes endpoints supplied
;   by the line profile manipulator and operation and creates a plot.
;   The plot represents the data of the selected visualization at the
;   location specified by the manipulator.
;
; CATEGORY:
;   DAVE Visualization
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Sep 2006
;-
;###############################################################################


;===============================================================================
; DAVEopPlotProfile::_CreatePlotProfile
; 
; PURPOSE:
;   Create Line Profile Plot. Simply overide the base class method
;
; PARAMETER:
;   oTool: iTool object reference
;
;   pt0/pt1:  endpoints of the line
;
; KEYWORDS:
;
pro DAVEopPlotProfile::_CreatePlotProfile, oTool, oLine, oSelVis, $
                                RECURSE=recurse

    compile_opt idl2

    ;; Make sure we have a tool.
    if not OBJ_VALID(oTool) then $
        return

    ;; Check the selected object.
    if not OBJ_VALID(oSelVis) then $
        return

    ; alternate case is for an image (~isSurface)
    isSurface = obj_isa(oSelVis, 'IDLITVISSURFACE')

    pt0 = oLine->GetVertex(0)
    pt1 = oLine->GetVertex(1)
    IF ~isSurface THEN BEGIN
        pt0[2] = 0
        pt1[2] = 0
    ENDIF

    nData = oSelVis->GetParameterDataByType(['IDLARRAY2D'], oDataObjs)
    if nData eq 0 then $
        return

    ;; Adjust points for image origin and size
    if ~isSurface then begin
        oSelVis->GetProperty, XORIGIN=XOrigin, YORIGIN=YOrigin, $
            PIXEL_XSIZE=pixelXSize, PIXEL_YSIZE=pixelYSize
        pt0[0] = (pt0[0]-XOrigin)/pixelXSize
        pt0[1] = (pt0[1]-YOrigin)/pixelYSize
        pt1[0] = (pt1[0]-XOrigin)/pixelXSize
        pt1[1] = (pt1[1]-YOrigin)/pixelYSize
    endif

    ;; get xData
    oXData = oSelVis->GetParameter('X')
    xSuccess = (obj_valid(oXData) ? oXData->getData(xData) : 0)
    ;; get yData
    oYData = oSelVis->GetParameter('Y')
    ySuccess = (obj_valid(oYData) ? oYData->getData(yData) : 0)

    ;; Take a quick look to see if any lines are not clipped out.
    ;; We don't want to create a plot tool with all the lines clipped out.
    vis = 0
    for iData=0, nData-1 do begin
        success = oDataObjs[iData]->GetData(pData, /POINTER)
        if success eq 0 then continue
        dims = SIZE(*pData, /DIMENSIONS)
        p0 = DOUBLE(pt0)
        p1 = DOUBLE(pt1)
        rect = dblarr(4)
        ;; if X and Y vectors exist use them, else use data dimensions
;        rect[0:1] = [0,dims[0]-1]
;        rect[2:3] = [0,dims[1]-1]
        rect[0:1] = ((xSuccess && isSurface) ? [min(xData,max=max),max] : [0,dims[0]-1])
        rect[2:3] = ((ySuccess && isSurface) ? [min(yData,max=max),max] : [0,dims[1]-1])
        vis = self->_clipLine( p0, p1, rect )
        if vis gt 0 then $
            break
    endfor
    if vis eq 0 then $
        return

    ;; Create/Get our Plot Profile tool as needed.
    oNewTool = oTool->GetByIdentifier(self._strToolID)
    newToolCreated = 0
    if not OBJ_VALID(oNewTool) then begin
;        self._strToolID = IDLitSys_CreateTool("Plot Tool", $
;                                              NAME="Plot Profile",$
;                                              TITLE='IDL Plot
 ;                                              Profile')
       nameTag = 'Plot Profile'
        self._strToolID = IDLitSys_CreateTool("DAVE VisTool" $
                                              ,title=nameTag $
                                              ,user_interface='DAVE VisTool Interface' $
;                                             ,NAME="Plot Profile" $
                                             )
        oNewTool = oTool->GetByIdentifier(self._strToolID)

        ;; The vistool should have the same project/file name as the maintool
        oTool->GetProperty, tool_filename=filename
        oNewTool->SetProperty, nameTag=nameTag,tool_filename=filename 

        newToolCreated = 1
    endif
; 
    ;; Get the create viz operation.
    oCreateVis = oNewTool->GetService("CREATE_VISUALIZATION")
    if (~OBJ_VALID(oCreateVis)) then $
        return

    colorTab = [ [255,0,0], [0,255,0], [0,0,255],$
                 [255,255,0], [255,0,255], [0,255,255] ]

    ;; Get manip line's OUTPUT LINE parameter so we can wire it to
    ;; each Plot Line's INPUT LINE parameter
    oLineParm = oLine->GetParameter('LINE')
    oLine3DParm = oLine->GetParameter('LINE3D')

    ;; If the assoc vis is a surface, then we really only want the
    ;; surface data, not any associated texture map planes.
    ;; A good thought for the future is to create two plot profile tools,
    ;; one with the surface data, and another with the texture map planes.
    if isSurface then $
        nData = 1

    ;; for each channel
    for iData=0, nData-1 do begin

        success = oDataObjs[iData]->GetData(pData, /POINTER)
        if success eq 0 then continue

        ;; Need to clip
        dims = SIZE(*pData, /DIMENSIONS)
        p0 = DOUBLE(pt0)
        p1 = DOUBLE(pt1)
        rect = DBLARR(4)
        ;; if X and Y vectors exist use them, else use data dimensions
;        rect[0:1] = [0,dims[0]]
;        rect[2:3] = [0,dims[1]]
        rect[0:1] = ((xSuccess && isSurface) ? [min(xData,max=max),max] : [0,dims[0]])
        rect[2:3] = ((ySuccess && isSurface) ? [min(yData,max=max),max] : [0,dims[1]])
        vis = self->_clipLine( p0, p1, rect )
        if vis eq 0 then continue

        ;; Build Parm Set
        oParmSet = OBJ_NEW('IDLitParameterSet', $
                           DESCRIPTION='Plot Profile', NAME='Plot Profile',$
                           TYPE='Plot', ICON='plot')

        ;; Wire up Image parm
        oParmSet->Add, oDataObjs[iData], PARAMETER_NAME='IMAGE', /PRESERVE_LOCATION

        ;; Set the line data and wire up the Line Parm
        if ~isSurface then begin
            p0[0] = p0[0] * pixelXSize + XOrigin
            p0[1] = p0[1] * pixelYSize + YOrigin
            p1[0] = p1[0] * pixelXSize + XOrigin
            p1[1] = p1[1] * pixelYSize + YOrigin
        endif
        line = [[p0[0], p0[1], 0], [p1[0], p1[1], 0]]
        success = oLineParm->SetData(line)
        oParmSet->Add, oLineParm, PARAMETER_NAME='LINE'

        ;; if associated visualization is a surface then handle the 3D case
        if isSurface THEN BEGIN
            oLine->create3DLine,[[pt0], [pt1]],outdata=outdata
            success = oLine3DParm->SetData(outdata)
            oParmSet->Add, oLine3DParm, PARAMETER_NAME='LINE3D'
        endif $
        else begin
            ;; Update the line visualization with the clipped version of the line.
            oLine->SetProperty, _DATA=line
        endelse

        ;; Dummy parm for the plot data
        oParmSet->Add, OBJ_NEW('IDLitDataIDLVector', NAME='X', /NULL), $
            PARAMETER_NAME='X'
        oParmSet->Add, OBJ_NEW('IDLitDataIDLVector', NAME='Y', /NULL), $
            PARAMETER_NAME='Y'

        ;; Figure out the color
        if nData gt 1 then $
            color = colorTab[*, iData mod N_ELEMENTS(colorTab[0,*])]
        ; Otherwise leave COLOR undefined so it picks up the current style.

        ;; Give the data a reasonable name
        lineName = "Plot Profile" + STRCOMPRESS(STRING(iData))
        oLineParm->SetProperty, NAME=lineName+" Data"

        ;; Create the plot line profile visualization
        oCommandSet = oCreateVis->CreateVisualization(oParmSet, $
            "PLOT PROFILE", $
            COLOR=color, $
            NAME=lineName, $
            ID_VISUALIZATION=idVis)
        OBJ_DESTROY, oCommandSet    ; not undoable

        ;; remove duplicate items
        oParmSet->Remove,oParmSet->GetByName('IMAGE')
        ;; Add to Data Manager so that Paste Special works
        oTool->AddByIdentifier, "/Data Manager", oParmSet

        ;; Set the associated vis prop in the plot profile
        oPlotProfile = oTool->GetByIdentifier(idVis)
        oPlotProfile->SetProperty, $
            ASSOCIATED_VISUALIZATION=oSelVis->GetFullIdentifier(), $
            LINE_PROFILE=oLine->GetFullIdentifier()

        ;; Tell the line profile about the plot profile we just created
        oLine->AddPlotProfile, idVis

        ;; This is needed to get the plot profile up to date after setting the
        ;; associated vis above
        oPlotProfile->OnDataChangeUpdate, self, 'LINE'
        oPlotProfile->OnDataChangeUpdate, self, 'LINE3D'

        ;; Make the line and plot profiles observers of the image.
        if ~isSurface then begin
            oLine->AddOnNotifyObserver, oLine->GetFullIdentifier(), $
                oSelVis->GetFullIdentifier()
            oLine->AddOnNotifyObserver, idVis, oSelVis->GetFullIdentifier()
        endif
    endfor ;; each channel

    ;; Now see if we create any plot profiles for any line profiles that
    ;; do not have plot profiles.  This can happen if the user destroys
    ;; the plot profile tool with line profiles still on target.
    if N_ELEMENTS(recurse) eq 0 then begin
        oPlotDataspace = oPlotProfile->GetDataspace()
        oLineDataspace = oLine->GetDataspace()
        oPlots = oPlotDataspace->Get(/ALL, ISA='IDLITVISPLOTPROFILE')
        oLines = oLineDataspace->Get(/ALL, ISA='IDLITVISLINEPROFILE')
        ;; See if there are any lines that don't have plots.
        for iLine=0, N_ELEMENTS(oLines)-1 do begin
            idLine = oLines[iLine]->GetFullIdentifier()
            match = 0
            for iplt=0, N_ELEMENTS(oPlots)-1 do begin
                oPlots[iplt]->GetProperty, LINE_PROFILE=idPlotLine
                if idPlotline eq idLine then begin
                    match = 1
                    break
                endif
            endfor
            if match eq 0 then begin
                self->_CreatePlotProfile, oTool, oLines[iLine], oSelVis, /RECURSE
            endif
        endfor
    endif

    if (newToolCreated) then begin
       ;; Create an entry for Plot Profile in the Visualization Browser
       ;;----------------------------------------------
       
       ;; Use a UI service. Note the unconventional call
       ;; here. Usually, the requester is the object reference of
       ;; a tool/operation and not the oUI; but this was the only
       ;; way to have access to this oUI from within the UI service.
       oSystem = oTool->GetSystem()
       oMainTool = oSystem->getByIdentifier('/TOOLS/DAVEMAIN')
       if (obj_valid(oMainTool) && obj_isa(oMainTool,'DAVETool')) then begin
          res = oNewTool->doUIService('AddVisItem',oMainTool->GetUI())
          oMainTool->_SetDirty, 1
       endif
    endif
end



;===============================================================================
pro DAVEopPlotProfile__define
compile_opt idl2

struc = {DAVEopPlotProfile,            $
         inherits IDLitOpPlotProfile    $
        }

end
