; $Id$
;
; Copyright (c) 2000-2004, Research Systems, Inc.  All rights reserved.
;   Unauthorized reproduction prohibited.
;----------------------------------------------------------------------------
;+
; CLASS_NAME:
;   IDLitToolbase
;
; PURPOSE:
;   This file implements the HFBS Tool object
;
; CATEGORY:
;   
;
; SUPERCLASSES:
;   IDLitTool
;
; SUBCLASSES:
;
; CREATION:
;   See IDLitToolbase::Init
;
; METHODS:
;   This class has the following methods:
;
;   IDLitToolbase::Init
;
; INTERFACES:
; IIDLProperty
;-



;===============================================================================
; HfbsFWSTool::GetSystem
; 
; PURPOSE:
;   Return the IDLitSystem object
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::GetSystem
compile_opt idl2

return, self->_GetSystem()

end


;===============================================================================
; HfbsFWSTool::GetUI
; 
; PURPOSE:
;   Return the IDLitUI object for this tool
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::GetUI
compile_opt idl2

return, self._oUIConnection

end


;-------------------------------------------------------------------------------
pro HfbsFWSTool::Cleanup
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::Cleanup: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle)

        catch, /cancel
        ptr_free, Self.prefsPtr
        ptr_free, Self.historyPtr
        Self->IDLitTool::Cleanup
        Self->EnableUpdates
        return
    endif
endif

Self->DisableUpdates

; save session preferences
status = Self->SavePreferences()

ptr_free, Self.prefsPtr
ptr_free, Self.historyPtr

; Clean up samp container
oItems = Self.sampContRef->Get(/all, count=n)
if (n gt 0) then begin
   for i=0,n-1 do begin
      void = oItems[i]->GetMetaData('DATAHASH',pdataHash)
      if (ptr_valid(pdataHash)) then heap_free, pdataHash
   endfor
endif

obj_destroy, Self.sampContRef

; Call base class cleanup
self->IDLitTool::Cleanup

Self->EnableUpdates

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; HfbsFWSTool::savePreferences
;
; PURPOSE:
;   Update and save current user preferences
;
; PARAMETERS:
;
; RETURN VALUE:
;-
function HfbsFWSTool::SavePreferences, skip_filesave=skip_filesave
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::savePreferences: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

; Save current user graphic customizations and preferences
if (Self->HasVisualizations()) then begin
   oWin = Self->GetCurrentWindow()
   oView = oWin->GetCurrentView()
   oLayer = oView->GetCurrentLayer()
   oWorld = oLayer->GetWorld()
   oDataSpace = oWorld->GetDataSpaces()
   oVis = oDataSpace->GetVisualizations(count = nVis)
   
   for j=nVis-1,0,-1 do begin
      oVis[j]->Select
      Self->saveVisProps, oVis[j]
   endfor
endif

Self->GetProperty, preferences=preferences

if keyword_set(skip_filesave) then return, 1

if (n_elements(preferences) gt 0) then $
   status = HFBSFWSPreferences(preferences, /save)

return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; HfbsFWSTool::Exit
;+
; PURPOSE:
;   Exit handler
;
; PARAMETERS:
;
; RETURN VALUE:
;-
function HfbsFWSTool::Exit
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::Exit: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wd_HfbsFWSTool_event, { WIDGET_KILL_REQUEST, ID:wTLB, TOP:wTLB, HANDLER:wTLB }

return, 1
end



;-------------------------------------------------------------------------------
pro HfbsFWSTool::GetProperty, nameTag=nameTag, prompt=prompt, prmptTitle=prmptTitle, prmptDesc=prmptDesc $
 ,monScaleFlag=monScaleFlag,nSamp=nSamp,midTempFlag=midTempFlag $
 , data_directory=dataDir, daveTool=daveTool $
 ,sampContRef=sampContRef, ftpObject=ftpObject $
 ,sampSelected=sampSelected, rebinFlag=rebinFlag, binWidth=binWidth $
 ,preferences=preferences,history=history, modifiedStatus=modifiedStatus $
 ,xAxisVar=xAxisVar,yAxisVar=yAxisVar, sampTypeFlag=sampTypeFlag $
 ,selectedIDs=selectedIDs,nSelectedIDs=nSelectedIDs, monNormFlag=monNormFlag $
 ,msdDetString=msdDetString,intDetString=intDetString, depVars=depVars, indVars=indVars, msdIndex=msdIndex $
 ,s_errorbar_capsize=s_errorbar_capsize $
 ,s_transparency=s_transparency $
 ,s_antialias=s_antialias $
 ,s_color=s_color $
 ,s_errorbar_color=s_errorbar_color $
 ,s_use_default_color=s_use_default_color $
 ,s_sym_increment=s_sym_increment $
 ,s_sym_index=s_sym_index $
 ,s_sym_size=s_sym_size $
 ,s_sym_color=s_sym_color $
 ,s_sym_thick=s_sym_thick $
 ,s_y_errorbars=s_y_errorbars $
 ,s_linestyle=s_linestyle $
 ,s_thick=s_thick $
 ,s_sym_filled=s_sym_filled $
 ,s_sym_fill_color=s_sym_fill_color $
 ,s_font_name=s_font_name  $
 ,s_font_size=s_font_size $
 ,s_text_color=s_text_color  $
 ,b_errorbar_capsize=b_errorbar_capsize $
 ,b_transparency=b_transparency $
 ,b_antialias=b_antialias $
 ,b_color=b_color $
 ,b_errorbar_color=b_errorbar_color $
 ,b_use_default_color=b_use_default_color $
 ,b_sym_increment=b_sym_increment $
 ,b_sym_index=b_sym_index $
 ,b_sym_size=b_sym_size $
 ,b_sym_color=b_sym_color $
 ,b_sym_thick=b_sym_thick $
 ,b_y_errorbars=b_y_errorbars $
 ,b_linestyle=b_linestyle $
 ,b_thick=b_thick $
 ,b_sym_filled=b_sym_filled $
 ,b_sym_fill_color=b_sym_fill_color $
 ,b_font_name=b_font_name  $
 ,b_font_size=b_font_size $
 ,b_text_color=b_text_color  $
 ,_REF_EXTRA=etc

compile_opt idl2

; daveTool
if (arg_present(daveTool)) then $
  daveTool = self.daveTool

if (arg_present(s_errorbar_capsize)) then s_errorbar_capsize=Self.s_errorbar_capsize
if (arg_present(s_transparency)) then s_transparency=Self.s_transparency
if (arg_present(s_antialias)) then s_antialias=Self.s_antialias
if (arg_present(s_color)) then s_color=Self.s_color
if (arg_present(s_errorbar_color)) then s_errorbar_color=Self.s_errorbar_color
if (arg_present(s_use_default_color)) then s_use_default_color=Self.s_use_default_color 
if (arg_present(s_sym_increment)) then s_sym_increment=Self.s_sym_increment 
if (arg_present(s_sym_index)) then s_sym_index=Self.s_sym_index 
if (arg_present(s_sym_size)) then s_sym_size=Self.s_sym_size 
if (arg_present(s_sym_color)) then s_sym_color=Self.s_sym_color 
if (arg_present(s_sym_thick)) then s_sym_thick=Self.s_sym_thick 
if (arg_present(s_y_errorbars)) then s_y_errorbars=Self.s_y_errorbars
if (arg_present(s_linestyle)) then s_linestyle=Self.s_linestyle
if (arg_present(s_thick)) then s_thick=Self.s_thick
if (arg_present(s_sym_filled)) then s_sym_filled=Self.s_sym_filled
if (arg_present(s_sym_fill_color)) then s_sym_fill_color=Self.s_sym_fill_color
if (arg_present(s_font_name)) then s_font_name=Self.s_font_name
if (arg_present(s_font_size)) then s_font_size=Self.s_font_size
if (arg_present(s_text_color)) then s_text_color=Self.s_text_color
if (arg_present(b_errorbar_capsize)) then b_errorbar_capsize=Self.b_errorbar_capsize
if (arg_present(b_transparency)) then b_transparency=Self.b_transparency
if (arg_present(b_antialias)) then b_antialias=Self.b_antialias
if (arg_present(b_color)) then b_color=Self.b_color
if (arg_present(b_errorbar_color)) then b_errorbar_color=Self.b_errorbar_color
if (arg_present(b_use_default_color)) then b_use_default_color=Self.b_use_default_color
if (arg_present(b_sym_increment)) then b_sym_increment=Self.b_sym_increment
if (arg_present(b_sym_index)) then b_sym_index=Self.b_sym_index
if (arg_present(b_sym_size)) then b_sym_size=Self.b_sym_size
if (arg_present(b_sym_color)) then b_sym_color=Self.b_sym_color
if (arg_present(b_sym_thick)) then b_sym_thick=Self.b_sym_thick
if (arg_present(b_y_errorbars)) then b_y_errorbars=Self.b_y_errorbars
if (arg_present(b_linestyle)) then b_linestyle=Self.b_linestyle
if (arg_present(b_thick)) then b_thick=Self.b_thick
if (arg_present(b_sym_filled)) then b_sym_filled=Self.b_sym_filled
if (arg_present(b_sym_fill_color)) then b_sym_fill_color=Self.b_sym_fill_color
if (arg_present(b_errorbar_capsize)) then b_errorbar_capsize=Self.b_errorbar_capsize
if (arg_present(b_font_name)) then b_font_name=Self.b_font_name
if (arg_present(b_font_size)) then b_font_size=Self.b_font_size
if (arg_present(b_text_color)) then b_text_color=Self.b_text_color


; Preferences structure
if (arg_present(preferences)) then begin
   if (ptr_valid(Self.prefsPtr)) then begin
      preferences = (*self.prefsPtr)

      ; update the preferences structure
      ntags = n_tags(preferences)
      tags = tag_names(preferences)
      for i=0,ntags-1 do begin
         status = Self->GetPropertyByIdentifier(tags[i], value)
         if (status) then preferences.(i) = strjoin(strtrim(string(value),2),',')
      endfor
   endif
endif

if (Arg_present(midTempFlag)) then midTempFlag = self.midTempFlag
; modifiedStatus
if (arg_present(modifiedStatus)) then modifiedStatus = self.modifiedFlag
; nameTag
if (arg_present(nameTag)) then nameTag = self.nameTag
; prompt
if (arg_present(prompt)) then prompt = self.prompt
; prompt Title
if (arg_present(prmptTitle)) then prmptTitle = self.promptTitle
; prompt Desc
if (arg_present(prmptDesc)) then prmptDesc = self.promptDesc
;
if (arg_present(nSamp)) then nSamp = Self.nSamp
;
if (arg_present(binWidth)) then binWidth = Self.binWidth
;
if (arg_present(rebinFlag)) then rebinFlag = Self.rebinFlag
;
if (arg_present(msdDetString)) then msdDetString = Self.msdDetString
;
if (arg_present(intDetString)) then intDetString = Self.intDetString
;
if (arg_present(depVars)) then depVars = (*Self.depVars)
;
if (arg_present(indVars)) then indVars = (*Self.indVars)
;
if (arg_present(msdIndex)) then msdIndex = Self.msdIndex
;
if (arg_present(sampSelected)) then sampSelected = Self.sampSelected
;
if (arg_present(sampContRef)) then sampContRef = Self.sampContRef
;
if (arg_present(dataDir)) then dataDir = Self.dataDir
;
;if (arg_present(oplotVanFitFlag)) then oplotVanFitFlag = Self.oplotVanFitFlag
;
;if (arg_present(oplotBkgdFlag)) then oplotBkgdFlag = Self.oplotBkgdFlag
;
if (arg_present(ftpObject)) then ftpObject = Self.ftpObject
;
;if (arg_present(sampTypeFlag)) then sampTypeFlag = Self.sampTypeFlag
;
if (arg_present(xAxisVar)) then xAxisVar = Self.xAxisVar
;
if (arg_present(yAxisVar)) then yAxisVar = Self.yAxisVar
;
if (ARG_PRESENT(nGroups)) then nGroups = Self.nGroups
;
if (ARG_PRESENT(grpNos)) then grpNos = Self.grpNos
;
if (ARG_PRESENT(monNormFlag)) then monNormFlag = Self.monNormFlag
;
if (ARG_PRESENT(detGroupingFlag)) then detGroupingFlag = Self.detGroupingFlag
;
;if (arg_present(indepVar1List)) then indepVar1List = (*Self.indepVars1Ptr)
;;
;if (arg_present(indepVar12List)) then indepVar2List = (*Self.indepVars2Ptr)
;;
;if (arg_present(indepVar2List)) then indepVar2List = (*Self.indepVars2Ptr)
;
if (arg_present(history)) then history = (*Self.historyPtr)
;
if (arg_present(detGroupings)) then detGroupings = (*Self.detGroupings)
;
if (arg_present(selectedIDs) || arg_present(nSelectedIDs)) then begin
   oUI = Self->GetUI()
   oUI->GetProperty, group_leader = wTLB
   wChild = WIDGET_INFO(wTLB, /CHILD)
   WIDGET_CONTROL, wChild, GET_UVALUE=sPtr
   ; retrieve list of all _selected_ datasets from the data manager
   wDM = (*sPtr).wDM ; the data manager widget ID
   widget_control, wDM, get_uvalue=sPtrDM ; the data manager state ptr
   selectedIDs = (*(*sPtrDM).pdataids)  ; currently selected identifiers
   nSelectedIDs = n_elements(selectedIDs)
   if (selectedIDs[0] eq '') then nSelectedIDs = 0
endif

; Call base class method
if (n_elements(etc) gt 0) then $
  self->IDLitTool::GetProperty, _EXTRA=etc

end


;-------------------------------------------------------------------------------
pro HfbsFWSTool::SetProperty, nameTag=nameTag, prompt=prompt,preferences=preferences, _EXTRA=etc $
 ,monScaleFlag=monScaleFlag,maskFlag=maskFlag,data_directory=dataDir, daveTool=daveTool $
 ,nSamp=nSamp,nBkgd=nBkgd,nVan=nVan,vanparams=vanparams, rebinFlag=rebinFlag, binWidth=binWidth $
 ,sampSelected=sampSelected, bkgdSelected=bkgdSelected, vanSelected=vanSelected, modifiedStatus=modifiedStatus $
 ,xAxisVar=xAxisVar,yAxisVar=yAxisVar, sampTypeFlag=sampTypeFlag,midTempFlag=midTempFlag $
 ,detGroupingFlag=detGroupingFlag,grpNos=grpNos, detGroupings=detGroupings $
 ,grpScaleFactors=grpScaleFactors, grpscaleFacStr=grpscaleFacStr, nGroups=nGroups, monNormFlag=monNormFlag $
 ,msdDetString=msdDetString,intDetString=intDetString, depVars=depVars, indVars=indVars, msdIndex=msdIndex $
 ,s_errorbar_capsize=s_errorbar_capsize $
 ,s_transparency=s_transparency $
 ,s_antialias=s_antialias $
 ,s_color=s_color $
 ,s_errorbar_color=s_errorbar_color $
 ,s_use_default_color=s_use_default_color $
 ,s_sym_increment=s_sym_increment $
 ,s_sym_index=s_sym_index $
 ,s_sym_size=s_sym_size $
 ,s_sym_color=s_sym_color $
 ,s_sym_thick=s_sym_thick $
 ,s_y_errorbars=s_y_errorbars $
 ,s_linestyle=s_linestyle $
 ,s_thick=s_thick $
 ,s_sym_filled=s_sym_filled $
 ,s_sym_fill_color=s_sym_fill_color $
 ,s_font_name=s_font_name  $
 ,s_font_size=s_font_size $
 ,s_text_color=s_text_color  $
 ,b_errorbar_capsize=b_errorbar_capsize $
 ,b_transparency=b_transparency $
 ,b_antialias=b_antialias $
 ,b_color=b_color $
 ,b_errorbar_color=b_errorbar_color $
 ,b_use_default_color=b_use_default_color $
 ,b_sym_increment=b_sym_increment $
 ,b_sym_index=b_sym_index $
 ,b_sym_size=b_sym_size $
 ,b_sym_color=b_sym_color $
 ,b_sym_thick=b_sym_thick $
 ,b_y_errorbars=b_y_errorbars $
 ,b_linestyle=b_linestyle $
 ,b_thick=b_thick $
 ,b_sym_filled=b_sym_filled $
 ,b_sym_fill_color=b_sym_fill_color $
 ,b_font_name=b_font_name  $
 ,b_font_size=b_font_size $
 ,b_text_color=b_text_color  $
 ,f_errorbar_capsize=f_errorbar_capsize $
 ,f_transparency=f_transparency $
 ,f_antialias=f_antialias $
 ,f_color=f_color $
 ,f_errorbar_color=f_errorbar_color $
 ,f_use_default_color=f_use_default_color $
 ,f_sym_increment=f_sym_increment $
 ,f_sym_index=f_sym_index $
 ,f_sym_size=f_sym_size $
 ,f_sym_color=f_sym_color $
 ,f_sym_thick=f_sym_thick $
 ,f_y_errorbars=f_y_errorbars $
 ,f_linestyle=f_linestyle $
 ,f_thick=f_thick $
 ,f_sym_filled=f_sym_filled $
 ,f_sym_fill_color=f_sym_fill_color $
 ,f_font_name=f_font_name  $
 ,f_font_size=f_font_size $
 ,f_text_color=f_text_color  $
 ,ff_transparency=ff_transparency $
 ,ff_antialias=ff_antialias $
 ,ff_color=ff_color $
 ,ff_use_default_color=ff_use_default_color $
 ,ff_sym_index=ff_sym_index $
 ,ff_linestyle=ff_linestyle $
 ,ff_thick=ff_thick $
 ,sb_transparency=sb_transparency $
 ,sb_antialias=sb_antialias $
 ,sb_color=sb_color $
 ,sb_use_default_color=sb_use_default_color $
 ,sb_sym_index=sb_sym_index $
 ,sb_linestyle=sb_linestyle $
 ,sb_thick=sb_thick

compile_opt idl2

if n_elements(s_errorbar_capsize) then Self.s_errorbar_capsize=s_errorbar_capsize
if n_elements(s_transparency) then Self.s_transparency=s_transparency
if n_elements(s_antialias) then Self.s_antialias=s_antialias
if n_elements(s_color) then Self.s_color=s_color
if n_elements(s_errorbar_color) then Self.s_errorbar_color=s_errorbar_color
if n_elements(s_use_default_color) then Self.s_use_default_color=s_use_default_color 
if n_elements(s_sym_increment) then Self.s_sym_increment=s_sym_increment 
if n_elements(s_sym_index) then Self.s_sym_index=s_sym_index 
if n_elements(s_sym_size) then Self.s_sym_size=s_sym_size 
if n_elements(s_sym_color) then Self.s_sym_color=s_sym_color 
if n_elements(s_sym_thick) then Self.s_sym_thick=s_sym_thick 
if n_elements(s_y_errorbars) then Self.s_y_errorbars=s_y_errorbars
if n_elements(s_linestyle) then Self.s_linestyle=s_linestyle
if n_elements(s_thick) then Self.s_thick=s_thick
if n_elements(s_sym_filled) then Self.s_sym_filled=s_sym_filled
if n_elements(s_sym_fill_color) then Self.s_sym_fill_color=s_sym_fill_color
if n_elements(s_font_name) then Self.s_font_name=s_font_name
if n_elements(s_font_size) then Self.s_font_size=s_font_size
if n_elements(s_text_color) then Self.s_text_color=s_text_color
if n_elements(b_errorbar_capsize) then Self.b_errorbar_capsize=b_errorbar_capsize
if n_elements(b_transparency) then Self.b_transparency=b_transparency
if n_elements(b_antialias) then Self.b_antialias=b_antialias
if n_elements(b_color) then Self.b_color=b_color
if n_elements(b_use_default_color) then Self.b_use_default_color=b_use_default_color
if n_elements(b_errorbar_color) then Self.b_errorbar_color=b_errorbar_color
if n_elements(b_sym_increment) then Self.b_sym_increment=b_sym_increment
if n_elements(b_sym_index) then Self.b_sym_index=b_sym_index
if n_elements(b_sym_size) then Self.b_sym_size=b_sym_size
if n_elements(b_sym_color) then Self.b_sym_color=b_sym_color
if n_elements(b_sym_thick) then Self.b_sym_thick=b_sym_thick
if n_elements(b_y_errorbars) then Self.b_y_errorbars=b_y_errorbars
if n_elements(b_linestyle) then Self.b_linestyle=b_linestyle
if n_elements(b_thick) then Self.b_thick=b_thick
if n_elements(b_sym_filled) then Self.b_sym_filled=b_sym_filled
if n_elements(b_sym_fill_color) then Self.b_sym_fill_color=b_sym_fill_color
if n_elements(b_font_name) then Self.b_font_name=b_font_name
if n_elements(b_font_size) then Self.b_font_size=b_font_size
if n_elements(b_text_color) then Self.b_text_color=b_text_color
if n_elements(f_errorbar_capsize) then Self.f_errorbar_capsize=f_errorbar_capsize
if n_elements(f_transparency) then Self.f_transparency=f_transparency
if n_elements(f_antialias) then Self.f_antialias=f_antialias
if n_elements(f_color) then Self.f_color=f_color
if n_elements(f_errorbar_color) then Self.f_errorbar_color=f_errorbar_color
if n_elements(f_use_default_color) then Self.f_use_default_color=f_use_default_color
if n_elements(f_sym_increment) then Self.f_sym_increment=f_sym_increment
if n_elements(f_sym_index) then Self.f_sym_index=f_sym_index
if n_elements(f_sym_size) then Self.f_sym_size=f_sym_size
if n_elements(f_sym_color) then Self.f_sym_color=f_sym_color
if n_elements(f_sym_thick) then Self.f_sym_thick=f_sym_thick
if n_elements(f_y_errorbars) then Self.f_y_errorbars=f_y_errorbars
if n_elements(f_linestyle) then Self.f_linestyle=f_linestyle
if n_elements(f_thick) then Self.f_thick=f_thick
if n_elements(f_sym_filled) then Self.f_sym_filled=f_sym_filled
if n_elements(f_sym_fill_color) then Self.f_sym_fill_color=f_sym_fill_color
if n_elements(f_font_name) then Self.f_font_name=f_font_name
if n_elements(f_font_size) then Self.f_font_size=f_font_size
if n_elements(f_text_color) then Self.f_text_color=f_text_color
if n_elements(ff_transparency) then Self.ff_transparency=ff_transparency
if n_elements(ff_antialias) then Self.ff_antialias=ff_antialias
if n_elements(ff_color) then Self.ff_color=ff_color
if n_elements(ff_use_default_color) then Self.ff_use_default_color=ff_use_default_color
if n_elements(ff_sym_index) then Self.ff_sym_index=ff_sym_index
if n_elements(ff_linestyle) then Self.ff_linestyle=ff_linestyle
if n_elements(ff_thick) then Self.ff_thick=ff_thick
if n_elements(sb_transparency) then Self.sb_transparency=sb_transparency
if n_elements(sb_antialias) then Self.sb_antialias=sb_antialias
if n_elements(sb_color) then Self.sb_color=sb_color
if n_elements(sb_use_default_color) then Self.sb_use_default_color=sb_use_default_color
if n_elements(sb_sym_index) then Self.sb_sym_index=sb_sym_index
if n_elements(sb_linestyle) then Self.sb_linestyle=sb_linestyle
if n_elements(sb_thick) then Self.sb_thick=sb_thick

; DAVETool
if (n_elements(daveTool) && obj_valid(daveTool) && obj_isa(daveTool,'DAVETool')) then $
  Self.daveTool = daveTool

; Preferences
if (n_elements(preferences)) then begin
   if (strcmp(size(preferences,/tname),'STRUCT')) then begin
      ntags = n_tags(preferences)
      tags = tag_names(preferences)
      index = indgen(ntags)
      
      ; handle color tags - convert string to byte array, eg '0,128,255' to [0,128,255]
      spIndex = where(strmatch(tags,'*color*',/fold) eq 1, spCnt, complement=index, ncomplement=theRest)
      if (spCnt gt 0) then begin
         for i=0,spCnt-1 do begin
            strColor = preferences.(spIndex[i])
            bytColor = fix(strsplit(strColor,',',/extract))
            Self->SetPropertyByIdentifier, tags[spIndex[i]], bytColor
         endfor
      endif

      ; all other tags
      for i=0,theRest-1 do begin
        Self->SetPropertyByIdentifier, tags[index[i]], preferences.(index[i])
      endfor
      
      ; Finally, store the preferences structure
      if (ptr_valid(Self.prefsPtr)) then (*Self.prefsPtr) = preferences $
      else Self.prefsPtr = ptr_new(preferences)
   endif
endif

if (N_elements(midTempFlag)) then Self.midTempFlag = midTempFlag
; modifiedStatus
if (n_elements(modifiedStatus)) then Self.modifiedFlag = (fix(modifiedStatus) gt 0)? 1 : 0
; nameTag
if (n_elements(nameTag)) then Self.nameTag = strtrim(nameTag,2)
; prompt
if (n_elements(prompt)) then begin
   Self.prompt = prompt
   if (prompt eq 0) then Self._bDirty = 0
endif
;
if (n_elements(rebinFlag)) then Self.rebinFlag = rebinFlag
;
if (n_elements(binWidth)) then begin
  if (binWidth le 0.0) then binWidth = 0.5
  Self.binWidth = binWidth
endif
;
if (n_elements(msdIndex)) then Self.msdIndex = msdIndex
;
if (n_elements(msdDetString)) then begin
  ; Check input and modify as needed
  if (strcmp(strtrim(msdDetString),'')) then msdDetString = '1'
  ; also, for HFBS, the detectors must range from 1 to 16 only
  void = compactString2IntVector(msdDetString,msdIndex)
  myindex = where(msdIndex ge 1 and msdIndex le 16, cnt)  ; note use of 'and' instead o the logical '&&'
  if (cnt lt 1) then begin
    msdDetString = '1'
  endif else begin
    msdIndex = msdIndex[myindex]
    status = intVector2CompactString(msdIndex, msdDetString)
  endelse
  Self.msdDetString = msdDetString
endif
;
if (n_elements(intDetString)) then begin
  ; Check input and modify as needed
  if (strcmp(strtrim(intDetString),'')) then intDetString = '1'
  ; also, for HFBS, the detectors must range from 1 to 16 only
  void = compactString2IntVector(intDetString,intIndex)
  myindex = where(intIndex ge 1 and intIndex le 16, cnt)  ; note use of 'and' instead o the logical '&&'
  if (cnt lt 1) then begin
    intDetString = '1'
  endif else begin
    intIndex = intIndex[myindex]
    status = intVector2CompactString(intIndex, intDetString)
  endelse  
  Self.intDetString = intDetString
endif
;
if (n_elements(depVars)) then (*Self.depVars) = depVars
;
if (n_elements(indVars)) then (*Self.indVars) = indVars
;
if (n_elements(monScaleFlag)) then Self.monScaleFlag = monScaleFlag
;
if (n_elements(dataDir)) then Self.dataDir = dataDir
;
if (n_elements(nSamp)) then Self.nSamp = nSamp
;
if (n_elements(sampSelected)) then Self.sampSelected = sampSelected
;
if (n_elements(xAxisVar)) then Self.xAxisVar = xAxisVar
;
if (n_elements(yAxisVar)) then begin
  Self.yAxisVar = yAxisVar
  Self->UpdatePropertySheet
endif
;
if (N_ELEMENTS(monNormFlag) gt 0) then Self.monNormFlag = monNormFlag

; Call base class method
if (n_elements(etc) gt 0) then $
  self->IDLitTool::SetProperty, _EXTRA=etc

end


;===============================================================================
; HfbsFWSTool::deleteAllDatasets
;
; PURPOSE:
;   Delete all datasets currently loaded
;
; PARAMETERS:
;
; RETURN VALUE:
;
function HfbsFWSTool::deleteAllDatasets, _EXTRA=etc
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

wDM = (*sPtr).wDM
widget_control, wDM, get_uvalue=sPtrDM

cw_HFBSFWSDataManager_DeleteData, sPtrDM, wDM, /delall, _EXTRA=etc

return, 1
end
;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::loadSamp
;
; PURPOSE:
;   Specify and load sample data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::GetFileList, title, count, fromFTP=fromFTP, filter=filter $
                                ,multiple_files=multiple_files, path=path, newPath=newPath

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

if (keyword_set(fromFTP)) then begin
  Self->GetProperty, ftpObject=oFTP
  if (n_elements(filter) eq 0) then filter = ['*.hscn','*.*']
  files = Dialog_NCNRpublicData(oFTP,title=title,filter=filter,group_leader=wTLB,count=count)
endif else begin
  if (n_elements(filter) eq 0) then filter = '*.hscn'
  if (n_elements(multiple_files) eq 0) then multiple_files=1
  if (n_elements(path) eq 0) then Self->GetProperty, data_directory=path
  if (n_elements(title) eq 0) then title='Specify files to load'
  
  files = dialog_pickfile(title=title,multiple_files=multiple_files,/must_exist, dialog_parent=wTLB $
                         ,filter=filter,path=path, get_path=newPath)
  
  if (files[0] eq '') then begin
     count = 0
     return, ''
  endif
  
  count = n_elements(files)
  ;Self->SetProperty, data_directory=newPath
endelse

return, files

end
;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::refreshPropertySheet
;
; PURPOSE:
;   Refresh the propertysheet UI
;
; PARAMETERS:
;
; KEYWORDS:
;
pro HfbsFWSTool::RefreshPropertySheet

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

widget_control, (*sPtr).wPS, /refresh_property

end
;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::loadSampFtp
;
; PURPOSE:
;   Specify and load sample data file(s) from NCNR ftp server
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::loadSampFtp

return, Self->LoadSamp(/fromFtp)

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; HfbsFWSTool::loadSamp
;
; PURPOSE:
;   Specify and load sample data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   dataObject - specifies the data object whose data is to be reloaded
;   
;   reload  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   fromFTP - set if the source of the datafile to be loaded is the FTP server
;-
function HfbsFWSTool::loadSamp, reload=reload, dataObject=oItem, fromFTP=fromFTP

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::loadSamp: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        Self->EnableUpdates
        return, 0
    endif
endif

reloading = keyword_set(reload)
if (reloading) then begin
   if (~obj_valid(oItem)) then return, 0
   oItem->GetProperty, description=files
   count = 1
   void = oItem->GetMetaData('FROMFTP',fromFtp)
   ;if (~Self->ReadData(filename, dataPtr)) then return, 0
endif else begin
   ; Let user specify files to load
   title = 'Select HSCN Files to Load'
   
   files = Self->GetFileList(title,count,fromFTP=fromFTP,newPath=newPath)
   if (count eq 0) then return, 0
   if (n_elements(newPath) gt 0) then Self->SetProperty, data_directory=newPath
endelse

; switch on the busy symbol for cursor
widget_control, /HOURGLASS

Self->DisableUpdates
; Read the contents of the data files and make entries in the 'Sample' tree of the data manager
Self->GetProperty, sampContRef=oCont, nSamp=nData

; Calculate the q-values
detAngles = [14.46,20.98,27.08,32.31,36.0,43.75,51.5,59.25,67.0,$
  74.75,82.5,90.25,98.0,105.75,113.5,121.5]
lamo = 6.27
qVals = (4.0*!pi/lamo)*sin(0.5*!dtor*detAngles)

for i=0, count-1 do begin
  
  filename = files[i]
  hfbs_readhscn,filename,time,ctemp,stemp,data,error,comments,$
     TBM,FC,header,xout,titles,$
     WBM = wbm, LIVETIME = livetime, $ ;, norm2Mon = self.norm2Mon
     ftpObject=Self.ftpObject
  if (n_elements(stemp) le 1) then begin
    Self->StatusMessage,file_basename(filename)+' contains unknown or incomplete scan! Aborting...'
    continue
  endif
  basename = file_basename(filename,'.hscn',/fold_case)
  if (float((strtok(!version.release,/extract))[0]) ge 8.4) then begin
    dataHash = hash(/fold_case) ; use fold_case keyword to make keys case insensitive for the hash
  endif else begin
    dataHash = hash()           ; fold_case keyword only for IDL 8.4 and newer
  endelse
  
  dataHash['scanMode'] = strtrim((strsplit(header[2],':',/extract))[1],2)
  dataHash['comments'] = strtrim((strsplit(header[0],':',/extract))[1],2)
  dataHash['date'] = strtrim(strjoin((strsplit(header[3],':',/extract))[1:*],':'),2)
  dataHash['users'] = strtrim((strsplit(header[1],':',/extract))[1],2)
  dataHash['timePerPoint'] = float((strsplit(header[4],':',/extract))[1])
  dataHash['Time'] = time
  dataHash['LiveTime'] = livetime
  dataHash['SampleTemp'] = reform(stemp)
  dataHash['ControlTemp'] = reform(ctemp)
  dataHash['Q'] = qVals
  dataHash['RawInt'] = reform(data[lindgen(16)+1,*])
  dataHash['IBM'] = reform(fc)
  dataHash['WBM'] = reform(wbm)
  dataHash['TBM'] = reform(tbm)
  dataHash['DataVersion'] = 1       ; used for longterm management of the contents of this data structure
  dataHash['nDet'] = 16
  dataHash['nPts'] = n_elements(stemp)
  
  if (~reloading) then begin
    ;; create new object and attach dataPtr
    oItem = obj_new('IDLitParameterSet',name=basename,description=filename,type='samp')   
    oItem->AddMetaData,'DATAHASH',ptr_new(dataHash) ; stuff the datahash as a heap var; note cannot store a hash directoy as a metadata!
    oItem->AddMetaData,'FROMFTP',keyword_set(fromFTP)
      
    ; iTools does not handle '_[0-9]' characters in identifiers well eg _0, _1, etc
    ; replace all '_' with '-'
    oItem->GetProperty, identifier=idd
    hasChanged = 0
    while ((pos = stregex(idd,'_[0-9]')) ne -1) do begin
       strput, idd,'-',pos
       hasChanged = 1
    endwhile
    if (strmatch(idd,'[0-9]*')) then begin   ; if first char is a number
        idd = 'id'+idd
        hasChanged = 1
    endif
    if (hasChanged) then oItem->SetProperty,identifier=idd 
    
    ; Add to the data manager
    Self->AddByIdentifier, 'Sample Data Manager', oItem, position=0
  endif else begin
    ;; simply replace previous dataHash with updated one
    void = oItem->GetMetaData('DATAHASH',pDataHash)
    if (ptr_valid(pDataHash)) then heap_free, pDataHash
    oItem->AddMetaData,'DATAHASH',ptr_new(dataHash)
  endelse
   
  ; Update calculations (q,e,intensity) for data
  Self->UpdateCalculation, oItem, /updateMSD
   
  if (~reloading) then begin
     oCont->Add, oItem
     Self.nSamp++


     ; Create Place holder objects for x-axis, y-axis and error to be used later for plotting
     oZ = IDLitData(name='Dependent')
     oZ->AddMetaData,'Long_name','Detector Counts'
     oZ->AddMetaData,'Units',''
     oZ->AddMetaData,'Signal', 1
     oZ->AddMetaData,'normFlag', nFlag

     oX = IDLitData(name='Independent')

     oE = IDLitData(name='Error')
     
     oCalc = IDLitData(name='Calc')

     oItem->Add, [oZ,oX,oE,oCalc], parameter_name=['Dependent','Independent','Error','Calc']

  endif
  
endfor

Self.modifiedFlag = 1
Self->EnableUpdates

return, 1
end
;-------------------------------------------------------------------------------


;===============================================================================
; HfbsFWSTool::UpdateCalculation
;
; PURPOSE:
;   Update the PSD wavevector and energy transfer for the dataset
;
; PARAMETERS:
;   oPSet [in|out] - object whose data is to be updated
;
; KEYWORDS:
;
pro HfbsFWSTool::UpdateCalculation, oPSet, updateMSD=updateMSD
compile_opt idl2

if (n_elements(updateMSD eq 0)) then updateMSD=1

oPSet->GetProperty, name=nameTag
Self->StatusMessage, 'Updating Calculation for '+nameTag

void = oPSet->GetMetaData('DATAHASH',pDataHash)
dataHash = *pDataHash
if (~obj_isa(dataHash,'hash')) then return

int = dataHash['RawInt']
s1 = size(int)
nDet = dataHash['nDet']
nPts = dataHash['nPts']

uVecx = fltarr(nDet)+1.0D
Self->GetProperty, monNormFlag=monFlag, msdDetString=msdString, intDetString=intDetString, xAxisVar = xVar
if (intDetString eq '') then intDetString='1'
void = compactString2IntVector(intDetString,detIndex)

case monFlag of
  1: begin
    normFactor = dataHash['IBM'] 
    nFlag='IBM' ; normalize to IBM - incident beam monitor
  end
  2: begin
    normFactor = dataHash['TBM']  
    nFlag='TBM' ; normalize to TBM - transmitted beam monitor
  end
  3: begin
    normFactor = dataHash['WBM']  
    nFlag='WBM' ; normalize to WBM - white beam monitor
  end
  4: begin
    normFactor = dataHash['LiveTime']  
    nFlag='Time' ; normalize to counting time
  end
  else: begin
    normFactor = fltarr(nPts) + 1.0D
    nFlag='' ; no normilzation  
  end
endcase

dataHash['normFactor'] = normFactor ; store monitor normalization factor
dataHash['normFlag'] = nFlag

; evaluate errors
error = int
index = where(error le 0.0, count)
if (count gt 0) then error[index] = 1.0
error = sqrt(error)

; normalize to monitor
normFactor = (uVecx # normFactor)
int = int / normFactor
error = error/normFactor
;; sum the intensity for the detectors specified by the user
;int = total(int[detIndex,*],1)
;error = total(error[detIndex,*],1)

; store normalized intensity and error
dataHash['Intensity'] = int
dataHash['Error'] = error

if (updateMSD) then begin
  ; Calculate MSD
  ; ===============
  
  ; the temperature
  temp = dataHash['SampleTemp']
  if (xVar eq 1) then temp = dataHash['ControlTemp']
  
  if (Self.midTempFlag) then begin
    ;; if flag is set then convert temp data to histogram from point mode
    n = n_elements(temp)
    temp = [0.5*(3*temp[0]-temp[1]),0.5*(temp[1:n-1] + temp[0:n-2])]
  endif
  
  ; Normalize the data to that of the lowest temperature and monitor
  minTemp = min(temp)
  minIndex = where(temp eq minTemp, cnt)
  minIndex = minIndex[0]
  minValues = int[*,minIndex]  ; returns the intensities at the lowest temp for each detector
  uVecy = (fltarr(nPts) + 1)
  normFactor = minValues # uVecy ; expand the same min values to all data points
  data = int/normFactor          ; and then scale the data to that of lowest temp
  derr = error/normFactor        ; and the error too
  
  ; take log of data and error
  derr = Abs(3.0/data)*derr         ; if y = ln(x) then dy = 1/x * dx
  data = -3.0*alog(data)
  void = Where(Finite(derr),void,ncomplement=nInf,complement=infIndex)  ; check for non-finite errors
  if (nInf gt 0) then derr[infIndex] = 1.0                              ; and set them to 1.0
  
  ; Loop through data points and fit 3*ln(intensity) vs Q^2 to a straight line.
  ; The fitted gradient gives <u^2> (MSD).
  xfit = (dataHash['Q'])^2    ; Q^2
  yfit = data*0.0
  chsq = temp*0.0
  msd = temp*0.0
  msdErr = msd
  if (msdString eq '') then msdString='1'
  void = compactString2IntVector(msdString,msdIndex)
  
  ; initial parameters for fit to straight line y = c + b*x
  parinfo = replicate({value:0.D, fixed:0, limited:[0,0], limits:[0.D,0]}, 2)
  parinfo[0].fixed = 1  ; fix bkgd constant
  parinfo[1].value = 1.0  ; start slope at 1.0
  
  For i = 0, nPts-1 do begin
    ;fittedParams = poly_fit(xFit[msdIndex], data[msdIndex,i], 1, measure_errors=derr[msdIndex,i], sigma=fitErr, yFit=fittedLine, chisq=csq,covar=cov,status=status)
    fittedParams = mpfitfun('pan_background',xFit[msdIndex], data[msdIndex,i], derr[msdIndex,i] $
                            ,parinfo=parinfo,perror=fitErr,yFit=fittedLine,status=status,/quiet)
;    if (status eq 0) then begin
      chsq[i] = TOTAL( (data[msdIndex,i]-fittedLine)^2 * ABS(1.0/derr[msdIndex,i]^2) )
      yfit[*,i] = fittedParams[0] + xFit * fittedParams[1]
      msd[i] = fittedParams[1]
      msdErr[i] = fitErr[1]      
;    endif
  Endfor
  
  dataHash['MSD'] = msd
  dataHash['MSDERROR'] = msdErr
  dataHash['MSDFITS'] = yFit
  dataHash['IntensityLog'] = data
  dataHash['IntensityLogError'] = derr
  dataHash['CHISQ'] = chsq
  Self->SetPropertyAttribute, 'msdIndex', valid_range=[1,nPts,1]
  Self->RefreshPropertySheet
endif

end
;-------------------------------------------------------------------------------


;===============================================================================
; HfbsFWSTool::MergeDatasets
;+
; PURPOSE:
;   Update all variables (dependent and independent) to reflect current user settings
;   Only applicable to powder samples
;
; PARAMETERS:
;   oPSet [in|out] - object whose data is to be updated
;
; KEYWORDS:
;-
function HfbsFWSTool::MergeDatasets, oPSet, merged_dataset=oDS1
compile_opt idl2

nDat = n_elements(oPSet)
if (nDat lt 2) then return, 0
oData = []
; weed out any objects that are inappropriate
for i = 0, nDat-1 do if obj_isa(oPSet[i],'IDLitParameterSet') then oData = [oData,oPSet[i]]
nDat = n_elements(oData)
if (nDat lt 2) then return, 0

; from first DS, determine whether cooling or warming scan
void = oData[0]->GetMetaData('DATAHASH',pDataHash)
dataHash = *pDataHash
stemp = dataHash['SampleTemp']
nPts = dataHash['nPts']
TDir = (stemp[nPts-1] gt stemp[0])? 'Warming' : 'Cooling'

; get the mean temp for each dataset
temp = fltarr(nDat)
for i=0, nDat-1 do begin
  void = oData[i]->GetMetaData('DATAHASH',pDataHash)
  dataHash = *pDataHash
  stemp = dataHash['SampleTemp']
  temp[i] = mean(stemp)
endfor

; if cooling, order the datasets in order of decreasing temp
; if warming, order the datasets in order of increasing temp
index = sort(temp)
if (strcmp(TDir,'Cooling')) then index = reverse(index)
oData = oData[index]

;; loop through find the largest dataset
;nLargest = 0
;iLargest = 0
;for i=0, nDat-1 do begin
;  void = oData->GetMetaData('DATAHASH',pDataHash)
;  dataHash = *pDataHash
;  if (dataHash['nPts'] gt nLarge) then begin
;    nLargest = dataHash['nPts']
;    iLargest = i
;  endif
;endfor

; use the first dataset as the base data for the merged dataset
oDS1 = oData[0]
void = oDS1->GetMetaData('DATAHASH',pDataHash)
dh1 = *pDataHash
stemp1 = dh1['SampleTemp']
nPts1 = dh1['nPts']
DS1_TDir = (stemp1[nPts1-1] gt stemp1[0])? 'Warming' : 'Cooling' 
for i=1,nDat-1 do begin
  ;if (i eq iLargest) then continue
  oDS2 = oData[i]
  void = oDS2->GetMetaData('DATAHASH',pDataHash)
  dh2 = *pDataHash
  stemp2 = dh2['SampleTemp']
  nPts2 = dh2['nPts']
  DS2_TDir = (stemp2[nPts2-1] gt stemp2[0])? 'Warming' : 'Cooling'
  
  time1 = dh1['Time']
  time2 = dh2['Time'] + dh2['timePerPoint'] + time1[nPts1-1]
  dh1['Time'] =  [time1,time2]
  if (strcmp(DS2_TDir,DS1_TDir)) then begin
    dh1['LiveTime'] = [dh1['LiveTime'],dh2['LiveTime']]
    dh1['SampleTemp'] = [dh1['SampleTemp'],dh2['SampleTemp']]
    dh1['ControlTemp'] = [dh1['ControlTemp'],dh2['ControlTemp']]
    dh1['IBM'] = [dh1['IBM'],dh2['IBM']]
    dh1['WBM'] = [dh1['WBM'],dh2['WBM']]
    dh1['nPts'] = dh1['nPts'] + dh2['nPts']
    dh1['RawInt'] = transpose([transpose(dh1['RawInt']),transpose(dh2['RawInt'])])    
  endif else begin
    dh1['LiveTime'] = [dh1['LiveTime'],reverse(dh2['LiveTime'])]
    dh1['SampleTemp'] = [dh1['SampleTemp'],reverse(dh2['SampleTemp'])]
    dh1['ControlTemp'] = [dh1['ControlTemp'],reverse(dh2['ControlTemp'])]
    dh1['IBM'] = [dh1['IBM'],reverse(dh2['IBM'])]
    dh1['WBM'] = [dh1['WBM'],reverse(dh2['WBM'])]
    dh1['nPts'] = dh1['nPts'] + dh2['nPts']
    dh1['RawInt'] = transpose([transpose(dh1['RawInt']),transpose(reverse(dh2['RawInt'],2))])    
  endelse
endfor

; Update the calculation in the merged dataset and edit the dataset's nametag
Self->UpdateCalculation, oDS1, /updateMSD
oDS1->GetProperty, name=nametag
ODS1->SetProperty, name=nameTag+'_merged'

; Finally, loop through and delete all the other individual datasets that were included in the merge
oDM = Self->GetService("DATA_MANAGER")
Self->GetProperty, nSamp=nData, sampContRef=oCont
for i=1,nDat-1 do begin
  oCont->Remove, oData[i] ; remove dataset ref. from container
  nData--
  void = oData[i]->GetMetaData('DATAHASH',pDataHash)
  heap_free, pDataHash  ; no longer needed
  void = oDM->DeleteData(oData[i]->GetFullIdentifier())   ; finally delete obj ref from data manager
  if (obj_valid(oData[i])) then obj_destroy, oData[i]   ; make sure it is really gone
endfor
Self->SetProperty, nSamp=nData  ; update nSamp

return, 1
end
;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::UpdateIntensityScale
;+
; PURPOSE:
;   Update the intensity scale - switch between linear/logbase10 scales
;
; PARAMETERS:
;   
;
; KEYWORDS:
;-
;pro HfbsFWSTool::UpdateIntensityScale
;compile_opt idl2
;
;Self->GetProperty, logFlag=logFlag, sampContRef=oCont, minIntensity=minval, maxIntensity=maxval
;
;; must be single crystal sample
;;if (Self.sampTypeFlag ne 1) then return
;
;nData = oCont->Count()
;if (nData eq 0) then return
; 
;oData = oCont->Get(/all)
;
;case logFlag of
;   0: begin ; switch to linear scale
;      minVal = 10.0^(minVal)  ; update max and min intensity to the same scale as the intensity
;      maxVal = 10.0^(maxVal)
;      for i = 0,nData-1 do begin
;         oZ = oData[i]->GetByName('Counts')
;         if (~obj_valid(oZ)) then continue
;         if (~oZ->GetMetaData('LinearCounts',counts)) then continue
;         status = oZ->SetData(counts,/no_copy);,/no_notify)      
;      endfor
;   end
;   
;   1: begin ; switch to log scale
;       minVal = (minVal gt 0.0)? alog10(minVal) : 0.0 ; update max and min intensity to the same scale as the intensity
;       maxVal = (maxVal gt 0.0)? alog10(maxVal) : 0.0 
;       for i = 0,nData-1 do begin
;         oZ = oData[i]->GetByName('Counts')
;         if (~obj_valid(oZ)) then continue
;         if (~oZ->GetMetaData('LinearCounts',counts)) then begin
;            ; probably data from an older session so create the metadata
;            status = oZ->Getdata(counts)
;            oZ->AddMetaData, 'linearCounts',counts
;         endif
;
;         index = where(counts le 0.0, nBad)
;         if (nBad gt 0) then counts[index] = !values.F_NAN
;         counts = alog10(temporary(counts))
;         status = oZ->SetData(counts,/no_copy);,/no_notify)      
;      endfor
;   
;   end
;   
;   else:
;endcase
;Self->SetProperty, minIntensity=minVal, maxIntensity=maxVal
;
;; Update the intensity range
;Self->UpdatePlotIntensityRange
;
;end
;-------------------------------------------------------------------------------


;;===============================================================================
;; HfbsFWSTool::UpdatePlotIntensityRange
;;+
;; PURPOSE:
;;   Update the max and min intensity of the existing plot
;;
;; PARAMETERS:
;;   
;;
;; KEYWORDS:
;;-
;pro HfbsFWSTool::UpdatePlotIntensityRange, oVis
;compile_opt idl2
;
;;Self->GetProperty, sampTypeFlag=sampTypeFlag
;
;; must be single crystal sample
;;if (Self.sampTypeFlag ne 1) then return
;
;if (Self.defaultIntensityFlag eq 1) then begin  ;using default intensity range
;   Self->GetProperty, selectedIDs=selIDs,nSelectedIDs=nSel
;   if (nSel eq 0) then return   ; exit if no dataset is currently selected
;   ; Retrieve intensity of selected data
;   intensity = []
;   for i=0,nSel - 1 do begin
;      oTest = Self->GetByIdentifier(selIDs[i])
;      if (~isa(oTest,'IDLitParameterSet')) then continue
;;      status = oTest->GetMetaData('sampTypeFlag',sTypeFlag) ; only consider single crystal data to Mslice
;;      if (sTypeFlag ne 1) then continue
;      oZ = oTest->GetByName('Counts')
;      status = oZ->GetData(counts)
;      intensity = [intensity,reform(counts,n_elements(counts))]
;   endfor
;   if (~isa(intensity)) then return
;   maxVal = max(intensity,min=minVal,/NAN)
;endif else begin
;   maxVal = Self.maxIntensity
;   minVal = Self.minIntensity
;endelse
;
;if (maxVal eq minVal) then return
;
;if (maxVal lt minVal) then begin
;   tempVal = maxVal
;   maxVal = minVal
;   minVal = tempVal
;endif
;
;if (obj_valid(oVis) && isa(oVis,'IDLitVisualization')) then begin
;   oVis->SetProperty, min_value=minVal, max_value=maxVal
;endif else begin
;   if (Self->HasVisualizations()) then begin
;      oWin = Self->GetCurrentWindow()
;      oView = oWin->GetCurrentView()
;      oLayer = oView->GetCurrentLayer()
;      oWorld = oLayer->GetWorld()
;      oDataSpace = oWorld->GetDataSpaces()
;      oVis = oDataSpace->GetVisualizations(count = nVis)
;   
;      for i=0, nVis-1 do oVis[i]->SetProperty, min_value=minVal, max_value=maxVal
;   endif
;endelse
;end
;;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::restoreDataVersion
;
; PURPOSE:
;   Restore the specified object to the latest version
;
; PARAMETERS:
;   oData [in|out] - object to be restored to the latest version
;
; KEYWORDS:
;
pro HfbsFWSTool::restoreDataVersion, oData
compile_opt idl2

; restore to latest IDLitData version
oData->Restore

; Pull the data pointer structure and ensure it is up to data
currentVersion = Self.dataVersion

void = oData->GetMetaData('DATAHASH',pDataHash)
dataHash = *pDataHash
if (~obj_isa(dataHash,'hash')) then return
oldDataVersion = dataHash['DataVersion']

if (currentVersion gt oldDataVersion) then begin
   ; Update the data as necessary
endif

end



;===============================================================================
; HfbsFWSTool::UpdatePropertySheet
;
; PURPOSE:
;   Update the propertysheet to reflect the current sample type
;
; PARAMETERS:
;
; KEYWORDS:
;
pro HfbsFWSTool::UpdatePropertySheet
compile_opt idl2

depVars = ['Detector Intensity','Mean Square Displacement','Examine MSD Fits','Der. of Det Intensity' $
  ,'WBM','IBM','TBM','TBM/IBM','Sample Temperature','Control Temperature']
yAxisVar = Self.yAxisVar
depVars = (*Self.depVars)
yAxis = depVars[yAxisVar]

rebinFlag = Self.rebinFlag
Self->SetPropertyAttribute, 'rebinFlag', hide=0
Self->SetPropertyAttribute, 'binWidth', hide=(rebinFlag eq 0)

case yAxis of
  'Detector Intensity': begin
    Self->SetPropertyAttribute, 'intDetString', hide=0
    Self->SetPropertyAttribute, 'msdDetString', hide=1
    Self->SetPropertyAttribute, 'msdIndex', hide=1
    Self->SetPropertyAttribute, 'monNormFlag', hide=0
  end
  
  'Der. of Det Intensity': begin
    Self->SetPropertyAttribute, 'intDetString', hide=0
    Self->SetPropertyAttribute, 'msdDetString', hide=1
    Self->SetPropertyAttribute, 'msdIndex', hide=1
    Self->SetPropertyAttribute, 'monNormFlag', hide=0
  end

  'Mean Square Displacement': begin
    Self->SetPropertyAttribute, 'intDetString', hide=1
    Self->SetPropertyAttribute, 'msdDetString', hide=0
    Self->SetPropertyAttribute, 'msdIndex', hide=1
    Self->SetPropertyAttribute, 'monNormFlag', hide=0
  end
  
  'Examine MSD Fits': begin
    Self->SetPropertyAttribute, 'intDetString', hide=1
    Self->SetPropertyAttribute, 'msdDetString', hide=0
    Self->SetPropertyAttribute, 'msdIndex', hide=0
    Self->SetPropertyAttribute, 'monNormFlag', hide=0
    Self->SetPropertyAttribute, 'rebinFlag', hide=1
    Self->SetPropertyAttribute, 'binWidth', hide=1
  end

  'TBM/IBM': begin
    Self->SetPropertyAttribute, 'intDetString', hide=1
    Self->SetPropertyAttribute, 'msdDetString', hide=1
    Self->SetPropertyAttribute, 'msdIndex', hide=1
    Self->SetPropertyAttribute, 'monNormFlag', hide=1
  end

  else: begin
    Self->SetPropertyAttribute, 'intDetString', hide=1
    Self->SetPropertyAttribute, 'msdDetString', hide=1
    Self->SetPropertyAttribute, 'msdIndex', hide=1
    Self->SetPropertyAttribute, 'monNormFlag', hide=0
  end
endcase


end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; HfbsFWSTool::Plot
;
; PURPOSE:
;   Generates a plot of the specified dataset
;
; PARAMETERS:
;   oItem   - dataset to be plotted
;   sPtr    - heap variable containing useful widget related info
;   overPlotFlag  - is overplotting on (1) or off (0)
;   lastPltFlag   - if overplotting, is this the last plot to be included
;
; KEYWORDS:
;-
pro HfbsFWSTool::Plot, oItem, sPtr, overPlotFlag, lastPlotFlag, position=pos
compile_opt idl2

if (~Self->RetrieveValues(oItem, xValue, yValue, yError, yFit $
                               , xTitle=xTitle, ytitle=yTitle, xUnits=xUnits, yUnits=yUnits, plotTitle=plotTitle $
                               , examiningFits=examiningFits)) then return
                               
; sanitize errors; replace zero with 1
index = where(yError le 0.0, zeroError)
if (zeroError gt 0) then yError[index] = 1.0

; retrieve the dependent data/error to be plotted based on the user selection
oX = oItem->GetByName('Independent',count=count)
oZ = oItem->GetByName('Dependent',count=count)
oE = oItem->GetByName('Error',count=count)
oC = oItem->GetByName('Calc',count=count)
oZ->AddMetaData,'Long_name', yTitle
oZ->AddMetaData,'Units', yUnits
oX->AddMetaData,'Long_name', xTitle
oX->AddMetaData,'Units', xUnits
ctype = partype(yValue)
status = oZ->SetData(yValue,/no_copy,/no_notify)
status = oE->SetData(yError,/no_copy,/no_notify)
status = oX->SetData(xValue,/no_copy,/no_notify)
oZ->SetProperty, type=ctype
oX->SetProperty, type=ctype
oE->SetProperty, type=ctype
if (examiningFits) then begin
  status = oC->SetData(yFit,/no_copy,/no_notify)
  oC->SetProperty, type=ctype
endif

axisHasChanged = 0
msg = []

strparms = ['Y','X','Y ERROR','X ERROR','VERTICES','PALETTE','VERTEX_COLORS']
idparms = [oZ->GetFullIdentifier() $
          ,oX->GetFullIdentifier() $
          ,oE->GetFullIdentifier() $
          ,'','','','']
visID = 'PLOT'
opID = 'Operations/Insert Visualization'
oRequester = (Self->GetByIdentifier(opID))->GetObjectInstance()
oRequester->_setTool, Self
oRequester->GetProperty, show_execution_ui = orig_showUI
oRequester->SetProperty, show_execution_ui = 0

oItem->GetProperty, name=plotName   ; retrieve the dataset name as shown in the data manager and set it as plotname
oRequester->SetProperty, name=plotName, parameter_names=strParms, data_ids=idParms, visualization_id=visID

oWin = Self->GetCurrentWindow()
oView = oWin->GetCurrentView()
activateManip = 0
rangeChange = 0

noOverplot = overplotFlag eq 0
if (Self->HasVisualizations() && noOverplot) then begin
   ;; store the active manipulator
   (*sPtr).manipID = (Self->GetCurrentManipulator())->GetFullIdentifier()
   activateManip = 1

   ;; ensure the existing visualization is selected
   oLayer = oView->GetCurrentLayer()
   oWorld = oLayer->GetWorld()
   oDataSpace = oWorld->GetDataSpaces()
   oVis = oDataSpace->GetVisualizations(count = nVis)
    
   ; retrieve current xrange and yrange to determine if user has zoomed plot
   oDataSpace->GetProperty, xrange=xr, yrange=yr
   xChange = ((*sPtr).xrange[0] ne xr[0]) || ((*sPtr).xrange[1] ne xr[1]) 
   yChange = ((*sPtr).yrange[0] ne yr[0]) || ((*sPtr).yrange[1] ne yr[1]) 
   rangeChange = xChange && yChange
   
   delID = 'Operations/Edit/Delete'
   oDelete = (Self->GetByIdentifier(delID))->GetObjectInstance()
   
   oDataSpace->Select
   void = oDelete->DoAction(Self)

   obj_destroy, [oVis,oDataSpace]
   
   ; delete any residual annotations that may have been added
   annID = oView->FindIdentifiers('*ANNOTATION LAYER*',count=nAnn)
   if (nAnn gt 2) then begin
     oVoid = []
     for i=2, nAnn-1 do begin
       oTmp = Self->GetByIdentifier(annID[i])
       oTmp->Select
       oVoid = [oVoid,oTmp]
     endfor
     void = oDelete->DoAction(Self) ; delete selected anootations
     obj_destroy, oVoid
   endif

endif

; create the new vis
oCmd = oRequester->DoAction(Self)

; add an overplot if plotFlag = 2,3,6
if (examiningFits) then begin
  ; add a line plot of the fit to the existing data plot
  idparms = [oC->GetFullIdentifier() $
    ,oX->GetFullIdentifier() $
    ,'','','','','']

  oRequester->SetProperty,parameter_names=strParms,data_ids=idParms,visualization_id=visID
  oVoid = oRequester->DoAction(Self)
endif

;; Get the newly created vis
oLayer = oView->GetCurrentLayer()
oWorld = oLayer->GetWorld()
oDataSpace = oWorld->GetDataSpaces()
oVis = oDataSpace->GetVisualizations(count = nVis)

; And Customize it
colors = [[0,0,0],[0,0,255],[0255,0,0],[0,255,0],[0,255,255],[255,0,255] $
  ,[128,255,0],[255,128,0],[0,255,128],[128,128,255],[255,128,128] $
  ,[128,255,128],[128,128,128],[255,255,0]]
nColors = N_elements(colors)/3
pos = pos mod nColors
Self->SetProperty, s_color=colors[*,pos], s_sym_color=colors[*,pos], s_use_default_color=1 $
  , s_errorbar_color=colors[*,pos], s_sym_fill_color=colors[*,pos]
if (examiningFits) then begin
  Self->CustomizeVis, oVis[nVis-2], plotName, 'SAMPPLOT'
  Self->CustomizeVis, oVis[nVis-1], plotName+'_fit', 'BKGDPLOT'
  ;plotName = [plotName,
endif else Self->CustomizeVis, oVis[nVis-1], plotName, 'SAMPPLOT'
if (noOverplot) then oDataSpace->Translate, 0.20, 0.0, 0.0 ; shift dataspace a bit to left for first plot only

;; customize axes
;oAxes = oDataSpace->GetAxes(/container)
;oAxes->SetProperty,axis_style=2  $    ; draw box axes
;                  ;,yTitle=ylabel, xTitle=xlabel $
;                  ,zTitle=zlabel

; Update Plot Title text to reflect name of dataset being plotted
; The plot title annotation was automatically created when visualization was first created so
; all we need to do is change the text string
annID = oView->Findidentifiers('*ANNOTATION LAYER*',count=nAnn)
if (nAnn gt 0) then begin
  oText = Self->Getbyidentifier(annID[1])
  ;oItem->Getproperty, name=visname
  if (Isa(oText,'IDLitvisText')) then oText->Setproperty, string=plotTitle ;plotName
endif

                  
if (activateManip) then Self->ActivateManipulator, (*sPtr).manipID
Self->RefreshCurrentWindow   ;oWin->Draw   ; force window to refresh

end
;-------------------------------------------------------------------------------



;===============================================================================
; HfbsFWSTool::saveAsASCII
;
; PURPOSE:
;   Save output intensities to file using ASCII format
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::SaveAsASCII, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for j=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[j], xVal, yVal, yErr, yFit $
    , xTitle=xTitle, ytitle=yTitle, xUnits=xUnits, yUnits=yUnits, plotTitle=plotTitle $
    , yAxisKey='Detector Intensity')) then continue
  
  if (firstTime) then begin
    firstTime = 0
    xValue = xVal
    yValue = yVal
    yError = yErr
    if (basename eq !null) then oItem[j]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
    continue
  endif
  
  xValue = [xValue,xVal]
  yValue = [yValue,yVal]
  yError = [yError,yErr]

  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xValue,ydat=yValue,yerr=yError,uniq_val=xValue[Uniq(xValue)],avgsum=0,/checkfinite
endfor

if (xValue eq !null) then return, 0
Self->Getproperty, working_directory=workDir
title = "Specify filename to store ASCII format Detector Intensities"
ext = 'txt'
filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
  ,/overwrite_prompt, file=basename+'.txt',path=workDir, get_path=new_workDir)

if (~Strcmp(filename,'')) then begin
  nd = N_elements(xvalue)
  Openw, lun, filename, /get_lun
  for i = 0,nd-1 do Printf, lun, format='(3(F11.4,2X))',xValue[i],yValue[i],yError[i]

  Free_lun, lun, /force

  if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

  msg = 'Output stored in: '+filename
  Self->Statusmessage, msg
endif

return, 1
end


;===============================================================================
; HfbsFWSTool::savePlotASCII
;
; PURPOSE:
;   Save output intensities to file using ASCII format
;
; PARAMETERS:
;
; KEYWORDS:
;
function Hfbsfwstool::SavePlotASCII, oItem
  compile_opt idl2

  ; Basic error Handler
  if (N_elements(!debug) && (!debug eq 0)) then begin
    Catch, catchError
    if (catchError ne 0) then begin
      Self->Statusmessage, !ERROR_STATE.msg
      Catch, /cancel
      Return, 0
    endif
  endif

  if (N_elements(oItem) eq 0) then begin
    ; probably got here from the main menu so try to retrieve the currently selected dataset
    Self->Getproperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
    if (nData le 0) then begin
      Self->Statusmessage, 'No data loaded currently! Please load raw data first'
      Return, 0
    endif
    if (nSel le 0) then begin
      Self->Statusmessage, 'No dataset is selected! Please select a dataset first'
      Return, 0
    endif
    oItem = Objarr(nSel)
    for i=0,nSel-1 do oItem[i] = Self->Getbyidentifier(selIDs[i]) ; get the selected dataset
  endif

  firstTime = 1
  for j=0,N_elements(oItem)-1 do begin
    if (~Self->Retrievevalues(oItem[j], xVal, yVal, yErr, yFit $
      , xTitle=xTitle, ytitle=yTitle, xUnits=xUnits, yUnits=yUnits, plotTitle=plotTitle)) then continue

    if (firstTime) then begin
      firstTime = 0
      xValue = xVal
      yValue = yVal
      yError = yErr
      if (basename eq !null) then oItem[j]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
      if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
      continue
    endif

    xValue = [xValue,xVal]
    yValue = [yValue,yVal]
    yError = [yError,yErr]

    ; Merge the selected datasets so that overlapping data are averaged
    ; the new x bins are set to the unique x values.
    Dm_step_bin,0.0,xValue,ydat=yValue,yerr=yError,uniq_val=xValue[Uniq(xValue)],avgsum=0,/checkfinite
  endfor

  if (xValue eq !null) then Return, 0
  Self->Getproperty, working_directory=workDir
  title = "Specify filename to store ASCII format Detector Intensities"
  ext = 'txt'
  filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
    ,/overwrite_prompt, file=basename+'.txt',path=workDir, get_path=new_workDir)

  if (~Strcmp(filename,'')) then begin
    nd = N_elements(xvalue)
    Openw, lun, filename, /get_lun
    for i = 0,nd-1 do Printf, lun, format='(3(F11.4,2X))',xValue[i],yValue[i],yError[i]

    Free_lun, lun, /force

    if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

    msg = 'Output stored in: '+filename
    Self->Statusmessage, msg
  endif

  Return, 1
end


;===============================================================================
; HfbsFWSTool::saveAsASCIIMSD
;
; PURPOSE:
;   Save output mean square displacements to file using ASCII format
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::SaveAsASCIIMSD, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for j=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[j], xVal, yVal, yErr, yFit $
    , xTitle=xTitle, ytitle=yTitle, xUnits=xUnits, yUnits=yUnits, plotTitle=plotTitle $
    , yAxisKey='Mean Square Displacement')) then continue
  
  if (firstTime) then begin
    firstTime = 0
    xValue = xVal
    yValue = yVal
    yError = yErr
    if (basename eq !null) then oItem[j]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
    continue
  endif

  xValue = [xValue,xVal]
  yValue = [yValue,yVal]
  yError = [yError,yErr]
  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xValue,ydat=yValue,yerr=yError,uniq_val=xValue[Uniq(xValue)],avgsum=0,/checkfinite
endfor

if (xValue eq !null) then Return, 0

Self->Getproperty, working_directory=workDir
title = "Specify filename to store ASCII format Calculated MSD"
ext = 'txt'
filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
  ,/overwrite_prompt, file=basename+'.txt',path=workDir, get_path=new_workDir)

if (~Strcmp(filename,'')) then begin
  nd = N_elements(xvalue)
  Openw, lun, filename, /get_lun
  for i = 0,nd-1 do Printf, lun, format='(3(F11.4,2X))',xValue[i],yValue[i],yError[i]

  Free_lun, lun, /force

  if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

  msg = 'Output stored in: '+filename
  Self->Statusmessage, msg
endif

return, 1
end


;===============================================================================
; HfbsFWSTool::saveAsDave
;
; PURPOSE:
;   Save output intensities to file using DAVE format
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::SaveAsDave, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for i=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[i], yAxisKey='Detector Intensity', davePtr=davePtr)) then continue
  if (~ptr_valid(davePtr)) then continue
  

  if (firstTime) then begin
    ; for the first dataset, simply store the davePtr in a new variable for later use
    Dave_beast_ops,"duplicate",davePtr,theDavePtr
    firstTime = 0
    if (basename eq !null) then oItem[i]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
    continue
  endif

  ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
  data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
  xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
  qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
  err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
  yvals = data.y

  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite

  data = {x:xvals,y:yvals,qty:qty,err:err}
  (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
endfor
if (~Ptr_valid(theDavePtr)) then Return, 0

Self->Getproperty, working_directory=workDir
title = "Specify filename to store DAVE format Detector Intensities"
ext = 'dave'
filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
  ,/overwrite_prompt, file=basename+'.dave',path=workDir, get_path=new_workDir)

if (~Strcmp(filename,'')) then begin
  davePtr = theDavePtr  ; must use the name davePtr when saving a .dave file
  Save, davePtr, filename=filename
  if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

  msg = 'Output stored in: '+filename
  Self->Statusmessage, msg
endif

Heap_free, davePtr

return, 1
end


;===============================================================================
; HfbsFWSTool::SavePlotDave
;
; PURPOSE:
;   Save output intensities to file using DAVE format
;
; PARAMETERS:
;
; KEYWORDS:
;
function Hfbsfwstool::SavePlotDave, oItem
  compile_opt idl2

  ; Basic error Handler
  if (N_elements(!debug) && (!debug eq 0)) then begin
    Catch, catchError
    if (catchError ne 0) then begin
      Self->Statusmessage, !ERROR_STATE.msg
      Catch, /cancel
      Return, 0
    endif
  endif

  if (N_elements(oItem) eq 0) then begin
    ; probably got here from the main menu so try to retrieve the currently selected dataset
    Self->Getproperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
    if (nData le 0) then begin
      Self->Statusmessage, 'No data loaded currently! Please load raw data first'
      Return, 0
    endif
    if (nSel le 0) then begin
      Self->Statusmessage, 'No dataset is selected! Please select a dataset first'
      Return, 0
    endif
    oItem = Objarr(nSel)
    for i=0,nSel-1 do oItem[i] = Self->Getbyidentifier(selIDs[i]) ; get the selected dataset
  endif

  firstTime = 1
  for i=0,N_elements(oItem)-1 do begin
    if (~Self->Retrievevalues(oItem[i], davePtr=davePtr)) then continue
    if (~Ptr_valid(davePtr)) then continue


    if (firstTime) then begin
      ; for the first dataset, simply store the davePtr in a new variable for later use
      Dave_beast_ops,"duplicate",davePtr,theDavePtr
      firstTime = 0
      if (basename eq !null) then oItem[i]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
      if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
      continue
    endif

    ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
    data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
    xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
    qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
    err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
    yvals = data.y

    ; Merge the selected datasets so that overlapping data are averaged
    ; the new x bins are set to the unique x values.
    Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite

    data = {x:xvals,y:yvals,qty:qty,err:err}
    (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
  endfor
  if (~Ptr_valid(theDavePtr)) then Return, 0

  Self->Getproperty, working_directory=workDir
  title = "Specify filename to store Plotted Data in DAVE format"
  ext = 'dave'
  filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
    ,/overwrite_prompt, file=basename+'.dave',path=workDir, get_path=new_workDir)

  if (~Strcmp(filename,'')) then begin
    davePtr = theDavePtr  ; must use the name davePtr when saving a .dave file
    Save, davePtr, filename=filename
    if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

    msg = 'Output stored in: '+filename
    Self->Statusmessage, msg
  endif

  Heap_free, davePtr

  Return, 1
end


;===============================================================================
; HfbsFWSTool::saveAsDaveMSD
;
; PURPOSE:
;   Save output intensities to file using DAVE format
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::SaveAsDaveMSD, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for i=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[i], yAxisKey='Mean Square Displacement', davePtr=davePtr)) then continue
  if (~ptr_valid(davePtr)) then continue
  
  if (firstTime) then begin
    ; for the first dataset, simply store the davePtr in a new variable for later use
    Dave_beast_ops,"duplicate",davePtr,theDavePtr
    firstTime = 0
    if (basename eq !null) then oItem[i]->Getproperty, name=basename   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(basename) le 0) || (Strtrim(basename,2) eq '')) then basename='Untitled'
    continue
  endif

  ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
  data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
  xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
  qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
  err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
  yvals = data.y

  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite

  data = {x:xvals,y:yvals,qty:qty,err:err}
  (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
endfor

if (~Ptr_valid(theDavePtr)) then Return, 0

Self->Getproperty, working_directory=workDir
title = "Specify filename to store DAVE format Calculated MSD"
ext = 'dave'
filename = Dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
  ,/overwrite_prompt, file=basename+'.dave',path=workDir, get_path=new_workDir)

if (~Strcmp(filename,'')) then begin
  davePtr = theDaveptr  ; must uset use davePtr name when saving .dave file!
  Save, theDavePtr, filename=filename
  if (File_test(new_workDir,/directory)) then Self->Setproperty, working_directory=new_workDir

  msg = 'Output stored in: '+filename
  Self->Statusmessage, msg
endif

Heap_free, theDavePtr

return, 1
end


;===============================================================================
; HfbsFWSTool::exportToDaveDM
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::exportToDaveDM, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for i=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[i], yAxisKey='Detector Intensity', davePtr=davePtr)) then continue
  if (~ptr_valid(davePtr)) then continue
    
  if (firstTime) then begin
    ; for the first dataset, simply store the davePtr in a new variable for later use
    dave_beast_ops,"duplicate",davePtr,theDavePtr
    firstTime = 0
    if (nameTag eq !null) then oItem[i]->Getproperty, name=nameTag   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(nameTag) le 0) || (Strtrim(nameTag,2) eq '')) then nameTag='Untitled'
    continue
  endif
  
  ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
  data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
  xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
  qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
  err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
  yvals = data.y

  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite

  data = {x:xvals,y:yvals,qty:qty,err:err}
  (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
endfor

if (~ptr_valid(theDavePtr)) then return, 0

; Send data to DAVE Data Manager Folder
Self.davetool->Adddaveptrtodatamanager, theDavePtr, nameTag

msg = "New extry '"+nameTag+"' created in DAVE Data Manager"
Self->Statusmessage, msg

Heap_free, theDavePtr

return, 1
end


;===============================================================================
; HfbsFWSTool::exportPlotToDM
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function Hfbsfwstool::exportPlotToDM, oItem
  compile_opt idl2

  ; Basic error Handler
  if (N_elements(!debug) && (!debug eq 0)) then begin
    Catch, catchError
    if (catchError ne 0) then begin
      Self->Statusmessage, !ERROR_STATE.msg
      Catch, /cancel
      Return, 0
    endif
  endif

  if (N_elements(oItem) eq 0) then begin
    ; probably got here from the main menu so try to retrieve the currently selected dataset
    Self->Getproperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
    if (nData le 0) then begin
      Self->Statusmessage, 'No data loaded currently! Please load raw data first'
      Return, 0
    endif
    if (nSel le 0) then begin
      Self->Statusmessage, 'No dataset is selected! Please select a dataset first'
      Return, 0
    endif
    oItem = Objarr(nSel)
    for i=0,nSel-1 do oItem[i] = Self->Getbyidentifier(selIDs[i]) ; get the selected dataset
  endif

  firstTime = 1
  for i=0,N_elements(oItem)-1 do begin
    if (~Self->Retrievevalues(oItem[i], davePtr=davePtr)) then continue
    if (~Ptr_valid(davePtr)) then continue

    if (firstTime) then begin
      ; for the first dataset, simply store the davePtr in a new variable for later use
      Dave_beast_ops,"duplicate",davePtr,theDavePtr
      firstTime = 0
      if (nameTag eq !null) then oItem[i]->Getproperty, name=nameTag   ; retrieve the dataset name as shown in the data manager and set it as plotname
      if ((N_elements(nameTag) le 0) || (Strtrim(nameTag,2) eq '')) then nameTag='Untitled'
      continue
    endif

    ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
    data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
    xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
    qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
    err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
    yvals = data.y
    
    ; Merge the selected datasets so that overlapping data are averaged
    ; the new x bins are set to the unique x values.
    Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite
    
    data = {x:xvals,y:yvals,qty:qty,err:err}
    (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
  endfor

  if (~Ptr_valid(theDavePtr)) then Return, 0

  ; Send data to DAVE Data Manager Folder
  Self.davetool->Adddaveptrtodatamanager, theDavePtr, nameTag

  msg = "New extry '"+nameTag+"' created in DAVE Data Manager"
  Self->Statusmessage, msg

  Heap_free, theDavePtr

  Return, 1
end


;===============================================================================
; HfbsFWSTool::exportToDaveDMMSD
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::exportToDaveDMMSD, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
  catch, catchError
  if (catchError ne 0) then begin
    Self->StatusMessage, !ERROR_STATE.msg
    catch, /cancel
    return, 0
  endif
endif

if (n_elements(oItem) eq 0) then begin
  ; probably got here from the main menu so try to retrieve the currently selected dataset
  Self->GetProperty, nSamp=nData, selectedIDs=selIDs,nSelectedIDs=nSel
  if (nData le 0) then begin
    Self->StatusMessage, 'No data loaded currently! Please load raw data first'
    return, 0
  endif
  if (nSel le 0) then begin
    Self->StatusMessage, 'No dataset is selected! Please select a dataset first'
    return, 0
  endif
  oItem = objarr(nSel)
  for i=0,nSel-1 do oItem[i] = Self->GetByIdentifier(selIDs[i]) ; get the selected dataset
endif

firstTime = 1
for i=0,n_elements(oItem)-1 do begin
  if (~Self->RetrieveValues(oItem[i], yAxisKey='Mean Square Displacement', davePtr=davePtr)) then continue
  if (~ptr_valid(davePtr)) then continue
  if (firstTime) then begin
    ; for the first dataset, simply store the davePtr in a new variable for later use
    Dave_beast_ops,"duplicate",davePtr,theDavePtr
    firstTime = 0
    if (nameTag eq !null) then oItem[i]->Getproperty, name=nameTag   ; retrieve the dataset name as shown in the data manager and set it as plotname
    if ((N_elements(nameTag) le 0) || (Strtrim(nameTag,2) eq '')) then nameTag='Untitled'
    continue
  endif
  
  ; For subsequent datasets, simply retrieve the data block and append it to the previous data.
  data = *(*(*theDavePtr).datastrptr).commonstr.histptr ; {x:xvals,y:yvals,qty:qty,err:err}
  xvals = [data.x,(*(*(*daveptr).datastrptr).commonstr.histptr).x]
  qty = [data.qty,(*(*(*daveptr).datastrptr).commonstr.histptr).qty]
  err = [data.err,(*(*(*daveptr).datastrptr).commonstr.histptr).err]
  yvals = data.y

  ; Merge the selected datasets so that overlapping data are averaged
  ; the new x bins are set to the unique x values.
  Dm_step_bin,0.0,xvals,ydat=qty,yerr=err,uniq_val=xvals[Uniq(xvals)],avgsum=0,/checkfinite

  data = {x:xvals,y:yvals,qty:qty,err:err}
  (*(*(*theDavePtr).datastrptr).commonstr.histptr) = data
endfor
if (~Ptr_valid(theDavePtr)) then Return, 0

; Send data to DAVE Data Manager Folder
Self.davetool->Adddaveptrtodatamanager, theDavePtr, nameTag

msg = "New extry '"+nameTag+"' created in DAVE Data Manager"
Self->Statusmessage, msg

Heap_free, theDavePtr

return, 1
end


;;===============================================================================
;; HfbsFWSTool::exportToPAN
;;
;; PURPOSE:
;;   export dataset to DAVE Data Manager
;;
;; PARAMETERS:
;;
;; KEYWORDS:
;;
;function HfbsFWSTool::ExportSelection, davePtr, nameTag=nameTag, useTexLabels=useTexLabels $
;                                    ,dave=dave, ascii=ascii
;compile_opt idl2
;
;; Basic error Handler
;if (n_elements(!debug) && (!debug eq 0)) then begin
;    catch, catchError
;    if (catchError ne 0) then begin
;        Self->StatusMessage, !ERROR_STATE.msg
;        catch, /cancel
;        return, 0
;    endif
;endif
;
;; display dialog to let user select variables to be exported to Mslice 
;Self->GetProperty, selectedIDs=selIDs,nSelectedIDs=nSel, xAxisVar=xAxisVar, indepVar2List=varList
;if (nSel eq 0) then begin
;   Self->StatusMessage, 'No dataset selected!'
;   return, 0  ; exit if no dataset is currently selected
;endif
;xvar = varList[xAxisVar]
;
;oData = []
;for i=0,nSel - 1 do begin
;   oTmp = Self->GetByIdentifier(selIDs[i])
;   status = oTmp->GetMetaData('sampTypeFlag',sTypeFlag) ; only export powder data to Mslice
;   if (sTypeFlag ne 0) then continue
;   oData = [oData,oTmp]
;endfor
;
;if (~obj_valid(oData[0])) then return, 0
;if (~Self->retrieveDavePtr(oData, xVar, davePtr, useTexLabels=useTexLabels, ascii=ascii, dave=dave)) then return, 0
;
;oData[0]->GetProperty, name = nameTag
;
;return, 1
;
;end


;===============================================================================
; HfbsFWSTool::exportToPAN
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::exportToPAN, oItem
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        return, 0
    endif
endif

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
notifyIDs = [wTLB,wTLB]
Self->GetProperty, working_directory=workDir

if (~Self->RetrieveValues(oItem, yAxisKey='MSD Fits', davePtr=davePtr, /noTexLabels)) then return, 0
if (~ptr_valid(davePtr)) then return, 0

if (ptr_valid(davePtr)) then  begin
   oVoid = obj_new('OPAN',notifyID=[wTLB,wTLB],group_leader=wTLB $
                ,davePtr=davePtr,workDir=workDir)

   heap_free, davePtr
endif

return, 1
end


;===============================================================================
; HfbsFWSTool::saveProperties
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::SaveProperties
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::saveProperties: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

print, 'properties saved...'

return, 1
end


;===============================================================================
; HfbsFWSTool::RetrieveDavePtr
;
; PURPOSE:
;   create a davePtr structure from the data in the specified dataset object
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::RetrieveValues, oData, xValue, yValue, yError, yFit, plotTitle=plotTitle $
                                    , xTitle=xTitle, ytitle=yTitle, xUnits=xUnits, yUnits=yUnits, examiningFits=examiningFits $
                                    , xAxisKey=xAxisKey, yAxisKey=yAxisKey, davePtr=davePtr, noTexLabels=noTexLabels
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        Self->StatusMessage, !ERROR_STATE.msg
        catch, /cancel
        if (ptr_valid(davePtr)) then heap_free, davePtr
        return, 0
    endif
endif
if (~obj_isa(oData,'IDLitParameterSet')) then return, 0
;if (~obj_valid(oData)) then return, 0

; Retrieve info from current property sheet seetings
Self->GetProperty, depVars=depVarList, indVars=indVarList, xAxisVar=indepVar, yAxisVar=depVar $
  ,msdIndex=dpIndex, intDetString=intDetString, msdDetString=msdDetString
dpIndex = dpIndex - 1 ; detector nos -> index
depVarName=depVarList[depVar]
indepVarName = indVarList[indepVar]

; if set, overide using info passed in by caller
if (n_elements(yAxisKey) ne 0) then depVarName = yAxisKey
if (n_elements(xAxisKey) ne 0) then indepVarName = xAxisKey

void = oData->GetMetaData('DATAHASH',pDataHash)
dataHash = *pDataHash
if (~obj_isa(dataHash,'hash')) then return, 0

; retrieve the relevent info requested
rebinFlag = Self.rebinFlag
binWidth = Self.binWidth
examiningFits = 0
plotTitle=''
case depVarName of
  'Detector Intensity': begin
    yvalue = dataHash['Intensity']
    yerror = dataHash['Error']
    status = compactString2IntVector(intDetString, intIndex)
    nDet = n_elements(intIndex)
    intIndex = IntIndex - 1 ; convert detector nos to indices
    if (nDet eq 1) then begin
      yValue = reform(yValue[intIndex,*])
      yerror = reform(yError[intIndex,*])
    endif else begin
      yValue = total(yValue[intIndex,*],1)
      yerror = sqrt(total(yError[intIndex,*]^2,1))
    endelse
    yTitle = 'Intensity'
    yUnits = 'arb. units'
    plotTitle = (nDet eq 1)? 'Detector '+intDetString : 'Detectors '+intDetString
  end

  'Mean Square Displacement': begin
    yValue = dataHash['MSD']
    yError = dataHash['MSDERROR']
    yTitle = '$\langle u^2\rangle(T) - \langle u^2\rangle(T=0)$'
    yUnits = ''
    status = compactString2IntVector(msdDetString, msdIndex)
    nDet = n_elements(msdIndex)
    plotTitle = (nDet eq 1)? 'Calculated using Detector '+msdDetString : 'Calculated using Detectors '+msdDetString
  end

  'MSD Fits': begin
    ;yFit = reform((dataHash['MSDFITS'])[*,dpIndex])
    yValue = reform(dataHash['IntensityLog'])
    yError = reform(dataHash['IntensityLogError'])
    yTitle = '$-3 ln(I(T)/I(T_{min}))$'
    yUnits = ''
    nPts = dataHash['nPts']
    status = compactString2IntVector(msdDetString, msdIndex)
    nDet = n_elements(msdIndex)
    plotTitle = (nDet eq 1)? 'Detector '+msdDetString : 'Detectors '+msdDetString
    plotTitle += ' included in fit'
    examiningFits = 1
  end

  'Examine MSD Fits': begin
    yFit = reform((dataHash['MSDFITS'])[*,dpIndex])
    yValue = reform((dataHash['IntensityLog'])[*,dpIndex])
    yError = reform((dataHash['IntensityLogError'])[*,dpIndex])
    yTitle = '$-3 ln(I(T)/I(T_{min}))$'
    yUnits = ''
    status = compactString2IntVector(msdDetString, msdIndex)
    nDet = n_elements(msdIndex)
    plotTitle = (nDet eq 1)? 'Detector '+msdDetString : 'Detectors '+msdDetString
    plotTitle += ' included in fit'
    tempTitle = 'Temp = '+string((dataHash['SampleTemp'])[dpIndex],format='(F7.3)') + ' K'
    plotTitle = [plotTitle,tempTitle]
    examiningFits = 1
    rebinFlag = 0 ; don't rebin the data
  end

  'Der. of Det Intensity': begin
    yvalue = dataHash['Intensity']
    yerror = dataHash['Error']
    status = compactString2IntVector(intDetString, intIndex)
    nDet = n_elements(intIndex)
    intIndex = IntIndex - 1 ; convert detector nos to indices
    if (nDet eq 1) then begin
      yValue = reform(yValue[intIndex,*])
      yerror = reform(yError[intIndex,*])
    endif else begin
      yValue = total(yValue[intIndex,*],1)
      yerror = sqrt(total(yError[intIndex,*]^2,1))
    endelse
    ; More to be done later after the independent axis has been determined!
  end

  'WBM': begin
    yValue = dataHash['WBM']
    yError = sqrt(yValue)
    yTitle = 'White Beam Monitor'
    yUnits = ''
  end

  'IBM': begin
    yValue = dataHash['IBM']
    yError = sqrt(yValue)
    yTitle = 'Incident Beam Monitor'
    yUnits = ''
  end

  'TBM': begin
    yValue = dataHash['TBM']
    yError = sqrt(yValue)
    yTitle = 'Transmitted Beam Monitor'
    yUnits = ''
  end

  'TBM/IBM': begin
    y1 = dataHash['TBM']
    y2 = dataHash['IBM']
    yValue = y1/y2
    yError = yValue * Quadrature(1/sqrt(y1), 1/sqrt(y2))  ; should be quadrature(sqrt(y1)/y1 , sqrt(y2)/y2) But sqrt(y1)/y1 = 1/sqrt(y1)
    yTitle = 'Transmitted/Incident Beam'
    yUnits = ''
  end

  'Sample Temperature': begin
    yValue = dataHash['SampleTemp']
    yError = sqrt(yValue)
    yTitle = 'Sample Temperature'
    yUnits = 'K'
  end

  'Control Temperature': begin
    yValue = dataHash['ControlTemp']
    yError = sqrt(yValue)
    yTitle = 'Control Temperature'
    yUnits = 'K'
  end

  else: return, 0
endcase

case indepVarName of
  'Time': begin
    xValue = dataHash['Time']/60.0    ; convert from secconds to minutes
    derT = '$\partialI/\partialt$'
    xTitle = 'Counting Time'
    xUnits = 'mins'
  end

  'Control Temperature': begin
    xValue = dataHash['ControlTemp']
    derT = '$\partialI/\partialT$'
    xTitle = 'Control Temperature'
    xUnits = 'K'
    if (Self.midtempflag) then begin
      ;; if flag is set then convert temp data to histogram from point mode
      n = N_elements(xValue)
      xValue = [0.5*(3*xValue[0]-xValue[1]),0.5*(xValue[1:n-1] + xValue[0:n-2])]
    endif
  end

  'Sample Temperature': begin
    xValue = dataHash['SampleTemp']
    derT = '$\partialI/\partialT$'
    xTitle = 'Sample Temperature'
    xUnits = 'K'
    if (Self.midtempflag) then begin
      ;; if flag is set then convert temp data to histogram from point mode
      n = N_elements(xValue)
      xValue = [0.5*(3*xValue[0]-xValue[1]),0.5*(xValue[1:n-1] + xValue[0:n-2])]
    endif
  end

  else: return, 0
endcase

if strcmp(depVarName,'Der. of Det Intensity') then begin
  ; now have both x and y to perform derivative
  der = deriv(xValue, yValue)
  derE = derivsig(xValue,yValue,0.0,yError)
  yValue = der
  yError = derE
  yTitle = derT
  yUnits = ''
  plotTitle = (nDet eq 1)? 'Detector '+intDetString : 'Detectors '+intDetString
endif
if (strcmp(depVarName,'Examine MSD Fits') || strcmp(depVarName,'MSD Fits')) then begin
  xValue = (dataHash['Q'])^2
  xTitle = '$Q^2$'
  xUnits = '$\AA^{-2}$'
endif

if (~strcmp(depVarName,'MSD Fits')) then begin
  ; don't sort for MSD Fits which is 2D data
  index = sort(xValue)
  xValue = xValue[index]
  yValue = yValue[index]
  yError = yError[index]  
  if (n_elements(yFit) gt 0) then yFit = yFit[index]
endif

if (keyword_set(noTexLabels)) then begin
   xTitle = tex2idl(xTitle)
   xUnits = tex2idl(xUnits)
   yTitle = tex2idl(yTitle)
   yUnits = tex2idl(yUnits)
endif

if (rebinFlag) then begin
  ; Locate duplicate x values within a tol=binWidth interval
  tol = Self.binWidth
  nElmts = n_elements(yValue)
  i = 0
  newx = []
  newc = []
  newe = []
  while (i lt nElmts-1) do begin
    index = i
    for j = i, nElmts-2 do begin
      diff = abs(xValue[i] - xValue[j+1])
      if (diff le tol) then begin
        index = [index,j+1]
      endif else begin
        break
      endelse
    endfor
    i = j + 1
    xmean = mean(xValue[index]) ;(moment(x[index],maxmoment=1))[0]
    newx = [newx,xmean]
    c1 = mean(yValue[index],/NAN) ;total(yValue[index],/NAN)/float(n_elements(index))
    e1 = sqrt(total(yError[index]^2,/NAN))/float(n_elements(index))
    newc = [newc,c1]
    newe = [newe,e1]
  endwhile
  lastIndex = index[n_elements(index)-1]
  if (lastIndex eq nElmts-2) then begin
    newx = [newx,xValue[nElmts-1]]
    newc = [newc,yValue[nElmts-1]]
    newe = [newe,yError[nElmts-1]]
  endif

  xValue = newx
  yValue = newc
  yError = newe
;  if (constBinFlag eq 0) then begin
;    xValue = newx
;    yValue = newc
;    yError = newe
;  endif else begin
;    Self->GetProperty, minbinWidth=tol
;    x = xmin + tol*findgen((xmax - xmin)/tol + 1)
;    drebin,newx,newc,newe,x,yValue,yError,/points,/to_points,err=err,emsg=emsg
;    if (err ne 0) then begin
;      x = newx
;      yValue = newc
;      yError = newe
;    endif
;  endelse  
endif


; convert to davePtr struct if required
if (arg_present(davePtr)) then begin
  oData->GetProperty, description=filename

  if strcmp(depVarName,'MSD Fits') then begin
    ; create a 2D davePtr
    status = create_dave_pointer(davePtr, qty=yValue, xvals=xValue, err=yError, yvals=indgen(nPts)+1)
    if (~status) then begin
      msg = "Unable to allocate memory!"
      Self->StatusMessage, msg
      if (ptr_valid(davePtr)) then heap_free, davePtr
      return, 0
    endif

    (*(*davePtr).dataStrPtr).commonStr.instrument = 'HFBS'
    (*(*davePtr).dataStrPtr).commonStr.xlabel = xTitle
    (*(*davePtr).dataStrPtr).commonStr.xunits = xunits
    (*(*davePtr).dataStrPtr).commonStr.xtype = 'points'
    (*(*davePtr).dataStrPtr).commonStr.ylabel = 'Data point nos'
    (*(*davePtr).dataStrPtr).commonStr.yunits = ''
    (*(*davePtr).dataStrPtr).commonStr.ytype = 'points'

    (*(*davePtr).dataStrPtr).commonStr.histlabel = yTitle
    (*(*davePtr).dataStrPtr).commonStr.histunits = yUnits
    (*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr) = ''
  endif else begin
    ; create a 1D davePtr
    status = create_dave_pointer(davePtr, qty=yValue, xvals=xValue, err=yError)
    if (~status) then begin
      msg = "Unable to allocate memory!"
      Self->StatusMessage, msg
      if (ptr_valid(davePtr)) then heap_free, davePtr
      return, 0
    endif

    (*(*davePtr).dataStrPtr).commonStr.instrument = 'HFBS'
    (*(*davePtr).dataStrPtr).commonStr.xlabel = xTitle
    (*(*davePtr).dataStrPtr).commonStr.xunits = xunits
    (*(*davePtr).dataStrPtr).commonStr.xtype = 'points'
    (*(*davePtr).dataStrPtr).commonStr.ylabel = ''
    (*(*davePtr).dataStrPtr).commonStr.yunits = ''
    (*(*davePtr).dataStrPtr).commonStr.ytype = 'points'

    (*(*davePtr).dataStrPtr).commonStr.histlabel = yTitle
    (*(*davePtr).dataStrPtr).commonStr.histunits = yUnits
    (*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr) = ''    
  endelse
  specificStr = { $
               originalfilename:filename, $
               scanmode:dataHash['scanMode'], $
               comments:dataHash['comments'], $
               dateofexperiment:dataHash['date'], $
               users:dataHash['users'], $
               timeperpoint:dataHash['timePerPoint'] $
              }
      
  (*(*davePtr).dataStrPtr).specificPtr = ptr_new(specificStr)
endif

return, 1
end


;===============================================================================
;+
; HfbsFWSTool::WriteProjectAs
;
; PURPOSE:
;   save project (data plus state) to file
;
; PARAMETERS:
;
; KEYWORDS:
;-
function HfbsFWSTool::WriteProjectAs
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::writeProjectAs: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

Self->GetProperty, nSamp=_nSamp, working_directory=workDir
if (_nSamp le 0) then begin
   msg = 'No data in current session so cannot proceed with save action!'
   Self->StatusMessage, msg
   Self->ErrorMessage, msg, severity=0, title='Save Unnecessary!'
   return, 0
endif


projName = file_basename(Self.projName,/fold_case,'.tisv')
oSamp = Self.sampContRef->Get(position = 0)
if (strcmp(projName,'Untitled',/fold_case) && obj_valid(oSamp)) then begin
   oSamp->GetProperty, description=desc
   projName = file_basename(desc,/fold_case,'.bt7')
   ; replace spaces in filename with _
   while ((pos = stregex(projName,' ')) ne -1) do strput, projName,'_',pos
endif
title = "Specify filename to store session"
ext = 'tisv'
filter = ['*.tisv']
projName = dialog_pickfile(title=title,filter=filter,default_extension=ext $
                          ,dialog_parent=wTLB, /write $
                          ,/overwrite_prompt, file=projName+'.tisv' $
                          ,path=workDir, get_path=new_workDir)

if (strcmp(projName,'')) then return, 0 

return, Self->writeProject(/noPrompt, new_workDir=file_dirname(projName,/mark_dir), new_projName=file_basename(projName,/fold_case,'.tisv'))

end


;===============================================================================
;+
; HfbsFWSTool::WriteProject
;
; PURPOSE:
;   save project (data plus state) to file
;
; PARAMETERS:
;
; KEYWORDS:
;-
function HfbsFWSTool::WriteProject, noPrompt=noPrompt, new_workDir=new_workDir,new_projName=new_projName 
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::writeProject: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

; save session preferences
status = Self->SavePreferences(/skip_filesave)

Self->GetProperty, nSamp=_nSamp $ ;,nBkgd=_nBkgd,nVan=_nVan $
                 , preferences=_prefStr, working_directory=workDir
noData = (_nSamp le 0) ;&& (_nBkgd le 0) && (_nVan le 0)
if (noData) then begin
   msg = 'No data in current session so cannot proceed with save action!'
   Self->StatusMessage, msg
   Self->ErrorMessage, msg, severity=0, title='Save Unnecessary!'
   return, 0
endif

projName = (keyword_set(new_projName))? new_projName : Self.projName
if (keyword_set(new_workDir)) then workDir = new_workDir

if (keyword_set(noPrompt)) then begin
   projName = workDir+projName+'.tisv'
endif else begin
   projName = Self.projName
   oSamp = Self.sampContRef->Get(position = 0)
   if (strcmp(projName,'Untitled',/fold_case) || strcmp(projName,'')) then begin
      oSamp->GetProperty, name=projName   ; retrieve the dataset name as shown in the data manager
      ;projName = file_basename(desc,/fold_case,'.hscn')
      
      ; replace spaces in projName with _
      while ((pos = stregex(projName,' ')) ne -1) do strput, projName,'_',pos

      title = "Specify filename to store session"
      ext = 'tisv'
      filter = ['*.tisv']
      projName = dialog_pickfile(title=title,filter=filter,default_extension=ext $
                                ,dialog_parent=wTLB, /write $
                                ,/overwrite_prompt, file=projName+'.tisv' $
                                ,path=workDir, get_path=new_workDir)
      ;projName = dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
      ;                       ,/overwrite_prompt, file=projName+'.tisv',path=workDir, get_path=new_workDir)

   endif else projName = workDir+projName+'.tisv'
endelse
                       
if (~strcmp(projName,'')) then begin
   if (_nSamp gt 0) then _datasets = Self.sampContRef->Get(/all)
   nd = n_elements(_datasets)
   _pdataHash = []
   for i=0,nd-1 do begin
      void = _datasets[i]->GetMetaData('DATAHASH',pDataHash)
      _pdataHash = [_pdataHash,pdataHash]
   endfor
   
   save, _datasets, _pdataHash, _nSamp, _prefStr $  ; , _nBkgd, _nVan
       , description = 'HFBS Fixed Window Scan Analysis Session' $
       , filename=projName, /compress
   if (isa(new_workDir) && file_test(new_workDir,/directory)) then Self->SetProperty, working_directory=new_workDir

   msg = 'Session stored in: '+projName
   Self->StatusMessage, msg
   
   projName = file_basename(projName,/fold_case,'.tisv')
   Self.projName = projName
   Self->SetProperty, tool_filename=projName
   
   Self.modifiedFlag = 0
endif

return, 1
end


;===============================================================================
; HfbsFWSTool::readProject
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function HfbsFWSTool::ReadProject
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::ReadProject: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        if (obj_valid(oSave)) then obj_destroy, oSave
        Self->EnableUpdates
        return, 0
    endif
endif

filter = '*.tisv'
Self->GetProperty, working_directory=workDir
if (n_elements(title) eq 0) then title='Specify session to load'

filename = dialog_pickfile(title=title,/must_exist, dialog_parent=wTLB $
                       ,filter=filter,path=workDir, get_path=newWorkDir)

if (filename eq '') then return, 0
Self->SetProperty, work_directory=newWorkDir

oSave = obj_new('IDL_Savefile', filename)
contents = oSave->Contents()
obj_destroy, oSave
expectedDesc='HFBS Fixed Window Scan Analysis Session'
if (~strcmp(contents.description,expectedDesc,/fold_case)) then begin
   msg = 'Not a valid HFBS Fixed Window Scan Analysis session file!'
   Self->StatusMessage, msg
   Self->ErrorMessage, msg, severity=1, title='Invalid file type!'   
   return, 0
endif

restore, filename, restored_objects=oObj, /relaxed_structure_assignment
nObj = n_elements(oObj)
if (nObj le 0) then return, 0

; Check whether or not to save current session before closing it
if (Self.modifiedFlag) then begin
   ; If our current state is dirty, prompt to save first.
   Self->GetProperty, working_directory=workDir
   outfile = workDir+Self.projName+'.tisv'
   status = self->PromptUserYesNo( $
       "Save changes to " + outfile + "?", $
       answer, $
       DEFAULT_NO=0, $
       TITLE='Save', $
       /CANCEL)

   if (status && (answer eq 1)) then begin
      void = Self->writeProject(/noPrompt)
   endif

endif

Self->DisableUpdates

; delete all datasets in current session
void = Self->DeleteAllDatasets(/noPrompt)

projName = file_basename(filename,/fold_case,'.tisv')
Self.projName = projName

; update preferences structure to most current definition
status = HFBSFWSPreferences(_prefStr,/updateVersion) 

Self->SetProperty, preferences=_prefStr, tool_filename=projName

; restore datasets
nd = n_elements(_datasets)
for i=0, nd-1 do begin
   oData = _datasets[i]
   Self->restoreDataVersion, oData
   oData->GetProperty, type=dataType
   case dataType of
      'samp': begin
         Self->AddByIdentifier, 'Sample Data Manager', oData, position=0
         Self.sampContRef->Add, oData
      end
      
      else:
   
   endcase
endfor
Self.nSamp = _nSamp

Self->refreshPropertySheet

;; Select a dataset and plot it
if (_nSamp gt 0) then begin
   oItem = Self.sampContRef->Get(position = 0)
   dataType='samp'
endif

if (obj_valid(oItem)) then begin
   (*sPtr).itemID = oItem->GetFullIdentifier()
   cw_HFBSFWSdatamanager_setselect, (*sPtr).wDM, (*sPtr).itemID, /clear, dataType=dataType ; select it

   wd_HfbsFWSTool_sendPlotEvent, oUI, oItem  ; plot it
endif

Self.modifiedFlag = 0
Self->EnableUpdates
return, 1
end


;---------------------------------------------------------------------------
; Lifecycle Routines
;---------------------------------------------------------------------------
; HfbsFWSTool::Init
;
; Purpose:
; The constructor of the HfbsFWSTool object.
;
; Parameters:
; None.
;
function HfbsFWSTool::Init, _REF_EXTRA=_EXTRA
compile_opt idl2

;; Call our super class
if (~self->IDLitTool::Init(_EXTRA=_extra)) then $
  return, 0

oSystem = self->_GetSystem()

; Specify current data version number
; Essentially, whevever the data structure used to store 
; datasets changes the version number should be incremented to reflect
; this. When the version number changes, restoreDataVersion() must also
; be modified to handle restoring older version datasets to the
; current version.
Self.dataVersion = 1

; Add a 'DAVEDATA_MANAGER' service
;self->AddService, OBJ_NEW("DAVEsrvDatamanager", name="DAVEDATA_MANAGER")

; Create the data manager folders
oSampFolder = Obj_New("IDLitDataManagerFolder", identifier='Sample Data Manager' $
                      ,name='FWS Datasets', description='Sample Data Manager Container',tool=self)
;oBkgdFolder = Obj_New("IDLitDataManagerFolder", identifier='Bkgd Data Manager' $
;                      ,name='Background', description='Bkgd Data Manager Container',tool=self)
;oVanFolder = Obj_New("IDLitDataManagerFolder", identifier='Van Data Manager' $
;                      ,name='Vanadium', description='Van Data Manager Container',tool=self)
;self->Add, [oSampFolder,oBkgdFolder,oVanFolder]
self->Add, oSampFolder

self->createfolders,'Operations/       ',NAME='', description='Horizontal Space'
;;---------------------------------------------------------------------
;;*** File Menu

;; create folders
self->createfolders,'Operations/File',NAME=IDLitLangCatQuery('Menu:File')

self->RegisterOperation, 'Open Session', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='File/readProject'

self->RegisterOperation, 'Save Session', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='File/writeProject'
self->RegisterOperation, 'Save Session As', $
  'DAVEopMenuItem',icon='Save_24', IDENTIFIER='File/writeProjectAs'

                                ;-----------------
Self->RegisterOperation,'Export Graphics...' $
  ,'DAVEopExportGraphics' $
  ,identifier='File/Export' $
  ,ACCELERATOR='Ctrl+E' $
  ,writertypes=['IDLDEST','IDLIMAGE'] $
  ,icon='export',/separator

self->RegisterOperation, IDLitLangCatQuery('Menu:File:PrintPreview'), $
  'IDLitopPrintPreview', $
  DESCRIPTION='Print the contents of the active window', $
  IDENTIFIER='File/PrintPreview', ICON='print'

self->RegisterOperation, IDLitLangCatQuery('Menu:File:Print'), $
  'IDLitopFilePrint', $
  ACCELERATOR='Ctrl+P', $
  DESCRIPTION='Print the contents of the active window', $
  IDENTIFIER='File/Print', ICON='print_24'


self->RegisterOperation, 'Save Preferences', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='File/savePreferences',/separator

self->RegisterOperation, 'Reset Preferences (Default Values)', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='File/resetPreferences'


                                ;-----------------
self->RegisterOperation, 'Exit', 'DAVEopMenuItem', ACCELERATOR='Ctrl+Q', $
  IDENTIFIER='File/Exit', /SEPARATOR

self->RegisterOperation, 'ExitOp', 'DAVEopFileExit', IDENTIFIER='ExitOp'


;-----------
; Data Input Menu
self->createfolders,'Operations/Input',NAME='Data Input'

;self->RegisterOperation, 'Load PSD Calibration Parameters file', $
;  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadCalibParams'

self->RegisterOperation, 'Load HSCN file(s)', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadSamp'
self->RegisterOperation, 'Load HSCN file(s) (NCNR ftp server)', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadSampFtp'

;self->RegisterOperation, 'Load Bkgd file(s)', $
;  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadBkgd', /separator
;self->RegisterOperation, 'Load Bkgd file(s) (NCNR ftp server)', $
;  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadBkgdFtp'
;
;self->RegisterOperation, 'Load Vanadium file(s)', $
;  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadVan', /separator
;self->RegisterOperation, 'Load Vanadium file(s) (NCNR ftp server)', $
;  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadVanFtp'

self->RegisterOperation, 'Delete All Loaded Data', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/deleteAllDatasets',/separator



;-----------
; Data Output Menu
self->createfolders,'Operations/Output',NAME='Data Output'
;
self->RegisterOperation, 'Save Selected Intensity as DAVE file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsDave', sensitive=0

self->RegisterOperation, 'Save Selected Intensity as ASCII file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsAscii', sensitive=0

self->RegisterOperation, 'Export Selected Intensity to DAVE Data Maneger', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/exportToDaveDM', sensitive=0

self->RegisterOperation, 'Save Selected MSD as DAVE file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsDaveMSD', sensitive=0, /separator

self->RegisterOperation, 'Save Selected MSD as ASCII file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsAsciiMSD', sensitive=0

self->RegisterOperation, 'Export Selected MSD to DAVE Data Maneger', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/exportToDaveDMMSD', sensitive=0

self->Registeroperation, 'Save Plotted Quantity as ASCII file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/savePlotAscii', sensitive=0, /separator

self->Registeroperation, 'Save Plotted Quantity as DAVE file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/savePlotDAVE', sensitive=0

self->Registeroperation, 'Export Plotted Quantity to DAVE Data Maneger', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/exportPlotToDM', sensitive=0

;;---------------------------------------------------------------------
;; Create our File toolbar container.
;;
self->Register, "Open Session", $
                PROXY='Operations/File/ReadProject', $
                IDENTIFIER='Toolbar/File/ReadProject'
self->Register, "Save Session", $
                PROXY='Operations/File/WriteProject', $
                IDENTIFIER='Toolbar/File/WriteProject'
self->Register, "Save Session As", $
                PROXY='Operations/File/WriteProjectAs', $
                IDENTIFIER='Toolbar/File/WriteProjectAs'
self->Register, IDLitLangCatQuery('Menu:File:Print'), $
                PROXY='Operations/File/Print', $
                IDENTIFIER='Toolbar/File/Print'
self->Register, IDLitLangCatQuery('Menu:File:PrintPreview'), $
                PROXY='Operations/File/PrintPreview', $
                IDENTIFIER='Toolbar/File/PrintPreview'
self->Register, IDLitLangCatQuery('Menu:File:Export'), $
                PROXY='Operations/File/Export', $
                IDENTIFIER='Toolbar/File/Export'


;;---------------------------------------------------------------------
;;*** Edit Menu

self->createfolders,'Operations/Edit',NAME=IDLitLangCatQuery('Menu:Edit')

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Undo'), $
  'IDLitopUndo', $
  ACCELERATOR='Ctrl+Z', $
  IDENTIFIER='Edit/Undo', ICON='undo', /disable, $
  /IGNORE_AVAILABILITY

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Redo'), $
  'IDLitopRedo', $
  ACCELERATOR='Ctrl+Y', $
  IDENTIFIER='Edit/Redo', ICON='redo',/disable, $
  /IGNORE_AVAILABILITY

;; Clipboard...note these are proxied from the system registry!

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Cut'), $
  PROXY="/REGISTRY/OPERATIONS/CUT",$
  IDENTIFIER='Edit/Cut'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Copy') , $
  PROXY='/REGISTRY/OPERATIONS/COPY', $
  IDENTIFIER='Edit/Copy'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Paste') , $
  PROXY='/REGISTRY/OPERATIONS/Paste', $
  IDENTIFIER='Edit/Paste'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:PasteSpecial') , $
  PROXY='/REGISTRY/OPERATIONS/PASTESPECIAL', $
  IDENTIFIER='Edit/PasteSpecial'

                                ;-----------------
self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Delete') , $
  PROXY='/REGISTRY/OPERATIONS/Delete', $
  IDENTIFIER='Edit/Delete'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:SelectAll') , $
  'IDLitopSelectAll', $
  DESCRIPTION="Select all visualizations in the current view.", $
  IDENTIFIER='Edit/SelectAll', $
  ICON='select'


                                ;-----------------
self->createfolders,'Operations/Edit/Grouping', $
                    NAME=IDLitLangCatQuery('Menu:Edit:Grouping')

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Group'), $
  'IDLitopGroup', $
  IDENTIFIER='Edit/Grouping/Group', $
  ICON='group', $
  /SEPARATOR

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Ungroup'), $
  'IDLitopUngroup', $
  IDENTIFIER='Edit/Grouping/Ungroup', $
  ICON='ungroup'


;;---------------------------------------------------------------------
;;*** Format Menu

self->createfolders,'Operations/Edit/Style', $
                    NAME=IDLitLangCatQuery('Menu:Style')

self->RegisterOperation, IDLitLangCatQuery('Menu:Style:ApplyStyle') , $
  PROXY='/REGISTRY/OPERATIONS/Apply Style', $
  IDENTIFIER='Edit/Style/Apply Style'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Style:CreateStylefromSelection'), $
  'IDLitopStyleCreate', $
  IDENTIFIER='Edit/Style/Create Style', $
  ICON='style'

self->RegisterOperation, IDLitLangCatQuery('Menu:Style:StyleEditor'), $
  'IDLitopStyleEditor', $
  IDENTIFIER='Edit/Style/StyleEditor', $
  ICON='style'

self->RegisterOperation, 'Edit Parameters','IDLitopEditParameters'

Self->RegisterOperation, 'SaveProperties','DAVEopMenuItem',identifier='Edit/SaveProperties'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Properties'), $
  'IDLitopPropertySheet', $
  DESCRIPTION='Display the Property Sheet for the item', $
  IDENTIFIER='Edit/Properties', $
  ICON='propsheet'


;;---------------------------------------------------------------------
;; Create our Edit toolbar container.
;;
self->Register, IDLitLangCatQuery('Menu:Edit:Undo'), $
                PROXY='Operations/Edit/Undo', $
                IDENTIFIER='Toolbar/Edit/Undo'
self->Register, IDLitLangCatQuery('Menu:Edit:Redo'), $
                PROXY='Operations/Edit/Redo', $
                IDENTIFIER='Toolbar/Edit/Redo'
self->Register, IDLitLangCatQuery('Menu:Edit:Cut'), $
                PROXY="/REGISTRY/OPERATIONS/CUT",$
                IDENTIFIER='Toolbar/Edit/Cut'
self->Register, IDLitLangCatQuery('Menu:Edit:Copy'), $
                PROXY='/REGISTRY/OPERATIONS/COPY', $
                IDENTIFIER='Toolbar/Edit/Copy'
self->Register, IDLitLangCatQuery('Menu:Edit:Paste'), $
                PROXY='/REGISTRY/OPERATIONS/Paste', $
                IDENTIFIER='Toolbar/Edit/Paste'


;;---------------------------------------------------------------------
;;*** Insert menu

self->createfolders,'Operations/Insert', $
                    NAME=IDLitLangCatQuery('Menu:Insert')

self->RegisterOperation, 'Insert Visualization','IDLitopInsertVis';, $
;  IDENTIFIER='Insert/Visualization', ICON='view'


self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:View'), $
  'IDLitopInsertView', $
  IDENTIFIER='Insert/View', ICON='view'

self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:DataSpace'), $
  'IDLitopInsertDataSpace';, $
  ;IDENTIFIER='Insert/Data Space', ICON='mcr'

;self->createfolders,'Operations/Insert/Axis', NAME=IDLitLangCatQuery('Menu:Insert:Axis')

self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:XAxis'), $
  'IDLitOpInsertAxisX';, IDENTIFIER='Insert/Axis/X Axis', ICON='mcr'
self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:YAxis'), $
  'IDLitOpInsertAxisY';, IDENTIFIER='Insert/Axis/Y Axis', ICON='mcr'
self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:ZAxis'), $
  'IDLitOpInsertAxisZ';, IDENTIFIER='Insert/Axis/Z Axis', ICON='mcr'


;;---------------------------------------------------------------------
;;*** Operations Menu
self->createfolders,'Operations/Operations', $
                    NAME=IDLitLangCatQuery('Menu:Operations')

;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Operations:OperationsBrowser'), $
;  'IDLitopBrowserOperation', $
;  IDENTIFIER='Operations/Operations Browser' ; ,ICON='mcr'

self->createfolders,'Operations/Operations/Macros', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Macros')

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RunMacro'), $
  PROXY='/Registry/MacroTools/Run Macro', $
  IDENTIFIER='Operations/Macros/Run Macro'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:StartRecording'), $
  PROXY='/Registry/MacroTools/Start Recording', $
  IDENTIFIER='Operations/Macros/Start Recording'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:StopRecording'), $
  PROXY='/Registry/MacroTools/Stop Recording', $
  IDENTIFIER='Operations/Macros/Stop Recording'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:MacroEditor'), $
  PROXY='/Registry/MacroTools/Macro Editor', $
  IDENTIFIER='Operations/Macros/Macro Editor'

self->createfolders,'Operations/Operations/Filter', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Filter')

;
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:Median'), $
  'IDLitopMedianFilter', $
  DESCRIPTION='Perform the median filter operation on the selected item', $
  IDENTIFIER='Operations/Filter/Median', ICON='sum'

self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:Smooth'), $
  'IDLitopSmooth', $
  DESCRIPTION='Perform the smooth operation on the selected item', $
  IDENTIFIER='Operations/Filter/Smooth', ICON='sum'

self->createfolders,'Operations/Operations/Rotate', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Rotate')

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateLeft'), $
  'IDLitopRotateLeft', $
  DESCRIPTION='Rotate left by 90 degrees', $
  IDENTIFIER='Operations/Rotate/RotateLeft', $
  ICON='rotate'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateRight'), $
  'IDLitopRotateRight', $
  DESCRIPTION='Rotate right by 90 degrees', $
  IDENTIFIER='Operations/Rotate/RotateRight', $
  ICON='rotate'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateByAngle'), $
  'IDLitopRotateByAngle', $
  DESCRIPTION='Rotate by a specified angle', $
  IDENTIFIER='Operations/Rotate/RotateByAngle', $
  ICON='rotate'


;;---------------------------------------------------------------------
;;*** Window Menu

self->createfolders,'Operations/Window', $
                    NAME=IDLitLangCatQuery('Menu:Window')

;;-----------------
Self->RegisterOperation, 'Reset to Default Data Range', 'IDLitopRangeReset' $
   ,description='Reset to default data range', identifier='Window/Resetrange' $
   ,icon='range_reset'

;;--
self->createfolders,'Operations/Toolbar/Dataspace',name='Dataspace'
self->Register, 'Reset to Default Data Range', $
                PROXY='Operations/Window/Resetrange', $
                IDENTIFIER='Toolbar/Dataspace/Resetrange'

self->createfolders,'Operations/Window/Canvas Zoom', $
                    NAME=IDLitLangCatQuery('Menu:Window:CanvasZoom')

self->RegisterOperation, '800%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/800%', $
  ICON='zoom', /CHECKED;, /SEPARATOR
self->RegisterOperation, '400%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/400%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '200%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/200%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '100%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/100%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '75%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/75%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '50%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/50%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '25%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/25%', $
  ICON='zoom', /CHECKED

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:ZoomonResize'), $
  'IDLitopZoomResize', $
  ICON='zoom', $
  IDENTIFIER='Window/ZoomResize', /CHECKED

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:Layout'), $
  'IDLitopWindowLayout', $
  IDENTIFIER='Window/Layout'

                                ;-----------------

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:FittoView'), $
  'IDLitopFitToView', $
  IDENTIFIER='Window/FitToView', $
  ICON='fitwindow', /SEPARATOR

;;---------------------------------------------------------------------
;;*** Help Menu
self->createfolders,'Operations/Help', $
                    NAME=IDLitLangCatQuery('Menu:Help')


self->RegisterOperation, "User's Manual", 'DAVEopHelp', $
  IDENTIFIER='Help/HFBSFWSManual',helptopic='HFBS_FWS'
;-----------------

self->RegisterOperation, 'About HFBS FWS', 'HFBSFWSopHelpAbout', $
  IDENTIFIER='Help/About HFBS FWS',/separator

self->RegisterOperation, 'About DAVE', 'DAVEopHelpAbout', $
  IDENTIFIER='Help/About DAVE',/separator


;;---------------------------------------------------------------------
;;*** Manipulators
self->RegisterManipulator, 'Arrow', 'IDLitManipArrow', $
  ICON='arrow', /DEFAULT, IDENTIFIER="ARROW", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Select')

self->RegisterManipulator, 'Rotate', 'IDLitManipRotate', $
  ICON='rotate', IDENTIFIER="ROTATE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Rotate')

self->RegisterManipulator, 'View Pan', 'IDLitManipViewPan', $
  ICON='hand', IDENTIFIER="VIEWPAN", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Pan')


;;---------------------------------------------------------------------
;;*** View Zoom Manipulator
self->RegisterManipulator, 'View Zoom', 'IDLitManipViewZoom', $
  IDENTIFIER='View/ViewZoom', ICON='zoom', $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:ViewZoom')

;;---------------------------------------------------------------------
;; *** View Zoom Combobox

                                ; Combobox is not available on True64 (OSF Alpha)
useCombobox = ~(!VERSION.os eq 'OSF')
self->Register, 'View Zoom', 'IDLitopViewZoom', $
                IDENTIFIER='Toolbar/View/ViewZoom', $
                DROPLIST_EDIT=useCombobox, $
                DROPLIST_ITEMS=['800%', $
                                '400%', $
                                '200%', $
                                '150%', $
                                '130%', $
                                '100%', $
                                '75%',  $
                                '50%',  $
                                '25%'], $
                DROPLIST_INDEX=4, $
                /SINGLETON

;;---------------------------------------------------------------------
;;*** Annotation Manipulators
self->RegisterManipulator, 'Text', 'IDLitAnnotateText', $
  ICON='text', IDENTIFIER="ANNOTATION/TEXT", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateText')

self->RegisterManipulator, 'Line', 'IDLitAnnotateLine', $
  ICON='line', IDENTIFIER="ANNOTATION/LINE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateLine')

self->RegisterManipulator, 'Rectangle', 'IDLitAnnotateRectangle', $
  ICON='rectangl', IDENTIFIER="ANNOTATION/RECTANGLE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateRectangle')

self->RegisterManipulator, 'Oval', 'IDLitAnnotateOval', $
  ICON='ellipse', IDENTIFIER="ANNOTATION/OVAL", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateOval')

self->RegisterManipulator, 'Polygon', 'IDLitAnnotatePolygon', $
  ICON='segpoly', IDENTIFIER="ANNOTATION/POLYGON", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotatePolygon')

self->RegisterManipulator, 'Freehand', 'IDLitAnnotateFreehand', $
  ICON='freehand', IDENTIFIER="ANNOTATION/FREEHAND", $
  DESCRIPTION='Click & drag to draw'



;;---------------------------------------------------------------------
;;*** DrawContext
;self->Register, 'Cut', $
;                PROXY="/REGISTRY/OPERATIONS/CUT",$
;                IDENTIFIER='ContextMenu/DrawContext/Cut'
;
;self->Register, 'Copy', $
;                PROXY='/REGISTRY/OPERATIONS/COPY', $
;                IDENTIFIER='ContextMenu/DrawContext/Copy'
;
;self->Register, 'Paste', $
;                PROXY='/REGISTRY/OPERATIONS/Paste', $
;                IDENTIFIER='ContextMenu/DrawContext/Paste'
;
;                                ;-----------------
self->Register, 'Delete', $
                PROXY='/REGISTRY/OPERATIONS/Delete', $
                IDENTIFIER='ContextMenu/DrawContext/Delete'

                                ;-----------------
self->Register, 'Group', $
                PROXY='Operations/Edit/Grouping/Group', $
                IDENTIFIER='ContextMenu/DrawContext/Grouping/Group'

self->Register, 'Ungroup', $
                PROXY='Operations/Edit/Grouping/Ungroup', $
                IDENTIFIER='ContextMenu/DrawContext/Grouping/Ungroup'

self->Register, 'Modify Properties...', $
                PROXY='Operations/Edit/Properties', $
                IDENTIFIER='ContextMenu/DrawContext/Properties'




Self.sampContRef = obj_new('IDL_Container') ;obj_new('IDLitContainer',name='SampContainer')

;; Initialise data and Register Properties

;Self.sampTypeFlag = 0
;Self->RegisterProperty, 'sampTypeFlag', enumlist=['Dynamics','Fixed Window'], name='Scan Type' $
;                      ,description='Scan Type', hide=1
;hideIfFWS = (Self.sampTypeFlag eq 0)? 1 : 0

Self.nSamp = 0
;Self.nBkgd = 0


; The dependent quantity to be plotted
Self.yAxisVar = 0
depVars = ['Detector Intensity','Mean Square Displacement','Examine MSD Fits','Der. of Det Intensity' $
  ,'WBM','IBM','TBM','TBM/IBM','Sample Temperature','Control Temperature']
Self->RegisterProperty, 'yAxisVar', enumlist=depVars, name='Dependent Variable'
Self.depVars = ptr_new(depVars)

; The independent quantity to be plotted
indVars = ['Time','Control Temperature','Sample Temperature']
Self.xAxisVar = 0
Self->RegisterProperty, 'xAxisVar', enumlist=indVars, name='Independent Variable'
Self.indVars = ptr_new(indVars)

Self.midTempFlag = 0
Self->Registerproperty, 'midTempFlag', enumlist=['Points','Histogram'], name='Treat Temp Data as'

; Detectors included in 'Detector Intensity' or 'MSD Calculation'
Self->RegisterProperty, 'msdDetString', /string, name='Detectors in MSD Calculation'
Self.msdDetString = '1-16'
Self->RegisterProperty, 'intDetString', /string, name='Detectors in Intensity Plot'
Self.intDetString = '1-16'

Self.msdIndex = 1
Self->RegisterProperty, 'msdIndex', /integer, valid_range=[1,10,1], name='MSD Fit: point to view'

Self.monNormFlag = 1
Self->RegisterProperty, 'monNormFlag' $
  ,enumlist=['No Normalization','Incident Beam Monitor','Transmitted Beam Monitor','White Beam Monitor','Counting Time'] $
  ,name='Normalize To?', description='Should mon normalization be applied?'

Self.rebinFlag = 0
Self.binWidth = 0.5
Self->RegisterProperty, 'rebinFlag', enumlist=['No','Yes'], name='Rebin Data?'
Self->RegisterProperty, 'binWidth', /float, name='Minimum bin width'




;Self.detGroupingFlag = 0
;Self->RegisterProperty, 'detGroupingflag', enumlist=['No','Yes'], name="Apply Detector Grouping?"
;
;Self.nGroups = 16
;detgrpings = indgen(2,16)
;detgrpings[0,*] = indgen(16)+1
;detgrpings[1,*] = indgen(16)+1
;Self.detGroupings = ptr_new(detgrpings)
;Self->RegisterProperty, 'nGroups', userdef=strtrim(string(Self.nGroups))+' Groups',name='Detector Groupings'
;
;Self.grpNos = 0
;Self.grpNosRange = [1,16,1]
;Self->RegisterProperty, 'grpNos', /integer, valid_range=Self.grpNosRange, name='Plotted Group Number', hide=1
;
;Self.bkgdFlag = 0
;Self->RegisterProperty, 'bkgdFlag', enumlist=['No','Yes'], name='Subtract Bkgd?' $
;                      ,sensitive=(Self.nBkgd gt 0)? 1 : 0, hide=1
;
;Self.vanFlag = 0
;Self->RegisterProperty, 'vanFlag', enumlist=['No','Yes'], name='Vanadium Normalization?' $
;                      ,description='Should van normalization be applied?', hide=1
;
;Self.grpScaleFactors = fltarr(16)+1
;Self->RegisterProperty, 'grpScaleFactors', userdef='', hide=1
;Self.grpscaleFacStr = strjoin(strtrim(string(Self.grpScaleFactors),2),',')
;Self->RegisterProperty, 'grpScaleFacStr', userdef=Self.grpscaleFacStr, name='(Group) Scale Factors' $
;                      ,description='Scale Factors for each group'
;
;
;Self.dynOnlyPropsPtr = ptr_new(['grpNos','bkgdFlag','vanFlag','grpScaleFacStr'])
;Self.fwsOnlyPropsPtr = ptr_new(['xAxisVar','yAxisVar'])
;
;
;
;
;;
;;Self.defaultIntensityFlag = 1
;;Self->RegisterProperty, 'defaultIntensityFlag', enumlist=['No','Yes'], name='Use Default Intensity Range?' $
;;                      ,description='Use Default Intensity Range?' ,hide=hideifpowder
;;Self.maxIntensity = 0.0
;;Self->RegisterProperty, 'maxIntensity', /float, name='Max Intensity Value' $
;;                      ,description='Max Intensity Value' ,hide=hideifpowder
;;Self.minIntensity = 0.0
;;Self->RegisterProperty, 'minIntensity', /float, name='Min Intensity Value' $
;;                      ,description='Min Intensity Value' ,hide=hideifpowder
;;
;;
;;Self.normalizeTo = 0
;;Self->RegisterProperty, 'normalizeTo', enumlist=['Monitor','Counting Time','No Normalization'], name='Normalize To' $
;;                      ,description='Normalize Counts To'
;;

Self->SetPropertyAttribute, 'NAME', /hide
Self->SetPropertyAttribute, 'DESCRIPTION', /hide
Self.projName = 'Untitled'

;; Finally, set the prompt property to 1 meaning display a warning dialog 
;; before deleting this tool
Self.prompt = 1
Self.promptDesc = 'HFBS Fixed Window Scan Analysis'
Self.promptTitle = 'Delete Main Window?'

IDLge90 = (float(!version.release) ge 9.0)? 1 : 0
Self.ftpObject =  (IDLge90)? DAVEHttpRequest() :  DAVEftpURL()
Self.ftpObject->SetProperty, currentDir = '/pub/ncnrdata/hfbs/'

Self.historyPtr = ptr_new('')

;Self.ekFac = 2.072194

return, 1

end


;-------------------------------------------------------------------------
; HfbsFWSTool::SaveVisProps
;+
; Purpose:
;    Retrieve the curent vis properties and save them
;
; Parameters:
;    oVis     - The visualization whose properties are to be saved
;    
;-
pro HfbsFWSTool::SaveVisProps, oVis
compile_opt idl2

if (~obj_valid(oVis)) then return
if (~obj_isa(oVis,'IDLitVisualization')) then return

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'HfbsFWSTool::SaveVisProps: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return
    endif
endif

oVis->GetProperty, description=plotDesc
case strupcase(plotDesc) of
   'SAMPPLOT': prefix = 's_'
   'BKGDPLOT': prefix = 'b_'
;   'FASTPLOT': prefix = 'f_'
;   'SAMPBKGDPLOT': prefix = 'sb_'
;   'FASTFITPLOT': prefix = 'ff_'
   else: return
endcase

switch strupcase(plotDesc) of
   'SAMPPLOT':
   'BKGDPLOT':
   'FASTPLOT': begin
      ;; store the vis props
      vTags = ['use_default_color','color','errorbar_color','sym_color','errorbar_capsize' $
              ,'sym_increment','sym_index','sym_size','sym_thick','y_errorbars','antialias' $
              ,'linestyle','nsum','thick','sym_filled','sym_fill_color','transparency']
;      vTags = ['errorbar_capsize','transparency','antialias','color','use_default_color' $
;              ,'sym_increment','sym_index','sym_size','sym_color','sym_thick','y_errorbars' $
;              ,'linestyle','nsum','thick','sym_filled','sym_fill_color']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = oVis->GetPropertyByIdentifier(vTags[i], value)
         if (status) then etc = create_struct(etc,tTags[i],value)
      endfor
      Self->SetProperty, _EXTRA=etc

      ;; store the axes props
      vTags = ['font_name','font_size','text_color']
      tTags = prefix + vTags
      nTags = n_elements(vTags)
      oDataSpace = oVis->GetDataSpace()
      oAxes = oDataSpace->GetAxes(count=naxes)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = oAxes[0]->GetPropertyByIdentifier(vTags[i], value)
         if (status) then etc = create_struct(etc,tTags[i],value)
      endfor
      Self->SetProperty, _EXTRA=etc   
   
      break
   end
   
;   'SAMPBKGDPLOT':
;   'FASTFITPLOT': begin
;      ;; Set the vis props
;      vTags = ['transparency','antialias','color','use_default_color','sym_index' $
;              ,'linestyle','thick']
;      tTags = prefix + vTags
;
;      nTags = n_elements(vTags)
;      etc = {dummy:''}
;      for i=0, nTags-1 do begin
;         status = oVis->GetPropertyByIdentifier(vTags[i], value)
;         if (status) then etc = create_struct(etc,tTags[i],value)
;      endfor
;      Self->SetProperty, _EXTRA=etc
;   
;      break
;   end
   
   else:
endswitch

end




;-------------------------------------------------------------------------
; HfbsFWSTool::customizeVis
;+
; Purpose:
;    Customize a newly created visualization using attributes obtained from the tool
;
; Parameters:
;    oVis     - The visualization to be customized
;    
;    plotname - a string name to be used to identify the visualization
;-
pro HfbsFWSTool::CustomizeVis, oVis, plotName, plotDesc
compile_opt idl2

if (~obj_valid(oVis)) then return
if (~obj_isa(oVis,'IDLitVisualization')) then return

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_HfbsFWSTool::customizeVis: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return
    endif
endif

case strupcase(plotDesc) of
   'SAMPPLOT': prefix = 's_'
   'BKGDPLOT': prefix = 'b_'
   else: return
endcase

switch strupcase(plotDesc) of
   'SAMPPLOT':
   'BKGDPLOT': begin
      ;; Set the vis props
      vTags = ['use_default_color','color','errorbar_color','sym_color','errorbar_capsize' $
              ,'sym_increment','sym_index','sym_size','sym_thick','y_errorbars','antialias' $
              ,'linestyle','nsum','thick','sym_filled','sym_fill_color','transparency']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = []  ;{dummy:''}
      for i=0, nTags-1 do begin
         status = Self->GetPropertyByIdentifier(tTags[i], value)
         if (status) then etc = create_struct(etc,vTags[i],value)
      endfor

      if (float((strtok(!version.release,/extract))[0]) lt 8.0) then begin
         ; workaround for bug in IDL 7.1 or lower; use_default_color is not honoured for errorbar_color
         if (etc.use_default_color eq 1) then etc.errorbar_color = etc.color
      endif

      oVis->SetProperty, name=strupcase(plotName), description=plotDesc, _EXTRA=etc
      oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account

      ;; Set the axes props
      vTags = ['font_name','font_size','text_color']
      tTags = prefix + vTags
      nTags = n_elements(vTags)
      oDataSpace = oVis->GetDataSpace()
      oAxes = oDataSpace->GetAxes(count=naxes)
      etc = []    ;{dummy:''}
      for i=0, nTags-1 do begin
         status = Self->GetPropertyByIdentifier(tTags[i], value)
         if (status) then etc = create_struct(etc,vTags[i],value)
      endfor
      for i=0,naxes-1 do oAxes[i]->SetProperty, _EXTRA=etc   
   
      break
   end
   
   else:
endswitch



end



;---------------------------------------------------------------------------
; HfbsFWSTool::InitPreferences
;
; Purpose:
;   This method
;
pro HfbsFWSTool::InitPreferences
compile_opt idl2

; Init working and data directory from the DAVETool values
;daveTool = Self.daveTool
;if (obj_valid(daveTool) and obj_isa(daveTool,'DAVETool')) then begin
;   daveTool->GetProperty, data_directory=datDir, working_directory=workDir
;   Self->SetProperty, data_directory=datDir, working_directory=workDir
;endif


; Retrieve user settings on disk and initialise current session
status = HFBSFWSPreferences(preferences)
if (status) then Self->SetProperty, preferences=preferences

;Self->GetProperty, maskFlag=mF, vanFlag=fF, xAxisVar=xAxisVar $
;                 , sampTypeFlag=sampTypeFlag
;noHide = (defIntFlag eq 0) && (sampTypeFlag eq 1)  ; don't hide if not using default intensity and single crystal sample
;Self->SetPropertyAttribute, 'maxIntensity', hide=~noHide
;Self->SetPropertyAttribute, 'minIntensity', hide=~noHide
;Self->SetPropertyAttribute, 'MASKEDDETS', sensitive=mF
;Self->SetPropertyAttribute, 'FASTVALUE', sensitive=fF
;Self->SetPropertyAttribute, 'MONCORFLAG', sensitive=fF
;Self->SetPropertyAttribute, 'RESCORFLAG', sensitive=mF
;Self->SetPropertyAttribute, 'DETAILEDBALANCEFLAG', sensitive=mF

; For safety, always initialise phiOffset to 0.0 b/c a user may not be fully aware
; about this setting. A value of 0.0 is harmless
;Self->SetProperty, phiOffset = 0.0

Self->refreshPropertySheet

; Update title to reflect session name
; setting tool_filename generates a callback on the registered UI callback
Self->SetProperty, tool_filename=Self.projName

; Retrieve then graphics window
; and make changes to the auto_resize and zoom-on-resize settings
oWin = Self->GetCurrentWindow()
oWin->SetProperty, auto_resize=1, zoom_on_resize=1

if (float((strtok(!version.release,/extract))[0]) ge 8.0) then begin
   ; For IDL 8.0 and newer, a title text is added to the graphics by default
   ; Hide it!

   idText = iGetID('*title*',tool=Self)
   oText = Self->GetByIdentifier(idText[0])
   if (isa(oText,'IDLitvisText')) then oText->setProperty, hide=0
   ;oWin = Self->GetCurrentWindow()
   ;oScene = oWin->GetScene()
   ;oView = oScene->GetCurrentView()
   ;atoms = oView->GetAll(count=natoms)
   ;for i=0,natoms-1 do begin
   ;   if (obj_isa(atoms[i],'IDLitvisText')) then atoms[i]->setProperty, hide=1
   ;endfor
   Self->RefreshCurrentWindow
endif

end


;---------------------------------------------------------------------------
; HfbsFWSTool::resetPreferences
;
; Purpose:
;   This method
;
function HfbsFWSTool::ResetPreferences
compile_opt idl2


; Retrieve user settings on disk and initialise current session
status = HFBSFWSPreferences(preferences,/reset)
if (status) then Self->SetProperty, preferences=preferences

;Self->GetProperty, maskFlag=mF, vanFlag=fF
;
;Self->SetPropertyAttribute, 'MASKEDDETS', sensitive=mF
;Self->SetPropertyAttribute, 'FASTVALUE', sensitive=fF
;Self->SetPropertyAttribute, 'MONCORFLAG', sensitive=fF
;Self->SetPropertyAttribute, 'RESCORFLAG', sensitive=mF
;Self->SetPropertyAttribute, 'DETAILEDBALANCEFLAG', sensitive=mF

Self->UpdatePropertySheet
Self->refreshPropertySheet

end


;---------------------------------------------------------------------------
; HfbsFWSTool::EditUserDefProperty
;
; Purpose:
;   This method
;
function HfbsFWSTool::EditUserDefProperty, oTool, PropID
compile_opt idl2

;; The API requires the oTool arg. It is redundant here since it is Self!

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

case PropID of
  'CALIBPARAMSBASEFILENAME': begin
    status = Self->loadCalibParams()
  end

;  'CHDELEFBASEFILENAME': begin
;    status = Self->loadChEf()
;    break
;  end
;
;  'CHEFFBASEFILENAME': begin
;    status = Self->loadChEfficiency()
;    break
;  end

  'MASKEDDETS': begin
     ; The currently selected dataset is registered as (*sPtr).itemID
     noSelection = strcmp(strtrim((*sPtr).itemID),'')
     if (noselection) then begin
       ; retrieve the first loaded dataset, if any
       Self->GetProperty, sampContRef=oCont
       oItem = oCont->Get(position=0,count=nItem)
     endif else begin
       oItem = Self->GetByIdentifier((*sPtr).itemID)
       nItem = 1
     endelse
     if (nItem ne 1) then return, 0
     
     wd_HFBSMaskChannels, oUI, oItem
  end
  
   
  else:
endcase

Self->refreshPropertySheet

return, 1
end


;---------------------------------------------------------------------------
; HfbsFWSTool__Define
;
; Purpose:
;   This method defines the HfbsFWSTool class.
;
pro HfbsFWSTool__Define
compile_opt idl2

void = { HfbsFWSTool,                 $
         inherits IDLitTool        $   ; Provides iTool interface
         ,dataDir:''               $   ; data directory (Note: working_directory is defined by the base class
         ,nSamp:0                  $   ; nos of valid Sample datasets
;         ,nBkgd:0                  $   ; nos of valid bkgd datasets
;         ,nVan:0                   $   ; nof of valid van bkgd datasets
         ,sampContRef:obj_new()    $   ; container for Sample datasets
         ,sampSelected:obj_new()   $   ; most recent selection from samp datasets
;         ,bkgdSelected:obj_new()   $   ; most recent selection from bkgd datasets
         ,rebinFlag:0              $   ; whether or not to rebin the data vs temp
         ,binWidth:0.0             $   ; bin width or interval
         ,DAVETool:obj_new()       $   ; DAVETool object
         ,projName:''              $   ; name of current project
         ,nameTag:''               $   ; Tag to be used for identification purposes
         ,prompt:1                 $
         ,promptTitle:''           $
         ,promptDesc:''            $
;         ,sumSampFlag:0            $   ; sum multiple sample files? 0=no, 1=yes
;         ,sumBkgdFlag:0            $   ; sum multiple bkgd files? 0=no, 1=yes
;         ,qUnits:0                 $   ; Units for QX, QY variables: 0->1/A, 1->r.l.u
         ,xAxisVar:0               $   ; Variable to plotted as the x-axis
         ,yAxisVar:0               $   ; variable to be plotted as the y-axis
;         ,indepVars1Ptr:ptr_new()  $   ; list of valid sxtal independent variables ('QX','QY','E','|Q|','TEMP','MAGFIELD')
;         ,indepVars2Ptr:ptr_new()  $   ; list of valid powder x-axis independent variables ('2THETA','Q','DSPACE','TEMP','MAGFIELD','DATAPOINT')
;         ,indepVars3Ptr:ptr_new()  $   ; list of valid powder y-axis independent variables ('2THETA','Q','DSPACE')
;         ,nGroups:16               $   ; number of detector groups
;         ,detGroupingFlag:0        $   ; apply detector grouping
;         ,detGroupings:ptr_new()   $   ; detector groupings info
;         ,grpNos:0                 $   ; current group to plot
;         ,grpNosRange:[1,16,1]     $   ; range of valid group numbers
;         ,detailedBalanceFlag:0    $   ; apply detailed balance correction 1=yes, 0=no
;         ,bkgdFlag:0               $   ; subtract bkgd data  1=yes, 0=no
;         ,vanFlag:0                $   ; apply van normilization? 1=yes, 0=no
         ,monNormFlag:0            $   ; apply mon normilization? 1=yes, 0=no
         ,midTempFlag:0            $   ; should the temperature be that between the data point? Corrects a bit for sample temperature lag!
;         ,sampTypeFlag:0           $   ; sample type: 1:single crystal, 0:powder
;         ,grpScaleFactors:fltarr(16) $ ; 
;         ,grpscaleFacStr:''        $
         ,prefsPtr:ptr_new()       $   ; points to preference structure
         ,historyPtr:ptr_new()     $
         ,modifiedFlag:0B          $   ; determines modification state of tool - unsufficient control over _bDirty property!
         ,dataVersion:0            $   ; current version number for dataset structure
         ,ftpObject:obj_new()      $   ; DAVEftpURL() object used to access data on ftp://ftp.ncnr.nist.gov/pub/ncnrdata/
;         ,chEff:fltarr(16)         $   ; Channel efficiencies
         ,logFlag:0                $   ; use linear (0) or log (1) intensity scale
;         ,ekFac:2.072194           $   ; Energy = ekFac * k^2
;         ,dynOnlyPropsPtr:ptr_new() $   ;
;         ,fwsOnlyPropsPtr:ptr_new() $  ;
         ,msdDetString:''          $   ; specifies the detectors to be used in the MSD calculation
         ,intDetString:''          $   ; specifies the detectors to be used in the detector intensity plot
         ,depVars:ptr_new()        $   ; list of dependent variables
         ,indVars:ptr_new()        $   ; list of independent variables
         ,msdIndex:0               $   ; the data point to be viewd when MSD Fits are on display

         ,s_errorbar_capsize:    0.1 $
         ,s_transparency:        50 $
         ,s_antialias:           3 $
         ,s_color:               [0,0,255] $
         ,s_errorbar_color:      [0,0,255] $
         ,s_use_default_color:   1   $
         ,s_sym_increment:       1   $
         ,s_sym_index:           4   $
         ,s_sym_size:            1.0   $          ; 
         ,s_sym_color:           [0,0,255] $   ;
         ,s_sym_thick:           1.0 $            ;
         ,s_y_errorbars:         1 $              ; show y errors
         ,s_linestyle:           0 $              ; 
         ,s_nsum:                1 $              ; no point everaging
         ,s_thick:               2.0 $            ; line thickness
         ,s_sym_filled:          1   $            ; use filled symbols
         ,s_sym_fill_color:      [0,0,255] $   ; symbol fill color
         ,s_font_name:           'Helvetica'  $   ;
         ,s_font_size:           13 $             ;
         ,s_text_color:          [0,0,0]  $    ;

         ,b_errorbar_capsize:    0.1 $
         ,b_transparency:        50 $
         ,b_antialias:           3 $
         ,b_color:               [0,0,255] $
         ,b_errorbar_color:      [0,0,255] $
         ,b_use_default_color:   1   $
         ,b_sym_increment:       1   $
         ,b_sym_index:           0   $
         ,b_sym_size:            1.0   $          ; 
         ,b_sym_color:           [0,0,255] $   ;
         ,b_sym_thick:           1.0 $            ;
         ,b_y_errorbars:         1 $              ; show y errors
         ,b_linestyle:           0 $              ; line
         ,b_nsum:                1 $              ; no point everaging
         ,b_thick:               1.0 $            ; line thickness
         ,b_sym_filled:          1   $            ; use filled symbols
         ,b_sym_fill_color:      [0,0,255] $   ; symbol fill color
         ,b_font_name:           'Helvetica'  $   ;
         ,b_font_size:           13 $             ;
         ,b_text_color:          [0,0,0]  $    ;
       }
end
