; $Id$
;
; Copyright (c) 2003-2010, ITT Visual Information Solutions. All
;       rights reserved. Unauthorized reproduction is prohibited.
;----------------------------------------------------------------------------
; Purpose:
;   This file implements the generic IDL Tool object that
;   implements the actions performed when a file is saved.
;
;---------------------------------------------------------------------------

; Richard Tumanjong Azuah
; September, 2010
; 
; Extend IDLitOpFileSave for use to export the graphics windiw into a variety of 
; image formats.
; 

;===============================================================================
; DAVEopExportGraphics::GetProperty
pro DAVEopExportGraphics::GetProperty, writerTypes=writerTypes, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(writerTypes)) then begin
    writerTypes =  (ptr_valid(self.writerTypes))? (*self.writerTypes) : ''
endif

if(n_elements(etc) gt 0) then $
  self->IDLitopFileSave::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopExportGraphics::SetProperty
pro DAVEopExportGraphics::SetProperty, writerTypes=writerTypes, _EXTRA=etc
compile_opt idl2

if (n_elements(writerTypes) gt 0) then begin
    if (ptr_valid(self.writerTypes)) then $
      (*self.writerTypes) = writerTypes $
    else $
      self.writerTypes = ptr_new(writerTypes)
endif

if(n_elements(etc) gt 0) then $
  self->IDLitopFileSave::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopExportGraphics::GetFilterList
; 
; PURPOSE:
;   Provide a list of file filters (extensions) that should be handled
;   by the current operation. Method is intended to be called by the
;   UI Service which launches the file selection dialog.
;
; PARAMETERS:
;
; KEYWORDS:
;   count [out] - The number of filter groups (one group per reader
;                 which may actually contain more than one filter)
;
; RETURN VALUE:
;    If successful, an array of reader name/filter(s) pairs
;    If unsuccessful, an empty string.
;
function DAVEopExportGraphics::GetFilterList, COUNT=count
compile_opt idl2

oTool = self->GetTool()
oWriteFile = oTool->GetService("WRITE_FILE")
if(not obj_valid(oWriteFile))then begin
    count = 0
    return,''
endif

self->GetProperty,writerTypes=writerTypes

return, oWriteFile->GetFilterListByType(writerTypes,COUNT=count)

end



;===============================================================================
; DAVEopExportGraphics::_Save
;
; Purpose:
;   Used to save the iTool state.
;
; Parameters:
;   oTool   - The tool we are operating in.
;
; Return Value
;   Success (1), Failure (0), or Cancel (-1).
;
function DAVEopExportGraphics::_Save, oTool, _EXTRA=_extra

    compile_opt idl2, hidden

;tryAgain:
    ; Do we have our File Writer service?
    oWriteFile = oTool->GetService("WRITE_FILE")
    if(not obj_valid(oWriteFile))then begin
        self->ErrorMessage, $
            [IDLitLangCatQuery('Error:Framework:CannotAccessWriterService')], $
            title=IDLitLangCatQuery('Error:InternalError:Title'), severity=2
        return, 0
    endif

    self->IDLitOperation::GetProperty, SHOW_EXECUTION_UI=showUI

    badName = (self._filename eq '') || $
        STRCMP(self._filename, 'untitled', /FOLD_CASE)

    ; If we don't have a valid name, see if the Tool does.
    if (badName) then begin
        oTool->GetProperty, TOOL_FILENAME=filename
        self._filename = filename
    endif

    filebase = FILE_BASENAME(self._filename)

    badName = (self._filename eq '') || $
        STRCMP(self._filename, 'untitled', /FOLD_CASE)

;    if (showUI || badName) then begin
;    if (showUI) then begin

        ; Ask the UI service to present the file selection dialog to the user.
        ; The caller sets my filename property before returning.
        ; This should also call my GetFilterList().
        success = oTool->DoUIService('FileSaveAs', self)

        if (success eq 0) then $
            return, -1  ; cancel

;    endif

    ; check our filename cache
    if (self._fileName eq '') then $
        return, -1  ; cancel

    idWriter = oWriteFile->FindMatchingWriter(self._fileName)
    if (idWriter eq '') then begin
      Self->ErrorMessage, $;/USE_LAST_ERROR, $
        title='Unknown file format', severity=1, $
        ['File format could not be determined from specified filename:', $
        Self._fileName,' ', $
        'Please include extension in filename eg tmp.png for PNG format', $
        'or .jpg for JPEG' ]
        
        return, 0
    endif
    
    oDesc = oTool->GetByIdentifier(idWriter)
    oWriter = oDesc->GetObjectInstance()
    oWriter->GetProperty, TYPES=types
    if ISA(_extra) then $
      oWriter->SetProperty, _EXTRA=_extra
    
    ; Get the graphics window
    oSaveItem = oTool->GetCurrentWindow()      
    
    oGeneral = oTool->GetByIdentifier('/REGISTRY/SETTINGS/GENERAL_SETTINGS')
    if (Obj_Valid(oGeneral)) then begin
      oGeneral->GetProperty, RESOLUTION=resolution
    endif

    status = oWriteFile->WriteFile(self._fileName, oSaveItem, $
      RESOLUTION=resolution, _EXTRA=_extra)

    if (status ne 1) then begin
        self->ErrorMessage, $;/USE_LAST_ERROR, $
          title=IDLitLangCatQuery('Error:InternalError:Title'), severity=2, $
          [IDLitLangCatQuery('Error:Framework:FileWriteError'), $
          self._fileName,' ', $
          'Please include extension in filename eg tmp.png for PNG format']
;        if (idWriter eq '') then begin
;          self._filename = ''
;          goto, tryAgain
;        endif
        return, 0
    endif

    return, 1 ; success

end



;===============================================================================
; Definition
;---------------------------------------------------------------------------
; Purpose:
;   Class definition.
;
pro DAVEopExportGraphics__define
compile_opt idl2

struc = {DAVEopExportGraphics $
         ,writerTypes:ptr_new('') $ ; names of valid writers to be used
         ,inherits IDLitopFileSave $
        }

end
