
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataSubtract
;
; PURPOSE:
;   A simple operation that performs a subtraction of one dataset from
;   another. The dataset must contain plottable data with a dependent
;   and independent components.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   ResetOpAttributes
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataSubtract::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DAVEopDataSubtract::GetProperty, dataset1=dataset1,dataset2=dataset2, ds2Norm=ds2Norm $
  ,resultFlag=resultFlag,resultName=resultName, useDS2ErrFlag=useDS2ErrFlag, _REF_EXTRA=etc
compile_opt idl2

; dataset1
if (arg_present(dataset1)) then dataset1 = self.dataset1

; dataset2
if (arg_present(dataset2)) then dataset2 = self.dataset2

if (arg_present(ds2Norm)) then ds2Norm = self.dataset2Norm

if (Arg_present(useDS2ErrFlag)) then useDS2ErrFlag = self.useDS2ErrFlag

if (arg_present(resultFlag)) then resultFlag = self.resultFlag

if (arg_present(resultName)) then resultName = self.resultName

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataSubtract::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DAVEopDataSubtract::SetProperty, dataset1=dataset1,dataset2=dataset2, ds2Norm=ds2Norm $
  ,resultFlag=resultFlag,resultName=resultName, useDS2ErrFlag=useDS2ErrFlag, _EXTRA=etc

compile_opt idl2

; dataset1 index
if (n_elements(dataset1)) then begin
  Self.dataset1=dataset1
  resultname = ptr_valid(Self.selNamesPtr)? (*Self.selNamesPtr)[dataset1]+'-sub' : 'untitled'
  Self.resultName = resultName 
endif

; dataset2 index
if (n_elements(dataset2)) then Self.dataset2 = dataset2

if (n_elements(ds2Norm)) then Self.dataset2Norm = ds2Norm

if (N_elements(useDS2ErrFlag)) then Self.useDS2ErrFlag = useDS2ErrFlag

if (n_elements(resultFlag)) then begin
  Self.resultFlag = resultFlag
  Self->SetPropertyAttribute, 'resultName', hide=~resultFlag
endif

if (n_elements(resultName)) then Self.resultName = resultName

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataSubtract::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSubtract::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store useful info. The target for the
; command is the operation itself. For the subtraction operation, only
; one subtraction will be performed per command set object created! If
; this were not the case, then it would be better to make a dataset
; object as the target.
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, ds2Norm=dataset2Norm, resultFlag=resultFlag, resultName=resultName, useDS2ErrFlag=useDS2ErrFlag
void = oCmd->AddItem('OLD_DATASET2NORM',dataset2Norm)
void = oCmd->Additem('OLD_USEDS2ERRFLAG',useDS2ErrFlag)
void = oCmd->AddItem('OLD_RESULTFLAG',resultFlag)
void = oCmd->AddItem('OLD_RESULTNAME',resultName)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataSubtract::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSubtract::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, ds2Norm=dataset2Norm, resultFlag=resultFlag, resultName=resultName, useDS2ErrFlag=useDS2ErrFlag
void = oCmd->AddItem('NEW_DATASET2NORM',dataset2Norm)
void = oCmd->Additem('NEW_USEDS2ERRFLAG',useDS2ErrFlag)
void = oCmd->AddItem('NEW_RESULTFLAG',resultFlag)
void = oCmd->AddItem('NEW_RESULTNAME',resultName)

return, 1

end


;===============================================================================
; DAVEopDataSubtract::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSubtract::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataSubtract::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSubtract::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Get the 'old' properties and revert to them
void = oCmds[0]->GetItem('OLD_DATASET2NORM',dataset2Norm)
void = oCmds[0]->Getitem('OLD_USEDS2ERRFLAG',useDS2ErrFlag)
void = oCmds[0]->GetItem('OLD_RESULTFLAG',resultFlag)
void = oCmds[0]->GetItem('OLD_RESULTNAME',resultName)

oCmdTarget->SetProperty, ds2Norm=dataset2Norm, resultFlag=resultFlag, resultName=resultName, useDS2ErrFlag=useDS2ErrFlag

void = oCmds[0]->GetItem('NEW_RESULTFLAG',resultFlag)
case resultFlag of
  0: begin
    ;; Subtracted result was stored in the first dataset of the subtraction eqn
    
;    ; Get the 'new' normalization to undo the operation
;    if (~oCmds[0]->GetItem('NEW_DATASET2NORM',ds2Norm)) then ds2Norm = 1.0
    
    ; Get various stored properties and restore the operation with them
    void = oCmds[0]->GetItem('IDDATA_DS1',idDep1)
    void = oCmds[0]->GetItem('IDDATA_DS2',idDep2)
    void = oCmds[0]->GetItem('ERROREXIST',errExist)
    oDep1 = oTool->GetByIdentifier(idDep1)
    oDep2 = oTool->GetByIdentifier(idDep2)
    if (errExist) then begin
      void = oCmds[0]->GetItem('IDERROR_DS1',idErr1)
      void = oCmds[0]->GetItem('IDERROR_DS2',idErr2)
      oErr1 = oTool->GetByIdentifier(idErr1)
      oErr2 = oTool->GetByIdentifier(idErr2)
    endif
    trmtExist = oCmds[0]->GetItem('ID_TREATMENT',idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then void = oCmds[0]->GetItem('OLD_TREATMENT',trmt) else trmtExist=0
    
    ;oDep1 = oTool->GetByIdentifier(d1ID)
    if (~obj_valid(oDep1) || ~obj_isa(oDep1,'IDLitData')) then begin
      msg = 'Subtraction: Missing dependent data 1!'
      oTool->StatusMessage,msg
      return, 0
    endif
    if (~oDep1->GetData(data1)) then return, 0
    
    ;oDep2 = oTool->GetByIdentifier(d2ID)
    if (~obj_valid(oDep2) || ~obj_isa(oDep2,'IDLitData')) then begin
      msg = 'Subtraction: Missing dependent data 2!'
      oTool->StatusMessage,msg
      return, 0
    endif
    if (~oDep2->GetData(data2)) then return, 0
    data2 = ds2Norm * data2   ; apply normalization factor as before
    
    if (errExist) then begin
      ;    oErr1 = oTool->GetByIdentifier(e1ID)
      if (~obj_valid(oErr1) || ~obj_isa(oErr1,'IDLitData')) then begin
        msg = 'Subtraction: Missing error data 1!'
        oTool->StatusMessage,msg
        return, 0
      endif
      if (~oErr1->GetData(err1)) then return, 0
      
      ;    oErr2 = oTool->GetByIdentifier(e2ID)
      if (~obj_valid(oErr2) || ~obj_isa(oErr2,'IDLitData')) then begin
        msg = 'Subtraction: Missing error data 2!'
        oTool->StatusMessage,msg
        return, 0
      endif
      if (~oErr2->GetData(err2)) then return, 0
      err2 = err2 * ds2Norm  ; apply normalization factor as before
    endif
    
    ; Undo subtraction and update the data objects
    if (errExist) then begin
      if (useDS2ErrFlag) then err1 = Sqrt(err1^2 - err2^2)
      void = oErr1->Setdata(err1,/no_copy,/no_notify)
    endif
    void = oDep1->SetData(data1 + data2,/no_copy,/no_notify)
    
    ; Treatment info
    if (trmtExist) then $
      void = oTrmt->SetData(trmt,/no_copy,/no_notify) ; modify treatment
      
    ; Notify observers about the change so they can update themselves!
    oDep1->notifyDataChange
    oDep1->notifyDataComplete
  end
  
  1: begin
    ;; a new dataset was created and the subtracted results stored in it

    ; Undo previous sum by deleting the previously generated output from the Data Manager
    void = oCmds[0]->GetItem('FULLOUTPUTID',idDataset)
    oDataset = oTool->GetByIdentifier(idDataset)
    oDM = oTool->GetService("DAVEDATA_MANAGER")
    status = oDM->DeleteData(oDataset->GetFullIdentifier())    
  end
  
  else:
endcase



; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DAVEopDataSubtract::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSubtract::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Get the 'new' properties and reset operation to them
void = oCmds[0]->GetItem('NEW_DATASET2NORM',ds2Norm)
void = oCmds[0]->Getitem('NEW_USEDS2ERRFLAG',useDS2ErrFlag)
void = oCmds[0]->GetItem('NEW_RESULTFLAG',resFlag)
void = oCmds[0]->GetItem('NEW_RESULTNAME',resName)
oCmdTarget->SetProperty, ds2Norm=ds2Norm, resultFlag=resFlag, resultName=resName, useDS2ErrFlag=useDS2ErrFlag

; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('IDDATA_DS1',idDep1)
void = oCmds[0]->GetItem('IDDATA_DS2',idDep2)
oDep1 = oTool->GetByIdentifier(idDep1)
oDep2 = oTool->GetByIdentifier(idDep2)
void = oCmds[0]->GetItem('ERROREXIST',errExist)
if (errExist) then begin
    void = oCmds[0]->GetItem('IDERROR_DS1',idErr1)
    void = oCmds[0]->GetItem('IDERROR_DS2',idErr2)
    oErr1 = oTool->GetByIdentifier(idErr1)
    oErr2 = oTool->GetByIdentifier(idErr2)
endif
trmtExist = oCmds[0]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then void = oCmds[0]->GetItem('NEW_TREATMENT',trmt) else trmtExist = 0

if (~obj_valid(oDep1) || ~obj_isa(oDep1,'IDLitData')) then begin
    msg = 'Subtraction: Missing dependent data!'
    oTool->StatusMessage,msg
    return, 0
endif
if (~oDep1->GetData(data1)) then return, 0

if (~obj_valid(oDep2) || ~obj_isa(oDep2,'IDLitData')) then begin
    msg = 'Subtraction: Missing dependent data!'
    oTool->StatusMessage,msg
    return, 0
endif
if (~oDep2->GetData(data2)) then return, 0
data2 = ds2Norm * data2   ; apply normalization factor as before

if (errExist) then begin
    if (~obj_valid(oErr1) || ~obj_isa(oErr1,'IDLitData')) then begin
        msg = 'Subtraction: Missing error in dependent data!'
        oTool->StatusMessage,msg
        return, 0
    endif
    if (~oErr1->GetData(err1)) then return, 0
    
    if (~obj_valid(oErr2) || ~obj_isa(oErr2,'IDLitData')) then begin
        msg = 'Subtraction: Missing error in dependent data!'
        oTool->StatusMessage,msg
        return, 0
    endif
    if (~oErr2->GetData(err2)) then return, 0
    err2 = err2 * ds2Norm  ; apply normalization factor as before
endif


case resFlag of
  0: ;; Results should be stored in the first dataset in the subtraction eqn
    
  1: begin
    ;; Results should be stored in a new dataset created
    
    ; locate the first dataset used in original subtraction eqn
    oTarget1 = getmaindataobjectparent(oDep1)
    if (~obj_valid(oTarget1)) then return, 0
    if (~obj_isa(oTarget1,'IDLitParameterSet')) then return, 0
    
    ; Create new dataset to contain subtracted data
    if (obj_hasmethod(oTarget1,'Clone')) then begin
      ; use clone method to replicate first dataset
      oRes = oTarget1->Clone()
      oRes->SetProperty, name=resName
      oTool->AddDatasetToDataManager, oRes
    endif else begin
      ; use DM to replicate first dataset
      oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
      if (~oDM->CopyData(oTarget1->GetFullIdentifier(), name=resName)) then return, 0
    endelse
    
    oRes->GetProperty, trmtRef=oTrmt, dataRef=oDep1, errorRef=oErr1    ; dependent data and error object refs
    trmtExist = oCmds[0]->GetItem('NEW_TREATMENT',trmt)
    
    ; Update 'FULLOUTPUTID' entry
    void = oCmds[0]->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier(),/overwrite)
    
  end
  
  else:
endcase

; Perform the subtraction and update the data objects
if (errExist) then begin
  if (useDS2ErrFlag) then err1 = sqrt(err1^2 + err2^2)
  void = oErr1->SetData(err1,/no_copy,/no_notify)
endif
void = oDep1->SetData(data1 - data2,/no_copy,/no_notify)

; Treatment info
if (trmtExist) then $
  void = oTrmt->SetData(trmt,/no_copy,/no_notify) ; modify treatment


; Notify observers about the change so they can update themselves!
oDep1->notifyDataChange
oDep1->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DAVEopDataSubtract::ResetOpAttributes
; 
; PURPOSE:
;   Reset attributes for this operation to reflect the currently
;   selected datasets
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function DAVEopDataSubtract::ResetOpAttributes
compile_opt idl2

; Get the selected data and populate the first and second dataset fields
if (self.selCount lt 2) then return, 0

self->SetPropertyAttribute, 'dataset1', enumlist=(*self.selNamesPtr)
self->SetPropertyAttribute, 'dataset2', enumlist=(*self.selNamesPtr)
self->SetProperty, dataset1=0, dataset2=1

return, 1

end


;===============================================================================
; DAVEopDataSubtract::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Take the
;   difference between two supplied datasets and store the results in
;   the first dataset or in a new dataset.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataSubtract::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt lt 2) then begin
    oTool->StatusMessage, 'Please, select at least two top-level datasets from the Data Browser tree.'
    return, obj_new()
endif

; Locate datasets containing dave1commonstr containers
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be 'DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP',...
                                    ; these are defined when this operation is registered!
nValid = n_elements(validTypes)
validName = 'DAVE Data Manager'
for i = 0,n_elements(oSelections)-1 do begin
    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oRes = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)
    if (~obj_valid(oRes)) then continue

    ; Record the obj ref. and identifier of the valid data as well as the selection it is found in
    ; Also the note the name property of the selection
    selID = oSelections[i]->GetFullIdentifier()
    selIDs = (n_elements(selIDs) gt 0)? [selIDs,selID] : selID
    oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
    oSelections[i]->GetProperty, name=selName
    selNames = (n_elements(selNames) gt 0)? [selNames,selName] : selName
endfor

if (n_elements(oSel) eq 0) then begin
    oTool->StatusMessage, 'Could not locate two valid datasets to perform subtraction!'
    return, obj_new()
endif

; Store the selected datasets and the associated target data objects
if (ptr_valid(self.targetIDsPtr)) then ptr_free, self.targetIDsPtr
if (ptr_valid(self.selNamesPtr)) then ptr_free, self.selNamesPtr

self.targetIDsPtr = ptr_new(selIDs) ;ptr_new(targetIDs)
self.selNamesPtr = ptr_new(selNames)
self.selCount = n_elements(selNames)
oTarget = oSel

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
;hasPropSet = 0b

if (doUI) then begin
    
    ;; Set operation properties based on the selected data
    if (~self->ResetOpAttributes()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Perform subtraction and record changes in undo/redo buffer
if (~self->Subtract(oTarget, oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataSubtract::Subtract
; 
; PURPOSE:
;   Perform difference b/n two datasets and record changes in the
;   operation's undo/redo buffer.
;
; PARAMETERS:
;   oTarget [in|out] - the target data - not used in this operation
;
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0
;
function DAVEopDataSubtract::Subtract, oTarget, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Operation settings
ds1 = Self.dataset1
ds2 = Self.dataset2
ds2Norm = Self.dataset2Norm
resFlag=Self.resultFlag
resName=Self.resultName
useDS2ErrFlag = Self.useDS2ErrFlag

; Get the data contained in ds1 and ds2
; and check that they have the same size (dims and elements)
targetID1 = (*self.targetIDsPtr)[ds1]
oTarget1 = oTool->GetByIdentifier(targetID1)
if (~obj_valid(oTarget1)) then begin
    msg = 'Subtraction: Could not retrieve target dataset!'
    oTool->StatusMessage,msg
    return, 0
endif
targetID2 = (*self.targetIDsPtr)[ds2]
oTarget2 = oTool->GetByIdentifier(targetID2)
if (~obj_valid(oTarget2)) then begin
    msg = 'Subtraction: Could not retrieve target dataset!'
    oTool->StatusMessage,msg
    return, 0
endif

;oDep1 = oTarget1->GetByName('2') ; dependent data
oTarget1->GetProperty, dataRef=oDep1    ; dependent data object ref
if (~obj_valid(oDep1) || ~obj_isa(oDep1,'IDLitData')) then begin
    msg = 'Subtraction: Could not retrieve any data from dataset!'
    oTool->StatusMessage,msg
    return, 0
endif
if (~oDep1->GetData(data1)) then return, 0

targetID = (*self.targetIDsPtr)[ds2]
;oDep2 = oTarget2->GetByName('2') ; dependent data
oTarget2->GetProperty, dataRef=oDep2    ; dependent data object ref
if (~obj_valid(oDep2) || ~obj_isa(oDep2,'IDLitData')) then begin
    msg = 'Subtraction: Could not retrieve any data from dataset!'
    oTool->StatusMessage,msg
    return, 0
endif
if (~oDep2->GetData(data2)) then return, 0
data2 = ds2Norm * data2   ; apply normalization factor

s1 = size(data1)
s2 = size(data2)
passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2]))
if (~passed) then begin
    msg = 'Subtraction: Two datasets are not of the same size!'
    oTool->StatusMessage,msg
    return, 0
endif

; Get the error data contained in ds1 and ds2
err1Exist = 1
;oErr1 = oTarget1->GetByName('3') ; error in dependent data
oTarget1->GetProperty, errorRef=oErr1    ; error in dependent data object ref
if (~obj_valid(oErr1) || ~obj_isa(oErr1,'IDLitData')) then err1Exist=0
err1Exist = (err1Exist)? oErr1->GetData(err1) : 0
err2Exist = 1
;oErr2 = oTarget2->GetByName('3') ; error in dependent data
oTarget2->GetProperty, errorRef=oErr2    ; error in dependent data object ref
if (~obj_valid(oErr2) || ~obj_isa(oErr2,'IDLitData')) then err2Exist=0
err2Exist = (err2Exist)? oErr2->GetData(err2) : 0
if (err2Exist) then err2 = ds2Norm*err2   ; apply normalization factor
errExist = (err1Exist && err2Exist)

; Retrieve the first command object from the command set to record
; details of what is done.
; This was created in RecordInitialValues()
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Stored useful info in the command obj
void = oCmd->AddItem('IDDATA_DS1',oDep1->GetFullIdentifier())
void = oCmd->AddItem('IDDATA_DS2',oDep2->GetFullIdentifier())
void = oCmd->AddItem('ERROREXIST',errExist)
if (errExist) then begin
  void = oCmd->AddItem('IDERROR_DS1',oErr1->GetFullIdentifier())
  void = oCmd->AddItem('IDERROR_DS2',oErr2->GetFullIdentifier())
endif

case resFlag of
  0: begin
    ;; Results should be stored in the first dataset in the subtraction eqn
    
    ; Get the treatment object
    trmtExist = 0
    oTarget1->GetProperty, trmtRef = oTrmt
    if (obj_valid(oTrmt)) then $
      trmtExist = oTrmt->GetData(trmt)     
   
    oRes = oTarget1
  end

  1: begin
    ;; Results should be stored in a new dataset created
    
    ; Create new dataset to contain subtracted data
    if (obj_hasmethod(oTarget1,'Clone')) then begin
      ; use clone method to replicate first dataset
      oRes = oTarget1->Clone()
      oRes->SetProperty, name=resName
      oTool->AddDatasetToDataManager, oRes
    endif else begin
      ; use DM to replicate first dataset
      oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
      if (~oDM->CopyData(oTarget1->GetFullIdentifier(), name=resName)) then return, 0
    endelse
    
    ; Update 'FULLOUTPUTID' entry
    void = oCmd->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier(),/overwrite)

  end
  
  else:
endcase

; Get the treatment info and update it
trmtExist = 0
oRes->GetProperty, trmtRef=oTrmt, dataRef=oDep1, errorRef=oErr1
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then begin
  void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
  void = oCmd->AddItem('OLD_TREATMENT',trmt)
  
  line = '____________________________________________________'
  newTrmt = 'Subtraction applied: '+(*self.selNamesPtr)[ds1]+' = '+(*self.selNamesPtr)[ds1]+' - '+(*self.selNamesPtr)[ds2]
  newTrmt = [newTrmt,'Normalization factor of '+strtrim(string(ds2Norm),2)+' applied to '+(*self.selNamesPtr)[ds2]]
  if (errExist) then newTrmt = [newTrmt,'Errors taken into account in quadrature!']
  newTrmt = [trmt,line,'Timestamp: '+systime(),newTrmt]
  
  void = oCmd->AddItem('NEW_TREATMENT',newTrmt) ; save modified treatment info
  void = oTrmt->SetData(newTrmt,/no_copy,/no_notify) ; modify treatment
endif

; Perform the subtraction and update the data objects
if (errExist) then begin
  if (useDS2ErrFlag) then err1 = sqrt(err1^2 + err2^2)
  void = oErr1->SetData(err1,/no_copy,/no_notify)
endif
void = oDep1->SetData(data1 - data2,/no_copy,/no_notify) ; /no_notify ==> don't notify observers!

; Notify observers about the change so they ca update themselves!
;oDep = (ds3Flag)? oDep3 : oDep1
oDep1->notifyDataChange
oDep1->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataSubtract::Cleanup
; 
; PURPOSE:
;   DAVEopDataSubtract class cleanup
;
pro DAVEopDataSubtract::Cleanup
compile_opt idl2

if (ptr_valid(self.selNamesPtr)) then ptr_free, self.selNamesPtr
if (ptr_valid(self.targetIDsPtr)) then ptr_free, self.targetIDsPtr
if (ptr_valid(self.selIDsPtr)) then ptr_free, self.selIDsPtr

;ptr_free, selNamesPtr,targetIDsPtr,selIDsPtr

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSubtract::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataSubtract::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is reversible
; This operation is not expensive
if (~self->IDLitOperation::Init(NAME='Data Subtraction' $
                                ,reversible_operation=1,expensive_computation=0 $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']

;self->SetProperty, reversible_operation=1, expensive_computation=0

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register properties for this operation
self->RegisterProperty, 'dataset1', enumlist=['empty'], description='Dataset A as in "A-B"' $
  ,name='Dataset 1 (input/output)'
self->RegisterProperty, 'dataset2', enumlist=['empty'], description='Dataset B as in "A-B"' $
  ,name='Dataset 2 (input)'
self->RegisterProperty, 'ds2Norm', /float, description='Dataset B Normalization factor' $
  ,name='Dataset 2 Normalization'
self->Registerproperty, 'useDS2ErrFlag', enumlist=['No','Yes'], description='Use Dataset 2 Errors in subtraction' $
  ,name='Use Dataset 2 Errors?'
resList = ['Overwrite Dataset 1','Create New Dataset']
self->RegisterProperty, 'resultFlag',enumlist=resList, description='Destination for subtracted results' $
  ,name='Save subtracted dataset in'
self->RegisterProperty, 'resultName',/string, description='Name of dataset to store result' $
  ,name='Subtracted dataset name',hide=1

Self.useDS2ErrFlag = 1
self.dataset2Norm=1.0

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSubtract__define
; 
; PURPOSE:
;   DAVEopDataSubtract class structure definition
;
pro DAVEopDataSubtract__define

compile_opt idl2

struc = {DAVEopDataSubtract $
         ,inherits IDLitOperation $
         ,dataset1:0B   $         ; first dataset index; A in (A - B)
         ,dataset2:1B   $         ; second dataset index; B in (A - B)
         ,resultFlag:0  $         ; destination for subtracted data: 0=overwrite dataset1, 1=create new dataset
         ,useDS2ErrFlag:0 $       ; 1=use errors in dataset2; 0=don't use Dataset2 errors in calc
         ,resultName:'' $         ; name of dataset to store the sum
         ,dataset2Norm:1.0 $      ; normalization factor for dataset 2
         ,selCount:0B $           ; nos of datasets currently selected
         ,selNamesPtr:ptr_new() $ ;  ptr to names of currently selected datasets
         ,targetIDsPtr:ptr_new() $; ptr to ids of target objects in selected datasets
         ,selIDsPtr:ptr_new() $   ; ptr to ids of selected datasets
        }

end
