
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataSimpleAxisTransform
;
; PURPOSE:
;   A simple operation that can transform an axis of a dataset. Only one axis is
;   operated on at a time except when the intensity/counts axis is being transformed
;   in which case a transform must also be applied to the error axis.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Oct 2010
;-
;###############################################################################


;===============================================================================
; DAVEopDataSimpleAxisTransform::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataSimpleAxisTransform::GetProperty, expression=expression,errorexp=errorexp $
  ,axis=axis,axisLabel=axisLabel,axisUnits=axisUnits $
  ,oldLabel=oldLabel,oldUnits=oldUnits $
  ,_REF_EXTRA=etc
  
compile_opt idl2

;
if (arg_present(expression)) then expression = self.expression

;
if (arg_present(errorexp)) then errorexp = self.errorexp

;
if (arg_present(axisLabel)) then axisLabel = self.axisLabel

;
if (arg_present(axisUnits)) then axisUnits = self.axisUnits

;
if (arg_present(oldLabel)) then oldLabel = self.oldLabel

;
if (arg_present(oldUnits)) then oldUnits = self.oldUnits

;
if (arg_present(axis)) then axis = self.axis

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataSimpleAxisTransform::SetProperty, expression=expression,errorexp=errorexp $
  ,axis=axis,axisLabel=axisLabel,axisUnits=axisUnits $
  ,_EXTRA=etc
compile_opt idl2

;
if (n_elements(axisLabel)) then self.axisLabel=axisLabel

;
if (n_elements(axisUnit)) then self.axisUnit=axisUnits

;
if (n_elements(expression)) then self.expression=expression

;
if (n_elements(errorexp)) then self.errorexp=errorexp


; axis
if (n_elements(axis)) then begin
  Self.axis = axis
  Self->_Update2CurrentData
endif


; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::_UpdateCurrentBinSettings
; 
; PURPOSE:
;   
;
; PARAMETERS:
;
; KEYWORDS:
;
;
pro DAVEopDataSimpleAxisTransform::_Update2CurrentData
compile_opt idl2

if (~obj_valid(self.oTarget)) then return
oParent = getmaindataobjectparent(self.oTarget)
axisLabel = ''
axisUnits = ''
case Self.axis of
  0: oParent->GetProperty, axis1Label=axisLabel, axis1Units=axisUnits
  1: oParent->GetProperty, axis2Label=axisLabel, axis2Units=axisUnits
  2: oParent->GetProperty, dataLabel=axisLabel, dataUnits=axisUnits
  else:
endcase

Self.axisLabel = axisLabel
Self.axisUnits = axisUnits
Self.oldLabel = axisLabel
Self.oldUnits = axisUnits

; If there is only one independent axis modify the axis drop-list.
oParent->GetProperty, nDimensions=ndim
Self->SetPropertyAttribute, 'axis', enumlist=(ndim eq 1)? ['x-axis','Intensity'] : ['x-axis','y-axis','Intensity']
Self->SetPropertyAttribute, 'errorexp', sensitive=(Self.axis eq 2)

;Self.oParent = oParent

end



;===============================================================================
; DAVEopDataSimpleAxisTransform::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSimpleAxisTransform::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values, This is the first one
; that is created for this oCmdSet
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty, expression=expression,errorexp=errorexp,axis=axis $
  ,oldLabel=oldLabel,oldUnits=oldUnits
  
void = oCmd->AddItem('OLD_EXP',expression)
void = oCmd->AddItem('OLD_UNITS',oldUnits)
void = oCmd->AddItem('OLD_LABEL',oldLabel)
void = oCmd->AddItem('OLD_AXIS',axis)
if (axis eq 2) then  void = oCmd->AddItem('OLD_EEXP',errorexp)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSimpleAxisTransform::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty, expression=expression,errorexp=errorexp,axis=axis $
  ,axisLabel=axisLabel,axisUnits=axisUnits
  
void = oCmd->AddItem('NEW_EXP',expression)
void = oCmd->AddItem('NEW_UNITS',axisUnits)
void = oCmd->AddItem('NEW_LABEL',axisLabel)
void = oCmd->AddItem('NEW_AXIS',axis)
if (axis eq 2) then  void = oCmd->AddItem('NEW_EEXP',errorexp)

return, 1

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSimpleAxisTransform::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSimpleAxisTransform::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

rebinAxis = self.axis
; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('OLD_MIN',min)
void = oCmds[0]->GetItem('OLD_MAX',max)
void = oCmds[0]->GetItem('OLD_BIN',bin)
void = oCmds[0]->GetItem('OLD_AXIS',axis)
void = oCmds[0]->GetItem('OLD_TRUNCATE',truncate)
self->SetProperty, min=min,max=max,bin=bin,axis=axis,truncate=truncate, /no_update


; Undo the changes that were made to the target
void = oCmds[1]->GetItem('ID_IND',idInd)
void = oCmds[1]->GetItem('OLD_INDDATA',vin)
void = oCmds[1]->GetItem('ID_DEP',idDep)
void = oCmds[1]->GetItem('OLD_DEPDATA',zin)
oDep = oTool->GetByIdentifier(idDep)
oInd = oTool->GetByIdentifier(idInd)

if (~obj_valid(oDep) || ~obj_valid(oInd)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif
void = oDep->SetData(zin,/no_copy,/no_notify)
void = oInd->SetData(vin,/no_copy,/no_notify)


void = oCmds[1]->GetItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmds[1]->GetItem('OLD_ERRDATA',zein)
    void = oCmds[1]->GetItem('ID_ERR',idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (~obj_valid(oErr)) then begin
        oTool->_RemoveCommand, oCmdSet
        return, 0
    endif
    void = oErr->SetData(zein,/no_copy,/no_notify)
endif

void = oCmds[1]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmds[1]->GetItem('OLD_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif
 
;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSimpleAxisTransform::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('NEW_MIN',min)
void = oCmds[0]->GetItem('NEW_MAX',max)
void = oCmds[0]->GetItem('NEW_BIN',bin)
void = oCmds[0]->GetItem('NEW_AXIS',axis)
void = oCmds[0]->GetItem('NEW_TRUNCATE',truncate)
self->SetProperty, min=min,max=max,bin=bin,axis=axis,truncate=truncate, /no_update

; Redo the changes that were made
void = oCmds[1]->GetItem('ID_IND',idInd)
void = oCmds[1]->GetItem('NEW_INDDATA',vout)
void = oCmds[1]->GetItem('ID_DEP',idDep)
void = oCmds[1]->GetItem('NEW_DEPDATA',zout)
oDep = oTool->GetByIdentifier(idDep)
oInd = oTool->GetByIdentifier(idInd)
if (~obj_valid(oDep) || ~obj_valid(oInd)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

void = oDep->SetData(zout,/no_copy,/no_notify)
void = oInd->SetData(vout,/no_copy,/no_notify)

void = oCmds[1]->GetItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmds[1]->GetItem('NEW_ERRDATA',zeout)
    void = oCmds[1]->GetItem('ID_ERR',idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (~obj_valid(oErr)) then begin
        oTool->_RemoveCommand, oCmdSet
        return, 0
    endif
    void = oErr->SetData(zeout,/no_copy,/no_notify)
endif

void = oCmds[1]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmds[1]->GetItem('NEW_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif

;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataSimpleAxisTransform::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be rebinned
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   ;; Valid types are containers with at least one independent and dependent data components
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
;   oTarget = (n_elements(oTarget) gt 0)? [oTarget,oRes] : oRes
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
oTarget = oSel[0]
if (n_elements(oTarget) eq 0) then begin
    oTool->StatusMessage, 'Selection does not contain data that can be transformed!'
    return, obj_new()
endif


; Make the first valid target the operation's target dataset
self.oTarget = oTarget

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Some operation properties depend on the selected data: update them
    self.setdefault = 1
    self->_Update2CurrentData

    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Rebin the data and record changes in undo/redo buffer
if (~self->Transform(oTarget, oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataSimpleAxisTransform::Rebin
; 
; PURPOSE:
;   Rebin the data conatined in the specified target and record
;   changes in the operation's undo/redo buffer.
;
; PARAMETERS:
;   oTarget [in|out] - the data container whose data is to be
;                      rebinned.
;
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0.
;
function DAVEopDataSimpleAxisTransform::Transform, oTarget, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get current operation settings
self->GetProperty,min=lolim,max=uplim,bin=binw,axis=axis,truncate=truncate $
                  ,reversible_operation=rev_op,expensive_computation=exp_comp

; uplim must be greater than lolim
if (lolim gt (uplim-binw)) then begin
    oTool->StatusMessage,'Data Rebin: lower limit should not be >= upper limit!'
    return, 0
endif

; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as 
; axes data or objects and the treatment buffer.
oParent = getmaindataobjectparent(oTarget)

; Get the independent data and it's distribution (ie whether histogram or point)
if (self.axis eq 0) then $
   oParent->GetProperty, axis1Ref=oInd, axis1Value=vin, axis1Distribution=distribution else $
   oParent->GetProperty, axis2Ref=oInd, axis2Value=vin, axis2Distribution=distribution
nin = n_elements(vin)
if (nin lt 2) then return, 0   ; vin must be a vector

; The dependent data
oParent->GetProperty, dataRef=oDep, dataValue=zin
if (n_elements(zin) eq 0) then return, 0

; The error in the dependent data
oParent->GetProperty, errorRef=oErr, errorValue=zein
errExist = (n_elements(zein) gt 0)

; Store copies of original data because rebin operation is not
; reversible.
oCmd = obj_new("IDLitCommand", TARGET_IDENTIFIER=oTarget->GetFullIdentifier())
oCmdSet->Add, oCmd
void = oCmd->AddItem('OLD_INDDATA',vin)
void = oCmd->AddItem('OLD_DEPDATA',zin)
;void = oCmd->AddItem('ID_TARGETPARENT',oParent->GetFullIdentifier())
void = oCmd->AddItem('ID_IND',oInd->GetFullIdentifier())
void = oCmd->AddItem('ID_DEP',oDep->GetFullIdentifier())
void = oCmd->AddItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmd->AddItem('OLD_ERRDATA',zein)
    void = oCmd->AddItem('ID_ERR',oErr->GetFullIdentifier())
endif

; Get treatment history
oParent->GetProperty, trmtRef = oTrmt    ; can also use: oTrmt = locatetreatmentobject(oTarget)
if (obj_valid(oTrmt)) then begin
   void = oTrmt->GetData(trmt)
   void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
   void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
   ;; modify treatment info accordingly
   line = '____________________________________________________'
   trmt = [trmt, $
           line, $
           'Timestamp: '+systime(), $
           'Perform a rebin of the data']
endif


; If 2D data and rebin axis is y-axis then transpoase data
transpose = 0
if ((self.axis eq 1) && ((size(zin))[0] eq 2)) then begin
    zin = transpose(zin)        ; transpose data so that y-axis data become the rows to be rebinned
    zein = transpose(zein)      ;
    transpose = 1
endif
        
lolim_mod = (lolim lt vin[0])? vin[0] : lolim
uplim_mod = (uplim gt vin[nin-1])? vin[nin-1] : uplim 
  
;; determine output bins
nout = fix((uplim_mod - lolim_mod)/binw) + 1
vout = lolim_mod + findgen(nout)*binw
;; do the rebinning
if (~errExist) then zein = 0.01*zin ; fake some error if necessary!
case strupcase(strmid(distribution,0,4)) of ; switch between histogram or points data rebinning
    'HIST': drebin,vin,zin,zein,vout,zout,zeout,/hist,/to_hist,err=err,emsg=emsg
    else: drebin,vin,zin,zein,vout,zout,zeout,/points,/to_points,err=err,emsg=emsg
endcase
if (err ne 0) then begin
    oTool->StatusMessage,emsg
    return, 0
endif

;; if the data outside the rebinning range is not to be
;; truncated then this data needs to be included in the output
if (~truncate) then begin
    ;; get any data below lolim
    res = where(vin lt lolim_mod, cnt)
    if (cnt gt 0) then begin
        vout = [vin[res],vout]
        zout = [zin[res],zout]
        zeout = [zein[res],zeout]
    endif

    ;; get any data above uplim
    res = where(vin gt uplim_mod, cnt)
    if (cnt gt 0) then begin
        vout = [vout,vin[res]]
        zout = [zout,zin[res]]
        zeout = [zeout,zein[res]]
    endif
endif


; transpose the data back, if necessary
if (transpose) then begin
    zout = transpose(zout)
    zeout = transpose(zeout)
endif

; Store rebinned data because rebin operation is expensive
void = oCmd->AddItem('NEW_INDDATA',vout)
void = oCmd->AddItem('NEW_DEPDATA',zout)
if (errExist) then void = oCmd->AddItem('NEW_ERRDATA',zeout)

; Update data objects with rebinned data using the no_notify keyword
; so that observers are not informed
void = oDep->setData(zout,/no_copy,/no_notify)
if (errExist) then void = oErr->SetData(zeout,/no_copy,/no_notify)
void = oInd->SetData(vout,/no_copy,/no_notify)

; Update treatment information
if (obj_valid(oTrmt)) then begin
    trmt = [trmt, $
            'From '+ $
            strtrim(string(lolim_mod),2)+' to '+strtrim(string(uplim_mod),2)+ $
            ' in steps of '+strtrim(string(binw),2)]

    void = oCmd->AddItem('NEW_TREATMENT',trmt) ; save modified treatment info
    void = oTrmt->SetData(trmt,/no_copy,/no_notify)
endif

; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataSimpleAxisTransform::Cleanup
; 
; PURPOSE:
;   DAVEopDataSimpleAxisTransform class cleanup
;
pro DAVEopDataSimpleAxisTransform::Cleanup
compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSimpleAxisTransform::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataSimpleAxisTransform::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is not reversible 
; This operation is expensive
if (~self->IDLitOperation::Init(NAME='Data Rebin' $
;                                ,types=['DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP'] $
                                ,reversible_operation=0,expensive_computation=1 $
                                ,_EXTRA=etc)) then return, 0

;self->SetProperty, reversible_operation=0, expensive_computation=1

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register an offset property for this operation
self->RegisterProperty, 'axis', enumlist=['x-axis','y-axis'], description='Transform Axis' $
  ,name='Transform Axis',sensitive=1
self->RegisterProperty, 'expression', /string, description='Transform Expression' $
  ,name='X = ',sensitive=1
self->RegisterProperty, 'errorexp', /string, description='Transform Expression for Error' $
  ,name='Error = ',/hide
self->RegisterProperty, 'oldlabel', /string, description='Current Axis Label' $
  ,name='Current Axis Label',sensitive=0
self->RegisterProperty, 'axislabel', /string, description='New Axis Label' $
  ,name='New Axis Label'
self->RegisterProperty, 'oldunits', /string, description='Current Axis Units' $
  ,name='Current Axis Units',sensitive=0
self->RegisterProperty, 'axisunits', /string, description='New Axis Units' $
  ,name='New Axis Units'

; init offset to 0.0
Self.axis = 0
Self.axisLabel = ''
Self.axisUnit = ''
Self.expression = ''


; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSimpleAxisTransform__define
; 
; PURPOSE:
;   DAVEopDataSimpleAxisTransform class structure definition
;
pro DAVEopDataSimpleAxisTransform__define

compile_opt idl2

struc = {DAVEopDataSimpleAxisTransform $
         ,inherits IDLitOperation $
         ,expression:'' $       ; transform expression to be applied
         ,errorexp:'' $         ; transform expression to be applied
         ,axisUnits:'' $        ; axis unit to use after transform
         ,axisLabel:'' $        ; axis label to use after transform
         ,oldUnits:'' $        ; axis unit to use after transform
         ,oldLabel:'' $        ; axis label to use after transform
         ,axis:0 $              ; axis to be transformed
         ,oTarget:obj_new() $   ;
         ,oParent:obj_new() $
        }

end
