
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataRebin
;
; PURPOSE:
;   A simple operation that rebins the plottable data of an
;   IDLitData object. The plottable data must consist of an dependent
;   component with at least one independent component.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataRebin::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataRebin::GetProperty, dmin=dmin,dmax=dmax,dbinmin=dbinmin, dbinmax=dbinmax $
  ,min=min,max=max,bin=bin,axis=axis,truncate=truncate, norebin=norebin, _REF_EXTRA=etc
compile_opt idl2

; dmin
if (arg_present(dmin)) then dmin = self.dmin

;
if (arg_present(dmax)) then  dmax = self.dmax

;
if (arg_present(dbinmin)) then dbinmin = self.dbinmin

;
if (arg_present(dbinmax)) then dbinmax = self.dbinmax

;
if (arg_present(min)) then min = self.min

;
if (arg_present(max)) then max = self.max

;
if (arg_present(bin)) then bin = self.bin

;
if (arg_present(axis)) then axis = self.axis

;
if (arg_present(truncate)) then truncate = self.truncate
;
if (Arg_present(norebin)) then norebin = self.norebin

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataRebin::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataRebin::SetProperty, dmin=dmin,dmax=dmax,dbinmin=dbinmin, dbinmax=dbinmax $
  ,min=min,max=max,bin=bin,axis=axis,truncate=truncate, no_update=no_update, norebin=norebin $ ;,types=types
  , _EXTRA=etc
compile_opt idl2


; dmin
if (n_elements(dmin)) then self.dmin=dmin

; dmax
if (n_elements(dmax)) then self.dmax=dmax

; dbinmin
if (n_elements(dbinmin)) then self.dbinmin = dbinmin

; dbinmax
if (n_elements(dbinmax)) then self.dbinmax = dbinmax

; min
if (n_elements(min)) then self.min = min > self.dmin

; max
if (n_elements(max)) then self.max = max < self.dmax

; bin width
if (n_elements(bin)) then self.bin = bin > self.dbinmin

; axis
if (n_elements(axis)) then begin
  ;oTool = self->GetTool()
  ;oTool->ErrorMessage, ['In SetProperty','axis = '+string(axis)], severity=0, title='Data Rebin'
    self.axis = axis
    self.setdefault = 1
endif

; 
if (n_elements(truncate)) then self.truncate=truncate
;
if (N_elements(norebin)) then self.norebin=norebin

; Update the current bin settings as they depend on the ind axis data
if (~keyword_set(no_update)) then self->_UpdateCurrentBinSettings

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataRebin::_UpdateCurrentBinSettings
; 
; PURPOSE:
;   
;
; PARAMETERS:
;
; KEYWORDS:
;
;
pro DAVEopDataRebin::_UpdateCurrentBinSettings
compile_opt idl2

if (~obj_valid(self.oTarget)) then return
oParent = getmaindataobjectparent(self.oTarget)
if (self.axis eq 0) then $
   oParent->GetProperty, axis1Value=axisData else $
   oParent->GetProperty, axis2Value=axisData

n = n_elements(axisData)
if (n lt 2) then return   ; axisData it must be a vector to proceed

dMin = min(axisData)
dMax = max(axisData)
bins = axisData[1:n-1] - axisData[0:n-2]
dBinMin = min(bins, max=dBinMax)

;self->SetProperty,dmin=dmin,dmax=dmax,dbinmin=dbinmin,dbinmax=dbinmax
self.dmin=dmin
self.dmax=dmax
self.dbinmin=dbinmin
self.dbinmax=dbinmax
if (self.setdefault) then begin
    self.setdefault = 0
    self.min = dmin
    self.max = dmax
    self.bin = dbinmax
endif

; If there is only one independent axis modify the axis drop-list.
oParent->GetProperty, nDimensions=ndim
self->SetPropertyAttribute, 'axis', enumlist=(ndim eq 1)? ['x-axis'] : ['x-axis','y-axis']

; Impose certain bounds
self.bin = self.bin > self.dbinmin
self.min = (self.min > self.dmin) < self.dmax
self.max = (self.max < self.dmax) > self.dmin
if (self.min eq self.dmax) then self.min = self.dmax - self.bin
if (self.max eq self.dmin) then self.max = self.dmin + self.bin
;self.min = self.min < (self.max-self.bin)
;self.max = self.max > (self.min+self.bin)

end



;===============================================================================
; DAVEopDataRebin::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataRebin::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values, This is the first one
; that is created for this oCmdSet
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty,min=min,max=max,bin=bin,axis=axis,truncate=truncate
void = oCmd->AddItem('OLD_MIN',min)
void = oCmd->AddItem('OLD_MAX',max)
void = oCmd->AddItem('OLD_BIN',bin)
void = oCmd->AddItem('OLD_AXIS',axis)
void = oCmd->AddItem('OLD_TRUNCATE',truncate)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataRebin::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataRebin::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty,min=min,max=max,bin=bin,axis=axis,truncate=truncate
void = oCmd->AddItem('NEW_MIN',min)
void = oCmd->AddItem('NEW_MAX',max)
void = oCmd->AddItem('NEW_BIN',bin)
void = oCmd->AddItem('NEW_AXIS',axis)
void = oCmd->AddItem('NEW_TRUNCATE',truncate)

return, 1

end


;===============================================================================
; DAVEopDataRebin::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataRebin::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataRebin::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataRebin::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

rebinAxis = self.axis
; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('OLD_MIN',min)
void = oCmds[0]->GetItem('OLD_MAX',max)
void = oCmds[0]->GetItem('OLD_BIN',bin)
void = oCmds[0]->GetItem('OLD_AXIS',axis)
void = oCmds[0]->GetItem('OLD_TRUNCATE',truncate)
self->SetProperty, min=min,max=max,bin=bin,axis=axis,truncate=truncate, /no_update


; Undo the changes that were made to the target
void = oCmds[1]->GetItem('ID_IND',idInd)
void = oCmds[1]->GetItem('OLD_INDDATA',vin)
void = oCmds[1]->GetItem('ID_DEP',idDep)
void = oCmds[1]->GetItem('OLD_DEPDATA',zin)
oDep = oTool->GetByIdentifier(idDep)
oInd = oTool->GetByIdentifier(idInd)

if (~obj_valid(oDep) || ~obj_valid(oInd)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif
void = oDep->SetData(zin,/no_copy,/no_notify)
void = oInd->SetData(vin,/no_copy,/no_notify)


void = oCmds[1]->GetItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmds[1]->GetItem('OLD_ERRDATA',zein)
    void = oCmds[1]->GetItem('ID_ERR',idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (~obj_valid(oErr)) then begin
        oTool->_RemoveCommand, oCmdSet
        return, 0
    endif
    void = oErr->SetData(zein,/no_copy,/no_notify)
endif

void = oCmds[1]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmds[1]->GetItem('OLD_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif
 
;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataRebin::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataRebin::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get various stored properties and restore the operation with them
void = oCmds[0]->GetItem('NEW_MIN',min)
void = oCmds[0]->GetItem('NEW_MAX',max)
void = oCmds[0]->GetItem('NEW_BIN',bin)
void = oCmds[0]->GetItem('NEW_AXIS',axis)
void = oCmds[0]->GetItem('NEW_TRUNCATE',truncate)
self->SetProperty, min=min,max=max,bin=bin,axis=axis,truncate=truncate, /no_update

; Redo the changes that were made
void = oCmds[1]->GetItem('ID_IND',idInd)
void = oCmds[1]->GetItem('NEW_INDDATA',vout)
void = oCmds[1]->GetItem('ID_DEP',idDep)
void = oCmds[1]->GetItem('NEW_DEPDATA',zout)
oDep = oTool->GetByIdentifier(idDep)
oInd = oTool->GetByIdentifier(idInd)
if (~obj_valid(oDep) || ~obj_valid(oInd)) then begin
    oTool->_RemoveCommand, oCmdSet
    return, 0
endif

void = oDep->SetData(zout,/no_copy,/no_notify)
void = oInd->SetData(vout,/no_copy,/no_notify)

void = oCmds[1]->GetItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmds[1]->GetItem('NEW_ERRDATA',zeout)
    void = oCmds[1]->GetItem('ID_ERR',idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (~obj_valid(oErr)) then begin
        oTool->_RemoveCommand, oCmdSet
        return, 0
    endif
    void = oErr->SetData(zeout,/no_copy,/no_notify)
endif

void = oCmds[1]->GetItem('ID_TREATMENT',idTrmt)
oTrmt = oTool->GetByIdentifier(idTrmt)
if (obj_valid(oTrmt)) then begin
    void = oCmds[1]->GetItem('NEW_TREATMENT',trmt)
    void = oTrmt->SetData(trmt,/no_copy)
endif

;; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

return, 1

end


;===============================================================================
; DAVEopDataRebin::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataRebin::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be rebinned
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   ;; Valid types are containers with at least one independent and dependent data components
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
;   oTarget = (n_elements(oTarget) gt 0)? [oTarget,oRes] : oRes
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
oTarget = oSel[0]
if (n_elements(oTarget) eq 0) then begin
    oTool->StatusMessage, 'Selection does not contain data that can be rebinned!'
    return, obj_new()
endif


; Make the first valid target the operation's target dataset
self.oTarget = oTarget
;self.oSel = oSel[0]             ; also record the selected data containing the target dataset

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Some operation properties depend on the selected data: update them
    self.setdefault = 1
    self->_UpdateCurrentBinSettings

    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Rebin the data and record changes in undo/redo buffer
if (~self->Rebin(oTarget, oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataRebin::Rebin
; 
; PURPOSE:
;   Rebin the data conatined in the specified target and record
;   changes in the operation's undo/redo buffer.
;
; PARAMETERS:
;   oTarget [in|out] - the data container whose data is to be
;                      rebinned.
;
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0.
;
function DAVEopDataRebin::Rebin, oTarget, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get current operation settings
self->GetProperty,min=lolim,max=uplim,bin=binw,axis=axis,truncate=truncate, norebin=norebin $
                  ,reversible_operation=rev_op,expensive_computation=exp_comp

; uplim must be greater than lolim
if (lolim gt (uplim-binw)) then begin
    oTool->StatusMessage,'Data Rebin: lower limit should not be >= upper limit!'
    return, 0
endif

; We need the parent dataset object that the target is contained in.
; The parent object can be used to retrieve all relevant info such as 
; axes data or objects and the treatment buffer.
oParent = getmaindataobjectparent(oTarget)

; Get the independent data and it's distribution (ie whether histogram or point)
if (self.axis eq 0) then $
   oParent->GetProperty, axis1Ref=oInd, axis1Value=vin, axis1Distribution=distribution else $
   oParent->GetProperty, axis2Ref=oInd, axis2Value=vin, axis2Distribution=distribution
nin = n_elements(vin)
if (nin lt 2) then return, 0   ; vin must be a vector

; The dependent data
oParent->GetProperty, dataRef=oDep, dataValue=zin
if (n_elements(zin) eq 0) then return, 0

; The error in the dependent data
oParent->GetProperty, errorRef=oErr, errorValue=zein
errExist = (n_elements(zein) gt 0)

; Store copies of original data because rebin operation is not
; reversible.
oCmd = obj_new("IDLitCommand", TARGET_IDENTIFIER=oTarget->GetFullIdentifier())
oCmdSet->Add, oCmd
void = oCmd->AddItem('OLD_INDDATA',vin)
void = oCmd->AddItem('OLD_DEPDATA',zin)
;void = oCmd->AddItem('ID_TARGETPARENT',oParent->GetFullIdentifier())
void = oCmd->AddItem('ID_IND',oInd->GetFullIdentifier())
void = oCmd->AddItem('ID_DEP',oDep->GetFullIdentifier())
void = oCmd->AddItem('ERREXIST',errExist)
if (errExist) then begin
    void = oCmd->AddItem('OLD_ERRDATA',zein)
    void = oCmd->AddItem('ID_ERR',oErr->GetFullIdentifier())
endif

; Get treatment history
oParent->GetProperty, trmtRef = oTrmt    ; can also use: oTrmt = locatetreatmentobject(oTarget)
if (obj_valid(oTrmt)) then begin
   void = oTrmt->GetData(trmt)
   void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
   void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
   ;; modify treatment info accordingly
   line = '____________________________________________________'
   trmt = [trmt, $
           line, $
           'Timestamp: '+systime(), $
           'Perform a rebin of the data']
endif


; If 2D data and rebin axis is y-axis then transpoase data
transpose = 0
if ((self.axis eq 1) && ((size(zin))[0] eq 2)) then begin
    zin = transpose(zin)        ; transpose data so that y-axis data become the rows to be rebinned
    zein = transpose(zein)      ;
    transpose = 1
endif
        
lolim_mod = (lolim lt vin[0])? vin[0] : lolim
uplim_mod = (uplim gt vin[nin-1])? vin[nin-1] : uplim 

if (norebin) then begin
  ; simply truncating without rebining
  truncate = 1
  index = where(vin ge lolim_mod and vin le uplim_mod, foundValid)
  if (foundValid gt 0) then begin
    vout = vin[index]
    case zin.ndim of
      1: begin
        zout = zin[index]
        zeout = zein[index]
      end
      
      2: begin
        zout = zin[index,*]
        zeout = zein[index,*]
      end
    endcase
  endif
endif else begin
  ;; determine output bins
  nout = Fix((uplim_mod - lolim_mod)/binw) + 1
  vout = lolim_mod + Findgen(nout)*binw
  ;; do the rebinning
  if (~errExist) then zein = 0.01*zin ; fake some error if necessary!
  case Strupcase(Strmid(distribution,0,4)) of ; switch between histogram or points data rebinning
    'HIST': Drebin,vin,zin,zein,vout,zout,zeout,/hist,/to_hist,err=err,emsg=emsg
    else: Drebin,vin,zin,zein,vout,zout,zeout,/points,/to_points,err=err,emsg=emsg
  endcase
  if (err ne 0) then begin
    oTool->Statusmessage,emsg
    Return, 0
  endif
endelse

;; if the data outside the rebinning range is not to be
;; truncated then this data needs to be included in the output
if (~truncate) then begin
    ;; get any data below lolim
    res = where(vin lt lolim_mod, cnt)
    if (cnt gt 0) then begin
        vout = [vin[res],vout]
        zout = [zin[res],zout]
        zeout = [zein[res],zeout]
    endif

    ;; get any data above uplim
    res = where(vin gt uplim_mod, cnt)
    if (cnt gt 0) then begin
        vout = [vout,vin[res]]
        zout = [zout,zin[res]]
        zeout = [zeout,zein[res]]
    endif
endif


; transpose the data back, if necessary
if (transpose) then begin
    zout = transpose(zout)
    zeout = transpose(zeout)
endif

; Store rebinned data because rebin operation is expensive
void = oCmd->AddItem('NEW_INDDATA',vout)
void = oCmd->AddItem('NEW_DEPDATA',zout)
if (errExist) then void = oCmd->AddItem('NEW_ERRDATA',zeout)

; Update data objects with rebinned data using the no_notify keyword
; so that observers are not informed
void = oDep->setData(zout,/no_copy,/no_notify)
if (errExist) then void = oErr->SetData(zeout,/no_copy,/no_notify)
void = oInd->SetData(vout,/no_copy,/no_notify)

; Update treatment information
if (obj_valid(oTrmt)) then begin
    trmt = [trmt, $
            'From '+ $
            strtrim(string(lolim_mod),2)+' to '+strtrim(string(uplim_mod),2)+ $
            ' in steps of '+strtrim(string(binw),2)]

    void = oCmd->AddItem('NEW_TREATMENT',trmt) ; save modified treatment info
    void = oTrmt->SetData(trmt,/no_copy,/no_notify)
endif

; Notify observers about the change!
oDep->notifyDataChange
oDep->notifyDataComplete
oInd->notifyDataChange
oInd->notifyDataComplete
oErr->notifyDataChange
oErr->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataRebin::Cleanup
; 
; PURPOSE:
;   DAVEopDataRebin class cleanup
;
pro DAVEopDataRebin::Cleanup
compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataRebin::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataRebin::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is not reversible
; This operation is expensive
if (~self->IDLitOperation::Init(NAME='Data Rebin' $
;                                ,types=['DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP'] $
                                ,reversible_operation=0,expensive_computation=1 $
                                ,_EXTRA=etc)) then return, 0

;self->SetProperty, reversible_operation=0, expensive_computation=1

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register an offset property for this operation
self->RegisterProperty, 'axis', enumlist=['x-axis','y-axis'], description='Rebin Axis' $
  ,name='Rebin Axis',sensitive=1
self->RegisterProperty, 'dmin', /float, description='Lower limit of data' $
  ,name='Current Lower Limit',sensitive=0
self->RegisterProperty, 'dmax', /float, description='Upper limit of data' $
  ,name='Current Upper Limit',sensitive=0
self->RegisterProperty, 'dbinmin', /float, description='Mininum bin width of data' $
  ,name='Current Min Bin Width',sensitive=0
self->RegisterProperty, 'dbinmax', /float, description='Maximum bin width of data' $
  ,name='Current Max Bin Width',sensitive=0
self->RegisterProperty, 'min', /float, description='Desired lower limit' $
  ,name='Desired Lower Limit',sensitive=1
self->RegisterProperty, 'max', /float, description='Desired upper limit' $
  ,name='Desired Upper Limit',sensitive=1
self->RegisterProperty, 'bin', /float, description='Desired bin width' $
  ,name='Desired Bin Width',sensitive=1
self->RegisterProperty, 'truncate', enumlist=['No','Yes'], description='Truncate data to rebin range?' $
  ,name='Truncate data to bin range?',sensitive=1
self->Registerproperty, 'norebin', enumlist=['No','Yes'], description='Truncate data, no rebinning?' $
  ,name='Truncate without rebinning?',sensitive=1

; init offset to 0.0
self.axis = 0
self.truncate = 1
Self.norebin=0
self.setdefault = 1


; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataRebin__define
; 
; PURPOSE:
;   DAVEopDataRebin class structure definition
;
pro DAVEopDataRebin__define

compile_opt idl2

struc = {DAVEopDataRebin $
         ,inherits IDLitOperation $
         ,dMin:0.0 $            ; bin axis minimum of existing data
         ,dMax:0.0 $            ; bin axis maxinum of existing data
         ,dBinMin:0.0 $         ; min bin width of existing data
         ,dBinMax:0.0 $         ; max bin width of existing data
         ,min:0.0 $             ; min of axis range to be rebin
         ,max:0.0 $             ; max of axis range to be rebin
         ,bin:0.0 $             ; bin width to be rebin to
         ,axis:0 $              ; rebin axis (for 2D)
         ,truncate:0 $          ; set to 1 if data to be truncated 
         ,noRebin:0  $          ; set to 1 if data is to be truncated without rebining
         ,setdefault:0 $        ; a control flag
         ,oTarget:obj_new() $   ; the target object.
;         ,oSel:obj_new() $      ; the selected data item
        }

end
