; $Id$
;###############################################################################
;
; NAME:
;  SPH_BESSEL
;
; PURPOSE:
;  Implementation of the spherical bessel function for use in fitting
;  data in RAINS.
;
; CATEGORY:
;  DAVE, Data Analysis, RAINS, surface fitting
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;function sph_bessel,x,n
;return,(sqrt(0.5*!dpi/x))*beselj(x,n+0.5,/double)
;end

;; Richard Tumanjong Azuah
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;===============================================================================
;+
; Sph_bessel
;
; PURPOSE:
;   Spherical Bessel Function of the First Kind, jn, defined in terms of the ordinary Bessel functions, Jn
;   https://mathworld.wolfram.com/SphericalBesselFunctionoftheFirstKind.html
;
; PARAMETERS:
;   x - A scalar or array specifying the values for which the Spherical Bessel function is required.
;   
;   n - A scalar or array specifying the order of the Bessel function to calculate. 
;       Values for n can be integers or real numbers. If N is negative then it must be an integer.
;
; KEYWORDS:
;
; RETURNS:
;   If both arguments are scalars, the function returns a scalar. If both arguments are arrays, 
;   the function matches up the corresponding elements of X and N, returning an array with the 
;   same dimensions as the smallest array. If one argument is a scalar and the other argument 
;   is an array, the function uses the scalar value with each element of the array, and returns 
;   an array with the same dimensions as the smallest input array.
;
;   If X is double-precision, the result is double-precision, otherwise the result is single-precision.
;-
function Sph_bessel,x,n

  ; special case, j0(x) = Sinc(x)
  if (n.ndim eq 0 && n eq 0) then begin
    return, Sinc(x)
  endif
  
  sph_j = (Sqrt(0.5*!dpi/x))*Beselj(x,n+0.5,/double)

  ; make j zero whenever x is zero
  index = where(x eq 0.0, cnt)
  if (cnt gt 0) then sph_j[index] = 0.0
  
  return, sph_j
end

pro test_sph_bessel
  x = findgen(51)*0.1
  p1 = plot(x,sph_bessel(x,0))
  p1 = Plot(x,Sph_bessel(x,1),/overplot)
  p1 = Plot(x,Sph_bessel(x,2),/overplot)
  p1 = Plot(x,Sph_bessel(x,3),/overplot)
  p1 = Plot(x,Sph_bessel(x,4),/overplot)
  p1 = Plot(x,Sph_bessel(x,5),/overplot)
end
