; $Id$
;+
;===============================================================================
; NAME:
;   Condensate Induced Singularity routines
;
; PURPOSE:
;   Define subroutines used to determine contributions to the condensate induced 
;   singularity in the momentum distribution function of helium in the presence of
;   Bose-Einstein condensation
;   
;   The model momentum distribution can be represented as
;     n(k) = n0[1 + f(k)] + n*(k)
;   where n0 is the condensate fraction
;         f(k) is the condensate induced peaking term
;         n*(k) represents the momentum distribution of the uncondensed atoms
;         k is the 3-D atomic momentom
;   
;   f(k) is given by
;     f(k) = B/k*Coth(D*k)*Exp(-k^2/kc) where
;   where B, D and kc are constants, kc is cut-off value of k (usually 0.5) included so
;   that the term Exp(-k^2/kc) restricts f(k) to small values of k only.
;   
;   However, n(k) is determined as a Fourier Transform of n(s), in which
;     n(s) = n0[1+f(s)] + A*n*(s)
;     s = distance travelled by the struck particle.
;   with the terms having the same meaning as in the k-space representation above.
;   s and k are FT conjugates.
;   Thus in this file, there are three functions defined as follows
;     - Fofk is f(k) above
;     - Fofs is f(s) above
;     - Fofks is an intermediate fuction used to caculate f(s) as a FT of f(k)
;
; CATEGORY:
;   Quantum Fluids Analysis
;-



;###############################################################################
;+
; NAME:
;   Fofs
;
; PURPOSE:
;   Define Fofs. This is f(s) or the the s-space representation of f(k).
;   f(s) is the FT of  f(k).
;
; CATEGORY:
;   Quantum Fluids Analysis
;
; PARAMETERS
;   s (in) - distance travelled by the struck particle.
;
; KEYWORDS:
;
; RETURNS the value of Fofs(s)
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; 2013
;-
;###############################################################################
function Fofs, s, B=B, D=D, F=F, nOrder=nOrder, kc=kc, T=T $
             , splineFlag=splineFlag, splinePtr=splinePtr, _Extra=etc
compile_opt idl2

if (n_elements(s) eq 0) then return, 0.0

; If using a spline interpolation then do it now and return the results
if (keyword_set(splineFlag) && ptr_valid(splinePtr)) then begin
  calc = s*0.0
  x = (*splinePtr).x
  index = where(s le x[n_elements(x)-1],cnt)
  if (n_tags(*splinePtr) eq 3 && cnt gt 0) then $
    calc[index] = SPL_INTERP((*splinePtr).x, (*splinePtr).y, (*splinePtr).y2, s[index], /double)
  return, calc
endif

; Setup default values for various parameters
if (n_elements(nOrder) eq 0) then nOrder = 1
if (n_elements(T) eq 0) then T = 1.0
if (n_elements(kc) eq 0) then kc=0.5
if (n_elements(B) eq 0 || n_elements(D) eq 0 || n_elements(F) eq 0) then begin
  mass = 4.0026           ; rel a. mass of He4
  hbar = 1.0545919        ; *10**(-34) Js
  ec   = 1.6021917        ; *10**(-19) C
  amu  = 1.660531         ; *10**(-27) Kg
  kB   = 1.380622         ; *10**(-23) J/K
  rho  = 0.6022169/27.0   ; 1/A**3         He4 at SVP
  Vs   = 240              ; m/s            He4 at SVP
  B = mass*amu*Vs/(2.0*hbar*(2.0*!pi)^3*rho)/1000.0  ; A**2
  D = hbar*Vs/(2.0*kb*T)/10.0                    ; A
  F = 2.0*kc^2                                     ; (1/A)**2
endif

calc = s*0.0
;index = where(s ge 0.0 and s le 500.0,cnt)
cnt = n_elements(s)
if (cnt gt 0) then begin  
  ; Perform the calculation
  etc = create_struct('s',s[0],'B',B,'D',D,'F',F,'nOrder',nOrder,'kc',kc,'T',T)
  for i = 0, cnt-1 do begin
    etc.s = s[i]
    
    calc[i] = qpint1d('Fofks',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)
  endfor
endif
;n = n_elements(s)
;if (n eq 1) then begin
;  sValue = s
;  return, qpint1d('Fofks',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)
;endif else begin
;  fofs = []
;  for i = 0,n-1 do begin
;    etc.s = s[i]
;    fofs = [fofs,qpint1d('Fofks',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)]
;  endfor
;endelse

return, calc
end


;###############################################################################
;+
; NAME:
;   Fofks
;
; PURPOSE:
;   Define Fofks. Used to calculate f(s) as the Fourier Transform of f(k). 
;   Must integrated over k, thus giving the FT. k and s are passed in as arguments
;
; CATEGORY:
;   Quantum Fluids Analysis
;
; PARAMETERS
;   k (in) - the 1-D atomic momentum distribution
;
; KEYWORDS:
;
; RETURNS the value of Fofks(k,s) and the specified k,s
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; 2013
;-
;###############################################################################
function Fofks, k, s=s, B=B, D=D, F=F, nOrder=nOrder, kc=kc, T=T, _Extra=etc
compile_opt idl2

; Setup default values for various parameters
if (n_elements(nOrder) eq 0) then nOrder = 1
if (n_elements(T) eq 0) then T = 1.0
if (n_elements(kc) eq 0) then kc=0.5
if (n_elements(B) eq 0 || n_elements(D) eq 0 || n_elements(F) eq 0) then begin
  mass = 4.0026           ; rel a. mass of He4
  hbar = 1.0545919        ; *10**(-34) Js
  ec   = 1.6021917        ; *10**(-19) C
  amu  = 1.660531         ; *10**(-27) Kg
  kB   = 1.380622         ; *10**(-23) J/K
  rho  = 0.6022169/27.0   ; 1/A**3         He4 at SVP
  Vs   = 240              ; m/s            He4 at SVP
  B = mass*amu*Vs/(2.0*hbar*(2.0*!pi)^3*rho)/1000.0  ; A**2
  D = hbar*Vs/(2.0*kb*T)/10.0                    ; A
  F = 2.0*kc^2                                     ; (1/A)**2
endif

Fofks = k*0.0  ; init calc to zero
index= where(k ne 0.0, nonZerok)
if (nonZerok gt 0) then begin
  kk = k[index]
  Fofks[index] = 4*!pi*B*kk^nOrder*Coth(D*kk)*EXP(-kk^2/F)*Sinc(kk*s)
endif

return, Fofks
end


;###############################################################################
;+
; NAME:
;   Fofk
;
; PURPOSE:
;   Define Fofk. This is the basic definition of f(k).
;
; CATEGORY:
;   Quantum Fluids Analysis
;
; PARAMETERS
;   k (in) - the 1-D atomic momentum distribution
;
; KEYWORDS:
;
; RETURNS the value of Fofk(k)
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; 2013
;-
;###############################################################################
function Fofk, k, B=B, D=D, F=F, kc=kc, T=T, _Extra=etc
compile_opt idl2

if (n_elements(k) eq 0) then return, 0.0

; Setup default values for various parameters
if (n_elements(T) eq 0) then T = 1.0
if (n_elements(kc) eq 0) then kc=0.5
if (n_elements(B) eq 0 || n_elements(D) eq 0 || n_elements(F) eq 0) then begin
  mass = 4.0026           ; rel a. mass of He4
  hbar = 1.0545919        ; *10**(-34) Js
  ec   = 1.6021917        ; *10**(-19) C
  amu  = 1.660531         ; *10**(-27) Kg
  kB   = 1.380622         ; *10**(-23) J/K
  rho  = 0.6022169/27.0   ; 1/A**3         He4 at SVP
  Vs   = 240              ; m/s            He4 at SVP
  B = mass*amu*Vs/(2.0*hbar*(2.0*!pi)^3*rho)/1000.0  ; A**2
  D = hbar*Vs/(2.0*kb*T)/10.0                    ; A
  F = 2.0*kc^2                                     ; (1/A)**2
endif

; Perform the calculation
if (F eq 0.0) then return, k*0.0

Fofk = k*0.0  ; init calc to zero
index= where(k ne 0.0, nonZerok)
if (nonZerok gt 0) then begin
  kk = k[index]
  Fofk[index] = (B/kk)*Coth(D*kk)*EXP(-kk^2/F)
endif

return, Fofk
end


;###############################################################################
;+
; NAME:
;   Coth
;
; PURPOSE:
;   Define a Hyperbolic Cotangent
;
; CATEGORY:
;   Quantum Fluids Analysis
;
; PARAMETERS
;   x (in) - the independent variable
;
; KEYWORDS:
;
; RETURNS the value of Coth(x)
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; 2013
;-
;###############################################################################
function Coth, x
compile_opt idl2

; coth is 0.0 for x = 0.0
calc = x*0.0  ; for x = 0.0

xabs = abs(x)

; coth is essentially 1.0 for abs(x) ge 9.0
index = where(xabs ge 9.0,cnt)
if (cnt gt 0) then calc[index] = x[index]/abs(x[index])

; else use formula
index = where(xabs gt 0.0 and xabs lt 9.0,cnt)
if (cnt gt 0) then begin
  arg = x[index]
  calc[index] = (exp(arg)+exp(-arg))/(exp(arg)-exp(-arg))
endif


return, calc

end


;###############################################################################
;+
; NAME:
;   Sinc
;
; PURPOSE:
;   Define Sin(x)/x
;
; CATEGORY:
;   Quantum Fluids Analysis
;
; PARAMETERS
;   x (in) - the independent variable
;
; KEYWORDS:
;
; RETURNS the value of Sinc(x)
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; 2013
;-
;###############################################################################
function Sinc, x
compile_opt idl2

calc = x*0.0 + 1.0
index = where(x ne 0.0,cnt)
if (cnt gt 0) then calc[index] = sin(x[index])/x[index]
return, calc
  
end
