; $Id$
;###############################################################################
;
; NAME:
;  RMD_HEAVISIDE
;
; PURPOSE:
;  Implementation of a unit step function for use in data fitting.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN, curve fitting
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;+
; NAME:
;       RMD_HEAVISIDE
;
; PURPOSE:
;
;       Implementation of the Heaviside step function.  This function
;		is defined as follows:
;
;		RMD_HEAVISIDE(X) = 0.0 for X < 0.0
;		RMD_HEAVISIDE(X) = 0.5 for X = 0.0
;		RMD_HEAVISIDE(X) = 1.0 for X > 0.0
;
;		This routine has been vectorized for speed.
;
;
; AUTHOR:
;
;       Robert M. Dimeo, Ph.D.
;		NIST Center for Neutron Research
;       100 Bureau Drive
;		Gaithersburg, MD 20899
;       Phone: (301) 975-8135
;       E-mail: robert.dimeo@nist.gov
;       http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;       Mathematics
;
; CALLING SEQUENCE:
;
;       Y = RMD_HEAVISIDE(X,COUNT = COUNT,/DOUBLE)
;
; INPUT PARAMETERS:
;
;       X -	A numerical array, in general.
;
; RETURNS:
;
;   The resulting array as defined above.
;
;   If no parameters are passed in a value of -1L is returned, and COUNT
;   is set to zero.  Note that the only way to recognize this error
;   is to examine COUNT.
;
; INPUT KEYWORDS:
;
;   DOUBLE -	Setting this keyword forces output with double precision.
;
; OUTPUT KEYWORDS:
;
;   COUNT - Upon return, this is the number of elements in the result.
;           This is only important when there is an error in the result,
;			in which case COUNT is set to zero.
;
; REQUIRED PROGRAMS:
;
;        NONE
;
; EXAMPLE
;
;	IDL>	xlo = -5.0 & xhi = 5.0 & npts = 100 & dx = (xhi-xlo)/(npts-1.0)
;	IDL>	x = xlo + dx*findgen(npts)
;	IDL>	y = rmd_heaviside(x-1.0)
;	IDL>	plot,x,y,yrange = [-2.0,2.0],ystyle = 1
;
; MODIFICATION HISTORY:
;
;       Written by Rob Dimeo, September 6, 2002.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function rmd_heaviside,x,count = count,double = double
if n_params() eq 0 then begin
  count = 0
  return,-1L
endif
if keyword_set(double) then begin
  a = 1D & b = 0D & c = 0.5D
endif else begin
  a = 1. & b = 0. & c = 0.5
endelse
y = (x gt 0)*a + ((x lt 0))*b + c*(x eq 0)
count = n_elements(y)
return,y
end