;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; NAME:
;  PAN_HydrogenRotRecoil
;
; PURPOSE:
;  PAN model function to determine hydrogen recoil scattering including 
;  rotational transitional terms for use in fitting data in the impulse 
;  approximation.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Timothy Prisk
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.

;+
; ARCSResWidth
;
; PURPOSE:
;   Evaluates the instrumental resolution width at any energy transfer
;   These are semi-emperical values determined for the ARCS instrument 
;   and then fitted to a polynomial of degree 4
;   W = b0 + b1*Et + b2*Et^2 + b3*Et^3 + b4*Et^4; 
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   eTransfer:  the neutron energy transfer at which the resolution width is required
;
; RETURN VALUE:
;   the instrumental resolution width
;-
function ARCSResWidth, eTransfer, _Extra=extra
compile_opt idl2

if (n_elements(eTransfer) eq 0) then eTransfer=0.0

; parameters for ARCS instrument at Ei = 500 meV
b0 =  15.53747
b1 = -0.03454
b2 =  1.96388E-5
b3 = -2.38325E-8
b4 =  5.60666E-11

Return, b0 + b1*eTransfer + b2*eTransfer^2 + b3*eTransfer^3 + b4*eTransfer^4

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; ARCSResWidth1
;
; PURPOSE:
;   Evaluates the instrumental resolution width at the peak centers of
;   the first 5 rotational peaks for para-hydrogen.
;   These are semi-emperical values determined for the ARCS instrument
;   and then fitted to a polynomial to obtain coeeficients that can be used 
;   to estimate the widths at other Q values for the first 5 transitions.
;   W = b0 + b1*Et + b2*Et^2 + b3*Et^3 + b4*Et^4;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   eTransfer:  the neutron energy transfer at which the resolution width is required
;
; RETURN VALUE:
;   the instrumental resolution width
;-
function Arcsreswidth1, Qgroup, _Extra=extra
compile_opt idl2

if (N_elements(Qgroup) eq 0) then Qgroup=1.0

; Calculate resolution at peak center.
R1 = 15.08155D0 - 0.02166D0*Qgroup - 0.03074D0*Qgroup^2 - 4.12848D-4*Qgroup^3 + 3.12092D-5*Qgroup^4
R2 = 14.11106D0 - 0.02666D0*Qgroup - 0.02881D0*Qgroup^2 - 5.03550D-4*Qgroup^3 + 3.49062D-5*Qgroup^4
R3 = 12.72221D0 - 0.03764D0*Qgroup - 0.02517D0*Qgroup^2 - 7.01154D-4*Qgroup^3 + 4.26839D-5*Qgroup^4
R4 = 11.00586D0 - 0.06322D0*Qgroup - 0.01802D0*Qgroup^2 - 0.00116D0*Qgroup^3 + 5.99657D-5*Qgroup^4
R5 = 9.1162D0  - 0.12902D0*Qgroup - 0.00205D0*Qgroup^2 - 0.00233D0*Qgroup^3 + 1.02931D-4*Qgroup^4

Return, [R1,R2,R3,R4,R5]

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilAA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. :
;   - the Impulse Approximation is not
;   - the Additive Approach model for the momentum distribution is used
;   - the momentum distribution is a Gaussian + a small correction
;   - Final State Effects (FSE) taken into account thorugh a an additive first order correction to the scattering
;   - The relative peak intensities of the peaks modified by the rotational transitions are determined by the
;     Young-Kopel theory. The expressions for the intensities are those specified by eqns (20) to (24) in 
;     Dawidowski et al, PRB vol 73, Pg 144203 (2006) DOI: https://doi.org/10.1103/PhysRevB.73.144203
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;
;
; RETURN VALUE:
;
;-
function Pan_HydrogenRecoilAAYK,x,parms,$
  parmnames = parmnames, $
  canDraw = canDraw, $
  drawMessage = drawMessage, $
  initParms = initParms, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, $
  qvals=qvals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, xrange=xrange, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  compile_opt idl2

  canDraw = 0
  twoDimFlag = 0
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if N_params() eq 0 then begin

    parmnames = ['KE [K]              ', 'delta','a3bar [1/(A^2 ps)]','a4bar [1/(A^2 ps^2)]', 'shift','scaleFac']
    initParms = [60.0,0.0,40.0,0.0,0.0,0.001]
    Return,-1
  endif

  ; Parameter definition and initalization.
  KE = parms[0]
  delta = parms[1]    ; ratio of alpha4/alpha2^2
  a3bar = parms[2]    ; 1/(A^2 ps)
  a4bar = parms[3]    ; 1/(A^2 ps^2)
;  A1    = parms[4]
;  A2    = parms[5]
;  A3    = parms[6]
;  A4    = parms[7]
;  A5    = parms[8]
  shift = parms[4]
  sf    = parms[5]   ; arb scale factor to scale model to data

  ; Define constants.
  hbarsquared = 4.18016 ; meV amu A^2
  m  = 1.007825*2.0     ; amu
  B0 = 7.3567           ; meV
  D0 = 0.00569          ; meV
  H0 = 6.4e-6           ; meV

  hbar    = 6.35078D0         ; amu A^2/ps
  hbar2   = 4.18016D0         ; meV amu A^2
  m       = 1.007825D0*2.0D0  ; amu
  B0      = 7.3567D0          ; meV
  D0      = 0.00569D0         ; meV
  H0      = 6.4D-6            ; meV
  KpermeV = 11.60451812D0     ; NIST CODATA value.
  scaleFac = 9.64853          ; 1 meV = 9.64853 amu A^2/ps^2

  ; Convert kinetic energy to peak width.
  KE_meV = KE/KpermeV
  sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
  sigmaS = (hbarsquared/m)*Qgroup*sigmaK

  ; Calculate cumulants.
  alpha2 = (2.0D0/3.0D0)*(m/hbarsquared)*(KE_meV)                   ; A^{-2}
  alpha4 = delta*alpha2^2                                           ; A^{-4}
  mu2    = hbarsquared*(hbarsquared*Qgroup^2/m^2)*alpha2            ; meV^2
  mu3    = (hbar^5*Qgroup^2/m^2)*a3bar*(1.0D0/scaleFac)^3           ; meV^3
  mu4    = (hbar^6)*((Qgroup/m)^2)*a4bar*(1.0D0/scaleFac)^4         ;
  mu4    = mu4 + (hbar^8)*((Qgroup/m)^4)*alpha4*(1.0D0/scaleFac)^4  ; meV^4


  ; Determine peak positions.
  Recoil = 0.5D0*(hbarsquared/m)*Qgroup^2
  Rot1 = B0*1.0D0*(1.0D0 + 1.0D0) - D0*(1.0D0*(1.0D0 + 1.0D0))^2 + H0*(1.0D0*(1.0D0 + 1.0D0))^3
  Rot2 = B0*2.0D0*(2.0D0 + 1.0D0) - D0*(2.0D0*(2.0D0 + 1.0D0))^2 + H0*(2.0D0*(2.0D0 + 1.0D0))^3
  Rot3 = B0*3.0D0*(3.0D0 + 1.0D0) - D0*(3.0D0*(3.0D0 + 1.0D0))^2 + H0*(3.0D0*(3.0D0 + 1.0D0))^3
  Rot4 = B0*4.0D0*(4.0D0 + 1.0D0) - D0*(4.0D0*(4.0D0 + 1.0D0))^2 + H0*(4.0D0*(4.0D0 + 1.0D0))^3
  Rot5 = B0*5.0D0*(5.0D0 + 1.0D0) - D0*(5.0D0*(5.0D0 + 1.0D0))^2 + H0*(5.0D0*(5.0D0 + 1.0D0))^3
  C1 = Recoil + Rot1 - shift
  C2 = Recoil + Rot2 - shift
  C3 = Recoil + Rot3 - shift
  C4 = Recoil + Rot4 - shift
  C5 = Recoil + Rot5 - shift


  ; Calculate resolution at peak center.
  ResWidths = Arcsreswidth1(qGroup)

  ; Observed with is intrinsic width and res width added in quadrature .
  StoW = Sqrt(8.0*Alog(2.0))
  mu2_p1 = mu2 + (ResWidths[0]/StoW)^2
  mu2_p2 = mu2 + (ResWidths[1]/StoW)^2
  mu2_p3 = mu2 + (ResWidths[2]/StoW)^2
  mu2_p4 = mu2 + (ResWidths[3]/StoW)^2
  mu2_p5 = mu2 + (ResWidths[4]/StoW)^2

  ; The model is a sum of five peaks.
  ; Consisting of a recoil peak (additive approach model with final state corrections included)
  ; The recoil peaks are then modulated by the the rotational transitions of molecular para-hydrogen


  inv_sqrt2pi = 1.0D0/Sqrt(2.0D0*!PI)   ; Precompute Gaussian normalization constant.

  ; Calculate AA final state effects to order 1/Q.
  IApeak1 = inv_sqrt2pi/Sqrt(mu2_p1)*Exp(-0.5D0*(x - C1)^2/mu2_p1)
  IApeak2 = inv_sqrt2pi/Sqrt(mu2_p2)*Exp(-0.5D0*(x - C2)^2/mu2_p2)
  IApeak3 = inv_sqrt2pi/Sqrt(mu2_p3)*Exp(-0.5D0*(x - C3)^2/mu2_p3)
  IApeak4 = inv_sqrt2pi/Sqrt(mu2_p4)*Exp(-0.5D0*(x - C4)^2/mu2_p4)
  IApeak5 = inv_sqrt2pi/Sqrt(mu2_p5)*Exp(-0.5D0*(x - C5)^2/mu2_p5)

  z1 = (x - C1)
  z2 = (x - C2)
  z3 = (x - C3)
  z4 = (x - C4)
  z5 = (x - C5)

  odd1 = -(0.5D0*mu3/mu2_p1^2)*z1*(1.0D0 - 0.5D0*z1^2/mu2_p1)
  odd2 = -(0.5D0*mu3/mu2_p2^2)*z2*(1.0D0 - 0.5D0*z2^2/mu2_p2)
  odd3 = -(0.5D0*mu3/mu2_p3^2)*z3*(1.0D0 - 0.5D0*z3^2/mu2_p3)
  odd4 = -(0.5D0*mu3/mu2_p4^2)*z4*(1.0D0 - 0.5D0*z4^2/mu2_p4)
  odd5 = -(0.5D0*mu3/mu2_p5^2)*z5*(1.0D0 - 0.5D0*z5^2/mu2_p5)

  even1 = (mu4/(8.0D0*mu2_p1^2))*(1.0D0 - 2.0D0*z1^2/mu2_p1 + z1^4/(3.0D0*mu2_p1^2))
  even2 = (mu4/(8.0D0*mu2_p2^2))*(1.0D0 - 2.0D0*z2^2/mu2_p2 + z2^4/(3.0D0*mu2_p2^2))
  even3 = (mu4/(8.0D0*mu2_p3^2))*(1.0D0 - 2.0D0*z3^2/mu2_p3 + z3^4/(3.0D0*mu2_p3^2))
  even4 = (mu4/(8.0D0*mu2_p4^2))*(1.0D0 - 2.0D0*z4^2/mu2_p4 + z4^4/(3.0D0*mu2_p4^2))
  even5 = (mu4/(8.0D0*mu2_p5^2))*(1.0D0 - 2.0D0*z5^2/mu2_p5 + z5^4/(3.0D0*mu2_p5^2))

  FSE1 = odd1 + even1
  FSE2 = odd2 + even2
  FSE3 = odd3 + even3
  FSE4 = odd4 + even4
  FSE5 = odd5 + even5
  
  ; Peak intensities based on Young-Kopel theory
  ; from Eqns (20) through (24) in Dawidowski et al, PRB vol 73, Pg 144203 (2006)
  ; DOI: https://doi.org/10.1103/PhysRevB.73.144203
  ; Only the para-contribution is used.
  ; Ortho contribution and coherent scattering evaluated and found to be small (Tim Prisk)
  Ak = fltarr(5)
  h2_radius = 0.37  ; in Angstrom
  arg = Qgroup*h2_radius
  for k=1,5 do Ak[k-1] = (2*k + 1)*(Sph_bessel(arg,k))^2    ; A_p = (2k + 1)*(j_k(Q*a))^2; k goes from 1 to 5

  ; The model is a sum of five peaks.
  peak1 = Ak[0]*IApeak1*(1.0D0 + FSE1)
  peak2 = Ak[1]*IApeak2*(1.0D0 + FSE2)
  peak3 = Ak[2]*IApeak3*(1.0D0 + FSE3)
  peak4 = Ak[3]*IApeak4*(1.0D0 + FSE4)
  peak5 = Ak[4]*IApeak5*(1.0D0 + FSE5)

  Return, sf*(peak1 + peak2 + peak3 + peak4 + peak5)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilAA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. :
;   - the Impulse Approximation is not
;   - the Additive Approach model for the momentum distribution is used
;   - the momentum distribution is a Gaussian + a small correction
;   - Final State Effects (FSE) taken into account thorugh a an additive first order correction to the scattering
;   - there are no restrictions in the relative peak intensities of the peaks modified by the rotational transitions
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;
;
; RETURN VALUE:
;
;-
function Pan_HydrogenRecoilAA,x,parms,$
  parmnames = parmnames, $
  canDraw = canDraw, $
  drawMessage = drawMessage, $
  initParms = initParms, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, $
  qvals=qvals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, xrange=xrange, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

compile_opt idl2

canDraw = 0
twoDimFlag = 0
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if N_params() eq 0 then begin

  parmnames = ['KE [K]', 'delta','a3bar [1/(A^2 ps)]','a4bar [1/(A^2 ps^2)]','A1', 'A2', 'A3', 'A4', 'A5', 'shift','scaleFac']
  initParms = [60.0,0.0,40.0,0.0,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
  Return,-1
endif

; Parameter definition and initalization.
KE = parms[0]
delta = parms[1]    ; ratio of alpha4/alpha2^2
a3bar = parms[2]    ; 1/(A^2 ps)
a4bar = parms[3]    ; 1/(A^2 ps^2)
A1    = parms[4]
A2    = parms[5]
A3    = parms[6]
A4    = parms[7]
A5    = parms[8]
shift = parms[9]
sf    = parms[10]   ; arb scale factor to scale model to data

; Define constants.
hbarsquared = 4.18016 ; meV amu A^2
m  = 1.007825*2.0     ; amu
B0 = 7.3567           ; meV
D0 = 0.00569          ; meV
H0 = 6.4e-6           ; meV

hbar    = 6.35078D0         ; amu A^2/ps
hbar2   = 4.18016D0         ; meV amu A^2
m       = 1.007825D0*2.0D0  ; amu
B0      = 7.3567D0          ; meV
D0      = 0.00569D0         ; meV
H0      = 6.4D-6            ; meV
KpermeV = 11.60451812D0     ; NIST CODATA value.
scaleFac = 9.64853          ; 1 meV = 9.64853 amu A^2/ps^2

; Convert kinetic energy to peak width.
KE_meV = KE/KpermeV
sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
sigmaS = (hbarsquared/m)*Qgroup*sigmaK

; Calculate cumulants.
alpha2 = (2.0D0/3.0D0)*(m/hbarsquared)*(KE_meV)                   ; A^{-2}
alpha4 = delta*alpha2^2                                           ; A^{-4}
mu2    = hbarsquared*(hbarsquared*Qgroup^2/m^2)*alpha2            ; meV^2
mu3    = (hbar^5*Qgroup^2/m^2)*a3bar*(1.0D0/scaleFac)^3           ; meV^3
mu4    = (hbar^6)*((Qgroup/m)^2)*a4bar*(1.0D0/scaleFac)^4         ;
mu4    = mu4 + (hbar^8)*((Qgroup/m)^4)*alpha4*(1.0D0/scaleFac)^4  ; meV^4


; Determine peak positions.
Recoil = 0.5D0*(hbarsquared/m)*Qgroup^2
Rot1 = B0*1.0D0*(1.0D0 + 1.0D0) - D0*(1.0D0*(1.0D0 + 1.0D0))^2 + H0*(1.0D0*(1.0D0 + 1.0D0))^3
Rot2 = B0*2.0D0*(2.0D0 + 1.0D0) - D0*(2.0D0*(2.0D0 + 1.0D0))^2 + H0*(2.0D0*(2.0D0 + 1.0D0))^3
Rot3 = B0*3.0D0*(3.0D0 + 1.0D0) - D0*(3.0D0*(3.0D0 + 1.0D0))^2 + H0*(3.0D0*(3.0D0 + 1.0D0))^3
Rot4 = B0*4.0D0*(4.0D0 + 1.0D0) - D0*(4.0D0*(4.0D0 + 1.0D0))^2 + H0*(4.0D0*(4.0D0 + 1.0D0))^3
Rot5 = B0*5.0D0*(5.0D0 + 1.0D0) - D0*(5.0D0*(5.0D0 + 1.0D0))^2 + H0*(5.0D0*(5.0D0 + 1.0D0))^3
C1 = Recoil + Rot1 - shift
C2 = Recoil + Rot2 - shift
C3 = Recoil + Rot3 - shift
C4 = Recoil + Rot4 - shift
C5 = Recoil + Rot5 - shift


; Calculate resolution at peak center.
ResWidths = ARCSResWidth1(qGroup)

; Observed with is intrinsic width and res width added in quadrature .
StoW = Sqrt(8.0*Alog(2.0))
mu2_p1 = mu2 + (ResWidths[0]/StoW)^2
mu2_p2 = mu2 + (ResWidths[1]/StoW)^2
mu2_p3 = mu2 + (ResWidths[2]/StoW)^2
mu2_p4 = mu2 + (ResWidths[3]/StoW)^2
mu2_p5 = mu2 + (ResWidths[4]/StoW)^2

; The model is a sum of five peaks.
; Consisting of a recoil peak (additive approach model with final state corrections included)
; The recoil peaks are then modulated by the the rotational transitions of molecular para-hydrogen


inv_sqrt2pi = 1.0D0/Sqrt(2.0D0*!PI)   ; Precompute Gaussian normalization constant.

; Calculate AA final state effects to order 1/Q.
IApeak1 = A1*inv_sqrt2pi/sqrt(mu2_p1)*exp(-0.5D0*(x - C1)^2/mu2_p1)
IApeak2 = A2*inv_sqrt2pi/sqrt(mu2_p2)*exp(-0.5D0*(x - C2)^2/mu2_p2)
IApeak3 = A3*inv_sqrt2pi/sqrt(mu2_p3)*exp(-0.5D0*(x - C3)^2/mu2_p3)
IApeak4 = A4*inv_sqrt2pi/sqrt(mu2_p4)*exp(-0.5D0*(x - C4)^2/mu2_p4)
IApeak5 = A5*inv_sqrt2pi/sqrt(mu2_p5)*exp(-0.5D0*(x - C5)^2/mu2_p5)

z1 = (x - C1)
z2 = (x - C2)
z3 = (x - C3)
z4 = (x - C4)
z5 = (x - C5)

odd1 = -(0.5D0*mu3/mu2_p1^2)*z1*(1.0D0 - 0.5D0*z1^2/mu2_p1)
odd2 = -(0.5D0*mu3/mu2_p2^2)*z2*(1.0D0 - 0.5D0*z2^2/mu2_p2)
odd3 = -(0.5D0*mu3/mu2_p3^2)*z3*(1.0D0 - 0.5D0*z3^2/mu2_p3)
odd4 = -(0.5D0*mu3/mu2_p4^2)*z4*(1.0D0 - 0.5D0*z4^2/mu2_p4)
odd5 = -(0.5D0*mu3/mu2_p5^2)*z5*(1.0D0 - 0.5D0*z5^2/mu2_p5)

even1 = (mu4/(8.0D0*mu2_p1^2))*(1.0D0 - 2.0D0*z1^2/mu2_p1 + z1^4/(3.0D0*mu2_p1^2))
even2 = (mu4/(8.0D0*mu2_p2^2))*(1.0D0 - 2.0D0*z2^2/mu2_p2 + z2^4/(3.0D0*mu2_p2^2))
even3 = (mu4/(8.0D0*mu2_p3^2))*(1.0D0 - 2.0D0*z3^2/mu2_p3 + z3^4/(3.0D0*mu2_p3^2))
even4 = (mu4/(8.0D0*mu2_p4^2))*(1.0D0 - 2.0D0*z4^2/mu2_p4 + z4^4/(3.0D0*mu2_p4^2))
even5 = (mu4/(8.0D0*mu2_p5^2))*(1.0D0 - 2.0D0*z5^2/mu2_p5 + z5^4/(3.0D0*mu2_p5^2))

FSE1 = odd1 + even1
FSE2 = odd2 + even2
FSE3 = odd3 + even3
FSE4 = odd4 + even4
FSE5 = odd5 + even5

; The model is a sum of five peaks.
peak1 = IApeak1*(1.0D0 + FSE1)
peak2 = IApeak2*(1.0D0 + FSE2)
peak3 = IApeak3*(1.0D0 + FSE3)
peak4 = IApeak4*(1.0D0 + FSE4)
peak5 = IApeak5*(1.0D0 + FSE5)

Return, sf*(peak1 + peak2 + peak3 + peak4 + peak5)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilIA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. Assumes:
;   - the momentum distribution is Gaussian
;   - the Impulse Approximation is valid
;   - there are no restrictions in the relative peak intensities of the peaks modified by the rotational transitions
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   
;
; RETURN VALUE:
;   
;-
function Pan_hydrogenRecoilIA,x,parms,$
  parmnames = parmnames, $
  canDraw = canDraw, $
  drawMessage = drawMessage, $
  initParms = initParms, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, $
  qvals=qvals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, xrange=xrange, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  compile_opt idl2

  canDraw = 0
  twoDimFlag = 0
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if N_params() eq 0 then begin

    parmnames = ['KE [K]', 'A1', 'A2', 'A3', 'A4', 'A5', 'A6', 'shift','scaleFac']
    initParms = [60.0,1.0,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
    Return,-1
  endif

  ; Parameter definition and initalization.
  KE = parms[0]
  A1 = parms[1]
  A2 = parms[2]
  A3 = parms[3]
  A4 = parms[4]
  A5 = parms[5]
  A6 = parms[6]
  shift = parms[7]
  sf = parms[8]

  ; Define constants.
  hbarsquared = 4.18016 ; meV amu A^2
  m  = 1.007825*2.0     ; amu
  B0 = 7.3567           ; meV
  D0 = 0.00569          ; meV
  H0 = 6.4e-6           ; meV

  ; Convert kinetic energy to peak width.
  KE_meV = KE/11.6
  sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
  sigmaS = (hbarsquared/m)*Qgroup*sigmaK

  ; Determine peak positions.
  Recoil = 0.5*(hbarsquared/m)*Qgroup^2.0

  Rot1 = B0*1.0*(1.0 + 1.0) - D0*(1.0*(1.0 + 1.0))^2.0 + H0*(1.0*(1.0 + 1.0))^3.0
  Rot2 = B0*2.0*(2.0 + 1.0) - D0*(2.0*(2.0 + 1.0))^2.0 + H0*(2.0*(2.0 + 1.0))^3.0
  Rot3 = B0*3.0*(3.0 + 1.0) - D0*(3.0*(3.0 + 1.0))^2.0 + H0*(3.0*(3.0 + 1.0))^3.0
  Rot4 = B0*4.0*(4.0 + 1.0) - D0*(4.0*(4.0 + 1.0))^2.0 + H0*(4.0*(4.0 + 1.0))^3.0
  Rot5 = B0*5.0*(5.0 + 1.0) - D0*(5.0*(5.0 + 1.0))^2.0 + H0*(5.0*(5.0 + 1.0))^3.0
  Rot6 = B0*6.0*(6.0 + 1.0) - D0*(6.0*(6.0 + 1.0))^2.0 + H0*(6.0*(6.0 + 1.0))^3.0

  C1 = Recoil + Rot1 - shift
  C2 = Recoil + Rot2 - shift
  C3 = Recoil + Rot3 - shift
  C4 = Recoil + Rot4 - shift
  C5 = Recoil + Rot5 - shift
  C6 = Recoil + Rot6 - shift

  ; Calculate resolution at peak center.
  ResWidths = Arcsreswidth([C1,C2,C3,C4,C5,C6])

  ; Observed with is intrinsic width and res width added in quadrature .
  StoW = Sqrt(8.0*Alog(2.0))
  sigmaO = Sqrt((ResWidths/StoW)^2 + sigmaS^2)

  ; The model is a sum of six peaks.
  peak1 = (A1/Sqrt(2.0*!dpi*sigmaO[0]^2))*Exp(-0.5*((x-C1)/sigmaO[0])^2)
  peak2 = (A2/Sqrt(2.0*!dpi*sigmaO[1]^2))*Exp(-0.5*((x-C2)/sigmaO[1])^2)
  peak3 = (A3/Sqrt(2.0*!dpi*sigmaO[2]^2))*Exp(-0.5*((x-C3)/sigmaO[2])^2)
  peak4 = (A4/Sqrt(2.0*!dpi*sigmaO[3]^2))*Exp(-0.5*((x-C4)/sigmaO[3])^2)
  peak5 = (A5/Sqrt(2.0*!dpi*sigmaO[4]^2))*Exp(-0.5*((x-C5)/sigmaO[4])^2)
  peak6 = (A6/Sqrt(2.0*!dpi*sigmaO[5]^2))*Exp(-0.5*((x-C6)/sigmaO[5])^2)

  Return, sf*(peak1 + peak2 + peak3 + peak4 + peak5 + peak6)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


pro Test_Pan_HydrogenRecoil, func=func_choice, qVal=qVal, yout=yout, plot=plot
  compile_opt idl2

  func_choice = (n_elements(func_choice) eq 0)? 0 : func_choice
  if (~Isa(qVal)) then qVal = 5.0
  x = Findgen(301)*2.0 - 50.0
  
  case func_choice of
    0: begin
      parms = [60.0,1.0,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
      void = Pan_HydrogenRecoilIA(parmnames=pnames)
      yout = Pan_HydrogenRecoilIA(x,parms,qgroup=qVal)
      if (Keyword_set(plot)) then begin
        title = 'Hydrogen Recoil Peaks IA'
        p = Plot(x,yout,title=title,symbol=0,linestyle=0)
      endif

      end  
        
    1: begin
      parms = [60.0,0.0,1.0,0.5,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
      void = Pan_HydrogenRecoilAA(parmnames=pnames)
      yout = Pan_HydrogenRecoilAA(x,parms,qgroup=qVal)
      if (Keyword_set(plot)) then begin
        title = 'Hydrogen Recoil Peaks AA'
        p = Plot(x,yout,title=title,symbol=0,linestyle=0)
      endif

      end

      2: begin
        parms = [60.0,0.0,1.0,0.5,0.0,0.001]
        void = Pan_hydrogenrecoilaayk(parmnames=pnames)
        yout = Pan_hydrogenrecoilaayk(x,parms,qgroup=qVal)
        if (Keyword_set(plot)) then begin
          title = 'Hydrogen Recoil Peaks AA+YK'
          p = Plot(x,yout,title=title,symbol=0,linestyle=0)
        endif

      end
          
    else:
  endcase



end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
