; $Id$
;###############################################################################
;
; NAME:
;  PAN_TRANSLATIONALDIFFUSION_2D
;
; PURPOSE:
;  Implementation of the translational diffusion model for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function pan_translationaldiffusion_2d,x,parms,				$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 1           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin  
  parmnames = ['Diffusion constant', $
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]
  single_parmnames = parmnames[0]
  multi_parmnames = ['Intensity','Center']
  
  ; Define hbar according to the energy units
  eUnits = xunits.Toupper()
  eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
  if (eUnitsKnown) then begin
    if (eUnits.Equals('UEV')) then hbar = 658.21        ; ueV.ps
    if (eUnits.Equals('MEV')) then hbar = 0.65821       ; meV.ns
  endif else begin
    ; Unknown energy units so let user specify
    msg = 'The unit of energy for the data could not be determined!'
    msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
    msg = [msg,'Click Yes if energy unit is ueV',' ']
    msg = [msg,'Or No if energy unit is meV']
    title = 'Translational Diffussion 2D -- Specify Energy Unit'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
    hbar = (decision.Equals('Yes'))? 658.0 : 0.658
  endelse
  if (Isa(func_dataHash,'hash')) then func_dataHash['hbar'] = hbar

	if arg_present(initParms) then begin
		intensity = max(yVals,dimension = 1,index)
		index = index - indgen(nq)*nx ; index contains the 1D subscripts so need to reform
		center = xvals[index]

    ; assuming a single main peak then estimate FWHM
    fwhm = fltarr(nq)
    for i=0,nq-1 do begin
      hm = 0.5*intensity[i]
      res = where(yVals[*,i] ge hm, found)
      if (found gt 0) then begin
        fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
      endif
    endfor
		qVal   = qVals[groupNumber-1]
		d = 0.5*fwhm[groupNumber-1]/(hbar*qVal^2)

		;if strupcase(xunits) eq 'MEV' then d = d*1.d-4 else d = d*1.d-7

		initParms = [d,intensity,center]
	endif
	
  return,-1
endif

hbar = 658.0  ; ueV.ps [microVolts . picoseconds]
              ; assumes energy units in microvolts!
if (Isa(func_dataHash,'hash') && func_dataHash.Haskey('hbar')) then hbar = func_dataHash['hbar']

D = ux#(parms[0])#uq
;if strupcase(xunits) eq 'MEV' then D = D*1.d4 else D = D*1.d7
gam = hbar*D*(ux#qVals)^2

index = indgen(nq) + 1
intensity = ux#(parms[index])
center = ux#(parms[index+nq])
yout = intensity*((gam/!dpi))/((x#uq-center)^2+(gam)^2)

;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
;  yout = Pan_convolute(x,yout,resPtr)
;endif

return,yout
end
