; $Id$
;###############################################################################
;
; NAME:
;  PAN_NFoldJumpDiffusion_2D
;
; PURPOSE:
;  Implementation of the n-fold rotational jump diffusion on a circle for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_NFoldJumpDiffusion_2D,x,parms,					$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  parmnames = ['N','Tau','Radius', $
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]
  single_parmnames = parmnames[0:2]
  multi_parmnames = ['Intensity','Center']

  ; Define hbar according to the energy units
  eUnits = xunits.Toupper()
  eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
  if (eUnitsKnown) then begin
    if (eUnits.Equals('UEV')) then hbar = 658.21
    if (eUnits.Equals('MEV')) then hbar = 0.65821
  endif else begin
    ; Unknown energy units so let user specify
    msg = 'The unit of energy for the data could not be determined!'
    msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
    msg = [msg,'Click Yes if energy unit is ueV',' ']
    msg = [msg,'Or No if energy unit is meV']
    title = 'Translational Diffussion 2D -- Specify Energy Unit'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
    hbar = (decision.Equals('Yes'))? 658.0 : 0.658
  endelse
  if (Isa(func_dataHash,'hash')) then func_dataHash['hbar'] = hbar

  if arg_present(initParms) then begin
    intensity = Max(yVals,dimension = 1,index)
    index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming a single main peak then estimate FWHM for current group
    hm = 0.5*intensity[groupNumber-1]
    res = Where(yVals[*,groupNumber-1] ge hm, found)
    if (found gt 0) then begin
      fwhm = xvals[res[found-1]] - xvals[res[0]]
    endif

		n = 3
		tau = 2.0*hbar/fwhm             ; convert FWHM for current group to tau 
		                                ; tau is in 1/ns for [E]=ueV and 1/ps for [E]=meV
		radius = 1.0                    ; in angstroms
		initParms = [n,tau,radius,intensity,center]
	endif
  return,-1
endif
if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

hbar = 658.0  ; ueV.ps [microVolts . picoseconds]
; assumes energy units in microvolts!
if (Isa(func_dataHash,'hash') && func_dataHash.Haskey('hbar')) then hbar = func_dataHash['hbar']

N = parms[0]
tau = parms[1]
R = parms[2]

; calculate the structure factors, Ai(Q), for i=1,N-1
nn = 1.0+findgen(N)               ; nn == index from 1 to N-1
rn = 2.0*R*sin(nn*!dpi/N)         ; jump distance as function of n (nn)
un = 1+bytarr(N-1)                ; unit vector along n
j0 = (sph_bessel((qVals#rn),0))	  ; j0 as function of Q and n (nn)

lmin = 1 & lmax = N-1             ; range of l
al = fltarr(N-1,nq)               ; 
lor = 0.0
index = findgen(nq)+3
intensity = ux#(parms[index])
center = ux#(parms[index+nq])
;center1 = parms[3+ny:3+2*ny-1]

; evaluate the Lorentzian and delta components
for l = lmin,lmax do begin
	ali = (total(j0*(uq#cos(2.0*l*nn*!dpi/N)),2))/N    ; al for each l value
	al[l-1,*] = transpose(ali)
	taul = 0.5*tau/(sin(!dpi*l/N))^2
	gam = hbar/taul
	lor = lor + (ux#ali)*(ux#(gam/!dpi)#uq)/((x#uq-center)^2+(ux#gam#uq)^2)
endfor

; delta function component
a0 = 1.0 - total(al,1)                  ; A0 = 1-sum(Al)
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = a0                  ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position

; Perform convolution with resolution
; the lor is convoluted seperatedly as well as the delta and 
; a sum of the two is returned by pan_convolute()
yout = intensity*pan_convolute(x,resPtr,lor,pseudo_delta = pseudo_delta)

return,yout
end
