; +
; NAME:
;
;  HC_MODEL
;
; PURPOSE:
;
;  Procedure that defines the two-dimensional diffusion
;  on a honeycomb lattice as derived by Ron Cappelletti.
;
; AUTHOR:
;
;  Robert Dimeo
;  National Institute of Standards and Technology
;  Center for Neutron Research
;  100 Bureau Drive, Mail Stop 8562
;  Gaithersburg, MD 20899
;  Tel: (301) 975-8135
;  Email: robert.dimeo@nist.gov
;
; CATEGORY:
;
;  Data analysis
;
; REQUIREMENTS
;
;  IDL 6.0 or higher
;
; ADDITIONAL REQUIRED PROGRAMS
;
;  HC_LORENTZIAN.PRO
;
; MODIFICATION HISTORY:
;
;  Writing began -- 6/21/04 (RMD)
;  Eliminated array copying as much as I could using
;     temporary -- 09/27/04 (RMD)
; -
; *************************************************** ;
function hc_grid,xlo,xhi,nx
return,xlo+((xhi-xlo)/(nx-1.0))*dindgen(nx)
end
; *************************************************** ;
pro hc_model,  xvals,            $
               parms,            $
               yfit,             $
               _Extra = extra
;tstart = systime(/seconds)
nx = extra.nx
ny = extra.ny
x = xvals[0:nx-1]
q = xvals[nx:nx+ny-1]

ue = 1+bytarr(nx)

; "Decode" the parameters
r = parms[0]               ; hopping rate in ueV
l = parms[1]               ; jump distance
eisf = parms[2]            ; elastic fraction
int = parms[3:3+ny-1]      ; integrated intensity
cen = parms[3+ny:3+2*ny-1] ; center of lineshape

; Get the resolution function parameters out and
; create the resolution function
res_func = dblarr(nx,ny,/nozero)
for i = 0,ny-1 do begin
   area = (*(*extra.res_ptr)(i).parea)
   center = (*(*extra.res_ptr)(i).pcenter)
   fwhm = (*(*extra.res_ptr)(i).pfwhm)

   for j = 0,(*extra.res_ptr)(i).ncurves-1 do begin
      if j eq 0 then $
         res_func[0,i] =   hc_lorentzian( x,area[j],              $
                                          center[j]+cen[i],       $
                                          fwhm[j])                $
      else $
         res_func[0,i] =   res_func[*,i]+                         $
                           hc_lorentzian( x,area[j],              $
                                          center[j]+cen[i],       $
                                          fwhm[j]                 )
   endfor
endfor

ntheta = extra.ntheta
nphi = extra.nphi
theta =  hc_grid(0.0,!pi,ntheta)
phi   =  hc_grid(0.0,2.0*!pi,nphi)
dtheta = theta[1]-theta[0]
dphi = phi[1]-phi[0]
pow_ave = dblarr(nx,ny,/nozero)
cos_term = sqrt(3.)/2.
for i = 0,ny-1 do begin
   area = (*(*extra.res_ptr)(i).parea)
   center = (*(*extra.res_ptr)(i).pcenter)
   fwhm = (*(*extra.res_ptr)(i).pfwhm)

   a = q[i]*l*sin(theta)#(cos(phi))                ; ntheta x nphi
;   b = q[i]*l*cos(!pi/6.)*(sin(theta))#(sin(phi))  ; ntheta x nphi
   b = q[i]*l*cos_term*(sin(theta))#(sin(phi))     ; ntheta x nphi
   ff = sqrt(1.+4.*cos(b)*cos(1.5*a)+4*(cos(b))^2) ; ntheta x nphi
;   mag_a = r*ff/3.0                                ; ntheta x nphi
   c = (cos(a)+2.0*cos(0.5*a)*cos(b))/ff           ; ntheta x nphi
   w1 = r*(1.0-ff/3.0)                             ; ntheta x nphi
   w2 = r*(1.0+ff/3.0)                             ; ntheta x nphi

;  We'll do the convolution here prior to doing the
;  powder average.  We can save a lot in computational
;  time doing this now with the resolution function
;  represented as a sum of Lorentzians.

   gam = 0.5*fwhm
   sd = dblarr(nx,ntheta,nphi,/nozero)

   for k = 0,(*extra.res_ptr)(i).ncurves-1 do begin
      for j = 0,nx-1 do begin
         if k eq 0 then $
            sd[j,*,*] = (hc_lorentzian(x[j],area[k]*(1.0+c),         $
                                       center[k],2.0*(w1+gam[k])) +  $
                        hc_lorentzian( x[j],area[k]*(1.0-c),         $
                                       center[k],2.0*(w2+gam[k])))/2.0  $
         else $
            sd[j,*,*] = sd[j,*,*] + $
                        (hc_lorentzian(x[j],area[k]*(1.0+c),         $
                                       center[k],2.0*(w1+gam[k])) +  $
                        hc_lorentzian( x[j],area[k]*(1.0-c),         $
                                       center[k],2.0*(w2+gam[k])))/2.0
      endfor
   endfor

;  Powder average the quantity SD
   integrand = dblarr(nx,ntheta,nphi,/nozero)
   for j = 0,nphi-1 do begin
      sin_term = ue#(sin(theta)) ; nx x ntheta
      integrand[*,*,j] = sin_term*sd[*,*,j]
   endfor
   pow_ave[0,i] = (total(total(temporary(integrand),2),2)*dtheta*dphi)/(4.0*!pi)

endfor
yfit = (ue#(temporary(int)))*(eisf*temporary(res_func)+(1.0-eisf)*temporary(pow_ave))
;print,'Elapsed time: ',systime(/seconds)-tstart
end

; *************************************************** ;