; +
; NAME:
;
;  HC_GUI
;
; PURPOSE:
;
;  User interface that wraps the functionality of the
;  class HC_DATA__DEFINE enables analysis of data
;  using the honeycomb model function.
;
; AUTHOR:
;
;  Robert Dimeo
;  National Institute of Standards and Technology
;  Center for Neutron Research
;  100 Bureau Drive, Mail Stop 8562
;  Gaithersburg, MD 20899
;  Tel: (301) 975-8135
;  Email: robert.dimeo@nist.gov
;
; CATEGORY:
;
;  Data analysis
;
; REQUIREMENTS
;
;  IDL 6.0 or higher
;
; ADDITIONAL REQUIRED PROGRAMS
;
;  HC_DATA__DEFINE and all programs required therein
;  SOURCEROOT.PRO
;  CENTERTLB.PRO
;  RAINS_DISPLAY_CORRELATION.PRO and all programs required therein
;
; MODIFICATION HISTORY:
;
;  Writing began -- 7/06/04 (RMD)
; -
; *************************************************** ;
pro hc_gui_cleanup,tlb
widget_control,tlb,get_uvalue = pstate
dir = (*pstate).dir
;save,filename = (*pstate).info_file,dir
wdelete,(*pstate).winpix
heap_free,pstate
help,/heap,/brief
end
; *************************************************** ;
pro hc_gui_about,event

text = [ 'This application was written by',  $
         'Rob Dimeo','',                     $
         'NIST Center for Neutron Research', $
         'July 7, 2004',                     $
         '***************',                  $
         'Modified (RMD): September 27, 2004']

void = dialog_message(dialog_parent = event.top,   $
   /information,text)
end
; *************************************************** ;
pro hc_gui_plot,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
slider_id = widget_info(event.top,find_by_uname = 'SLIDER')
widget_control,slider_id,get_value = index
index = index[0] - 1

ret = o->get_property(qty = qty,dqty = dqty, $
   xvals = xvals,yvals = yvals)

z = qty[*,index]
dz = dqty[*,index]

if (*pstate).autoscale then begin
   xlo = min(xvals,max = xhi)
   zlo = min(z,max = zhi)
   deltax = xhi-xlo
   deltaz = zhi+max(dz)
   xrange = [xlo-0.1*deltax,xhi+0.1*deltax]
   zrange = [zlo-0.1*deltaz,zhi+0.1*deltaz]
   (*pstate).xrange = xrange
   (*pstate).yrange = zrange
endif
ret = o->draw(index,xrange = (*pstate).xrange,/xsty,  $
   yrange = (*pstate).yrange,/ysty)
*(*pstate).xptr = !x
*(*pstate).yptr = !y
end
; *************************************************** ;
pro hc_gui_refresh_win,event
widget_control,event.top,get_uvalue = pstate
wset,(*pstate).winpix
hc_gui_plot,event
wset,(*pstate).winvis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pstate).winpix]
end
; *************************************************** ;
pro hc_gui_load_data,event
title = 'Please select a DAVE data file'
widget_control,event.top,get_uvalue = pstate
filename = dialog_pickfile(dialog_parent = event.top, $
   path = (*pstate).dir,get_path = out_path,          $
   filter = '*.dave',title = title)

if filename ne '' then begin
   (*pstate).dir = out_path
   (*pstate).data_file = filename
   data_id = widget_info(event.top,find_by_uname = 'DATA_FILE')
   display_name = file_basename(filename,'.dave')
   widget_control,data_id,set_value = display_name
   dir_id = widget_info(event.top,find_by_uname = 'DIR')
   widget_control,dir_id,set_value = out_path
   slider_id = widget_info(event.top,find_by_uname = 'SLIDER')
   if obj_valid((*pstate).odata) then obj_destroy,(*pstate).odata
   odata = obj_new('hc_data',filename)
   if obj_valid(odata) then begin
      (*pstate).odata = odata
      ret = odata->get_property(ny = ny)
      widget_control,slider_id,set_slider_max = ny
      hc_gui_refresh_win,event
   endif
endif
end
; *************************************************** ;
pro hc_gui_load_res_parms,event
title = 'Please select a resolution function parameter file'
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then begin
   msg = 'You must load in a data file first'
   void = dialog_message(dialog_parent = event.top,msg)
   res_id = widget_info(event.top,find_by_uname = 'RES_FILE')
   (*pstate).res_file = ''
   widget_control,res_id,set_value = (*pstate).res_file
   return
endif
filename = dialog_pickfile(dialog_parent = event.top, $
   path = (*pstate).dir,get_path = out_path,          $
   filter = '*.fit',title = title)

if filename ne '' then begin
   (*pstate).dir = out_path
   res_id = widget_info(event.top,find_by_uname = 'RES_FILE')
   display_name = file_basename(filename,'.fit')
   widget_control,res_id,set_value = display_name
   dir_id = widget_info(event.top,find_by_uname = 'DIR')
   widget_control,dir_id,set_value = out_path
   if obj_valid((*pstate).odata) then begin
      o = (*pstate).odata
      ret = o->load_resolution_params(filename)
      ret = o->get_property(ny = ny)
      (*pstate).res_file = filename
   endif

   slider_id = widget_info(event.top,find_by_uname = 'SLIDER')
   widget_control,slider_id,set_slider_max = ny
   hc_gui_refresh_win,event
endif
end
; *************************************************** ;
pro hc_gui_sel_dir,event
; User has ability to change the working directory
widget_control,event.top,get_uvalue = pstate
dir = dialog_pickfile(dialog_parent = event.top,   $
   /directory,path = (*pstate).dir,                $
   title = 'Please select working directory')
if dir ne '' then (*pstate).dir = dir
dir_id = widget_info(event.top,find_by_uname = 'DIR')
widget_control,dir_id,set_value = (*pstate).dir
end
; *************************************************** ;
pro hc_gui_build_param_base,base,init_parms
display_names = ['Hop Rate (meV)','Jump Distance (A)',   $
   'EISF']
unames = ['HOP_RATE','JUMP_DISTANCE','EISF']
n = n_elements(display_names)
row_base = lonarr(n)
check_base = lonarr(n)
for i = 0,n-1 do begin
   row_base[i] = widget_base(base,/row)
   void = cw_field(row_base[i],value = init_parms[i], $
      title = display_names[i],uname = unames[i])
   check_base[i] = widget_base(row_base[i],/nonexclusive)
   uname = unames[i]+'_FIXED'
   fix_button = widget_button(check_base[i],value = 'Fixed',  $
      uname = uname)
   if i eq 1 then $
      widget_control,fix_button,set_button = 1
endfor
end
; *************************************************** ;
pro hc_gui_win_event,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
case event.type of
0: begin    ; button press
      (*pstate).mouse = event.press
      if (*pstate).mouse eq 4 then begin
         (*pstate).autoscale = 1
         !x = *(*pstate).xptr & !y = *(*pstate).yptr
         hc_gui_refresh_win,event
      endif
      if (*pstate).mouse eq 1 then begin
         (*pstate).xbox[0] = event.x
         (*pstate).ybox[0] = event.y
         !x = *(*pstate).xptr
         !y = *(*pstate).yptr
         hc_gui_refresh_win,event
         empty
         (*pstate).autoscale = 0B
      endif
   end
1: begin ; button release
    if (*pstate).mouse eq 1 then begin
      xll = (*pstate).xbox[0] < (*pstate).xbox[1]
      yll = (*pstate).ybox[0] < (*pstate).ybox[1]
      w = abs((*pstate).xbox[1] - (*pstate).xbox[0])
      h = abs((*pstate).ybox[1] - (*pstate).ybox[0])
      xur = xll + w
      yur = yll + h
      !x = *(*pstate).xptr & !y = *(*pstate).yptr
      ll = convert_coord(xll,yll,/device,/to_data)
      ur = convert_coord(xur,yur,/device,/to_data)
      (*pstate).xrange = [ll[0],ur[0]]
      (*pstate).yrange = [ll[1],ur[1]]
      hc_gui_plot,event
      (*pstate).mouse = 0B
    endif
    if (*pstate).mouse eq 4 then begin
      hc_gui_plot,event
      (*pstate).mouse = 0B
    endif
   end
2: begin ; mouse motion
      if (*pstate).mouse eq 1 then begin
         (*pstate).xbox[1] = event.x
         (*pstate).ybox[1] = event.y
         xc = [(*pstate).xbox[0],event.x,event.x,$
              (*pstate).xbox[0],$
              (*pstate).xbox[0]]
         yc = [(*pstate).ybox[0],(*pstate).ybox[0],$
              event.y,event.y,$
              (*pstate).ybox[0]]
         !x = *(*pstate).xptr & !y = *(*pstate).yptr
         wset,(*pstate).winvis
         device,copy = [0,0,!d.x_size,!d.y_size,0,0,(*pstate).winpix]
         plots,xc,yc,/device,thick = 2.0
         empty
      endif
   end
else:
endcase
end
; *************************************************** ;
function hc_gui_show_message, event,tlb,msg,       $
                              realize = realize,   $
                              destroy = destroy

; This function displays a simple message and can
; be created and destroyed through simple calls via
; keyword setting: "REALIZE" and "DESTROY".

if keyword_set(realize) then begin
   title = 'Information'
   tlb = widget_base(group_leader = event.top,  $
      /tlb_frame_attr,title = title)
   void = widget_label(tlb,value = msg,/dynamic_resize)
   centertlb,tlb
   widget_control,tlb,/realize
endif

if keyword_set(destroy) then widget_control,tlb,/destroy

end
; *************************************************** ;
pro hc_gui_evaluate_model,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
ret = (*pstate).odata->get_property(fit_parms = fit_parms)
msg = 'Evaluating function...please wait'

; Get the messenger on the screen
ret = hc_gui_show_message(event,tlb,msg,/realize)

unames = ['HOP_RATE','JUMP_DISTANCE','EISF']

eisf_id = widget_info(event.top,find_by_uname = 'EISF')
widget_control,eisf_id,get_value = eisf & eisf = eisf[0]
hop_rate_id = widget_info(event.top,find_by_uname = 'HOP_RATE')
widget_control,hop_rate_id,get_value = hop_rate & hop_rate = hop_rate[0]
jump_dist_id = widget_info(event.top,find_by_uname = 'JUMP_DISTANCE')
widget_control,jump_dist_id,get_value = jump_dist
jump_dist = jump_dist[0]

ntheta_id = widget_info(event.top,find_by_uname = 'NTHETA')
nphi_id = widget_info(event.top,find_by_uname = 'NPHI')
widget_control,ntheta_id,get_value = ntheta
ntheta = ntheta[0]
widget_control,nphi_id,get_value = nphi
nphi = nphi[0]

o = (*pstate).odata
ret = o->set_property(ntheta = ntheta,nphi = nphi)
if n_elements(fit_parms) ne 1 then begin
   ret = o->evaluate_model(hop_rate,jump_dist,eisf,   $
      /use_fit_params)
endif else begin
   ret = o->evaluate_model(hop_rate,jump_dist,eisf)
endelse
; Kill the messenger
ret = hc_gui_show_message(event,tlb,msg,/destroy)

hc_gui_refresh_win,event
end
; *************************************************** ;
pro hc_gui_eval_function,p,event
widget_control,event.top,get_uvalue = pstate
ret = (*pstate).odata->set_property(parms = p)

jump_dist_id = widget_info(event.top,find_by_uname = 'JUMP_DISTANCE')
eisf_id = widget_info(event.top,find_by_uname = 'EISF')
hop_rate_id = widget_info(event.top,find_by_uname = 'HOP_RATE')

; Update the text boxes with the values for the parameters
widget_control,jump_dist_id,set_value = strtrim(string(p[1]),2)
widget_control,hop_rate_id,set_value = strtrim(string(p[0]),2)
widget_control,eisf_id,set_value = strtrim(string(p[2]),2)

hc_gui_evaluate_model,event
end
; *************************************************** ;
pro hc_gui_fit,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
niter_id = widget_info(event.top,find_by_uname = 'NITER')
widget_control,niter_id,get_value = val & niter = fix(val[0])

; Guess the initial parameters
eisf_id = widget_info(event.top,find_by_uname = 'EISF')
widget_control,eisf_id,get_value = eisf & eisf = eisf[0]
hop_rate_id = widget_info(event.top,find_by_uname = 'HOP_RATE')
widget_control,hop_rate_id,get_value = hop_rate & hop_rate = hop_rate[0]
jump_dist_id = widget_info(event.top,find_by_uname = 'JUMP_DISTANCE')
widget_control,jump_dist_id,get_value = jump_dist
jump_dist = jump_dist[0]

ntheta_id = widget_info(event.top,find_by_uname = 'NTHETA')
nphi_id = widget_info(event.top,find_by_uname = 'NPHI')
widget_control,ntheta_id,get_value = ntheta
ntheta = ntheta[0]
widget_control,nphi_id,get_value = nphi
nphi = nphi[0]
ret = o->set_property(ntheta = ntheta,nphi = nphi,niter = niter)
ret = o->get_property(qty = qty,ny = ny)
ret = o->get_property(fit_parms = these_parms)

; Get the initial guesses for the parameters
int = max(qty,dimension = 1,index)
cen = replicate(0d,ny)
if n_elements(these_parms) eq 1 then begin
   parms = [hop_rate,jump_dist,eisf,int,cen]
endif else begin
   parms = these_parms
   parms[0] = hop_rate
   parms[1] = jump_dist
   parms[2] = eisf
endelse

; Determine which parameters are to be held fixed
nparms = n_elements(parms)
fixed_array = replicate(0,nparms)
unames = ['HOP_RATE','JUMP_DISTANCE','EISF']
eisf_check_id = widget_info(event.top,find_by_uname = 'EISF_FIXED')
jump_rate_check_id = widget_info(event.top,find_by_uname = 'HOP_RATE_FIXED')
jump_dist_check_id = widget_info(event.top,find_by_uname = 'JUMP_DISTANCE_FIXED')
eisf_checked = widget_info(eisf_check_id,/button_set)
rate_checked = widget_info(jump_rate_check_id,/button_set)
dist_checked = widget_info(jump_dist_check_id,/button_set)

fixed_array[0:2] = [rate_checked,dist_checked,eisf_checked]

ret = o->set_property(parms = parms)

; Now set up the display that shows the current chi-squared
; and allows pausing.
stop_base = widget_base(group_leader = event.top,  $
   title = 'Honeycomb Model Analysis',/col)
centertlb,stop_base
if !d.name eq 'WIN' then begin
   thisFont = "Comic Sans MS*Bold*20"
   stop_but = widget_button(stop_base,value = 'Interrupt fitting', $
                              xsize = 150,ysize = 50,font = thisFont)
   void = widget_label(stop_base,value = 'Iteration',font = thisFont)
   txt_field = widget_text(stop_base,value = '',xsize = 5,font = thisFont)
   void = widget_label(stop_base,value = 'chi-squared',font = thisFont)
   chisq_field = widget_text(stop_base,value = '',font = thisFont)
endif else begin
   stop_but = widget_button(stop_base,value = 'Interrupt fitting', $
                              xsize = 150,ysize = 50)
   void = widget_label(stop_base,value = 'Fit iteration')
   txt_field = widget_text(stop_base,value = '',xsize = 5)
   void = widget_label(stop_base,value = 'chi-squared')
   chisq_field = widget_text(stop_base,value = '')
endelse
widget_control,stop_base,/realize

iterargs =  {  stopbut:stop_but, txt_field:txt_field,             $
               chisq_field:chisq_field,winpix:(*pstate).winpix,   $
               winvis:(*pstate).winvis,event:event                }

ret = o->fit(  msg = msg,                    $
               fixed_array = fixed_array,    $
               iterargs = iterargs           )

if widget_info(stop_base,/valid_id) then widget_control,stop_base,/destroy

if ret eq 0 then begin
   void = dialog_message(dialog_parent = event.top,msg)
   return
endif
ret = o->get_property(fit_parms = parms)
ret = o->show_parms()

; Update the text boxes with the values for the parameters
widget_control,jump_dist_id,set_value = strtrim(string(parms[1]),2)
widget_control,hop_rate_id,set_value = strtrim(string(parms[0]),2)
widget_control,eisf_id,set_value = strtrim(string(parms[2]),2)

hc_gui_refresh_win,event
end
; *************************************************** ;
pro hc_gui_load_fit_params,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
dir = (*pstate).dir
title = 'Please select a fit file '
filename =  dialog_pickfile(dialog_parent = event.top,   $
            /read,title = title, filter = '*.hcfit',     $
            path = dir                                   )
if file_test(filename) then begin
   msg = 'Loading in parameters and calculating function'
   ret = hc_gui_show_message(event,tlb,msg,/realize)
   restore,filename
   ret = o->get_property(ntheta = ntheta,nphi = nphi,niter = niter)
   ret = o->set_property(parms = parms)
   ret = o->evaluate_model(parms[0],parms[1],parms[2],   $
      /use_fit_params)
   ret = hc_gui_show_message(event,tlb,msg,/destroy)
   ; Update the GUI with the values just loaded in...
   hop_rate_id = widget_info(event.top,find_by_uname = 'HOP_RATE')
   jump_dist_id = widget_info(event.top,find_by_uname = 'JUMP_DISTANCE')
   eisf_id = widget_info(event.top,find_by_uname = 'EISF')
   ntheta_id = widget_info(event.top,find_by_uname = 'NTHETA')
   nphi_id = widget_info(event.top,find_by_uname = 'NPHI')
   niter_id = widget_info(event.top,find_by_uname = 'NITER')
   widget_control,niter_id,set_value = strtrim(string(niter),2)
   widget_control,ntheta_id,set_value = strtrim(string(ntheta),2)
   widget_control,nphi_id,set_value = strtrim(string(nphi),2)
   widget_control,hop_rate_id,set_value = strtrim(string(parms[0]),2)
   widget_control,jump_dist_id,set_value = strtrim(string(parms[1]),2)
   widget_control,eisf_id,set_value = strtrim(string(parms[2]),2)
   hc_gui_refresh_win,event
endif
end
; *************************************************** ;
pro hc_gui_save_fit_params,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
ret = o->get_property(fit_parms = parms)
nparms = n_elements(parms)
if nparms eq 1 then return
dir = (*pstate).dir
title = 'Please type the name of the fit file ' +  $
   '(no extension please)'
filename =  dialog_pickfile(dialog_parent = event.top,   $
            /write,title = title, filter = '*.hcfit',    $
            path = dir                                   )
if filename eq '' then return
filename = filename + '.hcfit'
save,filename = filename,parms
end
; *************************************************** ;
pro hc_gui_save_fits_as_text,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
ret = o->get_property(yfit = yfit)
help,yfit
if n_elements(yfit) eq 1 then return
ret = o->get_property(nx = nx,ny = ny,xvals = xvals,     $
   yvals = yvals,dqty = dqty,qty = qty,res_fun = res_fun,$
   yfit = yfit                                           )
ncols = 5*ny
format_statement = '('+strtrim(string(ncols),2)+'f15.6)'
nrows = nx
; Create a data array with the required number of rows and columns
; Ultimately this will be the array that is written out to a file
for i = 0,ny-1 do begin

   dat = reform(qty[*,i])
   fit = reform(yfit[*,i])
   res = reform(res_fun[*,i])
   daterr = reform(dqty[*,i])
   x = reform(xvals)
   thisformat = '(f15.3)'
   num_string = strtrim(string(yvals[i],format = thisformat),2)

   if i eq 0 then begin
      d = [transpose(x),transpose(dat),transpose(daterr),transpose(res),transpose(fit)]
      titles = ['x('+num_string+')','y('+num_string+')','dy('+num_string+')','res('+num_string+')','fit']
   endif else begin
      d = [d,transpose(x),transpose(dat),transpose(daterr),transpose(res),transpose(fit)]
      titles = [titles,'x('+num_string+')','y('+num_string+')','dy('+num_string+')','res('+num_string+')','fit']
   endelse
endfor

dir = (*pstate).dir
title = 'Please type the name of the file (no extension please)'
filename =  dialog_pickfile(dialog_parent = event.top,   $
            /write,title = title, filter = '*.txt',      $
            path = dir                                   )
if filename eq '' then return
filename = filename + '.txt'
openw,lun,filename,/get_lun
text_format = '('+strtrim(string(ncols),2)+'A15)'
printf,lun,titles,format = text_format

for i = 0,nrows-1 do printf,lun,d[*,i],format = format_statement
free_lun,lun,/force

end
; *************************************************** ;
pro hc_gui_view_corr_matrix,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return
o = (*pstate).odata
ret = o->get_property(cor = cor)
if n_elements(cor) le 1 then return

thisEvent = tag_names(event,/structure_name)
case thisEvent of
'WIDGET_BUTTON':  $
   begin
      ret = o->get_property(cor = cor,parmnames = parmnames,   $
               qty = qty)
      nparms = n_elements(parmnames)
      new_parmnames = strarr(nparms)
      for i = 0,nparms-1 do begin
         new_parmnames[i] = '#'+strtrim(string(i),2)+': '+parmnames[i]
      endfor
      bad_index = where(finite(cor) ne 1, count_bad)
      if count_bad gt 0 then cor[bad_index] = -1.e6

      rains_display_correlation, cor,                       $
                                 new_parmnames,             $
                                 group_leader = event.top,  $
                                 notify_ids = [event.id,event.top]
   end
else:
endcase
end
; *************************************************** ;
pro hc_gui_write_fit_parms_as_ascii,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return

o = (*pstate).odata
strout = o->show_fit_results()
if n_elements(strout) eq 1 then begin
   msg = 'Either no fits or no errors are present'
   void = dialog_message(dialog_parent = event.top,   $
      msg)
   return
endif

dir = (*pstate).dir
title = 'Please type the name of the file (no extension please)'
filename =  dialog_pickfile(dialog_parent = event.top,   $
            /write,title = title, filter = '*.txt',      $
            path = dir                                   )
if filename eq '' then return
filename = filename + '.txt'
data_file = (*pstate).data_file
res_file = (*pstate).res_file

openw,lun,filename,/get_lun
n = n_elements(strout)
printf,lun,'Data file: '+data_file
printf,lun,'Resolution file: '+res_file
for i = 0,n-1 do printf,lun,strout[i]
free_lun,lun,/force


end
; *************************************************** ;
pro hc_gui_show_results,event
widget_control,event.top,get_uvalue = pstate
if ~obj_valid((*pstate).odata) then return

o = (*pstate).odata
strout = o->show_fit_results()
if n_elements(strout) eq 1 then begin
   msg = 'Either no fits or no errors are present'
   void = dialog_message(dialog_parent = event.top,   $
      msg)
   return
endif
title = 'Fit results'
void = dialog_message(dialog_parent = event.top, strout, $
   /information,title = title)
end
; *************************************************** ;
pro hc_gui_event,event
uname = widget_info(event.id,/uname)

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'hc_gui_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

case uname of
'QUIT':              $
   widget_control,event.top,/destroy
'LOAD_DATA':         $
   hc_gui_load_data,event
'LOAD_RES_PARMS':    $
   hc_gui_load_res_parms,event
'SEL_DIR':           $
   hc_gui_sel_dir,event
'ABOUT':             $
   hc_gui_about,event
'SLIDER':            $
   hc_gui_refresh_win,event
'ZOOM_EVENTS':       $
   hc_gui_win_event,event
'EVALUATE_MODEL':    $
   hc_gui_evaluate_model,event
'FIT':               $
   hc_gui_fit,event
'LOAD_FIT_PARAMS':   $
   hc_gui_load_fit_params,event
'SAVE_FIT_PARAMS':   $
   hc_gui_save_fit_params,event
'SAVE_FITS_AS_TEXT': $
   hc_gui_save_fits_as_text,event
'VIEW_CORR_MATRIX':  $
   hc_gui_view_corr_matrix,event
'SHOW_FIT_RESULTS':  $
   hc_gui_show_results,event
'WRITE_FIT_RESULTS': $
   hc_gui_write_fit_parms_as_ascii,event
else:
endcase

end
; *************************************************** ;
pro hc_gui, group_leader = group_leader, workDir=dir, dataDir=dataDir, _EXTRA=etc
; Widget definition module
; Turn color decomposition off and load the B/W
; color table.
device,decomposed = 0
loadct,0,/silent
; Search for the hc_gui information file called
; hc_info.sav.  This contains the identity of the
; working directory and allows the user to only
; need to set the working directory once upon
; first use.
;info_file = filepath('hc_info.sav',root = sourceroot())
;file_exists = file_test(info_file)
;if file_exists then begin
;   restore,info_file
;endif else begin
;   cd,current = dir
;   save,filename = info_file,dir
;endelse

; Lay out the user-interface
base_title = 'Honeycomb Model Data Analysis Application'
if n_elements(group_leader) eq 0 then group_leader = 0L
tlb = widget_base(/col,tlb_frame_attr = 1,mbar = bar, $
   grid_layout = 0,title = base_title, group_leader=group_leader)
; Build the file menu items at the top of the UI
file_menu = widget_button(bar,value = 'FILE',/menu)
void = widget_button(file_menu,value = 'Select working directory', $
   uname = 'SEL_DIR')
void = widget_button(file_menu,value = 'Load data file', $
   uname = 'LOAD_DATA',/separator)
void = widget_button(file_menu,value = 'Load resolution parameters', $
   uname = 'LOAD_RES_PARMS')
void = widget_button(file_menu,value = 'QUIT',  $
   uname = 'QUIT',/separator)
fit_menu = widget_button(bar,value = 'FITTING',/menu)
void = widget_button(fit_menu,value = 'Load fit parameters',      $
   uname = 'LOAD_FIT_PARAMS')
void = widget_button(fit_menu,value = 'Save current fit parameters',      $
   uname = 'SAVE_FIT_PARAMS')
void = widget_button(fit_menu,value = 'View correlation matrix',      $
   uname = 'VIEW_CORR_MATRIX')
void = widget_button(fit_menu,value = 'Save data and fits as text',      $
   uname = 'SAVE_FITS_AS_TEXT',/separator)
void = widget_button(fit_menu,value = 'Save fit parameters as text',      $
   uname = 'WRITE_FIT_RESULTS')
misc_menu = widget_button(bar,value = 'MISC',/menu)
void = widget_button(misc_menu,value = 'About this application',  $
   uname = 'ABOUT')
field_size = 30
void = widget_label(tlb,value = dir,/dynamic_resize, $
   uname = 'DIR')

row_base = widget_base(tlb,/row)
ctrl_base = widget_base(row_base,/col,/base_align_right)
void = cw_field(ctrl_base,title = 'Resolution Parameter File', $
   value = '',/noedit,uname = 'RES_FILE',/col,  $
   xsize = field_size)
void = cw_field(ctrl_base,title = 'Data File', $
   value = '',/noedit,uname = 'DATA_FILE',/col, $
   xsize = field_size)

param_base = widget_base(ctrl_base,/col,/frame,/base_align_right)

init_parms = [0.0199,2.1465,0.12]
hc_gui_build_param_base,param_base,init_parms

bottom_base = widget_base(ctrl_base,/row,/base_align_center)
col1 = widget_base(bottom_base,/col,/base_align_right)
void = cw_field(col1,title = 'NTHETA',value = '5',  $
   uname = 'NTHETA',xsize = 3)
void = cw_field(col1,title = 'NPHI',value = '5',    $
   uname = 'NPHI',xsize = 3)
void = cw_field(col1,title = '# Iterations',value = '3',  $
   uname = 'NITER',xsize = 3)
col2 = widget_base(bottom_base,/col,/base_align_right)
void = widget_button(col2,value = 'Fit Model to Data',   $
   uname = 'FIT')
void = widget_button(col2,value = 'Show fit results',      $
   uname = 'SHOW_FIT_RESULTS',/separator)
void = widget_button(col2,value = 'Evaluate Model',   $
   uname = 'EVALUATE_MODEL')
void = widget_slider(col2,value = 1,min = 1,max = 2,   $
   title = 'Group to display',uname = 'SLIDER')

; Draw the plot window
xsize = 500 & ysize = xsize
win = widget_draw(row_base,xsize = xsize,ysize = ysize,  $
   /motion_events,/button_events,uname = 'ZOOM_EVENTS')

centertlb,tlb
widget_control,tlb,/realize
widget_control,win,get_value = winvis
window,/free,/pixmap,xsize = xsize,ysize = ysize
winpix = !d.window
odata = obj_new()
xptr = ptr_new(/allocate_heap)
yptr = ptr_new(/allocate_heap)
; Stuff the useful information into a state structure
state =  {  odata:odata,            $
            dir:dir,                $
;            info_file:info_file,    $
            winvis:winvis,          $
            winpix:winpix,          $
            autoscale:1B,           $
            xbox:[0,1],             $
            ybox:[0,1],             $
            xrange:[0.,1.],         $
            yrange:[0.,1.],         $
            mouse:0B,               $
            xptr:xptr,              $
            yptr:yptr,              $

            res_file:'',            $
            data_file:''            }

pstate = ptr_new(state,/no_copy)
widget_control,tlb,set_uvalue = pstate

reg_name = 'hc_gui'
xmanager,reg_name,tlb,cleanup = 'hc_gui_cleanup',  $
   event_handler = 'hc_gui_event';,/no_block
end
; *************************************************** ;