; +
; NAME:
;
;  HC_DATA__DEFINE
;
; PURPOSE:
;
;  Object class for the Zirconium Beryllide data collected by
;  Cappelletti, Udovic, and Chowdhuri.
;
; AUTHOR:
;
;  Robert Dimeo
;  National Institute of Standards and Technology
;  Center for Neutron Research
;  100 Bureau Drive, Mail Stop 8562
;  Gaithersburg, MD 20899
;  Tel: (301) 975-8135
;  Email: robert.dimeo@nist.gov
;
; CATEGORY:
;
;  Data analysis
;
; REQUIREMENTS
;
;  IDL 6.0 or higher
;
; ADDITIONAL REQUIRED PROGRAMS
;
;  MPFIT.PRO
;  MPCURVEFIT.PRO
;  GET_DAVE_PTR_CONTENTS.PRO
;  HC_MODEL.PRO
;  HC_LORENTZIAN.PRO
;
; MODIFICATION HISTORY:
;
;  Writing began -- 6/14/04 (RMD)
;  Fixed bug with loading resolution function for data fit in PAN
;  where the energy transfer is in ueV -- 10/04/04
; -
; *************************************************** ;
pro hc_data::cleanup
ptr_free,self.pqty,self.pdqty,self.px,self.py
ptr_free,self.pfit,self.pfitparms,self.pdfitparms
ptr_free,self.pcor_ptr,self.res_func_ptr
ptr_free,self.parmnames_ptr
heap_free,self.res_param_ptr
end
; *************************************************** ;
function hc_data::update_parmnames
ny = n_elements(*self.py)
q = *self.py
parmnames = ['Hop rate','Hop distance','EISF']
int_names = 'Intensity ('+strtrim(string(q),2)+')'
;int_names = 'Intensity '+strtrim(string(1+indgen(ny)),2)
cen_names = 'Center ('+strtrim(string(q),2)+')'
;cen_names = 'Center '+strtrim(string(1+indgen(ny)),2)
parmnames = [parmnames,int_names,cen_names]
*self.parmnames_ptr = parmnames
return,1
end
; *************************************************** ;
function hc_data::read_data
; This function reads in DAVE data files
restore,self.filename
ret = get_dave_ptr_contents(daveptr,   qty = qty,        $
                                       err = err,        $
                                       xvals = xvals,    $
                                       xtype = xtype,    $
                                       xunits = xunits,  $
                                       yvals = yvals     )
if strupcase(xunits) eq 'ENERGY:UEV' then $
   xvals = 1.e-3*xvals  ; convert to meV if necessary
self.xunits = xunits
if strupcase(xtype) eq 'HISTOGRAM' then begin
   ; Convert to point data
   nx = n_elements(xvals)
   x = 0.5*(xvals[0:nx-2]+xvals[1:nx-1])
   xvals = x
endif
*self.pqty = qty
*self.pdqty = err
*self.px = xvals
*self.py = yvals
heap_free,daveptr
ret = self->update_parmnames()
return,1
end
; *************************************************** ;
function hc_data::set_property,  ntheta = ntheta,  $
                                 nphi = nphi,      $
                                 parms = parms,    $
                                 niter = niter

if n_elements(niter) ne 0 then self.niter = niter
if n_elements(parms) ne 0 then *self.pfitparms = parms
if n_elements(ntheta) ne 0 then self.ntheta = ntheta
if n_elements(nphi) ne 0 then self.nphi = nphi
return,1
end
; *************************************************** ;
function hc_data::show_parms
if n_elements(*self.pfitparms) eq 0 then return,0
if n_elements(*self.pdfitparms) eq 0 then return,0
nx = n_elements(*self.px)
ny = n_elements(*self.py)
parmnames = *self.parmnames_ptr
parms = *self.pfitparms
dparms = *self.pdfitparms
for i = 0,n_elements(parmnames)-1 do $
   print,parmnames[i]+'='+strtrim(string(parms[i]),2)+'+/-' + $
   strtrim(string(dparms[i]),2)
ret = self->get_property(chisq = chisq)
print,'Chi-squared: ',chisq
return,1
end
; *************************************************** ;
function hc_data::get_property,  ny = ny,                   $
                                 nx = nx,                   $
                                 qty = qty,                 $
                                 dqty = dqty,               $
                                 res_str = res_str,         $
                                 xvals = xvals,             $
                                 yvals = yvals,             $
                                 fit_parms = fit_parms,     $
                                 dfit_parms = dfit_parms,   $
                                 cor = cor,                 $
                                 chisq = chisq,             $
                                 res_fun = res_fun,         $
                                 parmnames = parmnames,     $
                                 yfit = yfit,               $
                                 ntheta = ntheta,           $
                                 nphi = nphi,               $
                                 niter = niter

if n_elements(*self.pqty) eq 0 then return,0
dsize = size(*self.pqty)
ny = n_elements(*self.py)
nx = n_elements(*self.px)
ntheta = self.ntheta
nphi = self.nphi
niter = self.niter
parmnames = ['Hop rate','Hop distance','EISF']
int_names = 'Intensity '+strtrim(string(1+indgen(ny)),2)
cen_names = 'Center '+strtrim(string(1+indgen(ny)),2)
parmnames = [parmnames,int_names,cen_names]
if arg_present(cor) then begin
   if n_elements(*self.pcor_ptr) ne 0 then begin
      cor = *self.pcor_ptr
   endif else begin
      cor = 0
   endelse
endif
if arg_present(fit_parms) then begin
   if n_elements(*self.pfitparms) ne 0 then begin
      fit_parms = *self.pfitparms
   endif else begin
      fit_parms = 0
   endelse
endif
if arg_present(dfit_parms) then begin
   if n_elements(*self.pdfitparms) ne 0 then begin
      dfit_parms = *self.pdfitparms
   endif else begin
      dfit_parms = 0
   endelse
endif
if arg_present(res_str) then begin
   if n_elements(*self.res_param_ptr) ne 0 then begin
      res_str = *self.res_param_ptr
   endif else begin
      res_str = 0
   endelse
endif
if arg_present(res_fun) then begin
   if n_elements(*self.res_func_ptr) ne 0 then begin
      res_fun = *self.res_func_ptr
   endif else begin
      res_fun = 0
   endelse
endif
if arg_present(yfit) then begin
   if n_elements(*self.pfit) ne 0 then begin
      yfit = *self.pfit
   endif else begin
      yfit = 0
   endelse
endif
if arg_present(yvals) then begin
   if n_elements(*self.py) ne 0 then begin
      yvals = *self.py
   endif else begin
      yvals = 0
   endelse
endif
if arg_present(xvals) then begin
   if n_elements(*self.px) ne 0 then begin
      xvals = *self.px
   endif else begin
      xvals = 0
   endelse
endif
if arg_present(qty) then begin
   if n_elements(*self.pqty) ne 0 then begin
      qty = *self.pqty
   endif else begin
      qty = 0
   endelse
endif
if arg_present(dqty) then begin
   if n_elements(*self.pdqty) ne 0 then begin
      dqty = *self.pdqty
   endif else begin
      dqty = 0
   endelse
endif
if arg_present(chisq) then begin
   chisq = self.chisq
endif
return,1
end
; *************************************************** ;
pro hc_fit_params__define
define = {hc_fit_params,                     $
                        ncurves:0L,          $
                        group:0L,            $
                        parea:ptr_new(),     $
                        pcenter:ptr_new(),   $
                        pfwhm:ptr_new()      $
         }
end
; *************************************************** ;
function hc_data::load_resolution_params,filename,msg = msg
; This reads in the fit parameter output from PAN and
; stores the results in a data structure
msg = 'Resolution parameters loaded successfully'
if n_params() eq 0 then filename = ''
if ~file_test(filename) then begin
   msg = 'Resolution parameters file not found'
   return,0
endif

if strupcase(self.xunits) eq 'ENERGY:UEV' then  $
   efact = 1.e-3 else   $
   efact = 1.0

counter = 0L & dum = ''

openr,lun,filename,/get_lun
while not eof(lun) do begin
   readf,lun,dum
   if counter eq 0L then output = dum else $
      output = [output,dum]
   counter++
endwhile
free_lun,lun,/force

; Ok, now that the fit parameter file has been read in
; successfully parse out the information into a structure.
area_counter = 0L
center_counter = 0L
fwhm_counter = 0L
group_counter = 0L

for i = 0,counter-1 do begin
; Figure out which group this is
   group_pos = strpos(output[i],'Group:')
   if group_pos[0] ne -1 then begin
      this_string = strsplit(output[i],/extract)
      this_group = fix(this_string[1])
      cur_group = this_group
      if group_counter eq 0L then group = this_group else $
         group = [group,this_group]
      group_counter++
   endif
endfor

ngroups = n_elements(group)
group_hi = group[ngroups-1]   ; This is the highest group read in
param_str_arr = replicate({hc_fit_params},ngroups)
group_counter = 0L
curve_counter = 0
curve_array = intarr(group_hi)

for i = 0,counter-1 do begin
   c1_pos = strpos(output[i],'Curve 1:')
   cn_pos = strpos(output[i],'Curve')
   if c1_pos[0] ne -1 then begin
      curve_counter = 0
      curve_array[group[group_counter]-1] = curve_counter+1
   endif
   if (cn_pos[0] ne -1) and (c1_pos[0] eq -1) then begin
      curve_counter++
      curve_array[group[group_counter]-1] = curve_counter+1
   endif

   group_pos = strpos(output[i],'Group:')
   if (group_pos[0] ne -1) and (i gt 1) then begin
      group_counter++
   endif

   cur_group = group[group_counter]
   param_str_arr(group_counter).group = cur_group
   param_str_arr(group_counter).ncurves = curve_counter+1

; Load in the areas
   area_pos = strpos(output[i],'area:')
   if area_pos[0] ne -1 then begin
      this_string = strsplit(output[i],/extract)
      this_area = float(this_string[2])
      if area_counter eq 0 then area = this_area else $
         area = [area,this_area]
      area_counter++
   endif
; Load in the centers
   center_pos = strpos(output[i],'center:')
   if center_pos[0] ne -1 then begin
      this_string = strsplit(output[i],/extract)
      this_center = float(this_string[2])
      if center_counter eq 0 then center = efact*this_center else $
         center = [center,efact*this_center]
      center_counter++
   endif
; Load in the fwhms
   fwhm_pos = strpos(output[i],'FWHM:')
   if fwhm_pos[0] ne -1 then begin
      this_string = strsplit(output[i],/extract)
      this_fwhm = float(this_string[2])
      if fwhm_counter eq 0 then fwhm = efact*this_fwhm else $
         fwhm = [fwhm,efact*this_fwhm]
      fwhm_counter++
   endif
endfor

index = 0
for i = 0,ngroups-1 do begin
   ncurves = param_str_arr(i).ncurves
   ; Normalize the integrated intensity of the Lorentzians
;   total_area = total(area[index:index+ncurves-1])
;  Shortened the computation of the individual integrated intensity
;  fractions by one line (removing the total_area variable computation
   this_area = (area[index:index+ncurves-1])/(total(area[index:index+ncurves-1]))
   param_str_arr(i).parea = ptr_new(this_area)
   param_str_arr(i).pcenter = ptr_new(center[index:index+ncurves-1])
   param_str_arr(i).pfwhm = ptr_new(fwhm[index:index+ncurves-1])
   index = index+ncurves
endfor

; Must normalize the resolution function

*self.res_param_ptr = param_str_arr

; Now resize the qty values so that they are consistent with
; the available groups in the resolution function.

qty = *self.pqty
dqty = *self.pdqty
q = *self.py
res_str = param_str_arr
ngroups = n_elements(res_str)
groups = intarr(ngroups)
for i = 0,ngroups-1 do begin
   groups[i] = res_str(i).group
endfor
q = q[groups-1]
ny = n_elements(q)
*self.pqty = qty[*,groups-1]
*self.pdqty = dqty[*,groups-1]
*self.py = q

; Calculate the resolution function on this energy
; grid for plotting later.

x = *self.px
y = *self.py
nx = n_elements(x)
ny = n_elements(y)
res_func = dblarr(nx,ny)
for i = 0,ny-1 do begin
   ncurves = res_str(i).ncurves
   area = *res_str(i).parea
   center = *res_str(i).pcenter
   fwhm = *res_str(i).pfwhm
   for j = 0,ncurves-1 do begin
      if j eq 0 then begin
         res_func[*,i] =   hc_lorentzian(x,area[j],center[j],fwhm[j])
      endif else begin
         res_func[*,i] =   res_func[*,i]+ $
                           hc_lorentzian(x,area[j],center[j],fwhm[j])
      endelse
   endfor
endfor
*self.res_func_ptr = res_func
ret = self->update_parmnames()
return,1
end
; *************************************************** ;
function hc_data::draw_3d,index,_Extra = extra
if n_elements(*self.px) eq 0 then return,0
if n_elements(*self.pfit) eq 0 then return,0
xsize = 500 & ysize = 600
window,0,xsize = xsize,ysize = ysize
x = *self.px & nx = n_elements(x)
y = *self.py & ny = n_elements(y)
z = *self.pqty
dz = *self.pdqty
yfit = *self.pfit

uy = 1+bytarr(ny)
ux = 1+bytarr(nx)
xdisp = x#uy
ydisp = ux#y

xmin = min(x,max = xmax)
delx = xmax - xmin
ymin = min(y,max = ymax)
dely = ymax - ymin
zmin = min(z,max = zmax)
delz = zmax - zmin

xr = [xmin-0.1*delx,xmax+0.1*delx]
yr = [ymin-0.1*dely,ymax+0.1*dely]
zr = [zmin-0.1*delz,zmax+0.1*delz]
help,xdisp
help,ydisp
help,z
surface,z,xdisp,ydisp,        $
         charsize = 1.0,xrange = xr,   $
         yrange = yr, zrange = zr
;plots,xdisp,ydisp,z,psym = 4

return,1
end
; *************************************************** ;
function hc_data::draw,index,_Extra = extra,ps = ps
if n_params() eq 0 then index = 0
if n_elements(*self.pqty) eq 0 then return,0
xsize = 500 & ysize = 700

res_pos = [0.15,0.1,0.9,0.25]

data_pos = [0.15,0.3,0.9,0.9]


x = *self.px
ny = n_elements(*self.py)
if index lt 0 then index = 0
if index ge ny then index = ny-1
q = (*self.py)[index]
z = (*self.pqty)[*,index]
dz = (*self.pdqty)[*,index]

if n_elements((*self.res_func_ptr)) ne 0 then begin
   if n_elements((*self.res_func_ptr)[*,index]) ne 0 then begin
      res_fun = (*self.res_func_ptr)[*,index]
      sf = max(z)/max(res_fun)
      res_fun = res_fun*sf
   endif
endif

this_format = '(f15.3)'
title = 'Q='+strtrim(string(q,format = this_format),2)+  $
   ' (!3!sA!r!u!9 %!3!n!E-1!N)'
;if ~keyword_set(ps) then $
;   window,0,xsize = xsize,ysize = ysize
plot,x,z,psym = 4,title = title,_Extra = extra, $
   ytitle = 'I (arb units)',xtitle = 'E (meV)';,pos = data_pos
errplot,x,z-dz,z+dz,width = 0.0
if n_elements(res_fun) ne 0 then $
   oplot,x,res_fun,psym = 0,linestyle = 2

;if n_elements(*self.pfit) eq 0 then return,1

;yfit = (*self.pfit)[*,index]
;res = (yfit-z)/dz
;
if n_elements(*self.pfit) ne 0 then begin
   yfit = (*self.pfit)[*,index]
   oplot,x,yfit,psym = 0,thick = 2.0
endif
;plot,x,res,psym = 0,pos = res_pos,/noerase,  $
;   ytitle = 'Residuals',xtitle = 'E (meV)',  $
;   _Extra = extra

return,1
end
; *************************************************** ;
function hc_data::ps_output,dir = dir
if n_elements(dir) eq 0 then $
   dir = 'c:\bhd\dave_development\dave\programs\testing\cappelletti\output\ps_50x50\'
if n_elements(*self.pfit) eq 0 then return,0
ny = n_elements(*self.py)
y = *self.py
this_device = !d.name

for i = 0,ny-1 do begin
   yformat = '(f15.0)'
   yvalue = strtrim(string(y[i]*1000.0,format = yformat),2)
   name = dir+'output'+yvalue+'ps'
   set_plot,'PS'
   device,filename = name, xsize = 8.0,ysize = 11.0, $
      xoff = 0.0,yoff = 0.0,/inches

   ret = self->draw(xrange = [-0.5,0.5],/xsty,/ps,i)
   device,/close_file
   set_plot,this_device
endfor

return,1
end
; *************************************************** ;
pro hc_iterproc,fnc,p,iter,fnorm,   $
      FUNCTARGS = functargs,        $
      QUIET = quiet,                $
      _Extra = extra
; Sends some useful information to the output log as the algorithm progresses

dof = extra.dof
string_format = '(f15.2)'
print,'**********************************'
print,'Iteration: '+strtrim(string(iter),2)+$
   ', Chi-squared='+strtrim(string(fnorm/dof,format = string_format),2)
end

; *************************************************** ;
pro hc_new_iterproc,fnc,p,iter,fnorm,              $
      stopBut = stopBut,txt_field = txt_field,     $
      chisq_field = chisq_field,                   $
      winpix = winpix,                             $
      winvis = winvis,                             $
      event = event,                               $
      FUNCTARGS = functargs,                       $
      PARINFO =  parinfo,                          $
      QUIET = quiet,                               $
      _Extra = extra

if n_elements(stopBut) eq 0 then return
if widget_info(stopBut,/valid_id) eq 0 then return
if n_elements(winpix) ne 0 then $
   hc_gui_eval_function,p,event
widget_control,txt_field,set_value = strtrim(string(iter),2)
widget_control,chisq_field,set_value = strtrim(string(fnorm/extra.dof),2)
event = widget_event(/nowait,stopBut)
evname = tag_names(event,/structure_name)
if evname eq '' or event.id eq 0 then return

if evname eq 'WIDGET_BUTTON' and event.select then begin
  event1 = widget_event(/nowait,stopBut)
  common mpfit_error,mperr
  mperr = -1
endif

return
end
; *************************************************** ;
function hc_data::show_fit_results
if n_elements(*self.pfitparms) eq 0 then return,''
parmnames = *self.parmnames_ptr
parms = *self.pfitparms
if n_elements(*self.pdfitparms) eq 0 then return,''
dparms = *self.pdfitparms
chisq = self.chisq
nparms = n_elements(parms)
for i = 0,nparms-1 do begin
   new = parmnames[i]+'= '+strtrim(string(parms[i]),2)+$
      '+/-'+strtrim(string(dparms[i]),2)
   if i eq 0 then strout = new else $
      strout = [strout,new]
endfor
chi_text = 'Chi-squared (reduced): '+  $
   strtrim(string(self.chisq),2)
strout = [strout,'',chi_text]
return,strout
end
; *************************************************** ;
function hc_data::fit,  msg = msg,                    $
                        nofit = nofit,                $
                        fixed_array = fixed_array,    $
                        iterargs = iterargs
; This method executes the fitting of the model as
; written in HC_MODEL.PRO
;
if ~keyword_set(nofit) then nofit = 0 else nofit = 1
msg = 'No errors in fitting'
; Error checking first
if n_elements(*self.pqty) eq 0 then begin
   msg = 'No data currently in memory'
   return,0
endif
if n_elements(*self.res_param_ptr) eq 0 then begin
   msg = 'No resolution function currently in memory'
   return,0
endif
ret = self->get_property(nx = nx,ny = ny,res_str = res_str,qty = qty,dqty = dqty)
x = *self.px & y = *self.py
ny = n_elements(y)
xvals = [x,y]

res_ptr = ptr_new(res_str)

; Establish some initial guesses
r = 0.0199 & l = 2.1465 & eisf = 0.12
int = max((*self.pqty),dimension = 1,index)
for i = 0,ny-1 do begin
   qty_max = max((*self.pqty)[*,i],index)
   if n_elements(index_array) eq 0 then index_array = index else $
      index_array = [index_array,index]
endfor
cen = replicate(0d,n_elements(index_array))
if n_elements(*self.pfitparms) ne 0 then begin
   parms = *self.pfitparms
;   print,'Parameters have been defined'
endif else begin
   parms = [r,l,eisf,int,cen]
;   print,'No parameters defined'
endelse

fit_function = 'HC_MODEL'
;; Call the fit function to see if there are any errors (debugging)
;call_procedure,   fit_function,              $
;                  xvals,                     $
;                  parms,                     $
;                  yfit,                      $
;                  nx = nx,                   $
;                  ny = ny,                   $
;                  ntheta = self.ntheta,      $
;                  nphi = self.nphi,          $
;                  res_ptr = res_ptr

weights = 1d/dqty^2
quiet = 1
if n_elements(fixed_array) eq 0 then            $
   fixed_array = replicate(0,n_elements(parms))

dof = n_elements(qty)-n_elements(parms)+total(fixed_array)
self.dof = dof
if ~nofit then begin

parinfo = replicate({fixed:0,limited:[0,0],limits:[0.D,0.D]},n_elements(parms))
nx = n_elements(x) & ny = n_elements(y)
if n_elements(fixed_array) eq 0 then begin
   fixed_array = replicate(0,n_elements(parms))
endif else begin
   parinfo(*).fixed = fixed_array
endelse
; Fix the centers
parinfo(3+ny:3+2*ny-1).fixed = 1
; Set the lower limit on the EISF to 0.0
parinfo(2).limited(0) = 1
parinfo(2).limits(0) = 0.D
;; Set the upper limit on the EISF to 1.0
;parinfo(2).limited(1) = 1
;parinfo(2).limits(1) = 1.D
if n_elements(iterargs) eq 0 then begin
iterproc = 'hc_iterproc'
yfit = mpcurvefit(   xvals,qty,weights,parms,sigma,   $
                     function_name = fit_function,    $
                     /autoderivative,                 $
                     chisq = chisq,                   $
                     status = status,                 $
                     quiet = quiet,                   $
                     parinfo = parinfo,               $
                     covar = covar,                   $
                     iterproc = iterproc,             $
                     functargs = {nx:nx,ny:ny,        $
                                 ntheta:self.ntheta,  $
                                 nphi:self.nphi,      $
                                 res_ptr:res_ptr},    $
                     itmax = self.niter               )
endif else begin
iterproc = 'hc_new_iterproc';'hc_iterproc'
yfit = mpcurvefit(   xvals,qty,weights,parms,sigma,   $
                     function_name = fit_function,    $
                     /autoderivative,                 $
                     chisq = chisq,                   $
                     status = status,                 $
                     quiet = quiet,                   $
                     parinfo = parinfo,               $
                     covar = covar,                   $
                     iterproc = iterproc,             $
                     iterargs = iterargs,             $
                     functargs = {nx:nx,ny:ny,        $
                                 ntheta:self.ntheta,  $
                                 nphi:self.nphi,      $
                                 res_ptr:res_ptr},    $
                     itmax = self.niter               )
endelse

   if status le 0 then begin
      ;print,'Status = '+strtrim(string(status),2)
      return,0
   endif
   print,'Status: ',status
   self.chisq = chisq/dof
endif else begin
   ;print,'Evaluating function'
   covar = fltarr(n_elements(parms),n_elements(parms))
   sigma = replicate(0.0,n_elements(parms))
endelse
ptr_free,res_ptr

; Compute the correlation matrix
if n_elements(covar) gt 0 then begin
   pcor = covar * 0
   FOR i = 0, n_elements(parms)-1 DO FOR j = 0, n_elements(parms)-1 DO $
      pcor[i,j] = covar[i,j]/sqrt(covar[i,i]*covar[j,j])
   *self.pcor_ptr = pcor
endif
*self.pfitparms = parms
*self.pdfitparms = sigma
*self.pfit = yfit

return,1
end
; *************************************************** ;
function hc_data::evaluate_model,r,l,eisf,   $
   use_fit_params = use_fit_params
; Guess the amplitudes
if n_elements(*self.pqty) eq 0 then return,0

qty = *self.pqty

ret = self->get_property(nx = nx,ny = ny, $
   res_str = res_str,xvals = xvals,yvals = yvals)
int = max((*self.pqty),dimension = 1,index)
for i = 0,ny-1 do begin
   qty_max = max((*self.pqty)[*,i],index)
   if n_elements(index_array) eq 0 then index_array = index else $
      index_array = [index_array,index]
endfor
cen = replicate(0d,n_elements(index_array))
if keyword_set(use_fit_params) then begin
   parms = *self.pfitparms
   parms[0:2] = [r,l,eisf]
endif else begin
   parms = [r,l,eisf,int,cen]
endelse
fit_function = 'HC_MODEL'
res_ptr = ptr_new(res_str)

; Call the fit function to see if there are any errors (debugging)
call_procedure,   fit_function,              $
                  [xvals,yvals],             $
                  parms,                     $
                  yfit,                      $
                  nx = nx,                   $
                  ny = ny,                   $
                  ntheta = self.ntheta,      $
                  nphi = self.nphi,          $
                  res_ptr = res_ptr
ptr_free,res_ptr
*self.pfit = yfit
return,1
end
; *************************************************** ;
function hc_data::init,filename,parms = parms
if n_params() eq 0 then filename = ''
if ~file_test(filename) then begin
   msg = 'File not found'
   return,0
endif
if n_elements(parms) ne 0 then self.pfitparms = ptr_new(parms) else $
   self.pfitparms = ptr_new(/allocate_heap)
self.filename = filename
self.ntheta = 100
self.nphi = 10
self.niter = 3
self.pqty = ptr_new(/allocate_heap)
self.pdqty = ptr_new(/allocate_heap)
self.px = ptr_new(/allocate_heap)
self.py = ptr_new(/allocate_heap)
self.pfit = ptr_new(/allocate_heap)
self.pdfitparms = ptr_new(/allocate_heap)
self.res_param_ptr = ptr_new(/allocate_heap)
self.pcor_ptr = ptr_new(/allocate_heap)
self.res_func_ptr = ptr_new(/allocate_heap)

self.parmnames_ptr = ptr_new(/allocate_heap)
self.chisq = 0.0
self.dof = 1
; Read the file in
ret = self->read_data()

return,1
end
; *************************************************** ;
pro hc_data__define
define = {hc_data,                           $
                  pqty:ptr_new(),            $
                  pdqty:ptr_new(),           $
                  px:ptr_new(),              $
                  py:ptr_new(),              $
                  pfit:ptr_new(),            $
                  pcor_ptr:ptr_new(),        $
                  res_param_ptr:ptr_new(),   $
                  res_func_ptr:ptr_new(),    $
                  parmnames_ptr:ptr_new(),   $
                  ntheta:0,                  $
                  chisq:0.0,                 $
                  dof:0,                     $
                  pfitparms:ptr_new(),       $
                  pdfitparms:ptr_new(),      $
                  nphi:0,                    $
                  niter:0,                   $
                  xunits:'',                 $
                  filename:''                }

end
; *************************************************** ;
