; $Id$
;###############################################################################
;
; NAME:
;  DAVE_E_NOTES__DEFINE
;
; PURPOSE:
;  Electronic notebook class definition
;
; CATEGORY:
;  DAVE, electronic notebook
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro dave_e_notes::cleanup
ptr_free,self.content_ptr
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;PRO mhh_strict, txtlines
;;
;; Replaces any occurrence of HTML reserved characters (<,>,&,") in the
;; given text lines with the appropriate HTML counterpart.
;;
;; entry:
;;       txtlines - String array containing the text line(s) to be altered.
;; exit:
;;  txtlines - Same as input except that reserved characters have been
;;                  replaced with the appropriate HTML syntax.
;;
; COMPILE_OPT hidden
;
; count = N_ELEMENTS(txtlines)
; FOR i=0,count-1 DO BEGIN
;  txt = txtlines[i]
;
;  ; Ampersands get replaced with &amp.  Must do ampersands first because
;  ; they are used to replace other reserved characters in HTML.
;  spos = STRPOS(txt,'&')
;  WHILE (spos NE -1) DO BEGIN
;   newtxt = STRMID(txt,0,spos)+'&amp;'+STRMID(txt,spos+1,STRLEN(txt)-spos+1)
;   txt = newtxt
;   spos = STRPOS(txt,'&',spos+1)
;  ENDWHILE
;  txtlines[i] = txt
;
;  ; Left angle brackets get replaced with &lt;
;  spos = STRPOS(txt,'<')
;  WHILE (spos NE -1) DO BEGIN
;   newtxt = STRMID(txt,0,spos)+'&lt;'+STRMID(txt,spos+1,STRLEN(txt)-spos+1)
;   txt = newtxt
;   spos = STRPOS(txt,'<',spos+1)
;  ENDWHILE
;  txtlines[i] = txt
;
;  ; Right angle brackets get replaced with &gt;
;  spos = STRPOS(txt,'>')
;  WHILE (spos NE -1) DO BEGIN
;   newtxt = STRMID(txt,0,spos)+'&gt;'+STRMID(txt,spos+1,STRLEN(txt)-spos+1)
;   txt = newtxt
;   spos = STRPOS(txt,'>',spos+1)
;  ENDWHILE
;  txtlines[i] = txt
;
;  ; Double quotes get replaced with &quot;
;  spos = STRPOS(txt,'"')
;  WHILE (spos NE -1) DO BEGIN
;   newtxt = STRMID(txt,0,spos)+'&quot;'+STRMID(txt,spos+1,STRLEN(txt)-spos+1)
;   txt = newtxt
;   spos = STRPOS(txt,'"',spos+1)
;  ENDWHILE
;  txtlines[i] = txt
; ENDFOR
;END
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::add_line,line,STRICT = strict
if keyword_set(strict) then mhh_strict, line
ncontents = n_elements(*self.content_ptr)
if ncontents eq 0 then begin
   *self.content_ptr = line
endif else begin
   *self.content_ptr = [*self.content_ptr,'<p>',line,'<p>']
endelse
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::read_html
file = self.web_dir+path_sep()+self.html_filename
exists = file_test(file)
if not exists then return,0
dum = '' & counter = 0L
openr,lun,file,/get_lun
while not eof(lun) do begin
   readf,lun,dum
   if counter eq 0L then strout = dum else $
      strout = [strout,dum]
   counter = counter+1L
endwhile
free_lun,lun,/force
; Find the line containing the header....
body_begin = -999
body_end = -999
for i = 0,counter-1 do begin
   body_pos = strpos(strout[i],'<body>')
   if body_pos[0] ne -1 then body_begin = i
   body_end_pos = strpos(strout[i],'</body>')
   if body_end_pos[0] ne -1 then body_end = i
endfor
if (body_begin ne -999) and (body_end ne -999) then begin
   file_contents = strout[body_begin+1:body_end-1]
endif else begin
   file_contents = ''
endelse
return,file_contents
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::append_html_file
old_contents = self->read_html()
if n_elements(*self.content_ptr) gt 0 then begin
   ; Ok, there is new content so append this to the existing file.
   new_contents = *self.content_ptr
   n = n_elements(old_contents)
   for i = 0,n-1 do begin
      where_match = where(*self.content_ptr eq old_contents[i],count_match)
   endfor
   ; Test if the new contents match the old contents
   if count_match ne n then begin
      strout = [old_contents,'<p>',new_contents]
      *self.content_ptr = strout
   endif
endif else begin
   ; There is no new content so fill up with the old stuff
   *self.content_ptr = old_contents
endelse
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::create_html

; Test if the file was created.
; If so then the new content shall be appended to the old.  If not then
; just create a new file.
file = self.web_dir+self.html_filename
already_exists = file_test(file)

strout = '<!-- This file was generated by DAVE_E_NOTES.PRO -->'
strout = [strout,'<!-- using the DAVE software package -->']

; Header.
strout = [strout,'<html>']
strout = [strout,' ']

 ; Title.
strout = [strout,"<head>"]
strout = [strout,"<TITLE>",self.title,"</TITLE>"]
strout = [strout,"</head>"]
strout = [strout," "]
strout = [strout, "<center><h1>"+self.heading+"</h1></center>"]
strout = [strout,"<body>"]
if n_elements(*self.content_ptr) ne 0 then $
   strout = [strout,*self.content_ptr]
strout = [strout,'<p>','Creation date:'+systime()]
strout = [strout,'</body>']
strout = [strout,'</html>']

file = self.web_dir+path_sep()+self.html_filename
openw,lun,file,/get_lun
for i = 0,n_elements(strout)-1 do printf,lun,strout[i]
free_lun,lun,/force
self.saved = self.saved + 1L
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::add_image,image_filename
read_png,image_filename,img,r,g,b

pos = strpos(image_filename,path_sep(),/reverse_search)
rel_file_name = strmid(image_filename,pos+1)

imgSize = size(img)
sx = imgSize[2] & sy = imgSize[3]
aspect2 = 1.0*sy/(1.0*sx)
outSx = 500 & outSy = fix(1.0*outSx*aspect2)
outsx = sx & outsy = sy
img_line =  '<img src="'+rel_file_name+'" width="'+ $
            strtrim(string(outSx),2)+ $
            '" height="'+strtrim(string(outSy),2)+'"><br>'
ret = self->add_line(img_line)
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::get_property,   title = title,             $
                                       heading = heading,         $
                                       web_browser = web_browser, $
                                       web_dir = web_dir,         $
                                       saved = saved,             $
                                       html_filename = html_filename
saved = (self.saved gt 0)
html_filename = self.html_filename
title = self.title
heading = self.heading
web_browser = self.web_browser
web_dir = self.web_dir
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::set_property,   title = title,             $
                                       new_lines = new_lines,     $
                                       heading = heading,         $
                                       web_browser = web_browser, $
                                       web_dir = web_dir,         $
                                       html_filename = html_filename

if n_elements(web_dir) ne 0 then self.web_dir = web_dir
if n_elements(html_filename) ne 0 then self.html_filename = html_filename
if n_elements(web_browser) ne 0 then self.web_browser = web_browser
if n_elements(heading) ne 0 then self.heading = heading
if n_elements(title) ne 0 then self.title = title

if n_elements(new_lines) ne 0 then begin
   n = n_elements(new_lines)
   for i = 0,n-1 do ret = self->add_line(new_lines[i],/strict)
endif

return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::view_web_page, ERRMSG = errmsg
file = self.web_dir+self.html_filename
errmsg = ''
if not file_test(file) then begin
   errmsg = 'Electronic notebook file could not be found.'
   return,0
endif

case strupcase(!version.os_family) of
    'WINDOWS': begin
        cmd = self.web_browser+' '+file
        spawn,cmd,/noshell,/nowait
    end

    else: begin
        case strupcase(!version.os) of
            'DARWIN': cmd = 'open -a '+self.web_browser+' '+file
            else: cmd = self.web_browser+' '+file+' &'
        endcase
        spawn, cmd
    end
endcase

return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dave_e_notes::init,  html_filename = html_filename,   $
                              title = title,                   $
                              heading = heading,               $
                              contents = contents,             $
                              web_browser = web_browser,       $
                              web_dir = web_dir

self.content_ptr = ptr_new(/allocate_heap)
if n_elements(web_dir) ne 0 then self.web_dir = web_dir else $
   self.web_dir = sourcepath()
if n_elements(web_browser) ne 0 then self.web_browser = web_browser
if n_elements(contents) ne 0 then *self.content_ptr = contents
if n_elements(title) eq 0 then title = 'DAVE Electronic Notebook'
self.title = title
if n_elements(heading) eq 0 then heading = 'DAVE Electronic Notebook'
self.heading = heading
if n_elements(html_filename) eq 0 then html_filename = 'e_notes.html'
self.html_filename = html_filename
self.saved = 0L
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro dave_e_notes__define
define = {  dave_e_notes,           $
            html_filename:'',       $
            web_dir:'',             $  ; Directory holding the notebook material
            web_browser:'',         $  ; default web browser
            title:'',               $  ; title of the web page
            heading:'',             $  ; heading of the web page
            saved:0L,               $  ; number of times saved
            content_ptr:ptr_new()   }  ; body of the web page

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro test_html
mk_html_help      ; simply reference this procedure in other to force the compiler to resolve mhh_strict
                  ; which is defined in mk_html.pro in the IDL distribution
web_dir = 'c:\bhd\dave_development\dave\programs\e_notebook\web_directory\'
image_file = web_dir + 'tas_plot.png'
read_png,image_file,image,r,g,b
oenotes = obj_new('dave_e_notes')
ret = oenotes->add_line('This is my e-notebook web page',/strict)
;wb = '"c:\Program Files\Internet Explorer\IEXPLORE.EXE"' ; the web browser of choice
wb = '"c:\Program Files\Netscape\Netscape 6\Netscp.exe"' ; the web browser of choice

ret = oenotes->set_property(web_browser = wb,web_dir = web_dir)
ret = oenotes->add_image(image_file)
ret = oenotes->create_html()
ret = oenotes->view_web_page()
obj_destroy,oenotes
end
