; $Id$
;###############################################################################
; NAME:
;   CalibratePSDA4
;
; PURPOSE:
;   PSD detector calibration module. It performs the following tasks
;   ** Load a calibration data file
;   ** Determine the final energy for each horizontal pixel for PSD
;   ** Determine the sensitivity or efficiency for each horizontal pixel
;   ** Save results in an output file
;
; CATEGORY:
;   Triple Axis Data Reduction
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;   azuah@nist.gov; (301) 9755604
;   Aug, 2007
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;==============================================================================
pro CalibratePSDA4_cleanup, wTLB
compile_opt idl2

widget_control, wTLB, get_uvalue=sPtr
wDelete, (*sPtr).winPix

heap_free, sPtr

;print,'Cleaup called'

end

pro CalibratePSDA4_help, event
compile_opt idl2

void = launch_help(!DAVE_PDFHELP_DIR+'spinscalibpsd.pdf',tlb = event.top)
return

end


;==============================================================================
pro CalibratePSDA4_loadData, sPtr, status=status, fromFTP=fromFTP
compile_opt idl2

status = 0
tlb = (*sPtr).wTLB
curDir = (*sPtr).dataDir


title = 'Select calibration file to load'
filter = ['*.ng5']

if (keyword_set(fromFTP)) then begin
  oFTP = (*sPtr).ftpObject
  if (n_elements(filter) eq 0) then filter = ['*.ng5']
  filename = Dialog_NCNRpublicData(oFTP,title=title,filter=filter,group_leader=tlb,count=count)
endif else begin
  filename = dialog_pickfile(title=title,/must_exist, dialog_parent=tlb $
                         ,filter=filter,path=curDir, get_path=newDir)
endelse

if (filename[0] eq '') then return

oData = obj_new('ng5CalibDataA4',filename[0], fromFTP=fromFTP, ftpObject=oFTP)

if (obj_valid(oData)) then begin
  if (obj_valid((*sPtr).oData)) then obj_destroy, (*sPtr).oData
  (*sPtr).oData = oData
  wID = widget_info(tlb,find_by_uname='VIEWPSD')
  if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
  wID = widget_info(tlb,find_by_uname='CALCEF')
  if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
  wID = widget_info(tlb,find_by_uname='CALCSEN')
  if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1

  if (~fromFTP) then (*sPtr).dataDir = newDir
  title = 'SPINS PSD Calibration (A4) -- '+file_basename(filename[0])
  widget_control, (*sPtr).wTLB, base_set_title=title
  status = 1 ; success
endif else $
  void = dialog_message('Unable to read contents of '+filename,/error,dialog_parent=tlb)
  
end

;==============================================================================
pro CalibratePSDA4_dispPSDimage, sPtr, _Extra=etc
compile_opt idl2

; Get data to display
oData = (*sPtr).oData
if (~obj_valid(oData)) then return
oData->getProperty, data=data, A4vals=y
xs = (size(data,/dimensions))[1]
x = indgen(xs)+1
xTitle = 'Pixel'
yTitle = 'A4'

Contour, transpose(data),x,y,xtitle=xTitle,ytitle=yTitle,/fill,nlevels=50, _Extra=etc $
       ,background=255,color=0,charsize=2.0,charthick=1.75
;spectrogram,data,x,y,xtitle=xTitle,ytitle=yTitle,/noskipgaps

end

;==============================================================================
pro CalibratePSDA4_dispFits, sPtr, _Extra=etc, pixel=pixel
compile_opt idl2

pixelFlag = keyword_set(pixel)

; Get data to display
oData = (*sPtr).oData
if (~obj_valid(oData)) then return
oData->getProperty, data=data
if (pixelFlag) then begin
   uname = 'DGSLIDER1'
   xTitle = 'A4'
   oData->getProperty, A4vals=x
   endif else begin
   uname = 'DGSLIDER'
   xTitle = 'Pixel'
   xs = (size(data,/dimensions))[1]
   x = indgen(xs)+1
endelse

;get the current slider value
wID = widget_info((*sPtr).wTLB,find_by_uname=uname)
if (widget_info(wID,/valid)) then widget_control, wID, get_value=index
if (n_elements(index) eq 0) then index=1
y = (pixelFlag)? data[*,index-1] : data[index-1,*]


yTitle = 'Counts'

plot,x,y,psym=4,symsize=2,xtitle=xtitle,ytitle=ytitle, _Extra=etc $
    ,background=255,color=0,charsize=2.0,charthick=1.75 $
    ,ticklen=1.0,xgridstyle=1,ygridstyle=1
    
; errors
errplot,x,y-sqrt(y),y+sqrt(y),color=0

; fit
yFit = (pixelFlag)? (*(*sPtr).pFit)[*,index-1] : (*(*sPtr).pFit)[index-1,*]
oplot,x,yfit,color=0,thick=2.0

end

;==============================================================================
pro CalibratePSDA4_dispEf, sPtr, _Extra=etc
compile_opt idl2

; Get data to display
oData = (*sPtr).oData
if (~obj_valid(oData)) then return
oData->getProperty, nPix=nPix, A4p=A4p
x = indgen(nPix)+1

;get Ef
pixFits = (*(*sPtr).pPixFits)
a4Fits =  (*(*sPtr).pa4Fits)
xTitle = 'Pixel'
yTitle = 'A4[127] - A4'

plot,x,a4p[127]-a4p,xtitle=xtitle,ytitle=ytitle, _Extra=etc $
    ,background=255,color=0,charsize=2.0,charthick=1.75,ystyle=16 $
    ,ticklen=1.0,xgridstyle=1,ygridstyle=1

; actual fitted pix positions from gaussian peaks
oplot,pixFits,a4Fits[127]-a4Fits,color=0,psym=4,symsize=1.25

end

;==============================================================================
pro CalibratePSDA4_dispSen, sPtr, _Extra=etc
compile_opt idl2

; Get data to display
oData = (*sPtr).oData
if (~obj_valid(oData)) then return
oData->getProperty, nPix=nPix, sen=sen
x = indgen(nPix)+1

xTitle = 'Pixel'
yTitle = 'Sensitivity'

plot,x,sen,xtitle=xtitle,ytitle=ytitle,psym=4,symsize=2, _Extra=etc $
    ,background=255,color=0,charsize=2.0,charthick=1.75,ystyle=16 $
    ,ticklen=1.0,xgridstyle=1,ygridstyle=1

end

;==============================================================================
pro CalibratePSDA4_refresh, sPtr
compile_opt idl2
; save current device settings
device,get_decomposed = dc
tvlct,r,g,b,/get

; load a default color table
device,decomposed = 0 ; don't want color decomposition
;colors = hfbs_GetColor(/Load, Start=1) ; custom (16) color table
loadct,39,/silent

; Generate the plot
wset,(*sPtr).winPix
plotType = (*sPtr).plotType
;if (n_elements(plottype) eq 0) then plottype=0

; determine xrange and yrange
zp2=(*sPtr).zp2
etc = {dummy:0}
if (zp2[0] gt 0 || zp2[1] gt 0) then begin
    zp1 = (*sPtr).zp1
    x = min([zp1[0], zp2[0]])
    y = min([zp1[1], zp2[1]])
    w = abs(zp1[0] - zp2[0])
    h = abs(zp1[1] - zp2[1])
    x2 = x+w
    y2 = y+h
    lc = convert_coord(x, y, /device, /to_data)
    uc = convert_coord(x2, y2, /device, /to_data)
    etc = {xrange:[lc[0],uc[0]],yrange:[lc[1],uc[1]]}
endif

case plotType of
  0: CalibratePSDA4_dispPSDimage, sPtr, _Extra=etc
  1: CalibratePSDA4_dispFits, sPtr, _Extra=etc
  2: CalibratePSDA4_dispEf, sPtr, _Extra=etc
  3: CalibratePSDA4_dispSen, sPtr, _Extra=etc
  11: CalibratePSDA4_dispFits, sPtr, _Extra=etc, /pixel
endcase
wset,(*sPtr).winVis
device,copy = [0,0,(*sPtr).winXs,(*sPtr).winYs,0,0,(*sPtr).winPix]

; revert to original device settings
device,decomposed = dc
tvlct,r,g,b
end

;==============================================================================
;; Fit Gaussians to data. The fitted peak position at each data point is the pixel
;; which corresponds to the final energy for that data point.

; Fit Gaussians to data. The fitted peak position at each pixel is the Energy transfer
; which corresponds with that pixel
function CalibratePSDA4_doFits, sPtr
compile_opt idl2

oData = (*sPtr).oData
oData->getProperty,data=data,nDat=nDat,nPix=nPix,A4vals=A4,Efix=EfixAna $
                  ,meanInt=meanInt,sigmaInt=sigmaInt

if (ptr_valid((*sPtr).pFit)) then $
  (*(*sPtr).pFit) = data*0.0 $
else $
  (*sPtr).pFit = ptr_new(data*0.0)

;x = findgen(nDat)+1.0 
x = A4
n = n_elements(A4)
;Erange = E[n-1] - E[0]
;Epeaks = (findgen(nPix)+1) * (Erange/float(nPix)) + E[0]
;EpeaksMin = Epeaks*0.7 - 0.1*Erange 
;EpeaksMax = Epeaks*0.7 + 0.1*Erange ; +75%
;sigmaRange = 0.03 ; estimate of the range of widths for this PSD ie 0.02 -> 0.05
;Esigma = (findgen(nPix)+1) * (sigmaRange/float(nPix)) + 0.02
A4min = min(A4)
A4max = max(A4)
Esigma = fltarr(nPix)+1.0 ;assume widths of approx 1.0
EsigmaMin = Esigma * 0.1  ; -10%
EsigmaMax = Esigma * 2  ; +100%

;p = [0.0,0.0,1.0,0.0,1.0] ; bc,bs,area,center,sigma
p = [1.0,0.0,1.0,0.0,0.0] ; area,center,sigma,bc,bs

oDiffEvo = SPINSCalib_EPSDE()
a4PixFit = []
pixel = []
a4PixFitErr =[]
pixArea = []
sqrt2pi = sqrt(2*!pi)

; create a progressbart widget to show fit progress
wProg = nse_cwo_progress(title='SPINS PSD Calibration' $
                        ,label=['Fitting pixel:'] $
                        ,values=[0L],startvalues=[0L],endvalues=[nPix],steps=[1L] $
                        ,dialog_parent=(*sPtr).wTLB, group_leader=(*sPtr).wTLB $
                        ,obj=oProg,/nostop)
for i=0,nPix-1 do begin
  y = reform(data[*,i])
  yerror = sqrt(y)
  ind = where(yerror le 0.0,cnt)
  if (cnt gt 0) then yerror[ind] = 1.0
  Amin = max(y) * 0.5
  Amax = max(y) * 1.5
  BCmax = max(y) * 0.05
  BCmin = 0.0
  ;pMin = [Amin,EpeaksMin[i],EsigmaMin[i],BCmin]
  ;pMax = [Amax,EpeaksMax[i],EsigmaMax[i],BCmax]
  pMin = [Amin,A4min,EsigmaMin[i],BCmin]
  pMax = [Amax,A4max,EsigmaMax[i],BCmax]
  oDiffEvo->SetProperty, xData=x, yData=y, yError=yError, pMin=pMin, pMax=pMax, startingflag=0
  oProg->step, 0  ; update progress bar  
  status = oDiffEvo->Fit()  
  oDiffEvo->GetProperty, initparam=p0, bestparam=p1, bestfvalue=chisq, generationCnt=genCnt
  status = oDiffEvo->Model(p1,calc=yfit)  
;  if (i gt 110) then begin
;    print,'###### Fitting pixel ',i, '  chisq = ',chisq
;  endif else begin
;    print,'###### Fitting pixel ',i, '  chisq = ',chisq
;  endelse
  (*(*sPtr).pFit)[*,i] = yfit
    
  a4PixFit = [a4PixFit,p1[1]]  ; a4 at each pixel = peak pos
  a4PixFitErr = [a4PixFitErr,p1[1]*0.1]  ; fitted peak pos error
  pixArea = [pixArea,p1[0]*p1[2]*sqrt2pi]    ; area = Imax * sigma * sqrt(2*pi)
  pixel = [pixel,i+1]  ; note pixel that was fitted
endfor

widget_control, wProg, /destroy ; delete progress bar widget
obj_destroy, oDiffEvo

oData->SetProperty, sen=pixArea  ; store area in sensitivity field


; Now know the energy transfer (peak position) at each pixel that was fitted.
; Now need to extrapolate the results using a linear interpolation/fit to obtain 
; a smooth energy transfer for any pixel.
; pixel = pixels that were fitted
; a4PixFit = energy transfer for each fitted pixel
; a4PixFitErr = error in the energy transfer
; ePix  = effective energy transfer at each pixel
; efPix = effective final energy for each pixel


lparam = poly_fit(pixel,a4PixFit,1,measure_errors=a4PixFitErr,status=status) ; fit to poly of degree 1
x = findgen(npix) + 1.0
if (status eq 0) then begin
  a4Pix = lparam[0] + lparam[1]*x
endif else begin
  ; fit unsuccessful so use interpolation
  a4Pix = interpol(a4PixFit,pixel,x,/lsquadratic)
endelse

;; Now know the final energy at the pixel locations that coincide with the 
;; center of the Gaussian peaks. Now need to extrapolate the results using
;; a linear interpolation/fit to obtain final energy for any pixel.
;; pixDpt = pixel location of peak for each data point
;; efDpt = effective final energy for each data point
;; efp = effective final energy for each pixel
;
;
;;NB - pixError is actually the error in the pixel location and NOT Ef as it is used below!
;lparam = poly_fit(pixDpt,efDpt,1,measure_errors=pixError,status=status) ; fit to poly of degree 1
;if (status eq 0) then begin
;  efp = lparam[0] + lparam[1]*x
;endif else begin
;  ; fit unsuccessful so use interpolation
;  efp = interpol(efDpt,pixDpt,x,/lsquadratic)
;endelse


;; effective final energy at each pixel = Energy transfer at pixel + fixed analyzer energy
;efPix = ePix + EfixAna
;oData->setProperty, Efp=efPix
oData->setProperty, A4p=A4Pix ; a4 at each pixel relative to the central pixel

if (ptr_valid((*sPtr).pPixFits)) then $      ; store the pixels that were fitted
  (*(*sPtr).pPixFits) = pixel $
else $
  (*sPtr).pPixFits = ptr_new(pixel)
if (ptr_valid((*sPtr).pa4Fits)) then $
  ;(*(*sPtr).pa4Fits) = a4PixFit + EfixAna $    ; and the corresponding final energy
  (*(*sPtr).pa4Fits) = a4PixFit $    ; and the corresponding A4
else $
  ;(*sPtr).pa4Fits = ptr_new(a4PixFit + EfixAna)
  (*sPtr).pa4Fits = ptr_new(a4PixFit)

;if (ptr_valid((*sPtr).pPixFits)) then $
;  (*(*sPtr).pPixFits) = pixDpt $
;else $
;  (*sPtr).pPixFits = ptr_new(pixDpt)
;if (ptr_valid((*sPtr).pa4Fits)) then $
;  (*(*sPtr).pa4Fits) = EfDpt $
;else $
;  (*sPtr).pa4Fits = ptr_new(EfDpt)

return, 1
end


;==============================================================================
; Evaluate the sensitivity or efficiency for each pixel in the horizontal direction 
; (the data is already vertically integrated in hardware).
pro CalibratePSDA4_gaussian, x, p, yout, pDer
area = p[0]
cen = p[1]
sig = p[2]
bc = p[3]
bs = p[4]

yout = (area/sqrt(2.0*!pi*sig^2))*exp(-0.5*((x-cen)/sig)^2) + bc + bs*x

end


;==============================================================================
; Evaluate the sensitivity or efficiency for each pixel in the horizontal direction 
; (the data is already vertically integrated in hardware).
function CalibratePSDA4_calcSen, sPtr, minPix, maxPix 
compile_opt idl2

oData = (*sPtr).oData
oData->getProperty,data=data,nPix=nPix,A4p=A4p, A4vals=A4, sen=pixArea

;; For each of pixel, sum counts over all data points
;sumP = total(data,1) 

; Fit a Gaussian + linear bkgd to data to determine the area.
; The fits are done for each pixel as a function of data points. The fitted area of
; the Gaussian essentially gives the integrated intensity in each pixel.
x = A4

dA4dPix = abs(deriv(A4p)) ; dA4p/dp - gradient of A4p wrt pixels
normFactor = nPix/total(pixArea/dA4dPix)
sens = (pixArea/dA4dPix) * normFactor
sensError = sqrt(pixArea)/dA4dPix*normFactor

oData->SetProperty, sen=sens

return, 1

end


;==============================================================================
pro CalibratePSDA4_drawEvent, sPtr, event
compile_opt idl2

case event.type of
   0: begin                ; button press
        (*sPtr).mouseType = event.press
        if ((*sPtr).mouseType eq 4) then begin ; right-hand mb so reset plot to full range
            CalibratePSDA4_refresh, sPtr
            (*sPtr).mouseType = 0 ; set mouse type to undefined value
            return
        endif
        if ((*sPtr).mouseType eq 1) then begin ; left-hand mb so record mouse location
            (*sPtr).zp1 = [event.x,event.y]
        endif
    end
    1: begin                ; button release
        if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
            CalibratePSDA4_refresh, sPtr
            (*sPtr).zp2 = [0,0] ; reset to zero
        endif
        (*sPtr).mouseType = 0
    end
    2: begin                ; mouse motion
        if ((*sPtr).mouseType eq 1) then begin ; mouse press was a left-hand mb
            (*sPtr).zp2 = [event.x, event.y]
            zp1 = (*sPtr).zp1
            zp2 = (*sPtr).zp2
            xc = [zp1[0], event.x, event.x, zp1[0], zp1[0]]
            yc = [zp1[1], zp1[1], event.y, event.y, zp1[1]]
            device, copy=[0,0,(*sPtr).winXs,(*sPtr).winYs,0,0,(*sPtr).winPix] ; copy contents of winPix to winVis
;            device,get_decomposed = dc
;            device,decomposed = 0
            plots, xc, yc, /device, color=(*sPtr).colors.red
;            device,decomposed = dc
        endif
    end
    else:
endcase    



end


;==============================================================================
pro CalibratePSDA4_writeImage, sPtr, jpeg=jpeg, png=png
compile_opt idl2

writejpeg = (keyword_set(jpeg))? 1 : 0
writepng = (keyword_set(png))? 1 : 0
if (~writejpeg && ~writepng) then return

fname = (writejpeg)? 'psd_calibration.jpeg' : 'psd_calibration.png'
ext =  (writejpeg)? 'jpeg' : 'png'
flt =  (writejpeg)? '*.jpeg' : '*.png'

fname = dialog_pickfile(default_extension=ext,dialog_parent=(*sPtr).wTLB,filter=flt,file=fname $
                        ,/overwrite_prompt,path=(*sPtr).workDir,/write)
if (fname eq '') then return

wset, (*sPtr).winVis
image = tvrd()

if (writejpeg) then write_jpeg, fname,image else write_png, fname, image

end


;==============================================================================
pro CalibratePSDA4_save, sPtr
compile_opt idl2

title = 'Choose Effective A4 to Save'
msg = 'Do you wish to save the interpolated A4?'
msg = [msg,'Select "No" to save the exact A4']

answer = dialog_message(/question, title=title, msg, dialog_parent=(*sPtr).wTLB)

(*sPtr).oData->GetProperty, filename=rawFilename
fname = file_basename(rawfilename,'.ng5',/fold)+'_calibration.txt'
ext =  'txt'
flt = '*.txt'

fname = dialog_pickfile(default_extension=ext,dialog_parent=(*sPtr).wTLB,filter=flt,file=fname $
                        ,/overwrite_prompt,path=(*sPtr).workDir,/write)
if (fname eq '') then return

oData = (*sPtr).oData
oData->getProperty,NPIX=nPix,A4P=A4p,SEN=sen

if (~strcmp(answer,'Yes',/fold_case)) then begin
   pixFits = (*(*sPtr).pPixFits)
   A4p =  (*(*sPtr).pa4Fits)
   
   if (n_elements(pixFits) ne nPix) then begin
      rawEf = findgen(nPix)*0.0
      rawEf[pixFits] = A4p
      A4p = rawEf
   endif
endif

openw,lun,fname,/get_lun

printf,lun, '#SPINS PSD Calibration based on: '+rawFilename
printf,lun, '#Created on: '+systime()
printf,lun, '#number of Pixels'
printf,lun, '#A4 block - Delta A4 for each pixel relative to the A4 of the central pixel (127)'
printf,lun, '#Efficiency block - effieciency divisor for each pixel'
printf,lun,nPix
printf,lun,a4p[127]-a4p,format='(8G10.4)'
printf,lun,sen,format='(8G10.4)'

free_lun, lun,/force
end


;==============================================================================
pro CalibratePSDA4_event, event
compile_opt idl2

;begin error handler------------------------------------------------------------
defsysv,'!debug',exist=debugExist
if (debugExist && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'Error encountered in CalibratePSDA4_Event'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle)
        catch, /cancel
        return
    endif
endif
;end error handler
widget_control, event.top, get_uvalue=sPtr
tlb = (*sPtr).wTLB

uname = widget_info(event.id,/uname)

case uname of
  'EXIT': widget_control, event.top, /destroy
  
  'HELP': CalibratePSDA4_help, event
  
  'DRAW': CalibratePSDA4_DrawEvent, sPtr, event
  
  'JPEG': CalibratePSDA4_writeImage, sPtr, /jpeg

  'PNG': CalibratePSDA4_writeImage, sPtr, /png
  
  'LOAD': begin
    widget_control, event.id, get_uvalue=fromFTP
    CalibratePSDA4_loadData, sPtr, status=status, fromFTP=fromFTP 
    if (~status) then return
    
    ; Reset widget states

    wNames = ['VIEWFITS','DGSLIDER','VIEWFITS1','DGSLIDER1','VIEWEF','VIEWSEN'] ; de-sensitize these
    for i=0,n_elements(wNames)-1 do begin
        wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0
    endfor
    ; update slider widget range
    oData = (*sPtr).oData
    oData->getProperty, nDat=nDat, nPix=nPix
    wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER')
    if (widget_info(wID,/valid)) then widget_control, wID, set_slider_max=nDat, set_slider_min=1
    wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER1')
    if (widget_info(wID,/valid)) then widget_control, wID, set_slider_max=nPix, set_slider_min=1
    
    ; refresh plot window
    wID = widget_info((*sPtr).wTLB,find_by_uname='VIEWPSD')
    if (widget_info(wID,/valid)) then $
        widget_control, wID, /set_button,send_event={id:wID,top:tlb,handler:tlb,select:1}
    
;    (*sPtr).plotType = 0
;    CalibratePSDA4_refresh, sPtr;, type=0
  end
  
  'VIEWPSD': begin
    widget_control, event.id, set_uvalue=event.select
    if (event.select eq 1) then begin
      (*sPtr).plotType = 0
      CalibratePSDA4_refresh, sPtr;, type=0
      
      ; Uncheck these
      wNames = ['VIEWFITS' $
               ,'VIEWFITS1' $
               ,'VIEWEF' $
               ,'VIEWSEN']
      
      for i=0,n_elements(wNames)-1 do begin
          wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
          if (widget_info(wID,/valid)) then widget_control, wID, set_button=0
          widget_control, wID, set_uvalue=0
      endfor
    endif else begin
    
    endelse
  end
  
  'VIEWEF': begin
    widget_control, event.id, set_uvalue=event.select
    if (event.select eq 1) then begin

      ; Uncheck these
      wNames = ['VIEWFITS' $
               ,'VIEWFITS1' $
               ,'VIEWSEN' $
               ,'VIEWPSD']
      
      for i=0,n_elements(wNames)-1 do begin
          wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
          if (widget_info(wID,/valid)) then widget_control, wID, set_button=0
          widget_control, wID, set_uvalue=0
      endfor
      
      ; Update plot with Ef plot
      (*sPtr).plotType = 2
      CalibratePSDA4_refresh, sPtr;, type=2
    endif 
  end

  'VIEWSEN': begin
    widget_control, event.id, set_uvalue=event.select
    if (event.select eq 1) then begin

      ; Uncheck these
      wNames = ['VIEWFITS' $
               ,'VIEWFITS1' $
               ,'VIEWEF' $
               ,'VIEWPSD']
      
      for i=0,n_elements(wNames)-1 do begin
          wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
          if (widget_info(wID,/valid)) then widget_control, wID, set_button=0
          widget_control, wID, set_uvalue=0
      endfor
            
      ; Update display with sen plot
      (*sPtr).plotType = 3
      CalibratePSDA4_refresh, sPtr;, type=3
    endif 
  end


  'VIEWFITS': begin
    widget_control, event.id, set_uvalue=event.select
    if (event.select eq 1) then begin
      ; sensitize slider
      wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER')
      if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1

      ; Uncheck these
      wNames = ['VIEWSEN' $
               ,'VIEWFITS1' $
               ,'VIEWEF' $
               ,'VIEWPSD']
      
      for i=0,n_elements(wNames)-1 do begin
          wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
          if (widget_info(wID,/valid)) then widget_control, wID, set_button=0
          widget_control, wID, set_uvalue=0
      endfor

      ; Update plot with fits
      (*sPtr).plotType = 1
      CalibratePSDA4_refresh, sPtr;, type=1
    endif else begin
      wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER')
      if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0    
    endelse
  end
  
  'VIEWFITS1': begin
    widget_control, event.id, set_uvalue=event.select
    if (event.select eq 1) then begin
      ; sensitize slider
      wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER1')
      if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1

      ; Uncheck these
      wNames = ['VIEWSEN' $
               ,'VIEWFITS' $
               ,'VIEWEF' $
               ,'VIEWPSD']
      
      for i=0,n_elements(wNames)-1 do begin
          wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
          if (widget_info(wID,/valid)) then widget_control, wID, set_button=0
          widget_control, wID, set_uvalue=0
      endfor

      ; Update plot with fits
      (*sPtr).plotType = 11
      CalibratePSDA4_refresh, sPtr;, type=1
    endif else begin
      wID = widget_info((*sPtr).wTLB,find_by_uname='DGSLIDER1')
      if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0    
    endelse
  end
  
  'DGSLIDER': begin
    buttonIsSet = 0
    wID = widget_info((*sPtr).wTLB,find_by_uname='VIEWFITS')
    if (widget_info(wID,/valid)) then widget_control, wID, get_uvalue=buttonIsSet
    if (buttonIsSet) then begin
        (*sPtr).plotType = 1
        CalibratePSDA4_refresh, sPtr;, type=1
    endif
  end
  
  'DGSLIDER1': begin
    buttonIsSet = 0
    wID = widget_info((*sPtr).wTLB,find_by_uname='VIEWFITS1')
    if (widget_info(wID,/valid)) then widget_control, wID, get_uvalue=buttonIsSet
    if (buttonIsSet) then begin
        (*sPtr).plotType = 11
        CalibratePSDA4_refresh, sPtr;, type=1
    endif
  end
  
  'CALCEF': begin
    ok = CalibratePSDA4_doFits(sPtr)
    if (ok) then begin ; update some widget states

      ; Sensitize the following and refresh plot, if applicable
      wNames = ['VIEWFITS' $
               ,'VIEWFITS1' $
               ,'VIEWEF']

      ; sensitize view fits and refresh plot
      for i=0,n_elements(wNames)-1 do begin      
         wID = widget_info((*sPtr).wTLB,find_by_uname=wNames[i])
         if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
         widget_control, wID, get_uvalue=buttonIsSet
         if (buttonIsSet) then CalibratePSDA4_refresh, sPtr
      endfor
    endif
  end
  
  'CALCSEN': begin
    ok = CalibratePSDA4_calcSen(sPtr)

    if (ok) then begin ; update some widget states
      ; sensitize view sen and refresh plot
      wID = widget_info((*sPtr).wTLB,find_by_uname='VIEWSEN')
      if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
      widget_control, wID, get_uvalue=buttonIsSet
      if (buttonIsSet) then CalibratePSDA4_refresh, sPtr;, type=3
    endif    
  end
  
  'SAVE': begin
    CalibratePSDA4_save, sPtr
  end
  
  else: print,'Event generated by ',uname
endcase



end 

;==============================================================================
pro CalibratePSDA4, group_leader=group_leader, workDir=workDir, dataDir=dataDir, DAVETool=oDAVETool, _EXTRA=etc
compile_opt idl2

registeredName = 'CalibratePSDA4'
if xregistered(registeredName) then return

if (n_elements(workDir) eq 0) then workDir = ''
if (n_elements(dataDir) eq 0) then dataDir = ''
if (n_elements(DaveTool) eq 0) then DaveTool = obj_new()

; Create a custom color table for later use
tvlct,r,g,b,/get
device,decomposed = 0
colors = hfbs_GetColor(/Load, Start=1) ; custom (16) color table
device,decomposed = dc
tvlct,r,g,b

; define top-level base widget; make it resizeable
title = 'SPINS PSD Calibration'
wTLB = widget_base(group_leader=group_leader,/row,title=title,mbar=mbar,/tlb_size_events)

; menus
fileMenu = widget_button(mbar, value='File',/menu)
; fill the 'File' menu
void = widget_button(fileMenu,value='Load calibration file',uname='LOAD', uvalue=0)
void = widget_button(fileMenu,value='Load calibration file (NCNR ftp server)',uname='LOAD', uvalue=1)
;void = widget_button(fileMenu,value='Export plot to PS',uname='PS',/separator)
void = widget_button(fileMenu,value='Export plot to jpeg',uname='JPEG',/separator)
void = widget_button(fileMenu,value='Export plot to png',uname='PNG')
void = widget_button(fileMenu,value='Save calibration results',/separator,uname='SAVE')
void = widget_button(fileMenu,value='Exit',uname='EXIT',/separator)

helpMenu = widget_button(mbar, value='Help',/menu)
void = widget_button(helpMenu,value='Mini Manual', uname='HELP')

wCB1 = widget_base(wTLB,/col) ; controls base
wCB2 = widget_base(wTLB,/row) ; graphics base

wCB10 = widget_base(wCB1,/col,/frame)
void = cw_coloredlabel(wCB10,value='Raw Calibration File',/align_center,foreground_color=[255,0,0])
openBmp = filepath('open.bmp',subdirectory=['resource','bitmaps'])
tt = 'Load a PSD raw calibration dataset'
void = widget_button(wCB10,value=openBmp,/bitmap,/align_center,uname='LOAD',tooltip=tt)
;wCB10a = widget_base(wCB10,/row)
;void = widget_label(wCB10a,value='Loaded:')
;wLoadFile = widget_label(wCB10a,value='',/dynamic_resize,uname='FNAME')
wCB10a = widget_base(wCB10,/nonexclusive)
tt = 'View the raw PSD image data in the graphics window'
void = widget_button(wCB10a,value='View PSD Data',sensitive=0,uname='VIEWPSD',tooltip=tt)
widget_control, void, /set_button, set_uvalue=1

wCB11 = widget_base(wCB1,/col,/frame)
void = cw_coloredlabel(WCB11,value='Energy Calibration',/align_center,foreground_color=[255,0,0])
void = widget_label(WCB11,value='',/align_center)

void = widget_label(WCB11,value='Intensity Threshold',/align_center)
void = widget_label(WCB11,value='(Multiple of Std Dev',/align_center)
void = cw_field(wCB11,title='of Mean Intensity):',value=10.0,/col,/return_events $
                ,uname='THRESH',/float)
tt = 'Perform Gaussian fits of the peaks for each data point'
void = widget_button(wCB11,value='Fit Gaussians',/align_center,uname='CALCEF',tooltip=tt,sensitive=0)

wCB11a = widget_base(wCB11,/nonexclusive)
tt = 'View the Gaussian fits for each data point in the graphics window'
void = widget_button(wCB11a,value='View Fits: Data Point',sensitive=0,uname='VIEWFITS',tooltip=tt)
widget_control, void, set_button=0, set_uvalue=0
void = widget_slider(wCB11,title='Data point to view',sensitive=0,uname='DGSLIDER')

wCB11b = widget_base(wCB11,/nonexclusive)
tt = 'View the Gaussian fits for each pixel in the graphics window'
void = widget_button(wCB11b,value='View Fits',sensitive=0,uname='VIEWFITS1',tooltip=tt)
widget_control, void, set_button=0, set_uvalue=0
void = widget_slider(wCB11,title='Pixel to view',sensitive=0,uname='DGSLIDER1')

wCB11c = widget_base(wCB11,/nonexclusive)
tt = 'View the evaluated final energy for each pixel in the graphics window'
void = widget_button(wCB11c,value='View Calc. A4(pixel)',sensitive=0,uname='VIEWEF',tooltip=tt)
widget_control, void, set_button=0, set_uvalue=0

wCB12 = widget_base(wCB1,/col,/frame)
void = cw_coloredlabel(wCB12,value='Sensitivity Calibration',/align_center,foreground_color=[255,0,0])
void = widget_label(wCB12,value='')
tt = 'Evaluate the detector efficiency at each pixel'
void = widget_button(wCB12,value='Calibrate Sen.',uname='CALCSEN',tooltip=tt,sensitive=0)
wCB12a = widget_base(wCB12,/nonexclusive)
tt = 'View the evaluated sensitivity for each pixel in the graphics window'
void = widget_button(wCB12a,value='View Sen(pixel)',sensitive=0,uname='VIEWSEN',tooltip=tt)
widget_control, void, set_button=0, set_uvalue=0

; draw widget
xs = 650 & ys = 500
void = widget_draw(wCB2,xsize=xs,ysize=ys,uname='DRAW',/button_events,/motion_events,retain=2)


window,/free,/pixmap,xsize=xs,ysize=ys
winPix = !d.window

widget_control, wTLB,/realize
widget_control, void, get_value=winVis

IDLge90 = (Float(!version.release) ge 9.0)? 1 : 0
oFTP =  (IDLge90)? Davehttprequest() :  Daveftpurl()

oFTP->SetProperty, currentDir = '/pub/ncnrdata/ng5/'

sPtr = ptr_new({wTLB:wTLB $
                ,wCB1:wCB1 $
                ,wCB2:wCB2 $
                ,winVis:winVis $
                ,winPix:winPix $
                ,winXs:xs $
                ,winYs:ys $
                ,mouseType:0 $
                ,plotType:0 $
                ,zp1:[0,0] $
                ,zp2:[0,0] $
                ,colors:colors $
                ,oData:obj_new() $
                ,pFit:ptr_new() $
                ,pPixFits:ptr_new() $
                ,pa4Fits:ptr_new() $
                ,workDir:workDir $
                ,dataDir:dataDir $
                ,daveTool:DaveTool $
                ,ftpObject:oFTP $
               })
widget_control, wTLB, set_uvalue=sPtr

xmanager, registeredName, wTLB, cleanup='CalibratePSDA4_cleanup', /no_block

end

;
;;==============================================================================
;; Function to evalute a single Gaussian + sloping bkgd
;pro single_gaussian, x,p,yfit
;bkgd = p[0] + p[1]*x
;ggaus = p[2]*exp(-0.5*((x-p[3])/p[4])^2)
;;ggaus = p[2]/(sqrt(2.0*!dpi)*p[4]) * exp(-0.5*((x-p[3])/p[4])^2) ; normalized
;yfit = bkgd + ggaus
;
;end
