; $Id$
; FWS_ANALYSIS__DEFINE.PRO
;###############################################################################
;
; NAME:
;  FWS_ANALYSIS__DEFINE
;
; PURPOSE:
;  Class definition for the fixed window scan data analysis application.
;
; CATEGORY:
;  DAVE, HFBS, Fixed Window Scans, Analysis
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysisCleanup,tlb
compile_opt idl2,hidden
; This cleanup is executed after the quit method is invoked.
widget_control,tlb,get_uvalue = self
obj_destroy,self
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::cleanup
compile_opt idl2,hidden
; This cleanup is executed after fws_analysisCleanup
wdelete,self.winPix,self.fitWinPix

if ptr_valid(self.px) gt 0 then ptr_free,self.px
if ptr_valid(self.py) gt 0 then ptr_free,self.py
if ptr_valid(self.pxfit) gt 0 then ptr_free,self.pxfit
if ptr_valid(self.pyfit) gt 0 then ptr_free,self.pyfit

ptr_free,self.xpa,self.ypa,self.yerrpa,self.colorsPtr
ptr_free,self.xfitPtr,self.yfitPtr,self.tempPtr
ptr_free,self.xdatPtr,self.ydatPtr,self.ydatErrPtr
obj_destroy,[self.oContainer,self.legContainer]

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function fws_myPlotSym,index,_Extra = extra
compile_opt idl2,hidden
case index mod 10 of
0: sym = dave_plotSym(/circle,_Extra = extra)
1: sym = dave_plotSym(/triangle,_Extra = extra)
2: sym = dave_plotSym(/diamond,_Extra = extra)
3: sym = dave_plotSym(/box,_Extra = extra)
4: sym = 0
5: sym = dave_plotSym(/circle,/fill,_Extra = extra)
6: sym = dave_plotSym(/triangle,/fill,_Extra = extra)
7: sym = dave_plotSym(/diamond,/fill,_Extra = extra)
8: sym = dave_plotSym(/box,/fill,_Extra = extra)
9: sym = dave_plotSym(/triangle,/fill,angle = 180.0,_Extra = extra)
else:
endcase
return,sym
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::quit,event = event
compile_opt idl2,hidden
tvlct,self.r,self.g,self.b
widget_control,self.group_leader,sensitive = 1

if widget_info(self.tlb,/valid_id) gt 0 then $
    widget_control,self.tlb,/destroy
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::resize,event = event
compile_opt idl2,hidden
geom1 = widget_info(self.colBase1,/geometry)
geom2 = widget_info(self.colBase2,/geometry)

xsize = event.x
ysize = event.y

widget_control,self.win,draw_xsize = xsize-geom1.xsize-geom2.xsize, $
               draw_ysize = ysize
widget_control,self.fitWin,draw_xsize = xsize-geom1.xsize-geom2.xsize, $
         draw_ysize = ysize
; Don't forget to resize the pixmap window!
wdelete,self.winPix
window,/free,/pixmap,xsize = xsize-geom1.xsize-geom2.xsize,ysize = ysize
self.winPix = !d.window
wdelete,self.fitWinPix
window,/free,/pixmap,xsize = xsize-geom1.xsize-geom2.xsize,ysize = ysize
self.fitWinPix = !d.window

self->refreshScreen, event = event
self->drawFits
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::getDepVar
widget_control,self.depVarGroup,get_value = depVal
nbuffers = self.oContainer->count()
oall = self.oContainer->get(/all)
case depVal[0] of
0: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,time = time
       *self.xpa[i] = time
     endfor

     self.xlabel = 'time (min)'
   end
1: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,tsample = tsample
       *self.xpa[i] = tsample
     endfor
     self.xlabel = 'T (K) sample'
   end
2: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,tcontrol = tcontrol
       *self.xpa[i] = tcontrol
     endfor
     self.xlabel = 'T (K) control'
   end
3: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,q = q
       *self.xpa[i] = q
     endfor
     self.xlabel = 'Q (!3!sA!r!u!9 %!3!n!E-1!N)'
   end
else:
endcase
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::getinDepVar
widget_control,self.indepVarGroup,get_value = val

widget_control,self.depVarGroup,get_value = depVal

nbuffers = self.oContainer->count()
oall = self.oContainer->get(/all)

case val[0] of
0: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,intensity = intensity, error = error,q = q
       if depVal[0] ne 3 then begin
         widget_control,self.plotDetectors,get_value = strdets
         detectors = opan_selectgroups(strdets)
         d = intensity[detectors,*]
         de = error[detectors,*]
         eout = total(d,1)
         deout = sqrt(total(de^2,1))
         *self.ypa[i] = eout
         *self.yerrpa[i] = deout
       endif
     self.ylabel = 'Intensity (arb units)'
     endfor
   end
1: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,usq = usq, uerr = uerr
       if usq[0] ne -999 then begin
         *self.ypa[i] = usq
         *self.yerrpa[i] = uerr
       endif
     self.ylabel = '<u!u2!n>(T)-<u!u2!n>(T=0)'
     endfor
   end
2: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,wbm = wbm
     *self.ypa[i] = wbm
     *self.yerrpa[i] = sqrt(wbm)
     self.ylabel = 'White beam intensity (arb units)'
     endfor
   end
3: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,ibm = ibm
     *self.ypa[i] = ibm
     *self.yerrpa[i] = sqrt(ibm)
     self.ylabel = 'Incident beam intensity (arb units)'
     endfor
   end
4: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,tbm = tbm
     *self.ypa[i] = tbm
     *self.yerrpa[i] = sqrt(tbm)
     self.ylabel = 'Transmitted beam intensity (arb units)'
     endfor
   end
5: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,ratio = ratio,eratio = eratio
     *self.ypa[i] = ratio
     *self.yerrpa[i] = eratio
     self.ylabel = 'TBM/IBM'
     endfor
   end
6: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,time = time
     *self.ypa[i] = time
     ptr_free,self.yerrpa[i]
     self.yerrpa[i] = ptr_new(/allocate_heap)
     self.ylabel = 'Time (min)'
     endfor
   end
7: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,tsample = tsample
     *self.ypa[i] = tsample
     ptr_free,self.yerrpa[i]
     self.yerrpa[i] = ptr_new(/allocate_heap)
     self.ylabel = 'T (K) sample'
     endfor
   end
8: begin
     for i = 0,nbuffers-1 do begin
       oall[i]->getProperty,tcontrol = tcontrol
     *self.ypa[i] = tcontrol
     ptr_free,self.yerrpa[i]
     self.yerrpa[i] = ptr_new(/allocate_heap)
     self.ylabel = 'T (K) control'
     endfor
   end
else:
endcase

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::zoomEvents,event = event
if (self.oContainer->count()) eq 0 then return

if ptr_valid(self.px) ne 0 then !x = *self.px
if ptr_valid(self.py) ne 0 then !y = *self.py

case event.type of
0: begin    ; button press
     self.mouse = event.press
     if self.mouse eq 4 then begin
       self.autoscale = 1
     self->refreshScreen,event = event
     endif
     if self.mouse eq 1 then begin
       self.xbox[0] = event.x
       self.ybox[0] = event.y
       wset,self.winVis
       device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.winPix]
       empty
       self.autoscale = 0
       widget_control,self.win,/draw_motion_events
     endif
   end
1: begin ; button release
    if self.mouse eq 1 then begin
     xll = self.xbox[0] < self.xbox[1]
     yll = self.ybox[0] < self.ybox[1]
     w = abs(self.xbox[1] - self.xbox[0])
     h = abs(self.ybox[1] - self.ybox[0])
     xur = xll + w
     yur = yll + h
     ll = convert_coord(xll,yll,/device,/to_data)
     ur = convert_coord(xur,yur,/device,/to_data)
     self.xrange = [ll[0],ur[0]]
     self.yrange = [ll[1],ur[1]]
     self->refreshScreen,event = event
     self.mouse = 0B
     widget_control,self.win,draw_motion_events = 0
    endif
    if self.mouse eq 4 then begin
     self->refreshScreen, event = event
     self.mouse = 0B
     widget_control,self.win,draw_motion_events = 0
    endif
   end
2: begin ; mouse motion
     if self.mouse eq 1 then begin
        self.xbox[1] = event.x
        self.ybox[1] = event.y
        xc = [self.xbox[0],event.x,event.x,$
              self.xbox[0],$
              self.xbox[0]]
        yc = [self.ybox[0],self.ybox[0],$
              event.y,event.y,$
              self.ybox[0]]
        wset,self.winVis
        device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.winPix]
        plots,xc,yc,/device
        empty
     endif
   end
else:
endcase
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::ZoomEventsFit,event = event
if (self.oContainer->count()) eq 0 then return


if ptr_valid(self.pxfit) ne 0 then !x = *self.pxfit
if ptr_valid(self.pyfit) ne 0 then !y = *self.pyfit


case event.type of
0: begin    ; button press
     self.mouse = event.press
     if self.mouse eq 4 then begin
       self.autoscalefit = 1
     self->refreshFitScreen,event = event
     endif
     if self.mouse eq 1 then begin
       self.xbox[0] = event.x
       self.ybox[0] = event.y
       wset,self.fitwinVis
       device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.fitwinPix]
       empty
       self.autoscalefit = 0
       widget_control,self.fitwin,/draw_motion_events
     endif
   end
1: begin ; button release
    if self.mouse eq 1 then begin
     xll = self.xbox[0] < self.xbox[1]
     yll = self.ybox[0] < self.ybox[1]
     w = abs(self.xbox[1] - self.xbox[0])
     h = abs(self.ybox[1] - self.ybox[0])
     xur = xll + w
     yur = yll + h
     ll = convert_coord(xll,yll,/device,/to_data)
     ur = convert_coord(xur,yur,/device,/to_data)
     self.xrangefit = [ll[0],ur[0]]
     self.yrangefit = [ll[1],ur[1]]
     self->refreshFitScreen,event = event
     self.mouse = 0B
     widget_control,self.fitwin,draw_motion_events = 0
    endif
    if self.mouse eq 4 then begin
     self->refreshFitScreen, event = event
     self.mouse = 0B
     widget_control,self.fitwin,draw_motion_events = 0
    endif
   end
2: begin ; mouse motion
     if self.mouse eq 1 then begin
        self.xbox[1] = event.x
        self.ybox[1] = event.y
        xc = [self.xbox[0],event.x,event.x,$
              self.xbox[0],$
              self.xbox[0]]
        yc = [self.ybox[0],self.ybox[0],$
              event.y,event.y,$
              self.ybox[0]]
        wset,self.fitwinVis
        device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.fitwinPix]
        plots,xc,yc,/device
        empty
     endif
   end
else:
endcase
end;fws_analysis::zoomEventsFit
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::drawFits,event = event,ps_out=ps_out

device,get_decomposed=dc
device,decomposed=0
if n_elements(ps_out) eq 0 then ps_out = 0


widget_control,self.tempSlider,get_value = val
val = val[0] - 1
widget_control,self.buffSlider,get_value = buffer
buffer = buffer[0] - 1

oall = self.oContainer->get(/all)
if (self.oContainer->count()) eq 0 then return
oall[buffer]->getProperty,xfit = xfit,yfit = yfit, $
     xdat = xdat,ydat = ydat,yerr = yerr, $
     tcontrol = tcontrol,temperature = temperature

if n_elements(xfit) eq 1 then return

; Scale the axes
xmin = min(xfit)
xmax = max(xfit)
;dx = 0.2*(xmax-xmin)
;xrange = [xmin-dx,xmax+dx]
ymin = min([min(yfit),min(ydat)])
ymax = max([max(yfit),max(ydat)])
;dy = 0.2*(ymax-ymin)
;yrange = [ymin-dy,ymax+dy]



;NEED TO ADD FIT RESCALING HERE.

if self.autoscalefit eq 1 then begin
  ;dx = 0.1*(xmax - xmin)
  dx = 0.2*(xmax-xmin)
  xrange = [xmin-dx,xmax+dx]
  self.xrangefit = xrange
;  dy = 0.2*(ymax - ymin)
;  ymin = ymin < 0.0
;  yrange = [ymin,ymax+dy]

ymin = min([min(yfit),min(ydat)])
ymax = max([max(yfit),max(ydat)])
dy = 0.2*(ymax-ymin)
yrange = [ymin-dy,ymax+dy]

  self.yrangefit = yrange
endif




xfit = xfit[*,val]
yfit = yfit[*,val]
xdat = xdat[*,val]
ydat = ydat[*,val]
yerr = yerr[*,val]

xtitle = '!3!NQ!E2!N'
ytitle = '!3!N-3ln(I(T)/I(T!dmin!n))'
tFormat = '(f10.2)'
title = '!3T='+strtrim(string(temperature[val], $
   format = tFormat),2)+' K'
wset,self.fitWinPix

colors = *self.colorsPtr
ncolors = n_elements(colors)
if ps_out eq 1 then begin
  background = self.white
  first_color = self.black
endif else begin
  background = self.black
  first_color = self.white
endelse


;
;device,get_decomposed=dc
;device,decomposed=1
;plot,xdat,ydat,psym = fws_myPlotSym(5),xrange = xrange, $
;   yrange = yrange,/xsty,/ysty, $
;   xtitle = xtitle,ytitle = ytitle,title = title,charsize = 1.5, $
;   ;color = (2L^32)-1;(*self.colorsPtr)[0]
;   color = first_color, $
;   background = background
plot,xdat,ydat,psym = fws_myPlotSym(5),xrange = self.xrangefit, $
   yrange = self.yrangefit,/xsty,/ysty, $
   xtitle = xtitle,ytitle = ytitle,title = title,charsize = 1.5, $
   ;color = (2L^32)-1;(*self.colorsPtr)[0]
   color = first_color, $
   background = background
errplot,xdat,ydat-yerr,ydat+yerr,width = 0,color=first_color;(2L^32)-1

oplot,xfit,yfit,psym = 0,color=(2L^32)-1
wset,self.fitWinVis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.fitWinPix]
device,decomposed=dc

if ptr_valid(self.pxfit) gt 0 then ptr_free,self.pxfit
if ptr_valid(self.pyfit) gt 0 then ptr_free,self.pyfit
self.pxfit = ptr_new(!x)
self.pyfit = ptr_new(!y)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::changeBuffers,event = event
widget_control,self.buffSlider,get_value = val & val = val[0] - 1
nbuffers = self.oContainer->count()
if (nbuffers-1) lt val then return
oall = self.oContainer->get(/all)
oall[val]->getProperty,file = file
widget_control,self.fitFileField,set_value = file
nchannels = n_elements(*self.xpa[val])
widget_control,self.tempSlider,set_slider_max = nchannels,set_slider_min = 1, $
  set_value = 1

self->drawFits
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::calculate_msd,event = event
if (self.oContainer->count()) eq 0 then return
widget_control,self.plotDetectors,get_value = strdets
; Convert string to array
detectors = opan_selectgroups(strdets)
widget_control,self.depVarGroup,get_value = depVal

nbuffers = self.oContainer->count()
oall = self.oContainer->get(/all)
for i = 0,nbuffers-1 do begin
  oall[i]->getProperty,q = q
  if n_elements(q) ge max(detectors) then begin
    if depVal[0] eq 1 then begin
      result = oall[i]->calc_msd(detectors,/tsample)
;      oall[i]->getProperty,xfit = xfit,yfit = yfit, $
;            xdat = xdat,ydat = ydat,yerr = yerr, $
;            /tsample = temperature
;            *self.xfitPtr[i] = xfit
;            *self.yfitPtr[i] = yfit
;            *self.xdatPtr[i] = xdat
;            *self.ydatPtr[i] = ydat
;            *self.ydatErrPtr[i] = yerr
;            *self.tempPtr[i] = temperature
    endif else begin
      result = oall[i]->calc_msd(detectors,/tcontrol)
;      oall[i]->getProperty,xfit = xfit,yfit = yfit, $
;            xdat = xdat,ydat = ydat,yerr = yerr, $
;            tcontrol = temperature
;            *self.xfitPtr[i] = xfit
;            *self.yfitPtr[i] = yfit
;            *self.xdatPtr[i] = xdatPtr
;            *self.ydatPtr[i] = ydatPtr
;            *self.ydatErrPtr[i] = yerr
;            *self.tempPtr[i] = temperature
    endelse
  endif
endfor
self->selVar, event = event
self->changeBuffers
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::plotDetGroup,event = event
if (self.oContainer->count()) eq 0 then return
nbuffers = self.oContainer->count()
widget_control,self.detSlider,get_value = index
oall = self.oContainer->get(/all)

for i = 0,nbuffers-1 do begin
   oall[i]->getProperty,intensity = intensity, error = error
   *self.ypa[i] = intensity[index[0],*]
   *self.yerrpa[i] = error[index[0],*]
endfor
self.ylabel = 'Intensity (arb units)'
self->refreshScreen, event = event
self->changeBuffers
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;LRK - 01/26/09

pro fws_analysis::sendFitDataToPAN,event=event,_Extra=extra

    ;WANT TO MAKE f(I) v Q^2 DATA AVAILABLE FOR ALTERNATE FITTING METHODS
    ;FOR CASES WHEN LINEAR FITS ARE INADEQUATE.

;    widget_control,self.tempSlider,get_value = val
;    val = val[0] - 1


    if event.id eq self.sendTransposedDataToPAN then transposeData = 1 else transposedata = 0
    
    widget_control,self.buffSlider,get_value = buffer
    buffer = buffer[0] - 1

    oall = self.oContainer->get(/all)
    if (self.oContainer->count()) eq 0 then return

    oall[buffer]->getProperty,$;xfit = xfit,yfit = yfit, $
         xdat = xdat,ydat = ydat,yerr = yerr,temperature = temperature,$
         tcontrol = tcontrol
    
;    help,xdat,ydat,yerr,temperature
;    print,xdat[0,*];-xdat[2,*]
;    qty = transpose(ydat)
;    err = transpose(yerr)
;    x = xdat[*,0]
;    y = temperature
;    


  if transposeData eq 0 then begin
    qty = ydat
    err = yerr
    yvals = temperature;xdat[*,0]
    xvals = xdat[*,0];temperature    
;    help,x,y,qty,err
;    print,xdat[*,1]-xdat[*,0]    
    xlabel = '!3!NQ!E2!N'
    ylabel = 'T'
    ztitle = '!3!N-3ln(I(T)/I(T!dmin!n))'
    yunits ='K'
    xunits = 'Ang^-2'
  endif else begin
    qty = transpose(ydat)
    err = transpose(yerr)
    xvals = temperature;xdat[*,0]
    yvals = xdat[*,0];temperature    
;    help,x,y,qty,err
;    print,xdat[*,1]-xdat[*,0]    
    ylabel = '!3!NQ!E2!N'
    xlabel = 'T'
    ztitle = '!3!N-3ln(I(T)/I(T!dmin!n))'
    xunits ='K'
    yunits = 'Ang^-2'    
  endelse


;    tFormat = '(f10.2)'
;    title = '!3T='+strtrim(string(temperature[val], $
;       format = tFormat),2)+' K'

;help,x,y,qty,err
     retval = CREATE_DAVE_POINTER(  $
                   DAVEPTR ,         $
                   INSTRUMENT = 'FWS_ANALYSIS',  $
                   QTY = qty,          $
                   ERR = err,          $
                   XVALS = xvals,        $
                   XLABEL = xlabel,      $
                   YVALS = yvals,$        $
                   YLABEL = ylabel,$
                   yunits=yunits,$
                   xunits=xunits,$
                   qtlabel=ztitle)

;  ret_val = CREATE_DAVE_POINTER(  $
;               DAVEPTR ,         $
;               INSTRUMENT = instrument,  $
;               QTY = qty,          $
;               QTUNITS = qtunits,      $
;               QTLABEL = qtlabel,      $
;               ERR = err,          $
;               XVALS = xvals,        $
;               XTYPE = xtype,        $
;               XUNITS = xunits,      $
;               XLABEL = xlabel,      $
;               YVALS = yvals,        $
;               YTYPE = ytype,        $
;               YLABEL = ylabel,      $
;               YUNITS = yunits,      $
;               SPECIFICSTR = specificstr,  $
;               TREATMENT = treatment,    $
;               DNAME = dname,        $
;               DUNITS = dunits,      $
;               DLEGEND = dlegend,      $
;               DQTY = dqty,        $
;               DERR = derr,        $
;               ERMSG = errmsg        )
    if retval ne 0 then begin
      oall->getproperty,path=datadir,file=filename
      ;cd,current=cwd
      ;print,cwd
      ;if file_test(datadir) ne 
    
      t = getDaveTool()      
      if obj_valid(t) ne 0 then begin
        t->getProperty,working_directory=workdir
      endif
      ;msg = dialog_message(['Sending data to PAN:','datadir='+datadir,'workdir='+workdir],dialog_parent=self.tlb,title='Info:')
      p = obj_new('opan',group_leader = self.tlb, $
                         davePtr=davePtr,$
                         datadir=datadir,$
                         workdir=workdir,$
                         ;ytitle=ylabel,$
                         title='PAN: Fixed Window Scan Analysis  ---   '+filename,$
                         wintitle='PAN: Fixed Window Scan Analysis  ---   '+filename)

      ;IS davePtr NOW UNUSED? - Clean up.
      if ptr_valid(davePtr) gt 0 then ptr_free,davePtr
    endif else begin
      void = dialog_message('Unable to send data to PAN.',dialog_parent=self.tlb)
    endelse

end;fws_analysis::sendFitDataToPAN
pro fws_analysis::sendFitDataToExcelASCII,event=event,_Extra=extra

    ;WANT TO MAKE f(I) v Q^2 DATA AVAILABLE FOR ALTERNATE FITTING METHODS
    ;FOR CASES WHEN LINEAR FITS ARE INADEQUATE.

;HERE I WANT TO OUTPUT THE DATA IN COLUMNS WITH Y VALUES INDICATING THE ROWS
;RATHER THAN THE COLUMNS.

;THE PROBLEMS WAS THAT WITH TOO MANY GROUPS, IDL SEEMS TO MEET LINE
;LENGTH LIMITATIONS.


;    widget_control,self.tempSlider,get_value = val
;    val = val[0] - 1
    widget_control,self.buffSlider,get_value = buffer
    buffer = buffer[0] - 1

    oall = self.oContainer->get(/all)
    if (self.oContainer->count()) eq 0 then return

    oall[buffer]->getProperty,$;xfit = xfit,yfit = yfit, $
         xdat = xdat,ydat = ydat,yerr = yerr,temperature = temperature,$
         tcontrol = tcontrol
    
;    help,xdat,ydat,yerr,temperature
;    print,xdat[0,*];-xdat[2,*]
;    qty = transpose(ydat)
;    err = transpose(yerr)
;    x = xdat[*,0]
;    y = temperature
;    

    qty = ydat
    err = yerr
    y = temperature;xdat[*,0]
    x = xdat[*,0];temperature

    
;    help,x,y,qty,err
;    print,xdat[*,1]-xdat[*,0]

    
    xtitle = '!3!NQ!E2!N'
    ytitle = '!3!N-3ln(I(T)/I(T!dmin!n))'
;    tFormat = '(f10.2)'
;    title = '!3T='+strtrim(string(temperature[val], $
;       format = tFormat),2)+' K'

;help,x,y,qty,err
     retval = CREATE_DAVE_POINTER(  $
                   DAVEPTR ,         $
                   INSTRUMENT = 'FWS_ANALYSIS',  $
                   QTY = qty,          $
                   ERR = err,          $
                   XVALS = x,        $
                   XLABEL = xtitle,      $
                   YVALS = yvals,        $
                   YLABEL = ytitle)


    if retval ne 0 then begin
      p = obj_new('opan',group_leader = self.tlb, $
                         davePtr=davePtr)
      ;IS davePtr NOW UNUSED? - Clean up.
      if ptr_valid(davePtr) gt 0 then ptr_free,davePtr
    endif else begin
      void = dialog_message('Unable to send data to PAN.',dialog_parent=self.tlb)
    endelse

end;fws_analysis::sendFitDataToExcelASCII

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::write_data_to_text,event = event, _Extra = extra
compile_opt idl2,hidden
if (self.oContainer->count()) eq 0 then return
nbuffers = self.oContainer->count()

filename = dialog_pickfile(dialog_parent = event.top,/write, $
   title = 'Enter output filename', path = self.workDir)
if (filename eq '') or (filename eq ' ') then return
if (strpos(strupcase(filename),'.TXT'))[0] eq -1 then $
   filename = filename+'.txt'


if self.lines eq 0B then prefactor = 1 else prefactor = -1

widget_control,self.plotBuffers,get_value = strdets
if strdets[0] eq '1-1' then begin
  detectors = 0
endif else begin
  detectors = (opan_selectgroups(strdets))-1
endelse
ndets = n_elements(detectors)

if max(detectors) ge nbuffers then return

format = '(3f15.5)'
openw,lun,filename,/get_lun

for i = 0,ndets-1 do begin
   x = *self.xpa[detectors[i]]
   y = *self.ypa[detectors[i]]
   dy = *self.yerrpa[detectors[i]]
   nx = n_elements(x)
   label = strtrim(string(detectors[i]),2)
   printf,lun,'*****Group: '+label+'*****'
   for j = 0,nx-1 do $
      printf,lun,x[j],y[j],dy[j],format = format
   printf,lun,'**************************'
endfor
free_lun,lun,/force
void = dialog_message(['Output written to: ',filename],dialog_parent=event.top)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::draw,event = event,ps_out = ps_out,_Extra = extra
compile_opt idl2,hidden
device,get_decomposed=dc
device,decomposed=0
if n_elements(ps_out) eq 0 then ps_out = 0
if (self.oContainer->count()) eq 0 then return
nbuffers = self.oContainer->count()

if self.lines eq 0B then prefactor = 1 else prefactor = -1

widget_control,self.plotBuffers,get_value = strdets
if strdets[0] eq '1-1' then begin
  detectors = 0
endif else begin
  detectors = (opan_selectgroups(strdets))-1
endelse
ndets = n_elements(detectors)

if max(detectors) ge nbuffers then return
colors = *self.colorsPtr
ncolors = n_elements(colors)
if ps_out eq 1 then begin
  background = self.white
  first_color = self.black
endif else begin
  background = self.black
  first_color = self.white
endelse
xmin = (ymin = 1.e10)
xmax = (ymax = -1.e10)

for i = 0,ndets-1 do begin
  xmin = xmin < min(*self.xpa[detectors[i]])
  xmax = xmax > max(*self.xpa[detectors[i]])
  if n_elements(*self.ypa[detectors[i]]) ne 0 then begin
    if n_elements(*self.yerrpa[detectors[i]]) ne 0 then begin
      ymin = ymin < min(*self.ypa[detectors[i]]+*self.yerrpa[detectors[i]])
      ymax = ymax > max(*self.ypa[detectors[i]]+*self.yerrpa[detectors[i]])
    endif else begin
      ymin = ymin < min(*self.ypa[detectors[i]])
      ymax = ymax > max(*self.ypa[detectors[i]])
    endelse
  endif
endfor

if self.autoscale eq 1 then begin
  dx = 0.1*(xmax - xmin)
  self.xrange = [xmin-dx,xmax+dx]
  dy = 0.2*(ymax - ymin)
  ymin = ymin < 0.0
  self.yrange = [ymin,ymax+dy]
endif

for i = 0,ndets-1 do begin
  if n_elements(*self.ypa[detectors[i]]) ne 0 then begin
  ;thisColor = colors[i mod ncolors]
  if (ps_out and (i eq 0)) then thisColor = self.black else thisColor = colors[i mod ncolors]
    if i eq 0 then begin
      plot,*self.xpa[detectors[i]],*self.ypa[detectors[i]], $
         psym = prefactor*fws_myPlotSym(i),xrange = self.xrange, xstyle = 1, $
         yrange = self.yrange,ystyle = 1,xtitle = self.xlabel, $
         ytitle = self.ylabel,_Extra = extra,color = first_color, $
         background = background
    endif else begin
      oplot,*self.xpa[detectors[i]],*self.ypa[detectors[i]], $
         psym = prefactor*fws_myPlotSym(i), color = colors[i mod ncolors], $
         _Extra = extra
    endelse
   if (n_elements(*self.yerrpa[detectors[i]]) ne 0) and $
     self.error_bars then $
      hfbs_cerrplot,*self.xpa[detectors[i]], $
              *self.ypa[detectors[i]]-(*self.yerrpa[detectors[i]]), $
              *self.ypa[detectors[i]]+(*self.yerrpa[detectors[i]]), $
              width = 0.0,errcolors = {thisColor:thisColor}
  endif
endfor
device,decomposed=dc

if ptr_valid(self.px) gt 0 then ptr_free,self.px
if ptr_valid(self.py) gt 0 then ptr_free,self.py
self.px = ptr_new(!x)
self.py = ptr_new(!y)
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::refreshScreen,event = event
wset,self.winPix
self->draw,event = event,ps_out = 0
wset,self.winVis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.winPix]
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::refreshFitScreen,event = event
wset,self.fitwinPix
self->drawFits,event = event,ps_out = 0
wset,self.fitwinVis
device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.fitwinPix]
end;fws_analysis::refreshFitScreen
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::selVar,event = event
compile_opt idl2,hidden
if (self.oContainer->count()) eq 0 then return
self->getDepVar
self->getIndepVar
self->refreshScreen, event = event
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::setMonNorm,event = event
compile_opt idl2,hidden
widget_control,self.monNorm,get_value = val
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::displayFiles, event = event
compile_opt idl2,hidden
nfiles = self.oContainer->count()
if nfiles lt 1 then begin
  widget_control,self.fileBox,set_value = ''
  return
endif
oall = self.oContainer->get(/all)
fout = strarr(nfiles)
for i = 0,nfiles-1 do begin
  oall[i]->getProperty,file = file
  fout[i] = '#'+strtrim(string(i+1),2)+':  '+file
endfor
widget_control,self.fileBox,set_value = fout
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::clearBuffers,event = event
obj_destroy,self.oContainer
self.oContainer = obj_new('IDL_CONTAINER')
self->displayFiles
wset,self.winVis

nleg = self.legContainer->count()
if nleg eq 1 then begin
  oall = self.legContainer->get(/all)
  oall[0]->dismiss
endif

erase
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function fws_analysis::whichLine,fileString,pos,length
nstrings = n_elements(fileString)
thisLength = strlen(fileString)
for i = 0,nstrings-1 do begin
  if i eq 0 then begin
    start = 0
    finish = thisLength[i]
  endif else begin
    start = [start,finish[i-1]+2]
    finish = [finish,start[i]+thisLength[i]]
  endelse
endfor

for i = 0,nstrings-1 do begin
  if ((pos ge start[i]) and (pos le finish[i])) then $
    thisLine = i
endfor

return,thisLine
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::textMethod,event = event
name = tag_names(event,/structure_name)

; Thu Oct  5 12:11:18 EDT 2006
; Goran.Gasparovic@nist.gov
; Ref Begin gasparovic-061005.01
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'fws_analysis__define.pro:  Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,'Error reference: fws_analysis__define.pro, code 633.']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif
; End gasparovic-061005.01


if name eq 'WIDGET_TEXT_SEL' then begin
  widget_control,self.fileBox,get_value = fileString
  nstrings = n_elements(fileString)
  thisLen = strlen(fileString)
  start = event.offset
; Wed Oct  4 20:52:19 EDT 2006
; Goran.Gasparovic@nist.gov
; DAVE used to crash here if 'self->whichline(..)' not defined.  Fixed.
; Namely, if one clicked into an empty line in 'Plot detector group' window.
; Fixed.  Tested.  Works.
; Ref Begin gasparovic-061004.02
  if n_elements(self->whichLine(fileString,start,thisLen)) ne 0 then begin
	  rembuffer = self->whichLine(fileString,start,thisLen)
	  self.selBuffer = remBuffer
  end else begin
  	display_message='Automobile crash tests done here.'
  end
; End gasparovic-061004.02
endif

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::remSelBuffer,event = event
if self.selBuffer ne -1 then begin
  buffer = self.selBuffer
  self.selBuffer = -1
  oall = self.oContainer->get(/all)
  ptr_free,self.tempPtr[buffer]
  self.tempPtr[buffer] = ptr_new(/allocate_heap)
  ptr_free,self.xpa[buffer]
  self.xpa[buffer] = ptr_new(/allocate_heap)
  ptr_free,self.ypa[buffer]
  self.ypa[buffer] = ptr_new(/allocate_heap)
  ptr_free,self.yerrpa[buffer]
  self.yerrpa[buffer] = ptr_new(/allocate_heap)

  ptr_free,self.xfitPtr[buffer],self.yfitPtr[buffer],self.xdatPtr[buffer]
  ptr_free,self.ydatPtr[buffer],self.ydatErrPtr[buffer],self.tempPtr[buffer]
  self.tempPtr[buffer] = ptr_new(/allocate_heap)
  self.xfitPtr[buffer] = ptr_new(/allocate_heap)
  self.yfitPtr[buffer] = ptr_new(/allocate_heap)
  self.xdatPtr[buffer] = ptr_new(/allocate_heap)
  self.ydatPtr[buffer] = ptr_new(/allocate_heap)
  self.ydatErrPtr[buffer] = ptr_new(/allocate_heap)

  self.oContainer->remove,oall[buffer]
  obj_destroy,oall[buffer]
  nfiles = self.oContainer->count()
  if nfiles ge 1 then begin
    strout = '1-'+strtrim(string(nfiles),2)
   widget_control,self.buffSlider,set_slider_min = 1, $
      set_slider_max = (nfiles > 2),set_value = 1
    self->changeBuffers
  endif else begin
    strout = '1-1'
   widget_control,self.buffSlider,set_slider_min = 1, $
      set_slider_max = (nfiles > 2),set_value = 1
    wset,self.winVis
    erase
  endelse
  widget_control,self.plotBuffers,set_value = strout
  self->displayFiles,event = event
  self->selVar,event = event
  nleg = self.legContainer->count()
  if nleg eq 1 then begin
    oall = self.legContainer->get(/all)
    oall[0]->displayLegend
  endif
endif
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::loadHSCNFile,event = event
compile_opt idl2,hidden
files = dialog_pickfile(dialog_parent = self.tlb,title = 'Select file(s)', $
     filter = '*.hscn',path = self.datDir,get_path = outPath, $
     /multiple_files)
if files[0] eq '' then return
nfiles = n_elements(files)
thisLen = strlen(outPath)
widget_control,self.monNorm,get_value = normVal
for i = 0,nfiles-1 do begin
  file = strmid(files[i],thisLen)
  ; Instantiate a scan object
  o = obj_new('hfbs_scan',daveFile = file,davePath = outPath,norm2Mon = normVal[0])
  ; Invoke the read method
  o->readhscnfile
  self.oContainer->add,o
endfor
self->displayFiles
self.datDir = outPath
widget_control,self.plotDetectors,set_value = '1-16'
widget_control,self.detSlider,set_slider_min = 1,set_slider_max = 16
self->selVar, event = event

nfiles = self.oContainer->count()
strout = '1-'+strtrim(string(nfiles),2)
widget_control,self.plotBuffers,set_value = strout

widget_control,self.buffSlider,set_slider_min = 1, $
  set_slider_max = (nfiles > 2),set_value = 1

nleg = self.legContainer->count()
if nleg eq 1 then begin
  oall = self.legContainer->get(/all)
  oall[0] -> displayLegend
endif

end;fws_analysis::loadHSCNFile
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::loadSNSHSCNFile,event = event
compile_opt idl2,hidden

;LRK - 02/17/11
;THIS IS BEING ADDED TO ALLOW SNS Users TO READ AN HSCN FILE THAT HAS 
;UNCERTAINTIES IN THE FILE.

files = dialog_pickfile(dialog_parent = self.tlb,title = 'Select file(s)', $
     filter = '*.hscn',path = self.datDir,get_path = outPath, $
     /multiple_files)
if files[0] eq '' then return
nfiles = n_elements(files)
thisLen = strlen(outPath)
widget_control,self.monNorm,get_value = normVal
for i = 0,nfiles-1 do begin
  file = strmid(files[i],thisLen)
  ; Instantiate a scan object
  o = obj_new('hfbs_scan',daveFile = file,davePath = outPath,norm2Mon = normVal[0])
  ; Invoke the read method
  o->readhscnfile,/sns
  self.oContainer->add,o
endfor
self->displayFiles
self.datDir = outPath
widget_control,self.plotDetectors,set_value = '1-16'
widget_control,self.detSlider,set_slider_min = 1,set_slider_max = 16
self->selVar, event = event

nfiles = self.oContainer->count()
strout = '1-'+strtrim(string(nfiles),2)
widget_control,self.plotBuffers,set_value = strout

widget_control,self.buffSlider,set_slider_min = 1, $
  set_slider_max = (nfiles > 2),set_value = 1

nleg = self.legContainer->count()
if nleg eq 1 then begin
  oall = self.legContainer->get(/all)
  oall[0] -> displayLegend
endif

end;fws_analysis::loadSNSHSCNFile
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::loadDAVEFile,event = event
compile_opt idl2,hidden
files = dialog_pickfile(dialog_parent = self.tlb,title = 'Select file(s)', $
     filter = '*.dave',path = self.datDir,get_path = outPath, $
     /multiple_files)
if files[0] eq '' then return
nfiles = n_elements(files)
thisLen = strlen(outPath)
widget_control,self.monNorm,get_value = normVal
for i = 0,nfiles-1 do begin
  file = strmid(files[i],thisLen)
  ; Instantiate a scan object
  o = obj_new('hfbs_scan',daveFile = file,davePath = outPath,norm2Mon = normVal[0])
  ; Invoke the read method
  o->readdavefile
  self.oContainer->add,o
endfor
self->displayFiles
self.datDir = outPath
widget_control,self.plotDetectors,set_value = '1-16'
widget_control,self.detSlider,set_slider_min = 1,set_slider_max = 16
self->selVar, event = event

nfiles = self.oContainer->count()
strout = '1-'+strtrim(string(nfiles),2)
widget_control,self.plotBuffers,set_value = strout

widget_control,self.buffSlider,set_slider_min = 1, $
  set_slider_max = (nfiles > 2),set_value = 1

nleg = self.legContainer->count()
if nleg eq 1 then begin
  oall = self.legContainer->get(/all)
  oall[0] -> displayLegend
endif

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PRO eventHandler, event
compile_opt idl2,hidden


;begin error handler------------------------------------------------------------
; RTA - handler to catch misc. errors and provide a graceful exit so
;       that main app does not crash. Remember to switch off debug flag when debugging!
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif
;end error handler-------------------------------------------------------------


if dave_set_focus(event) then return
; This is the only event handler necessary for the program.  It makes a
; generic method call for an object, the informating begin provided
; by the UVALUE of the calling widgets.
;
; Check for resize events
thisEvent = tag_names(event,/structure_name)
case thisEvent of
'WIDGET_BASE': $
     begin
      widget_control,event.top,get_uvalue = self
      self->resize,event = event
      return
     end
'WIDGET_TAB':  return
else:
endcase

; Handle the methods
Widget_Control, event.id, Get_UValue=cmd
call_method, cmd.method,cmd.object,event = event
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::displayAll, event = event,ps_out = ps_out
nbuffers = self.oContainer -> count()
widget_control,self.detSlider,get_value = index
oall = self.oContainer->get(/all)
symsize = 0.25
self.ylabel = 'Intensity (arb units)'
!p.multi = [0,4,5]
for j = 1,16 do begin
  index = j
  for i = 0,nbuffers-1 do begin
   oall[i]->getProperty,intensity = intensity, error = error
   *self.ypa[i] = intensity[index[0],*]
   *self.yerrpa[i] = error[index[0],*]
  endfor
  self->draw,event = event,title = 'Group: '+strtrim(string(j),2),ps_out = ps_out,symsize = symsize
endfor

; Plot the WBM
widget_control,self.indepVarGroup,set_value = 2
self->getIndepVar
self->draw,event = event,title = 'White Beam Monitor',ps_out = ps_out,symsize = symsize

; Plot the IBM
widget_control,self.indepVarGroup,set_value = 3
self->getIndepVar
self->draw,event = event,title = 'Incident Beam Monitor',ps_out = ps_out,symsize = symsize

; Plot the TBM
widget_control,self.indepVarGroup,set_value = 4
self->getIndepVar
self->draw,event = event,title = 'Transmitted Beam Monitor',ps_out = ps_out,symsize = symsize

; Plot the sample temperature
widget_control,self.depVarGroup,set_value = 0
widget_control,self.indepVarGroup,set_value = 7
self->getIndepVar
self->getDepVar
self->draw,event = event,title = 'T!dsample!n (K)',ps_out = ps_out,symsize = symsize

!p.multi = 0
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::printAll2PS, event = event
nfiles = self.oContainer -> count()
if nfiles eq 0 then return

myFile = 'output.ps'
keywords = PSConfig(Cancel=cancelled,group_leader = event.top,$
           filename = myFile,color = 1,xsize = 7.5,ysize = 10.0, $
           directory = self.workDir,xoff = 0.5,yoff = 0.5)
IF cancelled THEN RETURN
thisDevice = !D.Name
Set_Plot, 'PS'
Device, _Extra=keywords
self->displayAll, event = event,ps_out = 1
Device, /Close_File
Set_Plot, thisDevice
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::print2PS, event = event
nfiles = self.oContainer -> count()
if nfiles eq 0 then return

myFile = 'output.ps';filename+'.ps'

keywords = PSConfig(Cancel=cancelled,group_leader = event.top,$
           filename = myFile,color = 1, $;xsize = 8.0,ysize = 11.0, $
           directory = self.workDir);,xoff = 0.0,yoff = 0.0)
IF cancelled THEN RETURN
thisDevice = !D.Name
Set_Plot, 'PS'
Device, _Extra=keywords

self->draw,event = event,ps_out = 1

Device, /Close_File
Set_Plot, thisDevice
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::calcDerivative,event = event
nbuffers = self.oContainer->count()
if nbuffers eq 0 then return
for i = 0,nbuffers-1 do begin
  x = *self.xpa[i]
  y = *self.ypa[i]
  if n_elements(*self.yerrpa[i]) ne 0 then begin
    yerr = *self.yerrpa[i]
    yed = derivsig(x,y,0.0,yerr)
    *self.yerrpa[i] = yed
  endif
  yd = deriv(x,y)
  *self.ypa[i] = yd
endfor
self->refreshScreen, event = event
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::emptyLegend, event = event
self.legContainer->remove,/all
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::writeMSD, event = event
if self.oContainer->count() eq 0 then return
nbuffers = self.oContainer->count()
oall = self.oContainer->get(/all)
self->getDepVar
for i = 0,nbuffers-1 do begin
   oall[i]->getProperty,usq = usq, uerr = uerr,file = file
   if usq[0] ne -999 then begin
   ; Ok, save the data
   ; What is the filename?
   ppos = strpos(file,'.')
   filename = strmid(file,0,ppos)+'msd.txt'
    totFile = self.workDir+filename
    t = *self.xpa[i]
    nt = n_elements(t)
    tsort = sort(t)
    t = t[tsort]
    usq = usq[tsort]
    uerr = uerr[tsort]
    openw,lun,totFile,/get_lun
    for j = 0,nt-1 do begin
      printf,lun,t[j],usq[j],uerr[j],format = '(3g15.5)'
   endfor
   free_lun,lun
   void = dialog_message(['Output written to: ',totFile],dialog_parent=event.top)
   endif
endfor

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::showLegend, event = event
widget_control,event.top,get_uvalue = object
nLegends = self.legContainer->count()
if nLegends gt 0 then begin
  ; Bring the current legend to the foreground
  oall = self.legContainer->get(/all)
  oall[0]->bringToFront
  return
endif
nbuffers = self.oContainer->count()
if nbuffers eq 0 then return

o = obj_new('fwsLegend',object,group_leader = event.top)
self.legContainer->add,o
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::connectPoints, event = event
if self.lines eq 1B then self.lines = 0B else self.lines = 1B
if (!version).release ge 5.6 then begin
   widget_control,event.id,set_button = self.lines
endif
self->refreshScreen,event = event
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::toggle_errors, event = event
if self.error_bars eq 1B then self.error_bars = 0B else self.error_bars = 1B
if (!version).release ge 5.6 then begin
   widget_control,event.id,set_button = self.error_bars
endif
self->refreshScreen,event = event

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::createWidgets
compile_opt idl2,hidden
; Widget construction module
if self.group_leader eq 0L then begin
  self.tlb = widget_base(/row,title = 'Fixed Window Scan Analysis', $
        /tlb_size_events,mbar = bar)
endif else begin
  self.tlb = widget_base(/row,title = 'Fixed Window Scan Analysis',mbar = bar, $
  group_leader = self.group_leader,/tlb_size_events)
endelse

button_labels = ["Main","Fits"]

if (!version).release ge 5.6 then begin
  tab_base = widget_tab(self.tlb)
  nlabels = n_elements(button_labels)
  tabit = lonarr(nlabels)
  for i = 0,nlabels-1 do begin
    tabit[i] = widget_base(tab_base,title = button_labels[i],/col)
  endfor
endif else begin
   tab_base = cw_tabbase(self.tlb,button_labels,bases = tabit,/col,/no_label,/frame)
endelse
tab = lonarr(2)
tab[0] = widget_base(tabit[0],/row)
tab[1] = widget_base(tabit[1],/row)

filemenu = widget_button(bar,value = 'File',/menu)
printmenu = widget_button(bar,value = 'Print',/menu)
plotmenu = widget_button(bar,value = 'Plot',/menu)
if (!version).release lt 5.6 then begin
   void = widget_button(plotMenu,value = 'Toggle connecting lines between points', $
      uvalue = {object:self,method:'connectPoints'},uname = 'CONNECT')
   void = widget_button(plotMenu,value = 'Toggle error bars', $
      uvalue = {object:self,method:'toggle_errors'},uname = 'TOGGLE')
endif else begin
   void = widget_button(plotMenu,value = 'Toggle connecting lines between points', $
      uvalue = {object:self,method:'connectPoints'}, $
      /checked_menu,uname = 'CONNECT')
   void = widget_button(plotMenu,value = 'Toggle error bars', $
      uvalue = {object:self,method:'toggle_errors'},/checked_menu, $
      uname = 'TOGGLE')
endelse

void = widget_button(printmenu,value = 'Print to PS file', $
     uvalue = {object:self,method:'print2PS'})
void = widget_button(printmenu,value = 'Print ALL to PS file', $
     uvalue = {object:self,method:'printAll2PS'})
void = widget_button(filemenu,value = 'Read HSCN file', $
     uvalue = {object:self,method:'loadHSCNFile'})
;void = widget_button(filemenu,value = 'Read SNS HSCN file', $
;     uvalue = {object:self,method:'loadSNSHSCNFile'})
void = widget_button(filemenu,value = 'Read reduced DAVE HSCN file', $
     uvalue = {object:self,method:'loadDAVEFile'})
void = widget_button(filemenu,value = 'Clear file buffers', $
     uvalue = {object:self,method:'clearBuffers'})
void = widget_button(filemenu,value = 'Write MSD to file', $
     uvalue = {object:self,method:'writeMSD'})
void = widget_button(filemenu,value = 'Write detector intensities to text file', $
     uvalue = {object:self,method:'write_data_to_text'})
void = widget_button(filemenu,value = 'Quit', $
     uvalue = {object:self,method:'quit'})

self.colBase1 = widget_base(tab[0],/col,/base_align_left,/base_align_top)

self.plotBuffers = cw_field(self.colBase1,/row,title = 'Plot buffers', $
         value = '1',xsize = 10, /return_events, $
           uvalue = {object:self,method:'selVar'})
self.plotDetectors = cw_field(self.colBase1,/row,title = 'Plot detector sum', $
           value = '2-14',xsize = 10, /return_events, $
           uvalue = {object:self,method:'selVar'})
void = widget_button(self.colBase1,value = 'Calculate MSD', $
     uvalue = {object:self,method:'calculate_msd'})
self.detSlider = widget_slider(self.colBase1,title = 'Plot detector group', $
     value = 1,maximum = 16,minimum = 1, $
     uvalue = {object:self,method:'plotDetGroup'})
self.fileBox = widget_text(self.colBase1,value = '',/scroll, $
         /all_events,xsize = self.xtextSize,ysize = 10, $
         uvalue = {object:self,method:'textMethod'})
void = widget_button(self.colBase1,value = 'Remove selected buffer', $
     uvalue = {object:self,method:'remSelBuffer'})
void = widget_button(self.colBase1,value = 'Derivative', $
     uvalue = {object:self,method:'calcDerivative'})
void = widget_button(self.colBase1,value = 'Legend', $
     uvalue = {object:self,method:'showLegend'})
;self.colBase2 = widget_base(self.tlb,/col,/base_align_left,/base_align_top)
self.colBase2 = widget_base(tab[0],/col,/base_align_left,/base_align_top)
mon = ['Normalize to monitor']
self.monNorm = cw_bgroup(self.colBase2, mon, /col, /nonexclusive,$
         label_top='Monitor Normalization',/no_release,$
         frame=1,set_value=1,/return_index, $
         uvalue = {object:self,method:'setMonNorm'})
indepVar = ['I(Q,T)','MSD','WBM','IBM','TBM','TBM/IBM', $
            'time (min)','T (K) sample', 'T (K) control'];, 'Fits']
self.indepVarGroup = cw_bgroup(self.colBase2, indepVar, /col, /exclusive,$
         label_top='Dependent variable',/no_release,$
         frame=1,set_value=0,/return_index, $
         uvalue = {object:self,method:'selVar'})
depVar = ['time (min)','T (K) sample', 'T (K) control'];, 'Q (A-1)']
self.depVarGroup = cw_bgroup(self.colBase2, depVar, /col, /exclusive,$
         label_top='Independent variable',/no_release,$
         frame=1,set_value=1,/return_index, $
         uvalue = {object:self,method:'selVar'})

;winBase = widget_base(self.tlb,/col)
winBase = widget_base(tab[0],/col)
winxsize = 500 & winysize = 400
self.win = widget_draw(winBase,xsize = winxsize, ysize = winysize, $
           /button_events,uvalue = {object:self,method:'zoomEvents'})

; Now put a new window in the 'FITS' tab
self.fitWin = widget_draw(tab[1],xsize = winxsize,ysize = winysize,$
                          /button_events,uvalue = {object:self,method:'zoomEventsFit'})
fitBase = widget_base(tab[1],/col)
self.tempSlider = widget_slider(fitBase,maximum = 100,minimum = 1,value = 1, $
      title = 'Temperature index', $
      uvalue = {object:self,method:'drawFits'})
self.buffSlider = widget_slider(fitBase,maximum = 100,minimum = 1,value = 1, $
      title = 'File Buffer', $
      uvalue = {object:self,method:'changeBuffers'})
self.fitFileField = cw_field(fitBase,/row,title = 'Scan file', $
         value = '',xsize = 25)

;LRK - 01/26/09
self.sendDataToPAN = widget_button(fitBase,value = 'Send Fit Data to PAN', $
                                   uvalue = {object:self,method:'sendFitDataToPAN'})
self.sendTransposedDataToPAN = widget_button(fitBase,value = 'Send Transposed Fit Data to PAN', $
                                   uvalue = {object:self,method:'sendFitDataToPAN'})

;self.sendDataToTransposedASCII = widget_button(fitBase,value = 'Send Fit Data to EXCEL Readable ASCII', $
;                                   uvalue = {object:self,method:'sendFitDataToExcelASCII'})


centertlb,self.tlb

widget_control,self.tlb,/realize

toggle_id = widget_info(self.tlb,find_by_uname = 'TOGGLE')
connect_id = widget_info(self.tlb,find_by_uname = 'CONNECT')
if (!version).release ge 5.6 then begin
  widget_control,toggle_id,set_button = 1
endif

widget_control,self.win,get_value = winVis
self.winVis = winVis
window,/free,/pixmap,xsize = winxsize, ysize = winysize
self.winPix = !d.window

widget_control,self.fitWin,get_value = fitWinVis
self.fitWinVis = fitWinVis
window,/free,/pixmap,xsize = winxsize, ysize = winysize
self.fitWinPix = !d.window

widget_control,self.tlb,set_uvalue = self
ret = dave_set_focus(self.tlb)
xmanager,'fws_analysis::createWidgets',self.tlb,/no_block, $
     event_handler = 'eventHandler', cleanup = 'fws_analysisCleanup'

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis::getProperty,   $
           nbuffers = nbuffers, $
           filenames = filenames

nbuffers = self.oContainer->count()
if nbuffers gt 0 then begin
  oall = self.oContainer->get(/all)
  fout = strarr(nbuffers)
  for i = 0,nbuffers-1 do begin
    oall[i]->getProperty,file = file
    if i eq 0 then begin
      filenames = file
    endif else begin
      filenames = [filenames,file]
    endelse
  endfor
endif else begin
  filenames = ''
endelse
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function fws_analysis::init,  group_leader = group_leader, $
                 datDir = datDir, $
                 workDir = workDir
compile_opt idl2,hidden
tvlct,r,g,b,/get
self.r = r
self.g = g
self.b = b

device,decomposed = 0
colors = hfbs_GetColor(/Load, Start=1)
self.colorsPtr = ptr_new(/allocate_heap)
in_colors = [  colors.white, colors.green, $
      colors.red, $
      colors.blue, colors.yellow, $
      colors.blue, colors.beige, $
      colors.sky, colors.orchid, $
      colors.aqua, colors.pink, $
      colors.cyan ]
*self.colorsPtr = in_colors
self.white = colors.white
self.black = colors.black

if n_elements(group_leader) eq 0 then group_leader = 0L
cd,current = thisDir
if n_elements(datDir) eq 0 then datDir = thisDir
if n_elements(workDir) eq 0 then workDir = thisDir
self.oContainer = obj_new('IDL_CONTAINER')
self.legContainer = obj_new('IDL_CONTAINER')
self.xlabel = ''
self.ylabel = ''
self.datDir = datDir
self.workDir = workDir
self.lines = 0B
self.error_bars = 1B

self.npointers = 100
self.tempPtr = ptrarr(self.npointers,/allocate_heap)
self.xdatPtr = ptrarr(self.npointers,/allocate_heap)
self.ydatPtr = ptrarr(self.npointers,/allocate_heap)
self.ydatErrPtr = ptrarr(self.npointers,/allocate_heap)
self.xfitPtr = ptrarr(self.npointers,/allocate_heap)
self.yfitPtr = ptrarr(self.npointers,/allocate_heap)

self.group_leader = group_leader
self->createWidgets
self.selBuffer = -1

self.xtextSize = 30
self.xpa = ptrarr(self.npointers,/allocate_heap)
self.ypa = ptrarr(self.npointers,/allocate_heap)
self.yerrpa = ptrarr(self.npointers,/allocate_heap)
self.xrange = [0.0,1.0]
self.yrange = [0.0,1.0]
self.xbox = [0.0,1.0]
self.ybox = [0.0,1.0]
self.mouse = 0B
self.autoscale = 1
;loadct,0,/silent

return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro fws_analysis__define
compile_opt idl2,hidden
define = {  fws_analysis,      $
      tlb:0L,         $
      colBase1:0L,      $
      colBase2:0L,      $
      group_leader:0L,   $
      r:bytarr(256),      $
      g:bytarr(256),      $
      b:bytarr(256),      $
      depVarGroup:0L,      $
      white:0,       $
      black:0,       $
      indepVarGroup:0L,    $
      legContainer:obj_new(), $
      lines:0B,         $
      detSlider:0L,      $
      selBuffer:0,      $
      fileBox:0L,       $
      xtextSize:0,      $
      monNorm:0L,       $
      plotBuffers:0L,      $
      plotDetectors:0L,    $
      error_bars:1L,      $
      win:0L,         $
      winVis:0L,        $
      winPix:0L,        $
      px:ptr_new(),$  ;AXIS POINTERS
      py:ptr_new(),$
      autoscale:1,      $
      xrange:fltarr(2),    $
      xbox:fltarr(2),      $
      yrange:fltarr(2),    $
      ybox:fltarr(2),      $
      mouse:0B,         $
      workDir:'',       $
      datDir:'',        $
      xlabel:'',        $
      ylabel:'',        $
      title:'',         $
      colorsPtr:ptr_new(/allocate_heap),  $
      npointers:100,      $
      tempPtr:ptrarr(100), $
      xfitPtr:ptrarr(100), $
      yfitPtr:ptrarr(100), $
      xdatPtr:ptrarr(100), $
      ydatPtr:ptrarr(100), $
      ydatErrPtr:ptrarr(100), $
      fitFileField:0L,   $
      fitWin:0L,        $
      fitWinVis:0L,      $
      fitWinPix:0L,      $
      pxfit:ptr_new(),$   ;AXIS POINTERS
      pyfit:ptr_new(),$
      autoscalefit:1,      $
      xrangefit:fltarr(2),    $
      xboxfit:fltarr(2),      $
      yrangefit:fltarr(2),    $
      yboxfit:fltarr(2),      $
      buffSlider:0L,      $
      tempSlider:0L,      $
      sendDataToPAN:0L,   $
      sendTransposedDataToPAN:0L,   $
      sendDataToTransposedASCII:0L,   $
      xpa:ptrarr(100),   $
      ypa:ptrarr(100),   $
      yerrpa:ptrarr(100),   $
      oContainer:obj_new() $
      }
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
