; $Id$
;
;
; NAME:
;	DenProBond
;
; PURPOSE:
;	This object serves as a graphical representation of an bond,
;	(or sphere), which subclasses from the IDLgrModel class.
;
; CATEGORY:
;	Object graphics.
;
; CALLING SEQUENCE:
;	To initially create:
;	       	oBond = OBJ_NEW('DenProBond') 
;
;	To retrieve a property value:
;		oBond->GetProperty
;
;	To set a property value:
;		oBond->SetProperty
;
;	To print to the standard output stream the current properties of 
;	the bond:
;		oBond->Print
;
;	To destroy:
;		OBJ_DESTROY, oBond
;
; KEYWORD PARAMETERS:
;   DenProBond::INIT:
;	<Note that keywords accepted by IDLgrModel::Init and/or
;	 IDLgrPolygon::Init are also accepted here.>
;	POS:	A three-element vector, [x,y,z], specifying the position
;               of the center of the orb, measured in data units . 
;		Defaults to [0,0,0].

; bondBegin:  A three-element vector, [x,y,z], specifying the position
;               of the beginning of the bond, measured in data units . 
;   Defaults to [0,0,0].
; bondEnd:  A three-element vector, [x,y,z], specifying the position
;               of the end of the bond, measured in data units . 
;   Defaults to [0,0,0].


;	RADIUS: A floating point number representing the radius of the
;               bond (measured in data units).  The default is 1.0.
;	DENSITY: A floating point number representing the density at which
;               the vertices should be generated along the surface of the
;               bond.  The default is 1.0.
;	TEX_COORDS: Set this keyword to a nonzero value if texture map
;               coordinates are to be generated for the bond.
;
;   DenProBond::GETPROPERTY:
; BondBegin:  Set this keyword to a named variable that upon return will
;   contain a three-element vector, [x,y,z], specifying the 
;   position of the beginning of the bond, measured in data units . 
; BondEnd:  Set this keyword to a named variable that upon return will
;   contain a three-element vector, [x,y,z], specifying the 
;   position of the end of the bond, measured in data units . 


;	RADIUS: Set this keyword to a named variable that upon return will
;		contain a floating point number representing the radius of the
;               bond (measured in data units).
;	DENSITY: Set this keyword to a named variable that upon return will
;		contain a floating point number representing the density at 
;		which the vertices are generated along the surface of the
;               bond.
;
;   DenProBond::SETPROPERTY:
;	<Note that keywords accepted by IDLgrModel::SetProperty and/or
;	 IDLgrPolygon::SetProperty are also accepted here.>
;	POS:	A three-element vector, [x,y,z], specifying the position
;               of the center of the orb. Defaults to [0,0,0].
;	RADIUS: A floating point number representing the radius of the
;               orb (measured in data units).  The default is 1.0.
;	DENSITY: A floating point number representing the density at which
;               the vertices should be generated along the surface of the
;               orb.  The default is 1.0.
;
; EXAMPLE:
;	Create an bond centered at the origin with a radius of 0.5:
;		oBond = OBJ_NEW('DenProBond', bondBegin=[0,0,0],bondEnd=[0,0,0],POS=[0,0,0], RADIUS=0.5) 
;

;----------------------------------------------------------------------------
; DenProBond::INIT
;
; Purpose:
;  Initializes an DenProBond object.
;
;  This function returns a 1 if initialization is successful, or 0 otherwise.
;
FUNCTION DenProBond::Init, POS=pos, bondbegin=bondbegin,bondend=bondend,$
                           RADIUS=radius, DENSITY=density, $
                           color=color,$
                           TEX_COORDS=tex_coords, _EXTRA=e

    IF (self->IDLgrModel::Init(_EXTRA=e) NE 1) THEN RETURN, 0

    self.color=[0b,0b,0b]
    self.pos = [0.0,0.0,0.0]
    self.bondbegin = [0.0,0.0,0.0]
    self.bondend = [1.0,1.0,1.0]
    self.radius = 1.0
    self.density = 1.0

    IF (N_ELEMENTS(pos) EQ 3) THEN $
        self.pos = pos

    IF (N_ELEMENTS(color) EQ 3) THEN begin
    
        ;print,'DenProBond::Init color=',color
        self.color = color

    endif


    IF (N_ELEMENTS(bondbegin) EQ 3) THEN $
        self.bondbegin = bondbegin
    IF (N_ELEMENTS(bondend) EQ 3) THEN $
        self.bondend = bondend


    IF (N_ELEMENTS(radius) EQ 1) THEN $
        self.radius = radius

    IF (N_ELEMENTS(density) EQ 1) THEN $
        self.density = density

    IF (N_ELEMENTS(tex_coords) EQ 1) THEN $
        self.texture = tex_coords

    ; Initialize the polygon object that will be used to represent
    ; the DenProBond.
    self.oPoly = OBJ_NEW('IDLgrPolygon', SHADING=1, /REJECT, _EXTRA=e)
    self.oLine = OBJ_NEW('IDLgrPolyline',_EXTRA=e)


    self->Add,self.oPoly
    self->Add,self.oLine

    ; Build the polygon vertices and connectivity based on property settings.
    self->BuildPoly

    RETURN, 1
END

;----------------------------------------------------------------------------
; DenProBond::CLEANUP
;
; Purpose:
;  Cleans up all memory associated with the DenProBond.
;
PRO DenProBond::Cleanup

    ; Cleanup the polygon object used to represent the DenProBond.
    OBJ_DESTROY, self.oPoly
    OBJ_DESTROY, self.oLine

    ; Cleanup the superclass.
    self->IDLgrModel::Cleanup
END;DenProBond::Cleanup

;----------------------------------------------------------------------------
; DenProBond::SETPROPERTY
;
; Purpose:
;  Sets the value of properties associated with the DenProBond object.
;
PRO DenProBond::SetProperty, $
    DENSITY=density, $
    PARENT=parent, $ ; Pass along to IDLgrModel only.
    bondbegin=bondbegin,$
    bondend=bondend,$
    POS=pos, $
    RADIUS=radius, $
    color=color,$
    _EXTRA=e

    ; Pass along extraneous keywords to the superclass and/or to the
    ; polygon used to represent the orb.
    self->IDLgrModel::SetProperty, _EXTRA=e
    self.oPoly->SetProperty, _EXTRA=e
    self.oLine->SetProperty, _EXTRA=e


    if n_elements(color) eq 3 then begin
      self.color=color
    endif

    IF (N_ELEMENTS(pos) EQ 3) THEN $
        self.pos = pos

    IF (N_ELEMENTS(BondBegin) EQ 3) THEN begin
        self.bondBegin = BondBegin
        print,'BondBegin=',bondBegin
    endif
        
    IF (N_ELEMENTS(BondEnd) EQ 3) THEN begin
        self.bondEnd = BondEnd
        ;print,'BondEnd=',bondEnd
    endif


    IF (N_ELEMENTS(radius) EQ 1) THEN $
        self.radius = radius

    IF (N_ELEMENTS(density) EQ 1) THEN $
        self.density = density

    ; Rebuild the polygon according to keyword settings.
    self->BuildPoly
END;DenProBond::SetProperty

;----------------------------------------------------------------------------
; DenProBond::GETPROPERTY
;
; Purpose:
;  Retrieves the value of properties associated with the DenProBond object.
;
PRO DenProBond::GetProperty, POS=pos, RADIUS=radius, DENSITY=density,$
                         bondbegin=bondbegin, bondend=bondend,$
                         POBJ=pobj, lobj=lobj,color=color,_REF_EXTRA=re
;print,'DenProBond::GetProperty,arg_present(color) =',arg_present(color)
    ; Retrieve extra properties from polygon first, then model
    ; so that the model settings (for common keywords) will prevail.
    self.oPoly->GetProperty, _EXTRA=re
    self->IDLgrModel::GetProperty, _EXTRA=re


    color=self.color
    pos = self.pos
    bondend = self.bondend
    bondbegin = self.bondbegin
    radius = self.radius 
    density = self.density 
    pobj = self.oPoly
    lobj = self.oLine
END;DenProBond::GetProperty

PRO DenProBond::Print
    PRINT, 'self.pos=',self.pos
    PRINT, 'self.bondbegin=',self.bondbegin
    PRINT, 'self.bondend=',self.bondend
    PRINT, 'self.redius=',self.radius
    PRINT, 'self.density=',self.density
END;DenProBond::Print

;BORROWED FROM g3dview:
;CREATES AN IDLgrPolygon AND AN IDLgrPolyline TO CREATE A CYLINDER TO REPRESENT A BOND.
function DenProBond::g3dviewmakeTube, v0, v1, tubedia, tubecolor, p1, _extra=e

;print,'DenProBond::g3dviewmakeTube'
;help,v0,v1,tubedia,tubecolor,p1,e


    ;  MESH_OBJ params
    p2 = v0
    p3 = v1 - v0

    ;  Calculate the x-product
    ;  You'll have to handle special case where p3==[0,0,1]
    if p3[0] eq 0 and p3[1] eq 0 then begin
    cp = CROSSP(p3,[0.,1.,0.])
    endif else begin
    cp = CROSSP(p3,[0.,0.,1.])
    endelse
    ;  Normalize x-product
    s = SQRT(TOTAL(cp^2))
    cp = cp / s
    ;  Create the line to rotate
    array1 = [[v0],[v1]] - REBIN(tubedia * cp, 3, 2)
    ;  Create the mesh
    MESH_OBJ, 6, verts, polys, array1, P1=p1, P2=p2, P3=p3
    ;MESH_OBJ, 3, verts, polys, array1, P1=p1, P2=p2, P3=p3

    ;  Create a polygon and a line representing the original vector
    if n_elements(tubecolor) ne 3 then tubecolor=[255b,0b,0b]
    ;print,'DenProBond::g3dviewmakeTube, tubecolor=',tubecolor
;help,/traceback
    oTube = OBJ_NEW('IDLgrPolygon', verts, POLYGONS=polys, $
        COLOR=tubecolor, STYLE=2, _extra=e)
    oVec = OBJ_NEW('IDLgrPolyline', [[v0],[v1]], COLOR=tubecolor)

    RETURN, [oTube, oVec]
end;DenProBond::g3dviewmakeTube


;----------------------------------------------------------------------------
; DenProBond::BUILDPOLY
;
; Purpose:
;  Sets the vertex and connectivity arrays for the polygon used to
;  represent the DenProBond.
;
PRO DenProBond::BuildPoly
    ; Build the DenProBond.
;print,'DenProBond::BuildPoly'
    p1=20
    objs = self->g3dviewmakeTube( self.bondbegin,self.bondend, self.radius,self.color, p1);, _extra=e)
;    objs = self->g3dviewmakeTube( [0.0,0.0,0.0], [5.0,5.0,5.0], 1.0, [255,0,0])

;011808
    ;FOR NOW, DESTROY THE OLD OBJECTS AND REPLACE THEM WITH THE NEW ONES.
    ;I DON'T THINK THIS WILL WORK BECAUSE OF THE Z-BUFFER ISSUE, 
    ;BUT USE IT FOR NOW TO GET THINGS ROLLING.
    self->remove,/all
    obj_destroy,self.oPoly
    obj_destroy,self.oLine
    
    self.oPoly = objs[0]
    self.oLine = objs[1]
    self->add,self.oPoly
    self->add,self.oLine
   
END;DenProBond::BuildPoly

;----------------------------------------------------------------------------
; DenProBond__DEFINE
;
; Purpose:
;  Defines the object structure for an DenProBond object.
;
PRO DenProBond__define
    struct = { DenProBond, $
               INHERITS IDLgrModel, $
               ;INHERITS IDLitVisualization, $
               pos: [0.0,0.0,0.0], $
               bondbegin: [0.0,0.0,0.0], $
               bondend: [0.0,0.0,0.0], $
               radius: 1.0, $
               density: 1.0, $
               texture: 0, $
               color:[0b,0b,0b],$
               oPoly: OBJ_NEW(), $
               oLine: OBJ_NEW() $
             }
END;DenProBond__define







