; $Id$
;#######################################################################
;
; NAME:
;  dm_write_spe
;
; PURPOSE:
;  write into a spe file xdat[nx] or [nx+1],ydat[ny] or [ny+1],zdat[nx,ny],zerr[nx,ny]
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  July, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;parameters:
;  file: a string of the file name be written in
;  xdat: [nx] or [nx+1] for histogram data
;  ydat: [ny] or [ny+1] for histogram data
;  zdat: [nx,ny]
;  zerr: [nx,ny]
;keywords:
;  xtit,ytit,ztit: labels for xdat, ydat, and zdat
;  error: returns 1 if input/output error occurs
;  group_leader: tlb for the dialog window
pro dm_write_spe,file,xdat,ydat,zdat,zerr,xtit=xtit,ytit=ytit,ztit=ztit,$
    error=error,group_leader=group_leader
    if n_elements(xtit) eq 0 then xtit='xdat' else xtit=strtrim(xtit[0],2)
    if n_elements(ytit) eq 0 then ytit='ydat' else ytit=strtrim(ytit[0],2)
    if n_elements(ztit) eq 0 then ztit='zdat' else ztit=strtrim(ztit[0],2)
    if n_elements(zerr) eq 0 or n_elements(zerr) ne n_elements(zdat) then begin
       error = 1b
       ok = dialog_message(['An error occurred when calling dm_write_spe.',' ',$
          'Syntax: dm_write_spe,file,xdat,ydat,zdat,zerr,xtit=xtit,ytit=ytit,ztit=ztit,error=error,group_leader=group_leader'],$
          /error,title='Syntax Error',dialog_parent=group_leader)
       return
    endif
    
    nx  = n_elements(xdat)
    ny  = n_elements(ydat)
    nz  = size(zdat,/dimension)
    
    if n_elements(nz) ne 2 then begin
       notgood = 1b
       errmsg  = 'zdat must be a two dimensional array.' 
       help,zdat,output=output
       errmsg  = [errmsg,output]
    endif else begin
       if (nx ne nz[0]) and (nx ne nz[0]+1) then begin
          notgood = 1b
          errmsg  = 'Incompatible xdat and zdat.'
          help,xdat,zdat,output=output
          errmsg  = [errmsg,output]
       endif 
       if (ny ne nz[1]) and (ny ne nz[1]+1) then begin
          notgood = 1b
          errmsg  = 'Incompatible ydat and zdat'
          help,ydat,zdat,output=output
          errmsg  = [errmsg,output]
        endif
    endelse
    if keyword_set(notgood) then begin
       error = 1b
       ok = dialog_message(errmsg,/error,title='Data Format Error',dialog_parent=group_leader)
       return
    endif
    
    openw,funit,file,/get_lun,error=error
    if error ne 0 then begin
       ok = dialog_message("Can't write in "+file,/error,dialog_parent=group_leader)
       return
    endif
    
    sformat = ['(8(g10.5))','(8(g10.4))','(8(g10.3))']
    
    if (nz[0] ge 1e6) or (nz[1] ge 1e6) then printf,funit,dm_to_string(nz[0],/int)+' '+dm_to_string(nz[1],/int) $
    else printf,funit,nz[0],nz[1],format='(2i5)'

    ;write xdat grid
    printf,funit,'### '+xtit
    ;search for the correct string format
    for fid=0,2 do begin
        ok  = 1b
        tmp = string(xdat,format=sformat[fid])
        for i=0,n_elements(tmp)-1 do begin
            if stregex(tmp[i],'\*',/boolean) then begin
               ok = 0b
               break
            endif
        endfor
        if ok then break
    endfor
    if nx eq nz[0] then $
       printf,funit,[xdat,0.0],format=sformat[fid] $
    else $
       printf,funit,xdat,format=sformat[fid]

    ;write ydat grid
    printf,funit,'### '+ytit
    ;search for the correct string format
    for fid=0,2 do begin
        ok  = 1b
        tmp = string(ydat,format=sformat[fid])
        for i=0,n_elements(tmp)-1 do begin
            if stregex(tmp[i],'\*',/boolean) then begin
               ok = 0b
               break
            endif
        endfor
        if ok then break
    endfor
    if ny eq nz[1] then $
       printf,funit,[ydat,0],format=sformat[fid] $
    else $
       printf,funit,ydat,format=sformat[fid]
    ;write zdat(nx,ny) and zerr(nx,ny)
    for i=0L,nz[0]-1L do begin
        printf,funit,'### '+ztit
        printf,funit,zdat[i,*],format=sformat[2]
        printf,funit,'### Errors'
        printf,funit,zerr[i,*],format=sformat[2]
    endfor
    free_lun,funit
end