; $Id: $
;#######################################################################
;
; NAME:
;  dm_load_dcsdata
;
; PURPOSE:
;  This program loads dcs file(s) for dcs_mslice and dcs_cryst_align program.
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2022
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; Calls:
;   dcs_read_binaryOctavefile   :   the program which actually reads in a dcs data
;   dm_center_kid               :   widget centering
;   dm_define_pointer           :   define a dave pointer
;   dm_ispower2                 :   determine if  a number is power of two
;   dm_locate_datastrptr        :   locate datastrPtr from the dave pointer
;   dm_to_string                :   string conversion
; Parameters:
;   open_path:  file path,string
;   open_file:  file name(s),string or string array
; Keywords:
;  input:
;   avgsum:     0:arithmetic average 1:sum, effective when diffuse keyword is set,default is 1
;   bitflip:    0:skip bitflip correction 1:yes(default)
;   checksum:   if set, do not bypass the bitflip checks for files later than 20090407, setting slice smooth level to -99 will set this flag
;   diffetyp:   0:single energy  1:multiple energy, only needed when diffuse keyword is set
;   diffuse:    if set, the output is intended for dcs_mslice diffuse scattering type
;   eadjust:    0: none 1:automatic 2:specified
;   epadjust:   for eadjust=2, epadjust specified the apparent elastic peak position
;   epadcheckfirst: if set, only check the first file for the shifting of elastic peak, and use it for all files
;   eint_yn:    only applicable for diffuse mode, if set, I=sum(I[i,j]*dE[i,j]) over the specified time channels
;   e_range:    energy range, effective when t_chan=1&2
;   fissionchamber: if set, fc2 and fc3 counts will be regarded as det 914 & 915, keyword raw should be set                
;   ftpobj:     ftp object for ftp files
;   ftpbufferdir: ftp buffer directory
;   monfit:     if set then monitor is fitted by a gaussian, and the monitor count is the integrated intensity, otherwise it's just the sum
;               default is set
;   montype:    0-fc0 1-bm1 2-duration, default is 1
;   nomessage:  if set, no message except for the file loading progress message will be displayed
;   parent:     message widget parent
;   raw:        if set, data will not be normalized, the output is intended for dcs_cryst_align program
;   t_chan:     0:all 1:elastic 2:specify erange
;   title:      message title
;  output:  all data types are float
;   qty:        [ntchan,ndet] or [ndet,nfiles]-raw or [ntchan,ndet,nfiles]-diffuse(same as raw if diffetyp=0)
;   dqty:       [ntchan,ndet] or [ndet,nfiles]-raw or [ntchan,ndet,nfiles]-diffuse
;   ang:        [12,nfiles], [*,0] contains [rotation,height,x-tran,y-tran,x-tilt,y-tilt,file index,hfield,T_sample,T_control,T_setpoint,monitor_count]
;               available when raw or diffuse keywords are set
;   bgrate:     [ndet] overall background rate in cts/hr/det, available when shutter closed files are present
;   dbgrate:    [ndet] error of bgrate
;   comment:    comment line of first file
;   duration:   duration of first file, available when diffuse or raw keywords are set
;   ehist:      [ntchan+1]
;   einc:       Ei
;   emean:      [ntchan]
;   e_range:    if initial values are not set, will save the e_range info
;   error:      1 if any error occurs, the results should be discarded
;               possible errors: 1.different file type 2.shutter closed for all nonbackground data
;                                3.shutter open for all dark count files 4. non existing filename
;   ewid:       diffuse:ewid[tchans]; powder,single:[ntchan], the corresponding energy width for the time channels
;   info:       {ch_wl:ch_wl,ch_ms:ch_ms,ch_srdenom:ch_srdenom,ch_res:ch_res,tsdmin:tsdmin,temperature:setpoint} is used to check
;               file sameness
;   kfactor:    [ntchan]-(ki/kf)  or a scalor
;   monrate:    monitor rate for normalizing dark count rate in projection calculation
;   shutter:    [nfiles] of shutter status, 0-closed 1-open
;   tmean:      [ntchan]    time channel in us
;   weight:     sum of integrated intensity of the beam monitor
;               or
;               [nfile] array of integrated intensity of the beam monitor if diffuse keyword is set
;   zero_error: return error bar for sqrt(intensity=1)
pro dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,ang=ang,avgsum=avgsum,bitflip=bitflip,bgrate=bgrate,dbgrate=dbgrate,checksum=checksum,comment=comment,$
    diffetyp=diffetyp,diffuse=diffuse,duration=duration,eadjust=eadjust,epadjust=epadjust,epadcheckfirst=epadcheckfirst,ehist=ehist,einc=einc,eint_yn=eint_yn,emean=emean,e_range=e_range,$
    error=error,ewid=ewid,fissionchamber=fissionchamber,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir,info=info,kfactor=kfactor,monfit=monfit,monrate=monrate,montype=montype,nomessage=nomessage,$
    parent=parent,raw=raw,shutter=shutter,t_chan=t_chan,title=title,tmean=tmean,weight=weight,zero_error=zero_error,debug=debug

    darkcount = arg_present(bgrate)
    if n_elements(eadjust)     eq 0 then eadjust  = 0
    if eadjust eq 2 then begin
       if n_elements(epadjust) eq 0 then eadjust  = 0 $
       else if (~ finite(epadjust)) then eadjust  = 0
    endif
    sch_ead = (eadjust gt 0)                              ;flag for elastic peak adjustment
    if (keyword_set(diffuse) or keyword_set(raw)) then begin
       if n_elements(avgsum)   eq 0 then avgsum   = 1     ;default is sum
       if n_elements(t_chan)   eq 0 then t_chan   = 1     ;default is elastic peak
       if n_elements(diffetyp) eq 0 then diffetyp = 0     ;default is single energy
       if diffetyp eq 0 then sch_ead = 0b
    endif else begin
       if n_elements(avgsum)   eq 0 then avgsum   = 0     ;default is average
       if n_elements(t_chan)   eq 0 then t_chan   = 0     ;default is all time channels
       if n_elements(diffetyp) eq 0 then diffetyp = 1     ;default is multiple energy
    endelse
    if t_chan eq 2 then begin                             ;if e_range is not given, regard it as elastic peak
       if n_elements(e_range) ne 2         then t_chan = 1 $
       else if total(finite(e_range)) ne 2 then t_chan = 1                
    endif
    if keyword_set(fissionchamber) and (~ keyword_set(raw)) then fissionchamber=0  ;disable that
    if n_elements(bitflip)     eq 0 then bitflip  = 1
    if n_elements(title)       eq 0 then title    = 'Loading data ...'
    if n_elements(monfit)      eq 0 then monfit   = 1     ;default is to fit monitor with a gaussian, and use the integrated intensity as normalization
    if n_elements(montype)     eq 0 then montype  = 1     ;default is BM1
    zero_error  = 0.0                           ;error bar for intensity=1
    n_dkfile    = 0
    error       = 0b                            ;initialization - no error
    sch_tch     = (t_chan gt 0)                 ;flag for locating time channels
    nofirstline = 0 & omithisto=0               ;dcs_read_binaryOctavefile keyword
    id_mon      = ([2,3,2])[montype]            ;index of monitor in histohigh[0:ntchan,*]
    n_files     = n_elements(open_file)
    n_read      = 0
    pathsep     = [dm_define_pointer(/getpathsep),'/']
    
    ;unexpected errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if ~keyword_set(nomessage) then ok = dialog_message(!error_state.msg,/error,/center)
       if obj_valid(mesg) then obj_destroy,mesg
       if ptr_valid(davePtr) then heap_free,davePtr & tmp=check_math(mask=32)
       return
    end
    
    daqdate = dm_to_number('07 Apr 2009',/date)     ;the date when the new data acquisition system was tested
    while(strmid(open_path,0,1,/reverse_offset) eq  pathsep[obj_valid(ftpobj)]) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    for i=0L,n_files-1L do begin
        file = open_path+pathsep[obj_valid(ftpobj)]+open_file[i]
        if (~ ptr_valid(davePtr)) then davePtr=dm_define_pointer()
        if i eq 0 then $
           mesg =  obj_new('dm_progress',title=title,message='Loading '+file,group_leader=parent) $
        else begin
           if (~ obj_valid(mesg)) then break
           mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)
        endelse
        if obj_valid(ftpobj) then begin
           ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+pathsep[0]+'ftptmp'+open_file[i])
           file = ftpbufferdir+pathsep[0]+'ftptmp'+open_file[i]
        endif
        ;read in data file
        dcs_read_binaryOctavefile,file,nofirstline,omithisto,var_read,unknown,readerror,davePtr=davePtr,/histodata_floating
        if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
        if strlen(readerror) ne 0 then begin
           error = 1b                           ;error occurs
           if obj_valid(mesg) then obj_destroy,mesg
           heap_free,davePtr & tmp=check_math(mask=32)
           if ~keyword_set(nomessage) then ok = dialog_message([file,readerror],/error,dialog_parent=parent,/center)
           return
        end
        this          = dm_locate_datastrptr(davePtr)                               ;get the data structure pointer
        old_daq       = ((dm_to_number((*(*this).specificPtr).start_date,/date) lt daqdate) or keyword_set(checksum))
        is_polanal    = (*(*this).specificPtr).polanal                              ;polarization analysis flag
        tsdmin        = float((*(*this).specificPtr).tsdmin)                        ;tsdmin
        ch_wl         = float((*(*this).specificPtr).ch_wl)                         ;wavelength
        ch_ms         = float((*(*this).specificPtr).ch_ms)                         ;chopper master speed
        ch_srdenom    = (*(*this).specificPtr).ch_srdenom                           ;chopper speed ratio denomenator
        ch_srnumer    = ([1,1>(ch_srdenom-1)])[(*(*this).specificPtr).ch_srmode-1]  ;chopper speed ratio numerator
        ch_res        = (*(*this).specificPtr).ch_res                               ;resolution mode
        setpoint      = float((*(*this).specificPtr).temp_setpoint)                 ;temperature set point
        motor_pos     = (*(*this).specificPtr).motor_pos                            ;motor position
        field         = strupcase(tag_names(*(*this).specificPtr))
        loc           = where(field eq 'FIELD_SAMPLE',count)                        ;check magnetic field exist or not
        if count eq 0 then hfield = !values.f_nan else hfield = float((*(*this).specificPtr).field_sample)
        duration1     = (*(*this).specificPtr).duration
        duration2     = double(dm_to_number((*(*this).specificPtr).stop_date,/epoch)-dm_to_number((*(*this).specificPtr).start_date,/epoch))
        this_duration = (abs(duration1-duration2) gt 0.1*duration1)?(duration2):(duration1)  ;duration of run in secs                            
        shut_stat     = (*(*this).specificPtr).shutter_stat                         ;shutter status 0:closed 1:open -1:not defined
        temp_sample   = (*(*this).specificPtr).temp_sample                          ;sample temperature
        temp_control  = (*(*this).specificPtr).temp_control    
        postemp       = where(temp_sample gt 0,count)
        if count ne 0 then begin
           temp_sample  = temp_sample[postemp]
           temp_control = temp_control[postemp]
        endif
        temp_sample   = mean(temp_sample)
        temp_control  = mean(temp_control)
        ;check sameness
        if n_elements(info) eq 0 then begin                      ;save file information
           info  = {ch_wl:ch_wl,ch_ms:ch_ms,ch_srdenom:ch_srdenom,ch_srnumer:ch_srnumer,ch_res:ch_res,tsdmin:tsdmin,temperature:setpoint}
           file0 = open_file[i]
        endif else if (~ keyword_set(raw)) then begin            ;check file sameness except when raw keyword is set
           if total(abs([info.ch_wl,info.ch_ms,info.ch_srdenom,info.ch_srnumer,info.tsdmin,info.ch_res]-[ch_wl,ch_ms,ch_srdenom,ch_srnumer,tsdmin,ch_res])) ne 0 then begin
              if ~keyword_set(nomessage) then begin
                 if n_elements(file0) eq 0 then file0 = 'previous'
                 resl = ['','low','med','high']
                 tmp  = [file0+':  '+dm_to_string(info.ch_wl)+' '+string('c5'XB)+'   '+resl[info.ch_res]+' resl   '+dm_to_string(info.ch_ms)+' rpm   '+$
                         dm_to_string(info.ch_srnumer)+'/'+dm_to_string(info.ch_srdenom)+'   tsdmin='+dm_to_string(info.tsdmin),$
                         open_file[i]+':  '+dm_to_string(ch_wl)+' '+string('c5'XB)+'   '+resl[ch_res]+' resl   '+dm_to_string(ch_ms)+' rpm   '+$
                         dm_to_string(ch_srnumer)+'/'+dm_to_string(ch_srdenom)+'   tsdmin='+dm_to_string(tsdmin)]
                 ok   = dialog_message(['Selected data files are of different wavelength settings.',tmp],/error,dialog_parent=parent,/center)
              endif 
              error = 1b                        ;error occurs
              if obj_valid(mesg) then obj_destroy,mesg
              heap_free,davePtr & tmp=check_math(mask=32)
              return                            ;exit
           endif
        endif
        if i eq 0 then begin
           nhedet   = (*(*this).specificPtr).nhedet               ;number of He-3 detectors
           comment  = strtrim((*(*this).specificPtr).comments,2)  ;comment
           ntchan   = (*(*this).specificPtr).ntchan               ;number of time channels
           speriod  = 6.0d7/ch_ms*ch_srdenom                      ;time between pulses at sample in micro second
           ; Define the number of good time channels for the He3 detectors, ngtchan0,
           ; and the number of good time channels for the high end inputs,  ngtchan1,
           twid     = (*(*this).specificPtr).tchanlook*0.05
           times0   = [0,reform(total(twid[0,*],/cumulative))]
           times1   = [0,reform(total(twid[1,*],/cumulative))]
           tstart   = times0[0:ntchan-1]
           tstop    = times0[1:ntchan] & times0 = 0      
           chan     = where(tstart le speriod-0.05 and tstop gt speriod-0.05) + 1
           ngtchan0 = chan[0]
           if ngtchan0 eq 0 then begin
              ngtchan0 = 1000L
              warning  = 1b 
           endif 
           tstart   = times1[0:ntchan-1]
           tstop    = times1[1:ntchan] & times1 = 0
           chan     = where(tstart le speriod-0.05 and tstop gt speriod-0.05) + 1
           ngtchan1 = chan[0] 
           if ngtchan1 eq 0 then begin
              ngtchan1 = 1000L
              warning  = 1b
           endif
           tstart = 0 & tstop = 0
           if keyword_set(warning) and (~keyword_set(nomessage)) then $
              ok = dialog_message('The time channel look-up table in this file might not be correct.',dialog_parent=parent,title='Warning: '+open_file[i],/center)
           conversion = !dcs_hsq2mn/(!dcs_hom/!dcs_dsd)^2
           tmean = ((*(*(*(*davePtr).dataStrPtr).specificPtr).time_propsPtr)[0:ngtchan0-1]+(*(*(*(*davePtr).dataStrPtr).specificPtr).time_propsPtr)[1:ngtchan0])*0.5
           emean = float(!dcs_hsq2mn/ch_wl^2-conversion/tmean^2)
           if (~ keyword_set(raw)) then begin     ;no use of twid for dcs_cryst_align data
              times = float((*(*(*this).specificPtr).time_propsPtr)[0:ngtchan0])
              twid  = times[1:ngtchan0]-times[0:ngtchan0-1] & times = 0
              w_max = max(twid,min=w_min)
              if w_max eq w_min then $
                 twid = twid[0] $
              else $
                 twid = twid#(fltarr(nhedet)+1.0)
           endif
           einc     = float(!dcs_hsq2mn/(ch_wl)^2)   ;incident energy
           ehist    = (*(*(*this).specificPtr).energy_propsPtr)[0:ngtchan0]
           emean0   = emean
           ewid     = (ehist[1:ngtchan0]-ehist[0:ngtchan0-1])
           ef       = einc-emean                ;final energy
           ki_kf    = sqrt(einc/ef)             ;ratio of ki/kf
           ntchan   = ngtchan0
           nitem    = ngtchan0
           if avgsum eq 1 then nitem = 1.0      ;1 for sum
           if darkcount then begin
              fb_weight = 0.0                   ;save the duration in hrs, for shutter-closed files
           endif
           if keyword_set(raw) or keyword_set(diffuse) then begin   ;initialization
              duration = this_duration          ;duration of first file
           endif else begin
              if n_elements(qty) eq 0 then begin
                 qty    = 0.0
                 dqty   = 0.0
                 weight = 0.0
              endif
           endelse
        endif
        tmp_qty = (*(*this).commonStr.histPtr).qty
        tmp_err = ((*(*this).commonStr.histPtr).err) 
        ptr_free,(*this).commonStr.histPtr
        ;check jumpy file
        if (~keyword_set(nomessage)) and (~is_polanal) then begin
           tmp_sum = total(tmp_qty,1)
           tmp_max = max(tmp_sum) & tmp_mean = mean(tmp_sum)
           tmp_ind = where(tmp_sum ge 10.0*tmp_mean,tmp_count)
           if tmp_count ne 0 then tmp_sum[tmp_ind] = tmp_max  ;enlarge the jumpy noise signal
           tmp = A_CORRELATE(tmp_sum,[31,248])                ;31 is the period, 248=31*8 is to avoid mistake
           if min(tmp) gt 0.3 then begin                      ;0.3 is the preset criterion
              if ((*(*this).specificPtr).ncycles eq 1) or (dm_to_number((*(*this).specificPtr).start_date,/epoch) lt dm_to_number('Jan 1 00:00:00 2015',/epoch)) then tmp_mesg = '' $
              else tmp_mesg = 'try patching this file using File->File Tools->Patch DCS Files or '  ;patching is only possible if ncycles> 1 and for files with each cycles saved
              ok = dialog_message(open_file[i]+' might be a "jumpy" data file. If this is the case, please '+tmp_mesg+'discard this file.',dialog_parent=parent,title='Warning: '+open_file[i],/center)
           endif
           tmp_sum = 0   
        endif
        ;correct bit flip errors
        if keyword_set(bitflip) and old_daq and (~is_polanal) then begin
           tdh_dsum = [long64(total(tmp_qty,1,/double)),long64(total((*(*this).specificPtr).histohigh,1,/double))]
           tdh_tsum = long64(total(tmp_qty,2,/double))+long64(total((*(*this).specificPtr).histohigh,2,/double))
           bad_t = where((tdh_tsum-(*(*this).specificPtr).timsum) ne 0,tcount)    ;disagreement in time sums over detector
           bad_d = where((tdh_dsum-(*(*this).specificPtr).detsum) ne 0,dcount)    ;disagreement in detector sums over time   
           if tcount lt 50 and dcount lt 50 then begin
              ;the following code can only correct isolated bitflip errors
              for ii=0,tcount-1 do begin
                  mesg->update,message='correcting bitflip errors...',title=open_file[i]
                  for jj=0,dcount-1 do begin
                      tmp1 = tdh_tsum[bad_t[ii]]-((*(*this).specificPtr).timsum)[bad_t[ii]]
                      tmp2 = tdh_dsum[bad_d[jj]]-((*(*this).specificPtr).detsum)[bad_d[jj]]
                      if (tmp1 eq tmp2) and (bad_t[ii] lt ngtchan0) and (bad_d[jj] lt nhedet) then begin
                         if (~dm_ispower2(abs(tmp1))) and (~keyword_set(nomessage)) then begin
                            ok = dialog_message(['Bitflip error at det='+dm_to_string(bad_d[jj])+', tchan='+dm_to_string(bad_t[ii])+$
                                 " may involve several bits.",'This bitflip error has been corrected.',$
                                 '','Detector number and time channel number start from 0.'],dialog_parent=parent,title='Warning: '+open_file[i],/center)
                         endif
                         tmp_qty[bad_t[ii],bad_d[jj]] = tmp_qty[bad_t[ii],bad_d[jj]]-tmp1
                         tmp_err[bad_t[ii],bad_d[jj]] = sqrt(0>(tmp_qty[bad_t[ii],bad_d[jj]]))
                         tdh_dsum = [long64(total(tmp_qty,1,/double)),long64(total((*(*this).specificPtr).histohigh,1,/double))]
                         tdh_tsum = long64(total(tmp_qty,2,/double))+long64(total((*(*this).specificPtr).histohigh,2,/double))
                      endif else begin
                         if n_elements(uncorr) eq 0 then uncorr = [bad_d[jj],bad_t[ii]] $
                         else uncorr = [[uncorr],[bad_d[jj],bad_t[ii]]]
                      endelse
                  endfor
              endfor
              if n_elements(uncorr) ne 0 then begin
                 tmp = n_elements(uncorr[0,*])
                 be_info = 'The following bitflip error'+(tmp gt 1?'s are':' is')+' not corrected:'
                 for kk=0,(30<(tmp-1)) do $
                     be_info = [be_info,string(uncorr[*,kk],format='("    det num = ",i4,"   tchan num = ",i4)')]
                 if kk lt tmp then $
                    be_info = [be_info,'            ...','            ...','            ...'] $
                 else  $
                    be_info[kk] = be_info[tmp]+' .'
                 be_info = [be_info,'','Detector number and time channel number start from 0.',$
                           'Detector number greater than '+dm_to_string(nhedet-1)+' or time channel number',$
                           'greater than '+dm_to_string(ngtchan0-1)+' is for histohigh data.']
                 if ~keyword_set(nomessage) then ok = dialog_message(be_info,dialog_parent=parent,title='Warning: '+open_file[i],/center)
                 tmp = temporary(uncorr) & tmp = -1
              endif
              tdh_dsum = -1 & tdh_tsum=-1
           endif else begin
              if (n_elements(skip_ok) eq 0) and (~keyword_set(nomessage)) then $
                 skip_ok = dialog_message('There are too many bit flip errors. Skip bit flip error correction.',dialog_parent=parent,title='Warning: '+open_file[i],/center)
           endelse
        endif
        if is_polanal then zeroerr1 = tmp_err[-1] else zeroerr1 = 1.0
        tmp_qty = tmp_qty[0:ngtchan0-1,*]
        tmp_err = tmp_err[0:ngtchan0-1,*]
        if keyword_set(fissionchamber) then begin  ;add FC2 and FC3 to the end of the data
           fc_qty  = (*(*this).specificPtr).histohigh[0:ngtchan1-1,4:5]
           fc_err  = sqrt(fc_qty)
           tmp_qty = [[tmp_qty],[fc_qty]]
           tmp_err = [[tmp_err],[fc_err]]
        endif
        ;check if shutter is closed
        mon_counts = (*(*this).specificPtr).histohigh[0:ngtchan1-1,id_mon]
        mon_sum = total(mon_counts)
        if shut_stat eq -1 then begin
           shutter_close = total(tmp_qty)/mon_sum
           shutter_close = (shutter_close gt 200)
        endif else $
           shutter_close = 1-shut_stat
        if montype eq 2 then begin
           mon_sum = this_duration
        endif
        if i eq 0 then first_mon_sum = mon_sum
        if ch_wl eq 0 then shutter_close=1  ;regard lambda=0 as shutter closed measurement
        if sch_ead then begin               ;elastic peak adjustment
           erotate = 0
           if shutter_close eq 1 then $
              sch_ead = 1b $                ;need to do it again
           else begin
              i_el = dm_findbetween(emean0,0)
              if (~ finite(i_el)) then begin
                 if ~keyword_set(nomessage) then ok = dialog_message('No E=0 in the original data. Elastic peak position adjustment is skipped.',dialog_parent=parent,title='Warning: '+open_file[i],/center) 
                 sch_ead = 0b
              endif
              if eadjust eq 1 then begin
                 tmp_y = total(tmp_qty,2,/double)
                 dm_gaussfit,tmp_y,params=params,fitnotgood=fitnotgood,debug=debug
                 if fitnotgood then begin;somehow the fitting is not right use all time channels
                    if ~keyword_set(nomessage) then ok = dialog_message(["Can't locate the elastic peak.","Elastic peak adjustment is skipped."],dialog_parent=parent,title='Warning: '+open_file[i],/center)
                    i_eadj = !values.f_nan
                 endif else begin
                    i_eadj = fix(params[1]+0.5)
                 endelse
              endif else if eadjust eq 2 then begin
                 i_eadj = dm_findbetween(emean0,epadjust)
                 if (~finite(i_eadj)) and (~keyword_set(nomessage)) then $
                    ok = dialog_message('No E='+dm_to_string(epadjust)+' in the original data. Elastic peak position adjustment is skipped.',$
                         dialog_parent=parent,title='Warning: '+open_file[i],/center)
              endif
              if finite(i_el) and finite(i_eadj) then erotate = fix(i_el-i_eadj) $
              else tmp = temporary(erotate)  ;destroy erotate
              if keyword_set(epadcheckfirst) and n_elements(erotate) ne 0 then sch_ead = 0b   
           endelse
        endif
        if n_elements(erotate) ne 0 then begin
           tmp_qty = shift(tmp_qty,erotate,0)
           tmp_err = shift(tmp_err,erotate,0)
        endif
        if sch_tch then begin           ;need to locate time channels
           sch_tch = 0                  ;no need to locate tchan again once this is done
           all_tch = 1                  ;show all time channels flag
           case t_chan of
              1: begin                      ;elastic peak
                 if shutter_close eq 1 then begin
                    sch_tch = 1             ;need to do it again
                    break
                 endif
                 tmp_y = total(tmp_qty,2,/double)
                 dm_gaussfit,tmp_y,params=params,ny=ny,fitnotgood=fitnotgood,debug=debug
                 if fitnotgood then begin
                    if ~keyword_set(nomessage) then ok = dialog_message(["Can't locate the elastic peak from the first file.","All time channels are used instead."],dialog_parent=parent,/center)
                 endif else begin
                    all_tch = 0     ;fitting is good
                    tchans  = round(params[1]-fix(2.5*params[2])+indgen(fix(5.0*params[2])+1))
                    index   = where(tchans lt 0, count)
                    if count ne 0 then tchans[index] = tchans[index]+ny
                    index   = where(tchans ge ny,count)
                    if count ne 0 then tchans[index] = tchans[index]-ny
                    emin    = min(emean[tchans],max=emax)
                    e_range = [emin,emax]
                 endelse
                 end
              2: begin             ;specify energy range
                 if n_elements(e_range) eq 2 then begin
                    if total(finite(e_range)) eq 2 then begin
                       tchans = where((emean ge e_range[0]) and (emean le e_range[1]),count)
                       if count ne 0 then begin
                          all_tch = 0
                          break
                       endif
                    endif
                    if ~keyword_set(nomessage) then ok = dialog_message(['The energy range specified is invalid.','All time channels are used instead.'],dialog_parent=parent,/center)
                 endif
                 end
              else:
           endcase
           if all_tch eq 1 then begin  ;all time channels
              tchans  = 0 & tmp = temporary(tchans) ;get rid of tchans
              emin    = min(emean,max=emax)
              e_range = [emin,emax]
           endif
           ntchan = n_elements(tchans)
           if (sch_tch eq 0) then begin
              if diffetyp eq 1 then begin
                 if ntchan ne 0 then begin
                    if n_elements(qty) gt 1 then begin
                       qty  = qty[tchans,*,*]
                       dqty = dqty[tchans,*,*]
                    endif
                 endif
              endif
              if ntchan ne 0 then begin
                 if n_elements(twid) ne 1 then twid = twid[tchans,*]
                 ewid  = ewid[tchans]
                 emean = emean[tchans]
                 ki_kf = ki_kf[tchans]
              endif
           endif
           if ntchan eq 0 then ntchan = ngtchan0
           nitem = (avgsum eq 1)?1.0:(1.0>ntchan)      ;1 for sum
        endif
        if ntchan ne ngtchan0 then begin
           tmp_qty = tmp_qty[tchans,*]
           tmp_err = tmp_err[tchans,*]
        endif 
        if n_elements(shutter) eq 0 then shutter = 1-shutter_close else shutter = [shutter,1-shutter_close]
        if keyword_set(raw) then begin  ;for dcs_cryst_align, no correction or normalization
           factor  = float(first_mon_sum/mon_sum)/nitem
           tmp_qty = total(tmp_qty,1)*factor
           tmp_dqty = sqrt(total(tmp_err^2,1))*factor    
           if i eq 0 then begin
              qty  = tmp_qty
              dqty = tmp_dqty
              zero_error = factor*zeroerr1
           endif else begin
              qty  = [[qty],[tmp_qty]]
              dqty = [[dqty],[tmp_dqty]]
              zero_error = (zero_error)<(factor*zeroerr1)
           endelse
           if n_elements(ang) eq 0 then $
              ang = [motor_pos[1:6],i,hfield,temp_sample,temp_control,setpoint,mon_sum] $
           else $
              ang = [[ang],[[motor_pos[1:6],i,hfield,temp_sample,temp_control,setpoint,mon_sum]]]
        endif else begin                ;for dcs_mslice
           if shutter_close then begin  ;shutter closed
              n_dkfile = n_dkfile+1     ;record the number of shutter closed files
              if darkcount then begin   ;add to darkcount rate if background keyword is set
                 fb_weight = fb_weight+float(this_duration/3600.0)   ;in hours
                 tmp_qty   = total(tmp_qty,1)
                 if n_elements(bgrate) eq 0 then bgrate=temporary(tmp_qty) $
                 else bgrate = temporary(bgrate)+temporary(tmp_qty)
              endif else begin          ;discard the data if not measuring dark count file
                 if ch_wl eq 0 then $
                    tmp = 'Wavelength is 0 for this file. ' $
                 else $
                    tmp = 'Shutter is closed for this file. '
                 if (~ obj_valid(mesg)) then break      
                 mesg->update,message=temporary(tmp)+'This file is discarded.',title=open_file[i]
                 if ~keyword_set(nomessage) then wait,2
              endelse
           endif else begin                 ;shutter open
              tmp_ind = where(tmp_qty eq 0,tmp_cnt)
              if tmp_cnt ne 0 then begin
                 tmp_err[tmp_ind] = 0.0
                 tmp_ind = 0
              endif 
              if darkcount then begin
                 if (~ obj_valid(mesg)) then break
                 mesg->update,message='Shutter is open. This file is discarded.',title=open_file[i]
                 if ~keyword_set(nomessage) then wait,2
                 continue
              endif
              normalization_values = mon_sum
              if keyword_set(monfit) and (montype eq 1) then begin  ;fit to a gaussian only for bm1
                 dm_gaussfit,mon_counts,x=tmean,params=params,debug=debug,fitnotgood=fitnotgood
                 if (~ fitnotgood) then normalization_values = float(abs(params[0]*params[2])*sqrt(2*!pi)/ch_srdenom) ;integrated intensity ~ monrate* npulse
                                                                                                                      ;divided by ch_srdenom so that mon_counts is time divided                                                                                             
              endif
              ;monrate = float(normalization_values*speriod/this_duration)*1e-6
              if n_elements(monrate) eq 0 then monrate = [normalization_values,speriod,this_duration] $
              else monrate = [[monrate],[[normalization_values,speriod,this_duration]]]
              ;divide (*histPtr).qty and (*histPtr).err by the time channel width
              if keyword_set(diffuse) then begin
                 if n_elements(weight) eq 0 then $
                    weight = normalization_values $
                 else $
                    weight = [weight,normalization_values]
                 if n_elements(ang) eq 0 then $
                    ang = [motor_pos[1:6],i,hfield,temp_sample,temp_control,setpoint,mon_sum] $
                 else $
                    ang = [[ang],[[motor_pos[1:6],i,hfield,temp_sample,temp_control,setpoint,mon_sum]]]
                 tmp_qty = tmp_qty/twid    ;~I * npulse
                 tmp_err = tmp_err/twid 
                 if normalization_values ne 0 then $
                    factor = 1e6/normalization_values $
                 else $
                    factor = 1.0
                 if diffetyp eq 0 then begin        ;single energy
                    if keyword_set(eint_yn) then begin
                       for j=0L,n_elements(tmp_qty[0,*])-1 do begin
                           tmp_qty[*,j] = tmp_qty[*,j]*ewid
                           tmp_err[*,j] = tmp_err[*,j]*ewid 
                       endfor
                    endif else begin
                       factor = factor/nitem
                       nmonrate = n_elements(monrate[0,*])
                       monrate[0,nmonrate-1] = monrate[0,nmonrate-1]*nitem
                    endelse
                    tmp_qty = total(tmp_qty,1)*factor
                    tmp_dqty = sqrt(total(tmp_err^2,1))*factor
                    if keyword_set(eint_yn) then factor = factor*min(abs(ewid))
                    if (i eq 0) or (n_elements(qty) eq 0) then begin
                       qty  = tmp_qty
                       dqty = tmp_dqty
                       zero_error = zeroerr1*factor/max(abs(twid))
                    endif else begin
                       qty  = [[qty],[tmp_qty]]
                       dqty = [[dqty],[tmp_dqty]]
                       zero_error = (zero_error)<(zeroerr1*factor/max(abs(twid)))
                    endelse
                 endif else begin                   ;multiple energy
                    if (i eq 0) or (n_elements(qty) eq 0) then begin
                       qty  = tmp_qty*factor
                       dqty = tmp_err*factor
                       zero_error = zeroerr1*factor/max(abs(twid))
                    endif else begin
                       qty  = [[[qty]],[[tmp_qty*factor]]]
                       dqty = [[[dqty]],[[tmp_err*factor]]]
                       zero_error = (zero_error)<(zeroerr1*factor/max(abs(twid)))
                    endelse
                 endelse
              endif else begin
                 if n_elements(ang) eq 0 then $
                    ang = [motor_pos[1:6],i,hfield,temp_sample,temp_control,setpoint,mon_sum]
                 weight = temporary(weight)+normalization_values
                 qty  = temporary(qty)+tmp_qty/twid
                 dqty = sqrt(temporary(dqty)^2+(tmp_err/twid)^2)
                 zero_error = zeroerr1/max(abs(twid))
              endelse
           endelse
        endelse
        tmp_qty = 0 & tmp_err = 0
        n_read = n_read+1
        if (~ obj_valid(mesg)) then break
    endfor
    
    twid = 0    ;clear memory
    tmp = check_math(mask=32)    ;Clear any floating-point underflows that has occurred
    if obj_valid(mesg) then obj_destroy,mesg
    heap_free,davePtr

    if keyword_set(raw) then begin
       monrate = first_mon_sum
       return
    endif

    ;monitor rate
    if n_elements(monrate) ne 0 then begin
       monrate = float(total(monrate[0,*])*monrate[1,0]/total(monrate[2,*]))*1e-6
    endif

    if darkcount then begin
       if n_elements(bgrate) ne 0 then begin
          dbgrate = sqrt(bgrate)/fb_weight
          bgrate  = bgrate/fb_weight
       endif else begin
          error = 1
          if ~keyword_set(nomessage) then ok = dialog_message('All files are with shutter open.',/error,dialog_parent=parent,/center)
       endelse
    endif else begin
       if n_dkfile ne n_read then begin
          kfactor = ki_kf
          if keyword_set(diffuse) then begin
             if diffetyp eq 0 then begin
                if t_chan eq 2 then begin
                   emean   = mean(emean)
                   kfactor = mean(kfactor)
                endif else begin
                   emean   = 0.0
                   kfactor = 1.0
                endelse
             endif
          endif else begin
             if weight ne 0 then $
                factor = float(1d6/weight) $
             else $
                factor = 1.0
             weight = float(weight)
             qty    = temporary(qty)*factor
             dqty   = temporary(dqty)*factor
             zero_error = zero_error*factor
          endelse
       endif else begin
          error = 1b
          if ~keyword_set(nomessage) then ok = dialog_message('All files are with shutter closed.',/error,dialog_parent=parent,/center)
       endelse
    endelse
end

;this function searchs for value in data, it returns !values.f_nan if value is not among data or in between two numbers of data
;data should be a sorted arrary
function dm_findbetween,data,value
    ind1 = where(data le value,count1)
    ind2 = where(data ge value,count2)
    if count1 ne 0 and count2 ne 0 then begin
       tmp = max(data[ind1],itmp) & ind1 = ind1[itmp[0]]
       tmp = min(data[ind2],itmp) & ind2 = ind2[itmp[0]]
       if abs(value-data[ind1]) le abs(data[ind2]-value) then index = ind1 $
       else index = ind2
    endif else index = !values.f_nan
    return,index   
end

;the following procedure is for loading empty can file for dcs_mslice
pro dcs_mslice::dm_load_dcs_emptycan,open_path,open_file,parent=parent,error=error,comment=comment,$
    qty=qty,dqty=dqty,ang=ang,weight=weight,adddata=adddata,info=info,ftpobj=ftpobj
    if self.samp_typ ge 2 then begin
       e_range = self.e_range
       t_chan  = self.e_bin[self.samp_typ]
       if keyword_set(adddata) and (t_chan ne 0) then t_chan = 2 ;to have consistent time channels
       dm_load_dcsdata,open_path,open_file,ang=ang,qty=qty,dqty=dqty,weight=weight,$
            comment=comment,ewid=ewid,t_chan=t_chan,e_range=e_range,info=info,$
            eint_yn=self.eint_yn,avgsum=self.bin_avgsum,parent=parent,error=error,$
            /diffuse,monfit=self.mon_sumfit[0],title='loading empty can file...',$
            eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
            diffetyp=(self.samp_typ eq 3),emean=emean,montype=self.mon_typ[0],bitflip=self.bitflip,ftpobj=ftpobj,ftpbufferdir=self.dirs[2]
       if (n_elements(e_range) ne 0) and (~ptr_valid(self.dataStrPtr)) then self.e_range = e_range
       ang = reform(ang[0,*])   ;only need a2 angle
    endif else begin
       ang = 0
       dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,parent=parent,info=info,$
            title='loading empty can file...',error=error,monfit=self.mon_sumfit[0],weight=weight,$
            comment=comment,eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
            montype=self.mon_typ[0],bitflip=self.bitflip,ftpobj=ftpobj,ftpbufferdir=self.dirs[2]
    endelse
end

;the following procedure is for loading dark count files for dcs_mslice
pro dcs_mslice::dm_load_dcs_darkcount,open_path,open_file,parent=parent,error=error,comment=comment,$
    bgrate=bgrate,dbgrate=dbgrate,ftpobj=ftpobj
    dm_load_dcsdata,open_path,open_file,bgrate=bgrate,dbgrate=dbgrate,error=error,comment=comment,$
        parent=parent,title='loading dark count file...',bitflip=self.bitflip,ftpobj=ftpobj,ftpbufferdir=self.dirs[2]
end

;the following procedure is for loading detector dependent background file for dcs_mslice
pro dcs_mslice::dm_load_dcs_detbackground,open_path,open_file,parent=parent,error=error,comment=comment,$
    qty=qty,dqty=dqty,ftpobj=ftpobj,weight=weight
    dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,$
        e_range=self.bgeran,avgsum=0,title='loading background file...',comment=comment,$
        parent=parent,error=error,monfit=self.mon_sumfit[0],bitflip=self.bitflip,$
        eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
        montype=self.mon_typ[0],ftpobj=ftpobj,ftpbufferdir=self.dirs[2],weight=weight
    ntch = n_elements(qty[*,0])
    qty  = total(qty,1)
    dqty = total(dqty^2,1)
    qty  = qty/ntch
    dqty = sqrt(dqty)/ntch
end

;the following procedure is for loading time channel background file for dcs_mslice
pro dcs_mslice::dm_load_dcs_tchanbackground,open_path,open_file,parent=parent,error=error,comment=comment,$
    qty=qty,dqty=dqty,emean=emean,ftpobj=ftpobj,weight=weight
    dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,$
        e_range=self.dcor_eqran[0:1],avgsum=0,title='loading background file...',comment=comment,$
        parent=parent,error=error,diffetyp=1,monfit=self.mon_sumfit[0],eadjust=self.eadjust,bitflip=self.bitflip,weight=weight,$
        epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,montype=self.mon_typ[0],emean=emean,ftpobj=ftpobj,ftpbufferdir=self.dirs[2]
end

;the following procedure is for loading vanadium file for dcs_mslice
pro dcs_mslice::dm_load_dcs_vanadium,open_path,open_file,parent=parent,error=error,$
    comment=comment,qty=qty,ei=ei,info=info,ftpobj=ftpobj
    dm_load_dcsdata,open_path,open_file,qty=qty,parent=parent,title='loading vanadium file...',error=error,$
                monfit=self.mon_sumfit[0],comment=comment,info=info,eadjust=self.eadjust,epadjust=self.eadjust_spec,$
                epadcheckfirst=self.eadjust_checkfirst,montype=self.mon_typ[0],einc=ei,bitflip=self.bitflip,ftpobj=ftpobj,ftpbufferdir=self.dirs[2]
end

;the following procedure is for loading dcs data when "Load Data" button is pressed in dcs_mslice
pro dcs_mslice::dm_load_dcs,open_path,open_file,parent=parent,error=error,comment=comment,$
    qty=qty,dqty=dqty,weight=weight,ang=ang,ei=ei,info=info,ewid=ewid,emean=emean,monrate=monrate,$
    kfactor=kfactor,title=title,tmean=tmean,adddata=adddata,ftpobj=ftpobj,temperature=temperature,hfield=hfield,zero_error=zero_error
    if self.samp_typ ge 2 then begin
       e_range = self.e_range
       t_chan  = self.e_bin[self.samp_typ]
       if keyword_set(adddata) and (t_chan ne 0) then t_chan = 2 ;to have consistent time channels
       dm_load_dcsdata,open_path,open_file,ang=ang,qty=qty,dqty=dqty,weight=weight,info=info,comment=comment,$
           monrate=monrate,ewid=ewid,kfactor=kfactor,t_chan=t_chan,e_range=e_range,eint_yn=self.eint_yn,$
           avgsum=self.bin_avgsum,parent=parent,error=error,/diffuse,monfit=self.mon_sumfit[0],title=title,$
           eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
           diffetyp=(self.samp_typ eq 3),emean=emean,montype=self.mon_typ[0],einc=ei,bitflip=self.bitflip,$
           checksum=(self.slice_smooth eq -99),ftpobj=ftpobj,ftpbufferdir=self.dirs[2],shutter=shutter,zero_error=zero_error
       if n_elements(e_range) ne 0 then self.e_range = e_range
       if n_elements(ang) ne 0 then begin
          temperature = ang[8:10,*]
          hfield = reform(ang[7,*])
          ang  = reform(ang[0,*])   ;only need a2 angle
       endif
    endif else begin
       if self.extravaxis_yn[1] or self.extravaxis_yn[2] then begin
          dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,weight=weight,emean=emean,info=info,ewid=ewid,comment=comment,/diffuse,t_chan=0,diffetyp=1,avgsum=self.bin_avgsum,$
              monrate=monrate,kfactor=kfactor,parent=parent,error=error,monfit=self.mon_sumfit[0],title=title,$
              eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
              montype=self.mon_typ[0],ang=ang,tmean=tmean,einc=ei,bitflip=self.bitflip,checksum=(self.slice_smooth eq -99),$
              ftpobj=ftpobj,ftpbufferdir=self.dirs[2],shutter=shutter,zero_error=zero_error
       endif else begin
          dm_load_dcsdata,open_path,open_file,qty=qty,dqty=dqty,weight=weight,emean=emean,info=info,ewid=ewid,comment=comment,$
              monrate=monrate,kfactor=kfactor,parent=parent,error=error,monfit=self.mon_sumfit[0],title=title,$
              eadjust=self.eadjust,epadjust=self.eadjust_spec,epadcheckfirst=self.eadjust_checkfirst,$
              montype=self.mon_typ[0],ang=ang,tmean=tmean,einc=ei,bitflip=self.bitflip,checksum=(self.slice_smooth eq -99),$
              ftpobj=ftpobj,ftpbufferdir=self.dirs[2],shutter=shutter,zero_error=zero_error
       endelse
       if n_elements(ang) ne 0 then begin
          temperature = ang[8:10,*]
          hfield = reform(ang[7,*])
          ang = ang[0]                  ;only need the first a2 angle
       endif
    endelse
    if self.extravaxis_yn[2] then begin ;need magnetic field values
       ind = where(shutter,count)       ;only for shutter open file
       if n_elements(hfield) eq 0 then begin
          if count eq 0 then hfield = !values.f_nan else hfield = fltarr(count)
          hfield[*] = !values.f_nan
       endif
       if total(finite(hfield,/nan)) ne 0 then begin
          repeat begin
              hfield = dm_dialog_input(open_file[ind]+':',/float,default=hfield,dialog_parent=self.tlb,/align_center,info="Please enter the missing magnetic field values in unit of tesla.",title='Missing magnetic field',cancel=cancel)
              if keyword_set(cancel) then message,'Missing magnetic field value. Go to Option->Viewing Axis->Allow Extra Viewing Axis to remove the magnetic field axis.',level=0,/noprint,/noname
          endrep until (total(finite(hfield,/nan)) eq 0)
       endif   
    endif
    if self.samp_typ ne 0 then tmean=0
end