; $Id: dm_gaussfit.pro,v 1.9 2014/11/26 18:59:20 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_gaussfit
;
; PURPOSE:
;  fit using gaussfit
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  October,2017
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;this procedure calls gaussfit with reasonable estimates, suitable for fitting just one gaussian
;Parameters:
;   y:           y-values, an [ny] array
;Keywords:
; input:
;   x:           x-values, an [ny] array, if not present, using the index array
;   nterms:      same as the nterms for gaussfit, default is 4 
;   lorentzian:  if set, fit to a lorentzian function
;   moffat:      if set, fit to a Moffat function
;   fixedwidth:  fixed width for fitting, a float, if fixedwidth=!values.f_nan, the width will not be fixed
;   debug:       if set, show debugging info
;   plot:        if set, the fit is shown as a plot
;   reverse:     if set, use reverse gaussian, ie, A0<0
;   auto:        if set, check both regualr gaussian and reverse gaussian, and use the best fit
; output:
;   fitnotgood:  returns 1 if the fit is determined as no good
;   chisq:       reduced chisq of the fitting
;   ny:          n_elements of y
;   params:      [A0,A1,A2,[A3],[A4],[A5],[A6]] 
;                Gaussian:   y = A0*exp[-(x-A1)^2/(2*A2^2)]+A3+A4*x+A5*x^2
;                Lorentzian: y = A0/(1+(x-A1)^2/A2^2)+A3+A4*x
;                Moffat:     y = A0/(1+(x-A1)^2/A2^2)^A3+A4+A5*x
;   sigma:       1-sigma error of returned params
;   fwhm:        params[2]*sqrt(8.*alog(2)) for gaussian, 2*params[2] for lorentzian, 2*params[2]*sqrt(2^(1/params[3])-1) for moffat
pro dm_gaussfit,y,x=x,ny=ny,params=params,fitnotgood=fitnotgood,fwhm=fwhm,lorentzian=lorentzian,moffat=moffat,fixedwidth=fixedwidth,nterms=nterms,chisq=chisq,sigma=sigma,plot=plot,auto=auto,reverse=reverse,_ref_extra=extra,debug=debug
    if n_elements(nterms) eq 0 then nterms=4          ;default is 4 terms
    if nterms gt 6 or nterms lt 3 then nterms=4
    if keyword_set(lorentzian) then nterms=(5<nterms)
    if keyword_set(moffat) then nterms=(4>nterms)
    if n_elements(fixedwidth) eq 0 then fixedwidth = !values.f_nan
    if arg_present(fitnotgood) then fitnotgood = 1b   ;initialize
    chisq = 0.0 & chisq1 = 0.0
    ny    = n_elements(y)
    ymax  = max(y,imax,min=ymin,subscript_min=imin) 
    if ymax eq ymin then begin
       params = fltarr(nterms)+!values.f_nan
       return
    endif
    if n_elements(x) ne ny then begin
       x  = findgen(ny)
       xw = 1.0
    endif else begin
       xw = x[1:ny-1]-x[0:ny-2] & xw=[xw[0],xw]
    endelse    
    ximax = x[imax] & ximax1 = x[imin] 
    fwhm  = total((y-ymin)*xw)/(ymax-ymin)
    fwhm1 = total((ymax-y)*xw)/(ymax-ymin)
    estimates = fltarr(nterms) 
    if keyword_set(moffat) then estimates[3] = 1.0
    estimates1 = estimates
    estimates[0:2]  = [ymax-([ymin,0])[nterms eq (3+keyword_set(moffat))],ximax,fwhm/((keyword_set(lorentzian) or keyword_set(moffat))?2:sqrt(8.*alog(2)))] 
    estimates1[0:2] = [ymin-([ymax,0])[nterms eq (3+keyword_set(moffat))],ximax1,fwhm1/((keyword_set(lorentzian) or keyword_set(moffat))?2:sqrt(8.*alog(2)))]
    if nterms ne 3+keyword_set(moffat) then begin
       estimates[3+keyword_set(moffat)]  = ymin
       estimates1[3+keyword_set(moffat)] = ymax
    endif
    if finite(fixedwidth) then begin
       parinfo = replicate({fixed:0b},nterms)
       parinfo[2].fixed = 1b
       estimates[2]  = fixedwidth
       estimates1[2] = fixedwidth
    endif
    if keyword_set(auto) or keyword_set(reverse) then begin
       if keyword_set(lorentzian) or keyword_set(moffat) or finite(fixedwidth) then begin
          result1 = call_function('mpfitpeak',x,y,params1,estimates=estimates1,nterms=nterms,chisq=chisq1,sigma=sigma1,dof=-1,lorentzian=lorentzian,moffat=moffat,parinfo=parinfo,_extra=extra)
          chisq1  = chisq1/(1>(ny-nterms+finite(fixedwidth)))    ;reduced chisq
       endif else begin
          result1 = gaussfit(x,y,params1,estimates=estimates1,nterms=nterms,chisq=chisq1,sigma=sigma1,_extra=extra)  ;keep gaussfit for nterms=6
       endelse
    endif
    if ~keyword_set(reverse) or keyword_set(auto) then begin  
       if keyword_set(lorentzian) or keyword_set(moffat) or finite(fixedwidth) then begin
          result = call_function('mpfitpeak',x,y,params,estimates=estimates,nterms=nterms,chisq=chisq,sigma=sigma,dof=-1,lorentzian=lorentzian,moffat=moffat,parinfo=parinfo,_extra=extra)
          chisq  = chisq/(1>(ny-nterms+finite(fixedwidth)))      ;reduced chisq
       endif else begin
          result = gaussfit(x,y,params,estimates=estimates,nterms=nterms,chisq=chisq,sigma=sigma,_extra=extra)
       endelse
    endif
    if keyword_set(reverse) or (keyword_set(auto) and (chisq1 lt chisq/5.0)) then begin ;only when reverse chisq is 5 times smaller
       result = result1
       chisq  = chisq1
       params = params1
       sigma  = sigma1
       ximax  = ximax1
       estimates = estimates1
    endif
    if size(params,/type) ne 4 then params = float(params)  ;mpfitpeak and gaussfit returns two different data type of params
    params[2] = abs(params[2])
    if keyword_set(lorentzian) then fwhm = params[2]*2 $
    else if keyword_set(moffat) then fwhm =  2*params[2]*sqrt(2^(1/params[3])-1) $
    else fwhm = params[2]*sqrt(8.*alog(2))
    if keyword_set(debug) then begin
       print,'estimates=',estimates
       print,'params=',params
       window,/free
       plot,x,y,psym=6,xr=params[1]+[-params[2],params[2]]*3.5,title='reduced chisq='+strtrim(string(chisq),2)
       oplot,x,result
       wait,2 & wdelete,!D.WINDOW
    endif
    ;check the fitting
    if arg_present(fitnotgood) then begin
       if total(finite(params)) ne nterms then fitnotgood = 1b else begin
          minxwid = min(x[1:ny-1]-x[0:ny-2])
          if (abs(params[1]-ximax) gt abs((x[ny-1]-x[0])/4.)) or (params[2] le minxwid/2.) then $
             fitnotgood = 1b $
          else $
             fitnotgood = 0b
       endelse
    endif
    if keyword_set(plot) then begin
       tmp = obj_new('dm_plot',x,y,psym='circle',linestyle='no line')
       tmp->getproperty,xran=xran,yran=yran
       nx = (ny>101)
       xx = xran[0]+findgen(nx)/(nx-1.)*(xran[1]-xran[0])
       ;[A0,A1,A2,[A3],[A4],[A5]] y=A0*exp[-(x-A1)^2/(2*A2^2)]+A3+A4*x+A5*x^2 for gaussian, y=A0/(1+(x-A1)^2/A2^2)+A3+A4*x for lorentzian, y=A0/(1+(x-A1)^2/A2^2)^A3+A4+A5*x for moffat
       if keyword_set(moffat) then yy = params[0]/(1.+((xx-params[1])/params[2])^2)^params[3] $
       else if keyword_set(lorentzian) then yy = params[0]/(1.+((xx-params[1])/params[2])^2) $
       else yy = params[0]*exp(-(xx-params[1])^2/(2*params[2]^2))
       if nterms ge 4+keyword_set(moffat) then yy = yy+params[3+keyword_set(moffat)]
       if nterms ge 5+keyword_set(moffat) then yy = yy+params[4+keyword_set(moffat)]*xx
       if nterms ge 6+keyword_set(moffat) then yy = yy+params[5+keyword_set(moffat)]*xx^2
       tmp->add_plot,xx,yy,linestyle='solid',color='green'
       tmp->draw
    endif
end