; $Id$
;#######################################################################
;
; NAME:
;  dm_dialog_password
;
; PURPOSE:
;  This program opens a dialog window for password input.
;
; CATEGORY:
;  general tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2020
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

pro dm_dialog_password_Exit,tlb,tmp=tmp
    ;do nothing, leave the job to obj_destroy
end

;widget event handler
pro dm_dialog_password_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;object event handler
pro dm_dialog_password::event, event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    widget_control,/hourglass
    ;error handling
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,/center,dialog_parent=self.tlb)
       return
    end
    eventname=widget_info(event.id,/uname)
    case eventname of
       'password':begin
                  case event.type of
                    0:  begin
                        if event.ch eq 10B then begin ;return
                           if strlen(self.input) eq 0 then self.input = ' '
                           widget_control,self.tlb,/destroy
                        endif else begin
                           self.input = strmid(self.input,0,event.offset-1)+string(event.ch)+strmid(self.input,event.offset-1)
                           tmp = string(bytarr(strlen(self.input))+(byte('*'))[0])
                           widget_control,event.id,set_value=tmp,set_text_select=strlen(self.input)
                        endelse
                        end
                    2:  begin
                        self.input = strmid(self.input,0,event.offset)+strmid(self.input,event.offset+event.length)
                        len = strlen(self.input) & tmp = ''
                        if len ne 0 then tmp = string(bytarr(len)+(byte('*'))[0])
                        widget_control,event.id,set_value=tmp,set_text_select=len
                        end
                    else:
                  endcase
                  end
       else:
    endcase
end

;get properties
pro dm_dialog_password::getproperty,pass=pass,input=input
    input = self.input
    if strlen(self.input) ne 0 then pass = (self.password eq self.input)
end

;destroy the object,life cycle method
pro dm_dialog_password::Cleanup
    if widget_info(self.extbase,/valid_id) then widget_control,self.extbase,/destroy
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
end

function dm_dialog_password::Init,password,dialog_parent=dialog_parent
    ;error handler
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,/center,dialog_parent=dialog_parent)
       return, 0
    end
    if n_elements(password) ne 0 then self.password= password[0]
    self.input   = ''
    self.extbase = widget_base(group_leader=dialog_parent,map=0,floating=(n_elements(dialog_parent) ne 0))
    self.tlb     = widget_base(title="Password",/col,group_leader=self.extbase,/modal,/align_center)
    void         = widget_label(self.tlb,value='  Please enter the password:  ',    /align_left)
    void         = widget_text(self.tlb,/all_events,uname='password')
    dm_center_kid,self.tlb,dialog_parent
    widget_control,self.tlb,/realize
    widget_control,void,/input_focus
    widget_control,self.tlb,set_uvalue=self
    xmanager,'dm_dialog_password',self.tlb,cleanup='dm_dialog_password_Exit'
    return,1
end

;class definition
pro dm_dialog_password__define
    void={dm_dialog_password     , $    ;thie object class name
        extbase:               0L, $    ;group leader of the modal tlb
        tlb:                   0L, $    ;top level base
        password:              '', $    ;correct password
        input:                 ''  $    ;typed in password
    }
end

;Usage:
;    result=dm_dialog_password(password)
;Parameter:
;    password:   the correct password
;Return:
; if parameter password is present
;    1:  if input matchs the password
;    0:  if input is incorrect
; if paramter password is not present
;    returns the input password string
function dm_dialog_password,password,dialog_parent=dialog_parent
    tmp = obj_new('dm_dialog_password',password,dialog_parent=dialog_parent)
    tmp->getproperty,pass=pass,input=input
    obj_destroy,tmp
    if n_elements(password) ne 0 then begin
       if (~keyword_set(pass)) and (n_elements(pass) ne 0) then ok = dialog_message("Incorrect password!",/error,/center,dialog_parent=dialog_parent)
       return,keyword_set(pass)
    endif else $
       return,input
end