; $Id: $
;#######################################################################
;
; NAME:
;  dm_4dgrid_bin
;
; PURPOSE:
;  bin 4d volume data according to given x1step, x2step, x3step, and x4step
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   in_x1(2,3,4)step: x1(2,3,4)step
;   x1(2,3,4)dat:     input x1(2,3,4)dat[ndat], output x1(2,3,4)dat[x1(2,3,4)size]
;   idat:             input idat[ndat],         output idat[x1size,x2size,x3size,x4size]
; keyword:
;   avgsum:           -1:weighted mean 0: arithmetic mean 1:sum, default 0
;   weight:           used as weight for avgsum = -1. if not present, use the error bar 
;   bintr:            binning threshold, bins with fewer data points than bintr*(max number of points in a bin) will be discarded
;   conststep:        0: use the average as the resulted binning values(default) 1: constant binning intervals
;   x1(2,3,4)start:   binning starting point, if absent, use the min(x1(2,3,4)dat) as the starting point
;   checkfinite:      flag for checking finite values of idat
;   leave_blank:      if set, leave empty column or rows as they are
;   ewid:             energy width, used when e-integrated intensity option is chose, input  dat=sum(I*ewid), output dat=sum(I*ewid)/sum(ewid)           
;   ierr:             input error bar of idat[ndat], ouput ierr[x1size,x2size,x3size,x4size]
;   zero_error:       if nonzero, use this value as error bar for zero intensities
;   bin_zerroerror:   flag for how to treat zero intenisty error, default 0 (user specify)
;   zoerrestrange:    used when bin_zerroerror=1, the zero intensity error bar statistical estimate range: 0-automatic search till non-zero counts are found; n(>=1)-specified search range 
;   estapplyall:      used when bin_zerroerror=1, if set, zoerrestrange is applied to all intensity error bar statistical estimate
;   foundrange:       output the range when zoerrestrange=0
pro dm_4dgrid_bin,in_x1step,in_x2step,in_x3step,in_x4step,x1dat,x2dat,x3dat,x4dat,idat,ierr=ierr,x1start=x1start,x2start=x2start,x3start=x3start,x4start=x4start,avgsum=avgsum,bintr=bintr,$
    checkfinite=checkfinite,conststep=conststep,ewid=ewid,leave_blank=leave_blank,debug=debug,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,zoerrestrange=zoerrestrange,$
    estapplyall=estapplyall,foundrange=foundrange,group_leader=group_leader

    if n_elements(avgsum) eq 0 then avgsum = 0        ;default using arithmetic mean in binning
    if n_elements(bintr)  eq 0 then bintr = 0
    if n_elements(conststep) eq 0 then conststep = 0  ;default is using average as the binned value
    if keyword_set(conststep) then leave_blank = 1
    if n_elements(zero_error) eq 0 then zero_error = 0.0
    if keyword_set(bin_zeroerror) then begin
       if n_elements(zoerrestrange) eq 0 then zoerrestrange = 0
       if zoerrestrange eq 0 then foundrange = 0
    endif
    if keyword_set(debug) then current = systime(/sec)
    ndat    = n_elements(idat)
    ok_ierr = n_elements(ierr) eq ndat
    ok_eint = n_elements(ewid) eq ndat
    ok_wght = n_elements(weight) eq ndat
    x1_type = size(x1dat,/type)
    x2_type = size(x2dat,/type)
    x3_type = size(x3dat,/type)
    x4_type = size(x4dat,/type)
    i_type  = size(idat,/type)
    if ok_ierr then e_type = size(ierr,/type)
    if ok_wght then w_type = size(weight,/type)
    if ok_eint then avgsum = 1
    if (avgsum eq -1) and (~ok_wght) and (~ok_ierr) then avgsum = 0    ;weighted average requires the weight or corresponding ierr, without it, switching to arithmetic mean
    if keyword_set(checkfinite) then begin
       ind = where(finite(idat),count) 
       if (count ne ndat) and (count ne 0) then begin
          x1dat = x1dat[ind]
          x2dat = x2dat[ind]
          x3dat = x3dat[ind]
          x4dat = x4dat[ind]
          idat  = idat[ind]
          if ok_ierr then ierr = ierr[ind]
          if ok_eint then ewid = ewid[ind]
          if ok_wght then weight = weight[ind]
          ind  = 0 & ndat = count
       endif
    endif
    x1step  = abs(in_x1step)
    x2step  = abs(in_x2step)
    x3step  = abs(in_x3step)
    x4step  = abs(in_x4step)
    x1max   = max(x1dat,min=x1min)
    x2max   = max(x2dat,min=x2min)
    x3max   = max(x3dat,min=x3min)
    x4max   = max(x4dat,min=x4min)
    if n_elements(x1start) eq 1 then begin
       if finite(x1start) then begin
          x1min = x1start
          ind = where(x1dat ge x1start,count)
          if count ne ndat then begin
             x1dat = x1dat[ind]
             x2dat = x2dat[ind]
             x3dat = x3dat[ind]
             x4dat = x4dat[ind]
             idat  = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if n_elements(x2start) eq 1 then begin
       if finite(x2start) then begin
          x2min = x2start
          ind = where(x2dat ge x2start,count)
          if count ne ndat then begin
             x1dat = x1dat[ind]
             x2dat = x2dat[ind]
             x3dat = x3dat[ind]
             x4dat = x4dat[ind]
             idat  = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if n_elements(x3start) eq 1 then begin
       if finite(x3start) then begin
          x3min = x3start
          ind = where(x3dat ge x3start,count)
          if count ne ndat then begin
             x1dat = x1dat[ind]
             x2dat = x2dat[ind]
             x3dat = x3dat[ind]
             x4dat = x4dat[ind]
             idat  = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if n_elements(x4start) eq 1 then begin
       if finite(x4start) then begin
          x4min = x4start
          ind = where(x4dat ge x4start,count)
          if count ne ndat then begin
             x1dat = x1dat[ind]
             x2dat = x2dat[ind]
             x3dat = x3dat[ind]
             x4dat = x4dat[ind]
             idat  = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if (avgsum eq -1) and (~ok_wght) then begin
       weight = ierr
       ind = where(ierr eq 0,complement=ind1,count)
       if count ne 0 then begin
          tmp = min(abs(ierr[ind1]))
          if zero_error ne 0 then tmp = (tmp)<(zero_error)
          weight[ind] = tmp        ;make sure no ierr=0 exists, otherwise replace them with smallest of nonzero value
       endif
       weight = 1./weight^2
       ok_wght = 1b
       ind = 0 & ind1 = 0
    endif
    if (x1step eq 0) then x1step = ((x1max eq x1min)?(0.1):((x1max-x1min)/100.))
    if (x2step eq 0) then x2step = ((x2max eq x2min)?(0.1):((x2max-x2min)/100.))
    if (x3step eq 0) then x3step = ((x3max eq x3min)?(0.1):((x3max-x3min)/100.))
    if (x4step eq 0) then x4step = ((x4max eq x4min)?(0.1):((x4max-x4min)/100.))
    x1max   = x1max+x1step/2.
    x1min   = x1min-x1step/2.
    x2max   = x2max+x2step/2.
    x2min   = x2min-x2step/2.
    x3max   = x3max+x3step/2.
    x3min   = x3min-x3step/2.
    x4max   = x4max+x4step/2.
    x4min   = x4min-x4step/2.
    x1size  = ceil((x1max-x1min)/x1step)
    x2size  = ceil((x2max-x2min)/x2step)
    x3size  = ceil((x3max-x3min)/x3step)
    x4size  = ceil((x4max-x4min)/x4step)
    tmp_x1  = dblarr(x1size) & num_x1 = lonarr(x1size)
    tmp_x2  = dblarr(x2size) & num_x2 = lonarr(x2size)
    tmp_x3  = dblarr(x3size) & num_x3 = lonarr(x3size)
    tmp_x4  = dblarr(x4size) & num_x4 = lonarr(x4size)
    tmp_i   = dblarr(x1size,x2size,x3size,x4size) & num_i = lonarr(x1size,x2size,x3size,x4size)
    if ok_eint then tmp_w = dblarr(x1size,x2size,x3size,x4size)
    if ok_ierr then tmp_e = dblarr(x1size,x2size,x3size,x4size)
    if ok_wght then tmp_m = dblarr(x1size,x2size,x3size,x4size)
  
    ;use histogram function to do the 4d-binning
    ;first construct a 1-d array for histogram function
    use64 = (x1size*x2size*x3size*x4size) lt 0 
    tmp_x4dat = long((x4dat-x4min)/x4step)
    if ~keyword_set(conststep) then begin
       num_x4 = histogram(tmp_x4dat,binsize=1L,min=0L,max=x4size-1L,reverse_indices=rindx)
       for i=0L,n_elements(num_x4)-1L do begin
           if num_x4[i] eq 0 then continue
           tmp_x4[i] = total(x4dat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
       endfor
    endif
    rindx = 0 & x4dat = 0
    if use64 then begin
       tmp = long64(x3size)*long64(temporary(tmp_x4dat))
       tmpmax = long64(x1size)*long64(x2size)*long64(x3size)*long64(x4size)-(1LL) 
    endif else begin
       tmp = long(x3size)*long(temporary(tmp_x4dat))
       tmpmax = long(x1size)*long(x2size)*long(x3size)*long(x4size)-(1L) 
    endelse
  
    tmp_x3dat = long((x3dat-x3min)/x3step)
    if ~keyword_set(conststep) then begin
       num_x3 = histogram(tmp_x3dat,binsize=1L,min=0L,max=x3size-1L,reverse_indices=rindx)
       for i=0L,n_elements(num_x3)-1L do begin
           if num_x3[i] eq 0 then continue
           tmp_x3[i] = total(x3dat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
       endfor
    endif
    rindx = 0 & x3dat = 0
    if use64 then tmp = long64(x2size)*(temporary(tmp)+long64(temporary(tmp_x3dat))) $
    else tmp = long(x2size)*(temporary(tmp)+long(temporary(tmp_x3dat)))
    
    tmp_x2dat = long((x2dat-x2min)/x2step)
    if ~keyword_set(conststep) then begin
      num_x2 = histogram(tmp_x2dat,binsize=1L,min=0L,max=x2size-1L,reverse_indices=rindx)
      for i=0L,n_elements(num_x2)-1L do begin
        if num_x2[i] eq 0 then continue
        tmp_x2[i] = total(x2dat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
      endfor
    endif
    rindx = 0 & x2dat = 0
    if use64 then tmp = long64(x1size)*(temporary(tmp)+long64(temporary(tmp_x2dat))) $
    else tmp = long(x1size)*(temporary(tmp)+long(temporary(tmp_x2dat)))
    
    tmp_x1dat = long((x1dat-x1min)/x1step)
    if ~keyword_set(conststep) then begin
       num_x1 = histogram(tmp_x1dat,binsize=1L,min=0L,max=x1size-1L,reverse_indices=rindx)
       for i=0L,n_elements(num_x1)-1L do begin
           if num_x1[i] eq 0 then continue
           tmp_x1[i] = total(x1dat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
       endfor
    endif
    rindx = 0 & x1dat = 0
    if use64 then tmp = temporary(tmp)+long64(temporary(tmp_x1dat)) $
    else tmp = temporary(tmp)+long(temporary(tmp_x1dat))
   
    num_i = histogram(temporary(tmp),binsize=1,min=0,max=tmpmax,reverse_indices=rindx)
    num_i = reform(num_i,x1size,x2size,x3size,x4size,/overwrite)
    if ok_ierr then begin
       ierr = ierr^2
       if (zero_error ne 0) and (avgsum eq -1) then begin
          ind = where(ierr eq 0,cnt)
          if cnt ne 0 then zero_error = zero_error*min(weight[ind])
       endif
    endif
    n_rindx = n_elements(rindx)
    if avgsum eq -1 then begin ;weighted mean
       if keyword_set(bin_zeroerror) then tmp_ierr = ierr  ;need another copy
       for i=0LL,n_elements(num_i)-1LL do begin
           if num_i[i] eq 0 then continue
           if keyword_set(bin_zeroerror) then begin  ;special care for zero intensities
              tmp_row1 = long64(i mod x1size)
              tmp_row2 = long64((i/x1size) mod x2size)
              tmp_row3 = long64((i/x1size/x2size) mod x3size)
              tmp_row4 = long64(i/x1size/x2size/x3size)
              tmp1     = 0d 
              tmp2     = 0d
              if keyword_set(estapplyall) and ((zoerrestrange) gt 0) then begin
                 for row1=(0LL)>(tmp_row1-zoerrestrange),(tmp_row1+zoerrestrange)<(x1size-1) do begin
                     for row2=(0LL)>(tmp_row2-zoerrestrange),(tmp_row2+zoerrestrange)<(x2size-1) do begin
                         for row3=(0LL)>(tmp_row3-zoerrestrange),(tmp_row3+zoerrestrange)<(x3size-1) do begin
                             for row4=(0LL)>(tmp_row4-zoerrestrange),(tmp_row4+zoerrestrange)<(x4size-1) do begin
                                 tmp_in = ((row4*x3size+row3)*x2size+row2)*x1size+row1
                                 if rindx[tmp_in] lt n_rindx then begin
                                    if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                       tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                       tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                    endif else begin
                                       tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                       tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                    endelse
                                 endif
                             endfor
                         endfor
                     endfor
                 endfor
                 if tmp2 gt 0 then begin
                    tmp = tmp1/tmp2
                    ierr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp
                 endif
              endif else begin
                 ind = where(tmp_ierr[rindx[rindx[i]:(rindx[i+1]-1)]] eq 0,cnt)
                 if cnt ne 0 then begin
                    if zoerrestrange eq 0 then begin
                       ;expanding the range gradually until finding non-zero intensities
                       found   = 0b
                       tmp_ran = 0l
                       tmp_max = max([tmp_row1,x1size-tmp_row1,tmp_row2,x2size-tmp_row2,tmp_row3,x3size-tmp_row3,tmp_row4,x4size-tmp_row4])
                       while (~found) do begin
                             for row1=(0LL)>(tmp_row1-tmp_ran),(tmp_row1+tmp_ran)<(x1size-1) do begin
                                 for row2=(0LL)>(tmp_row2-tmp_ran),(tmp_row2+tmp_ran)<(x2size-1) do begin
                                     for row3=(0LL)>(tmp_row3-tmp_ran),(tmp_row3+tmp_ran)<(x3size-1) do begin
                                         for row4=(0LL)>(tmp_row4-tmp_ran),(tmp_row4+tmp_ran)<(x4size-1) do begin
                                             if max(abs([row1-tmp_row1,row2-tmp_row2,row3-tmp_row3,row4-tmp_row4])) eq tmp_ran then begin
                                                tmp_in = ((row4*x3size+row3)*x2size+row2)*x1size+row1
                                                if rindx[tmp_in] lt n_rindx then begin
                                                   if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                                      tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                                      tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)                           
                                                   endif else begin
                                                      tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                                      tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                                   endelse
                                                endif
                                             endif
                                         endfor
                                     endfor
                                 endfor
                             endfor
                             if tmp1 gt 0 then begin
                                tmp = tmp1/tmp2
                                if keyword_set(estapplyall) then ierr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp $
                                else ierr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                                found = 1b
                             endif
                             tmp_ran = tmp_ran+1
                             if tmp_ran gt tmp_max then found = 1b
                       endwhile
                       foundrange = foundrange>(tmp_ran-1)
                    endif else begin
                       for row1=(0LL)>(tmp_row1-zoerrestrange),(tmp_row1+zoerrestrange)<(x1size-1) do begin
                           for row2=(0LL)>(tmp_row2-zoerrestrange),(tmp_row2+zoerrestrange)<(x2size-1) do begin
                               for row3=(0LL)>(tmp_row3-zoerrestrange),(tmp_row3+zoerrestrange)<(x3size-1) do begin
                                   for row4=(0LL)>(tmp_row4-zoerrestrange),(tmp_row4+zoerrestrange)<(x4size-1) do begin
                                       tmp_in = ((row4*x3size+row3)*x2size+row2)*x1size+row1
                                       if rindx[tmp_in] lt n_rindx then begin
                                          if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                             tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                             tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                          endif else begin
                                             tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                             tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                          endelse
                                       endif
                                   endfor
                               endfor
                           endfor
                       endfor
                       if tmp2 gt 0 then begin
                          tmp = tmp1/tmp2
                          ierr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                       endif 
                    endelse
                 endif
              endelse
           endif
           tmp_i[i] = total(idat[rindx[rindx[i]:(rindx[i+1]-1)]]*weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
           tmp_e[i] = total(ierr[rindx[rindx[i]:(rindx[i+1]-1)]]*(weight[rindx[rindx[i]:(rindx[i+1]-1)]])^2,/double)
           tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
       endfor
       if keyword_set(bin_zeroerror) then tmp_ierr = 0  ;release the memory
    endif else begin
       for i=0LL,n_elements(num_i)-1LL do begin
           if num_i[i] eq 0 then continue
           tmp_i[i] = total(idat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
           if ok_ierr then tmp_e[i] = total(ierr[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
           if ok_eint then tmp_w[i] = total(ewid[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
           if ok_wght then tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
       endfor
    endelse
    rindx = 0
    if keyword_set(conststep) then begin  ;use reform to avoid error when any xnsize=1
       if use64 then num_x1 = long64(total(reform(total(reform(total(num_i,2),x1size,x3size,x4size),2),x1size,x4size),2)) else num_x1 = long(total(reform(total(reform(total(num_i,2),x1size,x3size,x4size),2),x1size,x4size),2))
       if use64 then num_x2 = long64(total(reform(total(reform(total(num_i,3),x1size,x2size,x4size),1),x2size,x4size),2)) else num_x2 = long(total(reform(total(reform(total(num_i,3),x1size,x2size,x4size),1),x2size,x4size),2)) 
       if use64 then num_x3 = long64(total(reform(total(reform(total(num_i,4),x1size,x2size,x3size),1),x2size,x3size),1)) else num_x3 = long(total(reform(total(reform(total(num_i,4),x1size,x2size,x3size),1),x2size,x3size),1))
       if use64 then num_x4 = long64(total(reform(total(reform(total(num_i,1),x2size,x3size,x4size),1),x3size,x4size),1)) else num_x4 = long(total(reform(total(reform(total(num_i,1),x2size,x3size,x4size),1),x3size,x4size),1))
    endif

    idat = 0
    if ok_ierr then ierr = 0

    indx1 = where(num_x1 ne 0,countx1,complement=indx11,ncomplement=countx11)
    indx2 = where(num_x2 ne 0,countx2,complement=indx21,ncomplement=countx21)
    indx3 = where(num_x3 ne 0,countx3,complement=indx31,ncomplement=countx31)
    indx4 = where(num_x4 ne 0,countx4,complement=indx41,ncomplement=countx41)

    if keyword_set(leave_blank) then begin
       if countx11 ne 0 then begin
          num_x1[indx11]=1.0 & tmp_x1[indx11]=x1min+indx11*x1step
       endif
       if countx21 ne 0 then begin
          num_x2[indx21]=1.0 & tmp_x2[indx21]=x2min+indx21*x2step
       endif
       if countx31 ne 0 then begin
          num_x3[indx31]=1.0 & tmp_x3[indx31]=x3min+indx31*x3step
       endif
       if countx41 ne 0 then begin
          num_x4[indx41]=1.0 & tmp_x4[indx41]=x4min+indx41*x4step
       endif
    endif else begin
       if (countx1 ne 0) and (countx11 ne 0) then begin
          tmp_x1 = tmp_x1[indx1]   & num_x1 = num_x1[indx1]
          tmp_i = tmp_i[indx1,*,*,*] & num_i = num_i[indx1,*,*,*]
          if ok_ierr then tmp_e = tmp_e[indx1,*,*,*]
          if ok_eint then tmp_w = tmp_w[indx1,*,*,*]
          if ok_wght then tmp_m = tmp_m[indx1,*,*,*]
       endif
       if (countx2 ne 0) and (countx21 ne 0) then begin
          tmp_x2 = tmp_x2[indx2]   & num_x2 = num_x2[indx2]
          tmp_i = tmp_i[*,indx2,*,*] & num_i = num_i[*,indx2,*,*]
          if ok_ierr then tmp_e = tmp_e[*,indx2,*,*]
          if ok_eint then tmp_w = tmp_w[*,indx2,*,*]
          if ok_wght then tmp_m = tmp_m[*,indx2,*,*]
       endif
       if (countx3 ne 0) and (countx31 ne 0) then begin
          tmp_x3 = tmp_x3[indx3]   & num_x3 = num_x3[indx3]
          tmp_i = tmp_i[*,*,indx3,*] & num_i = num_i[*,*,indx3,*]
          if ok_ierr then tmp_e = tmp_e[*,*,indx3,*]
          if ok_eint then tmp_w = tmp_w[*,*,indx3,*]
          if ok_wght then tmp_m = tmp_m[*,*,indx3,*]
       endif
       if (countx4 ne 0) and (countx41 ne 0) then begin
          tmp_x4 = tmp_x4[indx4]   & num_x4 = num_x4[indx4]
          tmp_i = tmp_i[*,*,*,indx4] & num_i = num_i[*,*,*,indx4]
          if ok_ierr then tmp_e = tmp_e[*,*,*,indx4]
          if ok_eint then tmp_w = tmp_w[*,*,*,indx4]
          if ok_wght then tmp_m = tmp_m[*,*,*,indx4]
       endif
    endelse
    indx1=-1 & indx11=-1 & indx2=-1 & indx21=-1 & indx3=-1 & indx31=-1 & indx4=-1 & indx41=-1
    if keyword_set(conststep) then begin
       x1dat = x1min+(findgen(x1size)+0.5)*x1step
       x2dat = x2min+(findgen(x2size)+0.5)*x2step
       x3dat = x3min+(findgen(x3size)+0.5)*x3step
       x4dat = x4min+(findgen(x4size)+0.5)*x4step
       tmp_i = tmp_i[0:(x1size-1),0:(x2size-1),0:(x3size-1),0:(x4size-1)]
       num_i = num_i[0:(x1size-1),0:(x2size-1),0:(x3size-1),0:(x4size-1)]
       if ok_ierr then tmp_e = tmp_e[0:(x1size-1),0:(x2size-1),0:(x3size-1),0:(x4size-1)]
       if ok_eint then tmp_w = tmp_w[0:(x1size-1),0:(x2size-1),0:(x3size-1),0:(x4size-1)]
       if ok_wght then tmp_m = tmp_m[0:(x1size-1),0:(x2size-1),0:(x3size-1),0:(x4size-1)]
    endif else begin
       x1dat = temporary(tmp_x1)/temporary(num_x1)
       x2dat = temporary(tmp_x2)/temporary(num_x2)
       x3dat = temporary(tmp_x3)/temporary(num_x3)
       x4dat = temporary(tmp_x4)/temporary(num_x4)
    endelse
    idat = temporary(tmp_i)
    if ok_wght then weight = temporary(tmp_m)
    if ok_ierr then begin
       if avgsum ne -1 then begin
          ierr = sqrt(temporary(tmp_e))
          if n_elements(zero_error) ne 0 then begin
             ind = where(ierr eq 0,count)
             if count ne 0 then ierr[ind] = zero_error
             ind = 0
          endif
       endif else $
          ierr = fltarr(size(idat,/dimension))
    endif
    index1 = where(num_i ne 0,count1,complement=index0,ncomplement=count0)
    if (count1 ne 0) then begin
       if ok_eint then begin
          idat[index1] = idat[index1]/tmp_w[index1]
          if ok_ierr then ierr[index1] = ierr[index1]/tmp_w[index1]
       endif else if avgsum eq 0 then begin     ;arithmetic mean   
          idat[index1] = idat[index1]/num_i[index1]
          if ok_ierr then ierr[index1] = ierr[index1]/num_i[index1]
       endif else if avgsum eq -1 then begin    ;weighted mean
          idat[index1] = idat[index1]/weight[index1]
          ierr[index1] = sqrt(tmp_e[index1])/weight[index1]
          if (zero_error ne 0) then begin
             ind = where(ierr[index1] eq 0,count)
             if count ne 0 then ierr[index1[ind]] = zero_error/weight[index1[ind]]
          endif 
          tmp_e = 0
       endif
       if (bintr gt 0.0) and (bintr lt 1.0) then begin
          min_num_i = bintr*(max(num_i)>1)
          ind = where(num_i lt min_num_i,ct1)
          if ct1 gt 0 then begin
             idat[ind] = !values.f_nan
             if ok_ierr then ierr[ind] = !values.f_nan
          endif
       endif
    endif
    if (count0 ne 0) then begin
       idat[index0] = !values.f_nan
       if ok_ierr then ierr[index0] = !values.f_nan
    endif
    if size(x1dat,/type) ne x1_type then x1dat = fix(x1dat,type=x1_type)
    if size(x2dat,/type) ne x2_type then x2dat = fix(x2dat,type=x2_type)
    if size(x3dat,/type) ne x3_type then x3dat = fix(x3dat,type=x3_type)
    if size(x4dat,/type) ne x4_type then x4dat = fix(x4dat,type=x4_type)
    if size(idat,/type)  ne i_type  then idat  = fix(idat,type=i_type)
    if ok_ierr then begin
       if size(ierr,/type) ne e_type then ierr = fix(ierr,type=e_type)
    endif
    if ok_wght then begin
       if size(weight,/type) ne w_type then weight = fix(weight,type=w_type)
    endif
    if keyword_set(debug) then print,'4dgrid_bin finished in ',systime(/sec)-current,' secs.'
end