; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEwriteASCII
;
; PURPOSE:
;   File writer for saving datasets as ASCII files
;
; CATEGORY:
;   Input Output
;
; SUPERCLASSES:
;   IDLitWriter
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEwriteASCII::Init
;
; METHODS:
;   SetData
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; October 2006
;-
;###############################################################################


;===============================================================================
; DAVEwriteASCII::SetData
; 
; PURPOSE:
;   Responsible for actually writing the data to file
;
; PARAMETERS:
;   oData [in] - the data to be saved
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Successful Write
;    0 - Failure.
function DAVEwriteASCII::SetData, oData
compile_opt idl2

myTitle = 'Save as ASCII text file'
if (~obj_valid(oData) || n_elements(oData) lt 1) then begin
    self->SignalError,['No selected data','Cannot proceed with save!'] $
                        ,title= myTitle, severity=2
    return, 0
endif

if (self->GetFilename() eq '') then begin
    self->SignalError,['No output filename specified','Cannot proceed with save!'] $
                        ,title=myTitle, severity=2
    return, 0
endif


if (self->writeData(oData)) then  return, 1 $
else return, 0

end


;===============================================================================
; DAVEwriteASCII::writeData
; 
; PURPOSE:
;   Virtual function. Responsible for writing the appropriate contents
;   of the dataset to the ASCII file. To be implimented by the derived
;   classes.
;
; PARAMETERS:
;   oData [in] - The data object to be written
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Valid
;    0 - Invalid.
function DAVEwriteASCII::writeData, oData
compile_opt idl2

; This is a virtual function that must be implimented by derived classes
return, 1

end


;===============================================================================
; DAVEwriteASCII::isValid
; 
; PURPOSE:
;   Validate data object
;
; PARAMETERS:
;   oData [in] - The data object to be validated
;
;   validType [in] - The appropriate type
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Valid
;    0 - Invalid.
function DAVEwriteASCII::isValid, oData, validType
compile_opt idl2

;
if (~obj_valid(oData)) then return, 0

; Must be of type DAVE1DATASET
oData->getProperty, type=type
return, type eq validType

end


;===============================================================================
; DAVEwriteASCII::datasetType
; 
; PURPOSE:
;   Return the dataset type for the input dataset
;
; PARAMETERS:
;   oData [in] - The data object whose type is required
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Valid
;    0 - Invalid.
function DAVEwriteASCII::datasetType, oData
compile_opt idl2

;
if ((n_elements(oData) eq 0) || ~obj_valid(oData) || ~obj_isa(oData,'IDLitData')) then return, ''

oData->getProperty, type=type
return, type 

end


;===============================================================================
; DAVEwriteASCII::getDAVEPtr
; 
; PURPOSE:
;   Private function for converting an object hierarchy consisting of
;   IDLitDataContainer and/or IDLitData ojects into a davePtr
;   structure.
;
; PARAMETERS
;   davePtr [out] - The davePtr structure.
;
;   oData [in] - The data object containing data to be saved
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - If successful.
;    0 - If unsuccessful.
;
;function DAVEwriteASCII::getDavePtr, davePtr, oData
;compile_opt idl2
;
;retVal = 1
;myTitle = 'Save as DAVE 1.x file'
;
;
;;; Validate data object
;;if (~self->isValid(oData,'DAVE1DATASET')) then  begin
;;    self->SignalError,['Invalid dataset!','Cannot write selected dataset'] $
;;                        ,severity=2
;;    return, 0
;;endif
;
;switch self->datasetType(oData) of
;   'DAVE1DATASET': begin
;
;      ;; Get the top level contents of dataset
;      cont = oData->Get(/all,count=cnt)
;      if (cnt lt 1) then  begin
;         self->SignalError,['Empty dataset!','Cannot write selected dataset'] $
;                            ,severity=2
;         return, 0
;      endif
;      
;      for i=0,cnt-1 do begin
;         if (retVal eq 0) then continue
;         
;         cont[i]->getProperty, type=type
;         ;;    help,'****** type=',type
;         case type of
;            'DAVE1INST': begin
;               res = cont[i]->getData(inst)
;               ;;            help,inst
;            end
;            
;            'DAVE1HISTORY': begin
;               res = cont[i]->getData(trmt)
;                                ;            help,trmt
;            end
;            
;            'DESCRIPTR': begin
;               obj = cont[i]->getByName('Value')
;               if ((retVal = obj_valid(obj)) eq 0) then break
;               obj->getProperty, name=dptrName
;               void = obj->getData(dptrQty)
;               obj = cont[i]->getByName('Description')
;               void = obj->getData(dptrDesc)
;               obj = cont[i]->getByName('Error')
;               void = obj->getData(dptrErr)
;               obj = cont[i]->getByName('Units')
;               void = obj->getData(dptrUnits)
;            end
;
;            'DAVE1COMMONSTR': begin
;               ;; 
;               obj = cont[i]->getByName('2') ; the dependent axis
;               if ((retVal = obj_valid(obj)) eq 0) then break
;               void = obj->getData(qty)
;               if ((retVal =obj->GetMetaData('Long_name',qtyLabel)) eq 0 ) then qtyLabel=''
;               if ((retVal =obj->GetMetaData('Units',qtyUnits)) eq 0 ) then qtyUnits=''
;               if ((retVal =obj->GetMetaData('Axes',axes)) eq 0 ) then axes=''
;               
;               obj = cont[i]->getByName('3') ; the error in the dependent axis
;               if (obj_valid(obj)) then $
;                  void = obj->getData(err)
;               
;               obj = cont[i]->getByName('0') ; first independent axis
;               if (obj_valid(obj)) then begin
;                  void = obj->getData(xdata)
;                  if ((retVal =obj->GetMetaData('Long_name',xLabel)) eq 0 ) then xLabel=''
;                  if ((retVal =obj->GetMetaData('Units',xUnits)) eq 0 ) then xUnits=''
;                  if ((retVal =obj->GetMetaData('Distribution',xType)) eq 0 ) then xType='POINTS'
;                  ;;            if (strcmp(xType,'HISTOGRAM',/fold_case)) then xType = 'HISTO' 
;               endif
;               
;               if (n_elements(axes) eq 2) then begin
;                  obj = cont[i]->getByName('1') ; second independent axis
;                  if (obj_valid(obj)) then begin
;                     void = obj->getData(ydata)
;                     if ((retVal =obj->GetMetaData('Long_name',yLabel)) eq 0 ) then yLabel=''
;                     if ((retVal =obj->GetMetaData('Units',yUnits)) eq 0 ) then yUnits=''
;                     if ((retVal =obj->GetMetaData('Distribution',yType)) eq 0 ) then yType='POINTS'
;                                ;if (strcmp(yType,'HISTOGRAM',/fold_case)) then yType = 'HISTO' 
;                  endif
;               endif
;            end
;            
;            'DAVE1SPECIFICPTR': begin
;               retVal = self->writeSpec(specificStr, cont[i])
;            end
;            
;            else:
;         endcase
;      endfor
;
;      break
;   end
;   
;   'ASCIISPE': 
;   'ASCIIGRP': 
;   'ASCIICOL': begin
;
;      ;; Get the top level contents of dataset
;      void = oData->Get(/all,count=cnt)
;      if (cnt lt 1) then  begin
;         self->SignalError,['Empty dataset!','Cannot write selected dataset'] $
;                            ,severity=2
;         return, 0
;      endif
;
;      obj = oData->getByName('2') ; the dependent axis
;      if ((retVal = obj_valid(obj)) eq 0) then break
;      void = obj->getData(qty)
;      if ((retVal =obj->GetMetaData('Long_name',qtyLabel)) eq 0 ) then qtyLabel=''
;      if ((retVal =obj->GetMetaData('Units',qtyUnits)) eq 0 ) then qtyUnits=''
;      if ((retVal =obj->GetMetaData('Axes',axes)) eq 0 ) then axes=''
;      
;      obj = oData->getByName('3') ; the error in the dependent axis
;      if (obj_valid(obj)) then $
;         void = obj->getData(err)
;      
;      obj = oData->getByName('0') ; first independent axis
;      if (obj_valid(obj)) then begin
;         void = obj->getData(xdata)
;         if ((retVal =obj->GetMetaData('Long_name',xLabel)) eq 0 ) then xLabel=''
;         if ((retVal =obj->GetMetaData('Units',xUnits)) eq 0 ) then xUnits=''
;         if ((retVal =obj->GetMetaData('Distribution',xType)) eq 0 ) then xType='POINTS'
;         ;;            if (strcmp(xType,'HISTOGRAM',/fold_case)) then xType = 'HISTO' 
;      endif
;      
;      if (n_elements(axes) eq 2) then begin
;         obj = oData->getByName('1') ; second independent axis
;         if (obj_valid(obj)) then begin
;            void = obj->getData(ydata)
;            if ((retVal =obj->GetMetaData('Long_name',yLabel)) eq 0 ) then yLabel=''
;            if ((retVal =obj->GetMetaData('Units',yUnits)) eq 0 ) then yUnits=''
;            if ((retVal =obj->GetMetaData('Distribution',yType)) eq 0 ) then yType='POINTS'
;            ;;if (strcmp(yType,'HISTOGRAM',/fold_case)) then yType = 'HISTO' 
;         endif
;      endif
;      
;      break
;   end
;
;   else: begin
;      self->SignalError,['Invalid dataset!','Cannot write selected dataset'] $
;                         ,severity=2
;      return, 0
;   end
;
;endswitch
;
;
;
;
;if (retVal ne 0) then $
;  retVal = create_dave_pointer(davePtr $
;                           ,instrument = inst	$
;                           ,qty = qty $
;                           ,qtunits = qtyUnits $
;                           ,qtlabel = qtyLabel $
;                           ,err = err $
;                           ,xvals = xdata $
;                           ,xtype = xtype $
;                           ,xunits = xunits $
;                           ,xlabel = xlabel $
;                           ,yvals = ydata $
;                           ,ytype = ytype $
;                           ,yunits = yunits $
;                           ,ylabel = ylabel $
;                           ,specificstr = specificstr $
;                           ,treatment = trmt $
;                           ,dname = dptrName $
;                           ,dunits = dptrUnits $
;                           ,dlegend = dptrDesc $
;                           ,dqty = dptrQty $
;                           ,derr = dptrErr $
;                           ,ermsg = errmsg $
;                          )
;
;if (retVal eq 0) then begin
;    ;; cleanup before returning
;   if (n_elements(specificStr) gt 0) then heap_free, specificStr
;   if (ptr_valid(davePtr)) then heap_free, davePtr
;endif
;
;return, retVal
;
;end



;===============================================================================
; DAVEwriteASCII::WriteSpec
; 
; PURPOSE:
;   Convert contents of an IDLitData/Container object into a data structure.
;
; PARAMETERS
;   dataStr [out] - The data structure
;
;   obj [in] - The data object
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - success
;    0 - failure
;
;function DAVEwriteASCII::WriteSpec, dataStr, obj
;compile_opt hidden
;
;if (~obj_valid(obj)) then return, 0
;
;obj->getProperty, type=type, name=tag
;
;switch type of
;    '': break
;
;    'STRUCT':
;    'DAVE1SPECIFICPTR':
;    'POINTER': begin
;        if (obj_isa(obj,'IDL_CONTAINER')) then begin
;            cont = obj->get(/all,count=cnt)
;            if (cnt eq 0) then break 
;            for i = 0,cnt-1 do retVal = self->WriteSpec(dataStr,cont[i])
;            
;            
;            ;; store in heap if required
;            tmpStr = (type eq 'POINTER')? ptr_new(dataStr) : dataStr
;
;            if (type ne 'DAVE1SPECIFICPTR') then $
;              dataStr = (n_elements(dataStr) eq 0)? $
;                        create_struct(tag,tmpStr) : $
;                        create_struct(dataStr,tag,tmpStr)
;            
;            
;        endif else begin
;            if (~obj->getData(value)) then break
;            if (type ne 'STRUCT') then  $
;              value = ptr_new(value)
;            
;            dataStr = (n_elements(dataStr) eq 0)? $
;                      create_struct(tag,value) : $
;                      create_struct(dataStr,tag,value)
;            
;        endelse
;        
;        break
;    end
;    
;    else: begin
;        if (obj_isa(obj,'IDLITDATA')) then begin        
;            if (~obj->getData(value)) then break
;            dataStr = (n_elements(dataStr) eq 0)? $
;                      create_struct(tag,value) : $
;                      create_struct(dataStr,tag,value)
;        endif else if (obj_isa(obj,'IDL_CONTAINER')) then begin
;            cont = obj->get(/all,count=cnt)
;            for i = 0,cnt-1 do retVal = self->WriteSpec(dataStr,cont[i])
;        endif
;        
;        break
;    end
;    
;endswitch
;
;return, 1
;end


;===============================================================================
; DAVEwriteASCII::GetProperty
; 
; PURPOSE:
;   Get method for class
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVEwriteASCII::GetProperty, asciiType=asciiType, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(asciiType)) then begin
    asciiType =  self.asciiType
endif

if(n_elements(etc) gt 0) then $
  self->IDLitWriter::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEwriteASCII::SetProperty
; 
; PURPOSE:
;   Set method for class
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVEwriteASCII::SetProperty, asciiType=asciiType, _EXTRA=etc
compile_opt idl2

if (n_elements(asciiType) gt 0) then begin
   void = where(self.asciiTypes eq asciiType,count)
   if (count eq 1) then self.asciiType = asciiType
endif

if(n_elements(etc) gt 0) then $
  self->IDLitWriter::SetProperty, _EXTRA=etc 

end



;===============================================================================
; DAVEwriteASCII::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEwriteASCII class
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function DAVEwriteASCII::init, _REF_EXTRA=etc
compile_opt idl2

; Init superclass
if (self->IDLitWriter::Init(_EXTRA=etc) eq 0) then return, 0


; call setproperty method, if necessary.
if (n_elements(etc) gt 0) then $
  self->SetProperty, _EXTRA=etc


self.asciiTypes = ['ASCIICOL','ASCIIGRP','ASCIISPE']

return, 1

end




;===============================================================================
pro DAVEwriteASCII__Define
compile_opt idl2

void = {DAVEwriteASCII, $
        inherits IDLitWriter $
        ,asciiType:'' $           ; the ASCII file type to be written
        ,asciiTypes:['ASCIICOL','ASCIIGRP','ASCIISPE'] $ ; supported ASCII file types
       }
end
