; docformat = 'rst'

;+
; Represents an HDF 5 file.
;
; :Categories:
;   file i/o, hdf5, sdf
;
; :Requires:
;   IDL 8.0
;
; :Author:
;   Michael Galloy
;
; :Properties:
;   filename
;     filename of the HDF 5 file
;-


;= property access

;+
; Get properties
;-
pro mgffh5file::getProperty, filename=filename, _ref_extra=e
  compile_opt strictarr

  if (Arg_present(filename)) then filename = self.filename
  if (N_elements(e) gt 0L) then self->Mgffh5base::getproperty, _extra=e
end


;= helper routines

;+
; Dumps the contents of the file.
;-
pro mgffh5file::dump
  compile_opt strictarr

  mg_h5_dump, self.filename
end


;+
; Start the HDF 5 browser on the file.
;-
pro mgffh5file::browse
  compile_opt strictarr

  ok = H5_browser(self.filename)
end


;+
; Open an HDF 5 file.
;
; :Private:
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
pro mgffh5file::_open, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return
  endif

  self.id = H5f_open(self.filename)
end


;+
; Close an HDF 5 file.
;
; :Private:
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
pro mgffh5file::_close, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return
  endif

  H5f_close, self.id
end


;+
; Get object info for child object inside file.
;
; :Private:
;
; :Returns:
;   `H5F_STAT` structure
;
; :Params:
;   name : in, required, type=string
;     name of child object
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
function mgffh5file::_statObject, name, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return, -1
  endif

  for i = 0L, H5g_get_num_objs(self.id) - 1L do begin
    ; find first case-insensitive match
    objName = H5g_get_obj_name_by_idx(self.id, i)
    if (Strcmp(objName, name, /fold_case)) then begin
      name = objName
      break
    endif
  endfor

  Return, H5g_get_objinfo(self.id, name)
end


;= operator overload methods

;+
; Output for `HELP` for file.
;
; :Private:
;
; :Returns:
;   string
;
; :Params:
;   varname : in, required, type=string
;     name of variable containing group object
;-
function mgffh5file::_overloadHelp, varname
  compile_opt strictarr

  type = 'H5File'
  specs = String(self.filename, format='(%"<%s>")')
  Return, self->Mgffh5base::_overloadhelp(varname, type=type, specs=specs)
end


;+
; Output for `PRINT` for group.
;
; :Private:
;
; :Returns:
;   string
;-
function mgffh5file::_overloadPrint
  compile_opt strictarr

  ; TODO: show group name, not filename

  nmembers = H5g_get_num_objs(self.id)
  names = Strarr(nmembers)
  types = Strarr(nmembers)
  for g = 0L, nmembers - 1L do begin
    names[g] = H5g_get_obj_name_by_idx(self.id, g)
    info = H5g_get_objinfo(self.id, names[g])
    types[g] = info.type
  endfor

  listing = mg_strmerge('  ' + names + ' (' + Strlowcase(types) + ')')
  Return, String(self.filename, listing, format='(%"HDF5 file: %s\n%s")')
end


;+
; Handles accessing groups/variables, particularly those with case-sensitive
; names or spaces/other characters in their names.
;
; :Private:
;
; :Examples:
;   For example::
;
;     h = mg_h5(file_which('hdf5_test.h5'))
;     d = h['2D int array']
;
; :Returns:
;   group/variable object
;
; :Params:
;   isRange : in, required, type=lonarr(8)
;     indicates whether the i-th parameter is a index range or a scalar/array
;     of indices
;   ss1 : in, required, type=long/lonarr
;     scalar subscript index value, an index array, or a subscript range
;   ss2 : in, optional, type=any
;     not used
;   ss3 : in, optional, type=any
;     not used
;   ss4 : in, optional, type=any
;     not used
;   ss5 : in, optional, type=any
;     not used
;   ss6 : in, optional, type=any
;     not used
;   ss7 : in, optional, type=any
;     not used
;   ss8 : in, optional, type=any
;     not used
;-
function mgffh5file::_overloadBracketsRightSide, isRange, $
  ss1, ss2, ss3, ss4, $
  ss5, ss6, ss7, ss8
  compile_opt strictarr
  On_error, 2

  objInfo = self->_Statobject(ss1, error=error)
  if (error ne 0L) then Message, String(ss1, format='(%"object %s not found")')

  ; TODO: search existing children before creating a new one

  case objInfo.type of
    'GROUP': result = Obj_new('MGffH5Group', parent=self, name=ss1)
    'DATASET': result = Obj_new('MGffH5Dataset', parent=self, name=ss1)
    'TYPE': ; TODO: implement
    'LINK': ; TODO: implement
    'UNKNOWN': Message, String(ss1, format='(%"object %s unknown")')
    else: begin
      ; TODO: check for attribute
      Message, String(ss1, format='(%"object %s unknown")')
    end
  endcase

  self.children->Add, result

  if (N_elements(ss2) gt 0L) then begin
    Return, result->_Overloadbracketsrightside(isRange[1:*], ss2, ss3, ss4, ss5, ss6, ss7, ss8)
  endif else begin
    Return, result
  endelse
end


;= lifecycle method

;+
; Free resources of the HDF 5 file, including its children.
;-
pro mgffh5file::Cleanup
  compile_opt strictarr

  Obj_destroy, self.children
  self->_Close
end


;+
; Creates HDF 5 object.
;
; :Returns:
;   1 for success, 0 for failure
;
; :Keywords:
;   filename : in, optional, type=string
;     filename of the HDF 5 file
;   _extra : in, optional, type=keywords
;     keywords to `MGffH5Base::init`
;-
function mgffh5file::init, filename=filename, _extra=e
  compile_opt strictarr
  On_error, 2

  if (~self->Mgffh5base::init(_extra=e)) then Return, 0

  self.filename = N_elements(filename) eq 0L ? '' : filename
  self.children = Obj_new('IDL_Container')
  self.name = ''

  self->_Open, error=error
  if (error ne 0L) then Message, 'invalid HDF5 file'

  Return, 1
end


;+
; Define instance variables and class inheritance.
;
; :Fields:
;   filename
;     name of HDF 5 file
;   children
;     `IDL_Container` of children group/dataset objects
;-
pro mgffh5file__define
  compile_opt strictarr

  define = { Mgffh5file, inherits Mgffh5base, $
    filename: '', $
    children: Obj_new() $
  }
end


; main-level example program

Print, format='(%"\nUsing an HDF5 file:")'
h = mgffh5file(filename=File_which('hdf5_test.h5'))
Help, h
Print, h

Print, format='(%"\nUsing a group:")'
g1 = h['images']
Help, g1
Print, g1

Print, format='(%"\nUsing a dataset:")'
d = h['2D int array']
Help, d

Print, format='(%"\nUsing a dataset:")'
e = g1['eskimo']
Help, e
Help, Size(e, /structure), /structures
Window, /free, xsize=600, ysize=200, title='Eskimo profile'
Plot, e[*, 400], xstyle=9, ystyle=8
Print, e['IMAGE_COLORMODEL'], format='(%"IMAGE_COLORMODEL attribute = %s")'

ct = g1['eskimo_palette']
Tvlct, Transpose(ct[*])
Device, get_decomposed=odec
Device, decomposed=0
dims = Size(e, /dimensions)
Window, /free, xsize=dims[0], ysize=dims[1], title='Eskimo image'
Tv, e[*], order=1
Device, decomposed=odec

end