; $Id$
;+
; NAME:
;  GET_DAVE_PTR_CONTENTS
;
; PURPOSE:
;
;  This function retrieves the contents of a DAVE pointer.  If successful,
;  the function returns a 1 else it returns 0.
;
; AUTHOR:
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  100 Bureau Drive
;  Gaithersburg, MD 20899
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;  Utility, DAVE project
;
; CALLING SEQUENCE:
;
;  ret_val = GET_DAVE_PTR_CONTENTS(  $
;               DAVEPTR ,           $
;               INSTRUMENT = instrument,   $
;               QTY = qty,         $
;               QTUNITS = qtunits,      $
;               QTLABEL = qtlabel,      $
;               ERR = err,         $
;               XVALS = xvals,       $
;               XTYPE = xtype,       $
;               XUNITS = xunits,      $
;               XLABEL = xlabel,      $
;               YVALS = yvals,       $
;               YTYPE = ytype,       $
;               YUNITS = yunits,      $
;               YLABEL = ylabel,      $
;               SPECIFICSTR = specificstr, $
;               TREATMENT = treatment,   $
;               DNAME = dname,       $
;               DUNITS = dunits,      $
;               DLEGEND = dlegend,      $
;               DQTY = dqty,         $
;               DERR = derr,         $
;               ERMSG = errmsg       )
;
; INPUT PARAMETERS (REQUIRED)
;
;  DAVEPTR:  This must be a properly defined DAVE pointer as defined
;        in DAVE.PRO for instance.
;
;
; OPTIONAL OUTPUT KEYWORDS
;
;  QTY:     Double of float array containing the data variable.  Can be one or two
;        dimensional where the size can be one of the following
;        (nx,ny), (nx+1,ny), (nx,ny+1), (nx+1,ny+1),(nx),(nx+1) where
;        nx is the length of the first independent variable and ny is the
;        length of the second independent variable.
;
;  ERR:     Double of float array containing the error associated with the measured
;        quantity, QTY.  ERR must have the same dimension and size as QTY.
;  INSTRUMENT:  String variable describing instrument.
;  XVALS:    Double or float array containing the first independent variable for
;        QTY.
;  YVALS:    Double or float array containing the second independent variable for
;        QTY.
;  QTUNITS:  string variable specifying the units of QTY
;  QTLABEL:  string variable specifying the plot label for QTY
;  XTYPE:    String variable specifying if the first independent variable, XVALS,
;        is "POINTS" or "HISTO".
;  YTYPE:    String variable specifying if the second independent variable, YVALS,
;        is "POINTS" or "HISTO".
;  XLABEL:     string variable specifying the plot label for the first independent
;        variable, XVALS.
;  YLABEL:     string variable specifying the plot label for the second independent
;        variable, YVALS.
;  XUNITS:     string variable specifying the units of XVALS
;  YUNITS:     string variable specifying the units of YVALS
;  SPECIFICSTR: structure containing any instrument specific information the
;         user wishes to include in the DAVE pointer.  May contain any
;         variable type in its fields.
;  TREATMENT:   string array of any length containing the treatment of the data.
;
;
;  Note that if any of the following variables beginning with "D" are undefined then
;  the function returns a 0 with an error message.
;
;  DNAME:    string name of the variable/parameter to be described in the
;        descriptive pointer in the DAVEPTR called DESCRIPTR.
;  DUNITS:     string variable specifying the units of the descriptive variable.
;  DLEGEND:  string variable specifying a description of the variable
;  DQTY:   value of the descriptive variable.
;  DERR:   value specifying the uncertainty of the descriptive variable.
;
;  ERMSG:    string variable specifying the cause of an error in which the function
;        returns 0.
;
; COMMON BLOCKS:
;
;   None
;
; EXAMPLE USAGE:
;  In the following example we assume the presence of a particular DAVE
;  file, restore it, extract one cut at a particular y value and display it.
;
;  IDL>   dir = 'c:\dimeo\hfbs\data\ch3i\bulk\reduced\'
;  IDL>   name = '20010618_14dyn.dave'
;  IDL>   file = dir+name
;  IDL>   restore,file
;  IDL>   ret_val = get_dave_ptr_contents(daveptr,    $
;                     qty = qty,   $
;                     err = err,   $
;                     xvals = xvals, $
;                     yvals = yvals)
;  IDL>   index = 5
;  IDL>   plot,xvals,qty[*,index],psym = 4
;  IDL>   errplot,xvals,qty[*,index]-err[*,index],qty[*,index]+err[*,index],width = 0.0
;  IDL>   heap_free,daveptr
;
;
;###############################################################################
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;
; MODIFICATION HISTORY:
;
;  Written by Rob Dimeo, March 24, 2003
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function get_dave_ptr_contents,  $
               daveptr,           $
               instrument = instrument,  $
               qty = qty,          $
               qtunits = qtunits,      $
               qtlabel = qtlabel,      $
               err = err,          $
               xvals = xvals,         $
               xtype = xtype,         $
               xunits = xunits,      $
               xlabel = xlabel,      $
               yvals = yvals,         $
               ytype = ytype,         $
               yunits = yunits,      $
               ylabel = ylabel,      $
               specificstr = specificstr,   $
               treatment = treatment,    $
               dname = dname,         $
               dunits = dunits,      $
               dlegend = dlegend,      $
               dqty = dqty,        $
               derr = derr,        $
               ermsg = errmsg
errmsg = 'No errors'
; Was a DAVE pointer set as a parameter?
if n_params() eq 0 then begin
   errmsg = 'DAVEPTR is a required input parameter'
   return,0
endif

qty = (*(*(*daveptr).datastrptr).commonstr.histptr).qty
err = (*(*(*daveptr).datastrptr).commonstr.histptr).err
xvals = (*(*(*daveptr).datastrptr).commonstr.histptr).x
yvals = (*(*(*daveptr).datastrptr).commonstr.histptr).y

instrument = (*(*daveptr).datastrptr).commonstr.instrument
qtunits = (*(*daveptr).datastrptr).commonstr.histunits
qtlabel = (*(*daveptr).datastrptr).commonstr.histlabel
xunits = (*(*daveptr).datastrptr).commonstr.xunits
yunits = (*(*daveptr).datastrptr).commonstr.yunits
xlabel = (*(*daveptr).datastrptr).commonstr.xlabel
ylabel = (*(*daveptr).datastrptr).commonstr.ylabel
xtype = (*(*daveptr).datastrptr).commonstr.xtype
ytype = (*(*daveptr).datastrptr).commonstr.ytype
if n_elements(*(*(*daveptr).datastrptr).commonstr.treatmentptr) eq 0 then $
   treatment = 'No treatment found' else $
   treatment = *(*(*daveptr).datastrptr).commonstr.treatmentptr
; Did the user want to pull out a specific structures?

if arg_present(specificstr) then begin
  if ptr_valid((*(*daveptr).datastrptr).specificptr) then begin
   specificstr = *(*(*daveptr).datastrptr).specificptr
  endif else begin
    errmsg = 'SPECIFICSTR undefined for this DAVE pointer'
    return,0
  endelse
endif

; Is the descriptr a valid tag name?
if    (arg_present(dname)) or    $
   (arg_present(dunits)) or   $
   (arg_present(dlegend)) or    $
   (arg_present(dqty)) or      $
   (arg_present(derr)) then begin  $

   tags = tag_names(*daveptr)
   wherevalid = where(strupcase(tags) eq 'DESCRIPTR',count)
   if count ne 0 then begin
     if ptr_valid((*daveptr).descriptr) then begin
        if n_elements(*(*daveptr).descriptr) ne 0 then begin
         dname = (*(*daveptr).descriptr).name
         dunits = (*(*daveptr).descriptr).units
         dlegend = (*(*daveptr).descriptr).legend
         dqty = (*(*daveptr).descriptr).qty
         derr = (*(*daveptr).descriptr).err
        endif else begin
         errmsg = 'Descriptive variables are undefined for this DAVE pointer'
         return,0
        endelse
     endif else begin
      errmsg = 'Descriptive variables are undefined for this DAVE pointer'
      return,0
     endelse
   endif else begin
     errmsg = 'Descriptive variables are undefined for this DAVE pointer'
     return,0
   endelse
endif

; Return a 1 if successful
return,1
end
