; $Id$
;###############################################################################
;+
; NAME:
;   wd_DAVETool
;
; PURPOSE:
;   The main user interface widget creation routine for the DAVE Main
;   Tool. This is essentially what the user sees when the tool is
;   launched.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2004
;-
;###############################################################################



;===============================================================================
; wd_DAVETool_autoCreateVis
; 
; PURPOSE:
;   Automatically create a new visualization using the selected data from the DAVE Data
;   Manager
; PARAMETERS
;   oTool - the object ref of the main DAVE tool
;   
;   visType - the type of visualization to be generated
;
;   task - set to 'new' | 'add' for creating a new vis or ppending to an existing vis
;
; KEYWORDS:
;
;pro wd_DAVETool_autoCreateVis, oTool, visType
pro wd_DAVETool_autoVis, oTool, visType, task
compile_opt idl2

oSystem = _IDLitSys_GetSystem()

sPtrMain = oTool->GetStatePtr()
if (~ptr_valid(sPtrMain)) then return

wParm = (*sPtrMain).wParm
if (~widget_info(wParm,/valid)) then return

widget_control, widget_info(wParm,/child), get_uvalue=sPtrParm
if (~ptr_valid(sPtrParm)) then return

wProp = (*sPtrParm).wProp
if (~widget_info(wProp,/valid)) then return

task_uname = 'SET2NEW'
if (strcmp(task,'Add',/fold_case)) then begin
   task_uname = 'SET2ADD'

   ;; There must be an existing visualization and its visualization type must match
   ;; the type that is to be added.
   void = oTool->GetVisToolObjRefs(count=nVisObjs)
   if (nVisObjs le 0) then begin
      title = "Add "+visType+" Visualization "
      msg = ["There is no existing visualization to add to" $
            ,"Please, use this option only when adding to an existing visualization!"]
      oTool->ErrorMessage, msg,severity=1,title=title
      return
   endif
   
   ;; What visualization type is displayed in the current vis window?
   curVisTool = (*sPtrMain).oVisUI->GetTool()
   oWin = curVisTool->GetCurrentWindow()
   oView = oWin->GetCurrentView()
   oLayer = oView->GetCurrentLayer()
   oWorld = oLayer->GetWorld()
   oDS = oWorld->GetDataSpaces()
   oVis = oDS->GetVisualizations(count=nVis)
   isValidVisTool = 0
   if (nVis gt 0) then begin
     for i=0,nVis-1 do begin
       oVis[i]->GetProperty, type=foundType
       isValidVisTool = (strcmp(foundType,strupcase('IDL'+visType)))[0]
     endfor
   endif


;   ;; Does the visuslization to be added match what is currently selected in the parameter table
;   ;; droplist (ie what 
;   currentVisType = cw_itParameterPropertysheet_GetVisDesc(wParm)
;   if (strcmp(currentVisType,visType,/fold_case) eq 0) then begin
;      title = "Add "+visType+" Visualization "
;      msg = ["You are attempting to add a '"+visType+"' visualization to an existing" $
;            ,"visualization window that contains non-matching visualizations ("+foundType+")!" $
;            ,"Please, first select by clicking on a visualization window of the same type"]
;      oTool->ErrorMessage, msg,severity=1,title=title
;      return
;   endif

   ;; Does the visuslization to be added match the vis type in the most recent visualization window?
   if (~isValidVisTool) then begin
      title = "Add "+visType+" Visualization "
      msg = ["You are attempting to add a '"+visType+"' visualization to an existing" $
            ,"visualization window that contains non-matching visualizations ("+foundType+")!" $
            ,"Please, first select by clicking on a visualization window of the same type"]
      oTool->ErrorMessage, msg,severity=1,title=title
      return
   endif
endif


;; First ensure Visualization task is set to 'New'
wTLB = (*sPtrMain).wTLB
wID = widget_info(wTLB, find_by_uname=task_uname)
widget_control, wID, /set_button
event = {WIDGET_BUTTON,ID:wID,top:wTLB,handler:wTLB,select:1}
wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
                           ; widget_control, wID, send_event=event, since this will be executed
                           ; asyncronyously which is not desired.

;; Create a temporal visualization object from the specified viz type and assign this to the parameter editor table
oVisDescPtr = oTool->GetVisualization(visType)
if (~obj_valid(oVisDescPtr)) then return

oVis = oVisDescPtr->_InstantiateObject()
if (~obj_valid(oVis)) then return

cw_itParameterPropertySheet_SetValue, wParm, oVis $   ; insert vis into param table
          ,parameter_editor=(*sPtrMain).isParmEdit $
          ,insert_visualization=(*sPtrMain).isInsVis

obj_destroy, oVis   ; no longer needed

widget_control, wProp, /refresh_property   ; refresh param property sheet

oDatasets = oTool->GetSelectedData(count=nDatasets)

;;-------------------------------------
;; Create the viz using first dataset

;cw_itParameterPropertysheet_SetDataSelect, wParm, oDatasets[0]  ; assoc dataset with param table
oDataset = GetMainDataObjectParent(oDatasets[0])
oldDataset = oDataset
(*sPtrParm).dataID = oDataset->GetFullIdentifier()    ; insert data in Parameter Table's state ptr
oDataset->GetProperty, nDimensions=ndim
groupsToView = 0
plane = 0
if (strcmp(visType,'Plot3D',/fold_case) && (ndim eq 2)) then begin
    ;; If Plot3D and a 2D dataset, then let user select which group(s) to plot
    oDataset->GetProperty, axis1Value=xData, axis2Value=yData
      
    nx = n_elements(xData)
    ny = n_elements(yData)

    oPlot3DProps = oSystem->getByIdentifier('/Registry/Misc/Plot3D Options')

    if (~obj_valid(oPlot3dProps)) then return

    oPlot3DProps->setProperty, nxgrps=nx, nygrps=ny
    if (~oSystem->doUIService('PropertySheet',oPlot3dProps)) then return
    
    oPlot3DProps->GetProperty, grpIndices=groupsToView, plane=plane
endif


;; Attempt to match data items with visualization parameters
validDataTypes = (*sPtrMain).plotContDataType ; data types assigned to plottable dataset containers
matchFound = wd_davetool_parampropsheet_adddata(sPtrParm, validDataTypes, plane, groupsToView[0])
if (matchFound le 0) then return

;; Check if minimum requirements have been met for this visualization
;; If so then go ahead and create the visualization
requirementsMet = cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
if (RequirementsMet) then begin
   wID = (*sPtrMain).wAction
   widget_control, wID, sensitive= 1
   event = {WIDGET_BUTTON,ID:wID, top:wTLB, handler:wTLB,select:0}
   wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
                              ; widget_control, wID, send_event=event, since this will be executed
                              ; asyncronyously which is not desired.
endif else begin
   title = visType+" Visualization "
   msg = ["Could not match data with at least one required" $
         ,"parameter for the '"+visType+"' visualization." $
         ,"Hence the visualization could not be created!"]
   oTool->ErrorMessage, msg,severity=1,title=title
   
   return
endelse


;;----------------------------------------------
;; Process a multi-Plot3D
nGrps = n_elements(groupsToView)
if ((nGrps gt 1) && strcmp(visType,'Plot3D',/fold_case)) then begin

  wID = widget_info(wTLB, find_by_uname='SET2ADD')
  widget_control, wID, /set_button
  event = {WIDGET_BUTTON,ID:wID,top:wTLB,handler:wTLB,select:1}
  wd_DAVETool_event, event

   ;; Create a temporal visualization object from the specified viz type and assign
   ;; this to the parameter editor table
   oVisDescPtr = oTool->GetVisualization(visType)
   if (~obj_valid(oVisDescPtr)) then return
   
   oVis = oVisDescPtr->_InstantiateObject()
   if (~obj_valid(oVis)) then return

   for i = 1,nGrps-1 do begin    

     cw_itParameterPropertySheet_SetValue, wParm, oVis $   ; insert vis into param table
               ,parameter_editor=(*sPtrMain).isParmEdit $
               ,insert_visualization=(*sPtrMain).isInsVis
      
      
     widget_control, wProp, /refresh_property   ; refresh param property sheet

     matchFound = wd_davetool_parampropsheet_adddata(sPtrParm, validDataTypes,plane,groupsToView[i])
     if (matchFound le 0) then continue 

     ;; Check if minimum requirements have been met for this visualization
     ;; If so then go ahead and create the visualization
     requirementsMet = cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
     if (RequirementsMet) then begin
        wID = (*sPtrMain).wAction
        widget_control, wID, sensitive= 1
        event = {WIDGET_BUTTON,ID:wID, top:wTLB, handler:wTLB,select:i}
        wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
                                   ; widget_control, wID, send_event=event, since this will be executed
                                   ; asyncronyously which is not desired.
     endif else begin
        msg = "Could not match data with at least one required parameter" 
        oTool->StatusMessage, msg
     endelse
    
  
   endfor

   obj_destroy, oVis   ; no longer needed

endif

;;----------------------------------------------
;; Process a multi-plot
if (nDatasets gt 1 && strcmp(visType,'plot',/fold_case)) then begin
  wID = widget_info(wTLB, find_by_uname='SET2ADD')
  widget_control, wID, /set_button
  event = {WIDGET_BUTTON,ID:wID,top:wTLB,handler:wTLB,select:1}
  wd_DAVETool_event, event
  
  ;; Create a temporal visualization object from the specified viz type and assign
  ;; this to the parameter editor table
  oVisDescPtr = oTool->GetVisualization(visType)
  if (~obj_valid(oVisDescPtr)) then return
   
  oVis = oVisDescPtr->_InstantiateObject()
  if (~obj_valid(oVis)) then return
  
  nPlot = 1   ; plot index
  for i = 1,nDatasets-1 do begin
    oData = GetMainDataObjectParent(oDatasets[i])
    
    ; On X-windows, if user used the shift key to select multiple datasets, then the child nodes 
    ; of the dataset are also reported as selected. Hence the children appear as datasets! Since
    ; oData (the parent obtained above) will be the same for each child node, use the simple test 
    ; (next line) to only proceed to create a plot if the parent dataset has changed!
    if (oldDataset eq oData) then continue

    if (~obj_isa(oData,'GENERICDATASET')) then continue
    oData->GetProperty, nDimensions=ndim
    if (ndim ne 1) then continue
    
    ;cw_itParameterPropertysheet_SetDataSelect, wParm, oData  ; assoc dataset with param table
    (*sPtrParm).dataID = oData->GetFullIdentifier()    ; insert data in Parameter Table's state ptr
    
    cw_itParameterPropertySheet_SetValue, wParm, oVis $   ; insert vis into param table
              ,parameter_editor=(*sPtrMain).isParmEdit $
              ,insert_visualization=(*sPtrMain).isInsVis
      
    matchFound = wd_davetool_parampropsheet_adddata(sPtrParm, validDataTypes)
    if (matchFound le 0) then continue
    oldDataset = oData

    ;; Check if minimum requirements have been met for this visualization
    ;; If so then go ahead and create the visualization
    requirementsMet = cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
    if (RequirementsMet) then begin
       wID = (*sPtrMain).wAction
       widget_control, wID, sensitive= 1
       event = {WIDGET_BUTTON,ID:wID, top:wTLB, handler:wTLB,select:nPlot}
       wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
                                  ; widget_control, wID, send_event=event, since this will be executed
                                  ; asyncronyously which is not desired.
    endif else begin
       msg = "Could not match data with at least one required parameter" 
       oTool->StatusMessage, msg
    endelse
    nPlot++   ; increment plot index
  endfor

  obj_destroy, oVis   ; no longer needed
endif


end


;===============================================================================
; wd_DAVETool_autoAddVis
; 
; PURPOSE:
;   Automatically visualization the selected data in an existing visualization window
; PARAMETERS
;   oTool - the object ref of the main DAVE tool
;   
;   visType - the type of visualization to be generated
;
; KEYWORDS:
;
;pro wd_DAVETool_autoAddVis, oTool, visType
;compile_opt idl2
;
;sPtrMain = oTool->GetStatePtr()
;if (~ptr_valid(sPtrMain)) then return
;
;wParm = (*sPtrMain).wParm
;if (~widget_info(wParm,/valid)) then return
;
;widget_control, widget_info(wParm,/child), get_uvalue=sPtrParm
;if (~ptr_valid(sPtrParm)) then return
;
;wProp = (*sPtrParm).wProp
;if (~widget_info(wProp,/valid)) then return
;
;
;;; There must be an existing visualization and its visualization type must match
;;; the type that is to be added.
;void = oTool->GetVisToolObjRefs(count=nVisObjs)
;if (nVisObjs le 0) then begin
;   title = "Add "+visType+" Visualization "
;   msg = ["There is no existing visualization to add to" $
;         ,"Please, use this option only when adding to an existing visualization!"]
;   oTool->ErrorMessage, msg,severity=1,title=title
;   return
;endif
;
;;; Does the visuslization to be added match what is currently selected in the parameter table
;;; droplist (ie what 
;currentVisType = cw_itParameterPropertysheet_GetVisDesc(wParm)
;if (strcmp(currentVisType,visType,/fold_case) eq 0) then begin
;   title = "Add "+visType+" Visualization "
;   msg = ["You are attempting to add a '"+visType+"' visualization to an" $
;         ,"existing visualization window that is not of the same visualization type!" $
;         ,"Please, first select by clicking on a visualization window of the same type"]
;   oTool->ErrorMessage, msg,severity=1,title=title
;   return
;endif
;
;;; Ensure Visualization task is set to 'Add'
;wTLB = (*sPtrMain).wTLB
;wID = widget_info(wTLB, find_by_uname='SET2ADD')
;;wID2 = widget_info(wTLB, find_by_uname='SET2MOD')
;;wID3 = widget_info(wTLB, find_by_uname='SET2NEW')
;widget_control, wID, /set_button
;event = {WIDGET_BUTTON,ID:wID,top:wTLB,handler:wTLB,select:1}
;wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
;                           ; widget_control, wID, send_event=event, since this will be executed
;                           ; asyncronyously which is not desired.
;
;;; Create a temporal visualization object from the specified viz type and assign
;;; this to the parameter editor table
;oVisDescPtr = oTool->GetVisualization(visType)
;if (~obj_valid(oVisDescPtr)) then return
;
;oVis = oVisDescPtr->_InstantiateObject()
;if (~obj_valid(oVis)) then return
;
;cw_itParameterPropertySheet_SetValue, wParm, oVis $   ; insert vis into param table
;          ,parameter_editor=(*sPtrMain).isParmEdit $
;          ,insert_visualization=(*sPtrMain).isInsVis
;
;obj_destroy, oVis   ; no longer needed
;
;widget_control, wProp, /refresh_property   ; refresh param property sheet
;
;
;;; Attempt to match data items with visualization parameters
;validDataTypes = (*sPtrMain).plotContDataType ; data types assigned to plottable dataset containers
;matchFound = wd_davetool_parampropsheet_adddata(sPtrParm, validDataTypes)
;;matchFound = wd_davetool_parampropsheet_adddata(sPtrParm)
;if (matchFound le 0) then return
;
;;; Check if minimum requirements have been met for this visualization
;;; If so then go ahead and create the visualization
;requirementsMet = cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
;if (RequirementsMet) then begin
;   wID = (*sPtrMain).wAction
;   widget_control, wID, sensitive= 1
;   event = {WIDGET_BUTTON,ID:wID, top:wTLB, handler:wTLB,select:0}
;   wd_DAVETool_event, event   ; execute the event handler directly rather than relying on XMANAGER through
;                              ; widget_control, wID, send_event=event, since this will be executed
;                              ; asyncronyously which is not desired.
;endif else begin
;   title = "Add "+visType+" Visualization "
;   msg = ["Could not match data with at least one required" $
;         ,"parameter for the '"+visType+"' visualization." $
;         ,"Hence the visualization could not be added!"]
;   oTool->ErrorMessage, msg,severity=1,title=title
;endelse
;
;end


;===============================================================================
; wd_DAVETool_event
; 
; PURPOSE:
;   Main event handler for widget created by wd_DAVETool.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
pro wd_DAVETool_event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'WD_DAVETOOL: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

IDL_version8 = float(!version.RELEASE) ge 8.0

; Get our state info - stored in first child widget of event.handler.
wChild = widget_info(event.handler, /child)
widget_control, wChild, get_uvalue=sPtr
oTool = (*sPtr).oTool
oUI = (*sPtr).oUI
wParm = (*sPtr).wParm ;widget_info((*sPtr).wParamEditBase,/child)

case vartype(event) of          ; get the event structure name
        
    ;; Kill the widget?
    'WIDGET_KILL_REQUEST': begin
;        visToolArr = (*(*sPtr).visToolArrPtr)
;        for i=0,n_elements(visToolArr)-1 do begin
;            ;; switch off prompt so popup is not displayed when deleting vis tool
;            if (obj_valid(visToolArr[i])) then visToolArr[i]->SetProperty, prompt=0
;        endfor
;    
;        ;; Let the tool know that shutdown was requested
;        ;; This will shutdown the entire tool
;        oShutdown = oTool->GetService("SHUTDOWN")
;;         oSrvMacro = oTool->GetService('MACROS')
;;         if OBJ_VALID(oSrvMacro) then begin
;;                                 ; need to explicitly add this operation since after shutdown
;;                                 ; the tool is invalid
;;             oSrvMacro->GetProperty, CURRENT_NAME=currentName
;;             oSrvMacro->PasteMacroOperation, self, currentName
;;         endif
;
;        if(~obj_valid(oShutdown))then begin
;            oTool->ErrorMessage, title=IDLitLangCatQuery('Error:InternalError:Title'), $
;                                 [IDLitLangCatQuery('Error:Framework:CannotAccessShutdown'), $
;                                  IDLitLangCatQuery('Error:Framework:ForceShutdown')], severity=2
;            obj_destroy,self
;            return
;        endif
;        
;        void = oUI->DoAction(oShutdown->getFullIdentifier())

        exitID = 'Operations/File/Exit'
        exitOpDesc = oTool->GetByIdentifier(exitID)
        if (obj_valid(exitOpDesc)) then exitOp = exitOpDesc->GetObjectInstance()
        if (obj_valid(exitOp)) then void = exitOp->DoAction(oTool)
        
        return
    end

    ;; Show the context menu
;    'WIDGET_CONTEXT': widget_displaycontextmenu, event.id, event.x, event.y, (*sPtr).wContext

    ;; Focus change
    'WIDGET_KBRD_FOCUS': begin
        if (obj_valid(oUI) and event.enter eq 1) then begin
                                ;oTool = (*sPtr).oUI->GetTool()
            oCurrent = oTool->GetService("SET_AS_CURRENT_TOOL")
            void=oUI->DoAction(oCurrent->GetFullIdentifier())
            ;; DONT DO THIS our tools could enter a focus loop in the
            ;; following situation:
            ;;    IDL> Iplot & iPlot
            ;; widget_control, (*sPtr).wDraw, /input_Focus
        endif
    end

    ;; The panel has resized
    'CW_ITPANEL_RESIZE': begin
        ;; The panel resized, so we might need to resize the
        ;; status bar.
        geomDraw = widget_info((*sPtr).wDraw, /geometry)
        geomPanel = widget_info((*sPtr).wPanel, /geometry)
        newVisW = (geomdraw.scr_xsize + geomPanel.scr_xsize) > (*sPtr).minsize[0]
        
        cw_itStatusBar_Resize, (*sPtr).wStatus, newVisW
        widget_control, (*sPtr).wTLB, tlb_get_size=basesize
        (*sPtr).basesize = basesize
    end

    ;; The TLB was resized
    'WIDGET_BASE': begin
        ;; Compute the size change of the base relative to
        ;; its stored former size. Note use of update to prevent flash
                                ;widget_control, event.top, TLB_GET_SIZE=newSize
                                ;deltaW = newSize[0] - (*sPtr).basesize[0]
                                ;deltaH = newSize[1] - (*sPtr).basesize[1]
                                ;wd_DAVETool_resize, sPtr, deltaW, deltaH
        wd_DAVETool_resize, sPtr
    end

    ;; Tab widget clicked!
    'WIDGET_TAB' : begin
        ;; Whenever  the data/vis tab is clicked, the property sheet
        ;; has to be updated to reflect the item that is currently
        ;; selected in the tree widget of the selected tab.

        ;; Get the selected tab's base widget id from the event
;        tabID = (event.tab eq 0)? (*sPtr).wDM : (*sPtr).wVM
;        widget_control, tabID, get_uvalue=tab_sPtr
;        if (~ptr_valid(tab_sPtr)) then break

        ;; Treat data/vis tabs differently
        case event.tab of
            0: begin            ; the data tab
                
                (*sPtr).currentTab = 0 ; update so we know the data tab is in view

                ;; stop observing old data item
                oUI->RemoveOnNotifyObserver, (*sPtr).idSelf,(*sPtr).itemID

                ;; Query the root tree widget of the DM for the
                ;; currently selected item; The identifier of the item
                ;; is stored in its uname
                wSelected = widget_info(widget_info((*sPtr).wDM,/child),/tree_select)
                (*sPtr).itemID = (widget_info(wSelected[0],/valid_id))? $
                                 widget_info(wSelected[0],/uname) : ''
                
                
                ;; display properties in property sheet, after first
                ;; updating the oUI of the property sheet to that
                ;; of the appropriate tool
                widget_control, (*sPtr).wPS, get_uvalue=statePtr;, /no_copy
                (*statePtr).oUI = (*sPtr).oUI
;                widget_control, (*sPtr).wPS, set_uvalue=state, /no_copy

                widget_control, (*sPtr).wPS, set_value=(*sPtr).itemID

                ;; Update parameter editor, if applicable to selected item
                oItem =  oTool->GetByIdentifier((*sPtr).itemID)
                if obj_isa(oItem, 'IDLitData') then begin
                    ;; observe the data item
                    oUI->AddOnNotifyObserver,(*sPtr).idSelf,(*sPtr).itemID

                    ;; let table know which data item is currently selected
                    if widget_info(wParm,/valid_id) then $
                      cw_itParameterPropertySheet_SetDataSelect,wParm,oItem

                endif else begin
                    ;; let table know which data item is currently selected
                    if widget_info(wParm,/valid_id) then $
                      cw_itParameterPropertySheet_SetDataSelect,wParm,0
                endelse

            end

            1: begin            ; the vis tab

                (*sPtr).currentTab = 1 ; update so we know the vis tab is in view

                ;; Query the root tree widget of the VM for the
                ;; currently selected item; The identifier of the item
                ;; is stored in its uname
                wSelected = widget_info((*sPtr).wVM,/tree_select)
                (*sPtr).itemID = (widget_info(wSelected[0],/valid_id))? $
                                 widget_info(wSelected[0],/uname) : ''
                
                ;; display properties in property sheet, after first
                ;; updating the oUI of the property sheet to that
                ;; of the appropriate tool
                widget_control, (*sPtr).wPS, get_uvalue=statePtr;, /no_copy
                (*statePtr).oUI = (obj_valid((*sPtr).oVisUI))? (*sPtr).oVisUI : (*sPtr).oUI ; in case no vistool present!
;                widget_control, (*sPtr).wPS, set_uvalue=state, /no_copy

                widget_control, (*sPtr).wPS, set_value=(*sPtr).itemID

                ;; Update parameter editor

                ;; set selected data item to null (0)
                if widget_info(wParm,/valid_id) then $
                  cw_itParameterPropertySheet_SetDataSelect,wParm,0
            end
        endcase

        ;;help,event,/struct
    end

    else: ;; LET IT FALL THROUGH
endcase

; In addition, need to update the detail section of the screen if an
; item is selected from the data or visualization trees.
uname = widget_info(event.id, /uname)

; define a set of useful colors to use
colors = [[0,0,0] $     ; Black
         ,[255,0,0] $   ; Red
         ,[0,255,0] $   ; Green
         ,[0,0,255] $   ; Blue
         ,[100,0,0] $   ; dark red
         ,[0,100,0] $   ; dark Green
         ,[0,0,100] $   ; dark Blue
         ]

; define a set of useful symbols to use
symProps = replicate({symbol:'',sym_filled:1},13)
symProps[0].symbol = 'Circle'           & symProps[0].sym_filled = 1
symProps[1].symbol = 'Circle'           & symProps[1].sym_filled = 0
symProps[2].symbol = 'Square'           & symProps[2].sym_filled = 0
symProps[3].symbol = 'Square'           & symProps[3].sym_filled = 1
symProps[4].symbol = 'Diamond'          & symProps[4].sym_filled = 0
symProps[5].symbol = 'Diamond'          & symProps[5].sym_filled = 1
symProps[6].symbol = 'Triangle'         & symProps[6].sym_filled = 0
symProps[7].symbol = 'Triangle'         & symProps[7].sym_filled = 1
symProps[8].symbol = 'Star'             & symProps[8].sym_filled = 0
symProps[9].symbol = 'Star'             & symProps[9].sym_filled = 1
symProps[10].symbol = 'Triangle_down'   & symProps[10].sym_filled = 0
symProps[11].symbol = 'Triangle_down'   & symProps[11].sym_filled = 1
symProps[12].symbol = 'X'               & symProps[12].sym_filled = 0

case uname of
    'DAVEDATAMANAGER': begin
        ; Do nothing if new item is the same as old item
        if (strcmp((*sPtr).itemID, event.identifier,/fold_case)) then break
        
        ;; stop observing old data item
        oUI->RemoveOnNotifyObserver, (*sPtr).idSelf,(*sPtr).itemID
        ;; save ID of new item
        (*sPtr).itemID = event.identifier

        ;; the event.identifier field is inserted in
        ;; cw_daveDataManager whenever an item is selected.
        oItem =  oTool->GetByIdentifier(event.identifier)

        ;; Activate/De-activate operations in various top-level menus
        ;; based on the data type of the selected data.
        oItems = oTool->GetSelectedData()
        ;oItems=oItem
        selDataTypes = ''
        for i=0,n_elements(oItems)-1 do begin
            if (obj_hasmethod(oItems[i],'getTypes')) then $
              selDataTypes = [selDataTypes,oItems[i]->getTypes()]
        endfor
        contIDs = ['Operations/Analysis'] ;['Operations/Analysis','Operations/Misc']
        for i = 0,n_elements(contIDs)-1 do begin
            oAnaCont = oTool->GetByIdentifier(contIDs[i])
            oOperations = oAnaCont->IDL_Container::Get(/all)
            ;;  oTool->IDLitTool::UpdateAvailability is a dummy call to get routines in 
            ;; idlittool__updateavailability.pro to be resolved so
            ;; that the call to oTool->IDLitTool::_CheckItemsAvailability works
            if (~obj_hasmethod(oTool,'_CheckItemsAvailability')) then oTool->IDLitTool::UpdateAvailability
            oTool->IDLitTool::_CheckItemsAvailability, selDataTypes, oOperations
        endfor

        if obj_isa(oItem, 'IDLitData') then begin
            ;; Make the property sheet display the value of the data
            ;; by assigning it (or a portion of it) to the uvalue property
            ;; of the object so that when registered will
            ;; automatically be displayed by the property sheet.
            dsize = oItem->GetSize()
            isaContainer = obj_isa(oItem,'IDL_Container')
            if (dsize gt 10000.0 || isaContainer) then begin 
                oItem->GetProperty, uvalue=curUvalue
                fm = '(F5.1)'
                dsize = (dsize ge 1000000.0)? strtrim(string(dsize/1000000.0,format=fm),2)+'MB' : $
                        ((dsize lt 1000)? strtrim(string(dsize),2)+'B' : $
                         strtrim(string(dsize/1000.0,format=fm),2)+'KB')
                if (n_elements(curUvalue) lt 1) then begin
                    oItem->SetProperty, uvalue='Size = '+dsize
                    oItem->RegisterProperty,'UVALUE',/string,name='Value',sensitive=0
                endif else oItem->SetProperty, uvalue='Size = '+dsize
            endif else if (oItem->getdata(data)) then begin
                case size(data,/n_dimensions) of
                    0: begin
                        uvalue = string(data)
                    end

                    else: begin
                        case (n_elements(data) gt 3) of
                            0: uvalue = strjoin(string(data))
                            1: uvalue = strjoin([string(data[0:3]),' ...'])
                        endcase
                    end
                endcase
                oItem->GetProperty, uvalue=curUvalue
                if (n_elements(curUvalue) lt 1) then begin
                    oItem->SetProperty, uvalue=strtrim(uvalue,2)
                    oItem->RegisterProperty,'UVALUE',/string,name='Value',sensitive=0
                endif else oItem->SetProperty, uvalue=strtrim(uvalue,2)
            endif
        endif
        ;; display properties in property sheet
        widget_control, (*sPtr).wPS, set_value=(*sPtr).itemID

        if obj_isa(oItem, 'IDLitData') then begin
            ;; observe the data item
            oUI->AddOnNotifyObserver,(*sPtr).idSelf,(*sPtr).itemID

            ;; let table know which data item is currently selected
            if widget_info(wParm,/valid_id) then $
              cw_itParameterPropertySheet_SetDataSelect,wParm,oItem

            ;; if item was double clicked then add data to parm table
            if (event.clicks eq 2) && widget_info(wParm,/valid_id) then begin
                success = cw_itParameterPropertysheet_AddData(wParm)
                if success then begin
                    widget_control, $
                      (*sPtr).wAction, $
                      sensitive=cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
                    ;; do not allow DM to change insVis droplist
                    cw_itParameterPropertysheet_LockList,wParm
                endif
            endif
        endif else begin
            ;; let table know which data item is currently selected
            if widget_info(wParm,/valid_id) then $
              cw_itParameterPropertySheet_SetDataSelect,wParm,0
        endelse

    end

    'DAVEVISMANAGER': begin
        ;; stop observing old data item
;        oUI->RemoveOnNotifyObserver, (*sPtr).idSelf,(*sPtr).itemID

        ;; save ID of new item
        (*sPtr).itemID = event.identifier
        (*sPtr).oVisUI = event.oVisUI

        ;; display properties in property sheet
        widget_control, (*sPtr).wPS, set_value=(*sPtr).itemID

        ;; observe the data item
;        oUI->AddOnNotifyObserver,(*sPtr).idSelf,(*sPtr).itemID
    end

    'NEWVIS': begin             ; Create New visualization

        ;; get parameters from table
        cw_itParameterPropertysheet_GetParameters,wParm, strParms,idParms,count=count
        ;; get ID of current vis descriptor
        visID = cw_itParameterPropertysheet_GetVisDesc(wParm)

        ;; Essential checks
        
        if (strcmp(visID,'Image',/fold_case)) then begin
           ;; Can't create image plot if independent axes data do not
           ;; have uniform bins
           axes = ['X','Y']
           for i=0,n_elements(axes)-1 do begin
              index = where(strParms eq axes[i],axisFound)
              if (~axisFound) then continue
              axisID = idParms[index]
              oAxis = oTool->GetByIdentifier(axisID)
              if (~obj_valid(oAxis)) then continue
              
              if (~oAxis->GetData(axisData)) then continue
              
              index = lindgen(n_elements(axisData)-1)
              diff = axisData[index+1] - axisData[index]
              unit = (diff[0] ne 0.0)? diff/diff[0] : diff ; normalize to unity
              unit = float(string(unit,format='(G10.3)')) ; up to 3 dec place precision
              void = where(unit ne 1.00, binsNotUniform)
              if (binsNotUniform gt 0) then begin
                 msg = strtrim(axes[i],2)+'-axis data does not contain evenly spaced bins'
                 msg = [msg,'Cannot accurately represented data using an image plot!']
                 msg = [msg,'Please, use contour/surface plot or rebin the data first']
                 void = dialog_message(dialog_parent=event.top, msg, /info)
                 return
              endif
           endfor
        endif

        switch (*sPtr).visTask of
            'ADDVIS': begin     ; Use existing Vis Tool
                ;; Retrieve current Vis Tool                
                if (~obj_valid((*sPtr).oVisUI)) then return
                oVisTool = ((*sPtr).oVisUI)->GetTool()
                if (~obj_isa(oVisTool,'DAVEvisToolbase')) then return

                break
            end

            'NEWVIS':           ; Create a new Vis Tool
            else: begin
                ;; create new tool
                visCntStr = (*(*sPtr).visIDCountPtr)
                index = where(tag_names(visCntStr) eq strupcase(visID),cnt)
                if (cnt gt 0) then begin
                    visCnt = visCntStr.(index[0])
                    visCntStr.(index[0]) = visCnt + 1
                endif else begin
                    visCnt = 1
                    visCntStr = create_struct(visCntStr,visID,2)
                endelse
                (*(*sPtr).visIDCountPtr) = visCntStr
                
                nameTag = 'DAVE ' +strtrim(visID,2)+ ' ' +strtrim(string(visCnt),2)
                
                ;; if found, align vistool nametag with name of dataset or item being visualized
                dataNameTag = oTool->GetSelectedDataNameTag()
                nameTag = (dataNameTag[0] eq '')? nameTag : dataNameTag[0]

                oSystem = oTool->GetSystem()
                oVisTool = oSystem->GetByIdentifier( $
                           IDLitSys_CreateTool('DAVE VisTool' $
                                               ,title=nameTag $
                                               ,dimensions = [700,530] $
                                               ,user_interface='DAVE VisTool Interface' $
                                               ,/debug $  ; uncomment when debugging
;                                               ,/no_saveprompt $
                                              ))
                ;; The vistool should have the same filename as the maintool
                oTool->GetProperty, tool_filename=filename
                oVisTool->SetProperty, nameTag=nameTag,tool_filename=filename
                ; Retrieve the graphics window
                ; and set auto-resize and zoom-on-resize to 1 so the graphics window behaves properly when its size is changed
                oWin = oVisTool->GetCurrentWindow()
                oWin->SetProperty, auto_resize=1, zoom_on_resize=1
               
;                if (float((strtok(!version.release,/extract))[0]) ge 8.0) then begin
;                  ;; As of IDL 8.0, hide the visualization title text
;                  oWin = oVisTool->GetCurrentWindow()
;                  oScene = oWin->GetScene()
;                  oView = oScene->GetCurrentView()
;                  atoms = oView->GetAll(count=natoms)
;                  for i=0,natoms-1 do begin
;                     if (obj_isa(atoms[i],'IDLitvisText')) then atoms[i]->setProperty, hide=1
;                  endfor
;                endif
                
                break
            end

        endswitch



        ;; Get the insert visualization operation
;        insVisOpID = 'Operations/Insert/Visualization'
        insVisOpID = 'Operations/Insert Visualization'
        (*sPtr).oRequestor = (oVisTool->GetByIdentifier(insVisOpID))->GetObjectInstance()

        ;; if everything is valid then proceed with operation
        if obj_isa((*sPtr).oRequestor,'IDLitOpInsertVis') then begin
            if (count GT 0) then begin
                (*sPtr).oRequestor->SetProperty, $
                  PARAMETER_NAMES=strParms,DATA_IDS=idParms,VISUALIZATION_ID=visID
                (*sPtr).oRequestor->_SetTool,oVisTool
                ;; get macro service
                oSrvMacros = oVisTool->GetService('MACROS')
                ;; save current value and set show_ui to zero
                (*sPtr).oRequestor->GetProperty,SHOW_EXECUTION_UI=showUIOrig
                (*sPtr).oRequestor->SetProperty,SHOW_EXECUTION_UI=0
                ;; call doaction, which performs the operation and generates
                ;; the needed commands
                oCmd = (*sPtr).oRequestor->DoAction(oVisTool)
                ;; Add this to history explicitly
                if obj_valid(oCmd[0]) then begin
                    oSrvMacros->GetProperty, CURRENT_NAME=currentName
                    oSrvMacros->PasteMacroOperation, (*sPtr).oRequestor, currentName
                endif

                
                if (strcmp((*sPtr).visTask, 'ADDVIS', /fold_case)) then begin
                   ;; Add this to undo/redo explicitly when adding to an existing vis
                   oVisTool->_TransactCommand, oCmd
                endif

                ;; restore original values on the operation
                (*sPtr).oRequestor->SetProperty,SHOW_EXECUTION_UI=showUIOrig
            endif

            ;; Change the newly created is objects name to reflect the data from which
            ;; it was created from.
            oVis = oVisTool->GetSelectedItems() ; retrieve the vis object just created (it will be selected!)
            if (obj_valid(oVis)) then begin
               if (strcmp(visid,'PLOT3D',/fold_case)) then begin
                  dataNameTag = (oTool->GetSelectedDataNameTag())[0] ; for a multiplot, event.select specifies dataset being processed
                  ;; dataNameTag is ignored for plot3d vis
                  visNameTag = strtrim((*sPtr).plotLabel,2)
               endif else begin
                  dataNameTags = oTool->GetSelectedDataNameTag(nNameTags)
                  index = (nNameTags gt 1)? event.select : 0   ; for a multiplot, event.select specifies dataset being processed
                  dataNameTag = dataNameTags[index] 
                  visNameTag = strtrim(dataNameTag+" "+(*sPtr).plotLabel,2)
               endelse
               
               void = oVisTool->DoSetProperty(oVis->GetFullIdentifier(),'name',visNameTag)
               oVis->DoOnNotify,oVis->GetFullIdentifier(),'SETPROPERTY','NAME' ; broadcast name change message so
                                                                               ; that vis browser can update itself
            endif

            ;(*sPtr).isVisible = 0b

            if (strcmp((*sPtr).visTask, 'NEWVIS', /fold_case)) then begin
                ;; Create an entry in the Visualization Browser
                ;;----------------------------------------------
                
                ;; Use a UI service. Note the unconventional call
                ;; here. Usually, the requester is the object reference of
                ;; a tool/operation and not the oUI; but this was the only
                ;; way to have access to this oUI from within the UI service.
                res = oVisTool->doUIService('AddVisItem',oUI)
            endif

            ;; Change the default zoomFactor
            oWin = oVisTool->GetCurrentWindow()
            oScene = obj_valid(oWin) ? oWin->GetScene() : OBJ_NEW()
            oView = obj_valid(oScene) ? oScene->GetCurrentView() : OBJ_NEW()
                                ;if (obj_valid(oView)) then
                                ;oView->SetProperty, zoomFactor=1.30

            ;; Retrieve axes so fontsize can be set 12pt
            oVis = oVisTool->GetSelectedItems() ; retrieve the vis object just created
            if (obj_valid(oVis)) then oDSpace = oVis->GetDataspace()
            if (obj_valid(oDSpace)) then allAxes = oDSpace->GetAxes(count=nAxes)
            for i=0,nAxes-1 do allAxes[i]->SetProperty, font_size=13

            if (~strcmp(visid,'SURFACE',/fold_case) && strcmp((*sPtr).visTask, 'NEWVIS', /fold_case)) then begin
               ;Modify dataspace dimensions slightly to better use up available window
               ;However, does not work well for surface plots
               oDSpace->Scale, 0.9,1.1,1
               oDSpace->Translate, 0.15, 0.0, 0.0
            endif


            if (obj_valid(oView)) then begin
                switch (strupcase(visid)) of
                    'PLOT': begin
                        zoom = (IDL_version8)? 1.0 : 1.2
                        oView->SetCurrentZoom, zoom ;1.50
                        
                        oVis = oVisTool->GetSelectedItems() ; retrieve the vis object just created
                        
                        icol = (size(colors,/dimensions))[1]
                        cindex = event.select mod icol 
                        color = colors[*,cindex]                ; one from colors array above
         
;                        sym_index = event.select mod 25         ; event.select is always 0 for a single plot or 0 to n for a multiplot where n is the number of plots
;                        if (sym_index eq 0 || sym_index eq 3 || sym_index eq 8 || sym_index eq 9) $
;                           then sym_index = 24  ; ignore 0 (no symbol), 3 (Dot), 8 (>) and 9 (<)                        
                        symProp = symProps[event.select mod n_elements(symProps)] ; select one from earlier defined list
                        oVis->SetProperty ,errorbar_color=color $
                                          ,color=color $
                                          ;,sym_index=sym_index $
                                          ,_EXTRA=symProp
                                          
                        break
                    end
                    
                    'CONTOUR': begin
                        oVis = oVisTool->GetSelectedItems() ; retrieve the vis object just created
                        oVis->SetProperty, n_levels=20
                    end
                    'IMAGE': begin
                        zoom = (IDL_version8)? 1.0 : 1.2
                        oView->SetCurrentZoom, zoom
                        oLayer = oView->getcurrentlayer()
                        if (obj_valid(oLayer)) then $
                           oLayer->SetProperty, stretch_to_fit=1 ; set to true by default
                        break
                    end
                    
                    'PLOT3D': begin
                        oVis = oVisTool->GetSelectedItems() ; retrieve the vis object just created
                        
                        icol = (size(colors,/dimensions))[1]
                        cindex = event.select mod icol 
                        color = colors[*,cindex]                ; one from colors array above
;                        sym_index = event.select mod 9 
;                        if (sym_index eq 3) then sym_index = 8  ; ignore symbol 3 (Dot) 
;                        if (sym_index eq 0) then sym_index = 6  
                        symProp = symProps[event.select mod n_elements(symProps)] ; select one from earlier defined list
                        
                        oVis->SetProperty ,errorbar_color=color $
                                          ,sym_color=color $
                                          ,color=color $
                                          ;,sym_index=sym_index $
                                          ,_EXTRA=symProp
                        ; Note: there is no BREAK statement here!
                    end
                    'SURFACE': begin
                        zoom = (IDL_version8)? 1.0 : 1.1
                        oView->SetCurrentZoom, zoom 

                        ;; In addition, the z-axis is created in the
                        ;; foreground (at 0,0,0) by default. It is
                        ;; more desireable to have this in the
                        ;; background at the tip of the y-axis (0,1,0).
                        ;if (n_elements(allAxes) eq 3) then begin
                        if (nAxes eq 3) then begin
                            oAxes = oDSpace->GetAxes(/container)
                            allAxes[2]->SetProperty, norm_location=[0d,1d,0d] ; re-locate to tip of y-axis
                            oAxes->_updateAxesRanges
                        endif
                        
                        break
                    end

                    else:
                endswitch
            endif

            oVisTool->RefreshCurrentWindow
            oTool->_SetDirty, 1
        endif
    end

    'MODVIS': begin             ; Modify visualization
        ;; get the current Vis Tool
        if (~obj_valid((*sPtr).oVisUI)) then break
        oVisTool = ((*sPtr).oVisUI)->GetTool()
        if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break

        ;; Get the edit parameter operation
        ;modVisOpID = 'Operations/Edit/EditParameters' 
        modVisOpID = 'Operations/EditParameters' 
        (*sPtr).oRequestor = (oVisTool->GetByIdentifier(modVisOpID))->GetObjectInstance()

        if obj_isa((*sPtr).oRequestor,'IDLitOpEditParameters') then begin
            ;; get parameters from table
            cw_itParameterPropertysheet_GetParameters,wParm, strParms, $
              idParms, count=count
            if (count gt 0) then begin
                (*sPtr).oRequestor->SetProperty, $
                  PARAMETER_NAMES=strParms,DATA_IDS=idParms
                ;; set vis tool
                (*sPtr).oRequestor->_SetTool,oVisTool
                ;; get macro service
                oSrvMacros = oVisTool->GetService('MACROS')
                ;; save current value and set show_ui to zero
                (*sPtr).oRequestor->GetProperty,show_execution_ui=showUIOrig
                (*sPtr).oRequestor->SetProperty,show_execution_ui=0
                ;; call doaction, which performs the operation and generates
                ;; the needed commands
                oCmd = (*sPtr).oRequestor->DoAction(oVisTool)
                ;; Add this to history explicitly
                if obj_valid(oCmd[0]) then begin
                    oSrvMacros->GetProperty, current_name=currentName
                    oSrvMacros->PasteMacroOperation, (*sPtr).oRequestor, currentName
                endif
                ;; Add this to undo/redo explicitly
                oVisTool->_TransactCommand, oCmd
                ;; restore original values on the operation
                (*sPtr).oRequestor->SetProperty,show_execution_ui=showUIOrig
            endif
            widget_control,(*sPtr).wAction,sensitive=0

            oTool->_SetDirty, 1
        endif
        
    end


    'SET2NEW': begin            ; Set Vis Task to: Insert new visualization in New Vis Tool
        ;; do nothing of button is being de-selected
        if (event.select eq 0) then break

        (*sPtr).visTask = 'NEWVIS'
        (*sPtr).isInsVis = 1
        (*sPtr).isParmEdit = 0
        widget_control,(*sPtr).wAction,set_value='New Visualization' $
                       ,set_uname='NEWVIS',sensitive=0
        
        ;; Update parameter table
        cw_itParameterPropertySheet_SetValue,(*sPtr).wParm,0l $
          ,parameter_editor=(*sPtr).isParmEdit $
          ,insert_visualization=(*sPtr).isInsVis
        ;; update sensitivity of Apply button
        widget_control, (*sPtr).wAction, sensitive= $
                        cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)

    end

    'SET2ADD': begin            ; Set Vis Task to: Add new visualization in existing Vis Tool
        ;; do nothing of button is being de-selected
        if (event.select eq 0) then break

        (*sPtr).visTask = 'ADDVIS'
        (*sPtr).isInsVis = 1
        (*sPtr).isParmEdit = 0
        widget_control,(*sPtr).wAction,set_value='Add Visualization' $
                       ,set_uname='NEWVIS',sensitive=0

        ;; get the current Vis Tool
        if (~obj_valid((*sPtr).oVisUI)) then break
        oVisTool = ((*sPtr).oVisUI)->GetTool()
        if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break
        ;; Get major visualizations currently in the vis tool
        oVis = oVisTool->GetMainVisualizations(count=nVis)
        if (nVis eq 0) then break
        if (nVis gt 1) then oVis = oVis[0]

        ;; Update parameter table
        cw_itParameterPropertySheet_SetValue,(*sPtr).wParm,oVis $
          ,parameter_editor=(*sPtr).isParmEdit $
          ,insert_visualization=(*sPtr).isInsVis
        ;; update sensitivity of Apply button
        widget_control, (*sPtr).wAction, sensitive= $
                        cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)

    end

    'SET2MOD': begin            ; Set Vis Task to: Modify existing visualization
        ;; do nothing of button is being de-selected
        if (event.select eq 0) then break

        (*sPtr).visTask = 'MODVIS'
        (*sPtr).isInsVis = 0
        (*sPtr).isParmEdit = 1
        widget_control,(*sPtr).wAction,set_value='Modify Visualization' $
                       ,set_uname='MODVIS',sensitive=0

        ;; get the current Vis Tool
        if (~obj_valid((*sPtr).oVisUI)) then break
        oVisTool = ((*sPtr).oVisUI)->GetTool()
        if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break
        ;; Get selected item from the visualization
        oItem = (oVisTool->GetSelectedItems())[0]
        
        ;; Update parameter table
        cw_itParameterPropertySheet_SetValue,(*sPtr).wParm,oItem $
          ,parameter_editor=(*sPtr).isParmEdit $
          ,insert_visualization=(*sPtr).isInsVis
        ;; update sensitivity of Apply button
        widget_control, (*sPtr).wAction, sensitive= $
                        cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
    end


    'ADD': begin                ; add item to parameter table
        widget_control, $
          (*sPtr).wAction, $
          sensitive=cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
        ;; do not allow DM to change insVis droplist
        cw_itParameterPropertysheet_LockList,wParm
    end

    'REMOVE': begin             ; remove item from parameter table
        widget_control, $
          (*sPtr).wAction, $
          sensitive=cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
    end
    
    'REMOVE_ALL': begin         ; remove all items from parameter table
        widget_control, $
          (*sPtr).wAction, $
          sensitive=cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
    end
    
    'PROP_SHEET' : begin
        if (tag_names(event,/structure_name) EQ 'WIDGET_CONTEXT') then return
        if (event.identifier eq 'VIS_TYPE') && (event.type eq 0) then $
          widget_control, $
          (*sPtr).wAction, $
          sensitive=cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
    end
    
    'DISMISS': begin
        oShutdown = (*sPtr).oTool->GetService("SHUTDOWN")

        if(~obj_valid(oShutdown))then begin
            oTool->ErrorMessage, title=IDLitLangCatQuery('Error:InternalError:Title'), $
                                 [IDLitLangCatQuery('Error:Framework:CannotAccessShutdown'), $
                                  IDLitLangCatQuery('Error:Framework:ForceShutdown')], severity=2
            obj_destroy,self
            return
        endif
        void = oTool->DoAction(oShutdown->GetFullIdentifier())
        return
    end



    else:
endcase

;help,'In wd_davetool_event...'
;print,'Uname = ',uname
;help,event,/struct

end
;-------------------------------------------------------------------------------


