; $Id$
;###############################################################################
;+
; NAME:
;   wd_DAVETool
;
; PURPOSE:
;   The main user interface widget creation routine for the DAVE Main
;   Tool. This is essentially what the user sees when the tool is
;   launched.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2004
;-
;###############################################################################



;===============================================================================
; wd_DAVETool_cleanup
;
; PURPOSE:
;   Widget cleanup
;
; CATEGORY:
;   DAVE Main Tool
;
; PARAMETERS
;   wChild (in) - The id of the base widget to be destroyed. This
;                 would be the first child widget of the actual tlb
;                 since it was the one that registered this cleanup
;                 procedure.
;
; KEYWORDS:
;
pro wd_DAVETool_cleanup, wChild
compile_opt idl2

if(~widget_info(wChild, /valid))then return

widget_control, wChild, get_uvalue=sPtr

;; Destroy all Vis Tools first!
visToolArr = (*sptr).oTool->GetVisToolObjRefs(count=nVisTool)
exitID = 'Operations/File/Exit'
while (nVisTool--) do begin
    oVisTool = visToolArr[nVisTool]
    exitOpDesc = oVisTool->GetByIdentifier(exitID)
    if (obj_valid(exitOpDesc)) then exitOp = exitOpDesc->GetObjectInstance()
    if (obj_valid(exitOp)) then begin
        oVisTool->SetProperty, prompt=0 ; switch off prompt property so vis tool can be deleted without any prompting
        void = exitOp->DoAction(oVisTool)
    endif
endwhile

ptr_free, (*sPtr).visIDCountPtr,(*sPtr).visToolArrPtr ;, (*sPtr).visTLBListPtr, (*sPtr).visOUIListPtr
if (ptr_valid(sPtr)) then ptr_free, sPtr

end


;===============================================================================
; wd_DAVETool_resize
; 
; PURPOSE:
;   Resize the components of the TLB.
;
; PARAMETERS
;   sPtr (in)  - Pointer to the state structure containing info about
;                the widget components
;
; KEYWORDS:
;
pro wd_DAVETool_resize, sPtr
compile_opt idl2

; the new tlb widget size
widget_control, (*sPtr).wTLB, tlb_get_size=newSize

print,'Resize event handled...'

end


;===============================================================================
; wd_DAVETool_callback
; 
; PURPOSE:
;   Callback routine for handling messages received from the
;   underlying iTool framework.
;
; PARAMETERS
;   wTLB      - Base id of the TLB
;
;   strID     - ID of the message.
;
;   MessageIn - What is the message
;
;   userdata  - Data associated with the message
;
; KEYWORDS:
;
pro wd_DAVETool_callback, wTLB, strID, messageIn, userdata
compile_opt idl2

;print,'In cw_DAVETool_CALLBACK____________________________________________'
;print,'wTLB = ',wTLB
;print,'strID = ',strID
;print,'messageIn = ',messageIn
;print,'userdata = ',userdata
;print,'_________________________________________________________________________'

if (~widget_info(wTLB, /valid)) then return

; Grab the state of the widget from the first child
widget_control, widget_info(wTLB, /child), get_uvalue=sPtr

;; If the message is from the main tool, do the following and then exit
if (strcmp(strid,'/TOOLS/DAVEMAIN',15,/fold_case)) then begin
    if (strcmp(messageIn,'FILENAME',/fold_case)) then begin
        ;; Check the file name changes to display
        ;; Use the new filename to construct the title. 
        ;; Remove the path.
        if (n_elements(userdata) eq 0) then return
        filename = strsplit(userdata, '/\', /extract)
        filename = filename[n_elements(filename)-1]
        ;; Append the filename onto the base title.
        newTitle = filename + ' - ' + (*sPtr).title
        widget_control, wTLB, tlb_set_title=newTitle
    endif
    return
endif

; If this is a 'FOCUS_CHANGE' message then it is possible that a new
; Vis Tool has been selected so make necessary updates
if (strcmp(messageIn,'FOCUS_CHANGE',/fold_case) && (userdata eq 1)) then begin
    oSystem = _IDLitSys_GetSystem()
    oVisTool = oSystem->GetByIdentifier(strid)
    if (obj_isa(oVisTool,'DAVEvisToolbase')) then $
      (*sPtr).oVisUI = oVisTool->GetUI()

endif
oVisUI = obj_valid((*sPtr).oVisUI)? (*sPtr).oVisUI : (*sPtr).oUI
oVisTool = oVisUI->GetTool()
oUI = (*sPtr).oUI
oDAVETool =  (*sPtr).oTool
wParm = (*sPtr).wParm


; Handle messages
case strupcase(messageIn) of
    ;;; Do we need to add panels to the UI?
;     'ADDUIPANELS': begin
;         cw_itpanel_addPanel, (*sPtr).wPanel, userData
;         ;; Update the size of the status bar.
;         geom = widget_info(wTLB, /geometry)
;         cw_itStatusBar_Resize, (*sPtr).wStatus, geom.scr_xsize-2, -1
;         geom = widget_info((*sPtr).wDraw, /geometry)
;         cw_itpanel_resize, (*sPtr).wPanel, -1, geom.scr_ysize
;     end


    ;;; Virtual dims changed
;     'VIRTUAL_DIMENSIONS': begin
;         ;; Retrieve the original geometry (prior to the resize).
;         widget_control, wTLB, tlb_get_size=basesize
;         geom = widget_info((*sPtr).wDraw, /geometry)
;        
;         ;; See if the window shrank.
;         dx = (userdata[0] - geom.xsize) < 0
;         dy = (userdata[1] - geom.ysize) < 0
;        
;         ;; No shrinkage.
;         if ((dx eq 0) && (dy eq 0)) then $
;           break
;        
;         IDLitwdTool_resize, sPtr, dx, dy
;     end
    
    

    'FOCUS_CHANGE': begin
        ;if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break
        case userdata of 
            ;; Received focus
            1: begin

                ;; Get the wid of the node representing this Vis Tool in the Vis Browser
                oVisUI->GetProperty, group_leader=wTLB_VisTool
                widget_control, wTLB_VisTool, get_uvalue=wNode

                ;; Get the currently selected item on the Vis Tool
                ;; and update tree widget selection state to match.
                oItems = oVisTool->GetSelectedItems(count=nItems)
                if (nItems eq 0) then begin
                    ;; Use the window comp. It's id is stored as the uname
                    ;; of wNode
                    oItems = oVisTool->GetbyIdentifier(widget_info(wNode,/uname))
                    nItems = obj_valid(oItems) && obj_isa(oItems,'IDLitWindow')
                endif

                if (nItems gt 0) then begin
                    ;; Unselect any items currently selected in the
                    ;; tree widget
                    wTreeRoot = widget_info(wNode,/parent)
                    wSel = widget_info(wTreeRoot,/tree_select)
                    if (widget_info(wSel[0],/valid)) then $
                      for i = 0,n_elements(wSel)-1 do widget_control, wSel[i], set_tree_select=0

                    ;; Select items selected in the visualization
                    for i=0, nItems-1 do begin
                        idItem= widget_info(wTreeRoot,find_by_uname=oItems[i]->GetfullIdentifier())
                        if (idItem gt 0) then begin
                            selectionChanged = 1
                            widget_control,idItem, /set_tree_select, set_tree_visible=(i eq 0)
                            IDs = (n_elements(IDs) eq 0)? $
                                  oItems[i]->GetfullIdentifier() : $
                                  [IDs,oItems[i]->GetfullIdentifier()]
                        endif
                    endfor

                    ;; Notify observers about selection change message from 'Visualization'
                    if (n_elements(IDs) gt 0) then begin
                        if ((*sPtr).currentTab eq 1) then begin
                            oVisTool->DoOnNotify,'Visualization','SELECTIONCHANGED',IDs
                        endif ;else begin
                            
                        ;; update parameter table with current value
                        cw_itParameterPropertySheet_SetValue,(*sPtr).wParm,oItems[0] $
                          ,parameter_editor=(*sPtr).isParmEdit $
                          ,insert_visualization=(*sPtr).isInsVis

                        ;; update sensitivity of Apply button
                        widget_control, (*sPtr).wAction, sensitive= $
                                        cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
                        ;endelse
                    endif
                endif
                
                ;; Get the current Vis Tool's visualization object and update the parameter table with it
                if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break
                
                oVis = oVisTool->GetMainVisualizations(count=nVis)
                if (nVis le 0) then break
                if (nVis gt 1) then oVis = oVis[0]
                ;oVis->GetProperty, name=visName
                ;oVisTool->GetProperty, knownVis=knownVis
                ;void = where(strupcase(visName) eq strupcase(knownVis), valid)
                ;if (valid) then $
                cw_itParameterPropertySheet_SetValue,(*sPtr).wParm,oVis $; update parameter table with current value
                   ,parameter_editor=(*sPtr).isParmEdit $
                   ,insert_visualization=(*sPtr).isInsVis

            end


            ;; Lost Focus
            0:
            
            else:
        endcase
    end

    'SELECTIONCHANGED' : begin
        case strupcase(strID) of
            'VISUALIZATION': begin

                ;; update parameter table with current value
                cw_itParameterPropertySheet_SetValue,(*sPtr).wParm $
                  ,oVisTool->GetByIdentifier(userdata[0]) $
                  ,parameter_editor=(*sPtr).isParmEdit $
                  ,insert_visualization=(*sPtr).isInsVis

                ;; update sensitivity of Apply button
                widget_control, (*sPtr).wAction, sensitive= $
                                cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
            end

            else: begin
                return

                                ;if (~obj_isa(oVisTool,'DAVEvisToolbase')) then break
                ;; stop observing current object
                oDAVETool->RemoveOnNotifyObserver,(*sPtr).idSelf,(*sPtr).itemID
                ;; if we are in parameter editor mode, get new selection and
                ;; pass it to the parameter table widget
                if (*sPtr).isParmEdit then begin
                    ;; get parm table
                                ;wParm = widget_info((*sPtr).wParamEditBase,/child)
                    ;; get selected item
                    oItem = oDAVETool->GetByIdentifier(component[0])
                    ;; update parm table with current item
                    cw_itParameterPropertysheet_SetValue,wParm,oItem
                    ;; desensitize Apply button
                    widget_control,(*sPtr).wAction,sensitive=0
                    ;; observe current object
                    oVisTool->AddOnNotifyObserver,(*sPtr).idSelf,component[0]
                endif
                if (*sPtr).isInsVis then begin
                    ;; get parm table
                                ;wParm = widget_info((*sPtr).wParamEditBase,/child)
                    ;; ensure that all selected data does indeed exist
                    cw_itParameterPropertysheet_UpdateSensitivity,wID=wParm,/PARAMS_EXIST
                    ;; set sensitivity of Insert button
                    widget_control, (*sPtr).wAction, sensitive= $
                                    cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
                endif
            end
        endcase
    end

    'SETPROPERTY' : begin
        if ((*sPtr).isParmEdit || (*sPtr).isInsVis) then begin
            if (userdata eq 'NAME') then begin
                ;; update the names of all the data items in the propertysheet
                cw_itParameterPropertysheet_UpdateSensitivity,wID=wParm,/param_data
            endif else begin
                ;; get system
;                oSys = (*sPtr).oUI->GetTool()
                oVisTool = (*sPtr).oVisUI->GetTool()
                ;; get selected item
                oItem = oVisTool->GetByIdentifier(strid)
                ;; update parm table with current item
                if obj_isa(oItem,'IDLitParameter') then $
                  cw_itParameterPropertysheet_SetValue,wParm,oItem
            endelse
        endif
        if (*sPtr).isParmEdit then begin
            ;; desensitize Apply button
            widget_control,(*sPtr).wAction,sensitive=0
        endif
        if (*sPtr).isInsVis then begin
            ;; ensure that all selected data does indeed exist
            cw_itParameterPropertysheet_UpdateSensitivity,wID=wParm,/params_exist
            ;; set sensitivity of Insert button
            widget_control, (*sPtr).wAction, sensitive= $
                            cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
        endif
    end

    'FOCUS_GAIN' : begin
        ;; get system
;        oSys = (*sPtr).oUI->GetTool()
        ;; a different tool was selected
        if (*sPtr).isParmEdit then begin
            ;; get parm table
                                ;wParm = widget_info((*sPtr).wParamEditBase,/child)
            ;; get current item from DM and update table
            widget_control,(*sPtr).wDM,get_value=oItem
            cw_itParameterPropertysheet_SetDataSelect,wParm,oItem
            ;; get current tool
            oTool = oSys->_GetCurrentTool()
            ;; get current selected item
            oItem = (oTool->GetSelectedItems())[0]
            ;; update table
            cw_itParameterPropertysheet_SetValue,wParm,oItem
            ;; desensitize Apply button
            widget_control,(*sPtr).wAction,sensitive=0
            ;; observe current object
            if obj_valid(oItem) then $
              oVisTool->AddOnNotifyObserver,(*sPtr).idSelf,oItem->GetFullIdentifier()
        endif
    end

    'FOCUS_LOSS' : begin
        ;; get system
;        oSys = (*sPtr).oUI->GetTool()
        ;; stop observing current object
        oDAVETool->RemoveOnNotifyObserver,(*sPtr).idSelf,(*sPtr).idValue
        ;; clear parm edit table
        if (*sPtr).isParmEdit then begin
            ;; get parm table
                                ;wParm = widget_info((*sPtr).wParamEditBase,/child)
            ;; clear table
            cw_itParameterPropertysheet_SetValue,wParm,0
            cw_itParameterPropertysheet_SetDataSelect,wParm,0
            ;; desensitize Apply button
            widget_control,(*sPtr).wAction,sensitive=0
        endif
        if (*sPtr).isInsVis then begin
            ;; get parm table
                                ;wParm = widget_info((*sPtr).wParamEditBase,/child)
            ;; get current item from DM and update table
            widget_control,(*sPtr).wDM,get_value=oItem
            cw_itParameterPropertysheet_SetDataSelect,wParm,oItem
            ;; ensure that all selected data does indeed exist
            cw_itParameterPropertysheet_UpdateSensitivity, $
              wID=wParm,/REMOVE,/PARAMS_EXIST
            ;; set sensitivity of Insert button
            widget_control, (*sPtr).wAction, sensitive= $
                            cw_itParameterPropertysheet_IsRequiredFullfilled(wParm)
        endif
    end

    else :

endcase


end


;===============================================================================
; wd_DAVETool_PropSheet_Event
; 
; PURPOSE:
;   Event handler for property sheet component of widget created by
;   wd_DAVETool. This sheet displays detailed info about items
;   currently selected in the Data/Visualization tree widgets.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
function wd_DAVETool_PropSheet_Event, event
compile_opt idl2

;print,'In wd_DAVETool_PropSheet_Event____________________________________________'
;help,event,/struct
;print,'_________________________________________________________________________'

;; Retrieve sPtr from the first child of the TLB of wd_DAVETool
wTLB = widget_info(widget_info(widget_info(event.handler,/parent),/parent),/parent)
widget_control, widget_info(wTLB,/child), get_uvalue=sPtr

;; Get the state var for the PS
widget_control, event.id, get_uvalue=statePtr ;, /no_copy

;; Update the oUI tag of the PS state with that of either the Main
;; Tool or the current Vis Tool; depends on which tree widget tab
;; is currently in view.
case (*sPtr).currentTab of
    0: (*statePtr).oUI = (*sPtr).oUI
    1: (*statePtr).oUI = (obj_valid((*sPtr).oVisUI))? (*sPtr).oVisUI : (*sPtr).oUI
endcase
;widget_control, event.id, set_uvalue=state, /no_copy

;; Call the proper event handler for the PS widget.
ret_event = cw_itPropertySheet_Event(event)

if ((strcmp(event.identifier,'Name',/fold_case))[0] && $
    obj_isa(event.component,'IDLitDataContainer')) then begin
    ;;if the name property is being updated for an IDLitDataContainer
    ;;object then a metadata requires modification also
    event.component->GetProperty, type=objType, name=newValue
    switch objType of
        'DAVE1DATASET': begin
            oPSet = event.component->getByType('DAVE1COMMONSTR')
            if (obj_valid(oPSet)) then begin
                oPSet->AddMetaData,'DatasetName',newValue
            endif
            break
        end

        'ASCIICOL':
        'ASCIIGRP':
        'ASCIISPE': begin
            event.component->AddMetaData,'DatasetName',newValue
            break
        end

        else:
    endswitch

endif


case tag_names(event,/structure_name) of
    'WIDGET_PROPSHEET_CHANGE': begin
        if (obj_isa(event.component,'IDLITGRWINSCENE') && $
            (strcmp(event.identifier,'DESCRIPTION',/fold_case))[0]) then begin
            ;; if the 'description' property of the 'window' component
            ;; changes, then also modify the 'name' property. The
            ;; 'description' property is being used for this task b/c,
            ;; for some strange reason, the 'name' property could not
            ;; be altered directly from the property sheet. 

            ;; get the vis tool
            widget_control, event.id, get_uvalue=statePtr
            oVisTool = (*statePtr).oUI->GetTool()            
            
            ;; get the new value of the description property and
            ;; update the name property with it.
            (event.component)->GetProperty,description=newValue
            (event.component)->SetProperty,name=newValue
            
            ;; also update the vis tool's name property
            oVisTool->SetProperty, name=newValue

            ;; send out a notification message about the name property change
            ;; i) originating from the component
            oVisTool->DoOnNotify, (event.component)->GetFullIdentifier(), 'SETPROPERTY', 'NAME'
            ;; ii) originating from the vis tool
            oVisTool->DoOnNotify, oVisTool->GetFullIdentifier(), 'SETPROPERTY', 'NAME'

        endif
    end

    else:
endcase

return, ret_event

end
;-------------------------------------------------------------------------------


;===============================================================================
; wd_DAVETool_paramPropSheet_Event
; 
; PURPOSE:
;   Event handler. Pre-process events created by
;   cw_itParameterPropertySheet before calling the default event
;   handler.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
function wd_DAVETool_paramPropSheet_Event, event
compile_opt idl2

case widget_info(event.id, /uname) of
    'ADD': begin
        widget_control, widget_info(event.handler,/child), get_uvalue=sPtr ; Get the state
        if (~widget_info((*sPtr).wAdd,/sensitive)) then return, 0

        knownDataTypes = ['DAVE1COMMONSTR','ASCIICOL','ASCIIGRP','ASCIISPE']
        void = where(knownDataTypes eq (*sPtr).dataType, isKnown)

        propID = widget_info((*sPtr).wProp,/propertysheet_selected)

;        if ((propID eq 'VIS_TYPE') && (*sPtr).dataPset && isKnown) then begin
        if (propID eq 'VIS_TYPE') then begin 
           ;; ie if no specific vis parameter is selected try making
           ;; the assignment automatically based on the axis type of
           ;; the data item(s) selected.
           matchFound = wd_DAVETool_paramPropSheet_AddData(sPtr)

           ;; if matchFound then exit event handler else execute
           ;; default event handler
           return, (matchFound gt 0)? event : $
                   cw_itParameterPropertysheet_Event(event)
           
        endif else begin
           ;; execute default event handler
           return, cw_itParameterPropertysheet_Event(event)
        endelse
    end

    else: begin
        ;; execute default event handler
        return, cw_itParameterPropertysheet_Event(event)
    end
    
endcase

end
;-------------------------------------------------------------------------------



;===============================================================================
; wd_DAVETool_SetWidget
;
; Purpose:
;   Based on IDLitwdDataManager_SetWidget
;   Sets or destroys the subwidgets needed for the parameter editor
;   and the insert visualization dialogs.
;
; Parameter:
;   wID  - The widget ID of the base where the parameter editor dialog
;          is to be inserted.
;
pro wd_DAVETool_SetWidget,wID, wParamEditBase, $
  PARAMETER_EDITOR=parametereditor, $
  INSERT_VISUALIZATION=insertvisualization, $
  TITLE=title, $
  VALUE=oValue, REQUESTOR=oRequestor
compile_opt idl2

;; get state from first child of wID
widget_control,widget_info(wID,/child),get_uvalue=sPtr

;; turn off updates
widget_control,wID,update=0

;; save requesting operation
if n_elements(oRequestor) then (*sPtr).oRequestor = oRequestor

(*sPtr).isInsVis = keyword_set(insertvisualization)
(*sPtr).isParmEdit = keyword_set(parametereditor) && ~(*sPtr).isInsVis
(*sPtr).isVisible = 1b
;; observe visualizations for selection changes
(*sPtr).oUI->AddOnNotifyObserver, (*sPtr).idSelf, 'Visualization'

;; get parameter and ins vis widgets
;wParm = widget_info((*sPtr).wParamEditBase,/child)

if keyword_set(insertvisualization) || $
  keyword_set(parametereditor) then begin
    if ~widget_info((*sPtr).wParm,/valid_id) then $
      wParm = cw_itParameterPropertySheet(wParamEditBase $
                                          ,(*sPtr).oUI $
                                          ,value=oValue $
                                          ,L_XSIZE=(*sPtr).l_xsize $
                                          ,R_XSIZE=(*sPtr).r_xsize $
                                          ,YSIZE=(*sPtr).b_ysize $
                                          ,INSERT_VISUALIZATION=insertvisualization $
                                         )
    (*sPtr).wParm = wParm

    ;; change the default event handler for wParm. We need to
    ;; pre-process some events before calling the default handler.
    widget_control, wParm, event_func='wd_DAVETool_paramPropSheet_Event'

    ;; let table know which data item is current selected
    oSys = (*sPtr).oUI->GetTool()
    oItem = oSys->GetByIdentifier((*sPtr).itemID)
    cw_itParameterPropertysheet_setDataSelect,wParm,oItem
    
    if (*sPtr).isParmEdit then begin
        ;; observe current item
        (*sPtr).idValue = oValue->GetFullIdentifier()
        (*sPtr).oUI->AddOnNotifyObserver, (*sPtr).idSelf, (*sPtr).idValue
    endif
    
    ;; update table with current value
    cw_itParameterPropertySheet_SetValue,wParm,oValue, $
      PARAMETER_EDITOR=parametereditor, $
      INSERT_VISUALIZATION= $
      insertvisualization
    
    ;; if title was specified, save it for future use
;    if (keyword_set(TITLE)) then begin
;        CASE 1 OF
;            keyword_set(parametereditor) : (*sPtr).PEtitle = title
;            keyword_set(insertvisualization) : (*sPtr).IVtitle = title
;            else : (*sPtr).DMtitle = title
;        endCASE
;    endif
    
    ;; use appropriate title
;    CASE 1 OF
;        keyword_set(title) : title=title
;        keyword_set(parametereditor) : title=(*sPtr).PEtitle
;        keyword_set(insertvisualization) : title=(*sPtr).IVtitle
;        else : title=(*sPtr).DMtitle
;    endCASE
;    widget_control,wID,base_set_title=title
    
    ;; set value of DOACTION button
    ;; desensitise button because nothing has yet changed
    if keyword_set(parametereditor) then begin
        widget_control,(*sPtr).wAction,set_value='Modify Visualization' $
                       ,set_uname='MODVIS',sensitive=0
    endif else begin
        widget_control,(*sPtr).wAction,set_value='New Visualization' $
                       ,set_uname='NEWVIS',sensitive=0
    endelse
endif else begin
    ;; destroy parameter table and insert vis droplist
    if widget_info(wParm,/valid_id) then begin
        widget_control,wParm,/destroy
    endif
    ;; reset title and buttons
    widget_control,(*sPtr).wAction,set_value='New Visualization' $
                   ,set_uname='NEWVIS',sensitive=0
endelse

widget_control,wID,update=1

end


;===============================================================================
; wd_DAVETool
;
; PURPOSE:
;   The main user interface widget creation routine for the DAVE Main
;   Tool.
;
; CATEGORY:
;   DAVE Main Tool
;
; PARAMETERS
;   oTool (in) - The object of reference of the tool which owns this widget
;
;
; KEYWORDS:
;   title (in) - The widget title
;
;   USER_INTERFACE (out)  - Object reference of the UI tool.
;
pro wd_DAVETool, oTool, title=title, user_interface=oUI, _REF_EXTRA=_extra
compile_opt idl2

; oTool must be a valid object
if (~OBJ_VALID(oTool)) then $
  message, 'Tool is not a valid object.'

; If splash required, initialize it here.

; Change to busy cursor
WIDGET_CONTROL, /HOURGLASS

; Ensure title is set
title = (n_elements(title) gt 0)? title[0] : 'Data Analysis and Visualization Environment'
DMTitle = title
PETitle = title
IVTitle = title

; The tlb - plus menubar
wTLB = widget_base(/col,mbar=wMenubar,title=title,map=0,/context_events $
                   ,/kbrd_focus_events,/tlb_size_events,/tlb_kill_request_events $
                   ,_EXTRA=_extra)


;Create a new UI tool object.
oUI = obj_new('IDLitUI', oTool, group_leader=wTLB)

; Create top level menus
horizontalSpace = 'Operations/   '
wFile       = cw_Itmenu(wMenubar, oUI, 'Operations/File')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void      = cw_Itmenu(wMenubar, oUI, 'Operations/DataIO')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void      = cw_Itmenu(wMenubar, oUI, 'Operations/DataRed')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
;wEdit       = cw_Itmenu(wMenubar, oUI, 'Operations/Visualization')
;void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void       = cw_Itmenu(wMenubar, oUI, 'Operations/Analysis')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void       = cw_Itmenu(wMenubar, oUI, 'Operations/Planning')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void       = cw_Itmenu(wMenubar, oUI, 'Operations/Misc')
void        = cw_Itmenu(wMenubar, oUI, horizontalSpace)
void       = cw_Itmenu(wMenubar, oUI, 'Operations/Help')
;menuBarCharLen += 7             ; 'Tools'
;wInsert     = cw_Itmenu(wMenubar, oUI, 'Operations/Insert')
;wOperations = cw_Itmenu(wMenubar, oUI, 'Operations/Operations')
;wWindow     = cw_Itmenu(wMenubar, oUI, 'Operations/Window')


; Create Toolbar containers
wToolbar = widget_base(wTLB,/row,xpad=0,ypad=0,space=7,/context_events)

wTool1 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/File',sensitive=1)
wTool2 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/Edit',sensitive=1)
;wTool3 = CW_ITTOOLBAR(wToolbar, oUI, 'Operations/WindowTools',sensitive=0)
;wTool4 = CW_ITTOOLBAR(wToolbar, oUI, 'MANIPULATORS',/exclusive,sensitive=0)
;; Annotation portion of the toolbar.
;wTool4 = CW_ITTOOLBAR(wToolbar, oUI,'MANIPULATORS/ANNOTATION',/exclusive,sensitive=0)

; The main body (wBody) which consists of 
; - A tab base (wTabBase) on left containing
;   * a data browser represented by tree widgets
;   * a visualization browser represented by tree widgets
; - A base to the right (wPSBase) containing a property sheet for displaying
;   details of selected items in the data/vis browsers.
;
wBody = widget_base(wTLB,/row,/context_events,/frame)
wTabBase = widget_tab(wBody,tab_mode=0) ; set tab mode=0 for xplatform compt. since it is always 0 for motif.
wPSBase = widget_base(wBody,/frame,uname='PSBASE')
wDMTab = widget_base(wTabBase,/row,title='Data Browser',uname='DMTAB')
wVMTab = widget_base(wTabBase,/row,title='Visualization Browser',uname='VMTAB')

;; base for the possible parameter editor / insert vis widget
wParamEditBase = widget_base(wTLB,/frame)


; The data manager browser section
;-----------------------------------------------------------
xsDM = 350                      ; pixels
ysDM = 300 ;220
unDM = "DAVEDATAMANAGER"
wDM = cw_daveDataManager(wDMTab,oUI,uname=unDM,xsize=xsDM,ysize=ysDM,_extra=_extra)


; The visualization browser section
;-----------------------------------------------------------
xsVM = xsDM
ysVM = ysDM
wVM = cw_daveVisManager(wVMTab,oUI,xsize=xsVM,ysize=ysDM $
                        ,/construct_base)

; The Property Sheet section
;-----------------------------------------------------------
xsPS = xsDM
ysPS = 0.60*ysDM
wPS = CW_ITPROPERTYSHEET(wPSBase, oUI $
                         ,scr_xsize=xsPS $
                         ,scr_ysize=ysDM $
                         ,/commit $ 
                         ,/noRegisterVis $
                         ,type='Visualization Browser' $
                         ,event_func='wd_DAVEtool_PropSheet_event' $
                                ;,/sunken_frame $ ;,ysize=ysPS $
                        )


wButtonBase = widget_base(wTLB,/row,/frame)
text = 'Visualization Task:'

wButtonBase0 = widget_base(wButtonBase, row=1, scr_xsize=0.2*(xsDM+xsPS))
wButtonBase1 = widget_base(wButtonBase, col=3, scr_xsize=0.5*(xsDM+xsPS),/exclusive,/grid)
wButtonBase2 = widget_base(wButtonBase, row=1, scr_xsize=0.3*(xsDM+xsPS))

void = widget_label(wButtonBase0,value=text,/align_right)

tooltip = 'Select this option when creating a new visulization in a new window'
void = widget_button(wButtonBase1,value='New',tooltip=tooltip,uname='SET2NEW',/align_left)
widget_control, void, set_button=1
tooltip = 'Modify the parameters of an existing visulization'
void = widget_button(wButtonBase1,value='Modify',tooltip=tooltip,uname='SET2MOD',/align_left)
tooltip = 'Select to add a new visualization in an existing window'
void = widget_button(wButtonBase1,value='Add',tooltip=tooltip,uname='SET2ADD',/align_left)

tooltip = 'Press to create/modify visualization as specified in the parameter editor' 
wAction = widget_button(wButtonBase2,value='Create Visualization',tooltip=tooltip $
                        ,uname='NEWVIS',sensitive=0,/align_center)

; The status bar
tlbGeom = widget_info(wTLB, /geometry) ; get dimensions of tlb
wStatus = cw_itStatusBar(wTLB, oUI, xsize=tlbGeom.scr_xsize-9)
; wVM also needs a reference to wStatus.
; wStatus is actually a base widget; wVM really requires the id of the
; first widget_label that was created. Inspection of cw_itStatusBar,
; shows how to obtain this.
widget_control, widget_info(wStatus,/child), get_uvalue=stateStatus
widget_control, wVM, get_uvalue=state_VM, /no_copy
(state_VM).wStatus =  widget_info(wStatus, find_by_uname=stateStatus.segIds[0])
widget_control, wVM, set_uvalue=state_VM, /no_copy

;; Make our window
wDraw = CW_ITWINDOW(wTLB, oUI, $
                    DIMENSIONS=[1,1], $
                    VIRTUAL_DIMENSIONS=[1,1])



; Get the first child widget
wChild = widget_info(wTLB,/child)

; If applicable, relocate widget to requested location

; Realize TLB
centertlb, wTLB
widget_control, wTLB, /realize

; Retrieve start dimensions of TLB for storage
widget_control, wTLB, tlb_get_size=basesize

; Register the tlb as a widget with the UI tool object, creating an adapter object 
; whose identifier is returned.
; The specified callback function handles messages sent to this widget.
idSelf = oUI->RegisterWidget(wTLB, 'wdDAVETool','wd_DAVETool_callback' $
                             ,DESCRIPTION="DAVE Main Tool" $
                            )   ; /floating causes this widget to only be 
                                ; visible if the associated iTool is
                                ; "current," and the widget will be
                                ; destroyed when the iTool is destroyed.

; Make the above adapter (idSelf) listen to messages from the Main Tool
; Whenever messages originate from oTool, idSelf's callback function will be called!
oUI->AddOnNotifyObserver, idSelf, oTool->GetFullIdentifier()

; Known valid plottablable data types for dataset containers
plottableContainerDataTypes = ['DAVE1COMMONSTR','ASCIICOL','ASCIIGRP','ASCIISPE']

; Create state structure to store information needed by the other event handling procedures:      
sPtr=ptr_new({oTool:oTool $
              ,oUI:oUI $        ; UI object ref of the Main Tool
              ,oVisUI:obj_new() $ ; placeholder for the UI object ref of the current Vis Tool
              ,oRequestor:obj_new() $ ; operation that requested this
              ,wTLB:wTLB $      ; the TLB
              ,idSelf:idSelf $  ; object identifier of this TLB as registered with oUI
              ,wDM:wDM $        ; base widget id for data manager
              ,wTabBase:wTabBase $ ; widget id of the tab base
              ,wAction:wAction $ ; widget id of the button that initiates creation/modification of vis
              ,wVM:wVM $        ; base widget id for vis manager
              ,wPS:wPS $        ; base widget id for prop sheet
              ,wParm:0L $       ; 
              ,wStatus:wStatus $ ; widget id of status bar
              ,currentTab:0B $   ; which tab is current Data=0, Vis=1
              ,l_xsize: xsDM $
              ,r_xsize: xsPS $
              ,t_ysize: 6 $     ;9 Prop Sheet height in terms of nos of fields
              ,b_ysize: 6 $     ;9 " "
              ,title:title $
              ,basesize:basesize $
              ,itemID:'' $      ; selected item in DM
              ,visTask:'NEWVIS' $ ; 
              ,isParmEdit: 0b $ ; boolean flag
              ,isInsVis: 0b $   ; boolean flag
              ,isVisible: 1b $  ; is widget mapped?
              ,plotLabel: '' $  ; label associated with a new visualization
              ,plotContDataType:plottableContainerDataTypes $ ; Known valid plottablable data types for dataset containers
              ,visIDCountPtr:ptr_new({dummy:0}) $ ; Maintain a count of all vis types created so far
              ,visToolArrPtr:ptr_new(objarr(1)) $ ; Maintain a list of all Vis Tool object refs
             })

; Store state structure in first child widget of TLB
widget_control, wChild, set_uvalue=sPtr, kill_notify="wd_DAVETool_cleanup"


; Insert the Parameter Editor
;-----------------------------------------------------------
wd_DAVETool_SetWidget,wTLB, wParamEditBase $
  ,/insert_visualization $
  ,value=obj_new()

; Make the TLB visible
widget_control, wTLB, /map


; Finally, register with xmanager
xmanager, 'wd_DAVETool', wTLB, /just_reg, /no_block
; We need to use /JUST_REG so that we keep going in IDL Runtime/VM.
; IDLituiStartTool will actually start event processing for us.

; Update the app title to include the filename for this tool
oTool->GetProperty, tool_filename=filename
wd_DAVETool_Callback, wTLB, oTool->GetFullIdentifier(),'FILENAME',filename

end
