; $Id$
;###############################################################################
;+
; NAME:
;   wd_HFBSFWSTool
;
; PURPOSE:
;   The main user interface widget creation routine for the HFBS Reduction
;   Tool. This is essentially what the user sees when the tool is launched.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; September 2011
;-
;###############################################################################


;-------------------------------------------------------------------------
; wd_HFBSFWSTool_callback
;+
; Purpose:
;   Callback routine for the tool interface widget, allowing it to
;   receive update messages from the system.
;
; Parameters:
;   wBase     - Base id of this widget
;
;   strID     - ID of the message.
;
;   MessageIn - What is the message
;
;   userdata  - Data associated with the message
;-
pro wd_HFBSFWSTool_callback, wBase, strID, messageIn, userdata
compile_opt idl2

; print,'In wd_HFBSFWSTool_CALLBACK____________________________________________'
; print,'strID = ',strID
; print,'messageIn = ',messageIn
; print,'userdata = ',userdata
; print,'_________________________________________________________________________'


if (~WIDGET_INFO(wBase, /VALID)) then $
  return

;; Grab the state of the widget
WIDGET_CONTROL, WIDGET_INFO(wBase, /CHILD), GET_UVALUE=pState

case STRUPCASE(messageIn) of
    
    ;;
    'SETPROPERTY': begin
        if ((strcmp(userdata,'NAME',/fold_case))[0]) then begin
            (*pState).oTool->GetProperty, name=name,tool_filename=filename
            (*pState).title = name
            widget_control, wbase, tlb_set_title=name + ' [' + filename + ']'
        endif
    end
    
    ;; Check the file name changes to display
    'FILENAME': begin
        ;; Use the new filename to construct the title.
        ;; Remove the path.
        filename = STRSPLIT(userdata, '/\', /EXTRACT)
        filename = filename[N_ELEMENTS(filename)-1]
                                ; Append the filename onto the base title.
        newTitle =  filename + ' - ' + (*pState).title
        WIDGET_CONTROL, wBase, TLB_SET_TITLE=newTitle

        ;; if the tool has been modified (indicated by *) then the
        ;; DAVE Tool has to be notified.
        if (strpos(filename,'*',/reverse_search) ne -1) then begin
            widget_control, wBase, get_uvalue=wNode
            if ((n_elements(wNode) le 0) || ~widget_info(wNode,/valid)) then return
            widget_control, wNode, get_uvalue=sPtr
            wTLB_HFBSFWSTool = (*sPtr).wTLB_HFBSFWSTool
            widget_control, widget_info(wTLB_HFBSFWSTool,/child), get_uvalue=sPtr_HFBSFWSTool
            ((*sPtr_HFBSFWSTool).oTool)->_SetDirty, 1

            (*pState).oTool->_SetDirty, 0
        endif
    end
    
    ;; A panel was added to the tool. See if our size changed.
    'ADDUIPANELS': begin
        wd_HFBSFWSTool_resize, pState, 0, 0
    end
        
    ;; The show/hide was changed. See if our size changed.
    'SHOWUIPANELS': begin
        wd_HFBSFWSTool_resize, pState, 0, 0
    end
    
    ;; Virtual dims changed
    'VIRTUAL_DIMENSIONS': begin
        ;; Retrieve the original geometry (prior to the resize).
        WIDGET_CONTROL, wBase, TLB_GET_SIZE=basesize
        geom = WIDGET_INFO((*pState).wDraw, /GEOMETRY)
        
        ;; See if the window shrank.
        dx = (userdata[0] - geom.xsize) < 0
        dy = (userdata[1] - geom.ysize) < 0
        
        ;; No shrinkage.
        if ((dx eq 0) && (dy eq 0)) then $
          break
        
        IDLitwdTool_resize, pState, dx, dy
    end
    
    ;; The sensitivity is to be changed
    'SENSITIVE': begin
        WIDGET_CONTROL, wBase, SENSITIVE=userdata
    end
    
    else:                       ; do nothing
    
 endcase

; Fudge!
; To resolve issue when undo/redo buttons are desensitized when
; selections are made on visualizations. Setting the viz tool
; as the current tool appears to fix it.
oSystem = _IDLitSys_GetSystem()
oCurTool = oSystem->_GetCurrentTool()
if (obj_valid(oCurTool) && $
    (oCurTool eq (*pState).oTool)) then begin
   oCurrent = (*pState).oTool->GetService("SET_AS_CURRENT_TOOL")
   void = (*pState).oTool->DoAction(oCurrent->GetFullIdentifier())
endif

; Force viz tool state to appear unmodified
(*pState).oTool->_SetDirty, 0

end


;;-------------------------------------------------------------------------
;; wd_HFBSFWSTool_resize
;;
;; Purpose:
;;    Called when the user has resize the TLB of this tool interface.
;;    Will recalculate the size of the major elements in the
;;    interface.
;;
;; Parameters:
;;   sPtr   - pointer to the state struct for this widget.
;;
;;   deltaW   - The change in the width of the interface.
;;
;;   deltaH   - The change in the height of the interface.
;
pro wd_HFBSFWSTool_resize, sPtr, deltaW, deltaH
compile_opt idl2

; Retrieve the original geometry (prior to the resize)
; of the draw widget.
drawgeom = WIDGET_INFO((*sPtr).wDraw, /GEOMETRY)

; Compute the updated dimensions of the visible portion
; of the draw widget.
;newVisW = (drawgeom.draw_xsize + deltaW) > (*sPtr).minsize[0]
;newVisH = (drawgeom.draw_ysize + deltaH) > (*sPtr).minsize[1]
newVisW = (*sPtr).baseWin[0] + deltaW   ;drawgeom.draw_xsize + deltaW
newVisH = (*sPtr).baseWin[1] + deltaH   ;drawgeom.draw_ysize + deltaH

;different = newVisW ne drawgeom.draw_xsize || newVisH ne drawgeom.draw_ysize
;if (~different) then return

isUpdate = WIDGET_INFO((*sPtr).wBase, /UPDATE)

; If update turned off on unix, draw window won't resize properly.
; So just turn off update on Windows.
if (!version.os_family eq 'Windows') then begin
    if (isUpdate) then $
        widget_control, (*sPtr).wBase, UPDATE=0
endif else begin
   ; On Unix make sure update is on.
   if (~isUpdate) then $
       widget_control, (*sPtr).wBase, /UPDATE
endelse

; Update the draw widget dimensions and scrollbars.
CW_ITWINDOW_resize, (*sPtr).wDraw, newVisW, newVisH

; Update the toolbar row to be the same width as the status width.
;WIDGET_CONTROL, (*sPtr).wToolbar, SCR_XSIZE=deltaW + statusGeom.scr_xsize
WIDGET_CONTROL, (*sPtr).wToolbar, SCR_XSIZE=(*sPtr).baseSB[0] + deltaW

; Update the left panel: the data manager and the property sheet
cw_HFBSFWSDataManager_Resize, (*sPtr).wDM, (*sPtr).baseDM[0], deltaH/2+(*sPtr).baseDM[1]
;psGeom = widget_info((*sPtr).wPS, /geometry)
widget_control, (*sPtr).wPS, scr_ysize = deltaH/2 + (*sPtr).basePS[1]

; Update the statusbar to be the same width as the draw + panel.
;statusGeom = widget_info((*sPtr).wStatus, /geometry)
;cw_itStatusBar_Resize, (*sPtr).wStatus, deltaW + statusGeom.scr_xsize
;panelGeom = widget_info((*sPtr).wPanel,/geometry)
;cw_itStatusBar_Resize, (*sPtr).wStatus, (*sPtr).baseSB[0] + deltaW
cw_itStatusBar_Resize, (*sPtr).wStatus, newVisW + (*sPtr).baseDM[0]

if (isUpdate && ~WIDGET_INFO((*sPtr).wBase, /UPDATE)) then $
    widget_control, (*sPtr).wBase, /UPDATE


; Retrieve and store the new top-level base size.
if (WIDGET_INFO((*sPtr).wBase, /REALIZED)) then begin
    WIDGET_CONTROL, (*sPtr).wBase, TLB_GET_SIZE=basesize
    (*sPtr).basesize = basesize
endif
end


;;-------------------------------------------------------------------------
;; wd_HFBSFWSTool__cleanup
;;
;; Purpose:
;;   Called when the widget is dying, allowing the state ptr to be
;;   released.
;;
;; Parameters:
;;    wChild   - The id of the widget that contains this widgets
;;               state.
;;
pro wd_HFBSFWSTool_cleanup, wChild

compile_opt idl2

    if (~WIDGET_INFO(wChild, /VALID)) then $
        return

    WIDGET_CONTROL, wChild, GET_UVALUE=pState
    if (PTR_VALID(pState)) then $
        PTR_FREE, pState

end


;===============================================================================
; wd_HFBSPref_PropSheet_Event
; 
; PURPOSE:
;   Event handler for property sheet component of widget created by
;   wd_HFBSFWSTool. This sheet displays detailed info about items
;   currently selected in the Data/Visualization tree widgets.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
function wd_HFBSFWSTool_PropSheet_Event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_HFBSFWSTool_PropSheet_Event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;;print,'In wd_HFBSFWSTool_PropSheet_Event____________________________________________'
;;help,event,/struct
;;print,'_________________________________________________________________________'
;
;


;; Get our state var
wChild = widget_info(event.top, /child)
widget_control, wChild, get_uvalue=sPtr

oUI = (*sPtr).oUI
oTool = oUI->GetTool()
oSystem = oTool->GetSystem()

;; Call the proper event handler for the PS widget.
;help, event, /struct
;ret_event = cw_itPropertySheet_Event(event)

; switch on the busy symbol for cursor
widget_control, /HOURGLASS

;;******************************
;-- Record change to prop sheet
;-- Act on change to prop sheet
case tag_names(event, /structure_name) of

    'WIDGET_PROPSHEET_CHANGE': begin    
       ; Get the value of the changed property
       if (event.proptype eq 0) then begin
          ;; ==> a userdef property type so handle it appropriately
          status = oTool->EditUserDefProperty(oTool, event.identifier)
       endif else begin
          ; Update the property with the tool (event.component)
          value = widget_info(event.id, component=event.component,property_value=event.identifier)
;          if (strcmp(event.identifier,'LATTICEPARAMETERS')) then begin
;            ; check correctness of lattie parameter entered
;            toks = strsplit(value,' ',/extract,count=nToks)
;            if (ntoks ne 6) then begin
;               title = 'Lattice Parameters'
;               msg = 'Only '+strtrim(string(ntoks),2)+' values were entered!'
;               msg = [msg,'Specify ALL 6 lattice parameters: a b c alpha beta gamma']
;               oTool->ErrorMessage,msg,title=title,severity=0
;               return, 0
;;TODO - ensure this return, 0 does not inadvertently undo a previous property change 
;            endif
;          endif
          ; Set the component's property value. 
          event.component->SetPropertyByIdentifier, event.identifier, value 
       endelse


       id = event.identifier
       switch (id) of
;          'SAMPTYPEFLAG': begin
;            
;            break
;          end

          'DEFAULTINTENSITYFLAG': begin
            oTool->GetProperty, defaultIntensityFlag=defIntFlag, sampTypeFlag=sampTypeFlag
            noHide = (defIntFlag eq 0) && (sampTypeFlag eq 1)  ; don't hide if not using default intensity and single crystal sample
            oTool->SetPropertyAttribute, 'maxIntensity', hide=~noHide
            oTool->SetPropertyAttribute, 'minIntensity', hide=~noHide    
            ; continue to 'MAXINTENSITY' case
          end
          
          'REBINFLAG': begin
            oTool->GetProperty, rebinFlag = rebinFlag
            oTool->SetPropertyAttribute, 'binWidth', hide=~rebinFlag
            wd_HFBSFWSTool_sendPlotEvent, oUI
            break
          end

          'BINWIDTH': begin
            wd_HFBSFWSTool_sendPlotEvent, oUI
            break
          end

          'MININTENSITY':
          'MAXINTENSITY': begin
            oTool->UpdatePlotIntensityRange
            break
          end
    


          'MSDINDEX': begin
            ; this is only relevant when 'Examining MSD Fits' is the selected yAxis variable
            oTool->GetProperty, yAxisVar=yAxisVar
            if (yAxisVar ne 2) then break ; otherwise continue onto the next case which generates a new plot
          end
          'INTDETSTRING':
          'XAXISVAR':
          'YAXISVAR': begin
            ; the X or Y axis variable has changed so redo plots
            wd_HFBSFWSTool_sendPlotEvent, oUI
            break
          end

          'MIDTEMPFLAG':
          'MSDDETSTRING':
          'XAXISVAR':
          'MONNORMFLAG': begin
            ; get all the data and update the calculation
            oTool->GetProperty, sampContRef=oCont
            n = oCont->Count()
            for i=0,n-1 do begin
              oItem = oCont->Get(position=i)
              if (~obj_isa(oItem, "IDLitParameterSet")) then continue
              oItem->GetProperty, type = dataType
              isSamp = strcmp(dataType,'samp',/fold_case)
              if (isSamp) then oTool->UpdateCalculation, oItem, updateMSD=strcmp(id,'MSDDETSTRING',/fold_case) ; only update MSD if msddetstring changed
            endfor
            
            ; then refresh the plot
            wd_HFBSFWSTool_sendPlotEvent, oUI
            break
          end

         else: print, id
         
       endswitch
    end

    else:
endcase

oTool->RefreshPropertySheet
return, 1

end
;-------------------------------------------------------------------------------



;;-------------------------------------------------------------------------
;; wd_HFBSFWSTool_sendPlotEvent
;;
;; Purpose:
;;    Generate and send an event to update the plot window
;;
;; Parameters:
;;    event    - The widget event to process.
;;
pro wd_HFBSFWSTool_sendPlotEvent, oUI, oItem   ;, dataType
compile_opt idl2

oUI->GetProperty, group_leader=wTLB
wChild = WIDGET_INFO(wTLB, /CHILD)
WIDGET_CONTROL, wChild, GET_UVALUE=sPtr
oTool = oUI->Gettool()

if (n_elements(oItem) eq 0) then begin
   ; retrieve list of all _selected_ datasets from the data manager
   wDM = (*sPtr).wDM ; the data manager widget ID
   widget_control, wDM, get_uvalue=sPtrDM ; the data manager state ptr
   selIDs = (*(*sPtrDM).pdataids)  ; currently selected identifiers
   oItem = oTool->GetByIdentifier(selIDs[0])
   if (~obj_valid(oItem)) then return
   n = n_elements(selIDs)
endif else begin
   if (~obj_valid(oItem)) then return
   selIDs = oItem->GetfullIdentifier()
   n = 1
endelse

;; retrieve the widgetID of the DM tree root node
oItem->GetProperty, type=dataType
case strlowcase(dataType) of
   'bkgd': rootID = 'BKGDDATAMANAGER'
   'samp': rootID = 'SAMPDATAMANAGER'
   'van': rootID = 'VANDATAMANAGER'
   else: return
endcase
wRoot = cw_HFBSFWSdatamanager_getWidgetID((*sPtr).wDM, rootID)

oTool->DisableUpdates

overplot = 0  ; first plot should not be an overplot
for i = 0, n-1 do begin
   plotEvent = {cw_hfbsfwsDataManager, $
                ID:wRoot, $
                TOP:(*sPtr).wBase, HANDLER:(*sPtr).wBase, $
                clicks:1, $
                overplot:overplot, $
                lastplot:(i eq n-1), $
                identifier:selIDs[i], $
                position:i}
   wd_HFBSFWSTool_event, plotEvent
   overplot = 1  ; subsequent plots should be overplotted!
endfor

oTool->EnableUpdates

end


;;-------------------------------------------------------------------------
;; wd_HFBSFWSTool_Event
;;
;; Purpose:
;;    Event handler for the tool interface IDL widget.
;;
;; Parameters:
;;    event    - The widget event to process.
;;
pro wd_HFBSFWSTool_event, event
compile_opt idl2


; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_HFBSFWSTool_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;help,'wd_HFBSFWSTool_event called...'

;; Get our state
wChild = WIDGET_INFO(event.handler, /CHILD)
WIDGET_CONTROL, wChild, GET_UVALUE=sPtr

oTool = (*sPtr).oUI->GetTool()
oSystem = oTool->GetSystem()

;   help,event, /struct
;   help,''
;help,TAG_NAMES(event, /STRUCTURE_NAME)
case TAG_NAMES(event, /STRUCTURE_NAME) of

    ;; Kill the widget?
    'WIDGET_KILL_REQUEST': begin

       ; Check whether or not to save current session before closing it
       oTool->GetProperty, modifiedStatus=modifiedFlag, working_directory=workDir
       if (modifiedFlag) then begin
          ; If our current state is modified, prompt to save first.
          ;outfile = workDir+projName+'.fisv'
          status = oTool->PromptUserYesNo( $
              "Current session is modified! Save changes?", $
              answer, $
              DEFAULT_NO=0, $
              TITLE='Save', $
              /CANCEL)
      
          if (status && (answer eq 1)) then begin
             void = oTool->writeProject()
          endif
          oTool->SetProperty, prompt=0, modifiedStatus=0  ; don't prompt to exit next
       endif


        ;; Delegate the delete to the Exit operation that is defined for this tool
        exitID = 'Operations/ExitOp'
        exitOpDesc = oTool->GetByIdentifier(exitID)
        if (obj_valid(exitOpDesc)) then exitOp = exitOpDesc->GetObjectInstance()
        if (obj_valid(exitOp)) then void = exitOp->DoAction(oTool)
        return
    end

    ;; Focus change
    'WIDGET_KBRD_FOCUS': begin
        if (obj_valid((*sPtr).oUI) && event.enter) then begin
            oTool = (*sPtr).oUI->GetTool()
            oCurrent = oTool->GetService("SET_AS_CURRENT_TOOL")
            void = oTool->DoAction(oCurrent->GetFullIdentifier())
            ;; DONT DO THIS our tools could enter a focus loop in the
            ;; following situation:
            ;;    IDL> Iplot & iPlot
            ;; widget_control, (*sPtr).wDraw, /input_Focus
        endif
    end

    ;; The TLB was resized
    'WIDGET_BASE': begin
        ; Compute the size change of the base relative to
        ; its stored former size.
        WIDGET_CONTROL, event.top, TLB_GET_SIZE=newSize
;        newSize[0] = newSize[0] > (*sPtr).minSize[0]
;        newSize[1] = newSize[1] > (*sPtr).minSize[1]
;        deltaW = newSize[0] - (*sPtr).basesize[0]
;        deltaH = newSize[1] - (*sPtr).basesize[1]
        deltaW = (newSize[0] - (*sPtr).minSize[0]) > 0.0
        deltaH = (newSize[1] - (*sPtr).minSize[1]) > 0.0
        ; Bail if no change.
;        if (deltaW le 0 && deltaH le 0) then begin
;            widget_control, (*sPtr).wBase, xsize=(*sPtr).minSize[0], ysize=(*sPtr).minSize[1]
;            break
;        endif
        wd_HFBSFWSTool_resize, sPtr, deltaW, deltaH
        end
        

    else: ; do nothing

endcase


; In addition, need to deal with selections from the data manager.
uname = widget_info(event.id, /uname)

colors = [[0,0,0] $     ; Black
         ,[255,0,0] $   ; Red
         ,[0,255,0] $   ; Green
         ,[0,0,255] $   ; Blue
         ,[100,0,0] $   ; dark red
         ,[0,100,0] $   ; dark Green
         ,[0,0,100] $   ; dark Blue
         ]

switch strupcase(uname) of    
    'VANDATAMANAGER':
    'BKGDDATAMANAGER':
    'SAMPDATAMANAGER': begin
        ; Do nothing if new item is the same as old item
;        if (strcmp((*sPtr).itemID, event.identifier,/fold_case)) then break
        
        ;; stop observing old data item
        ;oUI->RemoveOnNotifyObserver, (*sPtr).idSelf,(*sPtr).itemID
        ;; save ID of new item
        if (~strcmp(event.identifier,'')) then (*sPtr).itemID = event.identifier
        isSamp = strcmp(uname,'SAMPDATAMANAGER')
        isBkgd = strcmp(uname,'BKGDDATAMANAGER')
        isFast = strcmp(uname,'VANDATAMANAGER')
        if (~isSamp && ~isBkgd && ~isFast) then return
        if (isSamp) then begin
          oTool->GetProperty, sampContRef=oCont, nSamp=nData;, sumSampFlag=sumFlag
          color = [0,0,255]
          symbol = 4
          plotname='SAMPPLOT'
        endif
        oItem =  oTool->GetByIdentifier(event.identifier) ; else use the selected dataset
        if (~obj_valid(oItem)) then break
        
        ; oItem must be an IDLitParameterSet
        if (~obj_isa(oItem,'IDLitParameterSet')) then break

        if (isSamp) then oTool->SetProperty, sampSelected=oItem;->GetFullIdentifier()
        
        ;; the event.identifier field is inserted in
        ;; cw_daveDataManager whenever an item is selected.

        
        oTool->Plot, oItem, sPtr, event.overplot, event.lastplot, position=event.position
         
        break
    end

    else:
endswitch

;oTool->RefreshPropertysheet
help,/heap,/brief
end





;-------------------------------------------------------------------------
; wd_HFBSFWSTool
;+
; Purpose:
;    This is the main entry point for the iTools common IDL Widget
;    user interface. This is passed a tool and the routine will then
;    build a UI that contains the contents of the tool object.
;
; Parameters:
;   oTool    - The tool object to use.
;
; Keywords:
;    TITLE          - The title for the tool. If not provided, IDL
;                     iTool is used.
;
;    LOCATION       - Where to place the new widget. X,Y
;
;    DIMENSIONS     - The size of the drawable.
;
;    VIRTUAL_DIMENSIONS - The virtual size of the drawing area
;
;    USER_INTERFACE - If set to an IDL variable, will return the user
;                     interface object built during this UI
;                     construction.
;-
pro wd_HFBSFWSTool, oTool, TITLE=titleIn, $
                 LOCATION=location, $
                 DIMENSIONS=dimensionsIn, $
                 VIRTUAL_DIMENSIONS=virtualDimensions, $
                 XSIZE=swallow1, $  ; should use DIMENSIONS
                 YSIZE=swallow2, $  ; should use DIMENSIONS
                 USER_INTERFACE=oUI, $  ; output keyword
                 _REF_EXTRA=_extra

    compile_opt idl2

@idlit_on_error2

if (~OBJ_VALID(oTool)) then $
    MESSAGE, 'Tool is not a valid object.'

title = (N_ELEMENTS(titleIn) gt 0) ? titleIn[0] : 'HFBS Fixed Window Scan Analysis'

WIDGET_CONTROL, /HOURGLASS

 ;*** Base to hold everything
 ;
wBase = WIDGET_BASE(/COLUMN, MBAR=wMenubar, $
                    TITLE=title, $
                    /TLB_KILL_REQUEST_EVENTS, $
                    /TLB_SIZE_EVENTS, $
                    /KBRD_FOCUS_EVENTS, $
                    _EXTRA=_extra)

 ;*** Create a new UI tool object, using our iTool.
 ;
 oUI = OBJ_NEW('IDLitUI', oTool, GROUP_LEADER=wBase)


 ;***  Menubars
 ;
 hspace = 'Operations/       '
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/File')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/Input')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/Output')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 ;wInsert     = CW_ITMENU(wMenubar, oUI, 'Operations/Edit')
 wInsert     = CW_ITMENU(wMenubar, oUI, 'Operations/Insert')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 ;wOperations = CW_ITMENU(wMenubar, oUI, 'Operations/Operations')
 ;wWindow     = CW_ITMENU(wMenubar, oUI, 'Operations/Window')
 wHelp       = CW_ITMENU(wMenubar, oUI, 'Operations/Help')

 ; Assume we were already at 90% at this point.
 ;void = IDLitwdSplash(PERCENT=92)

 ;***  Drawing area.
 ;
 screen = GET_SCREEN_SIZE(RESOLUTION=cm_per_pixel)
 hasDimensions = (N_ELEMENTS(dimensionsIn) eq 2)
if (~hasDimensions)then begin
    ;; Some multi monitor deployments will report a screen size
    ;; that includes all monitors, which can lead to some
    ;; interesting screen geometry. To take this into account, the
    ;; following logic is used to determine the default tool size.
    ;;  - Find the minimum dimension
    ;;  - minDim = dim * 0.4
    ;;  - The other maxDim = minDim * 1.4
    ;;
    ;; The only possible issue with this logic would be if the
    ;; logical screen is so large that this calculation will
    ;; result in a window exceeding a screen.
    dimensions = 0.4*screen
    if(screen[0] gt screen[1])then $ ;; x larger than Y
      dimensions[0] = dimensions[1] * 1.4 $
    else $
      dimensions[1] = dimensions[0] * 1.4
endif else $
    dimensions = dimensionsIn

; Make sure our dimensions are larger than the menubar.
; Otherwise strange things happen on Windows.
; We do this regardless of whether the user provided
; their own dimensions.
geom = WIDGET_INFO(wBase, /GEOMETRY)
minsize = [geom.scr_xsize + 10, 100]

dimensions >= minsize


;***  Toolbars
;
wToolbar = WIDGET_BASE(wBase, /ROW, XPAD=0, YPAD=0, SPACE=7)
wTool1 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/File')
wTool2 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/Edit')
wTool3 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators', /EXCLUSIVE)
wTool4 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/Dataspace')
wTool5 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators/View', /EXCLUSIVE)
wTool6 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/View') 
wTool7 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators/Annotation', /EXCLUSIVE)


; It may happen that our toolbar is just slightly larger than
; the specified dimensions. In this case, adjust the dimensions
; to fit the toolbar (unless the user has provided their
; own dimensions).
if (~hasDimensions) then begin
    geom = WIDGET_INFO(wToolbar, /GEOMETRY)
    if ((geom.xsize gt dimensions[0]) && $
        (geom.xsize lt (dimensions[0] + 40))) then $
    dimensions[0] = geom.xsize
endif

; Always adjust the toolbar to fit the base width.
if (dimensions[0] lt geom.xsize) then $
    WIDGET_CONTROL, wToolbar, SCR_XSIZE=dimensions[0]

;***  Panel and Drawing area
wRow = widget_base(wBase, /ROW, $
                    xpad=0, ypad=0, space=0)

wBasePanel  = widget_base(wRow, xpad=0, ypad=0, space=0,/col)
wBaseDraw = widget_base(wRow, xpad=0, ypad=0, space=0)

 ;; The Panel
xsDM = 300 
ysDM = 250
unDM = 'HFBSFWSDATAMANAGER'
wDM = cw_HFBSFWSDataManager(wBasePanel,oUI,uname=unDM,xsize=xsDM,ysize=ysDM,_extra=_extra)

wPSBase = widget_base(wBasePanel, xpad=0, ypad=0, space=0, /frame)
xsPS = xsDM
ysPS = ysDM*0.75
wPS = 0L
wPS = widget_propertysheet(wBasePanel $
                     ,value=oTool $
                     ,uname='HFBSFWSPROPSHEET' $
                     ,scr_xsize=xsPS $
                     ;,scr_ysize=ysPS $
                     ,event_func='wd_HFBSFWSTool_PropSheet_event' $
                    )

;oTool->GetProperty, sampType=sampType  ; sampType: 0=Powder, 1=Single Crystal
;wPowderBase = widget_base(wBasePanel, /row, xpad=0, ypad=0, space=0,map=~sampType)
;wSxtalBase = widget_base(wBasePanel, /col, xpad=0, ypad=0, space=0,map=sampType)

;tt = 'Click to export selected data to Mslice module (Single Crystal Only)'
;wMslice = widget_button(wSxtalBase,value='Export selected data to Mslice (Single Crystal samples)',uname='ExportToMslice',tooltip=tt $
;                        ,sensitive=0, scr_xsize=xsDM, yoffset=5)

;tt = 'Click to export selected data to DAVE Data Manager (Powder Only)'
;wDAVEDM = widget_button(wSxtalBase,value='Export to Data Manager',uname='ExportToDAVEDM',tooltip=tt $
;                        ,sensitive=0, scr_xsize=0.5*xsDM, yoffset=5)
;tt = 'Click to export selected data to PAN for fitting (Powder Only)'
;wPAN = widget_button(wSxtalBase,value='Export to Data Manager',uname='ExportToPAN',tooltip=tt $
;                        ,sensitive=0, scr_xsize=0.5*xsDM, yoffset=5)


;
;tt = 'Click to modify detectors to be masked'
;wMaskDet = widget_button(wButtonBase,value='Mask Detectors',uname='Mask',tooltip=tt $
;                        ,sensitive=1, xsize=xsDM, yoffset=5)

;tt = 'dummy tooltip'
;wDummy = widget_button(wButtonBase,value='Dummy',uname='Dummy',tooltip=tt $
;                        ,sensitive=0, xsize=xsDM, yoffset=5)

; If the user did not explicitly provide dimensions, and
; the panel height is greater than the default (half-screen)
; height, but less than half again the default height, then
; adjust our default height to match the panel height (so
; scrollbars will not be necessary).
if (~hasDimensions) then begin
    geom = WIDGET_INFO(wBasePanel, /GEOMETRY)
    if ((geom.scr_ysize gt dimensions[1]) && $
        (geom.scr_ysize lt (dimensions[1]*1.5))) then begin
        dimensions[1] = geom.scr_ysize
    endif
    minsize[1] = dimensions[1]
endif

;void = IDLitwdSplash(PERCENT=95)

;; Make our graphics window
if (hasDimensions && N_ELEMENTS(virtualDimensions) eq 0) then $
    virtualDimensions = dimensions

baseWin = dimensions
wDraw = CW_ITWINDOW(wBaseDraw, oUI, $
                     DIMENSIONS=dimensions, $
                     VIRTUAL_DIMENSIONS=virtualDimensions)

;*** Status bar.
;
dimensions[0] = dimensions[0] + geom.scr_xsize  ;  add the panel width
wStatus = CW_itStatusBar(wBase, oUI, XSIZE=dimensions[0])

; Cache some information.
;
wChild = WIDGET_INFO(wBase, /CHILD)

if(n_elements(location) eq 0)then begin
    location = [(screen[0] - dimensions[0])/2 - 10, $
                ((screen[1] - dimensions[1])/2 - 100) > 10]
endif
WIDGET_CONTROL, wBase, MAP=0, $
    TLB_SET_XOFFSET=location[0], TLB_SET_YOFFSET=location[1]

;; State structure for the widget
geomBase = widget_info(wBase,/geometry)
geomDM = widget_info(wDM,/geometry)
geomPS = widget_info(wPS,/geometry)
geomDraw = widget_info(wDraw,/geometry)
geomSB = widget_info(wStatus,/geometry)
State = { oTool     : oTool    $
          ,oUI       : oUI      $

          ,wBase      : wBase    $
          ,title     : title    $
          ,minsize   : [geomBase.xsize,geomBase.ysize]  $
          ,basesize  : [0L, 0L] $
          ,baseDM    : [xsDM,ysDM] $;[geomDM.scr_xsize, geomDM.scr_ysize] $  
          ,basePS    : [xsPS,ysPS] $;[geomPS.scr_xsize, geomPS.scr_ysize] $
          ,baseWin   : [geomDraw.draw_xsize, geomDraw.draw_ysize] $
          ,baseSB    : [geomSB.scr_xsize, geomSB.scr_ysize] $ $
          ,wToolbar  : wToolbar $
          ,wDraw     : wDraw    $
;          ,wMslice   : wMslice $
;          ,wDaveDM   : wDaveDM $
;          ,wPAN      : wPAN $
;          ,wPowderBase:wPowderBase $
;          ,wSxtalBase:wSxtalBase $
;          ,oMslice   : obj_new() $
          ,wPanel    : wBasePanel   $
          ,wDM       : wDM      $
          ,wPS       : wPS      $ 
          ,wStatus   : wStatus   $
          ,yDataID   : ''       $ ; identifier of the cuurent y-axis data
          ,xDataID   : ''       $
          ,eDataID   : ''       $
          ,xBkgdID   : ''       $ 
          ,itemID    : ''       $ ; id of item currently selected item in DM   
          ,manipID   : ''       $ ; id of active manipulator
          ,xrange    : [0.,0.]  $
          ,yrange    : [0.,0.]  $
          }
pState = PTR_NEW(state, /NO_COPY)
WIDGET_CONTROL, wChild, SET_UVALUE=pState

; Force an initial resize.
;wd_HFBSFWSTool_resize, pState, 0, 0

;void = IDLitwdSplash(PERCENT=100)

WIDGET_CONTROL, wBase, /REALIZE

; Retrieve the starting dimensions and store them.
; Used for window resizing in event processing.
WIDGET_CONTROL, wBase, TLB_GET_SIZE=basesize
(*pState).basesize = basesize

;; Register ourself as a widget with the UI object. 
;; Returns a string containing our identifier (a user interface adaptor is created  during reg)
myID = oUI->RegisterWidget(wBase, 'ToolBase', 'wd_HFBSFWSTool_callback')

;; Register for messages from HFBS tool (oTool).
oUI->AddOnNotifyObserver, myID, oTool->GetFullIdentifier()

; Also observe the system object for messages
oSys = oTool->_GetSystem()
oUI->AddOnNotifyObserver, myID, oSys->GetFullIdentifier()

WIDGET_CONTROL, wChild, KILL_NOTIFY="wd_HFBSFWSTool_cleanup"
WIDGET_CONTROL, wBase, /MAP ;show the user what we have made.

; Start event processing for the tool.
XMANAGER, 'wd_HFBSFWSTool', wBase, /NO_BLOCK

end
;------------------------------------------------------------------------------
