; $Id$
;###############################################################################
;+
; NAME:
;   cw_hfbsfwsDataManager
;
; PURPOSE:
;   Construct the widgets representing the data manager browser section of the
;   main DAVE user interface.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2004
;-
;###############################################################################


;===============================================================================
; cw_hfbsfwsDataManager_MergeData
;
; Purpose:
;   Merge the selected datasets into a single dataset, replacing the individual 
;   datasets with the newly created one
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_MergeData, sPtr, wTLB
compile_opt idl2

; get the selected dataset and pass them to the tool for merging
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count lt 1)then return
oTool = (*sPtr).oUI->Gettool()
oItem = []
for i=0,count-1 do begin
  obj = oTool->GetByIdentifier(id[i])
  if (obj_valid(obj) && obj_isa(obj,'IDLitParameterSet')) then oItem = [oItem,obj]
endfor

if (n_elements(oItem) gt 1) then begin
  oTool->DisableUpdates
  status = oTool->MergeDatasets(oItem, merged_dataset=oMerged)
  if (status && obj_isa(oMerged,'IDLitParameterSet')) then begin
    itemID = oMerged->GetFullIdentifier()
    (*sPtr).identifier = itemID
    cw_hfbsfwsDataManager_setselect, (*sPtr).wDM, itemID, /clear ; select the merged dataset
    wd_HFBSFWStool_sendPlotEvent, (*sPtr).oUI, oMerged   ; and plot it
  endif
  oTool->EnableUpdates
endif

end


;===============================================================================
; cw_hfbsfwsDataManager_ViewDetail 
;
; Purpose:
;   View the contents of a data element in the DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_ViewDetail, sPtr, wTLB
compile_opt idl2

; get the selected datasets, loop through and display their content
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count lt 1)then return
oTool = (*sPtr).oUI->Gettool()
for i=0,count-1 do begin
  oItem = oTool->GetByIdentifier(id[i])
  if(obj_valid(oItem))then begin
    ;idParent = cw_itTreeView_getParent((*sPtr).wDM, id[0])
    oItem->GetProperty, name=title, description=filename
    status = file_test(filename,/read)
    doneTxt = "Close "+title
    if (status) then begin
      font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
        '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
      font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
        '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
      font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
        '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'

      xdisplayfile, filename, done_button=doneTxt, /grow_to_screen, group=wTLB, font=font2
    endif else begin
      void = dialog_message("Could not retrieve contents",/error, $
        dialog_parent=wTLB, $
        title="No Data")
    endelse
    ;wd_viewDetail, oItem, group_leader=wTLB
  endif else $
    void = dialog_message("Could not retrieve contents",/error, $
    dialog_parent=wTLB, $
    title="No Data")
endfor  

end


;===============================================================================
; cw_hfbsfwsDataManager_ViewDetail 
;
; Purpose:
;   View the contents of a data element in the DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_ViewHist, sPtr, wTLB
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()
if(obj_valid(oTool))then begin
    oTool->GetProperty, history=history
    if (n_elements(history) gt 0) then begin
       font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
              '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
       font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
              '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
       font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
              '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'

        xdisplayfile, text=history, done_button=doneTxt, /grow_to_screen, group=wTLB, font=font2
    endif else begin
       void = dialog_message("Could not retrieve history",/error, $
                           dialog_parent=wTLB, $
                           title="No data")
    endelse
    ;wd_viewDetail, oItem, group_leader=wTLB
endif 

end


;===============================================================================
; cw_hfbsfwsDataManager_DeleteData 
;
; Purpose:
;   Encapsulates the functionality used to delete data. Based on
;   _cw_itDataManager_DeleteData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_DeleteData, sPtr, wTLB, deltree=deltree, delall=delall, noPrompt=noPrompt
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()

if (~keyword_set(delall)) then begin
   ;; Which data tree are we deleting from?
   ;; use the most recent selected item to find out
   if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then begin
      dataType='samp'
      wDM = (*sPtr).wDM
      oTool->GetProperty, nSamp=nData, sampContRef=oCont;, sumSampFlag=sumFlag
      ;sumLabel = 'SUMSAMPFLAG'
   endif
;   if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then begin
;      dataType='bkgd'
;      wDM = (*sPtr).wDM2
;      oTool->GetProperty, nBkgd=nData, bkgdContRef=oCont, sumBkgdFlag=sumFlag
;      sumLabel = 'SUMBKGDFLAG'
;   endif
;   if (stregex((*sPtr).identifier,'VAN',/fold_case) ge 0) then begin
;      dataType='van'
;      wDM = (*sPtr).wDM3
;      oTool->GetProperty, nVan=nData, vanContRef=oCont, sumVanFlag=sumFlag
;      sumLabel = 'SUMVANFLAG'
;   endif
   if (n_elements(dataType) eq 0) then return
endif

if (keyword_set(deltree)) then begin
   ;; Delete all data in samp or bkgd or vanadium branches
   msg = ["Deleting all "+dataType+" dataset(s):",'']
   oArr = oCont->Get(/all, count=count)
   if(count lt 1)then return
   for i=0,count-1 do begin
      id = oArr[i]->GetFullIdentifier()
      oArr[i]->GetProperty, name=name
      iArr = (n_elements(iArr) eq 0)? id : [iArr,id]
      msg = [msg,name]
   endfor
endif else if (keyword_set(delall)) then begin
   ;; Deleting all data in samp, bkgd and vanadium branches
   msg = ["Deleting all dataset(s):",'']
   oTool->GetProperty, sampContRef=oSampCont
;   oTool->GetProperty, bkgdContRef=oBkgdCont
;   oTool->GetProperty, vanContRef=oVanCont
   oArr1 = oSampCont->Get(/all, count=scount)
;   oArr2 = oBkgdCont->Get(/all, count=bcount)
;   oArr3 = oVanCont->Get(/all, count=fcount)
   if (scount gt 0) then $
      oArr = (n_elements(oArr) gt 0)? [oArr,oArr1] : oArr1
;   if (bcount gt 0) then $
;      oArr = (n_elements(oArr) gt 0)? [oArr,oArr2] : oArr2
;   if (fcount gt 0) then $
;      oArr = (n_elements(oArr) gt 0)? [oArr,oArr3] : oArr3
      count = n_elements(oArr)
   if(count lt 1)then return
   for i=0,count-1 do begin
      oArr[i]->GetProperty, name=name
      msg = [msg,name]
   endfor
endif else begin
   ;; Deleting only selected datasets

   ; Retrieve the detasets to be deleted
   msg = ["Deleting selected dataset(s):",'']
   if (n_elements(wDM) eq 0) then return
   id = cw_itTreeView_getSelect(wDM, count=count)
   if(count lt 1)then return
   for i=0,count-1 do begin
      oItem = oTool->GetByIdentifier(id[i])
      oItem->GetProperty, name=name
      oArr = (n_elements(oArr) eq 0)? oItem : [oArr,oItem]
      iArr = (n_elements(iArr) eq 0)? id[i] : [iArr,id[i]]
      msg = [msg,name]
   endfor
endelse


if (~keyword_set(noPrompt)) then begin
   title = "Delete Data"
   msg = [msg,' ','Are you sure?']
   status = dialog_message(title=title, msg, $
                           /question, /default_no, dialog_parent=wTLB)
   if (status eq 'No') then return
endif

oDM = oTool->GetService("DATA_MANAGER")

if (keyword_set(delTree)) then begin
   ;; Delete all datasets from specified tree
   for i=0,count-1 do begin
      ;void = oArr[i]->GetMetaData('DATAHASH',dataHash) ; the real data is in this dataPtr
      ;if (obj_valid(dataHash)) then obj_destroy, dataHash   ; so must free it!
      oCont->Remove, oArr[i]   ; remove obj ref from container
      ; sum data don't have a full identifier so don't decrement nData and can't use DM service to delete
      if (~strcmp(oArr[i]->GetFullIdentifier(),'')) then begin
         nData--
         void = oDM->DeleteData(iArr[i])   ; finally delete obj ref from data manager
      endif else $
         if (obj_valid(oArr[i])) then obj_destroy, oArr[i]   ; just a double check
   endfor

   case dataType of
      'samp': begin 
         oTool->SetProperty, nSamp=nData
       end 
;      'bkgd': begin
;         oTool->SetProperty, nBkgd=nData
;         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=0
;       end
;      'van': begin
;         oTool->SetProperty, nVan=nData
;         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMVANFLAG', sensitive=0
;       end
      else: return
   endcase

   oTool->refreshPropertySheet   ; force prop sheet to update

   ;; update context menu
   widget_control, (*sPtr).wView, sensitive=0
   widget_control, (*sPtr).wDel, sensitive=0
   widget_control, (*sPtr).wDelSamp, sensitive=0
   widget_control, (*sPtr).wMerge, sensitive=0
   widget_control, (*sPtr).wHist, sensitive=0
   widget_control, (*sPtr).wReload, sensitive=0
   widget_control, (*sPtr).wDAVE, sensitive=0
   widget_control, (*sPtr).wASCII, sensitive=0
   widget_control, (*sPtr).wDAVEDM, sensitive=0   
   widget_control, (*sPtr).wDAVEMSD, sensitive=0
   Widget_control, (*sPtr).wDAVEDMMSD, sensitive=0
   widget_control, (*sPtr).wASCIIMSD, sensitive=0
   widget_control, (*sPtr).wPAN, sensitive=0
   widget_control, (*sPtr).wFVan, sensitive=0

   oTool->SetProperty, modifiedStatus = 1
   return
endif

if (keyword_set(delAll)) then begin
   ;; Delete all datasets from all three trees
   if (scount gt 0) then begin
      oCont = oSampCont
      for i=0,scount-1 do begin
         ;void = oArr1[i]->GetMetaData('DATAHASH',dataHash) ; the real data is in this hash
         ;if (obj_valid(dataHash)) then obj_destroy, dataHash   ; so must free it!
         oCont->Remove, oArr1[i]   ; remove obj ref from container
         void = oDM->DeleteData(oArr1[i]->GetfullIdentifier())   ; finally delete obj ref from data manager
         if (obj_valid(oArr1[i])) then obj_destroy, oArr1[i]
      endfor
      oTool->SetProperty, nSamp=0
      ;oTool->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=0
   endif

   oTool->refreshPropertySheet   ; force prop sheet to update

   ;; update context menu
   widget_control, (*sPtr).wView, sensitive=0
   widget_control, (*sPtr).wDel, sensitive=0
   widget_control, (*sPtr).wDelSamp, sensitive=0
   widget_control, (*sPtr).wMerge, sensitive=0
   widget_control, (*sPtr).wHist, sensitive=0
   widget_control, (*sPtr).wReload, sensitive=0
   widget_control, (*sPtr).wDAVE, sensitive=0
   widget_control, (*sPtr).wASCII, sensitive=0
   widget_control, (*sPtr).wDAVEDM, sensitive=0
   widget_control, (*sPtr).wDAVEMSD, sensitive=0
   Widget_control, (*sPtr).wDAVEDMMSD, sensitive=0
   widget_control, (*sPtr).wASCIIMSD, sensitive=0
   widget_control, (*sPtr).wPAN, sensitive=0

   oTool->SetProperty, modifiedStatus = 1
   return
endif



; Delete selected datasets

;; proceed with the deleting
for i=0,count-1 do begin
   ;void = oArr1[i]->GetMetaData('DATAHASH',dataHash) ; the real data is in this hash
   ;if (obj_valid(dataHash)) then obj_destroy, dataHash   ; so must free it!
   oCont->Remove, oArr[i]   ; remove obj ref from container
   nData--
   case dataType of
      'samp': begin 
         oTool->SetProperty, nSamp=nData
       end 
      else: return
   endcase
   
   void = oDM->DeleteData(iArr[i])   ; finally delete obj ref from data manager
   if (obj_valid(oArr[i])) then obj_destroy, oArr[i]   ; just a double check
endfor

oTool->refreshPropertySheet   ; force prop sheet to update

if (nData gt 0) then begin
   ;; Update plot
   oItem = oCont->Get(position = 0)   ; Get the first dataset
   itemID = oItem->GetFullIdentifier()
   (*sPtr).identifier = itemID
   cw_hfbsfwsDataManager_setselect, wDM, itemID, /clear ; select it

   wd_HFBSFWStool_sendPlotEvent, (*sPtr).oUI, oItem   ; plot it
endif

oTool->SetProperty, modifiedStatus = 1

end


;===============================================================================
; cw_hfbsfwsDataManager_exportToDAVEDM
;
; Purpose:
;   Save selected data to file in DAVE format
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_exportToDAVEDM, sPtr, wTLB, quantity=quantity
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM
; Retrieve the detasets to be deleted

;
oTool = (*sPtr).oUI->Gettool()
oTool->GetProperty, sampContRef=oCont, nSamp=nData;, sumSampFlag=sumFlag

;if (sumFlag  && nData gt 1) then begin                        ;Is summed flag switched on?
;   if (oCont->Count() eq (nData+1)) then begin  ; ==> extra dataset is the summed data
;      oItem = oCont->Get(position=nData)        ; so use it instead
;      if (obj_valid(oItem)) then status = oTool->ExportToDaveDM(oItem)
;   endif else return
;endif else begin
   if (n_elements(wDM) eq 0) then return
   id = cw_itTreeView_getSelect(wDM, count=count) ; else use the selected datasets
   if(count lt 1)then return
   
   for i=0,count-1 do begin
      oItem = oTool->GetByIdentifier(id[i])
      if (~obj_valid(oItem)) then continue
      case strlowcase(quantity) of
        'intensities': status = oTool->ExportToDaveDM(oItem)
        'msd': status = oTool->ExportToDaveDMMSD(oItem)
        else:
      endcase
      
   endfor
;endelse


end


;===============================================================================
; cw_hfbsfwsDataManager_exportToPAN
;
; Purpose:
;   Save selected data to file in DAVE format
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_exportToPAN, sPtr, wTLB 
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM

;
oTool = (*sPtr).oUI->Gettool()
oTool->GetProperty, sampContRef=oCont, nSamp=nData;, sumSampFlag=sumFlag

;if (sumFlag  && nData gt 1) then begin                        ;Is summed flag switched on?
;   if (oCont->Count() eq (nData+1)) then begin  ; ==> extra dataset is the summed data
;      oItem = oCont->Get(position=nData)        ; so use it instead
;      if (obj_valid(oItem)) then status = oTool->ExportToPAN(oItem)
;   endif else return
;endif else begin
   if (n_elements(wDM) eq 0) then return
   id = cw_itTreeView_getSelect(wDM, count=count) ; else use the selected datasets
   if(count lt 1)then return
   
   for i=0,count-1 do begin
      oItem = oTool->GetByIdentifier(id[i])
      if (~obj_valid(oItem)) then continue
      status = oTool->ExportToPAN(oItem)
   endfor
;endelse


end


;===============================================================================
; cw_hfbsfwsDataManager_saveToDisk
;
; Purpose:
;   Save selected data to file in DAVE format
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_saveToDisk, sPtr, wTLB, format=format 
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM

; Retrieve the detasets to be deleted
oTool = (*sPtr).oUI->Gettool()
oTool->GetProperty, sampContRef=oCont, nSamp=nData;, sumSampFlag=sumFlag

;if (sumFlag  && nData gt 1) then begin                        ;Is summed flag switched on?
;   if (oCont->Count() eq (nData+1)) then begin  ; ==> extra dataset is the summed data
;      oItem = oCont->Get(position=nData)        ; so use it instead
;   endif else return
;endif else begin
   if (n_elements(wDM) eq 0) then return
   id = cw_itTreeView_getSelect(wDM, count=count) ; else use the selected datasets
   if(count lt 1)then return
   oItem = oTool->GetByIdentifier(id[0])   ; handle only first dataset
;endelse

if (~obj_valid(oItem)) then return

case strlowcase(format) of
   'dave': status = oTool->SaveAsDAVE(oItem)
   'ascii': status = oTool->SaveAsASCII(oItem)
   'msddave': status = oTool->SaveAsDAVEMSD(oItem)
   'msdascii': status = oTool->SaveAsASCIIMSD(oItem)
   else:
endcase

end


;===============================================================================
; cw_hfbsfwsDataManager_ReloadData 
;
; Purpose:
;   Encapsulates the functionality used to delete data. Based on
;   _cw_itDataManager_DeleteData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_ReloadData, sPtr, wTLB
compile_opt idl2

;; Which data tree are we deleting from?
;; use the most recent selected item to find out
if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then begin
   dataType='samp'
   wDM = (*sPtr).wDM
endif
;if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then begin
;   dataType='bkgd'
;   wDM = (*sPtr).wDM2
;endif
;if (stregex((*sPtr).identifier,'VAN',/fold_case) ge 0) then begin
;   dataType='van'
;   wDM = (*sPtr).wDM3
;endif

; Retrieve the detasets to be deleted
if (n_elements(wDM) eq 0) then return
id = cw_itTreeView_getSelect(wDM, count=count)
if(count lt 1)then return

oTool = (*sPtr).oUI->Gettool()

for i=0,count-1 do begin
   oItem = oTool->GetByIdentifier(id[i])
   case dataType of
      'samp': status = oTool->LoadSamp(dataObject=oItem, /reload)
;      'bkgd': status = oTool->LoadBkgd(dataObject=oItem, /reload)
;      'van': status = oTool->LoadVan(dataObject=oItem, /reload)
      else: return
   endcase
endfor

if (status) then begin
   ;; Update plot
   itemID = oItem->GetFullIdentifier()
   (*sPtr).identifier = itemID
   cw_hfbsfwsDataManager_setselect, wDM, itemID, /clear ; select it

   wd_HFBSFWStool_sendPlotEvent, (*sPtr).oUI, oItem   ; plot it
endif

end



;===============================================================================
; cw_hfbsfwsDataManager_RenameData 
;
; Purpose:
;   Encapsulates the functionality used to rename data. Based on
;   _cw_itDataManager_RenameData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_hfbsfwsDataManager_RenameData, sPtr, wTLB
compile_opt idl2

;; Which data tree are we renaming from?
;; use the most recent selected item to find out
if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then wDM = (*sPtr).wDM
;if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then wDM = (*sPtr).wDM2
;if (stregex((*sPtr).identifier,'VAN',/fold_case) ge 0) then wDM = (*sPtr).wDM3


;; Get the item
id = cw_itTreeView_getSelect(wDM, count=count)
if(count gt 0)then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id[0])              
    if(obj_valid(oItem))then begin
        oItem->GetProperty, name=name
        status = idlitwdprompttext(wTLB, $
                                   "Enter New Data Name:", $
                                   newName, $
                                   value=name, $
                                   title="Rename Data")
        if(status eq 1 and keyword_set(newName))then begin
            oItem->SetProperty, name=newName
            ;; Tell the tree to rebild
            cw_itTreeView_Callback, (*sPtr).wDM, id[0], "UPDATEITEM",id[0]

        endif
    endif else $
      void = dialog_message("Invalid Data Item.",/warning, $
                            dialog_parent=wTLB, $
                            title="Invalid Data")
endif else $
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")

end


;===============================================================================
; cw_hfbsfwsDataManager_Cleanup
; 
; PURPOSE:
;   Widget cleanup procedure.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be destroyed
;
; KEYWORDS:
;
pro cw_hfbsfwsDataManager_Cleanup, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr
ptr_free,(*sPtr).pDataIDs,sPtr

end


;===============================================================================
; cw_hfbsfwsDataManager_Realize
; 
; PURPOSE:
;   Called when the widget is realized - checks widget dimensions.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be realized
;
; KEYWORDS:
;
pro cw_hfbsfwsDataManager_Realize, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr

;; Make sure the geometry is all right.
sGeom = widget_info(widID, /geometry)
;cw_hfbsfwsDataManager_Resize, widID, sGeom.xsize, (*sPtr).ySize0

end


;===============================================================================
; cw_hfbsfwsDataManager_GetWidgetID
; 
; PURPOSE:
;   Called to retrieve the widget ID of the specified item identifier
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
;   idItem - The item whose widget ID is required
;
; Keywords:
;
function cw_hfbsfwsDataManager_GetWidgetID, widID, idItem
compile_opt idl2

return, widget_info(widID, find_by_uname=strupcase(idItem))

end


;===============================================================================
; cw_hfbsfwsDataManager_SetSelect
; 
; PURPOSE:
;   Called to set the selection state of the tree based on the
;   identifiers of the items passed in.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
;   idItem - The item to select. Optional if CLEAR is set.
;
; Keywords:
;   CLEAR: If set, clear all selected items first. If this keyword
;       is specified then idItem need not be supplied.
;
;   UNSELECT: Will unselect, not select;
;
pro cw_hfbsfwsDataManager_Setselect, wDM, idItem, UNSELECT=unselect, CLEAR=clear $
                                , dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
;      'bkgd': wDMC = (*sPtr).wDM2
;      'van': wDMC = (*sPtr).wDM3
      else: return
   endcase
endif

cw_itTreeView_SetSelect,  wDMC, idItem, UNSELECT=unselect, CLEAR=clear

end


;===============================================================================
; cw_hfbsfwsDataManager_Setvalue
; 
; PURPOSE:
;   Called by widget_control when the value of a widget is to be set.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
; KEYWORDS:
;
pro cw_hfbsfwsDataManager_Setvalue, wDM, value , dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
;      'bkgd': wDMC = (*sPtr).wDM2
;      'van': wDMC = (*sPtr).wDM3
      else: return
   endcase
endif

cw_itTreeView_SetSelect,  wDMC, value

end


;===============================================================================
; cw_hfbsfwsDataManager_Getvalue
; 
; PURPOSE:
;   Called by widget_control when the value of a widget is required.
;
; PARAMETERS
;   widID (in)  - The id of the widget whose value is required
;
; KEYWORDS:
;
; RETURN VALUE:
;   The required value of the widget
;
function cw_hfbsfwsDataManager_Getvalue, wDM, dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
;      'bkgd': wDMC = (*sPtr).wDM2
;      'van': wDMC = (*sPtr).wDM3
      else: return, 0L
   endcase
endif

return, cw_itTreeView_GetSelect(wDMC)

end


;===============================================================================
; cw_hfbsfwsDataManager_Event
; 
; PURPOSE:
;   Main event handler for widget created by cw_hfbsfwsDataManager.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
; RETURN VALUE:
;   0, if the event was completely handled here
;   Event Structure, if further processing is required.
;
function cw_hfbsfwsDataManager_Event, event
compile_opt idl2


; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'cw_hfbsfwsDataManager: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;; Get our state
widget_control, event.handler, get_uvalue=sPtr

sname = tag_names(event,/structure_name)
uname = widget_info(event.id, /uname)
;help, event, /struct

oUI = (*sPtr).oUI
oTool = (*sPtr).oUI->Gettool()

if (strcmp(sname,'WIDGET_TAB',/fold_case)) then begin
   ;; A new tab was selected so do the following
   ;; - check for any selected datasets
   ;; - if there are selected datasets, pick the first one and update the display with it
   case event.tab of
      0: wDM = (*sPtr).wDM   ; sample tab
;      1: wDM = (*sPtr).wDM2   ; bkgd tab
;      2: wDM = (*sPtr).wDM3   ; van bkgd tab
      else:
   endcase
   if (n_elements(wDM) eq 0) then return, 0
   id = cw_itTreeView_getSelect(wDM, count=count)
   if (count lt 1) then return, 0

   oItem = oTool->GetByIdentifier(id[0])
   isData = obj_isa(oItem, "IDLitData") && ~obj_isa(oItem,'IDL_CONTAINER')
   if (isData) then begin
      (*sPtr).identifier = id[0]   ; register data as currenty selected data
      wd_HFBSFWStool_sendPlotEvent, oUI, oItem   ; plot it
   endif

   ;; Update the context menu depending on 'selected' dataset 
   widget_control, (*sPtr).wView, sensitive=isData
   widget_control, (*sPtr).wDel, sensitive=isData
   widget_control, (*sPtr).wDelSamp, sensitive=isData
   widget_control, (*sPtr).wMerge, sensitive=isData
;   widget_control, (*sPtr).wHist, sensitive=isData
   widget_control, (*sPtr).wReload, sensitive=isData
   dataType = ''
   if (isData) then oItem->GetProperty, type=dataType
   isSamp = strcmp(dataType,'samp',/fold_case)
   widget_control, (*sPtr).wDAVE, sensitive=(isData && isSamp)
   widget_control, (*sPtr).wASCII, sensitive=(isData && isSamp)
   widget_control, (*sPtr).wDAVEDM, sensitive=(isData && isSamp)
   widget_control, (*sPtr).wDAVEMSD, sensitive=(isData && isSamp)
   Widget_control, (*sPtr).wDAVEDMMSD, sensitive=(isData && isSamp)
   widget_control, (*sPtr).wASCIIMSD, sensitive=(isData && isSamp)
   widget_control, (*sPtr).wPAN, sensitive=(isData && isSamp)
;   isVan = strcmp(dataType,'van',/fold_case)
;   isBkgd = strcmp(dataType,'bkgd',/fold_case)
   if (isSamp) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Data'
;   if (isBkgd) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Background Data'
;   if (isVan) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Vanadium Bkgd Data'

   return, 0
endif



switch strupcase(uname) of
    "VARIABLE": begin
        ;; Lets import a variable or two
        iStatus = IDLitwdCommandLineImport(oUI,group_leader=event.top)
        break
    end
    
    "FILE": begin
        ;; Lets import a file...
        iStatus = IDLitwdFileImport(oUI,group_leader=event.top)
        break
    end
    
    "BKGDDATAMANAGER":
    "VANDATAMANAGER":
    "SAMPDATAMANAGER": begin
        ;; Context Menu?
        if(tag_names(/structure, event) eq 'WIDGET_CONTEXT')then begin
            WIDGET_DISPLAYCONTEXTMENU, event.id, $
              event.x, event.y, (*sPtr).wContext 
        endif else begin
            id= (*event.value)[0]
            oItem = oTool->GetByIdentifier(id)
            isData = obj_isa(oItem, "IDLITPARAMETERSET") ;&& ~obj_isa(oItem,'IDL_CONTAINER')
            widget_control, (*sPtr).wView, sensitive=isData
            widget_control, (*sPtr).wDel, sensitive=isData
            widget_control, (*sPtr).wDelSamp, sensitive=isData
            widget_control, (*sPtr).wMerge, sensitive=isData
;            widget_control, (*sPtr).wHist, sensitive=isData
            widget_control, (*sPtr).wReload, sensitive=isData
            dataType = ''
            if (isData) then oItem->GetProperty, type=dataType
            isSamp = strcmp(dataType,'samp',/fold_case)
            widget_control, (*sPtr).wDAVE, sensitive=(isData && isSamp)
            widget_control, (*sPtr).wASCII, sensitive=(isData && isSamp)
            widget_control, (*sPtr).wDAVEDM, sensitive=(isData && isSamp)
            widget_control, (*sPtr).wDAVEMSD, sensitive=(isData && isSamp)
            Widget_control, (*sPtr).wDAVEDMMSD, sensitive=(isData && isSamp)
            widget_control, (*sPtr).wASCIIMSD, sensitive=(isData && isSamp)
            widget_control, (*sPtr).wPAN, sensitive=(isData && isSamp)
;            isVan = strcmp(dataType,'van',/fold_case)
;            isBkgd = strcmp(dataType,'bkgd',/fold_case)
            if (isSamp) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Data';, set_uvalue='samp'
;            if (isBkgd) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Background Data';, set_uvalue='bkgd'
;            if (isVan) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Vanadium Data';, set_uvalue='van'

            (*sPtr).identifier = id
            (*(*sPtr).pDataIDs) = (*event.value)
            heap_free, event


            oTool->UpdatePropertysheet ; some propertysheet elements depend on the number of selected datasets

            ;;
            ; (*(*sPtr).pDataIDs) contains a list of the selected ids in the data manager.
            ; However, this includes ids for child nodes not just the parents. Parent nodes
            ; represent datasets and we need to identify these and then send for plotting.
            ; Parent nodes are ParameterSet objects and have 'samp' datatype.
            oTool->DisableUpdates

            nPts = []
            iArr = (*(*sPtr).pDataIDs)
            count = n_elements(iArr)
            overplot = 0  ; first plot should not be an overplot
            (*sPtr).oUI->GetProperty, group_leader=wHandler ; the handler widget id for main widget
            for i=0,count-1 do begin
              oItem = oTool->GetByIdentifier(iArr[i])
              if (~obj_isa(oItem, "IDLitParameterSet")) then continue
              oItem->GetProperty, type = dataType
              isSamp = strcmp(dataType,'samp',/fold_case)
              if (isSamp) then begin
                plotEvent = {cw_hfbsfwsDataManager, $
                  ID:event.id, $
                  TOP:event.top, HANDLER:wHandler, $
                  clicks:event.clicks, $
                  overplot:overplot, $
                  lastplot:(i eq count-1), $
                  identifier:iArr[i], $
                  position:i}
                wd_HFBSFWSTool_event, plotEvent
                overplot = 1  ; subsequent plots should be overplotted!
                
                ; retrieve and store the nos of data points in dataset
                void = oItem->GetMetaData('DATAHASH',pDataHash)
                dataHash = *pDataHash
                nPts = [nPts,dataHash['nPts']]

              endif
            endfor
            
            ; update msdIndex so that it is consistent with the nos of datapoints for the selected datasets
            ; if more than one dataset is selected use the smallest nPts present
            if (n_elements(nPts) gt 0) then oTool->SetPropertyAttribute, 'msdIndex', valid_range=[1,min(nPts),1]

            oTool->EnableUpdates

        endelse
        break
    end

    'DELETE': begin
      cw_hfbsfwsDataManager_DeleteData, sPtr, event.top
      break
    end

    'DELETESAMP': begin
      cw_hfbsfwsDataManager_DeleteData, sPtr, event.top, /deltree
      break
    end

    'DELETEALL': begin
      cw_hfbsfwsDataManager_DeleteData, sPtr, event.top, /delall
      break
    end

    'MERGESELECTION': begin
      cw_hfbsfwsDataManager_MergeData, sPtr, event.top
      break
    end

    'RELOAD': begin
      cw_hfbsfwsDataManager_ReloadData, sPtr, event.handler
      break
    end

    'TODAVE': begin
      cw_hfbsfwsDataManager_saveToDisk, sPtr, event.handler, format='dave'
      break
    end

    'TOASCII': begin
      cw_hfbsfwsDataManager_saveToDisk, sPtr, event.handler, format='ascii'
      break
    end

    'TODAVEDM': begin
      cw_hfbsfwsDataManager_ExportToDAVEDM, sPtr, event.handler, quantity='intensities'
      break
    end

    'MSDTODAVEDM': begin
      cw_hfbsfwsDataManager_ExportToDAVEDM, sPtr, event.handler, quantity='msd'
      break
    end

    'MSDTOASCII': begin
      cw_hfbsfwsDataManager_saveToDisk, sPtr, event.handler, format='msdascii'
      break
    end

    'MSDTODAVE': begin
      cw_hfbsfwsDataManager_saveToDisk, sPtr, event.handler, format='msddave'
      break
    end

    'TOPAN': begin
      cw_hfbsfwsDataManager_ExportToPAN, sPtr, event.handler
      break
    end

;    'FITFAST': begin
;      if (~oTool->EditUserDefProperty(oTool, 'FASTPARAMS')) then break
;      
;      oData = oTool->GetByIdentifier((*sPtr).identifier)  ; get the last selected dataset
;      if (~obj_valid(oData)) then break
;      oData->GetProperty, type=dataType
;      if (strcmp(dataType,'van',/fold_case)) then $
;        wd_HFBSFWStool_sendPlotEvent, oUI, oData
;      break
;    end

    'VIEWHIST': begin
      cw_hfbsfwsDataManager_viewHist, sPtr, event.handler
      break
    end

    'VIEWDETAILS': begin
      cw_hfbsfwsDataManager_viewDetail, sPtr, event.handler
      break
    end

    else:
endswitch

return, 0

end


;===============================================================================
; cw_hfbsfwsDataManager_Resize
; 
; PURPOSE:
;   Resize the specified widget to the specified dimension.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be resized
;
;   newX (in)   - The new x size
;
;   newY (in)   - the new y size
;
; KEYWORDS:
;
pro cw_hfbsfwsDataManager_Resize, widID, newX, newY

compile_opt idl2

WIDGET_CONTROL, widID, get_uvalue=sPtr


;geomCW = Widget_Info(widID, /geometry)

;; Change height of dm
;widget_control, (*sPtr).wDM, SCR_XSIZE=newX-12, $ ;; why -12? It just works
;                SCR_YSIZE= newY - geomButs.scr_ysize-16
;widget_control, (*sPtr).wDM, scr_xsize=newX - 12 $ ;; why -12? It just works
;                ,scr_ysize= newY - 12

; there are three DM tabs so deal with each separately
wDM = (*sPtr).wDM
oldGeom = widget_info((*sPtr).wDM,/geometry)

;widget_control, (*sPtr).wDM, scr_xsize = oldGeom.scr_xsize + newX $
;                           , scr_ysize = oldGeom.scr_ysize + newY
;
;widget_control, (*sPtr).wDM2, scr_xsize = oldGeom.scr_xsize + newX $
;                           , scr_ysize = oldGeom.scr_ysize + newY
;
;widget_control, (*sPtr).wDM3, scr_xsize = oldGeom.scr_xsize + newX $
;                           , scr_ysize = oldGeom.scr_ysize + newY
widget_control, (*sPtr).wDM, scr_xsize = newX, scr_ysize = newY

;widget_control, (*sPtr).wDM2, scr_xsize = newX, scr_ysize = newY
;
;widget_control, (*sPtr).wDM3, scr_xsize = newX, scr_ysize = newY

end


;===============================================================================
; cw_hfbsfwsDataManager
; 
; PURPOSE:
;   Construct a compound widget representing the data manager browser section of the
;   main DAVE user interface.
;
; PARAMETERS
;   wParent (in) - id of the parent widget for the widget to be
;                  built. The parent widget embeds the new one.
;
;   oUI (in)     - object reference of the tool UI
;
; KEYWORDS:
;   uName (in)   - the uname to be assigned to the new widget
;
;   xSize (in)   - the width of the new widget
;
;   ySize (in)   - the height of the new widget
;
; RETURN VALUE:
;   the id of the base widget of created widget hierarchy.
;
function cw_hfbsfwsDataManager, wParent,oUI,uname=uName,xsize=xSize,ysize=ySize,_extra=_extra

compile_opt idl2

;; Check arguments.
if (N_PARAMS() lt 2) then $
  MESSAGE, 'Incorrect number of arguments.'

if (not WIDGET_INFO(wParent, /VALID)) then $
  MESSAGE, 'Invalid widget identifier.'

if (not keyword_set(ySize)) then ySize=300
if (not keyword_set(xSize)) then xSize=300

myname = 'cw_hfbsfwsDataManager'
wTabBase = widget_tab(wParent, tab_mode=0 $   ; mode eq 0 for Xplatform compt. since it is always 0 for motif
                       ,EVENT_FUNC= myname+'_event' $
                       ,PRO_SET_VALUE= myname+'_setvalue' $
                       ,FUNC_GET_VALUE= myname+'_getvalue' $
;                       ,uname=uName $
                       ,NOTIFY_REALIZE=myname+"_realize" $
                       ,KILL_NOTIFY=myname+"_cleanup" $
                      )
;w = widget_base(wParent, EVENT_FUNC= myname+'_event' $
;                       ,PRO_SET_VALUE= myname+'_setvalue' $
;                       ,FUNC_GET_VALUE= myname+'_getvalue' $
;                       ,uname=uName $
;                       ,NOTIFY_REALIZE=myname+"_realize" $
;                       ,KILL_NOTIFY=myname+"_cleanup" $
;                       ,/base_align_center, /row)

wSampTab = widget_base(wTabBase,/row,title='Data',uname='SAMPTAB',tab_mode=2)
;wBkgdTab = widget_base(wTabBase,/row,title='Background',uname='BKGDTAB',tab_mode=2)
;wVanTab = widget_base(wTabBase,/row,title='Vanadium',uname='VANTAB',tab_mode=2)

;; Now create the tree widgets
wDM1 = cw_ittreeview(wSampTab, oUI $
                    ,IDENTIFIER="Sample Data Manager" $  ; this is the identifier for an IDLitDataManagerFolder obj created in the Tool's Init()
                    ,uname="SAMPDATAMANAGER" $
                    ,xsize = xsize, ysize=ysize, /multiple);,/multiple)
;wDM2 = cw_ittreeview(wBkgdTab, oUI $
;                    ,IDENTIFIER="Bkgd Data Manager" $
;                    ,uname="BKGDDATAMANAGER" $
;                    ,xsize = xsize, ysize=ysize, /multiple);,/multiple)
;
;wDM3 = cw_ittreeview(wVanTab, oUI $
;                    ,IDENTIFIER="Van Data Manager" $
;                    ,uname="VANDATAMANAGER" $
;                    ,xsize = xsize, ysize=ysize, /multiple);,/multiple)

;; Build context menu to be applied to the tree widget
widget_control, wDM1, /context_events
;widget_control, wDM2, /context_events
;widget_control, wDM3, /context_events

;wContext = WIDGET_BASE(wTabBase,/CONTEXT_MENU,event_func=myname+'_event')  
wContext = WIDGET_BASE(wDM1,/CONTEXT_MENU,event_func=myname+'_event')  

tt1 = "Delete dataset"
tt2 = "Reload this dataset from disk"
tt3 = "View raw contents of selected item"
tt4 = "Merge Selected Datasets"
tt5 = "Save Detector intensities in DAVE format file"
tt6 = "Save Detector intensities in ASCII format file"
tt7 = "Export Detector intensities to DAVE Data Manager"
tt8 = "Fit mean square displacement data in PAN"
tt9 = "Save mean square displacements (MSD) in DAVE format file"
tt10 = "Save mean square displacements (MSD) in ASCII format file"
tt11 = "Export mean square displacements (MSD to DAVE Data Manager"
wDel = widget_button(wContext, value="Delete", uname="DELETE",tooltip=tt1,sensitive=0)
wReload = widget_button(wContext, value="Reload", uname="RELOAD",tooltip=tt2,sensitive=0)
wDelSamp = widget_button(wContext, value="Delete All Datasets", uname="DELETESAMP",tooltip=tt1,sensitive=0)
wMerge = widget_button(wContext, value="Merge Selected Datasets", uname="MERGESELECTION",tooltip=tt4,sensitive=0)
wView = widget_button(wContext, value="View Raw Contents", uname="VIEWDETAILS",tooltip=tt3,/separator,sensitive=0)
;wHist = widget_button(wContext, value="View Reduction History", uname="VIEWHIST",tooltip=tt4,sensitive=0)
wDAVE = widget_button(wContext, value="Save Intensities as DAVE file", uname="TODAVE",tooltip=tt5,/separator,sensitive=0)
wASCII = widget_button(wContext, value="Save Intensities as ASCII file", uname="TOASCII",tooltip=tt6,sensitive=0)
wDAVEDM = widget_button(wContext, value="Export Intensities to DAVE Data Manager", uname="TODAVEDM",tooltip=tt7,sensitive=0)
wASCIIMSD = widget_button(wContext, value="Save MSD as ASCII file", uname="MSDTOASCII",tooltip=tt10,sensitive=0,/separator)
wDAVEMSD = widget_button(wContext, value="Save MSD as DAVE file", uname="MSDTODAVE",tooltip=tt9,sensitive=0)
wDAVEDMMSD = widget_button(wContext, value="Export MSD to DAVE Data Manager", uname="MSDTODAVEDM",tooltip=tt11,sensitive=0)
wPAN = widget_button(wContext, value="Fit -3*ln(I/I_min) in PAN", uname="TOPAN",tooltip=tt8,sensitive=0)

state = { oUI:oUI $
          ,wDM:wDM1 $
;          ,wDM2:wDM2 $
;          ,wDM3:wDM3 $
          ,ysize0:ysize/3 $       ;initial Y size used in realize notify
          ,wContext: wContext $
          ,wDel:wDel $
          ,wDelSamp:wDelSamp $
          ,wMerge:wMerge $
          ,wReload:wReload $
          ,wDAVE:wDAVE $
          ,wASCII:wASCII $
          ,wDAVEDM:wDAVEDM $
          ,wPAN:wPAN $
          ,wDAVEMSD:wDAVEMSD $
          ,wASCIIMSD:wASCIIMSD $
          ,wDAVEDMMSD:wDAVEDMMSD $
          ,wView:wView $
;          ,wHist:wHist $
          ,identifier:'' $      ; used to store identifier of selected item
          ,pDataIDs:ptr_new('') $ ; store identifiers of multiple seleted items
        }

;; Normally, this would be the first child on the widget, but the
;; tree view widget is just a tree, and not wrapped and changes
;; could cause issues.
sPtr = ptr_new(State,/no_copy)
widget_control, wTabBase, set_uvalue=sPtr

;; Due to how the context menu is created, we place the state in the
;; context menu also
widget_control, wContext, set_uvalue=sPtr

return, wTabBase

end
