; $Id$
;
; Copyright (c) 2002-2004, Research Systems, Inc.  All rights reserved.
;       Unauthorized reproduction prohibited.
;+
; NAME:
;   wd_FANStool
;
; PURPOSE:
;   Create the IDL UI (widget) interface for an associated tool object.
;
; CALLING SEQUENCE:
;   wd_FANStool, Tool
;
; INPUTS:
;   Tool - Object reference to the tool object.
;
;-


;-------------------------------------------------------------------------
; wd_FANStool_callback
;+
; Purpose:
;   Callback routine for the tool interface widget, allowing it to
;   receive update messages from the system.
;
; Parameters:
;   wBase     - Base id of this widget
;
;   strID     - ID of the message.
;
;   MessageIn - What is the message
;
;   userdata  - Data associated with the message
;-
pro wd_FANStool_callback, wBase, strID, messageIn, userdata
compile_opt idl2

; print,'In wd_FANStool_CALLBACK____________________________________________'
; print,'strID = ',strID
; print,'messageIn = ',messageIn
; print,'userdata = ',userdata
; print,'_________________________________________________________________________'


if (~WIDGET_INFO(wBase, /VALID)) then $
  return

;; Grab the state of the widget
WIDGET_CONTROL, WIDGET_INFO(wBase, /CHILD), GET_UVALUE=pState

case STRUPCASE(messageIn) of
    
    ;;
    'SETPROPERTY': begin
        if ((strcmp(userdata,'NAME',/fold_case))[0]) then begin
            (*pState).oTool->GetProperty, name=name,tool_filename=filename
            (*pState).title = name
            widget_control, wbase, tlb_set_title=name + ' [' + filename + ']'
        endif
    end
    
    ;; Check the file name changes to display
    'FILENAME': begin
        ;; Use the new filename to construct the title.
        ;; Remove the path.
        filename = STRSPLIT(userdata, '/\', /EXTRACT)
        filename = filename[N_ELEMENTS(filename)-1]
                                ; Append the filename onto the base title.
        newTitle =  filename + ' - ' + (*pState).title
        WIDGET_CONTROL, wBase, TLB_SET_TITLE=newTitle

        ;; if the tool has been modified (indicated by *) then the
        ;; DAVE Tool has to be notified.
        if (strpos(filename,'*',/reverse_search) ne -1) then begin
            widget_control, wBase, get_uvalue=wNode
            if ((n_elements(wNode) le 0) || ~widget_info(wNode,/valid)) then return
            widget_control, wNode, get_uvalue=sPtr
            wTLB_FANSTool = (*sPtr).wTLB_FANSTool
            widget_control, widget_info(wTLB_FANSTool,/child), get_uvalue=sPtr_FANSTool
            ((*sPtr_FANSTool).oTool)->_SetDirty, 1

            (*pState).oTool->_SetDirty, 0
        endif
    end
    
    ;; A panel was added to the tool. See if our size changed.
    'ADDUIPANELS': begin
        wd_FANStool_resize, pState, 0, 0
    end
        
    ;; The show/hide was changed. See if our size changed.
    'SHOWUIPANELS': begin
        wd_FANStool_resize, pState, 0, 0
    end
    
    ;; Virtual dims changed
    'VIRTUAL_DIMENSIONS': begin
        ;; Retrieve the original geometry (prior to the resize).
        WIDGET_CONTROL, wBase, TLB_GET_SIZE=basesize
        geom = WIDGET_INFO((*pState).wDraw, /GEOMETRY)
        
        ;; See if the window shrank.
        dx = (userdata[0] - geom.xsize) < 0
        dy = (userdata[1] - geom.ysize) < 0
        
        ;; No shrinkage.
        if ((dx eq 0) && (dy eq 0)) then $
          break
        
        IDLitwdTool_resize, pState, dx, dy
    end
    
    ;; The sensitivity is to be changed
    'SENSITIVE': begin
        WIDGET_CONTROL, wBase, SENSITIVE=userdata
    end
    
    else:                       ; do nothing
    
 endcase

; Fudge!
; To resolve issue when undo/redo buttons are desensitized when
; selections are made on visualizations. Setting the viz tool
; as the current tool appears to fix it.
oSystem = _IDLitSys_GetSystem()
oCurTool = oSystem->_GetCurrentTool()
if (obj_valid(oCurTool) && $
    (oCurTool eq (*pState).oTool)) then begin
   oCurrent = (*pState).oTool->GetService("SET_AS_CURRENT_TOOL")
   void = (*pState).oTool->DoAction(oCurrent->GetFullIdentifier())
endif

; Force viz tool state to appear unmodified
(*pState).oTool->_SetDirty, 0

end


;;-------------------------------------------------------------------------
;; wd_FANStool_resize
;;
;; Purpose:
;;    Called when the user has resize the TLB of this tool interface.
;;    Will recalculate the size of the major elements in the
;;    interface.
;;
;; Parameters:
;;   sPtr   - pointer to the state struct for this widget.
;;
;;   deltaW   - The change in the width of the interface.
;;
;;   deltaH   - The change in the height of the interface.
;
pro wd_FANStool_resize, sPtr, deltaW, deltaH
compile_opt idl2

; Retrieve the original geometry (prior to the resize)
; of the draw widget.
drawgeom = WIDGET_INFO((*sPtr).wDraw, /GEOMETRY)

; Compute the updated dimensions of the visible portion
; of the draw widget.
;newVisW = (drawgeom.draw_xsize + deltaW) > (*sPtr).minsize[0]
;newVisH = (drawgeom.draw_ysize + deltaH) > (*sPtr).minsize[1]
newVisW = (*sPtr).baseWin[0] + deltaW   ;drawgeom.draw_xsize + deltaW
newVisH = (*sPtr).baseWin[1] + deltaH   ;drawgeom.draw_ysize + deltaH

;different = newVisW ne drawgeom.draw_xsize || newVisH ne drawgeom.draw_ysize
;if (~different) then return

isUpdate = WIDGET_INFO((*sPtr).wBase, /UPDATE)

; If update turned off on unix, draw window won't resize properly.
; So just turn off update on Windows.
if (!version.os_family eq 'Windows') then begin
    if (isUpdate) then $
        widget_control, (*sPtr).wBase, UPDATE=0
endif else begin
   ; On Unix make sure update is on.
   if (~isUpdate) then $
       widget_control, (*sPtr).wBase, /UPDATE
endelse

; Update the draw widget dimensions and scrollbars.
CW_ITWINDOW_resize, (*sPtr).wDraw, newVisW, newVisH

; Update the toolbar row to be the same width as the status width.
;WIDGET_CONTROL, (*sPtr).wToolbar, SCR_XSIZE=deltaW + statusGeom.scr_xsize
WIDGET_CONTROL, (*sPtr).wToolbar, SCR_XSIZE=(*sPtr).baseSB[0] + deltaW

; Update the left panel: the data manager and the property sheet
cw_fansDataManager_Resize, (*sPtr).wDM, (*sPtr).baseDM[0], deltaH/2+(*sPtr).baseDM[1]
;psGeom = widget_info((*sPtr).wPS, /geometry)
widget_control, (*sPtr).wPS, scr_ysize = deltaH/2 + (*sPtr).basePS[1]

; Update the statusbar to be the same width as the draw + panel.
;statusGeom = widget_info((*sPtr).wStatus, /geometry)
;cw_itStatusBar_Resize, (*sPtr).wStatus, deltaW + statusGeom.scr_xsize
;panelGeom = widget_info((*sPtr).wPanel,/geometry)
;cw_itStatusBar_Resize, (*sPtr).wStatus, (*sPtr).baseSB[0] + deltaW
cw_itStatusBar_Resize, (*sPtr).wStatus, newVisW + (*sPtr).baseDM[0]

if (isUpdate && ~WIDGET_INFO((*sPtr).wBase, /UPDATE)) then $
    widget_control, (*sPtr).wBase, /UPDATE


; Retrieve and store the new top-level base size.
if (WIDGET_INFO((*sPtr).wBase, /REALIZED)) then begin
    WIDGET_CONTROL, (*sPtr).wBase, TLB_GET_SIZE=basesize
    (*sPtr).basesize = basesize
endif
end


;;-------------------------------------------------------------------------
;; wd_FANStool__cleanup
;;
;; Purpose:
;;   Called when the widget is dying, allowing the state ptr to be
;;   released.
;;
;; Parameters:
;;    wChild   - The id of the widget that contains this widgets
;;               state.
;;
pro wd_FANStool_cleanup, wChild

compile_opt idl2

    if (~WIDGET_INFO(wChild, /VALID)) then $
        return

    WIDGET_CONTROL, wChild, GET_UVALUE=pState
    if (PTR_VALID(pState)) then $
        PTR_FREE, pState

end


;===============================================================================
; wd_FANSPref_PropSheet_Event
; 
; PURPOSE:
;   Event handler for property sheet component of widget created by
;   wd_FANSTool. This sheet displays detailed info about items
;   currently selected in the Data/Visualization tree widgets.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
function wd_FANSTool_PropSheet_Event, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_FANStool_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;;print,'In wd_FANSTool_PropSheet_Event____________________________________________'
;;help,event,/struct
;;print,'_________________________________________________________________________'
;
;


;; Get our state var
wChild = widget_info(event.top, /child)
widget_control, wChild, get_uvalue=sPtr

oUI = (*sPtr).oUI
oTool = oUI->GetTool()
oSystem = oTool->GetSystem()

;; Call the proper event handler for the PS widget.
;help, event, /struct
;ret_event = cw_itPropertySheet_Event(event)


;;******************************
;-- Record change to prop sheet
;-- Act on change to prop sheet
case tag_names(event, /structure_name) of

    'WIDGET_PROPSHEET_CHANGE': begin    
       ; Get the value of the changed property
       if (event.proptype eq 0) then begin
          ;; ==> a userdef property type so handle it appropriately
          status = oTool->EditUserDefProperty(oTool, event.identifier)
       endif else begin
          value = widget_info(event.id, component=event.component,property_value=event.identifier)
          ; Set the component's property value. 
          event.component->SetPropertyByIdentifier, event.identifier, value 
       endelse


       id = event.identifier
       switch (id) of
          'ENERGYUNIT': begin   ; x-axis energy unit has changed
             oTool->GetProperty, energyUnitFlag=eFlag, mev2wnos=mev2wnos
             oXData = oSystem->GetByIdentifier((*sPtr).xDataID)
             if (~obj_valid(oXData)) then break

             oTool->DisableUpdates

             status = oXData->GetData(data)
             data = (eFlag eq 0)? data/mev2wnos : data * mev2wnos
             status = oXData->SetData(data,/no_copy,/no_notify)
             units = (eFlag eq 0)? 'meV' : 'cm-1'
             oXData->AddMetaData,'UNITS', units
             oXData->NotifyDataChange
             oXData->NotifyDataComplete
             oXBkgd = oSystem->GetByIdentifier((*sPtr).xBkgdID)
             if (obj_valid(oXBkgd)) then begin
                status = oXBkgd->GetData(data)
                data = (eFlag eq 0)? data/mev2wnos : data * mev2wnos
                status = oXBkgd->SetData(data,/no_copy,/no_notify)             
                oXBkgd->AddMetaData,'UNITS', units
                oXBkgd->NotifyDataChange
                oXBkgd->NotifyDataComplete
             endif

             oTool->EnableUpdates
             ;oTool->RefreshCurrentWindow   ;oWin->Draw   ; force window to refresh
             
             oTool->SetProperty, modifiedStatus = 1
             break
          end
          
          'MASKEDDETS': begin
             oTool->GetProperty, nFast=nFast
             oData = oTool->GetByIdentifier((*sPtr).itemID)  ; get the last selected dataset
             if (~obj_valid(oData)) then break
             oData->GetProperty, type=dataType
             if (nFast gt 0) then begin
               msg = 'Please remember to update your Fast Fit Parameters also'
               oTool->ErrorMessage, severity=0, msg, title='FANS Data Reduction'
               oTool->StatusMessage, msg
               ;; update display if 'fast' data was on
               if (strcmp(dataType,'fast',/fold_case)) then begin
                 wd_FANStool_sendPlotEvent, oUI, oData
               endif
             endif else begin
               ;; update display
               wd_FANStool_sendPlotEvent, oUI, oData
             endelse

             oTool->SetProperty, modifiedStatus = 1
             break
          end

          'MASKFLAG': begin
             oTool->GetProperty, maskflag=mFlag
             oTool->SetPropertyAttribute, 'MASKEDDETS', sensitive=mFlag
             widget_control, event.id, /refresh_property  ; force prop sheet to update
             ; !! Note: no break statement; continue to MONSCALEVALUE case to update plot
          end
          
          'FASTPARAMS': ; !! Note: no break statement; continue to MONSCALEVALUE case to update plot
          'OPLOTFASTFITFLAG': begin
            oData = oTool->GetByIdentifier((*sPtr).itemID)
            wd_FANStool_sendPlotEvent, oUI, oData
            
            oTool->SetProperty, modifiedStatus = 1
            break
          end

          'OPLOTBKGDFLAG': begin
            oData = oTool->GetByIdentifier((*sPtr).itemID)
            wd_FANStool_sendPlotEvent, oUI, oData
            
            oTool->SetProperty, modifiedStatus = 1
            break
          end

          'FASTFLAG': ; !! Note: no break statement; continue to MONSCALEVALUE case to update plot
          'BKGDFLAG': ; !! Note: no break statement; continue to MONSCALEVALUE case to update plot
          'MONSCALEVALUE': begin
             widget_control, event.id, /refresh_property
             ;; update display
             oData = oTool->GetByIdentifier((*sPtr).itemID)  ; get the last selected dataset
             if (~obj_valid(oData)) then break
             oData->GetProperty, type=dataType
             if (strcmp(dataType,'samp',/fold_case) || strcmp(dataType,'bkgd',/fold_case)) then begin
               wd_FANStool_sendPlotEvent, oUI, oData
             endif
            
             oTool->SetProperty, modifiedStatus = 1
             break
          end
          
          'MONSCALEFLAG': begin
             oTool->SetPropertyAttribute, 'MONSCALEVALUE', sensitive=value
             widget_control, event.id, /refresh_property
             ;; update display
             oData = oTool->GetByIdentifier((*sPtr).itemID)  ; get the last selected dataset
             if (~obj_valid(oData)) then break
             oData->GetProperty, type=dataType
             if (strcmp(dataType,'samp',/fold_case) || strcmp(dataType,'bkgd',/fold_case)) then begin
               wd_FANStool_sendPlotEvent, oUI, oData
             endif
            
             oTool->SetProperty, modifiedStatus = 1
             break
          end

;          'MONSCALEFLAG': begin
;             oTool->GetProperty, monscaleflag=msFlag
;             oTool->SetPropertyAttribute, 'MONSCALEVALUE', sensitive=msFlag
;             widget_control, event.id, /refresh_property  ; force prop sheet to update
;             break
;          end

          'SUMFASTFLAG':
          'SUMBKGDFLAG':
          'SUMSAMPFLAG': begin
             oItem = obj_new()
             if (strcmp(id,'SUMFASTFLAG')) then begin
               oTool->GetProperty, fastContRef=oCont, nFast=nData
               oTool->GetProperty, sumFastFlag=sumFlag
               dataType='fast'
             endif
             if (strcmp(id,'SUMBKGDFLAG')) then begin
               oTool->GetProperty, bkgdContRef=oCont, nBkgd=nData
               oTool->GetProperty, sumBkgdFlag=sumFlag
               dataType='bkgd'
             endif
             if (strcmp(id,'SUMSAMPFLAG')) then begin
               oTool->GetProperty, sampContRef=oCont, nSamp=nData
               oTool->GetProperty, sumSampFlag=sumFlag
               dataType='samp'
             endif
             if (sumFlag eq 0) then begin
               ;; then remove any currently merged data from the container
               nContCnt = oCont->Count()
               if (nContCnt eq (nData+1)) then begin
                  oData = oCont->Get(position=nData)
                  if (obj_valid(oData)) then begin
                     void = oData->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
                     if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!

                     oCont->Remove, oData
                     obj_destroy, oData   ; oData is not part of any hierarchy so no deed to use DM service to destroy it!
                     ;oDM = oTool->GetService('DATA_MANAGER')
                     ;void = oDM->DeleteData(oData->GetFullIdentifier())
                  endif

                  ; Get the currently selected item(s)
                  itemID = (cw_fansdatamanager_getvalue((*sPtr).wDM, dataType=dataType))[0] ;(*sPtr).itemID   ;
                  if (itemID ne '') then begin
                     oItem = oTool->GetByIdentifier(itemID)
                  endif else begin
                     ; nothing was selected so select the first data in the DM tree
                     oItem = oCont->Get(position = 0)
                     if (~obj_valid(oItem)) then break
                     itemID = oItem->GetFullIdentifier()
                     ; and select it
                     cw_fansdatamanager_setselect, (*sPtr).wDM, itemID, dataType=dataType, /clear
                  endelse
               endif
             endif else begin
               oTool->MergeData, dataType
               oItem = oCont->Get(position = oCont->Count() - 1)
               ;if (~obj_valid(oItem)) then break
               ;itemID = oItem->GetFullIdentifier()
             endelse 

             ;; Update display with current data (oItem)
             wd_FANStool_sendPlotEvent, oUI, oItem  ;, dataType
             
             oTool->SetProperty, modifiedStatus = 1
             break   
          end

         else: print, id
         
       endswitch
    end

    else:
endcase

return, 1

end
;-------------------------------------------------------------------------------



;;-------------------------------------------------------------------------
;; wd_FANStool_sendPlotEvent
;;
;; Purpose:
;;    Generate and send an event to update the plot window
;;
;; Parameters:
;;    event    - The widget event to process.
;;
pro wd_FANStool_sendPlotEvent, oUI, oItem   ;, dataType
compile_opt idl2

if (~obj_valid(oItem)) then return

oUI->GetProperty, group_leader=wTLB
wChild = WIDGET_INFO(wTLB, /CHILD)
WIDGET_CONTROL, wChild, GET_UVALUE=sPtr


;; retrieve the widgetID of the DM tree root node
oItem->GetProperty, type=dataType
case strlowcase(dataType) of
   'fast': rootID = 'FANSFASTDATAMANAGER'
   'bkgd': rootID = 'FANSBKGDDATAMANAGER'
   'samp': rootID = 'FANSSAMPDATAMANAGER'
   else: return
endcase

wRoot = cw_fansdatamanager_getWidgetID((*sPtr).wDM, rootID)

;; followed by a plot update
updatePlotEvent = {wd_FANSTool_event, $
                ID:wRoot, $
                TOP:(*sPtr).wBase, HANDLER:(*sPtr).wBase, $
                clicks:1, $
                identifier:oItem->GetfullIdentifier()}

wd_FANSTool_event, updatePlotEvent


end


;;-------------------------------------------------------------------------
;; wd_FANStool_Event
;;
;; Purpose:
;;    Event handler for the tool interface IDL widget.
;;
;; Parameters:
;;    event    - The widget event to process.
;;
pro wd_FANStool_event, event
compile_opt idl2


; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_FANStool_event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;help,'wd_FANStool_event called...'

;; Get our state
wChild = WIDGET_INFO(event.handler, /CHILD)
WIDGET_CONTROL, wChild, GET_UVALUE=sPtr

oTool = (*sPtr).oUI->GetTool()
oSystem = oTool->GetSystem()

;   help,event, /struct
;   help,''
;help,TAG_NAMES(event, /STRUCTURE_NAME)
case TAG_NAMES(event, /STRUCTURE_NAME) of

    ;; Kill the widget?
    'WIDGET_KILL_REQUEST': begin

       ; Check whether or not to save current session before closing it
       oTool->GetProperty, modifiedStatus=modifiedFlag, working_directory=workDir
       if (modifiedFlag) then begin
          ; If our current state is modified, prompt to save first.
          ;outfile = workDir+projName+'.fisv'
          status = oTool->PromptUserYesNo( $
              "Current session is modified! Save changes?", $
              answer, $
              DEFAULT_NO=0, $
              TITLE='Save', $
              /CANCEL)
      
          if (status && (answer eq 1)) then begin
             void = oTool->writeProject()
          endif
          oTool->SetProperty, prompt=0, modifiedStatus=0  ; don't prompt to exit next
       endif


        ;; Delegate the delete to the Exit operation that is defined for this tool
        exitID = 'Operations/ExitOp'
        exitOpDesc = oTool->GetByIdentifier(exitID)
        if (obj_valid(exitOpDesc)) then exitOp = exitOpDesc->GetObjectInstance()
        if (obj_valid(exitOp)) then void = exitOp->DoAction(oTool)
        return
    end

    ;; Focus change
    'WIDGET_KBRD_FOCUS': begin
        if (obj_valid((*sPtr).oUI) && event.enter) then begin
            oTool = (*sPtr).oUI->GetTool()
            oCurrent = oTool->GetService("SET_AS_CURRENT_TOOL")
            void = oTool->DoAction(oCurrent->GetFullIdentifier())
            ;; DONT DO THIS our tools could enter a focus loop in the
            ;; following situation:
            ;;    IDL> Iplot & iPlot
            ;; widget_control, (*sPtr).wDraw, /input_Focus
        endif
    end

    ;; The TLB was resized
    'WIDGET_BASE': begin
        ; Compute the size change of the base relative to
        ; its stored former size.
        WIDGET_CONTROL, event.top, TLB_GET_SIZE=newSize
;        newSize[0] = newSize[0] > (*sPtr).minSize[0]
;        newSize[1] = newSize[1] > (*sPtr).minSize[1]
;        deltaW = newSize[0] - (*sPtr).basesize[0]
;        deltaH = newSize[1] - (*sPtr).basesize[1]
        deltaW = (newSize[0] - (*sPtr).minSize[0]) > 0.0
        deltaH = (newSize[1] - (*sPtr).minSize[1]) > 0.0
        ; Bail if no change.
;        if (deltaW le 0 && deltaH le 0) then begin
;            widget_control, (*sPtr).wBase, xsize=(*sPtr).minSize[0], ysize=(*sPtr).minSize[1]
;            break
;        endif
        wd_FANStool_resize, sPtr, deltaW, deltaH
        end
        

    else: ; do nothing

endcase


; In addition, need to deal with selections from the data manager.
uname = widget_info(event.id, /uname)

colors = [[0,0,0] $     ; Black
         ,[255,0,0] $   ; Red
         ,[0,255,0] $   ; Green
         ,[0,0,255] $   ; Blue
         ,[100,0,0] $   ; dark red
         ,[0,100,0] $   ; dark Green
         ,[0,0,100] $   ; dark Blue
         ]

switch strupcase(uname) of
;    'FITFAST': begin
;      ;; Launch the fast bkgd fit utility
;      wd_FANSFitFast, (*sPtr).oUI
;      break
;    end
    
;    'MASK': begin
;      ;; Launch the detector masking utility
;      if (strcmp(strtrim((*sPtr).itemID),'')) then break
;      oItem =  oTool->GetByIdentifier((*sPtr).itemID)
;      void = oItem->GetMetaData('DATAPTRREF',dataPtr)
;      if (~ptr_valid(dataPtr)) then break
;      wd_FANSMaskDetectors, (*sPtr).oUI, dataPtr=dataPtr
;      oTool->refreshPropertySheet
;      break
;    end
    
    'FANSFASTDATAMANAGER':
    'FANSBKGDDATAMANAGER':
    'FANSSAMPDATAMANAGER': begin
        ; Do nothing if new item is the same as old item
;        if (strcmp((*sPtr).itemID, event.identifier,/fold_case)) then break
        
        ;; stop observing old data item
        ;oUI->RemoveOnNotifyObserver, (*sPtr).idSelf,(*sPtr).itemID
        ;; save ID of new item
        if (~strcmp(event.identifier,'')) then (*sPtr).itemID = event.identifier
        isSamp = strcmp(uname,'FANSSAMPDATAMANAGER')
        isBkgd = strcmp(uname,'FANSBKGDDATAMANAGER')
        isFast = strcmp(uname,'FANSFASTDATAMANAGER')
        if (~isSamp && ~isBkgd && ~isFast) then return
        if (isFast) then begin
          oTool->GetProperty, fastContRef=oCont, nFast=nData, sumFastFlag=sumFlag
          color = [0,128,0]
          symbol = 6
          plotname='FASTPLOT'
        endif
        if (isBkgd) then begin
          oTool->GetProperty, bkgdContRef=oCont, nBkgd=nData, sumBkgdFlag=sumFlag
          color = [255,0,0]
          symbol = 5
          plotname='BKGDPLOT'
        endif
        if (isSamp) then begin
          oTool->GetProperty, sampContRef=oCont, nSamp=nData, sumSampFlag=sumFlag
          color = [0,0,255]
          symbol = 4
          plotname='SAMPPLOT'
        endif
        if (sumFlag  && nData gt 1) then begin                        ;Is summed flag switched on?
          if (oCont->Count() eq (nData+1)) then begin  ; ==> extra dataset is the summed data
             oItem = oCont->Get(position=nData)        ; so use it instead
             if (~obj_valid(oItem)) then break
          endif else break
        endif else begin
          oItem =  oTool->GetByIdentifier(event.identifier) ; else use the selected dataset
          if (~obj_valid(oItem)) then break
        endelse
        
        ; oItem must be an IDLitData and not a container
        if (obj_isa(oItem,'IDL_CONTAINER')) then break

        if (isSamp) then oTool->SetProperty, sampSelected=oItem;->GetFullIdentifier()
        if (isBkgd) then oTool->SetProperty, bkgdSelected=oItem;->GetFullIdentifier()
        if (isFast) then oTool->SetProperty, fastSelected=oItem;->GetFullIdentifier()
        
        ;; the event.identifier field is inserted in
        ;; cw_daveDataManager whenever an item is selected.

        if (obj_isa(oItem, 'IDLitData') && (oItem->GetMetaDataCount() gt 0))then begin
            ;; Plot the data

            void = oItem->GetMetaData('DATAPTRREF',dataPtr)
            if (~ptr_valid(dataPtr)) then return

            status = oTool->CalculateXYE(oItem, xvals, yvals, evals, xunits)

            (*sPtr).xrange = [xvals[0],xvals[n_elements(xvals)-1]]
            (*sPtr).yrange = [yvals[0],yvals[n_elements(yvals)-1]]
            
            oXVal = obj_new('IDLitData',xvals,name='xvalue',type=partype(xvals))
            oXVal->SetProperty, /auto_delete   ; tag for auto removal when no longer used
            oXVal->AddMetaData,'Long_name', (*dataPtr).xtitle
            oXVal->AddMetaData,'Units', xunits
            oXVal->AddMetaData,'Distribution','POINTS'

            oYVal = obj_new('IDLitData',yvals,name='yvalue',type=partype(yvals))
            oYVal->SetProperty, /auto_delete
            oYVal->AddMetaData,'Long_name', (*dataPtr).ytitle
            oYVal->AddMetaData,'Signal', 1

            oEVal = obj_new('IDLitData',evals,name='evalue',type=partype(evals))
            oEVal->SetProperty, /auto_delete
            
            oSystem->AddByIdentifier, '/Data Manager', oXval
            oSystem->AddByIdentifier, '/Data Manager', oYval
            oSystem->AddByIdentifier, '/Data Manager', oEval

            strparms = ['Y','X','VERTICES','Y ERROR','X ERROR','PALETTE','VERTEX_COLORS']
            idparms = [oYval->GetFullIdentifier() $
                      ,oXval->GetFullIdentifier() $
                      ,'' $
                      ,oEval->GetFullIdentifier(),'','','']
            visID = 'PLOT'
            opID = 'Operations/Insert Visualization'
            oRequester = (oTool->GetByIdentifier(opID))->GetObjectInstance()
            oRequester->_setTool, oTool
            oRequester->GetProperty, show_execution_ui = orig_showUI
            oRequester->SetProperty, show_execution_ui = 0

            oRequester->SetProperty, parameter_names=strParms, data_ids=idParms, visualization_id=visID
            
            oTool->DisableUpdates
            oWin = oTool->GetCurrentWindow()
            oView = oWin->GetCurrentView()
            activateManip = 0
            rangeChange = 0
            if (oTool->HasVisualizations()) then begin
               print, 'Vis deleted...'
               ;; store the active manipulator
               (*sPtr).manipID = (oTool->GetCurrentManipulator())->GetFullIdentifier()
               activateManip = 1

               ;; ensure the existing visualization is selected
               oLayer = oView->GetCurrentLayer()
               oWorld = oLayer->GetWorld()
               oDataSpace = oWorld->GetDataSpaces()
               oVis = oDataSpace->GetVisualizations(count = nVis)
               
               ; retrieve current xrange and yrange to determine if user has zoomed plot
               oDataSpace->GetProperty, xrange=xr, yrange=yr
               xChange = ((*sPtr).xrange[0] ne xr[0]) || ((*sPtr).xrange[1] ne xr[1]) 
               yChange = ((*sPtr).yrange[0] ne yr[0]) || ((*sPtr).yrange[1] ne yr[1]) 
               rangeChange = xChange && yChange
               
               delID = 'Operations/Edit/Delete'
               oDelete = (oTool->GetByIdentifier(delID))->GetObjectInstance()
               
               ;; then delete it
               for j=nVis-1,0,-1 do begin
                  oVis[j]->Select
                  oTool->saveVisProps, oVis[j]  ; save vis properties before deleting it
                  oPSet = oVis[j]->GetParameterSet()
                  void = oDelete->DoAction(oTool)   ; delete selected item ie the vis
   
                  ;; and also explicitly destroy the previous vis data
                  oDM = oTool->GetService('DATA_MANAGER')
                  oOldData = oPSet->Get(/all,count=nOldData)
                  if (nOldData gt 0) then begin
                     for i=0,nOldData-1 do begin
                        if (obj_valid(oOldData[i])) then void = oDM->DeleteData(oOldData[i]->GetFullIdentifier())
                     endfor
                  endif
                  if (obj_valid(oPSet)) then void = oDM->DeleteData(oPSet->GetFullIdentifier())
               endfor
;               oDataSpace->Select
;               void = oDelete->DoAction(oTool)   ; delete DS
               obj_destroy, [oVis,oDataSpace]
            endif

            oCmd = oRequester->DoAction(oTool)
            
            ;; Get the newly created vis
            oLayer = oView->GetCurrentLayer()
            oWorld = oLayer->GetWorld()
            oDataSpace = oWorld->GetDataSpaces()
            
            ; Customize default color and symbol
            oVis = oDataSpace->GetVisualizations(count = nVis)
            oVis = oVis[nVis-1]
;            oVis->SetProperty, sym_index=symbol, color=color, errorbar_color=color, name=plotname
;            oVis->SetProperty, name=plotname, _EXTRA=(*(*sPtr).sgprops)
;            oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account

            ; Set font_size to 12 (default is 16 for IDL 8 and newer)
;            oAxes = oDataSpace->GetAxes(count=naxes)
;            for i=0,naxes-1 do oAxes[i]->SetProperty, _EXTRA=(*(*sPtr).sgprops);font_size=12
            oTool->CustomizeVis, oVis, plotname

            ;; If fast bkgd data then overplot fit if selected
            oTool->GetProperty, fastFlag=fastFlag,fastparams=paramString,fastFunc=funcIndex $
                              , oplotFastFitFlag=opFFFlag, energyUnitFlag=eUFlag, mev2wnos=mev2wnos
            params = float(strsplit(paramString,',',/extract,count=nParams))
            zeroParam = (nParams lt 2) || (params[0] eq 0.0 && params[1] eq 0.0)
            if (isFast && opFFFlag && ~zeroParam) then begin
               omega = (eUFlag)? xvals/mev2wnos : xvals
               ;params[5] = (eUFlag)? mev2wnos : 1.0
               case funcIndex of
                   0: yfit=(fans_Arcsin(omega,params))[*,0] ;; funct returns eval + partial derivatives wrt fpar
                   1: yfit=(fans_Linear(omega,params))[*,0] 
                   2: yfit=(fans_Quad(omega,params))[*,0] 
                   3: yfit=(fans_Cubic(omega,params))[*,0]
               endcase

               oYFit = obj_new('IDLitData',yfit,name='fastfit',type=partype(yfit))
               oYFit->SetProperty, /auto_delete
               oSystem->AddByIdentifier, '/Data Manager', oYFit

               idparms = [oYFit->GetFullIdentifier() $
                         ,oXval->GetFullIdentifier() $
                         ,'','','','','']
   
               oRequester->SetProperty,parameter_names=strParms,data_ids=idParms,visualization_id=visID
               oCmd = oRequester->DoAction(oTool)
            
               ; Customize default color and linestyle
               oVis = oDataSpace->GetVisualizations(count = nVis)
               oVis = oVis[nVis-1]
;               oVis->SetProperty, name='FASTFITPLOT',sym_index=0, linestyle=0, color=color, alpha_channel=0.5, thick=5
;               oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account
               oTool->CustomizeVis, oVis, 'FASTFITPLOT'
            endif

            ;; If samp plot then overplot bkgd if selected
            oTool->GetProperty, oplotBkgdFlag=opBFlag, nBkgd=nBkgd, bkgdSelected=oBkgd, bkgdContRef=oBCont
            if (isSamp && opBFlag && (nBkgd gt 0)) then begin
               if (~obj_valid(oBkgd)) then begin
                  ;; try using the last bkgd data
                  oBkgd = oBCont->Get(position=(oBCont->Count()-1))
               endif
               if (obj_valid(oBkgd)) then begin
                  status = oTool->CalculateXYE(oBkgd, xvals, yvals);, evals, xunits)

                  oYB = obj_new('IDLitData',yvals,name='oplotybkgd',type=partype(yvals))
                  oYB->SetProperty, /auto_delete
                  oSystem->AddByIdentifier, '/Data Manager', oYB

                  oXB = obj_new('IDLitData',xvals,name='oplotxbkgd',type=partype(xvals))
                  oXB->SetProperty, /auto_delete
                  oSystem->AddByIdentifier, '/Data Manager', oXB

                  idparms = [oYB->GetFullIdentifier() $
                            ,oXB->GetFullIdentifier() $
                            ,'','','','','']
      
                  oRequester->SetProperty,parameter_names=strParms,data_ids=idParms,visualization_id=visID
                  oCmd = oRequester->DoAction(oTool)
                  
                  ; Customize default color and linestyle
                  oVis = oDataSpace->GetVisualizations(count = nVis)
                  oVis = oVis[nVis-1]
;                  oVis->SetProperty, name='SAMPBKGDPLOT', sym_index=0, linestyle=0, color=[255,0,0], alpha_channel=0.6, thick=3
;                  oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account
                  oTool->CustomizeVis, oVis, 'SAMPBKGDPLOT'
                  
                  (*sPtr).xBkgdID = oXB->GetFullIdentifier()
               endif
            endif
            oRequester->SetProperty, show_execution_ui = orig_showUI

            if (float((strtok(!version.release,/extract))[0]) lt 8.0) then begin
               ;Modify dimensions slightly to better use up available window
               oDataSpace->Scale, 1.40,1.5,1
               oDataSpace->Translate, 0.22, 0.06, 0.0
            endif else begin
               oDataSpace->Scale, 1.05,1.1,1
               oDataSpace->Translate, 0.20, 0.0, 0.0            
            endelse
;            if (rangeChange) then begin
;               oDataSpace->SetProperty, x_auto_update=0,y_auto_update=0
;               oDS = oDataSpace->getdataspace(/unnormalized)
;               if (obj_valid(oDS)) then $
;                  oDS->setproperty,x_minimum=xr[0],x_maximum=xr[1],y_minimum=yr[0],y_maximum=yr[1]
;            endif else oDataSpace->SetProperty, x_auto_update=1,y_auto_update=1
            
            if (activateManip) then oTool->ActivateManipulator, (*sPtr).manipID
            oTool->EnableUpdates
            oTool->RefreshCurrentWindow   ;oWin->Draw   ; force window to refresh

            ; store ID's of data currently used in plot
            (*sPtr).yDataID = oYVal->GetFullIdentifier()
            (*sPtr).xDataID = oXVal->GetFullIdentifier()
            (*sPtr).eDataID = oEVal->GetFullIdentifier()
        endif
        
        break
    end

    else:
endswitch

help,/heap,/brief
end





;-------------------------------------------------------------------------
; wd_FANStool
;+
; Purpose:
;    This is the main entry point for the iTools common IDL Widget
;    user interface. This is passed a tool and the routine will then
;    build a UI that contains the contents of the tool object.
;
; Parameters:
;   oTool    - The tool object to use.
;
; Keywords:
;    TITLE          - The title for the tool. If not provided, IDL
;                     iTool is used.
;
;    LOCATION       - Where to place the new widget. X,Y
;
;    DIMENSIONS     - The size of the drawable.
;
;    VIRTUAL_DIMENSIONS - The virtual size of the drawing area
;
;    USER_INTERFACE - If set to an IDL variable, will return the user
;                     interface object built during this UI
;                     construction.
;-
pro wd_FANStool, oTool, TITLE=titleIn, $
                 LOCATION=location, $
                 DIMENSIONS=dimensionsIn, $
                 VIRTUAL_DIMENSIONS=virtualDimensions, $
                 XSIZE=swallow1, $  ; should use DIMENSIONS
                 YSIZE=swallow2, $  ; should use DIMENSIONS
                 USER_INTERFACE=oUI, $  ; output keyword
                 _REF_EXTRA=_extra

    compile_opt idl2

@idlit_on_error2

if (~OBJ_VALID(oTool)) then $
    MESSAGE, 'Tool is not a valid object.'

title = (N_ELEMENTS(titleIn) gt 0) ? titleIn[0] : 'FANS Reduction'

WIDGET_CONTROL, /HOURGLASS

 ;*** Base to hold everything
 ;
wBase = WIDGET_BASE(/COLUMN, MBAR=wMenubar, $
                    TITLE=title, $
                    /TLB_KILL_REQUEST_EVENTS, $
                    /TLB_SIZE_EVENTS, $
                    /KBRD_FOCUS_EVENTS, $
                    _EXTRA=_extra)

 ;*** Create a new UI tool object, using our iTool.
 ;
 oUI = OBJ_NEW('IDLitUI', oTool, GROUP_LEADER=wBase)


 ;***  Menubars
 ;
 hspace = 'Operations/       '
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/File')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/Input')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 wFile       = CW_ITMENU(wMenubar, oUI, 'Operations/Output')
 wFile       = CW_ITMENU(wMenubar, oUI, hspace)
 ;wInsert     = CW_ITMENU(wMenubar, oUI, 'Operations/Edit')
 ;wInsert     = CW_ITMENU(wMenubar, oUI, 'Operations/Insert')
 ;wOperations = CW_ITMENU(wMenubar, oUI, 'Operations/Operations')
 ;wWindow     = CW_ITMENU(wMenubar, oUI, 'Operations/Window')
 wHelp       = CW_ITMENU(wMenubar, oUI, 'Operations/Help')

 ; Assume we were already at 90% at this point.
 ;void = IDLitwdSplash(PERCENT=92)

 ;***  Drawing area.
 ;
 screen = GET_SCREEN_SIZE(RESOLUTION=cm_per_pixel)
 hasDimensions = (N_ELEMENTS(dimensionsIn) eq 2)
if (~hasDimensions)then begin
    ;; Some multi monitor deployments will report a screen size
    ;; that includes all monitors, which can lead to some
    ;; interesting screen geometry. To take this into account, the
    ;; following logic is used to determine the default tool size.
    ;;  - Find the minimum dimension
    ;;  - minDim = dim * 0.4
    ;;  - The other maxDim = minDim * 1.4
    ;;
    ;; The only possible issue with this logic would be if the
    ;; logical screen is so large that this calculation will
    ;; result in a window exceeding a screen.
    dimensions = 0.4*screen
    if(screen[0] gt screen[1])then $ ;; x larger than Y
      dimensions[0] = dimensions[1] * 1.4 $
    else $
      dimensions[1] = dimensions[0] * 1.4
endif else $
    dimensions = dimensionsIn

; Make sure our dimensions are larger than the menubar.
; Otherwise strange things happen on Windows.
; We do this regardless of whether the user provided
; their own dimensions.
geom = WIDGET_INFO(wBase, /GEOMETRY)
minsize = [geom.scr_xsize + 10, 100]

dimensions >= minsize


;***  Toolbars
;
wToolbar = WIDGET_BASE(wBase, /ROW, XPAD=0, YPAD=0, SPACE=7)
wTool1 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/File')
wTool2 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/Edit')
wTool3 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators', /EXCLUSIVE)
wTool4 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/Dataspace')
wTool5 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators/View', /EXCLUSIVE)
wTool6 = CW_ITTOOLBAR(wToolbar, oUI, 'Toolbar/View') 
wTool7 = CW_ITTOOLBAR(wToolbar, oUI, 'Manipulators/Annotation', /EXCLUSIVE)


; It may happen that our toolbar is just slightly larger than
; the specified dimensions. In this case, adjust the dimensions
; to fit the toolbar (unless the user has provided their
; own dimensions).
if (~hasDimensions) then begin
    geom = WIDGET_INFO(wToolbar, /GEOMETRY)
    if ((geom.xsize gt dimensions[0]) && $
        (geom.xsize lt (dimensions[0] + 40))) then $
    dimensions[0] = geom.xsize
endif

; Always adjust the toolbar to fit the base width.
if (dimensions[0] lt geom.xsize) then $
    WIDGET_CONTROL, wToolbar, SCR_XSIZE=dimensions[0]

;***  Panel and Drawing area
wRow = widget_base(wBase, /ROW, $
                    xpad=0, ypad=0, space=0)

wBasePanel  = widget_base(wRow, xpad=0, ypad=0, space=0,/col)
wBaseDraw = widget_base(wRow, xpad=0, ypad=0, space=0)

 ;; The Panel
xsDM = 250 
ysDM = 250
unDM = 'FANSDATAMANAGER'
wDM = cw_FANSDataManager(wBasePanel,oUI,uname=unDM,xsize=xsDM,ysize=ysDM,_extra=_extra)

wPSBase = widget_base(wBasePanel, xpad=0, ypad=0, space=0, /frame)
xsPS = xsDM
ysPS = ysDM*1.0
wPS = 0L
wPS = widget_propertysheet(wBasePanel $
                     ,value=oTool $
                     ,uname='FANSPROPSHEET' $
                     ,scr_xsize=xsPS $
                     ,scr_ysize=ysPS $
                     ,event_func='wd_FANSTool_PropSheet_event' $
                    )

;wButtonBase = widget_base(wBasePanel, /col, xpad=0, ypad=0, space=0, /frame)
;
;tt = 'Click to fit available Fast Background data and extract parameters'
;wFitFast = widget_button(wButtonBase,value='Fit Fast Bkgd',uname='FitFast',tooltip=tt $
;                        ,sensitive=1, xsize=xsDM, yoffset=5)

;tt = 'Click to modify detectors to be masked'
;wMaskDet = widget_button(wButtonBase,value='Mask Detectors',uname='Mask',tooltip=tt $
;                        ,sensitive=1, xsize=xsDM, yoffset=5)

;tt = 'dummy tooltip'
;wDummy = widget_button(wButtonBase,value='Dummy',uname='Dummy',tooltip=tt $
;                        ,sensitive=0, xsize=xsDM, yoffset=5)

; If the user did not explicitly provide dimensions, and
; the panel height is greater than the default (half-screen)
; height, but less than half again the default height, then
; adjust our default height to match the panel height (so
; scrollbars will not be necessary).
if (~hasDimensions) then begin
    geom = WIDGET_INFO(wBasePanel, /GEOMETRY)
    if ((geom.scr_ysize gt dimensions[1]) && $
        (geom.scr_ysize lt (dimensions[1]*1.5))) then begin
        dimensions[1] = geom.scr_ysize
    endif
    minsize[1] = dimensions[1]
endif

;void = IDLitwdSplash(PERCENT=95)

;; Make our window
if (hasDimensions && N_ELEMENTS(virtualDimensions) eq 0) then $
    virtualDimensions = dimensions

;; Make our window
baseWin = dimensions
wDraw = CW_ITWINDOW(wBaseDraw, oUI, $
                     DIMENSIONS=dimensions, $
                     VIRTUAL_DIMENSIONS=virtualDimensions)

;*** Status bar.
;
dimensions[0] = dimensions[0] + geom.scr_xsize  ;  add the panel width
wStatus = CW_itStatusBar(wBase, oUI, XSIZE=dimensions[0])

; Cache some information.
;
wChild = WIDGET_INFO(wBase, /CHILD)

if(n_elements(location) eq 0)then begin
    location = [(screen[0] - dimensions[0])/2 - 10, $
                ((screen[1] - dimensions[1])/2 - 100) > 10]
endif
WIDGET_CONTROL, wBase, MAP=0, $
    TLB_SET_XOFFSET=location[0], TLB_SET_YOFFSET=location[1]

;; State structure for the widget
geomBase = widget_info(wBase,/geometry)
geomDM = widget_info(wDM,/geometry)
geomPS = widget_info(wPS,/geometry)
geomDraw = widget_info(wDraw,/geometry)
geomSB = widget_info(wStatus,/geometry)
State = { oTool     : oTool    $
          ,oUI       : oUI      $

          ,wBase      : wBase    $
          ,title     : title    $
          ,minsize   : [geomBase.xsize,geomBase.ysize]  $
          ,basesize  : [0L, 0L] $
          ,baseDM    : [xsDM,ysDM] $;[geomDM.scr_xsize, geomDM.scr_ysize] $  
          ,basePS    : [xsPS,ysPS] $;[geomPS.scr_xsize, geomPS.scr_ysize] $
          ,baseWin   : [geomDraw.draw_xsize, geomDraw.draw_ysize] $
          ,baseSB    : [geomSB.scr_xsize, geomSB.scr_ysize] $ $
          ,wToolbar  : wToolbar $
          ,wDraw     : wDraw    $
;          ,wFitFast  : wFitFast $
          ,wPanel    : wBasePanel   $
          ,wDM       : wDM      $
          ,wPS       : wPS      $ 
          ,wStatus   : wStatus   $
          ,yDataID   : ''       $ ; identifier of the cuurent y-axis data
          ,xDataID   : ''       $
          ,eDataID   : ''       $
          ,xBkgdID   : ''       $ 
          ,itemID    : ''       $ ; id of item currently selected in DM   
          ,manipID   : ''       $ ; id of active manipulator
          ,xrange    : [0.,0.]  $
          ,yrange    : [0.,0.]  $
          }
pState = PTR_NEW(state, /NO_COPY)
WIDGET_CONTROL, wChild, SET_UVALUE=pState

; Force an initial resize.
;wd_FANStool_resize, pState, 0, 0

;void = IDLitwdSplash(PERCENT=100)

WIDGET_CONTROL, wBase, /REALIZE

; Retrieve the starting dimensions and store them.
; Used for window resizing in event processing.
WIDGET_CONTROL, wBase, TLB_GET_SIZE=basesize
(*pState).basesize = basesize

;; Register ourself as a widget with the UI object. 
;; Returns a string containing our identifier (a user interface adaptor is created  during reg)
myID = oUI->RegisterWidget(wBase, 'ToolBase', 'wd_FANStool_callback')

;; Register for messages from the visualization tool (oTool).
oUI->AddOnNotifyObserver, myID, oTool->GetFullIdentifier()

; Also observe the system object for messages
oSys = oTool->_GetSystem()
oUI->AddOnNotifyObserver, myID, oSys->GetFullIdentifier()

WIDGET_CONTROL, wChild, KILL_NOTIFY="wd_FANStool_cleanup"
WIDGET_CONTROL, wBase, /MAP ;show the user what we have made.

; Start event processing for the tool.
XMANAGER, 'wd_FANStool', wBase, /NO_BLOCK

end
;------------------------------------------------------------------------------
