; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEvisPlot
;
; PURPOSE:
;   This file simply inherits and extends IDL's IDLitVisPlot visualization class
;
; CATEGORY:
;   DAVE Visualization
;
; SUPERCLASSES:
;   IDLitVisPlot
;
; SUBCLASSES:
;
; CREATION:
;
; METHODS:
;   OnDataChangeUpdate
;   _UpdateCapSize
;   _UpdateErrorBars
;   _UpdateFill
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; September 2004
;-
;###############################################################################


;===============================================================================
; DAVEvisPlot::OnDataChangeUpdate
; 
; PURPOSE:
;   Overide IDLitVisPlot::OnDataChangeUpdate. This procedure method is
;   called by a Subject via a Notifier when its data has changed.
;   This method obtains the data from the subject and updates the
;   internal IDLgrPlot object. 
;
; PARAMETERS:
;   oSubject [in] - The data item for the parameter that has changed.
;
;   parmName [in] - The name of the parameter that has changed.
;
; KEYWORDS:
;
pro DAVEvisPlot::OnDataChangeUpdate, oSubject, parmName, no_update=noUpdate ;skipBaseClass=skipBaseClass, _EXTRA=etc
compile_opt idl2

case strupcase(parmName) of
    '<PARAMETER SET>':begin
        ;; Get our data
        position = oSubject->Get(/ALL, count=nCount, NAME=name)
        for i=0, nCount-1 do begin
            if (name[i] eq '') then $
                continue
            oData = (oSubject->GetByName(name[i]))[0]
            if (~OBJ_VALID(oData)) then $
              continue
            if (oData->GetData(data, NAN=nan) le 0) then $
              continue

            case name[i] of

                'Y': begin
                    self->OnDataChangeUpdate, oData, 'Y', /NO_UPDATE

                    oData = oSubject->GetByName('X')
                    if (OBJ_VALID(oData)) then begin
                        self->OnDataChangeUpdate, oData, 'X', /NO_UPDATE
                    endif

                    self->_UpdateFill
                    self->_UpdateSymSize ; handle the case for an overplot with no range change

                end

                'X':            ; X is handled in the Y branch to control order

                ;; Pass all other parameters on to ourself.
                else: self->OnDataChangeUpdate, oData, name[i]

            endcase

        endfor
    end

    'X': BEGIN
        if (~oSubject->GetData(data)) then $
          break
        ;; Retrieve the range of the plot data
        void = self._oPlot->GetDataXYZRange(xRange, yRange, zRange)
        oDataSpace = self->GetDataSpace(/UNNORMALIZED)

        if (obj_valid(oDataSpace)) then begin
            ;; check for metadata and process them
            histogram = 0
            if (obj_isa(oSubject,'IDLitData') && (oSubject->GetMetadataCount() gt 0)) then begin
                
                ;; Axis Label
                xLabel = ''
                if (oSubject->GetMetaData('LONG_NAME',label)) then xLabel += label
                if (oSubject->GetMetaData('UNITS',units)) then $
                  xLabel = (strlen(units) gt 0)? xLabel + ' ('+units+')' : xLabel
                
                oAxes = oDataSpace->GetAxes(/container)
                
                if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
                  oAxes->SetProperty,xtitle=strtrim(xLabel)
                
                ;; Set plot mode: points / histograms
                if (oSubject->GetMetaData('Distribution',distribution)) then $
                  histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
                
                self->SetProperty, histogram=histogram
                
            endif
            
            if (histogram) then begin
                ;; convert to point mode
                nx = n_elements(data)
                index = lindgen(nx-1)
                points = 0.5*(data[index]+data[index+1]) ; ==> mid values of the bin boundaries
                data = temporary(points)
            endif
            
            oDataSpace->GetProperty, X_LOG=xLog, Y_LOG=yLog $
              ,XRANGE=xRange, YRANGE=yRange, ZRANGE=zRange
            if (xLog gt 0) then  data=alog10(data)
        endif

        self._oPlot->SetProperty, DATAX=data

        ;; update errors
        self->_UpdateErrorBars, 1

        if (~KEYWORD_SET(noUpdate)) then $
          self->OnDataRangeChange, self, xRange, yRange, zRange

    END

    'Y': BEGIN
        if (~oSubject->GetData(data, NAN=nan)) then $
          break
        ;; Retrieve the range of the plot data
        void = self._oPlot->GetDataXYZRange(xRange, yRange, zRange)
        oDataSpace = self->GetDataSpace(/UNNORMALIZED)

        if (OBJ_VALID(oDataSpace)) then begin
            oDataSpace->GetProperty, X_LOG=xLog, Y_LOG=yLog $
              ,XRANGE=xRange, YRANGE=yRange, ZRANGE=zRange
            if (yLog gt 0) then $
              data=alog10(data)
            
            ;; check for metadata and process them
            if (obj_isa(oSubject,'IDLitData') && (oSubject->GetMetadataCount() gt 0)) then begin
                ;; Y label
                yLabel = ''
                if (oSubject->GetMetaData('LONG_NAME',label)) then yLabel += label
                if (oSubject->GetMetaData('UNITS',units)) then $
                  yLabel = (strlen(units) gt 0)? yLabel + ' ('+units+')' : yLabel
                
                oDataSpace = self->GetDataSpace()
                oAxes = oDataSpace->GetAxes(/container)
                
                if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
                  oAxes->SetProperty,ytitle=strtrim(yLabel)
                
            endif
        endif

        self._oPlot->SetProperty, DATAY=data, HIDE=0
        self->SetPropertyAttribute, ['MIN_VALUE', 'MAX_VALUE'], HIDE=0

        ;; update errors
        self->_UpdateErrorBars, 1

        if (~KEYWORD_SET(noUpdate)) then $
          self->OnDataRangeChange, self, xRange, yRange, zRange

    END

    'VERTICES': BEGIN
        if (~oSubject->GetData(data)) then $
          break
        ;; Sanity check.
        if ((SIZE(data, /DIM))[0] gt 3) then $
          break

        ;; Retrieve the range of the plot data
        void = self._oPlot->GetDataXYZRange(xRange, yRange, zRange)
        oDataSpace = self->GetDataSpace(/UNNORMALIZED)
        if (OBJ_VALID(oDataSpace)) then $
          oDataSpace->GetProperty, XRANGE=xRange, YRANGE=yRange, ZRANGE=zRange

        self._oPlot->SetProperty, DATAY=data[1,*]
        self._oPlot->SetProperty, DATAX=data[0,*]
        if (~KEYWORD_SET(noUpdate)) then begin
            self->OnDataRangeChange, self, xRange, yRange, zRange
        endif
    END

    'X ERROR': self->_UpdateErrorBars, 0

    'Y ERROR': self->_UpdateErrorBars, 1

    'PALETTE': begin
        if (~oSubject->GetData(data)) then $
          break
        ;; Sanity check.
        if ((SIZE(data, /DIM))[0] gt 3) then $
          break
        if (size(data, /TYPE) ne 1) then data=bytscl(temporary(data))
        self._oPalette->SetProperty, $
          RED_VALUES=data[0,*], $
          GREEN_VALUES=data[1,*], $
          BLUE_VALUES=data[2,*]
        self->SetPropertyAttribute, 'VISUALIZATION_PALETTE', /SENSITIVE
        oVertColors = self->GetParameter('VERTEX_COLORS')
        if OBJ_VALID(oVertColors) then begin
            success = oVertColors->GetData(colors)
        endif
        if ~OBJ_VALID(oVertColors) || $
          (size(colors, /n_dimensions) gt 1) then begin
            ;; define default indices
            oVertColorsDefault = OBJ_NEW('IDLitDataIDLVector', BINDGEN(256), $
                                         NAME='<DEFAULT INDICES>')
            result = self->SetData(oVertColorsDefault, $
                                   PARAMETER_NAME='VERTEX_COLORS',/by_value)
        endif
    end

    'VERTEX_COLORS': begin
        if (~oSubject->GetData(data)) then $
          break
        if (size(data, /TYPE) ne 1) then data=bytscl(temporary(data))
        self._oPlot->SetProperty, VERT_COLORS=data

        oRgbTable = self->GetParameter('PALETTE')
        if ~OBJ_VALID(oRgbTable) && $
          (size(data, /n_dimensions) eq 1) then begin
            ;; define default palette, allows editing colors
            ;; only used if vertex colors parameter is supplied
            ;; and vertex colors are indices not colors.
            ramp = BINDGEN(256)
            colors = transpose([[ramp],[ramp],[ramp]])
            oColorTable = OBJ_NEW('IDLitDataIDLPalette', colors, NAME='RGB Table')

            ;; Set the data as by_value, so the parameter interface
            ;; will manage it.
            result = self->SetData(oColorTable, PARAMETER_NAME='PALETTE',/by_value)
        endif
    end
    
    else:                       ; ignore unknown parameters

endcase

self->_UpdateSelectionVisual

; Since we are changing a bunch of attributes, notify
; our observers in case the prop sheet is visible.
self->DoOnNotify, self->GetFullIdentifier(), 'SETPROPERTY', ''

end



;===============================================================================
; DAVEvisPlot::_UpdateErrorBars
; 
; PURPOSE:
;   Overide IDLitVisPlot::_UpdateErrorBars. Updates the error bar
;   geometry based on the plot data.
;
; PARAMETERS:
;   dim [in] -  0 for X, 1 for Y
;
; KEYWORDS:
;
pro DAVEvisPlot::_UpdateErrorBars, dim
compile_opt idl2

if (float(!version.release) ge 8.1) then begin
  Self->IDLitVisPlot::_UpdateErrorBars, dim
  return
endif


oErrData = self->GetParameter((dim ? 'Y' : 'X') + ' ERROR')
if (~OBJ_VALID(oErrData)) then $
  return
if (~oErrData->GetData(errdata)) then $
  return

; Retrieve X and Y data.
oDataY = self->GetParameter('Y')
if (~OBJ_VALID(oDataY) || ~oDataY->GetData(ydata)) then $
  return

self._oPlot->GetProperty, DOUBLE=isDouble

ndata = N_ELEMENTS(ydata)

oDataX = self->GetParameter('X')
if (obj_valid(oDataX) && oDataX->GetData(xdata)) then begin
    if (~oDataX->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        nx = n_elements(xdata)
        index = lindgen(nx-1)
        points = 0.5*(xdata[index]+xdata[index+1]) ; ==> mid values of the bin boundaries
        xdata = temporary(points)
    endif
endif

; Construct a findgen X vector if necessary.
if (N_ELEMENTS(xdata) ne ndata) then $
  xdata = (isDouble)? DINDGEN(ndata) : FINDGEN(ndata) ;FINDGEN(ndata)

self->_VerifyErrorBars, dim

ErrorData = (isDouble)? DBLARR(2, 2*ndata) : FLTARR(2, 2*ndata) ;fltarr(2, 2*ndata)

; set the other dimension's coordinates of polyline data
; same coordinate for both values of the specified dimension
ErrorData[~dim, 0:*:2] = dim ? xdata : ydata
ErrorData[~dim, 1:*:2] = dim ? xdata : ydata

case size(errdata, /n_dimensions) of
    1: begin
        ;; vector of error values applied as both + and - error
        ErrorData[dim, 0:*:2] = (dim ? ydata : xdata) - errdata
        ErrorData[dim, 1:*:2] = (dim ? ydata : xdata) + errdata
    end
    2: begin
        ;; 2xN array, [0,*] is negative error, [1,*] is positive error
        ErrorData[dim, 0:*:2] = (dim ? ydata : xdata) - errdata[0,*]
        ErrorData[dim, 1:*:2] = (dim ? ydata : xdata) + errdata[1,*]
    end
    else:
endcase

;; Handle logarithmic axes.
if (self._xVisLog) then begin
    ErrorData[0, *] = alog10(ErrorData[0, *])
endif
if (self._yVisLog) then begin
    ErrorData[1, *] = alog10(ErrorData[1, *])
endif

polylineDescript = LONARR(3*ndata)
polylineDescript[0:*:3] = 2
polylineDescript[1:*:3] = lindgen(ndata)*2
polylineDescript[2:*:3] = lindgen(ndata)*2+1

;; filter out non-finite values
infY = where(~finite(ErrorData[1,0:*:2]))
IF (infY[0] NE -1) THEN BEGIN
    ;; determine a valid data point to set the unneeded error bar
    ;; data.  This has to be done as the error bar data impacts the
    ;; range so using a generic value like 0 or 1d300 could change
    ;; the plot range
    validY = where(finite(ydata))
    validY = (validY[0] EQ -1) ? 0 : ydata[validY[0]]
    ErrorData[1,where(~finite(ErrorData[1,*]))] = validY
    nInf = n_elements(infY)
    polylineDescript[reform(infY*3##replicate(1,3),3*nInf)+ $
                     reform([0,1,2]#replicate(1,nInf),3*nInf)] = 0
ENDIF
infX = where(~finite(ErrorData[0,0:*:2]))
IF (infX[0] NE -1) THEN BEGIN
    ;; determine a valid data point to set the unneeded error bar
    ;; data.  This has to be done as the error bar data impacts the
    ;; range so using a generic value like 0 or 1d300 could change
    ;; the plot range
    validX = where(finite(xdata))
    validX = (validX[0] EQ -1) ? 0 : xdata[validX[0]]
    ErrorData[0,where(~finite(ErrorData[0,*]))] = validX
    nInf = n_elements(infX)
    polylineDescript[reform(infX*3##replicate(1,3),3*nInf)+ $
                     reform([0,1,2]#replicate(1,nInf),3*nInf)] = 0
ENDIF

oError = dim ? self._oYerror : self._oXerror

;; We save an extra copy of the polylines in the UVALUE,
;; for use in clipping to the plot range.
;; Retrieve HIDE property - it may be specified on command line
;; and set prior to processing of the parameter
oError->GetProperty, HIDE=hide
oError->SetProperty, DATA=temporary(ErrorData), $
                     HIDE=hide, $ ; may be hid from dataDisconnect
                     POLYLINES=polylineDescript, UVALUE=polylineDescript

oErrorContainer = dim ? $
                  self._oItYErrorBarContainer : self._oItXErrorBarContainer
oErrorContainer->SetProperty, /IMPACTS_RANGE

;; display the properties, even if the error bars themselves are hidden
self->SetPropertyAttribute, [(dim ? 'Y' : 'X') + '_ERRORBARS', $
                             'ERRORBAR_CAPSIZE', 'ERRORBAR_COLOR'], HIDE=0

self->_UpdateCapSize

end


;----------------------------------------------------------------------------
; METHODNAME:
;      IDLitVisPlot::_UpdateCapSize
;
; PURPOSE:
;      This procedure method scales the error bar geometry
;      to the dataspace data range.
;
; CALLING SEQUENCE:
;      Obj->[IDLitVisPlot::]_UpdateCapSize
;
; INPUTS:
;      There are no inputs for this method.
;
; KEYWORD PARAMETERS:
;      There are no keywords for this method.
;
pro DAVEvisPlot::_UpdateCapSize
compile_opt idl2, hidden

if (float(!version.release) ge 8.1) then begin
  Self->IDLitVisPlot::_UpdateCapSize
  return
endif


    oDataSpace = self->GetDataSpace(/UNNORMALIZED)
    if (OBJ_VALID(oDataSpace)) then begin
        ;;success = oDataSpace->GetXYZRange(xRange, yRange, zRange) !!! bug !
        self._oplot->getproperty,xrange=xRange,yrange=yRange
;        if (success) then begin
        if ((n_elements(xRange) gt 0) && (n_elements(yRange) gt 0)) then begin
            if (OBJ_VALID(self._oXErrorSym)) then $
              self._oXErrorSym->SetProperty, $
              SIZE=self._capSize*(yRange[1]-yRange[0])/10.0
            
            if (OBJ_VALID(self._oYErrorSym)) then $
              self._oYErrorSym->SetProperty, $
              SIZE=self._capSize*(xRange[1]-xRange[0])/10.0
            
        endif
    endif
end




;===============================================================================
; DAVEvisPlot::_UpdateFill
; 
; PURPOSE:
;   Overide IDLitVisPlot::_UpdateFill. This procedure method updates
;   the polygon representing the filled area under the plot.  It must
;   be updated when the fill level (the lower boundary) changes or
;   when going into or out of histogram mode, for example. 
;
; PARAMETERS:
;   dataspaceX/Yrange/Zrange - Optional args giving the dataspace ranges
;
; KEYWORDS:
;
pro DAVEvisPlot::_UpdateFill, dataspaceXrange, dataspaceYrange, dataspaceZrange
compile_opt idl2

if (float(!version.release) ge 8.1) then begin
  Self->IDLitVisPlot::_UpdateFill, dataspaceXrange, dataspaceYrange, dataspaceZrange
  return
endif

;; Don't bother to update if we havn't turned on filling.
if (~OBJ_VALID(self._oFill)) then $
  return

;; Retrieve X and Y data.
oDataY = self->GetParameter('Y')
if ((~OBJ_VALID(oDataY)) || (~oDataY->GetData(ydata))) then $
  return

ndata = N_ELEMENTS(ydata)

oDataX = self->GetParameter('X')
if (obj_valid(oDataX) && oDataX->GetData(xdata)) then begin
    if (~oDataX->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        nx = n_elements(xdata)
            index = lindgen(nx-1)
            points = 0.5*(xdata[index]+xdata[index+1]) ; ==> mid values of the bin boundaries
            xdata = temporary(points)
        endif
endif
;; Construct a findgen X vector if necessary.
if (N_ELEMENTS(xdata) ne ndata) then xdata = FINDGEN(ndata)

self._oPlot->GetProperty, $
  DOUBLE=isDouble, $
  HISTOGRAM=doHist, $
  MIN_VALUE=minValue, $
  MAX_VALUE=maxValue, $
  NSUM=nsum, $
  XRANGE=xrange, YRANGE=yrange


;; Check if we need to average points together.
if (nsum gt 1) then begin
    
    ;; If nsum is >= to the # of points, bail.
    if (nsum ge ndata) then begin
        ;; Setting the fill to a scalar will cause it to be hidden.
        self._oFill->SetProperty, DATA=0
        return                  ; we're done
    endif
    
    xdata = isDouble ? DOUBLE(xdata) : FLOAT(xdata)
    ydata = isDouble ? DOUBLE(ydata) : FLOAT(ydata)
    
    ;; Remove degen dimensions for REBIN
    xdata = REFORM(xdata, /OVERWRITE)
    ydata = REFORM(ydata, /OVERWRITE)
    
    if ((ndata mod nsum) eq 0) then begin
        xdata = REBIN(xdata, (ndata + nsum - 1)/nsum)
        ydata = REBIN(ydata, (ndata + nsum - 1)/nsum)
    endif else begin
        nint = (ndata/nsum)*nsum
        xdata = [REBIN(xdata[0:nint-1], ndata/nsum), $
                 TOTAL(xdata[nint:*])/(ndata-nint)]
        ydata = [REBIN(ydata[0:nint-1], ndata/nsum), $
                 TOTAL(ydata[nint:*])/(ndata-nint)]
    endelse

                                ; New number of data values.
    ndata = N_ELEMENTS(ydata)
endif


;; Create default FILL_LEVEL if necessary.
;; We use a tiny number as a flag to indicate it is currently undefined.
if (self._fillLevel eq 1d-300) then begin
    self._fillLevel = MIN(ydata)
    ;; Start showing the actual value.
    self->SetPropertyAttribute, 'FILL_LEVEL', UNDEFINED=0
endif


;; For a histogram style, duplicate the X and Y points.
if (doHist) then begin
    
    ;; Duplicate the X points, but subtract/add an offset
    ;; equal to half the distance between neighboring points.
    dx = (ndata gt 1) ? (xdata[1:*] - xdata[0:ndata-2])/2d : 1
    dx = [0, dx, 0]
    xdata1 = isDouble ? DBLARR(2*ndata) : FLTARR(2*ndata)
    xdata1[0:*:2] = xdata - dx[0:ndata-1]
    xdata1[1:*:2] = xdata + dx[1:*]
    xdata = TEMPORARY(xdata1)
    
    ;; Duplicate all of the y data.
    ydata = REFORM(REBIN(REFORM(ydata, 1, ndata), 2, ndata), 2*ndata)
    
endif

fillLevel = self._fillLevel

;; Convert data to logarithmic if necessary.
if (self._xVisLog) then begin
    xdata = alog10(xdata)
endif
if (self._yVisLog) then begin
    ydata = alog10(ydata)
    fillLevel = alog10(fillLevel)
endif

;; Clip the yrange at the min/max value. This isn't perfect,
;; because it still shows the fill in regions where the data has all
;; been set to NaN.
if (FINITE(minValue)) then $
  yrange[0] >= minValue
if (FINITE(maxValue)) then $
  yrange[1] <= maxValue

;; Restrict fill level to be within dataspace range.
if (N_ELEMENTS(dataspaceYrange) ne 2) then begin
    oDataSpace = self->GetDataSpace(/UNNORMALIZED)
    if (OBJ_VALID(oDataSpace)) then begin
        oDataSpace->GetProperty, Y_AUTO_UPDATE=yAutoUpdate, $
          Y_MINIMUM=yMin, Y_MAXIMUM=yMax
        ;; If y_auto_update is false then our fill needs
        ;; to match the dataspace. If y_auto_update is true then
        ;; because impacts_range=1 for our fill, the dataspace
        ;; y range should change to match our fill range.
        dataspaceYrange = [yMin, yMax]
    endif
endif

if (N_ELEMENTS(dataspaceYrange) eq 2) then begin
    fillLevel = dataspaceYrange[0] > fillLevel < dataspaceYrange[1]
endif

ndata = self->_FillClipX(xrange, xdata, ydata, HISTOGRAM=doHist)

if (ndata gt 1) then $
  ndata = self->_FillClipY(yrange, xdata, ydata)

;; If no points survived the clipping, bail early.
if (ndata eq 0) then begin      ; Clip everything
    ;; Setting the fill to a scalar will cause it to be hidden.
    self._oFill->SetProperty, DATA=0
    return                      ; we're done
endif


;; Construct our (2xN) data array.
;self._oPlot->GetProperty, USE_ZVALUE=useZvalue, ZVALUE=zvalue
self._oPlot->GetProperty, ZVALUE=zvalue
useZvalue = (zvalue ne 0)

filldata = isDouble ? DBLARR(2+useZvalue, ndata + 3) : $
           FLTARR(2+useZvalue, ndata + 3)

;; Fill in all the data points.
filldata[0,1:ndata] = xdata
filldata[1,1:ndata] = ydata

;; Fill in the first and last points.
filldata[0:1,0] = [xdata[0], fillLevel]
filldata[0:1,ndata+1] = [xdata[ndata-1], fillLevel]
filldata[0:1,ndata+2] = [xdata[0], fillLevel]

if (useZvalue) then $
  filldata[2,*] = zvalue

self._oFill->SetProperty, DATA=filldata

end


;===============================================================================
; DAVEvisPlot::Init
;
; PURPOSE:
;   Initialization method for objects of DAVEvisPlot class
;
; RETURN VALUE:
;   1 = success
;   0 = failure
;
function DAVEvisPlot::Init, _REF_EXTRA=etc

; Call base class
if (~self->IDLitVisPlot::Init(_EXTRA=etc)) then return, 0

; Change zoomFactor of View from 100% --> 130%
self->GetProperty, parent=parent
while (obj_valid(parent) && ~obj_isa(parent,'IDLitgrView')) do $
  parent->GetProperty,parent=parent 

if (obj_valid(parent) && obj_isa(parent,'IDLitgrView')) then $
  parent->SetProperty, zoomFactor = 1.30d
  
; Set some sensible defults
colors = [[0,0,0] $     ; Black
         ,[255,0,0] $   ; Red
         ,[0,255,0] $   ; Green
         ,[0,0,255] $   ; Blue
         ,[100,0,0] $   ; dark red
         ,[0,100,0] $   ; dark Green
         ,[0,0,100] $   ; dark Blue
         ]
icol = (size(colors,/dimensions))[1]
cindex = fix(icol * randomu(seed))   ; values 0 --> icol-1
color = colors[*,cindex]                ; one from colors array above
sym_index = fix(23*randomu(seed))+2      ; symbols 2 --> 24
if (sym_index eq 3 || sym_index eq 8 || sym_index eq 9) then sym_index = 24  ; ignore symbol 3 (Dot), 8 (>) and 9 (<)

self->SetProperty, linestyle=6 $                ; no lines
                  ,sym_index=sym_index $        ; randomly selected symbol
                  ,use_default_color=1 $        ; don't use seperate color for symbol
                  ,errorbar_color=color $       ; errorbar color
                  ,color=color         $       ; foreground color
                  ,filled=1                    ; use filled symbols
;print,'sym_index=',sym_index

return, 1
end

;===============================================================================
; DAVEvisPlot__define
; 
; PURPOSE:
;   DAVEvisPlot class structure definition
;
pro DAVEvisPlot__define

    compile_opt idl2

    struc = {DAVEvisPlot, $
        inherits IDLitVisPlot }

end
