; $Id$
;
; Copyright (c) 2000-2004, Research Systems, Inc.  All rights reserved.
;   Unauthorized reproduction prohibited.
;----------------------------------------------------------------------------
;+
; CLASS_NAME:
;   IDLitToolbase
;
; PURPOSE:
;   This file implements the FANS Tool object
;
; CATEGORY:
;   
;
; SUPERCLASSES:
;   IDLitTool
;
; SUBCLASSES:
;
; CREATION:
;   See IDLitToolbase::Init
;
; METHODS:
;   This class has the following methods:
;
;   IDLitToolbase::Init
;
; INTERFACES:
; IIDLProperty
;-


;===============================================================================
; fans_Arcsin
;
; PURPOSE:
;   Function definition used to fit fast background data. It is
;   defined as follows:
;   f = p[0]+p[1]/x+p[2]/x^2+p[3]invsin[sqrt(20.45/x)/p[4]]
;   where
;   x=indep vars, p=params, f=dep var, pder=partial derivatives
;   p[4] is the monocromator d-spacing and should be constant
;   p[5] is a constant for converting between meV <--> wavenumber
;
; PARAMETERS:
;   xvalue [in] - array of independent values.
;
;   par [in] - array of function parameters.
;
; RETURN VALUES:
;   The function and partial derivative evaluated for the specified xvalue
;
function fans_Arcsin, xvalue,par
; evaluate function
xval = xvalue/par[5]
xsq = xval^2
invx = 1.0/xval
invxsq = invx^2
arg = (1.0/par[4])*sqrt(20.451*invx)
res = where( ABS(arg) gt 1.0 )
if (res[0] ne -1) then begin
    if ( arg[res[0]] lt 0.0 ) then begin  ; ie if -ve
        arg[res] = -1.0
    endif else begin
        arg[res] = 1.0
    endelse 
endif
arcsin = asin(arg)
; evaluate function
yval = par[0] + par[1]*invx + par[2]*invxsq + par[3]*arcsin

; evaluate partial derivatives wrt each parameter
omarcsin2 = 1 - arcsin^2
res = where(omarcsin2 le 0.0)
if (res[0] ne -1) then $
  omarcsin2[res] = 0.0001
pder = [  [replicate(1.0,n_elements(xval))], $
          [invx], $
          [invxsq],$
          [arcsin], $
          [-1.0*arcsin/par[4]/sqrt(omarcsin2)],$
          [replicate(1.0,n_elements(xval))] ]

return, [[yval],[pder]]

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_Cubic
;
; PURPOSE:
;   Function definition used to fit fast background data. It is
;   defined as follows:
;   f = p[0] + p[1]*x + p[2]*x^2 + p[3]x^3
;   where
;   x=indep vars, p=params, f=dep var, pder=partial derivatives wrt p
;
; PARAMETERS:
;   xval [in] - array of independent values.
;
;   par [in] - array of function parameters.
;
; RETURN VALUES:
;   The function and partial derivative evaluated for the specified xvalue
;
function fans_Cubic, xval,par
x2 = xval^2
x3 = xval^3
yval = par[0] + par[1]*xval + par[2]*x2 + par[3]*x3 

; evaluate partial derivatives wrt each parameter
pder = [  [replicate(1.0,n_elements(xval))], $
          [xval], $
          [x2], $ 
          [x3] ]

return, [[yval],[pder]]

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_Quad
;
; PURPOSE:
;   Function definition used to fit fast background data. It is
;   defined as follows:
;   f = p[0] + p[1]*x + p[2]*x^2
;   where
;   x=indep vars, p=params, f=dep var, pder=partial derivatives wrt p
;
; PARAMETERS:
;   xval [in] - array of independent values.
;
;   par [in] - array of function parameters.
;
; RETURN VALUES:
;   The function and partial derivative evaluated for the specified xvalue
;
function fans_Quad, xval,par
yval = par[0] + par[1]*xval + par[2]*xval^2

; evaluate partial derivatives wrt each parameter
pder = [  [replicate(1.0,n_elements(xval))], $
          [xval], $
          [xval^2] ]

return, [[yval],[pder]]

end
;-------------------------------------------------------------------------------



;===============================================================================
; fans_Linear
;
; PURPOSE:
;   Function definition used to fit fast background data. It is
;   defined as follows:
;   f = p[0] + p[1]*x
;   where
;   x=indep vars, p=params, f=dep var, pder=partial derivatives wrt p
;
; PARAMETERS:
;   xval [in] - array of independent values.
;
;   par [in] - array of function parameters.
;
; RETURN VALUES:
;   The function and partial derivative evaluated for the specified xvalue
;
function fans_Linear, xval,par
yval = par[0] + par[1]*xval

; evaluate partial derivatives wrt each parameter
pder = [  [replicate(1.0,n_elements(xval))], $
          [xval] ]

return, [[yval],[pder]]

end
;-------------------------------------------------------------------------------

;===============================================================================
; FANSTool::GetSystem
; 
; PURPOSE:
;   Return the IDLitSystem object
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::GetSystem
compile_opt idl2

return, self->_GetSystem()

end


;===============================================================================
; FANSTool::GetUI
; 
; PURPOSE:
;   Return the IDLitUI object for this tool
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::GetUI
compile_opt idl2

return, self._oUIConnection

end


;-------------------------------------------------------------------------------
pro FANSTool::Cleanup
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::Cleanup: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle)

        catch, /cancel
        ptr_free, Self.prefsPtr
        ptr_free, Self.historyPtr
        Self->IDLitTool::Cleanup
        return
    endif
endif

;; Save current user graphic customizations and preferences
if (Self->HasVisualizations()) then begin
   oWin = Self->GetCurrentWindow()
   oView = oWin->GetCurrentView()
   oLayer = oView->GetCurrentLayer()
   oWorld = oLayer->GetWorld()
   oDataSpace = oWorld->GetDataSpaces()
   oVis = oDataSpace->GetVisualizations(count = nVis)
   
   for j=nVis-1,0,-1 do begin
      oVis[j]->Select
      Self->saveVisProps, oVis[j]
   endfor
endif

Self->GetProperty, preferences=preferences
if (n_elements(preferences) gt 0) then $
   status = fansReducPreferences(preferences, /save)

ptr_free, Self.prefsPtr
ptr_free, Self.historyPtr

; Clean up samp container
oItems = Self.sampContRef->Get(/all, count=n)
if (n gt 0) then begin
   for i=0,n-1 do begin
      void = oItems[i]->GetMetaData('DATAPTRREF',dataPtr)
      if (ptr_valid(dataPtr)) then heap_free, dataPtr
   endfor
endif

; Clean up samp container
oItems = Self.bkgdContRef->Get(/all, count=n)
if (n gt 0) then begin
   for i=0,n-1 do begin
      void = oItems[i]->GetMetaData('DATAPTRREF',dataPtr)
      if (ptr_valid(dataPtr)) then heap_free, dataPtr
   endfor
endif

; Clean up samp container
oItems = Self.fastContRef->Get(/all, count=n)
if (n gt 0) then begin
   for i=0,n-1 do begin
      void = oItems[i]->GetMetaData('DATAPTRREF',dataPtr)
      if (ptr_valid(dataPtr)) then heap_free, dataPtr
   endfor
endif

obj_destroy, [Self.sampContRef,Self.bkgdContRef,Self.fastContRef]

; Call base class cleanup
self->IDLitTool::Cleanup

end



;-------------------------------------------------------------------------------
function FANSTool::Exit
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::Exit: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wd_FANStool_event, { WIDGET_KILL_REQUEST, ID:wTLB, TOP:wTLB, HANDLER:wTLB }

return, 1
end


;-------------------------------------------------------------------------------
pro FANSTool::GetProperty, nameTag=nameTag, prompt=prompt, prmptTitle=prmptTitle, prmptDesc=prmptDesc $
 ,nActiveDetectors=nActiveDetectors,energyUnitFlag=energyUnitFlag $
 ,maskedDets=maskedDets,fastFunc=fastFunc,monScaleValue=monScaleValue $
 ,monScaleFlag=monScaleFlag,maskFlag=maskFlag,bkgdFlag=bkgdFlag $
 ,fastFlag=fastFlag,nSamp=nSamp,nBkgd=nBkgd,nFast=nFast,funcs=funcs $
 ,sumSampFlag=sumSampFlag,sumBkgdFlag=sumBkgdFlag,sumFastFlag=sumFastFlag $
 ,sampContRef=sampContRef,bkgdContRef=bkgdContRef,fastContRef=fastContRef $
 ,data_directory=dataDir,mev2wnos=mev2wnos,fastparams=fastparams, daveTool=daveTool $
 ,sampSelected=sampSelected, bkgdSelected=bkgdSelected, fastSelected=fastSelected $
 ,preferences=preferences,history=history,oplotFastFitFlag=oplotFastFitFlag $
 ,oplotBkgdFlag=oplotBkgdFlag, modifiedStatus=modifiedStatus, ftpObject=ftpObject $
 ,s_errorbar_capsize=s_errorbar_capsize $
 ,s_transparency=s_transparency $
 ,s_antialias=s_antialias $
 ,s_color=s_color $
 ,s_errorbar_color=s_errorbar_color $
 ,s_use_default_color=s_use_default_color $
 ,s_sym_increment=s_sym_increment $
 ,s_sym_index=s_sym_index $
 ,s_sym_size=s_sym_size $
 ,s_sym_color=s_sym_color $
 ,s_sym_thick=s_sym_thick $
 ,s_y_errorbars=s_y_errorbars $
 ,s_linestyle=s_linestyle $
 ,s_thick=s_thick $
 ,s_sym_filled=s_sym_filled $
 ,s_sym_fill_color=s_sym_fill_color $
 ,s_font_name=s_font_name  $
 ,s_font_size=s_font_size $
 ,s_text_color=s_text_color  $
 ,b_errorbar_capsize=b_errorbar_capsize $
 ,b_transparency=b_transparency $
 ,b_antialias=b_antialias $
 ,b_color=b_color $
 ,b_errorbar_color=b_errorbar_color $
 ,b_use_default_color=b_use_default_color $
 ,b_sym_increment=b_sym_increment $
 ,b_sym_index=b_sym_index $
 ,b_sym_size=b_sym_size $
 ,b_sym_color=b_sym_color $
 ,b_sym_thick=b_sym_thick $
 ,b_y_errorbars=b_y_errorbars $
 ,b_linestyle=b_linestyle $
 ,b_thick=b_thick $
 ,b_sym_filled=b_sym_filled $
 ,b_sym_fill_color=b_sym_fill_color $
 ,b_font_name=b_font_name  $
 ,b_font_size=b_font_size $
 ,b_text_color=b_text_color  $
 ,f_errorbar_capsize=f_errorbar_capsize $
 ,f_transparency=f_transparency $
 ,f_antialias=f_antialias $
 ,f_color=f_color $
 ,f_errorbar_color=f_errorbar_color $
 ,f_use_default_color=f_use_default_color $
 ,f_sym_increment=f_sym_increment $
 ,f_sym_index=f_sym_index $
 ,f_sym_size=f_sym_size $
 ,f_sym_color=f_sym_color $
 ,f_sym_thick=f_sym_thick $
 ,f_y_errorbars=f_y_errorbars $
 ,f_linestyle=f_linestyle $
 ,f_thick=f_thick $
 ,f_sym_filled=f_sym_filled $
 ,f_sym_fill_color=f_sym_fill_color $
 ,f_font_name=f_font_name  $
 ,f_font_size=f_font_size $
 ,f_text_color=f_text_color  $
 ,ff_transparency=ff_transparency $
 ,ff_antialias=ff_antialias $
 ,ff_color=ff_color $
 ,ff_use_default_color=ff_use_default_color $
 ,ff_sym_index=ff_sym_index $
 ,ff_linestyle=ff_linestyle $
 ,ff_thick=ff_thick $
 ,sb_transparency=sb_transparency $
 ,sb_antialias=sb_antialias $
 ,sb_color=sb_color $
 ,sb_use_default_color=sb_use_default_color $
 ,sb_sym_index=sb_sym_index $
 ,sb_linestyle=sb_linestyle $
 ,sb_thick=sb_thick $
 ,_REF_EXTRA=etc

compile_opt idl2

; daveTool
if (arg_present(daveTool)) then $
  daveTool = self.daveTool

if (arg_present(s_errorbar_capsize)) then s_errorbar_capsize=Self.s_errorbar_capsize
if (arg_present(s_transparency)) then s_transparency=Self.s_transparency
if (arg_present(s_antialias)) then s_antialias=Self.s_antialias
if (arg_present(s_color)) then s_color=Self.s_color
if (arg_present(s_errorbar_color)) then s_errorbar_color=Self.s_errorbar_color
if (arg_present(s_use_default_color)) then s_use_default_color=Self.s_use_default_color 
if (arg_present(s_sym_increment)) then s_sym_increment=Self.s_sym_increment 
if (arg_present(s_sym_index)) then s_sym_index=Self.s_sym_index 
if (arg_present(s_sym_size)) then s_sym_size=Self.s_sym_size 
if (arg_present(s_sym_color)) then s_sym_color=Self.s_sym_color 
if (arg_present(s_sym_thick)) then s_sym_thick=Self.s_sym_thick 
if (arg_present(s_y_errorbars)) then s_y_errorbars=Self.s_y_errorbars
if (arg_present(s_linestyle)) then s_linestyle=Self.s_linestyle
if (arg_present(s_thick)) then s_thick=Self.s_thick
if (arg_present(s_sym_filled)) then s_sym_filled=Self.s_sym_filled
if (arg_present(s_sym_fill_color)) then s_sym_fill_color=Self.s_sym_fill_color
if (arg_present(s_font_name)) then s_font_name=Self.s_font_name
if (arg_present(s_font_size)) then s_font_size=Self.s_font_size
if (arg_present(s_text_color)) then s_text_color=Self.s_text_color
if (arg_present(b_errorbar_capsize)) then b_errorbar_capsize=Self.b_errorbar_capsize
if (arg_present(b_transparency)) then b_transparency=Self.b_transparency
if (arg_present(b_antialias)) then b_antialias=Self.b_antialias
if (arg_present(b_color)) then b_color=Self.b_color
if (arg_present(b_errorbar_color)) then b_errorbar_color=Self.b_errorbar_color
if (arg_present(b_use_default_color)) then b_use_default_color=Self.b_use_default_color
if (arg_present(b_sym_increment)) then b_sym_increment=Self.b_sym_increment
if (arg_present(b_sym_index)) then b_sym_index=Self.b_sym_index
if (arg_present(b_sym_size)) then b_sym_size=Self.b_sym_size
if (arg_present(b_sym_color)) then b_sym_color=Self.b_sym_color
if (arg_present(b_sym_thick)) then b_sym_thick=Self.b_sym_thick
if (arg_present(b_y_errorbars)) then b_y_errorbars=Self.b_y_errorbars
if (arg_present(b_linestyle)) then b_linestyle=Self.b_linestyle
if (arg_present(b_thick)) then b_thick=Self.b_thick
if (arg_present(b_sym_filled)) then b_sym_filled=Self.b_sym_filled
if (arg_present(b_sym_fill_color)) then b_sym_fill_color=Self.b_sym_fill_color
if (arg_present(b_sym_fill_color)) then f_errorbar_capsize=Self.f_errorbar_capsize
if (arg_present(b_font_name)) then b_font_name=Self.b_font_name
if (arg_present(b_font_size)) then b_font_size=Self.b_font_size
if (arg_present(b_text_color)) then b_text_color=Self.b_text_color
if (arg_present(f_transparency)) then f_transparency=Self.f_transparency
if (arg_present(f_antialias)) then f_antialias=Self.f_antialias
if (arg_present(f_color)) then f_color=Self.f_color
if (arg_present(f_errorbar_color)) then f_errorbar_color=Self.f_errorbar_color
if (arg_present(f_use_default_color)) then f_use_default_color=Self.f_use_default_color
if (arg_present(f_sym_increment)) then f_sym_increment=Self.f_sym_increment
if (arg_present(f_sym_index)) then f_sym_index=Self.f_sym_index
if (arg_present(f_sym_size)) then f_sym_size=Self.f_sym_size
if (arg_present(f_sym_color)) then f_sym_color=Self.f_sym_color
if (arg_present(f_sym_thick)) then f_sym_thick=Self.f_sym_thick
if (arg_present(f_y_errorbars)) then f_y_errorbars=Self.f_y_errorbars
if (arg_present(f_linestyle)) then f_linestyle=Self.f_linestyle
if (arg_present(f_thick)) then f_thick=Self.f_thick
if (arg_present(f_sym_filled)) then f_sym_filled=Self.f_sym_filled
if (arg_present(f_sym_fill_color)) then f_sym_fill_color=Self.f_sym_fill_color
if (arg_present(f_font_name)) then f_font_name=Self.f_font_name
if (arg_present(f_font_size)) then f_font_size=Self.f_font_size
if (arg_present(f_text_color)) then f_text_color=Self.f_text_color
if (arg_present(ff_transparency)) then ff_transparency=Self.ff_transparency
if (arg_present(ff_antialias)) then ff_antialias=Self.ff_antialias
if (arg_present(ff_color)) then ff_color=Self.ff_color
if (arg_present(ff_use_default_color)) then ff_use_default_color=Self.ff_use_default_color
if (arg_present(ff_sym_index)) then ff_sym_index=Self.ff_sym_index
if (arg_present(ff_linestyle)) then ff_linestyle=Self.ff_linestyle
if (arg_present(ff_thick)) then ff_thick=Self.ff_thick
if (arg_present(sb_transparency)) then sb_transparency=Self.sb_transparency
if (arg_present(sb_antialias)) then sb_antialias=Self.sb_antialias
if (arg_present(sb_color)) then sb_color=Self.sb_color
if (arg_present(sb_use_default_color)) then sb_use_default_color=Self.sb_use_default_color
if (arg_present(sb_sym_index)) then sb_sym_index=Self.sb_sym_index
if (arg_present(sb_linestyle)) then sb_linestyle=Self.sb_linestyle
if (arg_present(sb_thick)) then sb_thick=Self.sb_thick


; Preferences structure
if (arg_present(preferences)) then begin
   if (ptr_valid(Self.prefsPtr)) then begin
      preferences = (*self.prefsPtr)

      ; update the preferences structure
      ntags = n_tags(preferences)
      tags = tag_names(preferences)
      for i=0,ntags-1 do begin
         status = Self->GetPropertyByIdentifier(tags[i], value)
         if (status) then preferences.(i) = strjoin(strtrim(string(value),2),',')
;         if (~status) then continue
;         if (strmatch(tags[i],'*color*',/fold_case)) then begin
;            ; for colors, first convert byte vector to a string eg [0,128,0] to '0,128,0'
;            preference.(i) = strjoin(value,',')
;         endif else $
;            preferences.(i) = value
      endfor
   endif
endif

; modifiedStatus
if (arg_present(modifiedStatus)) then modifiedStatus = self.modifiedFlag
; nameTag
if (arg_present(nameTag)) then nameTag = self.nameTag
; nameTag
if (arg_present(history)) then history = (*Self.historyPtr)
; prompt
if (arg_present(prompt)) then prompt = self.prompt
; prompt Title
if (arg_present(prmptTitle)) then prmptTitle = self.promptTitle
; prompt Desc
if (arg_present(prmptDesc)) then prmptDesc = self.promptDesc
;
if (arg_present(nSamp)) then nSamp = Self.nSamp
;
if (arg_present(nBkgd)) then nBkgd = Self.nBkgd
;
if (arg_present(nFast)) then nFast = Self.nFast
;
if (arg_present(sampSelected)) then sampSelected = Self.sampSelected
;
if (arg_present(bkgdSelected)) then bkgdSelected = Self.bkgdSelected
;
if (arg_present(fastSelected)) then fastSelected = Self.fastSelected
;
if (arg_present(sampContRef)) then sampContRef = Self.sampContRef
;
if (arg_present(bkgdContRef)) then bkgdContRef = Self.bkgdContRef
;
if (arg_present(fastContRef)) then fastContRef = Self.fastContRef
;
if (arg_present(nActiveDetectors)) then nActiveDetectors = Self.nActiveDetectors
;
if (arg_present(energyUnitFlag)) then energyUnitFlag = Self.energyUnitFlag
;
if (arg_present(maskedDets)) then maskedDets = Self.maskedDets
;
if (arg_present(fastparams)) then fastparams = Self.fastparams
;
if (arg_present(fastfunc)) then fastfunc = Self.fastfunc
;
if (arg_present(funcs)) then funcs = Self.fastfuncs
;
if (arg_present(monScaleValue)) then monScaleValue = Self.monScaleValue
;
if (arg_present(monScaleFlag)) then monScaleFlag = Self.monScaleFlag
;
if (arg_present(sumSampFlag)) then sumSampFlag = Self.sumSampFlag
;
if (arg_present(sumBkgdFlag)) then sumBkgdFlag = Self.sumBkgdFlag
;
if (arg_present(sumFastFlag)) then sumFastFlag = Self.sumFastFlag
;
if (arg_present(maskFlag)) then maskFlag = Self.maskFlag
;
if (arg_present(bkgdFlag)) then bkgdFlag = Self.bkgdFlag
;
if (arg_present(fastFlag)) then fastFlag = Self.fastFlag
;
if (arg_present(dataDir)) then dataDir = Self.dataDir
;
if (arg_present(mev2wnos)) then mev2wnos = Self.mev2wnos
;
if (arg_present(oplotFastFitFlag)) then oplotFastFitFlag = Self.oplotFastFitFlag
;
if (arg_present(oplotBkgdFlag)) then oplotBkgdFlag = Self.oplotBkgdFlag
;
if (arg_present(ftpObject)) then ftpObject = Self.ftpObject

; Call base class method
if (n_elements(etc) gt 0) then $
  self->IDLitTool::GetProperty, _EXTRA=etc

end


;-------------------------------------------------------------------------------
pro FANSTool::SetProperty, nameTag=nameTag, prompt=prompt,preferences=preferences, _EXTRA=etc $
 ,nActiveDetectors=nActiveDetectors,energyUnitFlag=energyUnitFlag $
 ,maskedDets=maskedDets,fastFunc=fastFunc,monScaleValue=monScaleValue $
 ,monScaleFlag=monScaleFlag,maskFlag=maskFlag,bkgdFlag=bkgdFlag $
 ,sumSampFlag=sumSampFlag,sumBkgdFlag=sumBkgdFlag,sumFastFlag=sumFastFlag $
 ,fastFlag=fastFlag, data_directory=dataDir, daveTool=daveTool $
 ,nSamp=nSamp,nBkgd=nBkgd,nFast=nFast,fastparams=fastparams $
 ,sampSelected=sampSelected, bkgdSelected=bkgdSelected, fastSelected=fastSelected $
 ,oplotFastFitFlag=oplotFastFitFlag,oplotBkgdFlag=oplotBkgdFlag $
 ,modifiedStatus=modifiedStatus $
 ,s_errorbar_capsize=s_errorbar_capsize $
 ,s_transparency=s_transparency $
 ,s_antialias=s_antialias $
 ,s_color=s_color $
 ,s_errorbar_color=s_errorbar_color $
 ,s_use_default_color=s_use_default_color $
 ,s_sym_increment=s_sym_increment $
 ,s_sym_index=s_sym_index $
 ,s_sym_size=s_sym_size $
 ,s_sym_color=s_sym_color $
 ,s_sym_thick=s_sym_thick $
 ,s_y_errorbars=s_y_errorbars $
 ,s_linestyle=s_linestyle $
 ,s_thick=s_thick $
 ,s_sym_filled=s_sym_filled $
 ,s_sym_fill_color=s_sym_fill_color $
 ,s_font_name=s_font_name  $
 ,s_font_size=s_font_size $
 ,s_text_color=s_text_color  $
 ,b_errorbar_capsize=b_errorbar_capsize $
 ,b_transparency=b_transparency $
 ,b_antialias=b_antialias $
 ,b_color=b_color $
 ,b_errorbar_color=b_errorbar_color $
 ,b_use_default_color=b_use_default_color $
 ,b_sym_increment=b_sym_increment $
 ,b_sym_index=b_sym_index $
 ,b_sym_size=b_sym_size $
 ,b_sym_color=b_sym_color $
 ,b_sym_thick=b_sym_thick $
 ,b_y_errorbars=b_y_errorbars $
 ,b_linestyle=b_linestyle $
 ,b_thick=b_thick $
 ,b_sym_filled=b_sym_filled $
 ,b_sym_fill_color=b_sym_fill_color $
 ,b_font_name=b_font_name  $
 ,b_font_size=b_font_size $
 ,b_text_color=b_text_color  $
 ,f_errorbar_capsize=f_errorbar_capsize $
 ,f_transparency=f_transparency $
 ,f_antialias=f_antialias $
 ,f_color=f_color $
 ,f_errorbar_color=f_errorbar_color $
 ,f_use_default_color=f_use_default_color $
 ,f_sym_increment=f_sym_increment $
 ,f_sym_index=f_sym_index $
 ,f_sym_size=f_sym_size $
 ,f_sym_color=f_sym_color $
 ,f_sym_thick=f_sym_thick $
 ,f_y_errorbars=f_y_errorbars $
 ,f_linestyle=f_linestyle $
 ,f_thick=f_thick $
 ,f_sym_filled=f_sym_filled $
 ,f_sym_fill_color=f_sym_fill_color $
 ,f_font_name=f_font_name  $
 ,f_font_size=f_font_size $
 ,f_text_color=f_text_color  $
 ,ff_transparency=ff_transparency $
 ,ff_antialias=ff_antialias $
 ,ff_color=ff_color $
 ,ff_use_default_color=ff_use_default_color $
 ,ff_sym_index=ff_sym_index $
 ,ff_linestyle=ff_linestyle $
 ,ff_thick=ff_thick $
 ,sb_transparency=sb_transparency $
 ,sb_antialias=sb_antialias $
 ,sb_color=sb_color $
 ,sb_use_default_color=sb_use_default_color $
 ,sb_sym_index=sb_sym_index $
 ,sb_linestyle=sb_linestyle $
 ,sb_thick=sb_thick

compile_opt idl2

if n_elements(s_errorbar_capsize) then Self.s_errorbar_capsize=s_errorbar_capsize
if n_elements(s_transparency) then Self.s_transparency=s_transparency
if n_elements(s_antialias) then Self.s_antialias=s_antialias
if n_elements(s_color) then Self.s_color=s_color
if n_elements(s_errorbar_color) then Self.s_errorbar_color=s_errorbar_color
if n_elements(s_use_default_color) then Self.s_use_default_color=s_use_default_color 
if n_elements(s_sym_increment) then Self.s_sym_increment=s_sym_increment 
if n_elements(s_sym_index) then Self.s_sym_index=s_sym_index 
if n_elements(s_sym_size) then Self.s_sym_size=s_sym_size 
if n_elements(s_sym_color) then Self.s_sym_color=s_sym_color 
if n_elements(s_sym_thick) then Self.s_sym_thick=s_sym_thick 
if n_elements(s_y_errorbars) then Self.s_y_errorbars=s_y_errorbars
if n_elements(s_linestyle) then Self.s_linestyle=s_linestyle
if n_elements(s_thick) then Self.s_thick=s_thick
if n_elements(s_sym_filled) then Self.s_sym_filled=s_sym_filled
if n_elements(s_sym_fill_color) then Self.s_sym_fill_color=s_sym_fill_color
if n_elements(s_font_name) then Self.s_font_name=s_font_name
if n_elements(s_font_size) then Self.s_font_size=s_font_size
if n_elements(s_text_color) then Self.s_text_color=s_text_color
if n_elements(b_errorbar_capsize) then Self.b_errorbar_capsize=b_errorbar_capsize
if n_elements(b_transparency) then Self.b_transparency=b_transparency
if n_elements(b_antialias) then Self.b_antialias=b_antialias
if n_elements(b_color) then Self.b_color=b_color
if n_elements(b_use_default_color) then Self.b_use_default_color=b_use_default_color
if n_elements(b_errorbar_color) then Self.b_errorbar_color=b_errorbar_color
if n_elements(b_sym_increment) then Self.b_sym_increment=b_sym_increment
if n_elements(b_sym_index) then Self.b_sym_index=b_sym_index
if n_elements(b_sym_size) then Self.b_sym_size=b_sym_size
if n_elements(b_sym_color) then Self.b_sym_color=b_sym_color
if n_elements(b_sym_thick) then Self.b_sym_thick=b_sym_thick
if n_elements(b_y_errorbars) then Self.b_y_errorbars=b_y_errorbars
if n_elements(b_linestyle) then Self.b_linestyle=b_linestyle
if n_elements(b_thick) then Self.b_thick=b_thick
if n_elements(b_sym_filled) then Self.b_sym_filled=b_sym_filled
if n_elements(b_sym_fill_color) then Self.b_sym_fill_color=b_sym_fill_color
if n_elements(b_font_name) then Self.b_font_name=b_font_name
if n_elements(b_font_size) then Self.b_font_size=b_font_size
if n_elements(b_text_color) then Self.b_text_color=b_text_color
if n_elements(f_errorbar_capsize) then Self.f_errorbar_capsize=f_errorbar_capsize
if n_elements(f_transparency) then Self.f_transparency=f_transparency
if n_elements(f_antialias) then Self.f_antialias=f_antialias
if n_elements(f_color) then Self.f_color=f_color
if n_elements(f_errorbar_color) then Self.f_errorbar_color=f_errorbar_color
if n_elements(f_use_default_color) then Self.f_use_default_color=f_use_default_color
if n_elements(f_sym_increment) then Self.f_sym_increment=f_sym_increment
if n_elements(f_sym_index) then Self.f_sym_index=f_sym_index
if n_elements(f_sym_size) then Self.f_sym_size=f_sym_size
if n_elements(f_sym_color) then Self.f_sym_color=f_sym_color
if n_elements(f_sym_thick) then Self.f_sym_thick=f_sym_thick
if n_elements(f_y_errorbars) then Self.f_y_errorbars=f_y_errorbars
if n_elements(f_linestyle) then Self.f_linestyle=f_linestyle
if n_elements(f_thick) then Self.f_thick=f_thick
if n_elements(f_sym_filled) then Self.f_sym_filled=f_sym_filled
if n_elements(f_sym_fill_color) then Self.f_sym_fill_color=f_sym_fill_color
if n_elements(f_font_name) then Self.f_font_name=f_font_name
if n_elements(f_font_size) then Self.f_font_size=f_font_size
if n_elements(f_text_color) then Self.f_text_color=f_text_color
if n_elements(ff_transparency) then Self.ff_transparency=ff_transparency
if n_elements(ff_antialias) then Self.ff_antialias=ff_antialias
if n_elements(ff_color) then Self.ff_color=ff_color
if n_elements(ff_use_default_color) then Self.ff_use_default_color=ff_use_default_color
if n_elements(ff_sym_index) then Self.ff_sym_index=ff_sym_index
if n_elements(ff_linestyle) then Self.ff_linestyle=ff_linestyle
if n_elements(ff_thick) then Self.ff_thick=ff_thick
if n_elements(sb_transparency) then Self.sb_transparency=sb_transparency
if n_elements(sb_antialias) then Self.sb_antialias=sb_antialias
if n_elements(sb_color) then Self.sb_color=sb_color
if n_elements(sb_use_default_color) then Self.sb_use_default_color=sb_use_default_color
if n_elements(sb_sym_index) then Self.sb_sym_index=sb_sym_index
if n_elements(sb_linestyle) then Self.sb_linestyle=sb_linestyle
if n_elements(sb_thick) then Self.sb_thick=sb_thick

; DAVETool
if (n_elements(daveTool) && obj_valid(daveTool) && obj_isa(daveTool,'DAVETool')) then $
  Self.daveTool = daveTool

; Preferences
if (n_elements(preferences)) then begin
   if (strcmp(size(preferences,/tname),'STRUCT')) then begin
      ntags = n_tags(preferences)
      tags = tag_names(preferences)
      index = indgen(ntags)
      
      ; handle color tags - convert string to byte array, eg '0,128,255' to [0,128,255]
      spIndex = where(strmatch(tags,'*color*',/fold) eq 1, spCnt, complement=index, ncomplement=theRest)
      if (spCnt gt 0) then begin
         for i=0,spCnt-1 do begin
            strColor = preferences.(spIndex[i])
            bytColor = fix(strsplit(strColor,',',/extract))
            Self->SetPropertyByIdentifier, tags[spIndex[i]], bytColor
         endfor
      endif
      
      ; all other tags
      for i=0,theRest-1 do Self->SetPropertyByIdentifier, tags[index[i]], preferences.(index[i])

      ; Need further to assign userdef attribute for user-defined properties
      ; in order for property sheet to reflect updated value
      uProps = ['MASKEDDETS','FASTPARAMS']
      n = n_elements(uProps)
      for i = 0,n-1 do begin
         index = where(strcmp(tags,uProps[i]), found)
         if (found) then $
            Self->SetPropertyAttribute, uProps[i],userdef=preferences.(index)
      endfor
      
      ; Finally, store the preferences structure
      if (ptr_valid(Self.prefsPtr)) then (*Self.prefsPtr) = preferences $
      else Self.prefsPtr = ptr_new(preferences)
   endif
endif

; modifiedStatus
if (n_elements(modifiedStatus)) then Self.modifiedFlag = (fix(modifiedStatus) gt 0)? 1 : 0
; nameTag
if (n_elements(nameTag)) then Self.nameTag = strtrim(nameTag,2)
; prompt
if (n_elements(prompt)) then begin
   Self.prompt = prompt
   if (prompt eq 0) then Self._bDirty = 0
endif
;
if (n_elements(nActiveDetectors)) then Self.nActiveDetectors = nActiveDetectors
;
if (n_elements(energyUnitFlag)) then Self.energyUnitFlag = energyUnitFlag
;
if (n_elements(maskedDets)) then Self.maskedDets = maskedDets
;
if (n_elements(fastfunc)) then Self.fastfunc = fastfunc
;
if (n_elements(fastparams)) then Self.fastparams = fastparams
;
if (n_elements(monScaleValue)) then Self.monScaleValue = (monScaleValue eq 0.0)? 1.0 : monScaleValue
;
if (n_elements(monScaleFlag)) then Self.monScaleFlag = monScaleFlag
;
if (n_elements(sumSampFlag)) then Self.sumSampFlag = sumSampFlag
;
if (n_elements(sumBkgdFlag)) then Self.sumBkgdFlag = sumBkgdFlag
;
if (n_elements(sumFastFlag)) then Self.sumFastFlag = sumFastFlag
;
if (n_elements(maskFlag)) then Self.maskFlag = maskFlag
;
if (n_elements(bkgdFlag)) then Self.bkgdFlag = bkgdFlag
;
if (n_elements(fastFlag)) then Self.fastFlag = fastFlag
;
if (n_elements(dataDir)) then Self.dataDir = dataDir
;
if (n_elements(nSamp)) then Self.nSamp = nSamp
;
if (n_elements(nBkgd)) then Self.nBkgd = nBkgd
;
if (n_elements(nFast)) then Self.nFast = nFast
;
if (n_elements(sampSelected)) then Self.sampSelected = sampSelected
;
if (n_elements(bkgdSelected)) then Self.bkgdSelected = bkgdSelected
;
if (n_elements(fastSelected)) then Self.fastSelected = fastSelected
;
if (n_elements(oplotFastFitFlag)) then Self.oplotFastFitFlag = oplotFastFitFlag
;
if (n_elements(oplotBkgdFlag)) then Self.oplotBkgdFlag = oplotBkgdFlag


; Call base class method
if (n_elements(etc) gt 0) then $
  self->IDLitTool::SetProperty, _EXTRA=etc

end


;===============================================================================
; FANSTool::deleteAllDatasets
;
; PURPOSE:
;   Delete all datasets currently loaded
;
; PARAMETERS:
;
; RETURN VALUE:
;
function FANSTool::deleteAllDatasets, _EXTRA=etc
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

wDM = (*sPtr).wDM
widget_control, wDM, get_uvalue=sPtrDM

cw_fansDataManager_DeleteData, sPtrDM, wDM, /delall, _EXTRA=etc

return, 1
end


;===============================================================================
; FANSTool::subtractBkgd
;
; PURPOSE:
;   Subtract bkgd data
;
; PARAMETERS:
;
; RETURN VALUE:
;
pro FANSTool::subtractBkgd
compile_opt idl2


end


;===============================================================================
; FANSTool::mergeData
;
; PURPOSE:
;   Merge multiple datasets into one, averaging or summing as needed. The composite dataset
;   is rescaled to the monitor value of the lowest energy dataset.
;
; PARAMETERS:
;   dataType [in] - 'samp' | 'bkgd' | 'fast'
;
; RETURN VALUE:
;
pro FANSTool::mergeData, dataType
compile_opt idl2

; get appropriate data; samp or bkgd
switch strlowcase(dataType) of

    'samp': begin
        nd = Self.nSamp
        dataCont = Self.sampContRef
        typeName = 'Sample '
;        id = 0
        break
    end

    'bkgd': begin
        nd = Self.nBkgd
        dataCont = Self.bkgdContRef
        typeName = 'Background '
;        id = 1
        break
    end

    'fast': begin
        nd = Self.nFast
        dataCont = Self.fastContRef
        typeName = 'Fast bkgd '
;        id = 2
        break
    end

endswitch

; Get the min and max energy values of the dataset
minima = fltarr(nd) - 1
maxima = minima
dataPtrArr = ptrarr(nd)
fnames = strarr(nd)
description = ''

for i = 0,nd-1 do begin
    oItem = dataCont->Get(position=i)
    void = oItem->GetMetaData('DATAPTRREF',dPtr)
    if (~ptr_valid(dPtr)) then return
    dataPtrArr[i] = dPtr
    oItem->GetProperty, name=name
    fnames[i] = name
    
    mine = min( (*(*dPtr).energyPtr), max=maxe)
    minima[i] = mine
    maxima[i] = maxe
    
    ; retrieve description for first filename and use later for the sum object
    if (i eq 0) then oItem->GetProperty, description=description
endfor 
ord = sort(minima)              ; find order of increasing min energy

; make a copy of the first dataset in the list. Note that dataPtrArr
; is an array of ptrs (each pointing to a dataset).
sumPtr = Self->initDataPtr(dataPtrArr[ord[0]])
monConstant = (*sumPtr).monitor * (*sumPtr).prefactor ; record that for first dataset
omega1 = (*(*sumPtr).energyPtr)
temp1 =  (*(*sumPtr).tempPtr)
(*(*sumPtr).historyPtr) = [typeName+'data is a sum of the following file(s):',(*(*sumPtr).historyPtr)]



; Monitor requires special attention! The monitor count is made of a
; product of the mon cnt and the prefactor and it is not possible to
; keep track of these separately when files are averaged. So set monitor
; count to mon*pref and pref to 1. Subsequent data are scaled to the
; mon*pref value of the first dataset.
;(*sumPtr).monitor = (*sumPtr).monitor * (*sumPtr).prefactor
;(*sumPtr).prefactor = 1

; loop through and sum all the datasets
; only the time, energy, detCnts, totCnts and totErrs components need
; to be averaged! - this is done in the inner loop
min1 = minima[ord[0]]  &  max1 = maxima[ord[0]]
for i = 1,nd-1 do begin         
    j = ord[i]

    ;; if dataset barely overlap or not at all then skip the current
    ;; dataset and advise user to select the merge option instead
    min2 = minima[j]  &  max2 = maxima[j]
    ;; get the current dataset to be dealt with
    curPtr = dataPtrArr[j]
    omega2 = (*(*curPtr).energyPtr)
    temp2 = (*(*curPtr).tempPtr)
    size1 = n_elements(omega1)
    size2 = n_elements(omega2)
    monitor1 = (*(*sumPtr).monPtr)
    monitor2 = (*(*curPtr).monPtr)
    
    ;; Examine various exclusive scenarios

    ;---------------------------------------------------------------
    ;; Scenario 1 - min2 > min1 ==> there is a 'no overlap' region in
    ;; dataset1. Have to copy this data into a buffer
    ibeg1 = 0 & iend1=0 & ibeg2=0 & iend2=0
    if (min1 ne min2) then begin
        res = where(omega1 ge min2) ; ge is deliberate
        iend1 = (res[0] le 0)? size1-1 : res[0] - 1
        if (iend1 ge 0) then begin
            omega = omega1[ibeg1:iend1]
            times = (*(*sumPtr).timePtr)[ibeg1:iend1]
            temperature = (*(*sumPtr).tempPtr)[ibeg1:iend1]
            totCnts = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
            detCnts = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
            monitor = monitor1[ibeg1:iend1]
            ibeg1=iend1+1
            iend1 = iend1 + 1
        endif
    endif
    
    ;----------------------------------------------------------------------
    ;; Scenario 2 - the overlap region;
    if (max1 lt min2) then goto, scenario3 ; inadequate overlap
    res = where(omega1 gt max2) ; the 'gt' is deliberate!
    iend1 = (res[0] lt 0)? size1-1 : res[0]-1 ; upper bound of overlap for ds1
    if (iend1 lt ibeg1) then iend1 = ibeg1
    ibeg2 = 0                   ; lower bound of overlap for ds2
    res = where(omega2 gt max1) 
    iend2 = (res[0] lt 0)? size2-1 : res[0]-1 ; upper bound of overlap for ds2
    if (iend2 lt ibeg2) then iend2 = ibeg2
    if ((where(omega1[ibeg1:iend1] ne omega2[ibeg2:iend2]))[0] ge 0) then begin 
        ;; ==> in overlap region, omega1 != omega2
        ;; hence need to merge (interleave) the data

        ;; determine which of dataset has a smaller min omega and use
        ;; that ds as base for inserting or appending the other ds 
        if (omega1[ibeg1] le omega2[ibeg2]) then begin
            om = omega1[ibeg1:iend1]
            mn = monitor1[ibeg1:iend1]
            temp = temp1[ibeg1:iend1]
            t = (*(*sumPtr).timePtr)[ibeg1:iend1]
            x = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
            xd = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
            om2 = omega2[ibeg2:iend2]
            mn2 = monitor2[ibeg2:iend2]
            tmp2 = temp2[ibeg2:iend2]
            t2 = (*(*curPtr).timePtr)[ibeg2:iend2] ;* mon_scale
            x2 = (*(*curPtr).totCntsPtr)[ibeg2:iend2] ;* mon_scale
            xd2 = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2] ;* mon_scale
            ibeg = 0
            iend = iend2 - ibeg2
            len = n_elements(om) - 1
            first_dset = 1 
        endif else begin
            om = omega2[ibeg2:iend2]
            mn = monitor2[ibeg2:iend2]
            temp = temp2[ibeg2:iend2]
            t = (*(*curPtr).timePtr)[ibeg2:iend2] ;* mon_scale
            x = (*(*curPtr).totCntsPtr)[ibeg2:iend2] ;* mon_scale
            xd = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2] ;* mon_scale
            om2 = omega1[ibeg1:iend1]
            mn2 = monitor1[ibeg1:iend1]
            tmp2 = temp1[ibeg1:iend1]
            t2 = (*(*sumPtr).timePtr)[ibeg1:iend1]
            x2 = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
            xd2 = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
            ibeg = 0
            iend = iend1 - ibeg1
            len = n_elements(om) - 1 ;iend2 - ibeg2
            first_dset = 2
        endelse
        for k=ibeg,iend do begin
            ;; loop thro and locate pos of om2[k] in om

            res = where(om ge om2[k])
            if ( res[0] ge 0 ) then begin 
                if (om2[k] eq om[res[0]]) then begin
                    ;; ==> energy value of om2[k] matches with one in om
                    ;; therefore perform an average.
                    kk = res[0]
                    x3 = x[kk]
                    x4 = x2[k]
                    t3 = t[kk]
                    t4 = t2[k]
                    xd3 = xd[*,kk]
                    xd4 = xd2[*,k]

                    ;; 1) Average total counts
                    x[kk] = x[kk] + x2[k]
                    mn[kk] = mn[kk] + mn2[k]
                    ;; 2) Average counting times
                    t[kk] = t[kk] + t2[k]
                    ;; 3) Average detector counts
                    xd[*,kk] = xd[*,kk] + xd2[*,k]



                    continue    ; move on to next for loop iteration
                endif
            endif  
            sz = size(xd)
            nw = sz[1]  
            ns = (sz[0] eq 1)? 1 : sz[2]
            tmpxd = fltarr(nw,ns+1)
            tmpwd = fltarr(nw,ns+1)
            if (res[0] lt 0) then begin
                ;; ==> need to _append_ ds2 values at end
                om = [om,om2[k]]
                temp = [temp,tmp2[k]]
                t = [t,t2[k]]
                x = [x,x2[k]]
                mn = [mn,mn2[k]]
                tmpxd[*,0:ns-1] = xd
                tmpxd[*,ns] = xd2[*,k]
                xd = tmpxd
                wd = tmpwd
            endif else begin
                ;; ==> need to _insert_ ds2 vals before res[0] index
                ;; in ds1
                kk = res[0]
                om = [om[0:kk-1],om2[k],om[kk:len]]
                temp = [temp[0:kk-1],tmp2[k],temp[kk:len]]
                t = [t[0:kk-1],t2[k],t[kk:len]]
                x = [x[0:kk-1],x2[k],x[kk:len]]
                mn = [mn[0:kk-1],mn2[k],mn[kk:len]]
                tmpxd[*,0:kk-1] = xd[*,0:kk-1]
                tmpxd[*,kk] = xd2[*,k]
                tmpxd[*,kk+1:ns] = xd[*,kk:len]
                xd = tmpxd
            endelse
            len = n_elements(om) - 1 ; update len to reflect expanded buffers
        endfor
        iend1 = iend1 + 1
        iend2 = iend2 + 1
    endif else begin 
        ;; ==> in overlap region, omega1 == omega2
        ;; hence do the averaging in the overlap region
        om = omega1[ibeg1:iend1]
        mn = monitor1[ibeg1:iend1]
        mn2 = monitor2[ibeg2:iend2]
        temp = temp1[ibeg1:iend1]

        ;; 1) Average total counts
        buf1 = (*(*sumPtr).totCntsPtr)[ibeg1:iend1]
        buf2 = (*(*curPtr).totCntsPtr)[ibeg2:iend2]
        x =  buf1 + buf2
        mn = mn + mn2
        
        ;; 2) Average counting times
        buf1 = (*(*sumPtr).timePtr)[ibeg1:iend1]
        buf2 = (*(*curPtr).timePtr)[ibeg2:iend2]; * mon_scale
        t = buf1 + buf2

        ;; 3) Average detector counts
        buf1 = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1]
        buf2 = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2]; * mon_scale
        xd = buf1 + buf2

        iend2 = iend2 + 1
        iend1 = iend1 + 1
    endelse

    ;; include the om,t,x,w,xd,wd calculated using either interleaving or
    ;; averaging into the sum buffers
    if (ibeg1 eq 0) then begin
        omega = om
        monitor = mn
        temperature = temp
        times = t
        totCnts = x
        detCnts = xd
    endif else begin
        ;; sum buffers already contain data from Scenario 1
        omega = [omega,om]
        monitor = [monitor,mn]
        temperature = [temperature,temp]
        times = [times,t]
        totCnts = [totCnts,x]
        if ((iend1 eq ibeg1) and (n_elements(om) eq 1)) then begin  
            ;; ie only a one data point overlap ==> xd,wd are 1D not 2D!
            sz = size(detCnts)
            nw = sz[1]
            ns = (sz[0] eq 1)? 1 : sz[2]
            tmpdetCnts = fltarr(nw,ns+1)
            tmpdetErrs = fltarr(nw,ns+1)
            tmpdetCnts[*,0:ns-1] = detCnts
            tmpdetCnts[*,ns] = xd
            detCnts = tmpdetCnts
        endif else begin
            sz1 = size(detCnts)
            sz2 = size(xd)
            nw = sz1[1]
            ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
            ns = (sz2[0] eq 1)? 1 : sz2[2]
            ns = ns + ns1
            tmpdetCnts = fltarr(nw,ns)
            tmpdetCnts[*,0:ns1-1] = detCnts
            tmpdetCnts[*,ns1:ns-1] = xd
            detCnts = tmpdetCnts
        endelse
    endelse
 

    ;----------------------------------------------------------------------
    ;; Scenario 3 - beyond the overlap region - include any remaining data
    ;; Only ds1 OR ds2 could still contain unaccounted data; NOT
    ;; BOTH. However, both could be all used up already
    scenario3:
;    if (((iend1 eq 0) and (iend1 lt size1-1)) or (iend1 lt size1-1))then begin ; pick up the remaining data in ds1
    if ((iend1 eq 0) or (iend1 le size1-1)) then begin ; pick up the remaining data in ds1
        res = where(omega1[iend1] le omega, count)
        ibeg1 = (count lt 1)? iend1 : iend1 + 1
        iend1 = size1 - 1
        omega = [omega,omega1[ibeg1:iend1]]
        monitor = [monitor,monitor1[ibeg1:iend1]]
        temperature = [temperature,temp1[ibeg1:iend1]]
        times = [times,(*(*sumPtr).timePtr)[ibeg1:iend1]]
        totCnts = [totCnts,(*(*sumPtr).totCntsPtr)[ibeg1:iend1]]
        sz1 = size(detCnts)
        sz2 = size((*(*sumPtr).detCntsPtr)[*,ibeg1:iend1])
        nw = sz1[1]
        ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
        ns = (sz2[0] eq 1)? 1 : sz2[2]
        ns = ns + ns1
        tmpdetCnts = fltarr(nw,ns)
        tmpdetCnts[*,0:ns1-1] = detCnts
        tmpdetCnts[*,ns1:ns-1] = (*(*sumPtr).detCntsPtr)[*,ibeg1:iend1] 
        detCnts = tmpdetCnts
        iend1 = iend1 + 1
    endif else if ((iend2 eq 0) or (iend2 le size2-1)) then begin ; pick up the remaining data in ds2
        res = where(omega2[iend2] le omega, count)
        ibeg2 = (count lt 1)? iend2 : iend2 + 1
        iend2 = size2 - 1
        omega = [omega,omega2[ibeg2:iend2]]
        monitor = [monitor,monitor2[ibeg2:iend2]]
        temperature = [temperature,temp2[ibeg2:iend2]]
        times = [times,(*(*curPtr).timePtr)[ibeg2:iend2]] 
        totCnts = [totCnts,(*(*curPtr).totCntsPtr)[ibeg2:iend2]]
        sz1 = size(detCnts)
        sz2 = size((*(*curPtr).detCntsPtr)[*,ibeg2:iend2])
        nw = sz1[1]
        ns1 = (sz1[0] eq 1)? 1 : sz1[2] 
        ns = (sz2[0] eq 1)? 1 : sz2[2]
        ns = ns + ns1
        tmpdetCnts = fltarr(nw,ns)
        tmpdetCnts[*,0:ns1-1] = detCnts
        tmpdetCnts[*,ns1:ns-1] = (*(*curPtr).detCntsPtr)[*,ibeg2:iend2]
        detCnts = tmpdetCnts
        iend2 = iend2 + 1
    endif 

    ;----------------------------------------------------------
    ;; Phew! Now, should have reached the end of both datasets
    ;; If not, then there is something wrong so inform user but do nothing
    if ( (iend1 le size1-1) or (iend2 le size2-1) ) then begin
        message = ['Processing data from: ', $
                   fnames[j], $
                   'It apeears not all the data was accounted for!']
        void = dialog_message(message,dialog_parent=(*sPtr).tlbID,/information)
    endif


    ;;------------------------------------------------------------
    ;; replace sum data with calculations
    (*(*sumPtr).energyPtr) = omega
    (*(*sumPtr).timePtr) = times
    (*(*sumPtr).tempPtr) = temperature
    (*(*sumPtr).totCntsPtr) = totCnts
    (*(*sumPtr).detCntsPtr) = detCnts
    (*(*sumPtr).monPtr) = monitor
    (*sumPtr).nspec = n_elements(omega)
  
    ;; update historyPtr
    (*(*sumPtr).historyPtr) = [(*(*sumPtr).historyPtr),(*(*curPtr).historyPtr)]

    ;; update certain quantities before the next loop
    omega1 = (*(*sumPtr).energyPtr)
    temp1 = (*(*sumPtr).tempPtr)
    min1 = min(omega1, max=max1)
endfor

; normalize spectra to a constant monitor
ndet = (size((*(*sumPtr).detCntsPtr)))[1]
factor = monConstant / (*(*sumPtr).monPtr)
(*(*sumPtr).totErrsPtr) = sqrt((*(*sumPtr).totCntsPtr)) * factor
(*(*sumPtr).totCntsPtr) = (*(*sumPtr).totCntsPtr) * factor
;(*(*sumPtr).detErrsPtr) = sqrt((*(*sumPtr).detCntsPtr)) * ((fltarr(ndet)+1)#factor)
(*(*sumPtr).detCntsPtr) = (*(*sumPtr).detCntsPtr) * ((fltarr(ndet)+1)#factor)
(*(*sumPtr).timePtr) = (*(*sumPtr).timePtr) * factor
(*(*sumPtr).monPtr) = (*(*sumPtr).monPtr) * factor
switch dataType of 
    'samp': begin               ; add the sumPtr to sample dataset
        basename = 'SampleSum'
        oItem = obj_new('IDLitData',basename,name=basename,type='samp',description=description)
        oItem->AddMetaData,'DATAPTRREF',sumPtr
        ;Self->AddByIdentifier, 'FANS Sample Data Manager', oItem
        Self.sampContRef->Add, oItem        
        break
    end
    'bkgd': begin               ; add the sumPtr to bkgd dataset
        basename = 'BkgdSum'
        oItem = obj_new('IDLitData',basename,name=basename,type='bkgd',description=description)
        oItem->AddMetaData,'DATAPTRREF',sumPtr
        Self.bkgdContRef->Add, oItem
        break
    end 
    'fast': begin               ; add the sumPtr to fast bkgd dataset
        basename = 'FastSum'
        oItem = obj_new('IDLitData',basename,name=basename,type='fast',description=description)
        oItem->AddMetaData,'DATAPTRREF',sumPtr
        Self.fastContRef->Add, oItem
        break
    end 
endswitch 



end
;-------------------------------------------------------------------------------





;===============================================================================
; FANSTool::initDataPtr
;
; PURPOSE:
;   Define a suitable data structure that can hold contents of FANS
;   data file.
;
; PARAMETERS:
;   rvaluePtr [in] - Pointer to an existed data structure. If
;                    specified, then initialise the new data structure
;                    with it's contents.
;
; RETURN VALUE:
;   a valid data structure for storing FANS data.
;
function FANSTool::initDataPtr, rvaluePtr

; If there is an rvaluePtr, 
fdStr = {filename:'', $
         date:'', $
         scan:'T', $
         monitor:0L, $   
         prefactor:0L, $
         base:'TIME', $
         type:'RAW', $
         title:'', $
         ytitle:'', $
         xtitle:'', $
         xunits:'', $
         yunits:'', $
         collimation:'00 00 00 00', $
         mosaic:'0  0  0', $
         orientation:'0 0 0 0.000 0 0 0', $
         latticeA:0.0, $
         latticeB:0.0, $
         latticeC:0.0, $
         latticeAl:0.0, $
         latticeBe:0.0, $
         latticeGa:0.0, $
         eStart:0.0, $
         eFinal:0.0, $
         eDelta:0.0, $
         monDSpace:0.0, $
         anaDSpace:0.0, $
         tStart:0.0, $
         tInc:0.0, $
         qhCenter:0.0, $
         qkCenter:0.0, $
         qlCenter:0.0, $
         qhDelta:0.0, $
         qkDelta:0.0, $
         qlDelta:0.0, $
         hfield:0.0, $          ; header info ends here
         tflag:0B, $
         ndets:128, $           ; however, it seems only 50 are currently active - see below
         nspec:0, $
         dataVersion:0, $   ; current data version number in use <== ADDED in dataVersion = 2
;******** Must add new non-pointer members immediately ABOVE this comment *****************
         monPtr:ptr_new(), $ ;monitor array
         qxPtr:ptr_new(), $
         qyPtr:ptr_new(), $
         qzPtr:ptr_new(), $
         tempPtr:ptr_new(), $
         timePtr:ptr_new(), $
         energyPtr:ptr_new(), $
         detCntsPtr:ptr_new(), $
         totCntsPtr:ptr_new(), $
         ;detErrsPtr:ptr_new(), $ ; error in detCnts
         historyPtr:ptr_new(), $ ; ptr to a string array containing history of file 
         totErrsPtr:ptr_new() $ ; error in totCnts <== ADDED in dataVersion = 2
;******** Must add new pointer members immediately above this comment *****************
        }


if ( ptr_valid(rvaluePtr) gt 0) then begin ; copy values from rvaluePtr
    
    rvalue = *rvaluePtr
    j = where(tag_names(rvalue) eq 'MONPTR', found)
    j = j[0]

    for i = 0,j-1 do $           ; copy the ordinary structure members
      fdStr.(i) = rvalue.(i)

    for i = j,n_tags(rvalue)-1 do $            ; copy the pointer structure members
      fdStr.(i) = ptr_new( *(rvalue.(i)) ) ; b/c ptrs in fdStr are initia. to NULL
endif

lvaluePtr = ptr_new(fdStr)

return, lvaluePtr               

end
;-------------------------------------------------------------------------------



;===============================================================================
; FANSTool::loadSamp
;
; PURPOSE:
;   Specify and load sample data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::GetFileList, title, count, fromFTP=fromFTP

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

if (keyword_set(fromFTP)) then begin
  Self->GetProperty, ftpObject=oFTP
  filter = ['*.bt4','*.*']
  files = Dialog_NCNRpublicData(oFTP,title=title,filter=filter,group_leader=wTLB,count=count)
endif else begin
  filter = '*.bt4'
  Self->GetProperty, data_directory=dataDir
  if (n_elements(title) eq 0) then title='Specify files to load'
  
  files = dialog_pickfile(title=title,/multiple_files,/must_exist, dialog_parent=wTLB $
                         ,filter=filter,path=dataDir, get_path=newDataDir)
  
  if (files[0] eq '') then begin
     count = 0
     return, ''
  endif
  
  count = n_elements(files)
  Self->SetProperty, data_directory=newDataDir
endelse

return, files

end
;-------------------------------------------------------------------------------



;===============================================================================
; FANSTool::refreshPropertySheet
;
; PURPOSE:
;   Refresh the propertysheet UI
;
; PARAMETERS:
;
; KEYWORDS:
;
pro FANSTool::refreshPropertySheet

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

widget_control, (*sPtr).wPS, /refresh_property

end
;-------------------------------------------------------------------------------


;===============================================================================
; FANSTool::loadFastFtp
;
; PURPOSE:
;   Specify and load fast bkgd data file(s) from NCNR ftp server
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::loadFastFtp

return, Self->LoadFast(/fromFtp)

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; FANSTool::loadFast
;
; PURPOSE:
;   Select and load Fast Bkgd data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   dataObject - specifies the data object whose data is to be reloaded
;   
;   reload  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   fromFTP - set if the source of the datafile to be loaded is the FTP server
;-
function FANSTool::loadFast, reload=reload, dataObject=oItem, fromFTP=fromFTP

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::loadFast: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

reloading = keyword_set(reload)
if (reloading) then begin
   if (~obj_valid(oItem)) then return, 0
   oItem->GetProperty, description=files
   count = 1
   void = oItem->GetMetaData('FROMFTP',fromFtp)
endif else begin
   ; Let user specify files to load
   title = 'Select Fast Background Files to Load'
   files = Self->GetFileList(title,count,fromFTP=fromFTP)
   if (count eq 0) then return, 0
endelse

; switch on the busy symbol for cursor
widget_control, /HOURGLASS

; Read the contents of the data files and make entries in the 'Fast Bkgd' tree of the data manager
Self->GetProperty, sumFastFlag=sumFlag, fastContRef=oCont, nFast=nData
for i=0, count-1 do begin
   filename = files[i]
   if (~Self->readData(filename, dataPtr, fromFTP=fromFTP)) then continue

   basename = file_basename(filename,'.bt4',/fold_case)
   
   if (~reloading) then begin
      ;; create new object and attach dataPtr
      oItem = obj_new('IDLitData',basename,name=basename,description=filename,type='fast')   
      oItem->AddMetaData,'DATAPTRREF',dataPtr
      oItem->AddMetaData,'FROMFTP',keyword_set(fromFTP)
      
      ; iTools does not handle '_[0-9]' characters in identifiers well eg _0, _1, etc
      ; replace all '_' with '-'
      oItem->GetProperty, identifier=idd
      hasChanged = 0
      while ((pos = stregex(idd,'_[0-9]')) ne -1) do begin
         strput, idd,'-',pos
         hasChanged = 1
      endwhile
      if (strmatch(idd,'[0-9]*')) then begin   ; if first char is a number
         idd = 'id'+idd
         hasChanged = 1
      endif
      if (hasChanged) then oItem->SetProperty,identifier=idd 

      ; Add to the data manager
      Self->AddByIdentifier, 'FANS Fast Data Manager', oItem
   endif else begin
      ;; simply replace previous dataPtr with updated one
      void = oItem->GetMetaData('DATAPTRREF',tempPtr)
      if (ptr_valid(tempPtr)) then heap_free, tempPtr
      oItem->AddMetaData,'DATAPTRREF',dataPtr
   endelse
   
   if (i eq 0 && sumFlag && (nData gt 0)) then begin
      ;; if data previously read and summed, then delete the sum
      if (oCont->Count() eq nData+1) then begin
         oSum = oCont->Get(position=nData)
         if (obj_valid(oSum)) then begin
            void = oSum->GetMetaData('DATAPTRREF',sumDataPtr) ; the real data is in this dataPtr
            if (ptr_valid(sumDataPtr)) then heap_free, sumDataPtr   ; so must free it!

            oCont->Remove, oSum
            obj_destroy, oSum   ; oSum is not part of any hierarchy so no deed to use DM service to destroy it!
         endif
      endif
   endif
   
   if (~reloading) then begin
      oCont->Add, oItem
      Self.nFast++
   endif

   dataPtr = 0   ; make dataPtr invalid before it is used again when looping
endfor

if ((Self.nFast gt 1) && sumFlag) then Self->MergeData, 'fast'

Self->SetPropertyAttribute, 'SUMFASTFLAG', sensitive=(Self.nFast gt 0)
Self->SetPropertyAttribute, 'FASTFLAG', sensitive=(Self.nFast gt 0)

Self->refreshPropertySheet   ; force prop sheet to update

Self.modifiedFlag = 1
return, 1
end
;-------------------------------------------------------------------------------


;===============================================================================
; FANSTool::loadBkgdFtp
;
; PURPOSE:
;   Specify and load bkgd data file(s) from NCNR ftp server
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::loadBkgdFtp

return, Self->LoadBkgd(/fromFtp)

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; FANSTool::loadBkgd
;
; PURPOSE:
;   Select and load Bkgd data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   dataObject - specifies the data object whose data is to be reloaded
;   
;   reload  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   fromFTP - set if the source of the datafile to be loaded is the FTP server
;-
function FANSTool::loadBkgd, reload=reload, dataObject=oItem, fromFTP=fromFTP

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::loadBkgd: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

reloading = keyword_set(reload)
if (reloading) then begin
   if (~obj_valid(oItem)) then return, 0
   oItem->GetProperty, description=files
   count = 1
   void = oItem->GetMetaData('FROMFTP',fromFtp)
endif else begin
   ; Let user specify files to load
   title = 'Select Background Files to Load'
   files = Self->GetFileList(title,count,fromFTP=fromFTP)
   if (count eq 0) then return, 0
endelse

; switch on the busy symbol for cursor
widget_control, /HOURGLASS

; Read the contents of the data files and make entries in the 'Sample' tree of the data manager
Self->GetProperty, sumBkgdFlag=sumFlag, bkgdContRef=oCont, nBkgd=nData
for i=0, count-1 do begin
   filename = files[i]
   if (~Self->readData(filename, dataPtr, fromFTP=fromFTP)) then continue

   basename = file_basename(filename,'.bt4',/fold_case)
   
   if (~reloading) then begin
      ;; create new object and attach dataPtr
      oItem = obj_new('IDLitData',basename,name=basename,description=filename,type='bkgd')   
      oItem->AddMetaData,'DATAPTRREF',dataPtr
      oItem->AddMetaData,'FROMFTP',keyword_set(fromFTP)
      
      ; iTools does not handle '_[0-9]' characters in identifiers well eg _0, _1, etc
      ; replace all '_' with '-'
      oItem->GetProperty, identifier=idd
      hasChanged = 0
      while ((pos = stregex(idd,'_[0-9]')) ne -1) do begin
         strput, idd,'-',pos
         hasChanged = 1
      endwhile
      if (strmatch(idd,'[0-9]*')) then begin   ; if first char is a number
         idd = 'id'+idd
         hasChanged = 1
      endif
      if (hasChanged) then oItem->SetProperty,identifier=idd 

      ; Add to the data manager
      Self->AddByIdentifier, 'FANS Bkgd Data Manager', oItem
   endif else begin
      ;; simply replace previous dataPtr with updated one
      void = oItem->GetMetaData('DATAPTRREF',tempPtr)
      if (ptr_valid(tempPtr)) then heap_free, tempPtr
      oItem->AddMetaData,'DATAPTRREF',dataPtr
   endelse
   
   if (i eq 0 && sumFlag && (nData gt 0)) then begin
      ;; if data previously read and summed, then delete the sum
      if (oCont->Count() eq nData+1) then begin
         oSum = oCont->Get(position=nData)
         if (obj_valid(oSum)) then begin
            void = oSum->GetMetaData('DATAPTRREF',sumDataPtr) ; the real data is in this dataPtr
            if (ptr_valid(sumDataPtr)) then heap_free, sumDataPtr   ; so must free it!

            oCont->Remove, oSum
            obj_destroy, oSum   ; oSum is not part of any hierarchy so no deed to use DM service to destroy it!
         endif
      endif
   endif
   
   if (~reloading) then begin
      oCont->Add, oItem
      Self.nBkgd++
   endif

   dataPtr = 0   ; make dataPtr invalid before it is used again when looping
endfor

if ((Self.nBkgd gt 1) && sumFlag) then Self->MergeData, 'bkgd'

Self->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=(Self.nBkgd gt 0)
Self->SetPropertyAttribute, 'BKGDFLAG', sensitive=(Self.nBkgd gt 0)

Self->refreshPropertySheet   ; force prop sheet to update

Self.modifiedFlag = 1
return, 1
end
;-------------------------------------------------------------------------------


;===============================================================================
; FANSTool::loadSampFtp
;
; PURPOSE:
;   Specify and load sample data file(s) from NCNR ftp server
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::loadSampFtp

return, Self->LoadSamp(/fromFtp)

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; FANSTool::loadSamp
;
; PURPOSE:
;   Specify and load sample data file(s)
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   dataObject - specifies the data object whose data is to be reloaded
;   
;   reload  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   fromFTP - set if the source of the datafile to be loaded is the FTP server
;-
function FANSTool::loadSamp, reload=reload, dataObject=oItem, fromFTP=fromFTP

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::loadSamp: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle, dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

reloading = keyword_set(reload)
if (reloading) then begin
   if (~obj_valid(oItem)) then return, 0
   oItem->GetProperty, description=files
   count = 1
   void = oItem->GetMetaData('FROMFTP',fromFtp)
   ;if (~Self->ReadData(filename, dataPtr)) then return, 0
endif else begin
   ; Let user specify files to load
   title = 'Select Sample Files to Load'
   
   files = Self->GetFileList(title,count,fromFTP=fromFTP)
   if (count eq 0) then return, 0
endelse

; switch on the busy symbol for cursor
widget_control, /HOURGLASS

; Read the contents of the data files and make entries in the 'Sample' tree of the data manager
Self->GetProperty, sumSampFlag=sumFlag, sampContRef=oCont, nSamp=nData
for i=0, count-1 do begin

   filename = files[i]
   if (~Self->readData(filename, dataPtr, fromFTP=fromFTP)) then continue

   basename = file_basename(filename,'.bt4',/fold_case)

   if (~reloading) then begin
      ;; create new object and attach dataPtr
      oItem = obj_new('IDLitData',basename,name=basename,description=filename,type='samp')   
      oItem->AddMetaData,'DATAPTRREF',dataPtr
      oItem->AddMetaData,'FROMFTP',keyword_set(fromFTP)
      
      ; iTools does not handle '_[0-9]' characters in identifiers well eg _0, _1, etc
      ; replace all '_' with '-'
      oItem->GetProperty, identifier=idd
      hasChanged = 0
      while ((pos = stregex(idd,'_[0-9]')) ne -1) do begin
         strput, idd,'-',pos
         hasChanged = 1
      endwhile
      if (strmatch(idd,'[0-9]*')) then begin   ; if first char is a number
         idd = 'id'+idd
         hasChanged = 1
      endif
      if (hasChanged) then oItem->SetProperty,identifier=idd 

      ; Add to the data manager
      Self->AddByIdentifier, 'FANS Sample Data Manager', oItem
   endif else begin
      ;; simply replace previous dataPtr with updated one
      void = oItem->GetMetaData('DATAPTRREF',tempPtr)
      if (ptr_valid(tempPtr)) then heap_free, tempPtr
      oItem->AddMetaData,'DATAPTRREF',dataPtr
   endelse
   
   if (i eq 0 && sumFlag && (nData gt 0)) then begin
      ;; if data previously read and summed, then delete the sum
      if (oCont->Count() eq nData+1) then begin
         oSum = oCont->Get(position=nData)
         if (obj_valid(oSum)) then begin
            void = oSum->GetMetaData('DATAPTRREF',sumDataPtr) ; the real data is in this dataPtr
            if (ptr_valid(sumDataPtr)) then heap_free, sumDataPtr   ; so must free it!

            oCont->Remove, oSum
            obj_destroy, oSum   ; oSum is not part of any hierarchy so no deed to use DM service to destroy it!
         endif
      endif
   endif
   
   if (~reloading) then begin
      oCont->Add, oItem
      Self.nSamp++
   endif

   dataPtr = 0   ; make dataPtr invalid before it is used again when looping
endfor

if ((Self.nSamp gt 1) && sumFlag) then Self->MergeData, 'samp'

Self->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=(Self.nSamp gt 0)
Self->refreshPropertySheet   ; force prop sheet to update

Self.modifiedFlag = 1

return, 1
end
;-------------------------------------------------------------------------------



;===============================================================================
; FANSTool::restoreDataVersion
;
; PURPOSE:
;   Restore the specified object to the latest version
;
; PARAMETERS:
;   oData [in|out] - object to be restored to the latest version
;
; KEYWORDS:
;
pro FANSTool::restoreDataVersion, oData
compile_opt idl2

; restore to latest IDLitData version
oData->Restore

; Pull the data pointer structure and ensure it is up to data
currentVersion = Self.dataVersion

void = oData->GetMetaData('DATAPTRREF',tempPtr)
updatefromVersion1 = 1  ; by default
tnames = tag_names(*tempPtr)
void = where(tnames eq 'DATAVERSION',versionTagIsPresent)
if (versionTagIsPresent) then begin
  case (*tempPtr).dataVersion of
    1: begin
      updateFromVersion1 = 1
      updateFromVersion2 = 0
    end
    
    2: begin
      updateFromVersion2 = 1
      updateFromVersion1 = 0
    end
    
    else: 
  endcase
endif

if (updateFromVersion1) then begin
  ;; version 1 lack the following fields:
  ;;  dataVersion 
  ;;  totErrsPtr field (ptr)
  dataPtr = Self->initDataPtr()
  nn = n_tags(*dataPtr)
  no = n_tags(*tempPtr)
  
  j = where(tnames eq 'MONPTR', found)
  j = j[0]
  
  for i = 0, j-1 do (*dataPtr).(i) = (*tempPtr).(i)
  (*dataPtr).(j) = 2    ; set value of dataVersion to 2

  for i = j, no-1 do  (*dataPtr).(i+1) = ptr_new((*(*tempPtr).(i)))
  j = where(tnames eq 'TOTCNTSPTR', found)
  (*dataPtr).(i+1) = ptr_new(sqrt((*(*tempPtr).(j))))  ; set the totErrsPtr field to sqrt(totCntsPtr)
  
  if (ptr_valid(tempPtr)) then heap_free, tempPtr
  oData->AddMetaData,'DATAPTRREF',dataPtr
end


end


;===============================================================================
; FANSTool::readData
;
; PURPOSE:
;   Read the contents of a .bt4 file
;
; PARAMETERS:
;   file [in] - filename to be read.
;
;   dataPtr [in|out] - data structure to read file contents into
;
; KEYWORDS:
;
function FANSTool::readData, file, dataPtr, fromFTP=fromFTP
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::readData: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        heap_free, dataPtr
        if (n_elements(lun) gt 0) then free_lun, lun, /force
        return, 0
    endif
endif


; check parameter count
if (n_params() ne 2) then return, 0

; check file parameter is defined
if (n_elements(file) le 0) then return, 0

filename=file
if (keyword_set(fromFTP)) then begin
  Self->GetProperty, ftpObject=oFTP
  
  ; locate a suitable temporal directory
;  tmpDir = (!version.OS_FAMILY eq 'Windows')? GETENV('TMP') : GETENV('TMPDIR')
;  if (tmpDir eq '') then return, 0
  tmpDir = !home_dir
  ; and copy file from the server into tmpfile on the host
  ; use this tmpfile below to load the data from
  tmpfile = tmpDir + path_sep() + 'FANSftpData.txt' 
  filename = oFTP->GetFileContent(file,localfilename=tmpfile)
  if (~file_test(filename,/read)) then return, 0  ; check file exists and is readable
endif

; B/c of unusual error resulting from running out of available logical units
; (probably a bug in get_lun); add the following 4 lines to force the use of 
; the same first available file unit every time
lun = 100
while ((fstat(lun)).open && (lun lt 129)) do begin
  lun++
endwhile

; open input file
openr, lun, filename, error=err ;, /get_lun
if (err ne 0) then begin
   free_lun, lun, /force
   return, 0
endif

;nlines = file_lines(filename)
buffer = strarr(12)
readf, lun, buffer

; initialised dataPtr to point to a FANS data structure if necessary
if (~ptr_valid(dataPtr)) then dataPtr = Self->initDataPtr()

; Set current data version number
(*dataPtr).dataVersion = Self.dataVersion

; get file header and sort out the useful info from it
(*dataPtr).filename = strmid(buffer[0],1,12)
(*dataPtr).date = strmid(buffer[0],16,17)
(*dataPtr).scan = strmid(buffer[0],36,1)
(*dataPtr).monitor = long(strmid(buffer[0],39,12))
(*dataPtr).prefactor = long(strmid(buffer[0],52,4))
(*dataPtr).base = strmid(buffer[0],58,4)
npts = fix(strmid(buffer[0],64,6))
(*dataPtr).type = strmid(buffer[0],71,3)
(*dataPtr).title = strtrim(buffer[2],2)

(*dataPtr).collimation = strmid(buffer[3],1,11)
(*dataPtr).mosaic = strmid(buffer[3],18,8)
(*dataPtr).orientation = strmid(buffer[3],36,23)

toks = float(strsplit(buffer[5],' ',/extract, count=ntok))
if (ntok eq 6) then begin
   (*dataPtr).latticeA = toks[0]
   (*dataPtr).latticeB = toks[1]
   (*dataPtr).latticeC = toks[2]
   (*dataPtr).latticeAl = toks[3]
   (*dataPtr).latticeBe = toks[4]
   (*dataPtr).latticeGa = toks[5]
endif

toks = float(strsplit(buffer[7],' ',/extract, count=ntok))
if (ntok eq 7) then begin
   (*dataPtr).eStart = toks[0]
   (*dataPtr).eDelta = toks[1]
   (*dataPtr).eFinal = toks[2]
   (*dataPtr).monDSpace = toks[3]
   (*dataPtr).anaDSpace = toks[4] 
   (*dataPtr).tStart = toks[5]
   (*dataPtr).tInc = toks[6]
endif

toks = float(strsplit(buffer[9],' ',/extract, count=ntok))
if (ntok eq 7) then begin
   (*dataPtr).qhCenter = toks[0]
   (*dataPtr).qkCenter = toks[1]
   (*dataPtr).qlCenter = toks[2]
   (*dataPtr).qhDelta = toks[3]
   (*dataPtr).qkDelta = toks[4]
   (*dataPtr).qlDelta = toks[5]
   (*dataPtr).hfield = toks[6]
endif

if (strmid(buffer[11],37,1) eq 'T') then (*dataPtr).tflag = 1

; now the data section.
hbuf = ((*dataPtr).tflag)? fltarr(7) : fltarr(6)
dbuf = lonarr((*dataPtr).ndets)
npts = (npts gt 0)? npts : 2000
i = 0 
time = (temp = (totErrs = (totCnts =  (omega = (qx = (qy = (qz = fltarr(npts))))))))
nosActDets = Self.nActiveDetectors
detCnts = fltarr(nosActDets,npts)
detErrs = fltarr(nosActDets,npts)
 
while (not eof(lun)) do begin
    readf, lun, hbuf,dbuf       ; read one data block (per energy transfer)
    qx[i] = hbuf[0]
    qy[i] = hbuf[1]
    qz[i] = hbuf[2]
    omega[i] = hbuf[3]
    if ( (*dataPtr).tflag) then begin
        temp[i] = hbuf[4]
        time[i] = hbuf[5]
        totCnts[i] = hbuf[6]
    endif else begin
        temp[i] = (*dataPtr).tStart
        time[i] = hbuf[4]
        totCnts[i] = hbuf[5]
    endelse
    totErrs[i] = (totCnts[i] le 0)? 1 : sqrt(totCnts[i]) ; assign 1 to erros if cnts <= 0
    detCnts[*,i] = dbuf[0:nosActDets-1] ; trim to active detectors only
;    res = where(dbuf le 0)      ; locate indices where detcts <= 0
;    if (res[0] ne -1) then begin ; and assign an error value of 1 to
;        dbuf[res] = 1
;    endif
;    detErrs[*,i] = sqrt(dbuf[0:nosActDets-1])
    i = i + 1
endwhile

free_lun, lun, /force
if (keyword_set(fromFTP)) then FILE_DELETE, filename  ;  delete temp data file

; move data into the heap and save pointers in dataPtr; trim spectra
; to number of points read from file
;conv_factor = (Self.energyUnitFlag eq 0)? 1.0 : Self.mev2wnos
(*dataPtr).nspec = i 
(*dataPtr).ndets = nosActDets

omega = omega[0:i-1]
sortIndex = sort(omega)
omega  = omega[sortIndex]

(*dataPtr).energyPtr = ptr_new(omega,/no_copy) ;ptr_new(omega[0:i-1]*conv_factor,/no_copy)
(*dataPtr).qxPtr = ptr_new(qx[sortIndex],/no_copy)
(*dataPtr).qyPtr = ptr_new(qy[sortIndex],/no_copy)
(*dataPtr).qzPtr = ptr_new(qz[sortIndex],/no_copy)
(*dataPtr).tempPtr = ptr_new(temp[sortIndex],/no_copy)
(*dataPtr).timePtr = ptr_new(time[sortIndex],/no_copy)
(*dataPtr).totCntsPtr = ptr_new(totCnts[sortIndex],/no_copy)
(*dataPtr).totErrsPtr = ptr_new(totErrs[sortIndex],/no_copy)
(*dataPtr).detCntsPtr = ptr_new(detCnts[*,sortIndex],/no_copy)
;(*dataPtr).detErrsPtr = ptr_new(detErrs[*,sortIndex],/no_copy)
(*dataPtr).monPtr = ptr_new(fltarr(i) + (*dataPtr).monitor * (*dataPtr).prefactor)
(*dataPtr).historyPtr = ptr_new(file)

;(*dataPtr).energyPtr = ptr_new(omega[0:i-1],/no_copy) ;ptr_new(omega[0:i-1]*conv_factor,/no_copy)
;(*dataPtr).qxPtr = ptr_new(qx[0:i-1],/no_copy)
;(*dataPtr).qyPtr = ptr_new(qy[0:i-1],/no_copy)
;(*dataPtr).qzPtr = ptr_new(qz[0:i-1],/no_copy)
;(*dataPtr).tempPtr = ptr_new(temp[0:i-1],/no_copy)
;(*dataPtr).timePtr = ptr_new(time[0:i-1],/no_copy)
;(*dataPtr).totCntsPtr = ptr_new(totCnts[0:i-1],/no_copy)
;;(*dataPtr).totErrsPtr = ptr_new(totErrs[0:i-1],/no_copy)
;(*dataPtr).detCntsPtr = ptr_new(detCnts[*,0:i-1],/no_copy)
;;(*dataPtr).detErrsPtr = ptr_new(detErrs[*,0:i-1],/no_copy)
;(*dataPtr).monPtr = ptr_new(fltarr(i) + (*dataPtr).monitor * (*dataPtr).prefactor)
;(*dataPtr).historyPtr = ptr_new(file)

; Default labels
(*dataPtr).xtitle = 'Energy Transfer'
(*dataPtr).xunits = 'meV'
case ((*dataPtr).base) of
   'TIME': begin
      value = fix((*(*dataPtr).timePtr)[0] * 60)
      tt = ' secs'
   end
   
   'NEUT': begin
      value = (*dataPtr).monitor * (*dataPtr).prefactor
      tt = ' mon'
   end
endcase
(*dataPtr).ytitle = 'Counts per '+strtrim(string(value),2)+tt

return, 1
end
;-------------------------------------------------------------------------------



;===============================================================================
; FANSTool::saveAsDave
;
; PURPOSE:
;   Save output to file using DAVE format
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::saveAsDave, oSamp
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsDave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        if (n_elements(lun) gt 0) then free_lun, lun, /force
        return, 0
    endif
endif

; Get the selected data
if (~obj_valid(oSamp)) then Self->GetProperty, sampSelected=oSamp

if (~obj_valid(oSamp)) then begin
   msg = "Please first select a sample dataset from the 'Sample' tree"
   Self->StatusMessage, msg
   return, 0
endif

oSamp->GetProperty, type=dataType
if (~strcmp(dataType, 'samp', /fold_case)) then begin
   msg = "Could not find a valid 'Sample' dataset to output"
   Self->StatusMessage, msg
   return, 0
endif

void = oSamp->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

if (~Self->retrieveDavePtr(oSamp, davePtr)) then return, 0

oSamp->GetProperty, description=desc
filename = file_basename(desc,/fold_case,'.bt4')
; replace spaces in filename with _
while ((pos = stregex(filename,' ')) ne -1) do strput, filename,'_',pos

Self->GetProperty, working_directory=workDir
title = "Specify filename to store DAVE format data"
ext = 'dave'
filename = dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
                       ,/overwrite_prompt, file=filename+'.dave',path=workDir, get_path=new_workDir)
                       
if (~strcmp(filename,'')) then begin
   save, davePtr, filename=filename
   if (file_test(new_workDir,/directory)) then Self->SetProperty, working_directory=new_workDir

   msg = 'Output stored in: '+filename
   Self->StatusMessage, msg
endif
 
return, 1
end


;===============================================================================
; FANSTool::exportToDaveDM
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::exportToDaveDM, oSamp
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsDave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

; Get the selected data
if (~obj_valid(oSamp)) then Self->GetProperty, sampSelected=oSamp

if (~obj_valid(oSamp)) then begin
   msg = "Please first select a sample dataset from the 'Sample' tree"
   Self->StatusMessage, msg
   return, 0
endif

oSamp->GetProperty, type=dataType
if (~strcmp(dataType, 'samp', /fold_case)) then begin
   msg = "Could not find a valid 'Sample' dataset to output"
   Self->StatusMessage, msg
   return, 0
endif

void = oSamp->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

if (~Self->retrieveDavePtr(oSamp, davePtr)) then return, 0

oSamp->GetProperty, description=desc
nametag = file_basename(desc,/fold_case,'.bt4')

if ((n_elements(nameTag) le 0) || (strtrim(nameTag,2) eq '')) then nameTag='Untitled'

; Send data to DAVE Data Manager Folder
Self.DAVETool->AddDavePtrToDataManager, davePtr, nameTag

msg = "New extry '"+nameTag+"' created in DAVE Data Manager"
Self->StatusMessage, msg


return, 1
end


;===============================================================================
; FANSTool::exportToPAN
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::exportToPAN, oSamp
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsDave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

; Get the selected data
if (~obj_valid(oSamp)) then Self->GetProperty, sampSelected=oSamp

if (~obj_valid(oSamp)) then begin
   msg = "Please first select a sample dataset from the 'Sample' tree"
   Self->StatusMessage, msg
   return, 0
endif

oSamp->GetProperty, type=dataType
if (~strcmp(dataType, 'samp', /fold_case)) then begin
   msg = "Could not find a valid 'Sample' dataset to output"
   Self->StatusMessage, msg
   return, 0
endif

void = oSamp->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

if (~Self->retrieveDavePtr(oSamp, davePtr)) then return, 0

notifyIDs = [wTLB,wTLB]
Self->GetProperty, working_directory=workDir
oVoid = obj_new('OPAN',notifyID=[wTLB,wTLB],group_leader=wTLB $
                ,davePtr=davePtr,workDir=workDir)

heap_free, davePtr

return, 1
end


;===============================================================================
; FANSTool::saveProperties
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::saveProperties
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveProperties: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif

print, 'properties saved...'

return, 1
end


;===============================================================================
; FANSTool::retrieveDavePtr
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::retrieveDavePtr, oData, davePtr
compile_opt idl2

; Get the selected data
if (~obj_valid(oData)) then return, 0

void = oData->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

status = Self->CalculateXYE(oData, xvals, yvals, evals, xunits)

status = create_dave_pointer(davePtr, qty=yvals, xvals=xvals, err=evals)
if (~status) then begin
   msg = "Unable to allocate memory!"
   Self->StatusMessage, msg
   if (ptr_valid(davePtr)) then heap_free, davePtr
   return, 0
endif

(*(*davePtr).dataStrPtr).commonStr.instrument = 'FANS'
(*(*davePtr).dataStrPtr).commonStr.xlabel = (*dataPtr).xtitle
(*(*davePtr).dataStrPtr).commonStr.xunits = xunits
(*(*davePtr).dataStrPtr).commonStr.xtype = 'points'
(*(*davePtr).dataStrPtr).commonStr.ylabel = ''
(*(*davePtr).dataStrPtr).commonStr.yunits = ''
(*(*davePtr).dataStrPtr).commonStr.ytype = 'points'

(*(*davePtr).dataStrPtr).commonStr.histlabel = (*dataPtr).ytitle
(*(*davePtr).dataStrPtr).commonStr.histunits = ''
(*(*(*davePtr).dataStrPtr).commonStr.treatmentPtr) = (*Self.historyPtr)

specificStr = { $
            filename:(*dataPtr).filename, $
            date:(*dataPtr).date, $
            scan:(*dataPtr).scan, $
            monitor:(*dataPtr).monitor, $
            prefactor:(*dataPtr).prefactor, $
            base:(*dataPtr).base, $
            type:(*dataPtr).type, $
            title:(*dataPtr).title, $
            collimation:(*dataPtr).collimation, $
            mosaic:(*dataPtr).mosaic, $
            orientation:(*dataPtr).orientation, $
            latticeA:(*dataPtr).latticeA, $
            latticeB:(*dataPtr).latticeB, $
            latticeC:(*dataPtr).latticeC, $
            latticeAl:(*dataPtr).latticeAl, $
            latticeBe:(*dataPtr).latticeBe, $
            latticeGa:(*dataPtr).latticeGa, $
            eStart:(*dataPtr).eStart, $
            eFinal:(*dataPtr).eFinal, $
            eDelta:(*dataPtr).eDelta, $
            monDSpace:(*dataPtr).monDSpace, $
            anaDSpace:(*dataPtr).anaDSpace, $
            tStart:(*dataPtr).tStart, $
            tInc:(*dataPtr).tInc, $
            qhCenter:(*dataPtr).qhCenter, $
            qkCenter:(*dataPtr).qkCenter, $
            qlCenter:(*dataPtr).qlCenter, $
            qhDelta:(*dataPtr).qhDelta, $
            qkDelta:(*dataPtr).qkDelta, $
            qlDelta:(*dataPtr).qlDelta, $
            hfield:(*dataPtr).hfield, $
            tflag:(*dataPtr).tflag, $
            ndets:(*dataPtr).ndets, $
            nspec:(*dataPtr).nspec $
           }

(*(*davePtr).dataStrPtr).specificPtr = ptr_new(specificStr)

return, 1
end


;===============================================================================
; FANSTool::writeProject
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::writeProject, noPrompt=noPrompt
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsDave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return, 0
    endif
endif


Self->GetProperty, nSamp=_nSamp,nBkgd=_nBkgd,nFast=_nFast $
                 , preferences=_prefStr, working_directory=workDir
noData = (_nSamp le 0) && (_nBkgd le 0) && (_nFast le 0)
if (noData) then begin
   msg = 'No data in current session so cannot proceed with save action!'
   Self->StatusMessage, msg
   Self->ErrorMessage, msg, severity=0, title='Save Unnecessary!'
   return, 0
endif

projName = Self.projName
if (keyword_set(noPrompt)) then begin
   projName = workDir+projName+'.fisv'
endif else begin
   oSamp = Self.sampContRef->Get(position = 0)
   if (strcmp(projName,'Untitled',/fold_case) && obj_valid(oSamp)) then begin
      oSamp->GetProperty, description=desc
      projName = file_basename(desc,/fold_case,'.bt4')
      ; replace spaces in filename with _
      while ((pos = stregex(projName,' ')) ne -1) do strput, projName,'_',pos
   endif
   title = "Specify filename to store session"
   ext = 'fisv'
   projName = dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
                          ,/overwrite_prompt, file=projName+'.fisv',path=workDir, get_path=new_workDir)
endelse
                       
if (~strcmp(projName,'')) then begin
   if (_nSamp gt 0) then _datasets = (Self.sampContRef->Get(/all))[0:_nSamp-1]
   if (_nBkgd gt 0) then _datasets = (n_elements(_datasets) gt 0)? $
                                     [_datasets,(Self.bkgdContRef->Get(/all))[0:_nBkgd-1]] : $
                                     (Self.bkgdContRef->Get(/all))[0:_nBkgd-1]
   if (_nFast gt 0) then _datasets = (n_elements(_datasets) gt 0)? $
                                     [_datasets,(Self.fastContRef->Get(/all))[0:_nFast-1]] : $
                                     (Self.fastContRef->Get(/all))[0:_nFast-1]
   nd = n_elements(_datasets)
   _dataPtrs = ptrarr(nd)
   for i=0,nd-1 do begin
      void = _datasets[i]->GetMetaData('DATAPTRREF',dataPtr)
      _dataPtrs[i] = dataPtr
   endfor
   save, _datasets, _dataPtrs, _nSamp, _nBkgd, _nFast, _prefStr $
       , description = 'FANS Data Reduction Session' $
       , filename=projName, /compress
   if (file_test(new_workDir,/directory)) then Self->SetProperty, working_directory=new_workDir

   msg = 'Session stored in: '+projName
   Self->StatusMessage, msg
   
   projName = file_basename(projName,/fold_case,'.fisv')
   Self.projName = projName
   Self->SetProperty, tool_filename=projName
   
   Self.modifiedFlag = 0
endif

return, 1
end


;===============================================================================
; FANSTool::readProject
;
; PURPOSE:
;   export dataset to DAVE Data Manager
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::readProject
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsDave: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        if (obj_valid(oSave)) then obj_destroy, oSave
        return, 0
    endif
endif



filter = '*.fisv'
Self->GetProperty, working_directory=workDir
if (n_elements(title) eq 0) then title='Specify session to load'

filename = dialog_pickfile(title=title,/must_exist, dialog_parent=wTLB $
                       ,filter=filter,path=workDir, get_path=newWorkDir)

if (filename eq '') then return, 0
Self->SetProperty, work_directory=newWorkDir

oSave = obj_new('IDL_Savefile', filename)
contents = oSave->Contents()
obj_destroy, oSave
expectedDesc='FANS Data Reduction Session'
if (~strcmp(contents.description,expectedDesc,/fold_case)) then begin
   msg = 'Not a valid FANS Reduction session file!'
   Self->StatusMessage, msg
   Self->ErrorMessage, msg, severity=1, title='Invalid file type!'   
   return, 0
endif

restore, filename, restored_objects=oObj, /relaxed_structure_assignment
nObj = n_elements(oObj)
if (nObj le 0) then return, 0

; Check whether or not to save current session before closing it
if (Self.modifiedFlag) then begin
   ; If our current state is dirty, prompt to save first.
   Self->GetProperty, working_directory=workDir
   outfile = workDir+Self.projName+'.fisv'
   status = self->PromptUserYesNo( $
       "Save changes to " + outfile + "?", $
       answer, $
       DEFAULT_NO=0, $
       TITLE='Save', $
       /CANCEL)

   if (status && (answer eq 1)) then begin
      void = Self->writeProject(/noPrompt)
   endif

endif

; delete all datasets in current session
void = Self->DeleteAllDatasets(/noPrompt)

; restore datasets
nd = n_elements(_datasets)
for i=0, nd-1 do begin
   oData = _datasets[i]
   Self->restoreDataVersion, oData
   oData->GetProperty, type=dataType
   case dataType of
      'samp': begin
         Self->AddByIdentifier, 'FANS Sample Data Manager', oData
         Self.sampContRef->Add, oData
      end
      
      'bkgd': begin
         Self->AddByIdentifier, 'FANS Bkgd Data Manager', oData
         Self.bkgdContRef->Add, oData
      end
      
      'fast': begin
         Self->AddByIdentifier, 'FANS Fast Data Manager', oData
         Self.fastContRef->Add, oData
      end
      
      else:
   
   endcase
endfor
Self.nSamp = _nSamp
Self.nBkgd = _nBkgd
Self.nFast = _nFast
if (Self.nSamp gt 1) then Self->MergeData, 'samp'
if (Self.nBkgd gt 1) then Self->MergeData, 'bkgd'
if (Self.nFast gt 1) then Self->MergeData, 'fast'

projName = file_basename(filename,/fold_case,'.fisv')
Self.projName = projName
Self->SetProperty, preferences=_prefStr, tool_filename=projName

Self->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=(Self.nSamp gt 0)

Self->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=(Self.nBkgd gt 0)
Self->SetPropertyAttribute, 'BKGDFLAG', sensitive=(Self.nBkgd gt 0)

Self->SetPropertyAttribute, 'SUMFASTFLAG', sensitive=(Self.nFast gt 0)
Self->SetPropertyAttribute, 'FASTFLAG', sensitive=(Self.nFast gt 0)

Self->refreshPropertySheet

;; Select a dataset and plot it
if (_nSamp gt 0) then begin
   oItem = Self.sampContRef->Get(position = 0)
   dataType='samp'
endif else if (_nBkgd gt 0) then begin
   oItem = Self.sampContRef->Get(position = 0)
   dataType='bkgd'
endif else if (_nFast gt 0) then begin
   oItem = Self.sampContRef->Get(position = 0)
   dataType='fast'
endif

if (obj_valid(oItem)) then begin
   (*sPtr).itemID = oItem->GetFullIdentifier()
   cw_fansdatamanager_setselect, (*sPtr).wDM, (*sPtr).itemID, /clear, dataType=dataType ; select it

   wd_FANStool_sendPlotEvent, oUI, oItem   ; plot it
endif

Self.modifiedFlag = 0
return, 1
end


;===============================================================================
; FANSTool::saveAsAscii
;
; PURPOSE:
;   Save output to file in ASCII format
;
; PARAMETERS:
;
; KEYWORDS:
;
function FANSTool::saveAsAscii, oSamp
compile_opt idl2

oUI = Self->GetUI()
oUI->GetProperty, group_leader = wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANSTool::saveAsAscii: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        if (n_elements(lun) gt 0) then free_lun, lun, /force
        catch, /cancel
        return, 0
    endif
endif


; Get the selected data
if (~obj_valid(oSamp)) then Self->GetProperty, sampSelected=oSamp

if (~obj_valid(oSamp)) then begin
   msg = "Please first select a sample dataset from the 'Sample' tree"
   Self->StatusMessage, msg
   return, 0
endif

oSamp->GetProperty, type=dataType
if (~strcmp(dataType, 'samp', /fold_case)) then begin
   msg = "Could not find a valid 'Sample' dataset to output"
   Self->StatusMessage, msg
   return, 0
endif

void = oSamp->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

status = Self->CalculateXYE(oSamp, xvals, yvals, evals, xunits)

time = (*(*dataPtr).timePtr)
temp = (*(*dataPtr).tempPtr)

oSamp->GetProperty, description=desc
filename = file_basename(desc,/fold_case,'.bt4')
; replace spaces in filename with _
while ((pos = stregex(filename,' ')) ne -1) do strput, filename,'_',pos

Self->GetProperty, working_directory=workDir
title = "Specify filename to store ASCII format data"
ext = 'txt'
filename = dialog_pickfile(title=title,default_extension=ext, dialog_parent=wTLB $
                       ,/overwrite_prompt, file=filename+'.txt',path=workDir, get_path=new_workDir)
                       
if (~strcmp(filename,'')) then begin
   openw, lun, filename, /get_lun

   nd = n_elements(xvals)
   for i = 0,nd-1 do printf, lun, format='(5(F11.4,2X))',xvals[i],yvals[i],evals[i],time[i],temp[i]
   
   free_lun, lun, /force
   if (file_test(new_workDir,/directory)) then Self->SetProperty, working_directory=new_workDir
endif

msg = 'Output stored in: '+filename
Self->StatusMessage, msg
 
return, 1
end


;===============================================================================
function FANSTool::CalculateXYE, oData, xvals, yvals, evals, xunits
compile_opt idl2

if (~obj_valid(oData) || ~obj_isa(oData, 'IDLitData')) then return, 0

void = oData->GetMetaData('DATAPTRREF',dataPtr)
if (~ptr_valid(dataPtr)) then return, 0

xvals = (*(*dataPtr).energyPtr)
;yvals = (*(*dataPtr).totCntsPtr)

(*Self.historyPtr) = (*(*dataPtr).historyPtr)

oData->GetProperty, type=dataType

case strlowcase(dataType) of
   'samp': status = Self->Correct4BkgdMaskFast(dataPtr, yvals, evals)
   'bkgd': status = Self->Correct4MaskFast(dataPtr, yvals, evals)
   'fast': status = Self->Correct4Mask(dataPtr, yvals, evals)
   else: return, 0
endcase

;if (n_elements(evals) eq 0 && n_elements(yvals) gt 0) then begin
;   evals = yvals
;   ind = where(evals le 0.0, zeroCnt)
;   if (zeroCnt) then evals[ind] = 1.0
;   evals = sqrt(evals)            
;endif

; Monitor rescale if required
Self->GetProperty, monScaleFlag=monFlag, monScaleValue=monValue
if (monFlag && ~strcmp(dataType,'fast',/fold_case)) then begin
   yvals = (yvals / (*(*dataPtr).monPtr)[0]) * monValue 
   evals = (evals / (*(*dataPtr).monPtr)[0]) * monValue 

   history = ['','Monitor Rescaling','Counts rescaled to a new monitor']
   history = [history,'Original monitor value: '+strtrim(string((*(*dataPtr).monPtr)[0]),2)]
   history = [history,'New monitor value: '+strtrim(string(monValue),2)]
   (*Self.historyPtr) = [(*Self.historyPtr),history]

   ;; update ytitle
   (*dataPtr).ytitle = 'Counts per '+strtrim(string(monValue),2)+' mon'
endif else begin
   ;; reset ytitle to its default
   case ((*dataPtr).base) of
      'TIME': begin
         value = fix((*(*dataPtr).timePtr)[0] * 60)
         tt = ' secs'
      end
      
      'NEUT': begin
         value = (*dataPtr).monitor * (*dataPtr).prefactor
         tt = ' mon'
      end
   endcase
   (*dataPtr).ytitle = 'Counts per '+strtrim(string(value),2)+tt
endelse

Self->GetProperty, energyUnitFlag=eFlag, mev2wnos=mev2wnos
if (eFlag eq 0) then begin
   xunits = 'meV'
endif else begin
   xunits = 'cm-1'
   xvals = xvals*mev2wnos
endelse

history = ['','Energy Units: '+xunits]
(*Self.historyPtr) = [(*Self.historyPtr),history]


return, 1

end


;===============================================================================
function FANSTool::Correct4BkgdMaskFast, sampPtr, CorrectedCounts, CorrectedErrors
compile_opt idl2

; Correct for detector masking and fast bkgd
status = Self->Correct4MaskFast(sampPtr, ySamp, eSamp)
correctedCounts = ySamp
correctedErrors = eSamp

xSamp = (*(*sampPtr).energyPtr)
mSamp = (*(*sampPtr).monPtr)

Self->GetProperty, bkgdFlag=bkgdFlag, bkgdSelected=oBkgd
if (bkgdFlag && ~obj_valid(oBkgd)) then begin
   ;; try using the last bkgd data
   oBCont = Self.bkgdContRef
   oBkgd = oBCont->Get(position=(oBCont->Count()-1))
endif
if (~bkgdFlag || ~obj_valid(oBkgd)) then return, 0
void = oBkgd->GetMetaData('DATAPTRREF',bkgdPtr)
if (~ptr_valid(bkgdPtr)) then return, 0

line = '_______________________________________________________'
history = ['',line,'Begin Background Subtraction Section','Background File:',(*(*bkgdPtr).historyPtr)]
(*Self.historyPtr) = [(*Self.historyPtr),history]

status = Self->Correct4MaskFast(bkgdPtr, yBkgd, eBkgd)
xBkgd = (*(*bkgdPtr).energyPtr)
mBkgd = (*(*bkgdPtr).monPtr)
;eBkgd = yBkgd
;ind = where(eBkgd le 0.0, zeroCnt)
;if (zeroCnt) then eBkgd[ind] = 1.0
;eBkgd = sqrt(eBkgd)


;; rebin the bkgd to the same xvals as the samp

;; do the rebinning
drebin,xBkgd,yBkgd,eBkgd,xSamp,zout,zeout,/points,/to_points,err=err,emsg=emsg
if (err ne 0) then begin
   Self->StatusMessage, emsg
   return, 0
endif

; Set zout count to zero for values of energy where 
; yBkgd was undefined. This is obviously the case when
; - xBkgd[0] gt xSamp[0]
; - xBkgd[nb-1] lt xSamp[ns-1]
; - there are gabs in the scan eventhough it is not easy to determine them
; For now, deal with first two cases.
nb = n_elements(xBkgd)
index = where((xSamp lt xBkgd[0]) or (xSamp gt xBkgd[nb-1]), count)
if (count gt 0) then begin
   zout[index] = 0.0
   zeout[index] = 1.0
endif

; subtract bkgd from samp
sf = msamp[0] / mBkgd[0]   ; Rescale the bkgd to the same monitor as the samp
CorrectedCounts = ySamp - zout *sf
CorrectedErrors = sqrt(eSamp^2 + sf*(zeout)^2) 


;history = ['',line,'Background Subtraction',(*(*bkgdPtr).historyPtr),line]
history = ['','End Background Subtraction Section',line]
(*Self.historyPtr) = [(*Self.historyPtr),history]


return, 1
end



;===============================================================================
function FANSTool::Correct4MaskFast, dataPtr, CorrectedCounts, CorrectedErrors
compile_opt idl2

; Correct for detector masking
status = Self->Correct4Mask(dataPtr, CorrectedCounts, CorrectedErrors)

Self->GetProperty, fastFlag=fastFlag, fastparams=paramString $
                 , fastFunc=funcIndex, fastSelected=oFast

; Correct for fast bkgd
params = float(strsplit(paramString,',',/extract,count=nParams))
isZero = 0
if (nParams gt 0) then isZero = (params[0] eq 0.0 && params[1] eq 0.0)
if (fastFlag && (nParams eq 0 || isZero)) then begin
   msg = 'Perhaps you have not specified the Fast Fit Parameters!'
   Self->ErrorMessage, severity=0, msg, title='FANS Data Reduction'
   Self->StatusMessage, msg

   return, 1
endif
if (fastFlag && ~obj_valid(oFast)) then begin
   ;; try using the last fast data
   oFCont = Self.fastContRef
   oFast = oFCont->Get(position=(oFCont->Count()-1))
endif

if (fastFlag && obj_valid(oFast)) then begin
   omega = (*(*dataPtr).energyPtr)
   case funcIndex of
       0: yfit=(fans_Arcsin(omega,params))[*,0] ;; funct returns eval + partial derivatives wrt fpar  
       1: yfit=(fans_Linear(omega,params))[*,0] 
       2: yfit=(fans_Quad(omega,params))[*,0] 
       3: yfit=(fans_Cubic(omega,params))[*,0]
   endcase

   void = oFast->GetMetaData('DATAPTRREF',fastPtr)
   if (~ptr_valid(fastPtr)) then return, 0

   ;; subtract fast bkgd (scaled to counting times)
   correctedCounts = correctedCounts - yfit * (*(*dataPtr).timePtr) / ((*(*fastPtr).timePtr)[0]) 

   history = ['','FAST Bkgd Correction','Fast Background function:'+Self.fastFuncs[funcIndex]]
   history = [history,'Function parameters:'+paramString]
   (*Self.historyPtr) = [(*Self.historyPtr),history]

endif

return, 1
end



;===============================================================================
function FANSTool::Correct4Mask, dataPtr, CorrectedCounts, CorrectedErrors
compile_opt idl2

correctedCounts = (*(*dataPtr).totCntsPtr)
correctedErrors = (*(*dataPtr).totErrsPtr)

Self->GetProperty, maskFlag=maskFlag

ndets = Self.nActiveDetectors
goodDets = indgen(ndets-2)+3    ; leave out monitors (dets 1 and 2)
sBadDets = Self.maskedDets
if (strcmp(sBadDets,'')) then return, 1

if (~compactstring2intvector(sBadDets, badDets)) then return, 0

if (~maskFlag) then return, 1

;; remove bad dets from good dets
nbad = n_elements(badDets)
if (nbad ge ndets-2) then return, 0 ; can't mask detectors!!!

for i = 0,nbad-1 do $
   goodDets = goodDets[where(goodDets ne badDets[i])]

detCounts = (*(*dataPtr).detCntsPtr)[goodDets-1,*]
correctedCounts = total(detCounts,1)
CorrectedErrors = correctedCounts
index = where(CorrectedErrors le 0.0, zeroError)
if (zeroError gt 0) then CorrectedErrors[index] = 1
CorrectedErrors = sqrt(CorrectedErrors)

bads = strtrim(string(badDets),2)
history = ['','Detector masking','The following detectors were excluded:',bads]
(*Self.historyPtr) = [(*Self.historyPtr),history]

return, 1
end



;---------------------------------------------------------------------------
; Lifecycle Routines
;---------------------------------------------------------------------------
; FANSTool::Init
;
; Purpose:
; The constructor of the FANSTool object.
;
; Parameters:
; None.
;
function FANSTool::Init, _REF_EXTRA=_EXTRA
compile_opt idl2

;; Call our super class
if (~self->IDLitTool::Init(_EXTRA=_extra)) then $
  return, 0

oSystem = self->_GetSystem()

; Specify current data version number
; Essentially, whevever the data structure used to store 
; datasets changes the version number should be incremented to reflect
; this. When the version number changes, restoreDataVersion() must also
; be modified to handle restoring older version datasets to the
; current version.
Self.dataVersion = 2

; Add a 'DAVEDATA_MANAGER' service
;self->AddService, OBJ_NEW("DAVEsrvDatamanager", name="DAVEDATA_MANAGER")

; Create the data manager folders
oSampFolder = Obj_New("IDLitDataManagerFolder", identifier='FANS Sample Data Manager' $
                      ,name='Sample', description='FANS Sample Data Manager Container',tool=self)
oBkgdFolder = Obj_New("IDLitDataManagerFolder", identifier='FANS Bkgd Data Manager' $
                      ,name='Background', description='FANS Bkgd Data Manager Container',tool=self)
oFastFolder = Obj_New("IDLitDataManagerFolder", identifier='FANS Fast Data Manager' $
                      ,name='Fast Bkgd', description='FANS Fast Bkgd Data Manager Container',tool=self)
self->Add, [oSampFolder,oBkgdFolder,oFastFolder]


self->createfolders,'Operations/       ',NAME='', description='Horizontal Space'
;;---------------------------------------------------------------------
;;*** File Menu

;; create folders
self->createfolders,'Operations/File',NAME=IDLitLangCatQuery('Menu:File')
;   self->createfolders,'Operations/File/New', $
;                       NAME=IDLitLangCatQuery('Menu:File:New')

;     self->RegisterOperation, 'iPlot', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Plot'

;     self->RegisterOperation, 'iSurface', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Surface'

;     self->RegisterOperation, 'iContour', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Contour'

;     self->RegisterOperation, 'iImage', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Image'

;     self->RegisterOperation, 'iVolume', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Volume'

;     self->RegisterOperation, 'iMap', 'IDLitOpNewTool', $
;         IDENTIFIER='File/New/Map'

;self->RegisterOperation, IDLitLangCatQuery('Menu:File:Open'), $
;  'IDLitopFileOpen', $
;  ACCELERATOR='Ctrl+O', $
;  DESCRIPTION='Open an existing data or image file', $
;  IDENTIFIER='File/Open', ICON='open'


                                ;-----------------
;self->RegisterOperation, IDLitLangCatQuery('Menu:File:Import'), $
;  'IDLitopImportData', $
;  IDENTIFIER='File/Import', $
;  /SEPARATOR
;self->RegisterOperation, IDLitLangCatQuery('Menu:File:Export'), $
;  'IDLitopExportData',icon='camera', $
;  IDENTIFIER='File/Export',/separator



self->RegisterOperation, 'Open Session', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='File/readProject'

self->RegisterOperation, 'Save Current Session', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='File/writeProject'

                                ;-----------------
;self->RegisterOperation, IDLitLangCatQuery('Menu:File:Save'), $
;  'IDLitopFileSave', $
;  ACCELERATOR='Ctrl+S', $
;  IDENTIFIER='File/Save', ICON='save', /SEPARATOR

;self->RegisterOperation, IDLitLangCatQuery('Menu:File:SaveAs'), $
;  'IDLitopFileSaveAs', $
;  IDENTIFIER='File/SaveAs', ICON='save'

                                ;-----------------
Self->RegisterOperation,'Export Graphics...' $
  ,'DAVEopExportGraphics' $
  ,identifier='File/Export' $
  ,ACCELERATOR='Ctrl+E' $
  ,writertypes=['IDLDEST','IDLIMAGE'] $
  ,icon='export',/separator

self->RegisterOperation, IDLitLangCatQuery('Menu:File:PrintPreview'), $
  'IDLitopPrintPreview', $
  DESCRIPTION='Print the contents of the active window', $
  IDENTIFIER='File/PrintPreview', ICON='print'

self->RegisterOperation, IDLitLangCatQuery('Menu:File:Print'), $
  'IDLitopFilePrint', $
  ACCELERATOR='Ctrl+P', $
  DESCRIPTION='Print the contents of the active window', $
  IDENTIFIER='File/Print', ICON='print_24'


self->RegisterOperation, 'Reset Preferences', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='File/resetPreferences',/separator

;self->RegisterOperation, IDLitLangCatQuery('Menu:File:Preferences'), $
;  'IDLitopBrowserPrefs', $
;  IDENTIFIER='File/Preferences', /SEPARATOR

                                ;-----------------
self->RegisterOperation, 'Exit', 'DAVEopMenuItem', ACCELERATOR='Ctrl+Q', $
  IDENTIFIER='File/Exit', /SEPARATOR

self->RegisterOperation, 'ExitOp', 'DAVEopFileExit', IDENTIFIER='ExitOp'


;-----------
; Data Input Menu
self->createfolders,'Operations/Input',NAME='Data Input'

self->RegisterOperation, 'Load Sample Data', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadSamp'

self->RegisterOperation, 'Load Sample Data (NCNR ftp server)', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadSampFtp'

self->RegisterOperation, 'Load Background Data', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadBkgd', /separator

self->RegisterOperation, 'Load Background Data (NCNR ftp server)', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadBkgdFtp'

self->RegisterOperation, 'Load Fast Background Data', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadFast', /separator

self->RegisterOperation, 'Load Fast Background Data (NCNR ftp server)', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/loadFastFtp'

self->RegisterOperation, 'Delete All Loaded Data', $
  'DAVEopMenuItem',icon='open', IDENTIFIER='Input/deleteAllDatasets',/separator



;-----------
; Data Output Menu
self->createfolders,'Operations/Output',NAME='Data Output'

self->RegisterOperation, 'Save output as DAVE file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsDave'

self->RegisterOperation, 'Save output as ASCII file', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/saveAsAscii'

self->RegisterOperation, 'Export output to DAVE Data Maneger', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/exportToDaveDM'

self->RegisterOperation, 'Fit output in PAN', $
  'DAVEopMenuItem',icon='save', IDENTIFIER='Output/exportToPAN',/separator




;;---------------------------------------------------------------------
;; Create our File toolbar container.
;;
;self->Register, IDLitLangCatQuery('Menu:File:New'), 'IDLitOpNewTool', $
;                IDENTIFIER='Toolbar/File/NewTool', ICON='new'
;self->Register, IDLitLangCatQuery('Menu:File:Open'), $
;                PROXY='Operations/File/Open', $
;                IDENTIFIER='Toolbar/File/Open'
;self->Register, IDLitLangCatQuery('Menu:File:Save'), $
;                PROXY='Operations/File/Save', $
;                IDENTIFIER='Toolbar/File/Save'
self->Register, IDLitLangCatQuery('Menu:File:Print'), $
                PROXY='Operations/File/Print', $
                IDENTIFIER='Toolbar/File/Print'
self->Register, IDLitLangCatQuery('Menu:File:PrintPreview'), $
                PROXY='Operations/File/PrintPreview', $
                IDENTIFIER='Toolbar/File/PrintPreview'
self->Register, IDLitLangCatQuery('Menu:File:Export'), $
                PROXY='Operations/File/Export', $
                IDENTIFIER='Toolbar/File/Export'


;;---------------------------------------------------------------------
;;*** Edit Menu

self->createfolders,'Operations/Edit',NAME=IDLitLangCatQuery('Menu:Edit')

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Undo'), $
  'IDLitopUndo', $
  ACCELERATOR='Ctrl+Z', $
  IDENTIFIER='Edit/Undo', ICON='undo', /disable, $
  /IGNORE_AVAILABILITY

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Redo'), $
  'IDLitopRedo', $
  ACCELERATOR='Ctrl+Y', $
  IDENTIFIER='Edit/Redo', ICON='redo',/disable, $
  /IGNORE_AVAILABILITY

;; Clipboard...note these are proxied from the system registry!

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Cut'), $
  PROXY="/REGISTRY/OPERATIONS/CUT",$
  IDENTIFIER='Edit/Cut'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Copy') , $
  PROXY='/REGISTRY/OPERATIONS/COPY', $
  IDENTIFIER='Edit/Copy'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Paste') , $
  PROXY='/REGISTRY/OPERATIONS/Paste', $
  IDENTIFIER='Edit/Paste'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:PasteSpecial') , $
  PROXY='/REGISTRY/OPERATIONS/PASTESPECIAL', $
  IDENTIFIER='Edit/PasteSpecial'

                                ;-----------------
self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Delete') , $
  PROXY='/REGISTRY/OPERATIONS/Delete', $
  IDENTIFIER='Edit/Delete'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:SelectAll') , $
  'IDLitopSelectAll', $
  DESCRIPTION="Select all visualizations in the current view.", $
  IDENTIFIER='Edit/SelectAll', $
  ICON='select'


                                ;-----------------
;self->createfolders,'Operations/Edit/Grouping', $
;                    NAME=IDLitLangCatQuery('Menu:Edit:Grouping')
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Group'), $
;  'IDLitopGroup', $
;  IDENTIFIER='Edit/Grouping/Group', $
;  ICON='group', $
;  /SEPARATOR
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Ungroup'), $
;  'IDLitopUngroup', $
;  IDENTIFIER='Edit/Grouping/Ungroup', $
;  ICON='ungroup'

;self->createfolders,'Operations/Edit/Order', $
;                    NAME=IDLitLangCatQuery('Menu:Edit:Order')
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:BringToFront'), $
;  'IDLitopBringToFront', $
;  IDENTIFIER='Edit/Order/BringToFront', $
;  ICON='front'
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:SendToBack'), $
;  'IDLitopSendToBack', $
;  IDENTIFIER='Edit/Order/SendToBack', $
;  ICON='back'
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:BringForward'), $
;  'IDLitopBringForward', $
;  IDENTIFIER='Edit/Order/BringForward', $
;  ICON='front'
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:SendBackward'), $
;  'IDLitopSendBackward', $
;  IDENTIFIER='Edit/Order/SendBackward', $
;  ICON='back'


;;---------------------------------------------------------------------
;;*** Format Menu

self->createfolders,'Operations/Edit/Style', $
                    NAME=IDLitLangCatQuery('Menu:Style')

self->RegisterOperation, IDLitLangCatQuery('Menu:Style:ApplyStyle') , $
  PROXY='/REGISTRY/OPERATIONS/Apply Style', $
  IDENTIFIER='Edit/Style/Apply Style'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Style:CreateStylefromSelection'), $
  'IDLitopStyleCreate', $
  IDENTIFIER='Edit/Style/Create Style', $
  ICON='style'

self->RegisterOperation, IDLitLangCatQuery('Menu:Style:StyleEditor'), $
  'IDLitopStyleEditor', $
  IDENTIFIER='Edit/Style/StyleEditor', $
  ICON='style'


;;-----------------
;; Register here but make private. We only expose this operation
;; in the context menu, but we want it to be (de)sensitized along with
;; the rest of the Edit menu.
;self->RegisterOperation, 'Export to IDL...', 'IDLitopclExport', $
;  /PRIVATE, $
;  DESCRIPTION='Export Visualization Parameters to IDL Variables. ', $
;  IDENTIFIER='Edit/CLExport', $
;  /separator


;;-----------------
;self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Parameters'), $
;  'IDLitopEditParameters', $
;  IDENTIFIER='Edit/EditParameters', $
;  ICON='binary', $
;  /separator
self->RegisterOperation, 'Edit Parameters','IDLitopEditParameters'

Self->RegisterOperation, 'SaveProperties','DAVEopMenuItem',identifier='Edit/SaveProperties'

self->RegisterOperation, IDLitLangCatQuery('Menu:Edit:Properties'), $
  'IDLitopPropertySheet', $
  DESCRIPTION='Display the Property Sheet for the item', $
  IDENTIFIER='Edit/Properties', $
  ICON='propsheet'


;;---------------------------------------------------------------------
;; Create our Edit toolbar container.
;;
self->Register, IDLitLangCatQuery('Menu:Edit:Undo'), $
                PROXY='Operations/Edit/Undo', $
                IDENTIFIER='Toolbar/Edit/Undo'
self->Register, IDLitLangCatQuery('Menu:Edit:Redo'), $
                PROXY='Operations/Edit/Redo', $
                IDENTIFIER='Toolbar/Edit/Redo'
self->Register, IDLitLangCatQuery('Menu:Edit:Cut'), $
                PROXY="/REGISTRY/OPERATIONS/CUT",$
                IDENTIFIER='Toolbar/Edit/Cut'
self->Register, IDLitLangCatQuery('Menu:Edit:Copy'), $
                PROXY='/REGISTRY/OPERATIONS/COPY', $
                IDENTIFIER='Toolbar/Edit/Copy'
self->Register, IDLitLangCatQuery('Menu:Edit:Paste'), $
                PROXY='/REGISTRY/OPERATIONS/Paste', $
                IDENTIFIER='Toolbar/Edit/Paste'


;;---------------------------------------------------------------------
;;*** Insert menu

self->createfolders,'Operations/Insert', $
                    NAME=IDLitLangCatQuery('Menu:Insert')

;self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:Visualization'), 'IDLitopInsertVis', $
;;  IDENTIFIER='Insert/Visualization', ICON='view'

self->RegisterOperation, 'Insert Visualization','IDLitopInsertVis';, $
;  IDENTIFIER='Insert/Visualization', ICON='view'


self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:View'), $
  'IDLitopInsertView', $
  IDENTIFIER='Insert/View', ICON='view'

self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:DataSpace'), $
  'IDLitopInsertDataSpace', $
  IDENTIFIER='Insert/Data Space', ICON='mcr'

self->createfolders,'Operations/Insert/Axis', $
                    NAME=IDLitLangCatQuery('Menu:Insert:Axis')

self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:XAxis'), $
  'IDLitOpInsertAxisX', $
  IDENTIFIER='Insert/Axis/X Axis', ICON='mcr'
self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:YAxis'), $
  'IDLitOpInsertAxisY', $
  IDENTIFIER='Insert/Axis/Y Axis', ICON='mcr'
self->RegisterOperation, IDLitLangCatQuery('Menu:Insert:ZAxis'), $
  'IDLitOpInsertAxisZ', $
  IDENTIFIER='Insert/Axis/Z Axis', ICON='mcr'


;;---------------------------------------------------------------------
;;*** Operations Menu
self->createfolders,'Operations/Operations', $
                    NAME=IDLitLangCatQuery('Menu:Operations')

;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Operations:OperationsBrowser'), $
;  'IDLitopBrowserOperation', $
;  IDENTIFIER='Operations/Operations Browser' ; ,ICON='mcr'

self->createfolders,'Operations/Operations/Macros', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Macros')

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RunMacro'), $
  PROXY='/Registry/MacroTools/Run Macro', $
  IDENTIFIER='Operations/Macros/Run Macro'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:StartRecording'), $
  PROXY='/Registry/MacroTools/Start Recording', $
  IDENTIFIER='Operations/Macros/Start Recording'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:StopRecording'), $
  PROXY='/Registry/MacroTools/Stop Recording', $
  IDENTIFIER='Operations/Macros/Stop Recording'
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:MacroEditor'), $
  PROXY='/Registry/MacroTools/Macro Editor', $
  IDENTIFIER='Operations/Macros/Macro Editor'

;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Operations:Statistics'), $
;  'IDLitopStatistics', $
;  DESCRIPTION='Display statistics for the selected item', $
;  IDENTIFIER='Operations/Statistics', ICON='sum', /SEPARATOR
;
;;self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:Histogram'), $
;;  'IDLitopHistogram', $
;;  DESCRIPTION='Perform the histogram operation on the selected item', $
;;  IDENTIFIER='Operations/Histogram', ICON='hist'
;
self->createfolders,'Operations/Operations/Filter', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Filter')

;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Operations:Convolution'), $
;  'IDLitopConvolution', $
;  DESCRIPTION='Perform the convolution operation on the selected item', $
;  IDENTIFIER='Operations/Filter/Convolution', ICON='sum'
;
self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:Median'), $
  'IDLitopMedianFilter', $
  DESCRIPTION='Perform the median filter operation on the selected item', $
  IDENTIFIER='Operations/Filter/Median', ICON='sum'

self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:Smooth'), $
  'IDLitopSmooth', $
  DESCRIPTION='Perform the smooth operation on the selected item', $
  IDENTIFIER='Operations/Filter/Smooth', ICON='sum'

self->createfolders,'Operations/Operations/Rotate', $
                    NAME=IDLitLangCatQuery('Menu:Operations:Rotate')

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateLeft'), $
  'IDLitopRotateLeft', $
  DESCRIPTION='Rotate left by 90 degrees', $
  IDENTIFIER='Operations/Rotate/RotateLeft', $
  ICON='rotate'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateRight'), $
  'IDLitopRotateRight', $
  DESCRIPTION='Rotate right by 90 degrees', $
  IDENTIFIER='Operations/Rotate/RotateRight', $
  ICON='rotate'

self->RegisterOperation, $
  IDLitLangCatQuery('Menu:Operations:RotateByAngle'), $
  'IDLitopRotateByAngle', $
  DESCRIPTION='Rotate by a specified angle', $
  IDENTIFIER='Operations/Rotate/RotateByAngle', $
  ICON='rotate'

;self->createfolders,'Operations/Operations/Transform', $
;                    NAME=IDLitLangCatQuery('Menu:Operations:Transform')
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:Resample'), $
;  'IDLitopResample', $
;  IDENTIFIER='Operations/Transform/Resample', ICON='sum'
;
;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Operations:RotateData'), $
;  'IDLitopRotateData', $
;  DESCRIPTION='Rotate the data by a specified angle', $
;  IDENTIFIER='Operations/Transform/RotateData', $
;  ICON='sum'
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:ScaleData'), $
;  'IDLitopScaleFactor', $
;  DESCRIPTION='Scale the data by a given factor', $
;  IDENTIFIER='Operations/Transform/ScaleFactor', ICON='sum'
;

;self->RegisterOperation, IDLitLangCatQuery('Menu:Operations:MapProjection'), $
;  'IDLitopMapProjection', $
;  IDENTIFIER='Operations/Map Projection', ICON='surface'


;;---------------------------------------------------------------------
;;*** Window Menu

self->createfolders,'Operations/Window', $
                    NAME=IDLitLangCatQuery('Menu:Window')

;;  Note: temporarily disabling ICON settings for browsers.
;self->RegisterOperation, IDLitLangCatQuery('Menu:Window:DataManager'), $
;  'IDLitopDataManager', $
;  IDENTIFIER='Window/Data Manager', $
;  ICON='prop'                   ; , ICON='mcr'

;     self->RegisterOperation,'Manipulator Browser','IDLitopBrowserManip', $
;         IDENTIFIER='Window/Manipulator Browser'; , ICON='mcr'

;     self->RegisterOperation, 'Tool Browser', 'IDLitopBrowserTool', $
;         IDENTIFIER='Window/Tool Browser'; , ICON='mcr'

;self->RegisterOperation, $
;  IDLitLangCatQuery('Menu:Window:VisualizationBrowser'), $
;  'IDLitopBrowserVis', $
;  IDENTIFIER='Window/Visualization Browser' ; , ICON='mcr'

;;-----------------
Self->RegisterOperation, 'Reset to Default Data Range', 'IDLitopRangeReset' $
   ,description='Reset to default data range', identifier='Window/Resetrange' $
   ,icon='range_reset'

;;--
self->createfolders,'Operations/Toolbar/Dataspace',name='Dataspace'
self->Register, 'Reset to Default Data Range', $
                PROXY='Operations/Window/Resetrange', $
                IDENTIFIER='Toolbar/Dataspace/Resetrange'

self->createfolders,'Operations/Window/Canvas Zoom', $
                    NAME=IDLitLangCatQuery('Menu:Window:CanvasZoom')

self->RegisterOperation, '800%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/800%', $
  ICON='zoom', /CHECKED;, /SEPARATOR
self->RegisterOperation, '400%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/400%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '200%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/200%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '100%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/100%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '75%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/75%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '50%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/50%', $
  ICON='zoom', /CHECKED
self->RegisterOperation, '25%', 'IDLitopCanvasZoom', $
  IDENTIFIER='Window/Canvas Zoom/25%', $
  ICON='zoom', /CHECKED

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:ZoomonResize'), $
  'IDLitopZoomResize', $
  ICON='zoom', $
  IDENTIFIER='Window/ZoomResize', /CHECKED

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:Layout'), $
  'IDLitopWindowLayout', $
  IDENTIFIER='Window/Layout'

                                ;-----------------

self->RegisterOperation, IDLitLangCatQuery('Menu:Window:FittoView'), $
  'IDLitopFitToView', $
  IDENTIFIER='Window/FitToView', $
  ICON='fitwindow', /SEPARATOR

;;---------------------------------------------------------------------
;;*** Help Menu
self->createfolders,'Operations/Help', $
                    NAME=IDLitLangCatQuery('Menu:Help')


self->RegisterOperation, "User's Manual", 'DAVEopHelp', $
  IDENTIFIER='Help/FANSReductionManual',helptopic='FANS_REDUCTION'
;-----------------

self->RegisterOperation, 'About FANS Data Reduction', 'FANSopHelpAbout', $
  IDENTIFIER='Help/About FANS',/separator

self->RegisterOperation, 'About DAVE', 'DAVEopHelpAbout', $
  IDENTIFIER='Help/About DAVE',/separator




; self->RegisterOperation, IDLitLangCatQuery('Menu:Help:HelponiTools'), $
;   'IDLitopHelpiTools', $
;   ACCELERATOR='F1', $
;   IDENTIFIER='Help/HelpiTools'
;
; self->RegisterOperation, $
;   IDLitLangCatQuery('Menu:Help:HelpontheiToolsDataManager'), $
;   'IDLitopHelpDataManager', IDENTIFIER='Help/HelpDataManager'
;
; self->RegisterOperation, $
;   IDLitLangCatQuery('Menu:Help:HelpontheiToolsParameterEditor'), $
;   'IDLitopHelpParamEditor', IDENTIFIER='Help/HelpParamEditor'
;
; self->RegisterOperation, $
;   IDLitLangCatQuery('Menu:Help:HelponSelectedItem'), $
;   'IDLitopHelpSelection', $
;   IDENTIFIER='Help/HelpSelection', /SEPARATOR
;
; self->RegisterOperation, IDLitLangCatQuery('Menu:Help:HelponthisiTool'), $
;   'IDLitopHelpTool', $
;   IDENTIFIER='Help/HelpTool'
;
;self->RegisterOperation, IDLitLangCatQuery('Menu:Help:AboutiTools'), $
;  'IDLitopHelpAbout', $
;  IDENTIFIER='Help/About iTools'


;;---------------------------------------------------------------------
;;*** Manipulators
self->RegisterManipulator, 'Arrow', 'IDLitManipArrow', $
  ICON='arrow', /DEFAULT, IDENTIFIER="ARROW", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Select')

self->RegisterManipulator, 'Rotate', 'IDLitManipRotate', $
  ICON='rotate', IDENTIFIER="ROTATE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Rotate')

self->RegisterManipulator, 'View Pan', 'IDLitManipViewPan', $
  ICON='hand', IDENTIFIER="VIEWPAN", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:Pan')


;;---------------------------------------------------------------------
;;*** View Zoom Manipulator
self->RegisterManipulator, 'View Zoom', 'IDLitManipViewZoom', $
  IDENTIFIER='View/ViewZoom', ICON='zoom', $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:ViewZoom')

;;---------------------------------------------------------------------
;; *** View Zoom Combobox

                                ; Combobox is not available on True64 (OSF Alpha)
useCombobox = ~(!VERSION.os eq 'OSF')
self->Register, 'View Zoom', 'IDLitopViewZoom', $
                IDENTIFIER='Toolbar/View/ViewZoom', $
                DROPLIST_EDIT=useCombobox, $
                DROPLIST_ITEMS=['800%', $
                                '400%', $
                                '200%', $
                                '150%', $
                                '130%', $
                                '100%', $
                                '75%',  $
                                '50%',  $
                                '25%'], $
                DROPLIST_INDEX=4, $
                /SINGLETON

;;---------------------------------------------------------------------
;;*** Annotation Manipulators
self->RegisterManipulator, 'Text', 'IDLitAnnotateText', $
  ICON='text', IDENTIFIER="ANNOTATION/TEXT", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateText')

self->RegisterManipulator, 'Line', 'IDLitAnnotateLine', $
  ICON='line', IDENTIFIER="ANNOTATION/LINE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateLine')

self->RegisterManipulator, 'Rectangle', 'IDLitAnnotateRectangle', $
  ICON='rectangl', IDENTIFIER="ANNOTATION/RECTANGLE", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateRectangle')

self->RegisterManipulator, 'Oval', 'IDLitAnnotateOval', $
  ICON='ellipse', IDENTIFIER="ANNOTATION/OVAL", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotateOval')

self->RegisterManipulator, 'Polygon', 'IDLitAnnotatePolygon', $
  ICON='segpoly', IDENTIFIER="ANNOTATION/POLYGON", $
  DESCRIPTION=IDLitLangCatQuery('Status:Framework:AnnotatePolygon')

self->RegisterManipulator, 'Freehand', 'IDLitAnnotateFreehand', $
  ICON='freehand', IDENTIFIER="ANNOTATION/FREEHAND", $
  DESCRIPTION='Click & drag to draw'



;;;---------------------------------------------------------------------
;;;*** DrawContext
;self->Register, 'Cut', $
;                PROXY="/REGISTRY/OPERATIONS/CUT",$
;                IDENTIFIER='ContextMenu/DrawContext/Cut'
;
;self->Register, 'Copy', $
;                PROXY='/REGISTRY/OPERATIONS/COPY', $
;                IDENTIFIER='ContextMenu/DrawContext/Copy'
;
;self->Register, 'Paste', $
;                PROXY='/REGISTRY/OPERATIONS/Paste', $
;                IDENTIFIER='ContextMenu/DrawContext/Paste'
;
;                                ;-----------------
;self->Register, 'Delete', $
;                PROXY='/REGISTRY/OPERATIONS/Delete', $
;                IDENTIFIER='ContextMenu/DrawContext/Delete'
;
;                                ;-----------------
;self->Register, 'Group', $
;                PROXY='Operations/Edit/Grouping/Group', $
;                IDENTIFIER='ContextMenu/DrawContext/Grouping/Group'
;
;self->Register, 'Ungroup', $
;                PROXY='Operations/Edit/Grouping/Ungroup', $
;                IDENTIFIER='ContextMenu/DrawContext/Grouping/Ungroup'
;
;self->Register, 'Bring To Front', $
;                PROXY='Operations/Edit/Order/BringToFront', $
;                IDENTIFIER='ContextMenu/DrawContext/Order/BringToFront'
;
;self->Register, 'Send To Back', $
;                PROXY='Operations/Edit/Order/SendToBack', $
;                IDENTIFIER='ContextMenu/DrawContext/Order/SendToBack'
;
;self->Register, 'Bring Forward', $
;                PROXY='Operations/Edit/Order/BringForward', $
;                IDENTIFIER='ContextMenu/DrawContext/Order/BringForward'
;
;self->Register, 'Send Backward', $
;                PROXY='Operations/Edit/Order/SendBackward', $
;                IDENTIFIER='ContextMenu/DrawContext/Order/SendBackward'
;
;                                ;-----------------
;self->Register, 'Export to IDL...', $
;                PROXY='Operations/Edit/CLExport', $
;                IDENTIFIER='ContextMenu/DrawContext/CLExport'

                                ;-----------------
;self->Register, 'Parameters...', $
;                PROXY='Operations/Edit/EditParameters', $
;                IDENTIFIER='ContextMenu/DrawContext/EditParameters'

;self->Register, 'Save Graph Properties...', $
;                PROXY='Operations/Edit/SaveProperties', $
;                IDENTIFIER='ContextMenu/DrawContext/SaveProperties'

self->Register, 'Modify Properties...', $
                PROXY='Operations/Edit/Properties', $
                IDENTIFIER='ContextMenu/DrawContext/Properties'

;;---------------------------------------------------------------------
;;*** CropContext
;self->Register, 'Crop...', 'IDLitopCropImage', $
;                DESCRIPTION='Crop the selected image', $
;                IDENTIFIER='ContextMenu/CropDrawContext/Crop', $
;                ICON='crop', $
;                /SINGLETON

Self.sampContRef = obj_new('IDL_Container') ;obj_new('IDLitContainer',name='SampContainer')
Self.bkgdContRef = obj_new('IDL_Container') ;obj_new('IDLitContainer',name='BkgdContainer')
Self.fastContRef = obj_new('IDL_Container') ;obj_new('IDLitContainer',name='FastContainer')

;; Initialise data and Register Properties
Self.nActiveDetectors = 50

Self.mev2wnos = 8.065541
;Self.wnos2mev = 0.123984   ; 1/Self.mev2wnos
Self.energyUnitFlag = 0
;Self.convFactor = (Self.energyUnitFlag eq 0)? Self.mev2wnos : Self.wnos2mev
Self->RegisterProperty, 'EnergyUnit', enumlist=['meV','cm-1'],name='Energy Units' $
                      ,description='Specify units for energy dimension'

Self.sumSampFlag = 0
Self->RegisterProperty, 'sumSampFlag', enumlist=['No','Yes'], name='Sum Sample Data?' $
                      ,description='Sum all loaded Sample data?', sensitive=0

Self.sumBkgdFlag = 0
Self->RegisterProperty, 'sumBkgdFlag', enumlist=['No','Yes'], name='Sum Bkgd Data?' $
                      ,description='Sum all loaded Bkgd data?', sensitive=0

Self.sumFastFlag = 0
Self->RegisterProperty, 'sumFastFlag', enumlist=['No','Yes'], name='Sum Fast Data?' $
                      ,description='Sum all loaded Fast data?', sensitive=0

Self.maskFlag = 0
Self->RegisterProperty, 'maskFlag', enumlist=['No','Yes'],name='Rem Masked Dets?' $
                      ,description='Remove masked detectors'
Self.maskedDets = ''
Self->RegisterProperty, 'maskedDets', userdef='',name='Masked Detectors' $
                      ,description='Detectors to be masked';, sensitive=Self.maskFlag
;Self->RegisterProperty, 'maskedDets', /string,name='Masked Detectors' $
;                      ,description='Detectors to be masked', sensitive=Self.maskFlag
                      

Self.fastFuncs = ['Arcsin','Linear','Quadratic','Cubic']

Self->RegisterProperty, 'fastfunc', enumlist=Self.fastFuncs, name='Fast Fit Func' $
                        ,description='Function for fitting fast bkgd'
                        
Self->RegisterProperty, 'fastparams', userdef='', name='Fast Fit params' $
                        ,description='Parameters after fitting fast bkgd'

Self.nSamp = 0
Self.nBkgd = 0
Self.nFast = 0
;Self.sampSelected = ''
;Self.bkgdSelected = ''
;Self.fastSelected = ''

Self.bkgdFlag = 0
Self->RegisterProperty, 'bkgdFlag', enumlist=['No','Yes'], name='Sub. Bkgd?' $
                      ,description='Should bkgd data be subtracted' $
                      ,sensitive=(Self.nBkgd gt 0)? 1 : 0

Self.fastFlag = 0
Self->RegisterProperty, 'fastFlag', enumlist=['No','Yes'], name='Sub. Fast Bkgd?' $
                      ,description='Should fast bkgd data be subtracted' $
                      ,sensitive=(Self.nFast gt 0)? 1 : 0

Self.monScaleFlag = 0
Self->RegisterProperty, 'monScaleFlag', enumlist=['No','Yes'], name='Rescale Data?' $
                      ,description='Rescale data by user monitor?'

Self.monScaleValue = 1
Self->RegisterProperty, 'monScaleValue', /float, name='New Monitor scale' $
                      ,description='Rescale data by this monitor', sensitive=Self.monScaleFlag
                      
Self->RegisterProperty, 'oplotFastFitFlag', enumlist=['No','Yes'], name='Overplot Fast Fit?' $
                      ,description='Overplot Fast fit with Fast bkgd?'

Self->RegisterProperty, 'oplotBkgdFlag', enumlist=['No','Yes'], name='Overplot Bkgd?' $
                      ,description='Overplot Bkgd data in samp plot?'

Self->SetPropertyAttribute, 'NAME', /hide
Self->SetPropertyAttribute, 'DESCRIPTION', /hide
Self.projName = 'Untitled'

;; Finally, set the prompt property to 1 meaning display a warning dialog 
;; before deleting this tool
Self.prompt = 1
Self.promptDesc = 'FANS Reduction Window'
Self.promptTitle = 'Delete Reduction Window?'

IDLge90 = (float(!version.release) ge 9.0)? 1 : 0
Self.ftpObject =  (IDLge90)? DAVEHttpRequest() :  DAVEftpURL()
Self.ftpObject->SetProperty, currentDir = '/pub/ncnrdata/bt4/'

Self.historyPtr = ptr_new('')
return, 1

end


;-------------------------------------------------------------------------
; FANStool::SaveVisProps
;+
; Purpose:
;    Retrieve the curent vis properties and save them
;
; Parameters:
;    oVis     - The visualization whose properties are to be saved
;    
;-
pro FANStool::SaveVisProps, oVis
compile_opt idl2

if (~obj_valid(oVis)) then return
if (~obj_isa(oVis,'IDLitVisualization')) then return

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'FANStool::SaveVisProps: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return
    endif
endif

oVis->GetProperty, name=plotname
case strupcase(plotname) of
   'SAMPPLOT': prefix = 's_'
   'BKGDPLOT': prefix = 'b_'
   'FASTPLOT': prefix = 'f_'
   'SAMPBKGDPLOT': prefix = 'sb_'
   'FASTFITPLOT': prefix = 'ff_'
   else: return
endcase

switch strupcase(plotname) of
   'SAMPPLOT':
   'BKGDPLOT':
   'FASTPLOT': begin
      ;; store the vis props
      vTags = ['use_default_color','color','errorbar_color','sym_color','errorbar_capsize' $
              ,'sym_increment','sym_index','sym_size','sym_thick','y_errorbars','antialias' $
              ,'linestyle','nsum','thick','sym_filled','sym_fill_color','transparency']
;      vTags = ['errorbar_capsize','transparency','antialias','color','use_default_color' $
;              ,'sym_increment','sym_index','sym_size','sym_color','sym_thick','y_errorbars' $
;              ,'linestyle','nsum','thick','sym_filled','sym_fill_color']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = oVis->GetPropertyByIdentifier(vTags[i], value)
         if (status) then etc = create_struct(etc,tTags[i],value)
      endfor
      Self->SetProperty, _EXTRA=etc

      ;; store the axes props
      vTags = ['font_name','font_size','text_color']
      tTags = prefix + vTags
      nTags = n_elements(vTags)
      oDataSpace = oVis->GetDataSpace()
      oAxes = oDataSpace->GetAxes(count=naxes)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = oAxes[0]->GetPropertyByIdentifier(vTags[i], value)
         if (status) then etc = create_struct(etc,tTags[i],value)
      endfor
      Self->SetProperty, _EXTRA=etc   
   
      break
   end
   
   'SAMPBKGDPLOT':
   'FASTFITPLOT': begin
      ;; Set the vis props
      vTags = ['transparency','antialias','color','use_default_color','sym_index' $
              ,'linestyle','thick']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = oVis->GetPropertyByIdentifier(vTags[i], value)
         if (status) then etc = create_struct(etc,tTags[i],value)
      endfor
      Self->SetProperty, _EXTRA=etc
   
      break
   end
   
   else:
endswitch

end




;-------------------------------------------------------------------------
; FANStool::customizeVis
;+
; Purpose:
;    Customize a newly created visualization using attributes obtained from the tool
;
; Parameters:
;    oVis     - The visualization to be customized
;    
;    plotname - a string name to be used to identify the visualization
;-
pro FANStool::CustomizeVis, oVis, plotname
compile_opt idl2

if (~obj_valid(oVis)) then return
if (~obj_isa(oVis,'IDLitVisualization')) then return

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'wd_FANStool_customizeVis: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=wTLB)
        catch, /cancel
        return
    endif
endif

case strupcase(plotname) of
   'SAMPPLOT': prefix = 's_'
   'BKGDPLOT': prefix = 'b_'
   'FASTPLOT': prefix = 'f_'
   'SAMPBKGDPLOT': prefix = 'sb_'
   'FASTFITPLOT': prefix = 'ff_'
   else: return
endcase

switch strupcase(plotname) of
   'SAMPPLOT':
   'BKGDPLOT':
   'FASTPLOT': begin
      ;; Set the vis props
      vTags = ['use_default_color','color','errorbar_color','sym_color','errorbar_capsize' $
              ,'sym_increment','sym_index','sym_size','sym_thick','y_errorbars','antialias' $
              ,'linestyle','nsum','thick','sym_filled','sym_fill_color','transparency']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = Self->GetPropertyByIdentifier(tTags[i], value)
         if (status) then etc = create_struct(etc,vTags[i],value)
      endfor

      if (float((strtok(!version.release,/extract))[0]) lt 8.0) then begin
         ; workaround for bug in IDL 7.1 or lower; use_default_color is not honoured for errorbar_color
         if (etc.use_default_color eq 1) then etc.errorbar_color = etc.color
      endif

      oVis->SetProperty, name=strupcase(plotname), _EXTRA=etc
      oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account

      ;; Set the axes props
      vTags = ['font_name','font_size','text_color']
      tTags = prefix + vTags
      nTags = n_elements(vTags)
      oDataSpace = oVis->GetDataSpace()
      oAxes = oDataSpace->GetAxes(count=naxes)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = Self->GetPropertyByIdentifier(tTags[i], value)
         if (status) then etc = create_struct(etc,vTags[i],value)
      endfor
      for i=0,naxes-1 do oAxes[i]->SetProperty, _EXTRA=etc   
   
      break
   end
   
   'SAMPBKGDPLOT':
   'FASTFITPLOT': begin
      ;; Set the vis props
      vTags = ['transparency','antialias','color','use_default_color','sym_index' $
              ,'linestyle','thick']
      tTags = prefix + vTags

      nTags = n_elements(vTags)
      etc = {dummy:''}
      for i=0, nTags-1 do begin
         status = Self->GetPropertyByIdentifier(tTags[i], value)
         if (status) then etc = create_struct(etc,vTags[i],value)
      endfor
      oVis->SetProperty, name=strupcase(plotname), _EXTRA=etc
      oVis->SetPropertyAttribute, 'NSUM', sensitive=0   ; disable point averaging/smoothing b/c errors are not taken into account
   
      break
   end
   
   else:
endswitch



end



;---------------------------------------------------------------------------
; FANSTool::InitPreferences
;
; Purpose:
;   This method
;
pro FANSTool::InitPreferences
compile_opt idl2

;; !!!!!!! For debugging! Remove this !!!!!!!!
;Self->SetProperty, data_directory = '/home/azuah/prog/idlwork/data/fans'

; Init working and data directory from the DAVETool values
;daveTool = Self.daveTool
;if (obj_valid(daveTool) and obj_isa(daveTool,'DAVETool')) then begin
;   daveTool->GetProperty, data_directory=datDir, working_directory=workDir
;   Self->SetProperty, data_directory=datDir, working_directory=workDir
;endif


Self->SetProperty, sumFastFlag=1

; Retrieve user settings on disk and initialise current session
status = fansReducPreferences(preferences)
if (status) then Self->SetProperty, preferences=preferences

Self->GetProperty, maskFlag=mF, monScaleFlag=msF

Self->SetPropertyAttribute, 'MONSCALEVALUE', sensitive=msF
Self->SetPropertyAttribute, 'MASKEDDETS', sensitive=mF

Self->refreshPropertySheet

; Update title to reflect session name
; setting tool_filename generates a callback on the registered UI callback
Self->SetProperty, tool_filename=Self.projName

; Retrieve the graphics window
; and set auto-resize and zoom-on-resize to 1 so the graphics window behaves properly when its size is changed
oWin = Self->GetCurrentWindow()
oWin->SetProperty, auto_resize=1, zoom_on_resize=1

if (float((strtok(!version.release,/extract))[0]) ge 8.0) then begin
   ; For IDL 8.0 and newer, a title text is added to the graphics by default
   ; Hide it!

   idText = iGetID('*title*',tool=Self)
   oText = Self->GetByIdentifier(idText[0])
   if (isa(oText,'IDLitvisText')) then oText->setProperty, hide=1
   ;oWin = Self->GetCurrentWindow()
   ;oScene = oWin->GetScene()
   ;oView = oScene->GetCurrentView()
   ;atoms = oView->GetAll(count=natoms)
   ;for i=0,natoms-1 do begin
   ;   if (obj_isa(atoms[i],'IDLitvisText')) then atoms[i]->setProperty, hide=1
   ;endfor
   Self->RefreshCurrentWindow
endif

end


;---------------------------------------------------------------------------
; FANSTool::resetPreferences
;
; Purpose:
;   This method
;
function FANSTool::resetPreferences
compile_opt idl2


; Retrieve user settings on disk and initialise current session
status = fansReducPreferences(preferences,/reset)
if (status) then Self->SetProperty, preferences=preferences

Self->GetProperty, maskFlag=mF, monScaleFlag=msF

Self->SetPropertyAttribute, 'MONSCALEVALUE', sensitive=msF
Self->SetPropertyAttribute, 'MASKEDDETS', sensitive=mF

Self->refreshPropertySheet

end


;---------------------------------------------------------------------------
; FANSTool::EditUserDefProperty
;
; Purpose:
;   This method
;
function FANSTool::EditUserDefProperty, oTool, PropID
compile_opt idl2

;; The API requires the oTool arg. It is redundant here since it is Self!

oUI = Self->GetUI()
oUI->GetProperty, group_leader=wTLB
wChild = widget_info(wTLB, /child)
widget_control, wChild, get_uvalue=sPtr

case PropID of
   'MASKEDDETS': begin
      ; The currently selected dataset is registered as (*sPtr).itemID
      if (strcmp(strtrim((*sPtr).itemID),'')) then return, 0
      oItem =  Self->GetByIdentifier((*sPtr).itemID)
      void = oItem->GetMetaData('DATAPTRREF',dataPtr)
      if (~ptr_valid(dataPtr)) then break
      
      wd_FANSMaskDetectors, oUI, dataPtr=dataPtr
      Self->refreshPropertySheet   
   end
   
   'FASTPARAMS': begin
      Self->GetProperty, FastSelected = oItem, nFast=nFast
      if (~obj_valid(oItem) && nFast gt 0) then begin
         oRes = Self.fastContRef->Get(/all,count=cnt)
         if (cnt gt 0) then oItem = oRes[cnt-1]
      endif
      if (~obj_valid(oItem)) then return, 0

      void = oItem->GetMetaData('DATAPTRREF',dataPtr)
      if (~ptr_valid(dataPtr)) then break

      counts = (*(*dataPtr).totCntsPtr)
      errors = (*(*dataPtr).totErrsPtr)
      status = Self->Correct4Mask(dataPtr, correctedCounts, correctedErrors)
      if (status) then begin
         counts = correctedCounts
         errors = correctedErrors
      endif

      tmpDataPtr = ptr_new({ $
         counts:counts $
        ,errors:errors $
        ,energy:(*(*dataPtr).energyPtr) $
        ,dspace:(*dataPtr).monDSpace $
        ,date:(*dataPtr).date $
        ,title:(*dataPtr).title $
        ,xtitle:(*dataPtr).xtitle $
        ,ytitle:(*dataPtr).ytitle $
      })

      wd_FANSFitFast, oUI, dataPtr=tmpDataPtr
      ptr_free, tmpDataPtr
      Self->refreshPropertySheet
   end
   
   else:
endcase


return, 1
end


;---------------------------------------------------------------------------
; FANSTool__Define
;
; Purpose:
;   This method defines the FANSTool class.
;
pro FANSTool__Define
compile_opt idl2

void = { FANSTool,                 $
         inherits IDLitTool        $   ; Provides iTool interface
         ,dataDir:''               $   ; data directory (Note: working_directory is defined by the base class
;         ,nameTag:''               $   ; Tag to be used for identification purposes
;         ,prompt:1                 $
         ,nSamp:0                  $   ; nos of valid Sample datasets
         ,nBkgd:0                  $   ; nos of valid bkgd datasets
         ,nFast:0                  $   ; nof of valid fast bkgd datasets
         ,sampContRef:obj_new()    $   ; container for Sample datasets
         ,bkgdContRef:obj_new()    $   ; container for bkgd datasets
         ,fastContRef:obj_new()    $   ; container for fast bkgd datasets
         ,sampSelected:obj_new()   $   ; most recent selection from samp datasets
         ,bkgdSelected:obj_new()   $   ; most recent selection from bkgd datasets
         ,fastSelected:obj_new()   $   ; most recent selection from fast datasets
         ,DAVETool:obj_new()       $   ; DAVETool object
         ,projName:''              $   ; name of current project
         ,nameTag:''               $   ; Tag to be used for identification purposes
         ,prompt:1                 $
         ,promptTitle:''           $
         ,promptDesc:''            $
         ,mev2wnos:8.065541        $   ; meV to wavenumber conversion factor
         ,nActiveDetectors:50      $   ; nos of active detectors
         ,energyUnitFlag:0         $   ; energy unit flag 0=meV, 1=cm-1
         ,sumSampFlag:0            $   ; sum multiple sample files? 0=no, 1=yes
         ,sumBkgdFlag:0            $   ; sum multiple bkgd files? 0=no, 1=yes
         ,sumFastFlag:0            $   ; sum multiple fast files? 0=no, 1=yes
         ,maskedDets:''            $   ; masked detectors (stored as a string)
         ,fastFuncs:strarr(4)      $   ; available functions to fit fast bkgd
         ,fastFunc:0               $   ; function to be used to fit fast bkgd 0=arcsin, 1=linear, 2=quadratic, 3=cubic
         ,fastparams:''            $   ; current fast function fit parameters
         ,monScaleValue:1          $   ; user specified mon scale factor
         ,monScaleFlag:0           $   ; apply userMonNorm scale factor 1=yes, 0=no
         ,maskFlag:0               $   ; remove masked detectors 1=yes, 0=no
         ,bkgdFlag:0               $   ; subtract bkgd data  1=yes, 0=no
         ,fastFlag:0               $   ; apply fast bkgd correction   1=yes, 0=no
         ,oplotFastFitFlag:0       $   ; over plot fast bkgd fit in fast bkgd plot?
         ,oplotBkgdFlag:0          $   ; over plot background data in sample plot?
         ,prefsPtr:ptr_new()       $   ; points to preference structure
         ,historyPtr:ptr_new()     $
         ,modifiedFlag:0B          $   ; determines modification state of tool - unsufficient control over _bDirty property!
         ,dataVersion:0            $   ; current version number for dataset structure
         ,ftpObject:obj_new()      $   ; DAVEftpURL() object used to access data on ftp://ftp.ncnr.nist.gov/pub/ncnrdata/
         
         ,s_errorbar_capsize:    0.1 $
         ,s_transparency:        50 $
         ,s_antialias:           3 $
         ,s_color:               [0,0,255] $
         ,s_errorbar_color:      [0,0,255] $
         ,s_use_default_color:   1   $
         ,s_sym_increment:       1   $
         ,s_sym_index:           4   $
         ,s_sym_size:            1.0   $          ; 
         ,s_sym_color:           [0,0,255] $   ;
         ,s_sym_thick:           1.0 $            ;
         ,s_y_errorbars:         1 $              ; show y errors
         ,s_linestyle:           6 $              ; no line
         ,s_nsum:                1 $              ; no point everaging
         ,s_thick:               1.0 $            ; line thickness
         ,s_sym_filled:          1   $            ; use filled symbols
         ,s_sym_fill_color:      [0,0,255] $   ; symbol fill color
         ,s_font_name:           'Helvetica'  $   ;
         ,s_font_size:           13 $             ;
         ,s_text_color:          [0,0,0]  $    ;

         ,b_errorbar_capsize:    0.1 $
         ,b_transparency:        50 $
         ,b_antialias:           3 $
         ,b_color:               [0,0,255] $
         ,b_errorbar_color:      [0,0,255] $
         ,b_use_default_color:   1   $
         ,b_sym_increment:       1   $
         ,b_sym_index:           4   $
         ,b_sym_size:            1.0   $          ; 
         ,b_sym_color:           [0,0,255] $   ;
         ,b_sym_thick:           1.0 $            ;
         ,b_y_errorbars:         1 $              ; show y errors
         ,b_linestyle:           6 $              ; no line
         ,b_nsum:                1 $              ; no point everaging
         ,b_thick:               1.0 $            ; line thickness
         ,b_sym_filled:          1   $            ; use filled symbols
         ,b_sym_fill_color:      [0,0,255] $   ; symbol fill color
         ,b_font_name:           'Helvetica'  $   ;
         ,b_font_size:           13 $             ;
         ,b_text_color:          [0,0,0]  $    ;

         ,f_errorbar_capsize:    0.1 $
         ,f_transparency:        50 $
         ,f_antialias:           3 $
         ,f_color:               [0,0,255] $
         ,f_errorbar_color:      [0,0,255] $
         ,f_use_default_color:   1   $
         ,f_sym_increment:       1   $
         ,f_sym_index:           4   $
         ,f_sym_size:            1.0   $          ; 
         ,f_sym_color:           [0,0,255] $   ;
         ,f_sym_thick:           1.0 $            ;
         ,f_y_errorbars:         1 $              ; show y errors
         ,f_linestyle:           6 $              ; no line
         ,f_nsum:                1 $              ; no point everaging
         ,f_thick:               1.0 $            ; line thickness
         ,f_sym_filled:          1   $            ; use filled symbols
         ,f_sym_fill_color:      [0,0,255] $   ; symbol fill color
         ,f_font_name:           'Helvetica'  $   ;
         ,f_font_size:           13 $             ;
         ,f_text_color:          [0,0,0]  $    ;

         ,sb_transparency:        60 $
         ,sb_antialias:           3 $
         ,sb_color:               [0,128,0] $
         ,sb_use_default_color:   1   $
         ,sb_sym_index:           0   $            ; no symbol
         ,sb_linestyle:           0 $              ;
         ,sb_thick:               5.0 $            ; line thickness

         ,ff_transparency:        60 $
         ,ff_antialias:           3 $
         ,ff_color:               [0,128,0] $
         ,ff_use_default_color:   1   $
         ,ff_sym_index:           0   $            ; no symbol
         ,ff_linestyle:           0 $              ;
         ,ff_thick:               5.0 $            ; line thickness


       }
end
