; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataScale
;
; PURPOSE:
;   A simple operation that multiplies the data of the selected
;   IDLitData object(s) by a scalefactor.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataScale::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [out] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataScale::GetProperty, scalefactor=scalefactor, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(scalefactor)) then $
  scalefactor =  self._scalefactor

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataScale::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [in] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataScale::SetProperty, scalefactor=scalefactor, _EXTRA=etc
compile_opt idl2

; The UI dialog should be resposible for checking the integrity of the
; value of scalefactor so will not be repeated here
if (n_elements(scalefactor) gt 0) then $
  self._scalefactor = scalefactor

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataScale::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataScale::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, scalefactor=scalefactor
void = oCmd->AddItem('OLD_SCALEFACTOR',scalefactor)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataScale::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataScale::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, scalefactor=scalefactor
void = oCmd->AddItem('NEW_SCALEFACTOR',scalefactor)

return, 1

end


;===============================================================================
; DAVEopDataScale::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataScale::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataScale::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataScale::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'old' scalefactor value and undo the change on the target
if (oCmds[0]->GetItem('OLD_SCALEFACTOR',scalefactor)) then begin
    oTarget->SetProperty, scalefactor=scalefactor
endif

; Loop through rest of commands and undo the changes that were made
; to the targets
if (nCmds gt 1) then begin
    void = oCmds[0]->GetItem('NEW_SCALEFACTOR',scalefactor) ; retrieve scalefactor that was applied to data
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Undo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin
            ;; modify the data
            void = oTarget->SetData(data/scalefactor) ; undo offet
            ;; revert to previous treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('OLD_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; DAVEopDataScale::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataScale::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'new' scalefactor value and redo the change on the target
; (self)
if (oCmds[0]->GetItem('NEW_SCALEFACTOR',scalefactor)) then begin
    oTarget->SetProperty, scalefactor=scalefactor
endif

; Loop through rest of commands and redo the changes that were made
; to the targets (datasets)
if (nCmds gt 1) then begin
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Redo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin
            ;; modify the data
            void = oTarget->SetData(data * scalefactor) ; re-apply scalefactor
            ;; re-apply the new treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('NEW_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; DAVEopDataScale::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's scalefactor to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataScale::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oTarget = oTool->GetSelectedData()
void = where(obj_valid(oTarget),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate all those that are IDLitData objects, but not
; containers.
for i=0,n_elements(oTarget)-1 do begin
    if (obj_valid(oTarget[i]) && $
        obj_isa(oTarget[i],'IDLitData') && $
      ~obj_isa(oTarget[i],'IDL_Container') ) then $
        index = (n_elements(index) gt 0)? [index,i] : i
endfor
if (n_elements(index) eq 0) then begin
    oTool->StatusMessage, 'Invalid selection(s). Select data items not folders!'
    return, obj_new()
endif

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Record initial value for the scalefactor property of this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ; Don't allow a scalefactor of 0.0
    if (self._scalefactor eq 0.0) then begin
        ;; Undo the change made by user in DoExecuteUI()
        oCmd = oCmdSet->Get(position=0)
        void = oCmd->GetItem('OLD_SCALEFACTOR',scalefactor)
        self._scalefactor = scalefactor
        ;; Display a status message and exit
        oTool->StatusMessage, 'Scalefactor cannot be set to 0.0!'
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final value for the scalefactor property of this operation.
    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

; Apply the scalefactor to the data
; NB: no properties are being modified for the target objects hence
; there are no initial/final values to be recorded
for i = 0,n_elements(index)-1 do begin
   if (oTarget[i]->GetData(data)) then begin
      ;; modify the data
      void = oTarget[i]->SetData(data * self._scalefactor)
      oCmd = obj_new('IDLitCommand',target_identifier=oTarget[i]->getfullidentifier())

      ;; Handle treatment history
      oTrmt = LocateTreatmentObject(oTarget[i])
      if (obj_valid(oTrmt)) then begin
         oTarget[i]->GetProperty, name=name
         void = oTrmt->GetData(trmt)         ; retrieve existing trmt details
         
         ; create new treatment info
         line = '____________________________________________________'
         newtrmt = [trmt, $
                   line, $
                   'Timestamp: '+systime(), $
                   'Apply an scalefactor of '+strtrim(string(self._scalefactor),2)+" to '"+name+"'"]

         void = oTrmt->SetData(newtrmt)   ; modify the tratment details
         
         ; record info to enable undo/redo functionality
         void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier()) ; store old treatment object reference
         void = oCmd->AddItem('OLD_TREATMENT',trmt) ; store old treatment info
         void = oCmd->AddItem('NEW_TREATMENT',newtrmt) ; and new treatment info
      endif

      oCmdSet->Add, oCmd
    endif
endfor

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataScale::Cleanup
; 
; PURPOSE:
;   DAVEopDataScale class cleanup
;
pro DAVEopDataScale::Cleanup

compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataScale::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataScale::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
if (~self->IDLitOperation::Init(NAME='Scalefactor' $
                                ,_EXTRA=etc)) then return, 0
;types=['IDLVECTOR','IDLARRAY2D','IDLARRAY3D','IDLIMAGE'] $
                                

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; This operation is reversible
; This operation is not expensive
self->SetProperty, reversible_operation=1, expensive_operation=0

; Register an scalefactor property for this operation
self->RegisterProperty, 'Scalefactor', /float, description='Scalefactor to be applied to data' $
  ,name='Scalefactor',sensitive=1

; init scalefactor to 1.0
self._scalefactor = 1.0

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataScale__define
; 
; PURPOSE:
;   DAVEopDataScale class structure definition
;
pro DAVEopDataScale__define

compile_opt idl2

struc = {DAVEopDataScale $
         ,inherits IDLitOperation $
         ,_scalefactor:1.0 $         ; scale factor to be applied to data
        }

end
