; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataOffset
;
; PURPOSE:
;   A simple operation that adds an additive offset to the data of an
;   IDLitData object.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataOffset::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataOffset::GetProperty, offset=offset, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(offset)) then $
  offset =  self._offset

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataOffset::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataOffset::SetProperty, offset=offset, _EXTRA=etc
compile_opt idl2

; The UI dialog should be resposible for checking the integrity of the
; value of offset so will not be repeated here
if (n_elements(offset) gt 0) then $
  self._offset = offset

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataOffset::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataOffset::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, offset=offset
void = oCmd->AddItem('OLD_OFFSET',offset)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DAVEopDataOffset::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataOffset::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, offset=offset
void = oCmd->AddItem('NEW_OFFSET',offset)

return, 1

end


;===============================================================================
; DAVEopDataOffset::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataOffset::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDataOffset::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataOffset::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'old' offset value and undo the change on the target
if (oCmds[0]->GetItem('OLD_OFFSET',offset)) then begin
    oTarget->SetProperty, offset=offset
endif

; Loop through rest of commands and undo the changes that were made
; to the targets
if (nCmds gt 1) then begin
    void = oCmds[0]->GetItem('NEW_OFFSET',offset) ; retrieve offset that was applied to data
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Undo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin
            ;; modify the data
            void = oTarget->SetData(data - offset) ; undo offet
            ;; revert to previous treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('OLD_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; DAVEopDataOffset::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataOffset::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)

; Get the 'new' offset value and redo the change on the target
; (self)
if (oCmds[0]->GetItem('NEW_OFFSET',offset)) then begin
    oTarget->SetProperty, offset=offset
endif

; Loop through rest of commands and redo the changes that were made
; to the targets (datasets)
if (nCmds gt 1) then begin
    for i=1,nCmds-1 do begin
        oCmds[i]->GetProperty, target_identifier=idTarget
        oTarget = oTool->GetByIdentifier(idTarget)
        if (~obj_valid(oTarget)) then begin
            oTool->StatusMessage,'Missing dataset! Redo could not be completed'
            continue
        endif
        if (oTarget->GetData(data)) then begin
            ;; modify the data
            void = oTarget->SetData(data + offset) ; re-apply offset
            ;; re-apply the new treatment history
            if (oCmds[i]->GetItem('ID_TREATMENT',idTrmt)) then begin
               oTrmt = oTool->GetByIdentifier(idTrmt)
               if (oCmds[i]->GetItem('NEW_TREATMENT',trmt)  && obj_valid(oTrmt)) then $
                  void = oTrmt->SetData(trmt)
            endif
        endif
    endfor
endif

return, 1

end


;===============================================================================
; DAVEopDataOffset::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataOffset::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oTarget = oTool->GetSelectedData()
void = where(obj_valid(oTarget),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate all those that are IDLitData objects, and not
; containers.
for i=0,n_elements(oTarget)-1 do begin
    if (obj_valid(oTarget[i]) && $
        obj_isa(oTarget[i],'IDLitData') && $
      ~obj_isa(oTarget[i],'IDL_Container') ) then $
        index = (n_elements(index) gt 0)? [index,i] : i
endfor
if (n_elements(index) eq 0) then begin
    oTool->StatusMessage, 'Invalid selection(s). Select data items not folders!'
    return, obj_new()
endif

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    
    ;; Record initial value for the offset property of this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final value for the offset property of this operation.
    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

; Apply the offset to the data
; NB: no properties are being modified for the target objects hence
; there are no initial/final values to be recorded
for i = 0,n_elements(index)-1 do begin
   if (oTarget[i]->GetData(data)) then begin
      ;; modify the data
      void = oTarget[i]->SetData(data + self._offset)
      ;; create a command object to record this change.
      oCmd = obj_new('IDLitCommand',target_identifier=oTarget[i]->getfullidentifier())

      ;; Handle treatment history
      oTrmt = LocateTreatmentObject(oTarget[i])
      if (obj_valid(oTrmt)) then begin
         oTarget[i]->GetProperty, name=name
         void = oTrmt->GetData(trmt)         ; retrieve existing trmt details
         
         ; create new treatment info
         line = '____________________________________________________'
         newtrmt = [trmt, $
                   line, $
                   'Timestamp: '+systime(), $
                   'Apply an offset of '+strtrim(string(self._offset),2)+" to '"+name+"'"]

         void = oTrmt->SetData(newtrmt)   ; modify the tratment details
         
         ; record info to enable undo/redo functionality
         void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier()) ; store old treatment object reference
         void = oCmd->AddItem('OLD_TREATMENT',trmt) ; store old treatment info
         void = oCmd->AddItem('NEW_TREATMENT',newtrmt) ; and new treatment info
      endif

      oCmdSet->Add, oCmd
   endif
endfor

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataOffset::Cleanup
; 
; PURPOSE:
;   DAVEopDataOffset class cleanup
;
pro DAVEopDataOffset::Cleanup

compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataOffset::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataOffset::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
if (~self->IDLitOperation::Init(NAME='Data Offset' $
                                ,_EXTRA=etc)) then return, 0
;types=['IDLVECTOR','IDLARRAY2D','IDLARRAY3D','IDLIMAGE'] 

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; This operation is reversible
; This operation is not expensive
self->SetProperty, reversible_operation=1, expensive_operation=0

; Register an offset property for this operation
self->RegisterProperty, 'Offset', /float, description='Offset to be applied to data' $
  ,name='Offset',sensitive=1

; init offset to 0.0
self._offset = 0.0

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataOffset__define
; 
; PURPOSE:
;   DAVEopDataOffset class structure definition
;
pro DAVEopDataOffset__define

compile_opt idl2

struc = {DAVEopDataOffset $
         ,inherits IDLitOperation $
         ,_offset:0.0 $         ; offset to be applied to data
        }

end
